// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19010 $
//  $Date: 2007-12-10 03:54:57 -0500 (Mon, 10 Dec 2007) $
//  $Author: andre $


// Rosetta Headers
#include "pose_rms.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "docking_ns.h"
#include "enzyme.h"
#include "maxsub.h"
#include "maxsub_threshold.h"
#include "orient_rms.h"
#include "pose.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/Time_Date.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


//Utility Headers
#include <utility/basic_sys_util.hh>

///////////////////////////////////////////////////////////////////////////////
// mapping is a mapping from the pose sequence to the native sequence

void
calc_homolog_rms(
	pose_ns::Pose & pose,
	pose_ns::Pose const & native_pose,
	FArray1D_int const & mapping
)
{
	using namespace pose_ns;
	assert( int(mapping.size1()) >= pose.total_residue() );

	pose.set_0D_score( RMSD, CA_rmsd_by_mapping( pose, native_pose, mapping));
	pose.set_0D_score( MAXSUB, CA_maxsub_by_mapping(pose,native_pose,mapping));
}

/////////////////////////////////////////////////////////////////
void
fill_rmsd_coordinates(
			int & natoms,
			FArray2D_double & p1a,
			FArray2D_double & p2a,
			const pose_ns::Pose & pose1,
			const pose_ns::Pose & pose2,
			const FArray1D_int & mapping ) {

	using namespace param_aa;

	int const nres1 = pose1.total_residue();
	int const nres2 = pose2.total_residue();

	FArray3D_float const & Epos1 ( pose1.Eposition() );
	FArray3D_float const & Epos2 ( pose2.Eposition() );
	FArray3D_float const & full_coord1 ( pose1.full_coord() );
	FArray3D_float const & full_coord2 ( pose2.full_coord() );

	natoms = 0;
	for ( int i = 1; i <= nres1; ++i ) {
		const int imap( mapping(i) );
		if ( imap >= 1 ) {
			if ( imap > nres2 ) {
				std::cout << "STOP: bad mapping in rms calculation. " <<
					imap << ' ' << nres2 << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if ( pose1.pseudo( i ) || pose2.pseudo( imap ) ) continue;
			++natoms;

			int const aa = pose1.res(i);
			for ( int k = 1; k <= 3; ++k ) {
				if (is_protein( aa )) {
					static int const atom_index_protein ( 2 ); // CA
					p1a(k,natoms) = Epos1( k, atom_index_protein, i );
					p2a(k,natoms) = Epos2( k, atom_index_protein, imap );
				}
				if (is_NA( aa )) {
					static int const atom_index_NA( 6 ); //C4*
					assert( pose1.fullatom() && pose2.fullatom());
					p1a(k,natoms) = full_coord1( k, atom_index_NA, i );
					p2a(k,natoms) = full_coord2( k, atom_index_NA, imap );
				}
				if (is_ligand( aa )) {
					static int const atom_index_LG( 1 );
					assert( pose1.fullatom() && pose2.fullatom());
					p1a(k,natoms) = full_coord1( k, atom_index_LG, i );
					p2a(k,natoms) = full_coord2( k, atom_index_LG, imap );
				}
			}
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
//
// TO DO: might be good to make following function fill p1a and p2a
//  using fill_rms_coordinates, which treats RNA right.
//
float
CA_rmsd_by_subset(
	const pose_ns::Pose & pose1,
	const pose_ns::Pose & pose2,
	const FArray1D_bool & subset
)
{
	assert( pose1.total_residue() == pose2.total_residue() );

	const int nres( pose1.total_residue() );
	int natoms(0);
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );

	FArray1D_int mapping( nres, 0);
	for ( int i = 1; i <= nres; i++ ) {
		if (subset(i)) mapping(i) = i;
	}
	fill_rmsd_coordinates( natoms, p1a, p2a, pose1, pose2, mapping );

	if (natoms == 0) return 0.0;

	// calc rms
	FArray1D_double const ww( natoms, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	findUU( p1a, p2a, ww, natoms, uu, ctx );
	float fast_rms;
	calc_rms_fast( fast_rms, p1a, p2a, ww, natoms, ctx );
	return fast_rms;
}

/////////////////////////////////////////////////////////////////
void
fill_rmsd_coordinates(
			FArray2D_double & p1a,
			FArray2D_double & p2a,
			const pose_ns::Pose & pose1,
			const pose_ns::Pose & pose2
			){
	int natoms( 0 );
	int const nres1 = pose1.total_residue();
	FArray1D_int mapping(nres1, 0);
	for (int i = 1; i <= nres1; i++) mapping(i) = i;
	fill_rmsd_coordinates( natoms, p1a, p2a, pose1, pose2, mapping);
}


///////////////////////////////////////////////////////////////////////////////
float
CA_rmsd_by_mapping(
	const pose_ns::Pose & pose1,
	const pose_ns::Pose & pose2,
	const FArray1D_int & mapping
)
{
	const int nres1( pose1.total_residue() );

	int natoms(0);
	FArray2D_double p1a( 3, nres1 );
	FArray2D_double p2a( 3, nres1 );

	fill_rmsd_coordinates( natoms, p1a, p2a, pose1, pose2, mapping );

	if (natoms == 0) return 0.0;

	// calc rms
	FArray1D_double const ww( natoms, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	findUU( p1a, p2a, ww, natoms, uu, ctx );
	float fast_rms;
	calc_rms_fast( fast_rms, p1a, p2a, ww, natoms, ctx );
	return fast_rms;
}

///////////////////////////////////////////////////////////////////////////////
int
CA_maxsub_by_mapping(
	const pose_ns::Pose & pose1,
	const pose_ns::Pose & pose2,
	const FArray1D_int & mapping
)
{
	const int nres1( pose1.total_residue() );
	assert( int(mapping.size1()) >= nres1 );

	int natoms(0);
	FArray2D_double p1a( 3, nres1 );
	FArray2D_double p2a( 3, nres1 );
	fill_rmsd_coordinates( natoms, p1a, p2a, pose1, pose2, mapping );

	double mxrms, mxpsi, mxzscore, mxscore, mxeval;
	int nali;
	maxsub( natoms, p1a, p2a, mxrms, mxpsi, nali, mxzscore, mxeval, mxscore );
	//logeval = std::log(mxeval);
	return nali;
}

int
CA_maxsub(
	const pose_ns::Pose & pose1,
	const pose_ns::Pose & pose2
	){
	int const nres1 = pose1.total_residue();
	FArray1D_int mapping(nres1, 0);
	for (int i = 1; i <= nres1; i++) mapping(i) = i;
	return CA_maxsub_by_mapping( pose1, pose2, mapping);
}

///////////////////////////////////////////////////////////////////////////////
float
calc_rms(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	float nat_rms(0.0);
	int nat_maxsub(0);
	if ( pose.native_pose_exists() ) {
		nat_rms = CA_rmsd( pose.native_pose(), pose );
		nat_maxsub = CA_maxsub( pose.native_pose(), pose );
	}
	pose.set_0D_score( RMSD, nat_rms );
	pose.set_0D_score( MAXSUB, nat_maxsub );

	if ( pose.start_pose_exists() ) {
		pose.set_0D_score( START_RMSD, CA_rmsd( pose, pose.start_pose() ) );
	}

	return nat_rms;
}

///////////////////////////////////////////////////////////////////////////////
float
CA_rmsd(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2
)
{
	int const total_residue( pose1.total_residue() );
	//	if ( total_residue != pose2.total_residue() ) {
	//		std::cout << "length mismatch in CA_rmsd(...) " <<
	//			total_residue << ' ' << pose2.total_residue() << std::endl;
	//		return 0.0;
	//	}

	// copy coords into double arrays
	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	fill_rmsd_coordinates( p1a, p2a, pose1, pose2 );

	// Calc rms
	FArray1D_double const ww( total_residue, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	float fast_rms;

	if ( docking::docking_pose_symm_full && pose2.symmetric() ){

		int const nres_monomer ( pose2.symmetry_info().nres_monomer_bb() );
		int const N ( pose2.symmetry_info().N_bb() );
		int const nres ( N*nres_monomer );
		FArray2D_double p1a_shuffle(3,nres);
		float fast_rms_shuffle;

		fast_rms = 1e3; //Since fast_rms has not been evaluated yet

//		static bool const combinatorial_rms_sym = truefalseoption("combinatorial_rms_sym");
		if (docking::combinatorial_rms_sym ) {
			// Create a vector of all combinations of n number of integers
			// output is found in shuffle_map. The map is used to shuffle
			// chains and the lowest rms is saved
			std::vector< std::vector<int> > shuffle_map;
			create_shuffle_map_recursive_rms(std::vector<int>(), N,shuffle_map);
			for (int j=1; j < int (shuffle_map.size()); j++ ){
				for (int i=0; i < N; ++i ) {
					int const begin ( shuffle_map.at(j).at(i)*nres_monomer*3);
					for ( int j = 0; j < nres_monomer*3; ++j ) {
						int const begin_shuffled (i*nres_monomer*3);
            p1a_shuffle[begin_shuffled+j] = p1a[begin+j];
				  }
				}
				findUU( p1a_shuffle, p2a, ww, total_residue, uu, ctx );
				calc_rms_fast( fast_rms_shuffle, p1a_shuffle, p2a, ww, total_residue, ctx );
				if ( fast_rms_shuffle < fast_rms ) {
				fast_rms = fast_rms_shuffle;
				}
			}
		} else {

			shuffle_chains_for_symmetry(p1a, p1a_shuffle,N,nres,nres_monomer);
			findUU( p1a, p2a, ww, total_residue, uu, ctx );
			float fast_rms_shuffle;
			calc_rms_fast( fast_rms, p1a, p2a, ww, total_residue, ctx );
			findUU( p1a_shuffle, p2a, ww, total_residue, uu, ctx );
			calc_rms_fast( fast_rms_shuffle, p1a_shuffle, p2a, ww, total_residue, ctx );
			if ( fast_rms_shuffle < fast_rms ) fast_rms = fast_rms_shuffle;
		}
	} else {
		findUU( p1a, p2a, ww, total_residue, uu, ctx );
		calc_rms_fast( fast_rms, p1a, p2a, ww, total_residue, ctx );
	}

	return fast_rms;
}

///////////////////////////////////////////////////////////////////////////////
float
allatom_rmsd(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2
)
{
	int const nres1 = pose1.total_residue();
	FArray1D_int mapping(nres1, 0);
	for (int i = 1; i <= nres1; i++) mapping(i) = i;
	return allatom_rmsd_by_mapping( pose1, pose2, mapping, true );
}

///////////////////////////////////////////////////////////////////////////////
float
LG_rmsd( pose_ns::Pose const & pose ) {
	float nat_rms(0.0);
	if ( pose.native_pose_exists() ) {
		nat_rms = LG_rmsd( pose.native_pose(), pose );
	}
	return nat_rms;
}

///////////////////////////////////////////////////////////////////////////////
float
LG_rmsd(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2
)
{
	int const nres1 = pose1.total_residue();
	FArray1D_int mapping(nres1, 0);
	for (int i = 1; i <= nres1; i++) {
		if( param_aa::is_ligand(pose1.res(i)) ) mapping(i) = i;
		else mapping(i) = 0;
	}
	return allatom_rmsd_by_mapping( pose1, pose2, mapping, false );
}

///////////////////////////////////////////////////////////////////////////////
float
allatom_rmsd_by_mapping(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2,
	FArray1D_int const & mapping,
	bool fit_rms
)
{
	int const total_residue( pose1.total_residue() );
	if ( total_residue != pose2.total_residue() ) {
		std::cout << "length mismatch in allatom_rmsd(...) " <<
			total_residue << ' ' << pose2.total_residue() << std::endl;
		return 0.0;
	}

	// copy coords into double arrays
	FArray3D_float const & fcoord1 ( pose1.full_coord() );
	FArray3D_float const & fcoord2 ( pose2.full_coord() );
	FArray2D_double p1a( 3, total_residue*param::MAX_ATOM()() );
	FArray2D_double p2a( 3, total_residue*param::MAX_ATOM()() );
	int natoms(0);
	for ( int i = 1; i <= total_residue; ++i ) {
		const int imap( mapping(i) );
		if( imap < 1 ) continue;
		assert( pose1.res(i) == pose2.res(imap) &&
						pose1.res_variant(i) == pose2.res_variant(imap) );
		int const natoms_aa_aav
			( aaproperties_pack::natoms(pose1.res(i), pose1.res_variant(i) ) );
		for ( int j = 1; j <= natoms_aa_aav; ++j ) {
			++natoms;
			for ( int k = 1; k <= 3; ++k ) {
				p1a(k,natoms) = fcoord1( k, j, i );
				p2a(k,natoms) = fcoord2( k, j, imap );
			}
		}
	}

	// calc rms
	FArray1D_double const ww( natoms, 1.0 );
	FArray2D_double uu( 3, 3, 0.0 );
	double ctx ( 0.0 );
	if( fit_rms ) {
		findUU( p1a, p2a, ww, natoms, uu, ctx );
		float fast_rms;
		calc_rms_fast( fast_rms, p1a, p2a, ww, natoms, ctx );
		return fast_rms;
	} else {
		float fix_rms ( 0.0 );
		for ( int k = 1; k <= natoms; ++k ) {
			double const ww_k = ww(k);
			for ( int j = 1; j <= 3; ++j ) {
				double const xx_jk = p1a(j,k);
				double const yy_jk = p2a(j,k);
				fix_rms += ww_k * ( xx_jk - yy_jk ) * ( xx_jk - yy_jk );
			}
		}
		return std::sqrt(fix_rms/natoms);
	}
	return 0;//should not be here
}

/////////////////////////////////////////////////////////////////////////////
void
per_res_CA_rmsd(
	pose_ns::Pose const & pose1,
	pose_ns::Pose const & pose2,
	std::vector< float > & per_res_rmsd
)
{
	int const total_residue( pose1.total_residue() );
	if ( total_residue != pose2.total_residue() ) {
		std::cout << "length mismatch in per_res_CA_rmsd(...) " <<
			total_residue << ' ' << pose2.total_residue() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::vector<float> residue_dist;
		// copy coords into double arrays
	FArray3D_float Epos1 ( pose1.Eposition() );
	FArray3D_float full1 ( pose1.full_coord() );
	FArray3D_float const & Epos2 ( pose2.Eposition() );

	int orient_start = { 1 };
	int orient_end = { total_residue };

	float local_rms;

	orient_region(total_residue,Epos1,full1,Epos2,1,
	 orient_start,orient_end,1,1,total_residue,local_rms);

	for ( int i = 1; i <= total_residue; ++i ) {
		per_res_rmsd.push_back( 0.0 );
		for ( int j = 1; j <= 3; ++j ) {
			float const d = Epos1(j,2,i) - Epos2(j,2,i);
			per_res_rmsd.back() += d*d;
		}
		per_res_rmsd.back() = std::sqrt(per_res_rmsd.back());
	}
}

///////////////////////////////////////////////////////////////////////////////
void
maxsub_orient_pose(
	pose_ns::Pose & pose1,
	pose_ns::Pose const & pose2,
	bool const ideal_pos
)
{
	int const total_residue( pose1.total_residue() );
	if ( total_residue != pose2.total_residue() ) {
		std::cout << "length mismatch in maxsub_orient_pose... " <<
			total_residue << ' ' << pose2.total_residue() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}


	FArray1D_bool maxsub_alignment( total_residue );

	// copy coords into double arrays
	FArray3D_float Epos1 ( pose1.Eposition() );
	FArray3D_float const & Epos2 ( pose2.Eposition() );
	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}


	// now find out the maxsub alignment
	double mxrms,mxpsi,mxzscore,mxscore,mxeval,rmstol;
	int nali;
	rmstol = 2.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment);

	// find out the aligned regions
	FArray1D_int alignstart, alignend;
	std::vector< int > starts, ends;

	bool aligned( false );
	int loop_start(1);
	int loop_end(1);
	int total_aligned(0);
	for( int i = 1; i <= total_residue; ++i ){
		if ( maxsub_alignment(i) && !aligned ){
			loop_start = i;
			loop_end = i;
			aligned = true;
			total_aligned++;
		}
		else if( maxsub_alignment(i) && aligned ){
			loop_end = i;
			total_aligned++;
		}
		else if( !maxsub_alignment(i) && aligned ){
			starts.push_back( loop_start );
			ends.push_back( loop_end );
			aligned = false;
		}
	}
	if ( aligned ) { // last loop
		starts.push_back( loop_start );
		ends.push_back( loop_end );
	}

	int const aligned_regions( int(starts.size() ) );
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	for( int i = 0; i < aligned_regions; ++i ){
		alignstart(i+1) = starts[i];
		alignend(i+1) = ends[i];
	}


	FArray3D_float full1 ( pose1.full_coord() );

	float local_rms;
	orient_region(total_residue,Epos1,full1,Epos2,aligned_regions,
	 alignstart,alignend,aligned_regions,alignstart,alignend,local_rms);

	pose1.set_coords( ideal_pos, Epos1, full1 );

}

///////////////////////////////////////////////////////////////////////////////
float
simple_rms(
	int const npoints,
	FArray2Da_float p1,
	FArray2Da_float p2
)
{
	p1.dimension(3,npoints);
	p2.dimension(3,npoints);

	static FArray2D_double p1a;
	static FArray2D_double p2a;
	if ( int(p1a.size2()) < npoints ) {
		p1a.dimension(3,npoints);
		p2a.dimension(3,npoints);
	}

	for ( int i = 1; i <= npoints; ++i ) {
		for ( int k=1; k<= 3; ++k ) {
			p1a(k,i) = p1(k,i);
			p2a(k,i) = p2(k,i);
		}
	}

	// calc rms
	FArray1D_double const ww( npoints, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	findUU( p1a, p2a, ww, npoints, uu, ctx );
	float fast_rms;
	calc_rms_fast( fast_rms, p1a, p2a, ww, npoints, ctx );
	return fast_rms;
}


///////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
namespace pose_ns {

	void Pose::CA_orient(
											 Pose const & pose
											 ) {
		FArray1D_bool subset( total_residue_, true );
		CA_orient_subset( pose, subset );
	}

///////////////////////////////////////////////////////////////////////////////
	void Pose::CA_orient_subset(
															Pose const & pose,
															FArray1D_bool const & subset
	)
	{

		// param: which atom to orient on
		//		int const atom_index ( 2 ); // CA //use more general machinery in fill_rmsd_coordinates.
		int const nres = pose.total_residue();

		if ( total_residue_ != nres ) {
			std::cout << "length mismatch in Pose::CA_rmsd(...) " <<
				total_residue_ << ' ' << pose.total_residue() << std::endl;
			return;
		}

		// signal change to Eposition array
		new_Eposition();

		// copy coords into double arrays
		FArray2D_double p1( 3, total_residue_ );
		FArray2D_double p2( 3, total_residue_ );
		int natoms(0);

		FArray1D_int mapping( nres, 0);
		for ( int i = 1; i <= nres; i++ ) {
			if (subset(i)) mapping(i) = i;
		}
		fill_rmsd_coordinates( natoms, p1, p2, *this, pose, mapping );

		// calc rms
		FArray1D_double p1_offset( 3 );
		FArray1D_double p2_offset( 3 );
		FArray2D_double uu( 3, 3 );
		double sigma3;
		FArray1D_double const ww( natoms, 1.0 );

		// according to the usage in orient_region
		// p1 is the moving array, so this is from Eposition
		findUU_trans( p1, p2, ww, natoms, uu, sigma3, p1_offset, p2_offset );

		UU_rotate( Eposition_, Eposition_.size2() * Eposition_.size3(), p1_offset,
							 p2_offset, uu );
		UU_rotate( centroid_, centroid_.size2(), p1_offset, p2_offset, uu );
		if ( fullatom_ ) {
			UU_rotate( full_coord_, full_coord_.size2()*full_coord_.size3(),
								 p1_offset, p2_offset, uu );
		}
	}


	///////////////////////////////////////////////////////////////////////////////
	void Pose::allatom_orient(
														Pose const & pose
	)
	{
		FArray1D_bool subset( total_residue_, true );
		allatom_orient_subset( pose, subset );
	}


	///////////////////////////////////////////////////////////////////////////////
	void Pose::allatom_orient_subset(
															Pose const & pose,
															FArray1D_bool const & subset
	)
	{
		//Sorry, this is horrendously inefficient code.
		Pose & pose1 ( *this );
		Pose const & pose2 (  pose );

		int const nres = total_residue_;
		FArray1D_int mapping( nres, 0);
		for ( int i = 1; i <= nres; i++ ) {
			if (subset(i)) mapping(i) = i;
		}

		// copied from allatom_rmsd :
		FArray3D_float const & fcoord1 ( pose1.full_coord() );
		FArray3D_float const & fcoord2 ( pose2.full_coord() );
		FArray2D_double p1( 3, total_residue_*param::MAX_ATOM()() );
		FArray2D_double p2( 3, total_residue_*param::MAX_ATOM()() );
		int natoms(0);
		for ( int i = 1; i <= total_residue_; ++i ) {
			const int imap( mapping(i) );
			if( imap < 1 ) continue;
			assert( pose1.res(i) == pose2.res(imap) &&
							pose1.res_variant(i) == pose2.res_variant(imap) );
			int const natoms_aa_aav
				( aaproperties_pack::natoms(pose1.res(i), pose1.res_variant(i) ) );
			for ( int j = 1; j <= natoms_aa_aav; ++j ) {
				++natoms;
				for ( int k = 1; k <= 3; ++k ) {
					p1(k,natoms) = fcoord1( k, j, i );
					p2(k,natoms) = fcoord2( k, j, imap );
				}
			}
		}

		// calc rms
		FArray1D_double p1_offset( 3 );
		FArray1D_double p2_offset( 3 );
		FArray2D_double uu( 3, 3 );
		double sigma3;
		FArray1D_double const ww( natoms, 1.0 );

		// according to the usage in orient_region
		// p1 is the moving array, so this is from Eposition
		findUU_trans( p1, p2, ww, natoms, uu, sigma3, p1_offset, p2_offset );

		UU_rotate( Eposition_, Eposition_.size2() * Eposition_.size3(), p1_offset,
							 p2_offset, uu );
		UU_rotate( centroid_, centroid_.size2(), p1_offset, p2_offset, uu );
		if ( fullatom_ ) {
			UU_rotate( full_coord_, full_coord_.size2()*full_coord_.size3(),
								 p1_offset, p2_offset, uu );
		}
	}

} //namespace pose_ns

// ///////////////////////////////////////////////////////////////////////////////
// void Pose::bb_orient_mapping(
// 														 Pose const & fixed_pose,
// 														 FArray1D_int const & mapping
// 														 )
// {
// 	const FArray3D_float & fixed_Epos ( fixed_pose.Eposition() );
// 	const int fixed_nres( fixed_pose.total_residue() );
// 	assert( int(mapping.size1()) >= total_residue );

// 	int natoms(0);
// 	FArray2D_double p1( 3, total_residue*param::MAX_POS );
// 	FArray2D_double p2( 3, total_residue*param::MAX_POS );
// 	for ( int i = 1; i <= total_residue; ++i ) {
// 		const int imap( mapping(i) );
// 		if ( imap >= 1 ) {
// 			if ( imap > fixed_nres ) {
// 				std::cout << "STOP: bad mapping to bb_orient_mapping: " <<
// 					imap << ' ' << fixed_nres << std::endl;
// 				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 			}
// 			for ( int j=1; j<= param::MAX_POS; ++j ) {
// 				if ( j==3 ) continue; // SKIP C-BETA
// 				++natoms;
// 				for ( int k = 1; k <= 3; ++k ) {
// 					p1(k,natoms) = Eposition ( k, j, i );
// 					p2(k,natoms) = fixed_Epos( k, j, imap );
// 				}
// 			}
// 		}
// 	}

// 	// calc rms
// 	FArray1D_double p1_offset( 3 );
// 	FArray1D_double p2_offset( 3 );
// 	FArray2D_double uu( 3, 3 );
// 	double sigma3;
// 	FArray1D_double const ww( natoms, 1.0 );

// 	// according to the usage in orient_region
// 	// p1 is the moving array, so this is from Eposition
// 	findUU_trans( p1, p2, ww, natoms, uu, sigma3, p1_offset, p2_offset );

// 	UU_rotate( Eposition, Eposition.size2() * Eposition.size3(),
// 						 p1_offset, p2_offset, uu );
// 	UU_rotate( centroid, centroid.size2(), p1_offset, p2_offset, uu );
// 	if ( fullatom ) {
// 		UU_rotate( full_coord, full_coord.size2()*full_coord.size3(),
// 							 p1_offset, p2_offset, uu );
// 	}
// }

// ///////////////////////////////////////////////////////////////////////////////
// // CA-rmsd over subset of the residues
// float Pose:: calc_subset_rms(
// 														 Pose const & pose,
// 														 FArray1D_bool const & subset
// 														 ) const
// {
// 	if ( total_residue != pose.total_residue() ) {
// 		std::cout << "length mismatch in Pose::calc_subset_rms(...) " <<
// 			total_residue << ' ' << pose.total_residue() << std::endl;
// 		return 0.0;
// 	}

// 	// copy coords into double arrays
// 	FArray3D_float const & Epos ( pose.Eposition() );
// 	FArray2D_double p1a( 3, total_residue );
// 	FArray2D_double p2a( 3, total_residue );
// 	int const atom_index ( 2 ); // CA
// 	int natoms(0);
// 	for ( int i = 1; i <= total_residue; ++i ) {
// 		if ( subset(i) ) {
// 			++natoms;
// 			for ( int k = 1; k <= 3; ++k ) {
// 				p1a(k,natoms) = Eposition( k, atom_index, i );
// 				p2a(k,natoms) = Epos     ( k, atom_index, i );
// 			}
// 		}
// 	}

// 	// calc rms
// 	FArray1D_double const ww( natoms, 1.0 );
// 	FArray2D_double uu( 3, 3 );
// 	double ctx;
// 	findUU( p1a, p2a, ww, natoms, uu, ctx );
// 	float fast_rms;
// 	calc_rms_fast( fast_rms, p1a, p2a, ww, natoms, ctx );
// 	return fast_rms;
// }

void
create_shuffle_map_recursive_rms(
        std::vector<int> sequence,
				int const N,
				std::vector< std::vector<int> > & map
)
{
				// This is a recursive algorithm to generate all combinations of
				// n digits where a number can only occur once in the sequence.
				// The size scales as N! so don't use this for large values of N!!!

        if ( int(sequence.size()) == N ){
								map.push_back(sequence);
                return;
        }
        for (int i=0; i< N; i++) {
                bool exist (false);
                for (int j=0; j < int(sequence.size()); j++) {
                        if (sequence.at(j) == i )
                                exist = true;
                }
                if (!exist) {
                        std::vector<int> sequence_tmp (sequence);
                        sequence_tmp.push_back(i);
                        create_shuffle_map_recursive_rms(sequence_tmp,N,map);
                }
        }
}

