// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: rhiju $


// Rosetta Headers
#include "pose_rna.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_tree_routines.h"
#include "current_pose.h"
#include "dna.h"
#include "dna_ns.h"
#include "files_paths.h" //for setting up initial rna pose from fasta.
#include "force_barcode.h"
#include "fragments_pose.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "hbonds.h"
#include "job_distributor.h"
#include "jumping_util.h" // debug
#include "jumping_loops.h"
#include "kin_stub.h"
#include "minimize.h"
#include "nblist.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_disulfides.h" //for copy_jumps
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_rna_base_doublet_classes.h"
#include "pose_rna_csa.h"
#include "pose_rna_featurizer.h"
#include "pose_rna_fragments.h"
#include "pose_rna_fullatom.h"
#include "pose_rna_jumping.h"
#include "pose_rna_ns.h"
#include "pose_rna_pdbstats.h"
#include "pose_vdw.h"
#include "read_aaproperties.h"
#include "random_numbers.h"
#include "read_aa_ss.h"
#include "refold.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h"
#include "silent_input.h"

#include "prof.h"

#include "misc.h"
#include "orient_rms.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.io.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

// BOINC
#ifdef BOINC
#include "boinc_rosetta_util.h"
#include "counters.h"
#include "trajectory.h"
#endif

using namespace pose_ns;

namespace rna_scoring_user_defined {
	FArray4D_float user_defined_basepair( param::MAX_RES(), param::MAX_RES(),
																				rna_scoring::NUM_EDGES, rna_scoring::NUM_EDGES, 0.0f );
	FArray2D_float user_defined_long_range_contact( param::MAX_RES(), param::MAX_RES(), 0.0f );

	std::vector< std::pair<int,int> > long_range_contact_residues;

	FArray1D_float user_defined_sasa( param::MAX_RES(), 0.0f );
	bool user_defined_sasa_flag = false;
}


inline
float
arccos(
	float const x
)
{
	return std::acos( numeric::sin_cos_range( x ) );
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Look in rna_vdw_pdb_stas [in pose_rna_pdbstats.cc ]
// for details. The parameters are in
// pose_rna_ns.cc, and are the thir largest atom-atom
// distance seen in the large ribosomal subunit
// crystal structure 1ffk.pdb.
float
rna_rna_vdw(pose_ns::Pose & pose, int const j, int const i, float const cendist_ij){
	if (i==j) return 0.0; // For now, don't worry about bumps within the residues... trust the fragments.

	using namespace param_aa;
	using namespace rna_scoring;
	using namespace numeric;

	static bool init( false );
	if (!init) {
		//		setup_rna_scoring();
		initialize_rna_vdw_atoms();
		init = true;
	}

	int const res_i( pose.res(i) );
	assert( is_RNA(res_i));

	int const res_j( pose.res(j) );
	assert( is_RNA(res_j));

	float vdw_score = 0.0;
	FArray3D_float const & full_coord( pose.full_coord() );

	bool local_verbose = false;

	for (int m = 1; m <= num_rna_vdw_atoms_check; m++) {
		for (int n = 1; n <= num_rna_vdw_atoms_check; n++) {

			int const atomnum_i = rna_vdw_atom(m, res_i);
 			int const atomnum_j = rna_vdw_atom(n, res_j);

			xyzVector_float const r_i ( & full_coord(1, atomnum_i, i));
			xyzVector_float const r_j ( & full_coord(1, atomnum_j, j));

			float const atomvdw = rna_vdw_parameter( m, n, res_i, res_j) * rna_vdw_scale;
			float const dist2 = atomvdw - distance_squared( r_i, r_j );
			if ( dist2 > 0.0 ) {
				vdw_score += ( dist2 * dist2 ) / atomvdw;
				if (local_verbose) std::cout << "Found a bump! " <<
														 aa_name3(res_i) << i << " "  <<
														 aaproperties_pack::atom_name(atomnum_i, res_i, 1) << " ---- " <<
														 aa_name3(res_j) << j << "  "  <<
														 aaproperties_pack::atom_name(atomnum_j, res_j, 1) <<
														 ": Penalty " << ( dist2 * dist2 ) / atomvdw <<
														 "   sqrt(atomvdw) " << std::sqrt(atomvdw) <<
														 "   dist " << std::sqrt( distance_squared(r_i,r_j)) << std::endl;
			}
		}
	}

	//centroid-centroid bump check
	float const atomvdw = 3.0 * 3.0;
	float dist2 = atomvdw - cendist_ij;
	if ( dist2 > 0.0 ) vdw_score += ( dist2 * dist2 ) / atomvdw;

	//	std::cout << "VDW: " << i << "  " << j << " " << vdw_score << std::endl;

	return vdw_score;
}

///////////////////////////////////////////////////////////////////////////////
bool
get_harmonic_constraint(){
	static bool init = false;
	static bool harmonic_constraint = false;
	if (!init){
		harmonic_constraint = truefalseoption( "harmonic_constraint" );
		init = true;
	}
	return harmonic_constraint;
}
///////////////////////////////////////////////////////////////////////////////
bool
get_rna_verbose(){
	static bool init = false;
	static bool rna_verbose = false;
	if (!init){
		rna_verbose = truefalseoption( "rna_verbose" );
		init = true;
	}
	return rna_verbose;
}

///////////////////////////////////////////////////////////////////////////////
void
eval_rna_long_range_contact_score( float & rna_long_range_contact_score,
																	 float & rna_contact_score,
																	 pose_ns::Pose & pose )
{
	using namespace rna_scoring_user_defined;
	pose_update_cendist( pose ); //may not be necessary if computing VDW anyway.
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );
	float cendist_ij;


	//This used to be computed based on centroid-centroid.
	// For MOHCA purposes, I really want long range contact score
	// to be based on backbone-backbone (C4'-C4')
	float old_rna_long_range_contact_score( 0.0 );

	int const total_residue = pose.total_residue();
	for (int i = 1; i <= total_residue; i++ ){
		for (int j = 1; j <= total_residue; j++ ){

			cendist_ij = cendist(i,j);

			//Apply user-defined constraints
			float const sqrt_cendist_ij = std::sqrt( cendist_ij );
			if (user_defined_long_range_contact(i,j) > 0.1 &&
					sqrt_cendist_ij > user_defined_long_range_contact(i,j)) {
				float const distance_offset = sqrt_cendist_ij - user_defined_long_range_contact(i,j) ;
				old_rna_long_range_contact_score += 0.1 * distance_offset * distance_offset;
			}

			//For user defined base-pairs, see eval_rna_base_pair_score() below. This
			// block here will only activate an additional long range harmonic constraint
			// to guide each pair of bases together if "-harmonic_constraint" is specified.
			if (get_harmonic_constraint()) {
				bool user_defined_basepair_exists = false;
				float distance_cutoff( 0.0 );
				for (int k = 1; k <= rna_scoring::NUM_EDGES; k++) {
					for (int m = 1; m <= rna_scoring::NUM_EDGES; m++) {
						if (user_defined_basepair(i,j,k,m) > 0.1 &&
								sqrt_cendist_ij > user_defined_basepair(i,j,k,m))	{
							user_defined_basepair_exists = true;
							distance_cutoff = user_defined_basepair(i,j,k,m);
							break;
						}
					}
				}
				//total hack. Quadratic potential is mighty strong!
				//default basepair cutoff is 8.0 A
				if (user_defined_basepair_exists){
					float const distance_offset = sqrt_cendist_ij - distance_cutoff;
					rna_contact_score += 0.1 * distance_offset * distance_offset;
				}
				//				std::cout << "DIAGNOSTIC FOR NATIVE: Missing: " << i << " " << j  << std::endl;
			}
		}
	}

	static bool const use_old_rna_long_range_contact_score = truefalseoption("use_old_rna_long_range_contact_score");
	if (use_old_rna_long_range_contact_score){
		rna_long_range_contact_score = old_rna_long_range_contact_score;
		return;
	}

	pose.set_extra_score( "OLDRNALR", old_rna_long_range_contact_score );

	//////////////////////////////////////////////////////////////////
	//OK, the real long-range score, based on C4' <--> C4' distances.
	//////////////////////////////////////////////////////////////////
	FArray3D_float const & full_coord( pose.full_coord() );

	int unsatisfied_long_range_constraints( 0 );

	for (int i = 1; i <= total_residue; i++ ){
		for (int j = i+1; j <= total_residue; j++ ){

			//Apply user-defined constraints
			if (user_defined_long_range_contact(i,j) > 0.1){

				numeric::xyzVector_float const r_i ( & full_coord(1, rna_variables::c4star, i));
				numeric::xyzVector_float const r_j ( & full_coord(1, rna_variables::c4star, j));

				float C4_C4_dist = (r_i - r_j).length();

				if (C4_C4_dist > user_defined_long_range_contact(i,j)) {

					unsatisfied_long_range_constraints++;
					float const distance_offset = C4_C4_dist - user_defined_long_range_contact(i,j) ;
					rna_long_range_contact_score += 0.2 * distance_offset * distance_offset;

				}

			}

		}
	}

	pose.set_extra_score( "UNS_LR", unsatisfied_long_range_constraints);

}


///////////////////////////////////////////////////////////////////////////////
void
evaluate_long_range_contact_distances( pose_ns::Pose & pose ){

	using namespace rna_scoring_user_defined;

	pose_update_cendist( pose );
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );

	int const num_long_range_contacts = long_range_contact_residues.size();
	for (int n = 0; n < num_long_range_contacts; n++ ){
		int const i = long_range_contact_residues[n].first;
		int const j = long_range_contact_residues[n].second;
		pose.set_extra_score(  "DIST"+string_of(n+1), std::sqrt(cendist(i,j)) );
	}

}

///////////////////////////////////////////////////////////////////////////////
// Useful for visualizing (and making a figure for a paper)
void
print_pdb_centroid( numeric::xyzVector_float const & centroid_i,
										numeric::xyzVector_float const & x_i,
										numeric::xyzVector_float const & y_i,
										numeric::xyzVector_float const & z_i,
										int const i)
{
	numeric::xyzVector_float blah;
	blah = centroid_i;
	std::cerr << "HETATM    1  MG   rG      " << i+2 << "   " <<
		F(8,3,blah(1)) << F(8,3,blah(2)) << F(8,3,blah(3)) << "  1.00 51.50" << std::endl;
	blah = centroid_i + 10.0f * x_i;
	std::cerr << "HETATM    1  MG   rG      " << i+2 << "   " <<
		F(8,3,blah(1)) << F(8,3,blah(2)) << F(8,3,blah(3)) << "  1.00 51.50" << std::endl;
	blah = centroid_i + 10.0f * y_i;
	std::cerr << "HETATM    1  MG   rG      " << i+2 << "   " <<
			F(8,3,blah(1)) << F(8,3,blah(2)) << F(8,3,blah(3)) << "  1.00 51.50" << std::endl;
	blah = centroid_i + 10.0f * z_i;
	std::cerr << "HETATM    1  MG   rG      " << i+2 << "   " <<
		F(8,3,blah(1)) << F(8,3,blah(2)) << F(8,3,blah(3)) << "  1.00 51.50" << std::endl;
}

///////////////////////////////////////////////////////////////////////////////
void
update_base_pair_array( pose_ns::Pose & pose,
												FArray3D_float & base_pair_array,
												FArray3D_float & base_axis_array,
												FArray3D_float & base_stagger_array,
												FArray2D_float & base_stack_array,
												FArray2D_float & base_stack_axis_array,
												FArray2D_float & base_geometry_orientation_array,
												FArray2D_float & base_geometry_height_array,
												bool const rna_array_state_ok
											)
{
	using namespace rna_scoring;
	using namespace pose_ns;
	using namespace param_aa;
	using namespace numeric;
	using namespace kin;
  using numeric::conversions::degrees;

	pose_update_cendist( pose ); //may not be necessary if computing VDW anyway.
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );
	float cendist_ij;

	FArray2D_bool const & pair_moved( pose.get_pair_moved() );

	static bool const output_coordinates = false;
	if (output_coordinates) pose.dump_pdb( "TEMP.pdb");

	float const CENDIST2_CUTOFF = 12.0*12.0;

	float const BASEPAIR_HEIGHT  = rna_basepair_stagger_cutoff; // 3.0
	float const BASEPAIR_RADIUS2 = rna_basepair_radius_cutoff * rna_basepair_radius_cutoff; //8.0

	//	float const BASESTACK_HEIGHT   = 6.2;
	float const BASESTACK_MIN_HEIGHT   = 2.4;
	float const BASESTACK_MAX_HEIGHT   = 6.0;
	float const BASESTACK_RADIUS2  = 4.0 * 4.0;

	float const Z_CUTOFF( 2.5 );

	int const total_residue = pose.total_residue();
	static bool const rna_verbose = get_rna_verbose();
	FArray3D_float const & full_coord( pose.full_coord() );

	static bool const disallow_zero_score_base_pair = truefalseoption( "disallow_zero_score_base_pair" );

	//	base_pair_array  = 0.0;
	//	base_stack_array = 0.0;
	//	base_axis_array  = 0.0;
	//	base_stagger_array  = 0.0;

	//Main loop
	// NOTE: Following evaluates each base pair from both sides, because base pair
	// terms are asymmetric in i<->j. So loop over the j index starts at 1, not i+1.

	for (int i = 1; i <= total_residue; i++ ){
		int const res_i = pose.res( i );

    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );
    Stub stub_i = get_base_coordinate_system(res_i, full_coord(1, 1, i), centroid_i );
    xyzMatrix_float const M_i( stub_i.M );
    xyzVector_float const x_i = M_i.col_x();
    xyzVector_float const y_i = M_i.col_y();
    xyzVector_float const z_i = M_i.col_z();

		if (output_coordinates) print_pdb_centroid( centroid_i, x_i, y_i, z_i, i);

		for (int j = 1; j <= total_residue; j++ ){
			if (i==j) continue;
			if ( !pair_moved(i,j) && rna_array_state_ok ) continue;

			//Initialize to zero.
			base_stack_array(i, j) = 0.0;
			base_stack_axis_array(i, j) = 0.0;
			for (int k = 1; k <= NUM_EDGES; k++) {
				base_pair_array   (i, j, k) = 0.0;
				base_axis_array   (i, j, k) = 0.0;
				base_stagger_array(i, j, k) = 0.0;
			}
			base_geometry_orientation_array(i, j) = 0.0;
			base_geometry_height_array(i, j) = 0.0;

			int const res_j = pose.res( j );
			if (!is_RNA( res_j)) continue;

			cendist_ij = cendist(i,j);

			//OK, actually check for base pairs.
			if ( cendist_ij < CENDIST2_CUTOFF ) {
				xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );

				xyzVector_float d_ij = centroid_j - centroid_i;
				float const dist_x = dot_product( d_ij, x_i );
				float const dist_y = dot_product( d_ij, y_i );
				float const dist_z = dot_product( d_ij, z_i );
				float const rho2 = dist_x*dist_x + dist_y*dist_y;

				Stub stub_j = get_base_coordinate_system(res_j, full_coord(1, 1, j), centroid_j );
				xyzMatrix_float const M_j( stub_j.M );
				xyzVector_float const x_j = M_j.col_x();
				xyzVector_float const y_j = M_j.col_y();
				xyzVector_float const z_j = M_j.col_z();
				float const cos_theta = dot_product( z_i, z_j );
				float const theta = degrees( arccos( cos_theta ) );

				//Following block of variables is only used if we're doing a really
				// stringent test of base pair geometry, conditional on the base-base distance,
				// and the relative angles that each base x axis makes to the inter-base vector.
				//
				//For efficiency, we really should only calculate this stuff if we're doing that style of
				// calculation.
				//
				//Also, if we're doing this new style of calculation, its totally pairwise symmetric --
				// so could avoid about half the base pair computation! Hmmm. Let's see how it works.
				static bool const use_rho_omega1_omega2 = truefalseoption( "use_rho_omega1_omega2" );
				float const omega_i   = degrees( std::atan2( dist_y, dist_x ) );
				xyzVector_float d_ji = -1.0f * d_ij;
				float const dist_x_j = dot_product( d_ji, x_j );
				float const dist_y_j = dot_product( d_ji, y_j );
				float const dist_z_j = dot_product( d_ji, z_j );
				float const omega_j   = degrees( std::atan2( dist_y_j, dist_x_j ) );
				float const avg_z = ( abs(dist_z) + abs(dist_z_j) ) / 2.0 ;
				float const rho = std::sqrt( rho2 );

				//Is it a base-pair, a base-stack, or do we ignore it?
				int edge_bin( 1 );

				//Slightly different cutoffs on base coplanarity depending on style of base
				// pair energy calculation.
				if ( ( !use_rho_omega1_omega2 & std::abs(dist_z) < BASEPAIR_HEIGHT) |
						 ( use_rho_omega1_omega2 & avg_z < Z_CUTOFF )){

					if ( rho2 < BASEPAIR_RADIUS2){
						//BASE PAIR
						float temp_rna_bp_score( 0.0 );

						if  (use_rho_omega1_omega2) {
							temp_rna_bp_score = get_rna_basepair_rho_omega1_omega2( rho, omega_i, omega_j, theta, res_i, res_j );
						} else {
							temp_rna_bp_score = get_rna_basepair_xy(dist_x, dist_y, theta, res_i, res_j);
						}

						// Some of the base pairing scores are 0.0. By default, pad by
						// a tiny tiny amount to make sure that they're still
						// counted as very weak base pairs during later book-keeping.
						if (!disallow_zero_score_base_pair) temp_rna_bp_score -= 0.0001;

						float const zeta = degrees( std::atan2( dist_y, dist_x) );
						if (abs(zeta) < 60.0)      edge_bin = WATSON_CRICK;  //Watson-Crick edge
						else if ( zeta > +60.0 )   edge_bin = HOOGSTEEN; // Hoogsteen edge
						else                       edge_bin = SUGAR; // Sugar edge

						if ( rna_verbose ){
							std::cout << " Possible base pair: "
												<< aa_name3( res_i) << I(3,i) << "-"
												<< aa_name3( res_j) << I(3,j)
												<< " edge: " << I(1,edge_bin)
												<< "  dists " << F(4,2,dist_x)
												<< " " << F(4,2,dist_y)
												<< " " << F(4,2,dist_z)
												<< " theta " << F(5,1,theta)
												<< "; rho " << F(4,2,rho)
												<< " omega1 " << F(5,1,omega_i)
												<< " omega2 " << F(5,1,omega_j)
												<< " : SCORE " << F(6,4,temp_rna_bp_score)
								//												<< " " << get_rna_axis_score( theta) << " " << get_rna_stagger_score( dist_z )
												<<	std::endl;
						}

						base_pair_array    ( i, j, edge_bin ) = temp_rna_bp_score;
						base_axis_array    ( i, j, edge_bin ) = get_rna_axis_score( theta );

						if (use_rho_omega1_omega2) {
							//Want symmetry.
							base_stagger_array ( i, j, edge_bin ) =
								0.5 * (get_rna_stagger_score( dist_z ) + get_rna_stagger_score( dist_z_j ) );
						} else {
							base_stagger_array ( i, j, edge_bin ) = get_rna_stagger_score( dist_z );
						}

					}
				}
				if ( std::abs(dist_z) >= BASESTACK_MIN_HEIGHT &&
						 std::abs(dist_z) <= BASESTACK_MAX_HEIGHT &&
							rho2 < BASESTACK_RADIUS2 ) {
					//BASE STACK
					base_stack_array( i, j ) = -0.5;
					base_stack_axis_array ( i, j ) = get_rna_axis_score( theta );
				} //basepair

				base_geometry_orientation_array( i, j ) = cos_theta;
				base_geometry_height_array     ( i, j ) = dist_z;
			} //cendist_ij
		} //j
	}//i

}

///////////////////////////////////////////////////////////////////////////////
// Sort list of base pairs by energy, and go down list -- don't allow
// any base pairs that are mutually exclusive!
///////////////////////////////////////////////////////////////////////////////
void
eval_rna_base_pair_score( float & rna_bp_w_score, float & rna_bp_h_score, float & rna_bp_s_score,
													float & rna_axis_score, float & rna_stagger_score,
													float & rna_contact_score,
													FArray3D_float const & base_pair_array,
													FArray3D_float const & base_axis_array,
													FArray3D_float const & base_stagger_array,
													FArray2D_float const & base_geometry_orientation_array,
													FArray2D_bool & scored_base_pair,
													FArray2D_bool & edge_is_base_pairing,
													Energy_base_pair_list & scored_base_pair_list,
													int const total_residue
													)
{
	using namespace rna_scoring;
	using namespace rna_scoring_user_defined;

	static bool const rna_verbose = get_rna_verbose();

	//A rigorous list of base pairs, for scoring.
	Energy_base_pair_list energy_base_pair_list;

	scored_base_pair = false;
	scored_base_pair_list.clear();

	float const SCORE_CUTOFF = -0.001;

	for (int i = 1; i <= total_residue; i++ ){
		for (int k = 1; k <= NUM_EDGES; k++ ){

			for (int j = i+1; j <= total_residue; j++ ){
			//A base pair is only real if each partner called the other one a true partner.
				if (base_pair_array(i,j,k) < SCORE_CUTOFF ){

					int found_match( 0 );
					float tmp_energy = 0.0;
					for (int m = 1; m <= NUM_EDGES; m++){
						if (base_pair_array(j,i,m) < tmp_energy){
							found_match = m;
							tmp_energy = base_pair_array(j, i, m);
						}
					}//m
					if (found_match == 0) continue;

					float const total_base_pair_energy =
						base_pair_array(i,j,k) + base_pair_array(j,i,found_match);

					Base_pair base_pair;
					base_pair.res1 = i;
					base_pair.edge1 = k;

					base_pair.res2 = j;
					base_pair.edge2 = found_match;

					//orientations are cos( theta ) and should be symmetric!
					assert( std::abs( base_geometry_orientation_array( i, j ) - base_geometry_orientation_array( j, i ) ) < 1.0e-2 );
					base_pair.orientation = ( base_geometry_orientation_array( i, j ) + base_geometry_orientation_array( j, i ) < 0.0 ? 1 : 2);

					energy_base_pair_list.push_back( std::make_pair( total_base_pair_energy, base_pair )  );

				}//is it a basepair?
			} //j

		} //k
	}//i

	energy_base_pair_list.sort(); //Start with the lowest energy base pairs.

	static bool const scale_axis_stagger_by_xy_score = truefalseoption( "scale_axis_stagger_by_xy_score" );

	for ( Energy_base_pair_list::const_iterator it = energy_base_pair_list.begin();
				it != energy_base_pair_list.end(); ++it ){
		float const energy = it->first;
		Base_pair const base_pair = it->second;

		int const i = base_pair.res1;
		int const k = base_pair.edge1;

		int const j = base_pair.res2;
		int const m = base_pair.edge2;

		if (edge_is_base_pairing( i, k )) continue;
		if (edge_is_base_pairing( j, m )) continue;

		edge_is_base_pairing( i, k ) = true;
		edge_is_base_pairing( j, m ) = true;

		//Made it! This is a base pair!
		if (k==1) rna_bp_w_score += base_pair_array(i,j,k);
		if (k==2) rna_bp_h_score += base_pair_array(i,j,k);
		if (k==3) rna_bp_s_score += base_pair_array(i,j,k);

		if (m==1) rna_bp_w_score += base_pair_array(j,i,m);
		if (m==2) rna_bp_h_score += base_pair_array(j,i,m);
		if (m==3) rna_bp_s_score += base_pair_array(j,i,m);

		float scalefactor ( 1.0f ) ;
		if ( scale_axis_stagger_by_xy_score ) scalefactor = 0.1 * std::abs( energy );

		float const scaled_axis_energy = scalefactor * ( base_axis_array(i,j,k) + base_axis_array(j,i,m) );
		float const scaled_stagger_energy = scalefactor * ( base_stagger_array(i,j,k) + base_stagger_array(j,i,m) );
		rna_axis_score    += scaled_axis_energy;
		rna_stagger_score += scaled_stagger_energy;

		if (rna_verbose) {
			//			std::cout << "BASE PAIR: " << I(3,i) << " " << I(3,j) << "   edges: " << I(3,k) << " " << I(3,m) <<
			//				"  ==> " <<
			//				F(5,3,base_pair_array(i,j,k)) << " " << F(5,3,base_pair_array(j,i,m))  << " " <<
			//				//											 F(5,3,energy) << " " << F(5,3,scaled_axis_energy) << " " << F(5,3,scaled_stagger_energy) <<
			//				std::endl;

			std::cout << "BASE PAIR: " << I(3,i) << " " << I(3,j) << " "
								<< get_edge_from_num( k ) << " "
								<< get_edge_from_num( m ) << " "
								<< get_orientation_from_num( base_pair.orientation )
								<< "  ==> "
								<< F(5,3,base_pair_array(i,j,k)) << " " << F(5,3,base_pair_array(j,i,m))  << " "
								<< std::endl;

		}

		if ( user_defined_basepair(i,j,k,m) > 0.1 ) rna_contact_score += -1.0;

		scored_base_pair(i,j) = true;
		scored_base_pair(j,i) = true;

		scored_base_pair_list.push_back( *it );
	}

}

///////////////////////////////////////////////////////////////////////////////
void
eval_rna_base_stack_score( float & rna_stack_score, float & rna_axis_score,
													 FArray2D_float const & base_stack_array,
													 FArray2D_float const & base_stack_axis_array,
													 FArray2D_float const & base_geometry_orientation_array,
													 FArray2D_float const & base_geometry_height_array,
													 FArray2D_bool const & scored_base_pair,
													 Energy_base_stack_list & scored_base_stack_list,
													 int const total_residue )
{
	static bool const rna_verbose = get_rna_verbose();
	static bool const include_neighbor_base_stacks = truefalseoption("include_neighbor_base_stacks");

	scored_base_stack_list.clear();

	for (int i = 1; i <= total_residue; i++ ){
		for (int j = i+1; j <= total_residue; j++ ){

			//			if (exclude_neighbor_base_stacks  &&  j == i+1) continue;

			//Base pairs and base stacks are mutually exclusive!
			if (scored_base_pair(i,j)) continue;

			//Both partners should think they are stacked onto each other.
			if ( base_stack_array( i, j ) < 0.0 &&
					 base_stack_array( j, i ) < 0.0 ){
				if (rna_verbose) std::cout << "BASE STACK: " << i << " " << j << std::endl;

				Base_stack base_stack;
				base_stack.res1 = i;
				base_stack.res2 = j;

				//orientations are cos( theta ) and should be symmetric!
				assert( std::abs( base_geometry_orientation_array( i, j ) - base_geometry_orientation_array( j, i ) ) < 1.0e-2 );
				base_stack.orientation = ( base_geometry_orientation_array( i, j ) + base_geometry_orientation_array( j, i ) ) < 0.0 ? 1 : 2;

				// height is not necessarily (anti)-symmetric if the planes of the two bases aren't co-planar.
				base_stack.which_side = ( base_geometry_height_array( i, j ) > 0.0 ) ? 1 : 2;

				float const total_base_stack_energy = base_stack_array( i, j ) + base_stack_array( j, i);
				scored_base_stack_list.push_back( std::make_pair( total_base_stack_energy,  base_stack ));

				//By default, don't count stacks between neighboring nucleotides, since that
				// interaction is captured by fragments.
				if (!include_neighbor_base_stacks  &&  j == i+1) continue;

				rna_stack_score += base_stack_array( i, j ) + base_stack_array( j, i);
				rna_axis_score  += base_stack_axis_array(i, j) + base_stack_axis_array(j, i);

			}
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
void
set_residue_stacked_array( FArray1D_bool & stacked, Energy_base_stack_list const scored_base_stack_list ){

	stacked = false;

	for ( Energy_base_stack_list::const_iterator it = scored_base_stack_list.begin();
				it != scored_base_stack_list.end(); ++it ){
		Base_stack base_stack = it->second;
		stacked( base_stack.res1 ) = true;
		stacked( base_stack.res2 ) = true;
	}

}

///////////////////////////////////////////////////////////////////////////////
// Empirically, fragment assembly produces too few bulged compared to
// crystal structures. The easiest fix is to give bulges a bonus, presumably
// an entropic advantage that is not properly captured by fragments.
//
void
eval_rna_bulge_score( float & rna_bulge_score,
											Energy_base_stack_list const scored_base_stack_list,
											FArray2D_bool & edge_is_base_pairing,
											int const total_residue )
{
	static bool const rna_verbose = get_rna_verbose();

	FArray1D_bool stacked( total_residue, false );

	set_residue_stacked_array( stacked, scored_base_stack_list );

	rna_bulge_score = 0.0;

	for (int i = 1; i <= total_residue; i++ ) {

		if (stacked(i)) continue;

		//Not stacked?
		//Also check that its not base paired.

		bool base_paired = false;

		for (int k = 1; k <= rna_scoring::NUM_EDGES; k++ ){
			if ( edge_is_base_pairing( i, k ) ) {
				base_paired = true;
				break;
			}
		}

		if (!base_paired) {
			rna_bulge_score += -1.0;
			if (rna_verbose) std::cout << "BULGE: " << i << std::endl;
		}
	}

}

///////////////////////////////////////////////////////////////////////////////
int
convert_acgu_to_1234( int const res_i )
{
	using namespace param_aa;

	assert( is_RNA( res_i ) );

	int res_i_bin = 0;
	if ( res_i == na_rad ) res_i_bin = 1;
	if ( res_i == na_rcy ) res_i_bin = 2;
	if ( res_i == na_rgu ) res_i_bin = 3;
	if ( res_i == na_ura ) res_i_bin = 4;

	assert( res_i_bin > 0);
	return res_i_bin;
}

int get_atom_num_j_bin( int const atom_num_j )
{
	using namespace rna_variables;

	int atom_num_j_bin = 0;
	if ( atom_num_j == o1p )    atom_num_j_bin = 1;
	if ( atom_num_j == o2p )    atom_num_j_bin = 2;
	if ( atom_num_j == o5star ) atom_num_j_bin = 3;
	if ( atom_num_j == o4star ) atom_num_j_bin = 4;
	if ( atom_num_j == o3star ) atom_num_j_bin = 5;
	if ( atom_num_j == o2star ) atom_num_j_bin = 6;
	assert( atom_num_j_bin > 0);

	return atom_num_j_bin;

}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
float
get_rna_nonbasebase_xy( float const x,
												float const y,
												int const res_i,
												int const atom_num_j )
{
	using namespace param_aa;
	using namespace rna_scoring;

	int x_bin, y_bin;

	static int const dist_cutoff = 8; // Magic number!!!

	x_bin = int( (x + dist_cutoff) ) + 1;
	y_bin = int( (y + dist_cutoff) ) + 1;

	if (x_bin < 1 ) x_bin = 1;
	if (y_bin < 1 ) y_bin = 1;
	if (x_bin > 2 * dist_cutoff) x_bin = 2 * dist_cutoff;
	if (y_bin > 2 * dist_cutoff) y_bin = 2 * dist_cutoff;

	int const res_i_bin = convert_acgu_to_1234( res_i );

	int const atom_num_j_bin = get_atom_num_j_bin( atom_num_j );

	assert( res_i_bin > 0);
	assert( atom_num_j_bin > 0);

	return rna_nonbasebase_xy( x_bin, y_bin, res_i_bin, atom_num_j_bin );
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// New insight: interaction between purines and 2'-OH (+other backbone oxygens) are as prevalent as basepairs
// in tertiary folded RNAs, like the ribosome!
void
eval_rna_nonbasebase_score( pose_ns::Pose & pose, float & rna_nonbasebase_score)
{

	//Could use pair-moved to make this faster!

 using namespace param_aa;
  using namespace numeric;
	using namespace kin;
	using namespace rna_scoring;

	rna_nonbasebase_score = 0.0;

	static bool init = {false};
	if (!init){
		initialize_rna_nonbasebase_xy();
		initialize_RNA_backbone_oxygen_atoms();
		init = true;
	}

  static float const dist_cutoff ( 12.0 );

	static float const Z_CUTOFF( 2.0 );
	static float const RHO_CUTOFF( 8.0 );

	int const total_residue = pose.total_residue();
	FArray3D_float const & full_coord( pose.full_coord() );
	static bool const rna_verbose = get_rna_verbose();

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = pose.res(i);
    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );
    Stub stub_i = get_base_coordinate_system(res_i, full_coord(1, 1, i), centroid_i );
    xyzMatrix_float const M_i( stub_i.M );
		//    assert( is_orthonormal( M_i, 1e-3) );
    xyzVector_float const x_i = M_i.col_x();
    xyzVector_float const y_i = M_i.col_y();
    xyzVector_float const z_i = M_i.col_z();

		int const num_heavy_atoms = aaproperties_pack::nheavyatoms( res_i, 1 );
		static int const  rna_base_start( 13 ); // magic number!!

		for ( int j = 1; j <= total_residue; j++ ){

			if ( i == j ) continue;
			if ( i == j+1 ) continue;

      int const res_j = pose.res(j);
      if (!is_RNA( res_j)) continue;

			// Go over sugar and phosphate oxygen atoms!
			for (int m = 1; m <= num_RNA_backbone_oxygen_atoms; m++ ){

				int const atom_num_j = RNA_backbone_oxygen_atom( m );

				xyzVector_float const heavy_atom_j( & full_coord( 1, atom_num_j, j ) );

				xyzVector_float d_ij = heavy_atom_j - centroid_i;

				float const dist_ij = d_ij.length();

				if (dist_ij < dist_cutoff){

					float const dist_x = dot_product( d_ij, x_i );
					float const dist_y = dot_product( d_ij, y_i );
					float const dist_z = dot_product( d_ij, z_i );

					float const rho = std::sqrt( dist_x * dist_x + dist_y * dist_y);

					if ( abs(dist_z) > Z_CUTOFF ) continue; // Look for atoms in the base plane
					if ( rho > RHO_CUTOFF ) continue; // Look for atoms in the base plane

					static float const atom_dist_cutoff = 4.0; //make sure we're in H-bonding distance of some base atom.
					{ //sanity check...
						bool found_a_neighbor = false;
						for (int atom_num_i = rna_base_start; atom_num_i <= num_heavy_atoms; atom_num_i++){
							xyzVector_float base_heavy_atom_i( &full_coord(1, atom_num_i, i) );
							if ( (heavy_atom_j - base_heavy_atom_i ).length() < atom_dist_cutoff){
								found_a_neighbor = true;
								break;
							}
						}
						if (!found_a_neighbor) continue;
					}

					float const score_contribution =
						get_rna_nonbasebase_xy( dist_x, dist_y, res_i, atom_num_j );

					rna_nonbasebase_score += score_contribution;

					if (rna_verbose ){
						std::cout <<
							"BASE NONBASE " <<
							param_aa::aa_name3(res_i) <<
							I(3,i) << " " <<
							aaproperties_pack::atom_name( atom_num_j, pose.res(j), 1 ) <<  " " <<
							I(3, j) << " " <<
							" [" << F(4,2,rho) << ", " << F(4,2,dist_z) << "]:  " <<
							F(6,2,score_contribution) <<
							std::endl;
					}

				}

			}
		}
	}

}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
float
get_rna_o2star_score( float const r,
											int const atom_num_j )
{

	using namespace param_aa;
	using namespace rna_scoring;

	static float const BIN_WIDTH = 0.25 ; // Magic number!!!

	int r_bin = int( r/BIN_WIDTH ) + 1;

	if (r_bin < 1 ) r_bin = 1;
	if (r_bin > 20) r_bin = 20;

	int const atom_num_j_bin = get_atom_num_j_bin( atom_num_j );

	return rna_o2star_potential( r_bin ) * rna_o2star_weight( atom_num_j_bin );
}



///////////////////////////////////////////////////////////////////////////////
void
initialize_rna_o2star_weights(){
	using namespace rna_scoring;

	rna_o2star_weight( 1 ) = 1.0; // O1P
	rna_o2star_weight( 2 ) = 1.0; // O2P
	rna_o2star_weight( 6 ) = 2.0; // O2*
}

///////////////////////////////////////////////////////////////////////////////
void
eval_rna_o2star_score( pose_ns::Pose & pose, float & rna_o2star_score)
{

	//Could use pair-moved to make this faster.

  using namespace param_aa;
  using namespace numeric;
	using namespace rna_scoring;
	using namespace rna_variables;

	static bool init( false );
	if (!init){
		initialize_rna_o2star_score();
		initialize_rna_o2star_weights();
		init = true;
	}

	rna_o2star_score = 0.0;

	int const total_residue = pose.total_residue();
	FArray3D_float const & full_coord( pose.full_coord() );
	static bool const rna_verbose = get_rna_verbose();

  float const dist_cutoff ( 6.0 );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = pose.res(i);
    if (!is_RNA( res_i)) continue;

		int const atom_num_i = o2star;

		xyzVector_float const heavy_atom_i( & full_coord( 1, atom_num_i, i ) );

		for ( int j = 1; j <= total_residue; j++ ){

			if ( abs(i-j) <= 2 ) continue;

			int const res_j = pose.res(j);
			if (!is_RNA( res_j)) continue;

			// Go over sugar and phosphate oxygen atoms!
			for (int m = 1; m <= num_RNA_backbone_oxygen_atoms; m++ ){

				if ( rna_o2star_weight( m ) < 0.001f ) continue;

				int const atom_num_j = RNA_backbone_oxygen_atom( m );

				//Don't double-count 2'-OH <--> 2'-OH interactions
				if (atom_num_j == atom_num_i && j < i) continue;

				xyzVector_float const heavy_atom_j( & full_coord( 1, atom_num_j, j ) );
				xyzVector_float d_ij = heavy_atom_j - heavy_atom_i;
				float const dist_ij = d_ij.length();

				if ( dist_ij > dist_cutoff ) continue;

				float const score_contribution = get_rna_o2star_score( dist_ij, atom_num_j );
				rna_o2star_score += score_contribution;

				if ( rna_verbose && score_contribution < -0.01 ){
					std::cout <<
						"2'-OH " <<
						param_aa::aa_name3(res_i) <<
						I(3,i) << " " <<
						aaproperties_pack::atom_name( atom_num_j, pose.res(j), 1 ) <<  " " <<
						I(3, j) << " " <<
						F(6,2,score_contribution) << " [" << dist_ij << "]" <<
						std::endl;
				}

			} // m

		} // k
	} // i


}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
float
get_rna_phosphate_score( float const r,
												 int const atom_num_j )
{
	using namespace rna_scoring;

	//This might go faster if exponential is precalculated into the rna_o2star_potential table..
	int const atom_num_j_bin = get_atom_num_j_bin( atom_num_j );

	static float const MAX_PENALTY = 8.0; //In kT.
	static float const SCREEN_SCALE = 2.5; //In Angstroms
	static float const DIST_CUTOFF = 8.0; //In Angstroms

	static float offset = MAX_PENALTY * exp( -1.0 * DIST_CUTOFF / SCREEN_SCALE );

	if (r < DIST_CUTOFF ){
		float const potential = MAX_PENALTY * exp( -1.0 * r/ SCREEN_SCALE ) - offset;
		return potential * rna_o2star_weight( atom_num_j_bin );
	} else {
		return 0.0;
	}
}



///////////////////////////////////////////////////////////////////////////////
void
initialize_rna_phosphate_weights(){
	using namespace rna_scoring;

	// Note that unless specified by user "-rna_phosphate_repulse_all"
	//  only O2P-O2P repulsion will be calculated. See eval_rna_phosphate_score().
	rna_phosphate_weight( 1 ) = 1.0; // O1P
	rna_phosphate_weight( 2 ) = 1.0; // O2P
	rna_phosphate_weight( 3 ) = 1.0; // O5*
	rna_phosphate_weight( 4 ) = 1.0; // O4*
	rna_phosphate_weight( 5 ) = 1.0; // O3*
	// O2* --> weight stays at zero.

}


///////////////////////////////////////////////////////////////////////////////
void
eval_rna_phosphate_score( pose_ns::Pose & pose, float & rna_phosphate_score)
{

	//Could use pair-moved to make this faster.

  using namespace param_aa;
  using namespace numeric;
	using namespace rna_scoring;
	using namespace rna_variables;

	static bool init( false );
	if (!init){
		// initialize_rna_phosphate_score(); // would make a table of the exponential
		initialize_rna_phosphate_weights();
		init = true;
	}

	rna_phosphate_score = 0.0;

	int const total_residue = pose.total_residue();
	FArray3D_float const & full_coord( pose.full_coord() );
	static bool const rna_verbose = get_rna_verbose();

  float const dist_cutoff ( 12.0 );

	static bool const rna_phosphate_repulse_all = truefalseoption( "rna_phosphate_repulse_all" );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = pose.res(i);
    if (!is_RNA( res_i)) continue;

		int const atom_num_i = o2p;

		xyzVector_float const heavy_atom_i( & full_coord( 1, atom_num_i, i ) );

		for ( int j = 1; j <= total_residue; j++ ){

			if ( abs(i-j) <= 2 ) continue;

			int const res_j = pose.res(j);
			if (!is_RNA( res_j)) continue;

			// Go over sugar and phosphate oxygen atoms!
			for (int m = 1; m <= num_RNA_backbone_oxygen_atoms; m++ ){

				if ( rna_phosphate_weight( m ) < 0.001f ) continue;

				int const atom_num_j = RNA_backbone_oxygen_atom( m );

				//By default only repel o2p.
				if (!rna_phosphate_repulse_all && atom_num_j != o2p ) continue;

				//Don't double-count O2P <--> O2P interactions
				if (atom_num_j == atom_num_i && j < i) continue;

				xyzVector_float const heavy_atom_j( & full_coord( 1, atom_num_j, j ) );
				xyzVector_float d_ij = heavy_atom_j - heavy_atom_i;
				float const dist_ij = d_ij.length();

				if ( dist_ij > dist_cutoff ) continue;

				float const score_contribution = get_rna_phosphate_score( dist_ij, atom_num_j );
				rna_phosphate_score += score_contribution;

				if ( rna_verbose && score_contribution > 0.1 ){
					std::cout <<
						"PHOSPHATE " <<
						param_aa::aa_name3(res_i) <<
						I(3,i) << " " <<
						aaproperties_pack::atom_name( atom_num_j, pose.res(j), 1 ) <<  " " <<
						I(3, j) << " " <<
						F(6,2,score_contribution) << " [" << dist_ij << "]" <<
						std::endl;
				}

			} // m

		} // k
	} // i


}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Try generic base-pairing and base-stacking rewards.
///////////////////////////////////////////////////////////////////////////////
void
eval_rna_score(
	pose_ns::Pose & pose,
	float & rna_bs_score,
	float & rna_bp_w_score,
	float & rna_bp_h_score,
	float & rna_bp_s_score,
	float & rna_axis_score,
	float & rna_stagger_score,
	float & rna_bulge_score,
	float & rna_contact_score,
	float & rna_long_range_contact_score,
	FArray2D_bool & scored_base_pair,
	FArray2D_bool & edge_is_base_pairing,
	Energy_base_pair_list  & scored_base_pair_list,
	Energy_base_stack_list & scored_base_stack_list
)
{
	using namespace rna_scoring;

	rna_bs_score = 0.0;
	rna_bp_w_score = 0.0;
	rna_bp_h_score = 0.0;
	rna_bp_s_score = 0.0;
	rna_axis_score    = 0.0;
	rna_stagger_score = 0.0;
	rna_contact_score = 0.0;
	rna_long_range_contact_score = 0.0;
	rna_bulge_score = 0.0;

	//Contact scores.
	read_user_defined_contact_files(); //returns if already initialized.
	eval_rna_long_range_contact_score( rna_long_range_contact_score, rna_contact_score, pose );

	int const total_residue = pose.total_residue();

	//Read in arrays of pre-computed values.
	Score_state rna_base_pair_array_state, rna_base_axis_array_state, rna_base_stagger_array_state;
	Score_state rna_base_stack_array_state, rna_base_stack_axis_array_state;
	FArray3D_float & base_pair_array    ( pose.set_3D_score( RNA_BASE_PAIR_ARRAY, rna_base_pair_array_state,
																													 total_residue, total_residue, NUM_EDGES ) );
	FArray3D_float & base_axis_array    ( pose.set_3D_score( RNA_BASE_AXIS_ARRAY, rna_base_axis_array_state,
																													 total_residue, total_residue, NUM_EDGES ) );
	FArray3D_float & base_stagger_array ( pose.set_3D_score( RNA_BASE_STAGGER_ARRAY, rna_base_stagger_array_state,
																													 total_residue, total_residue, NUM_EDGES ) );
	FArray2D_float & base_stack_array      ( pose.set_2D_score( RNA_BASE_STACK_ARRAY, rna_base_stack_array_state ) );
	FArray2D_float & base_stack_axis_array ( pose.set_2D_score( RNA_BASE_STACK_AXIS_ARRAY, rna_base_stack_axis_array_state ) );

	//Added in later. Could also keep x and y to get a full geometric description!
	FArray2D_float & base_geometry_orientation_array ( pose.set_2D_score( RNA_BASE_GEOMETRY_ORIENTATION_ARRAY, rna_base_pair_array_state ) );
	FArray2D_float & base_geometry_height_array      ( pose.set_2D_score( RNA_BASE_GEOMETRY_HEIGHT_ARRAY, rna_base_pair_array_state ) );

	if (rna_base_pair_array_state != GOOD ||
			rna_base_axis_array_state != GOOD ||
			rna_base_stagger_array_state != GOOD ||
			rna_base_stack_array_state != GOOD ||
			rna_base_stack_axis_array_state != GOOD
			) {
		bool const rna_array_state_ok = (rna_base_pair_array_state == OK);
		update_base_pair_array( pose, base_pair_array, base_axis_array, base_stagger_array,
														base_stack_array, base_stack_axis_array,
														base_geometry_orientation_array, base_geometry_height_array,
														rna_array_state_ok);
	}

	//Following function is a little tricky --
	// A base can only make three interactions, one from each base edge.
	// The routine picks the optimal interaction to include in the rna_bp_score.
	edge_is_base_pairing = false;
	scored_base_pair_list.clear();
	eval_rna_base_pair_score( rna_bp_w_score, rna_bp_h_score, rna_bp_s_score,
														rna_axis_score, rna_stagger_score,
														rna_contact_score,
														base_pair_array,
														base_axis_array,
														base_stagger_array,
														base_geometry_orientation_array,
														scored_base_pair,
														edge_is_base_pairing,
													  scored_base_pair_list,
														total_residue);

	// Note that having two bases with pairing between any two given edges should be
	// mutually exclusive with stacking.
	scored_base_stack_list.clear();
	eval_rna_base_stack_score( rna_bs_score, rna_axis_score,
														 base_stack_array, base_stack_axis_array,
														 base_geometry_orientation_array, base_geometry_height_array,
														 scored_base_pair,
														 scored_base_stack_list,
														 total_residue);

	// Didn't really work -- bonus to bulges, since the should have more conformational entropy.
	eval_rna_bulge_score( rna_bulge_score, scored_base_stack_list, edge_is_base_pairing, total_residue );

}


/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
eval_rna_score(
	pose_ns::Pose & pose,
	float & rna_bs_score,
	float & rna_bp_w_score,
	float & rna_bp_h_score,
	float & rna_bp_s_score,
	float & rna_axis_score,
	float & rna_stagger_score,
	float & rna_bulge_score,
	float & rna_contact_score,
	float & rna_long_range_contact_score
)
{
	using namespace rna_scoring;
	int const total_residue = pose.total_residue();
	FArray2D_bool scored_base_pair( total_residue, total_residue );
	FArray2D_bool edge_is_base_pairing( total_residue, NUM_EDGES, false );

	Energy_base_pair_list  scored_base_pair_list;
	Energy_base_stack_list scored_base_stack_list;

	eval_rna_score( pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list, scored_base_stack_list);
}

void
eval_rna_score(
	pose_ns::Pose & pose,
	float & rna_bs_score,
	float & rna_bp_w_score,
	float & rna_bp_h_score,
	float & rna_bp_s_score,
	float & rna_axis_score,
	float & rna_stagger_score,
	float & rna_bulge_score,
	float & rna_contact_score,
	float & rna_long_range_contact_score,
	FArray2D_bool & scored_base_pair,
	FArray2D_bool & edge_is_base_pairing
 ) {

	Energy_base_pair_list  scored_base_pair_list;
	Energy_base_stack_list scored_base_stack_list;
	eval_rna_score( pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list, scored_base_stack_list);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
get_scored_base_pair_array(
	pose_ns::Pose & pose,
	FArray2D_bool & scored_base_pair
)
{
	pose.new_score_pose(); //Just in case.
	float rna_bs_score,	rna_bp_w_score,	rna_bp_h_score,	rna_bp_s_score,	rna_axis_score,	rna_stagger_score,	rna_contact_score,	rna_long_range_contact_score, rna_bulge_score;

	FArray2D_bool edge_is_base_pairing( pose.total_residue(), rna_scoring::NUM_EDGES, false );
	Energy_base_pair_list  scored_base_pair_list;
	Energy_base_stack_list scored_base_stack_list;

	eval_rna_score( pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list, scored_base_stack_list);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////
bool check_chainbreak(int i, pose_ns::Pose & pose) {
	using namespace param_aa;
	using namespace numeric;

	if (i==pose.total_residue()) return false;

  float const CHAINBREAK_CUTOFF2 = 7.5*7.5;

	static const int atomindex = LookupByName( pose.res(i), 1, " C4*");

	FArray3D_float const & xyz_full = pose.full_coord();

	xyzVector_float vec( &xyz_full(1,atomindex,i+1) );
	xyzVector_float vec_prev( &xyz_full(1,atomindex, i) );
    const float dist2 = (vec-vec_prev).length_squared();
    if ( dist2 > CHAINBREAK_CUTOFF2) {
      return true;
    }

    return false;
  }

/////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// The following copies code from fill_native_base_pair_info() --
// probably shouldn't do that.
//
// Maybe the right thing would be to create an RNA_features class (which could
// be filled by pose_rna_featurizer.cc), and a single function that returns
// unshared features.
//
int
num_unshared_basepairs(
		pose_ns::Pose & pose1 /* reference pose*/,
		pose_ns::Pose & pose2 /*scored pose*/
)
{

	int const total_residue = pose1.total_residue();
	Energy_base_pair_list  scored_base_pair_list1, scored_base_pair_list2;
	Energy_base_stack_list scored_base_stack_list1, scored_base_stack_list2;

	////////////////////////////////////////////////////////////
	// Use scoring to fill lists of base pairs and base stacks
	////////////////////////////////////////////////////////////
	//Dummy variables...
	FArray2D_bool scored_base_pair( total_residue, total_residue );
	float rna_bs_score,	rna_bp_w_score,	rna_bp_h_score,	rna_bp_s_score,	rna_axis_score,	rna_stagger_score;
	float rna_contact_score,	rna_long_range_contact_score, rna_bulge_score;
	FArray2D_bool edge_is_base_pairing( total_residue, rna_scoring::NUM_EDGES, false );

	eval_rna_score( pose1, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list1, scored_base_stack_list1);

	eval_rna_score( pose2, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list2, scored_base_stack_list2);

	int count( 0 );

	for ( Energy_base_pair_list::const_iterator it1 = scored_base_pair_list1.begin();
				it1 != scored_base_pair_list1.end(); ++it1 ){

		Base_pair const base_pair1 = it1->second;

		bool found_match( false );

		for ( Energy_base_pair_list::const_iterator it2 = scored_base_pair_list2.begin();
					it2 != scored_base_pair_list2.end(); ++it2 ){

			Base_pair const base_pair2 = it2->second;

			if ( base_pair1 == base_pair2 ) {
				found_match = true;
				break;
			}

		}

		if (!found_match) {
			count++;
			if (get_rna_verbose()) std::cout << "Missing base pair: " << base_pair1 << std::endl;
		}

	}

	return count;

}


/////////////////////////////////////////////////////////////////////////////////////////////////////////
bool
is_anti( int const i, pose_ns::Pose & pose)
{
	static int const CHI_NUM = 7;
	float chi;

	using namespace param_aa;
	//Stupid, chi angle definition different for purine  and
	// pyrimidine. There was a (bad) reason for this in
	// the atom tree setup.
	if (pose.res(i) == na_rad || pose.res(i) == na_rgu ){
		static const float good_chi = 80.0;
		chi =  pose.get_torsion_by_number( i, CHI_NUM );
		return ( abs(chi -  good_chi ) < 40.0 );
	} else {
		static const float good_chi = -90.0;
		chi =  pose.get_torsion_by_number( i, CHI_NUM );
		return ( abs(chi -  good_chi ) < 40.0 );
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
bool count_as_base_pair( int const i,
												 int const j,
												 int const k,
												 int const m,
												 FArray3D_float const &  decoy_base_pair_array)
{

	bool base_pair_present = false;
	if (decoy_base_pair_array(i, j, k) < 0.0) {
		//The view from one side looks good!
		base_pair_present = true;

		for (int f = 1; f <= rna_scoring::NUM_EDGES; f++){
			// Make sure the other side doesn't disagree about
			// which edge is making the interaction.
			if (decoy_base_pair_array(j, i, m) -
					decoy_base_pair_array(j, i, f) > 0.5 ) {
				base_pair_present = false;
				break;
			}
		}
	}

	if (!base_pair_present) {
		//Try it from the other base's point of view.
		if (decoy_base_pair_array(j, i, m) < 0.0) {
			base_pair_present = true;

			for (int f = 1; f <= rna_scoring::NUM_EDGES; f++){
				// Make sure the other side doesn't disagree about
				// which edge is making the interaction.
				if (decoy_base_pair_array(i, j, k) -
						decoy_base_pair_array(i, j, f) > 0.5 ) {
					base_pair_present = false;
					break;
				}
			}
		}
	}

	//	std::cout << "TESTING: " << i << " " << j << " " << k << " " << m << " " << base_pair_present << std::endl;
	return base_pair_present;

}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
// Basic featurizing for diagnostics.
void
fill_native_base_pair_info(
	pose_ns::Pose & pose
)
{

	if (!pose.native_pose_exists()) return;

	Pose native_pose;
	//This actually shold make a copy, which we need for scoring, etc.
	native_pose = pose.native_pose();

	int const total_residue = pose.total_residue();
	Energy_base_pair_list  native_scored_base_pair_list, decoy_scored_base_pair_list;
	Energy_base_stack_list native_scored_base_stack_list, decoy_scored_base_stack_list;

	////////////////////////////////////////////////////////////
	// Use scoring to fill lists of base pairs and base stacks
	////////////////////////////////////////////////////////////
	//Dummy variables...
	FArray2D_bool scored_base_pair( total_residue, total_residue );
	float rna_bs_score,	rna_bp_w_score,	rna_bp_h_score,	rna_bp_s_score,	rna_axis_score,	rna_stagger_score;
	float rna_contact_score,	rna_long_range_contact_score, rna_bulge_score;
	FArray2D_bool edge_is_base_pairing( pose.total_residue(), rna_scoring::NUM_EDGES, false );

	eval_rna_score( pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									decoy_scored_base_pair_list, decoy_scored_base_stack_list);

	FArray3D_float const & decoy_base_pair_array( pose.get_3D_score( RNA_BASE_PAIR_ARRAY ) );

	eval_rna_score( native_pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score,
									rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									native_scored_base_pair_list, native_scored_base_stack_list);

	//////////////////////////////////////////////////
	// What native base pairs are present in decoy?
	//////////////////////////////////////////////////

	//This is inefficient but I only need to do it once per run, diagnostics...
	int num_native_WC_base_pairs( 0 ), num_native_NWC_base_pairs( 0 ), num_native_bulges( 0 );
	int num_decoy_WC_base_pairs( 0 ), num_decoy_NWC_base_pairs( 0 ), num_decoy_bulges( 0 );
	float frac_WC_base_pairs( 1.0 ), frac_NWC_base_pairs( 1.0 ), frac_bulges( 1.0 );
	bool is_WC( false );

	FArray1D_bool residue_makes_WC_basepair( total_residue, false );
	FArray1D_bool residue_makes_NWC_basepair( total_residue, false );

	using namespace param_aa;
	for ( Energy_base_pair_list::const_iterator it_N = native_scored_base_pair_list.begin();
				it_N != native_scored_base_pair_list.end(); ++it_N ){

		//Unpack it. This should probably just be a class.
		Base_pair const base_pair_N = it_N->second;

		int const i_N = base_pair_N.res1;
		int const k_N = base_pair_N.edge1;

		int const j_N = base_pair_N.res2;
		int const m_N = base_pair_N.edge2;

		int const res1 = pose.res( i_N );
		int const res2 = pose.res( j_N );

		if (k_N == 1 && m_N == 1  //Both edges are Watson-Crick.
				&& is_canonical( res1, res2 )
				&& is_anti( i_N, native_pose) && is_anti( j_N, native_pose) ){
			is_WC = true;
			num_native_WC_base_pairs++;
			residue_makes_WC_basepair( i_N ) = true;
			residue_makes_WC_basepair( j_N ) = true;
		}	else {
			is_WC = false;
			num_native_NWC_base_pairs++;
			residue_makes_NWC_basepair( i_N ) = true;
			residue_makes_NWC_basepair( j_N ) = true;
		}


		if ( count_as_base_pair( i_N, j_N, k_N, m_N, decoy_base_pair_array) ) {
			if (is_WC) {
				num_decoy_WC_base_pairs++;
			} else {
				num_decoy_NWC_base_pairs++;
			}
		}

// 		for ( Energy_base_pair_list::const_iterator it_D = decoy_full_base_pair_list.begin();
// 					it_D != decoy_full_base_pair_list.end(); ++it_D ){
// 			Base_pair const base_pair_D = it_D->second;
// 			Index_edge_pair const index_edge_pair1_D = base_pair_D.first;
// 			Index_edge_pair const index_edge_pair2_D = base_pair_D.second;

// 			if ((index_edge_pair1_D == index_edge_pair1_N && index_edge_pair2_D == index_edge_pair2_N) ||
// 					(index_edge_pair1_D == index_edge_pair2_N && index_edge_pair2_D == index_edge_pair1_N)) {
// 				if (is_WC) {
// 					num_decoy_WC_base_pairs++;
// 				} else {
// 					num_decoy_NWC_base_pairs++;
// 				}

// 			}
// 		}



	}


	//////////////////////////////////////////////////
	// What native bulges are present in decoy?
	//////////////////////////////////////////////////
	FArray1D_bool native_stacked( total_residue, false );
	FArray1D_bool decoy_stacked ( total_residue, false );


	set_residue_stacked_array( native_stacked, native_scored_base_stack_list );
	set_residue_stacked_array( decoy_stacked,  decoy_scored_base_stack_list );

	//Ignore termini and chainbreaks; they're always a little wacky.
	for (int i = 2; i <= total_residue-1 ; i++){
		if (!native_stacked(i) && !check_chainbreak(i-1, pose) && !check_chainbreak(i, pose)) {
			num_native_bulges++;
			if (!decoy_stacked(i)) {
				num_decoy_bulges++;
			}
		}
	}


	pose.set_extra_score( "N_WC" , num_decoy_WC_base_pairs) ;
	pose.set_extra_score( "N_NWC", num_decoy_NWC_base_pairs) ;
	pose.set_extra_score( "N_BUL", num_decoy_bulges) ;

	if (num_native_WC_base_pairs > 0) {
		frac_WC_base_pairs =  num_decoy_WC_base_pairs / (1.0 * num_native_WC_base_pairs);
	}

	if (num_native_NWC_base_pairs > 0) {
		frac_NWC_base_pairs =  num_decoy_NWC_base_pairs / (1.0 * num_native_NWC_base_pairs);
	}

	if (num_native_bulges > 0) {
		frac_bulges =  num_decoy_bulges / (1.0 * num_native_bulges);
	}

	//Do we really need this scoreline?
	//	pose.set_extra_score( "FRAC_WC" , frac_WC_base_pairs );
	//	pose.set_extra_score( "FRAC_NWC", frac_NWC_base_pairs );
	//	pose.set_extra_score( "FRAC_BUL", frac_bulges );


	//Calculate RMSD over Watson-Crick and non-Watson-Crick regions,
	// as has been requested by reviewers of paper.
	static const bool subset_rmsd = truefalseoption( "subset_rmsd" );
	if (subset_rmsd) {
		FArray1D_bool residue_tertiary( total_residue, false );
		int res_WC( 0 ), res_NWC( 0 ), res_TERT( 0 );
		for (int i = 1; i <= total_residue; i++) {
			residue_tertiary(i) = residue_makes_NWC_basepair(i) || (!residue_makes_WC_basepair(i));

			if (residue_makes_WC_basepair(i)) res_WC++;
			if (residue_makes_NWC_basepair(i)) res_NWC++;
			if (residue_tertiary(i)) res_TERT++;

		}

		float const rmsd_WC   = CA_rmsd_by_subset( pose, native_pose, residue_makes_WC_basepair );
		float const rmsd_NWC  = CA_rmsd_by_subset( pose, native_pose, residue_makes_NWC_basepair );
		float const rmsd_TERT = CA_rmsd_by_subset( pose, native_pose, residue_tertiary );
		pose.set_extra_score("nres_WC",  res_WC);
		pose.set_extra_score("nres_NWC", res_NWC);
		pose.set_extra_score("nres_TERT", res_TERT);
		pose.set_extra_score("rmsd_WC",  rmsd_WC);
		pose.set_extra_score("rmsd_NWC",  rmsd_NWC);
		pose.set_extra_score("rmsd_TERT",  rmsd_TERT);

		//Output assignments of whether residue is making WC base pair or not
		static bool init ( false );
		if (!init){
			std::ofstream outstream( "residue_WC_NWC.txt" );
			for (int i = 1; i <= total_residue; i++){
				outstream << i << " " << residue_makes_WC_basepair(i) << " " << residue_makes_NWC_basepair(i) << std::endl;
			}
			outstream.close();
			init = true;
		}
	}


	int const miss_nbp = num_unshared_basepairs( native_pose, pose );
	pose.set_extra_score( "MISS_NBP", miss_nbp );


}


/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
read_rna_contact_file(){
	using namespace rna_scoring_user_defined;

	std::string file_flag = "rna_contact_file";
 	if (!truefalseoption( file_flag )) return;

	std::string filename = stringafteroption(file_flag, "blah.contacts");

	// open file
  utility::io::izstream data_stream( filename );

	if ( !data_stream ) {
		std::cerr << "Can't find specified contact file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading contact file: " << filename << std::endl;;
	// read data
	int res1, res2, edge1, edge2;
	int count( 0 );
	float basepair_cutoff = 8.0;
	std::string line;
	while ( getline( data_stream, line)  ) {
		std::istringstream line_stream ( line );
		line_stream >> res1;
		float temp;
		line_stream >> temp ;
		res2 = static_cast<int>( temp + 0.5 );

		int tempint;
		line_stream >> tempint ;
		if (!line_stream.fail()) {
			edge1 = tempint;
			line_stream >> edge2;
			user_defined_basepair( res1, res2, edge1, edge2 ) = basepair_cutoff;
			user_defined_basepair( res2, res1, edge2, edge1 ) = basepair_cutoff;
			std::cout << " Give bonus to contact between " << res1 << " and " << res2 << std::endl;
		} else {
			for (edge1 = 1; edge1 <= rna_scoring::NUM_EDGES; edge1++){
				for (edge2 = 1; edge2 <= rna_scoring::NUM_EDGES; edge2++){
					user_defined_basepair( res1, res2, edge1, edge2 ) = basepair_cutoff;
					user_defined_basepair( res2, res1, edge2, edge1 ) = basepair_cutoff;
				}
			}
		}
		count++;
	}
	std::cout << "Read in: " << count << " possible base pair bonuses." << std::endl;

	//close file
	data_stream.close();
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
read_rna_long_range_contact_file(){
	using namespace rna_scoring_user_defined;

	std::string file_flag = "rna_long_range_contact_file";
 	if (!truefalseoption( file_flag )) return;

	std::string filename = stringafteroption(file_flag, "blah.contacts");

	// open file
  utility::io::izstream data_stream( filename );

	if ( !data_stream ) {
		std::cerr << "Can't find specified contact file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading contact file: " << filename << std::endl;;
	// read data
	int res1, res2;
	int count( 0 );
	std::string line;
	while ( getline( data_stream, line)  ) {
		float basepair_cutoff = 8.0;
		std::istringstream line_stream ( line );
		line_stream >> res1;
		float temp;
		line_stream >> temp ;
		res2 = static_cast<int>( temp + 0.5 );
		line_stream >> temp;
		if (!line_stream.fail()) {
			basepair_cutoff = temp;
		}

		user_defined_long_range_contact( res1, res2 ) = basepair_cutoff;
		user_defined_long_range_contact( res2, res1 ) = basepair_cutoff;

		long_range_contact_residues.push_back( std::make_pair( res1, res2) );

		count++;
	}
	std::cout << "Read in: " << count << " possible long range contact bonuses." << std::endl;

	//close file
	data_stream.close();
}

///////////////////////////////////////////////////////////////////////////////

void
read_user_defined_contact_files(){
	static bool init = false;
	if (init) return;

	read_rna_contact_file();
	read_rna_long_range_contact_file();

	init = true;
}

///////////////////////////////////////////////////////////////////////////////
float
get_rna_basepair_xy( float const x, float const y, float const theta,
										 int const res_i, int const res_j)
{
	using namespace param_aa;
	using namespace rna_scoring;

	static bool init = {false};
	int x_bin, y_bin, theta_bin;

	if (!init){
		initialize_rna_basepair_xy();
		init = true;
	}

	x_bin = int( (x + 10.0) / 2.0 ) + 1;
	y_bin = int( (y + 10.0) / 2.0 ) + 1;

	if (x_bin < 1 ) x_bin = 1;
	if (y_bin < 1 ) y_bin = 1;
	if (x_bin > 10) x_bin = 10;
	if (y_bin > 10) y_bin = 10;

	theta_bin = (theta > 90.0) ?  1 : 2;

	assert( is_RNA(res_i));
	assert( is_RNA(res_j));

	int const res_i_bin = convert_acgu_to_1234( res_i );
	int const res_j_bin = convert_acgu_to_1234( res_j );

	assert( res_i_bin > 0);
	assert( res_j_bin > 0);

	return rna_basepair_xy( x_bin, y_bin, theta_bin, res_i_bin, res_j_bin );
}


///////////////////////////////////////////////////////////////////////////////
float
get_rna_basepair_rho_omega1_omega2( float const rho, float const omega1, float const omega2,
																		float const theta, int const res_i, int const res_j)
{
	using namespace rna_scoring;

	static bool init = {false};
	int r_bin, omega1_bin, omega2_bin;

	if (!init){
		initialize_rna_basepair_rho_omega1_omega2();
		init = true;
	}

	r_bin = int( rho ) + 1;
	if (r_bin > 8) r_bin = 8;

	omega1_bin = int( mod( omega1 + 180.0f, 360.0f) / 10.0 ) + 1;
	if (omega1_bin < 1)  omega1_bin = 1;
	if (omega1_bin > 36) omega1_bin = 36;

	omega2_bin = int( mod( omega2 + 180.0f, 360.0f) / 10.0 ) + 1;
	if (omega2_bin < 1)  omega2_bin = 1;
	if (omega2_bin > 36) omega2_bin = 36;

	int const theta_bin = (theta > 90.0) ?  1 : 2;
	int const res_i_bin = convert_acgu_to_1234( res_i );
	int const res_j_bin = convert_acgu_to_1234( res_j );

	rnaBasepairKey which_potential = std::make_pair( theta_bin, std::make_pair( res_i_bin, res_j_bin ) );

	return rna_basepair_rho_omega1_omega2[ which_potential ]( r_bin, omega1_bin, omega2_bin );
}


///////////////////////////////////////////////////////////////////////////////
float
get_rna_axis_score( float const theta ){
	using namespace rna_scoring;
	using numeric::conversions::radians;

	int cos_theta_bin( 1 );
	cos_theta_bin = static_cast<int> (5 *  (std::cos(radians(theta)) + 1.0f) ) + 1;
	//		cos_theta_bin = static_cast<int> (10 *  (std::cos(radians(theta)) + 1.0f) ) + 1;

	return rna_axis( cos_theta_bin );
}

///////////////////////////////////////////////////////////////////////////////
float
get_rna_stagger_score( float const height ){
	using namespace rna_scoring;

	int height_bin = static_cast<int> ( 2.5 * (height + 2.0) ) + 1;
	if ( height_bin < 1 || height_bin > 11 ) height_bin = 1;

	return rna_stagger( height_bin );
}

///////////////////////////////////////////////////////////////////////////////
float
eval_rna_other_score(
	pose_ns::Pose & pose
	)
{
	float rna_score( 0.0 );

	int stupid_place_holder_no_warning;
	stupid_place_holder_no_warning = pose.total_residue();

	// I can probably get rid of this function now. Was testing
	// new energy terms in here.

	return rna_score;
}

///////////////////////////////////////////////////////////////////////////////
void
read_user_defined_sasa( int const nres ) //returns if already read in.
{
	using namespace rna_scoring_user_defined;

	static bool init = false;
	if (init) return;

	init = true;
	user_defined_sasa_flag = truefalseoption("rna_sasa_file");
	if (!user_defined_sasa_flag) return;

	std::string filename = stringafteroption("rna_sasa_file","blah.sasa");
	utility::io::izstream sasa_stream( filename );
	if ( !sasa_stream ) {
		std::cerr << "Can't find specified SASA file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}
	std::cout << "Reading SASA file: " << filename << std::endl;

	//read data
	float res_float;
	int res, count( 0 );
	float sasa;
	while ( sasa_stream >> res_float ) {
		res = static_cast<int>( res_float + 0.5 );
		sasa_stream >> sasa >> skip;
		user_defined_sasa( res ) = sasa;
		if (res > nres || res < 1){
			std::cerr << "Invalid residue in SASA file: " << filename << " " << res << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			return;
		}
		count++;
	}
	std::cout << "Read in: " << count << " SASA data points." << std::endl;

}

///////////////////////////////////////////////////////////////////////////////
// void get_approximate_ring_sasa( pose_ns::Pose & pose, FArray1D_float & user_defined_sasa,
// 																FArray1D_float & decoy_sasa, float const neighbor_radius){
// 	using namespace aaproperties_pack;
// 	using namespace param_aa;
// 	using namespace pose_ns;
// 	using namespace numeric;
// 	using namespace rna_variables;

// 	pose_update_cendist( pose ); //may not be necessary if computing VDW anyway.
// 	FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );
// 	float const CENDIST2_CUTOFF = 12.0*12.0;

// 	int const nres = pose.total_residue();
// 	float const neighbor_radius2 = neighbor_radius * neighbor_radius;
// 	FArray3D_float const full_coord( pose.full_coord() );

// 	for (int i=1; i<=nres; i++) {
// 		if (user_defined_sasa(i) == 0.0f)  continue;
// 		int const res_i = pose.res( i );
//     if (!is_RNA( res_i)) continue;

// 		int count( 0 );

// 		for (int j=1; j<= nres; j++ ){
// 			int const res_j = pose.res( j );
// 			if (!is_RNA( res_j)) continue;
// 			if ( cendist(i,j) >= CENDIST2_CUTOFF ) continue;

// 			//loop over ring atoms.
// 			int const NUM_RING_ATOMS( 5 );
// 			FArray1D_int ring_atoms( NUM_RING_ATOMS );
// 			ring_atoms(1) = c1star;
// 			ring_atoms(2) = c2star;
// 			ring_atoms(3) = c3star;
// 			ring_atoms(4) = c4star;
// 			ring_atoms(5) = c5star;

// 			for (int m=1; m <= NUM_RING_ATOMS; m++){
// 				int const atom_i = ring_atoms(m);
// 				xyzVector_float const r_i( & full_coord(1,atom_i,i) );

// 				for (int atom_j=1; atom_j <= nheavyatoms( res_j, 1); atom_j++){
// 					xyzVector_float const r_j( & full_coord(1,atom_j,j) );
// 					if ( (r_i-r_j).length_squared() < neighbor_radius2 ) count++;
// 				} // atom_j

// 			} // m, atom_i
// 		} // j


// 		//NEED TO ACTUALLY TAKE  A - B * count, but I didn't really get that far.
// 		// Appear to lose some information in going to number of nearest neighbors...
// 		decoy_sasa(i) = count;
// 	} // i

// }

void get_ring_sasa( pose_ns::Pose & pose, FArray1D_float & user_defined_sasa,
										FArray1D_float & decoy_sasa, bool const fast_sasa){
	using namespace rna_variables;
	using namespace param;

	pose_update_cendist( pose );
	pose.copy_to_misc();

	//Use Rosetta's built in SASA calculator.
	FArray2D_float atom_sasa( MAX_ATOM(), MAX_RES());
	FArray1D_float rsd_sasa( MAX_RES());
	float const probe_radius = 1.400;
	//	calc_aprox_sasa( atom_sasa );
	calc_per_atom_sasa( atom_sasa, rsd_sasa, probe_radius,
											true /*ignore hydrogens*/, false /*self_chain_only*/,
											fast_sasa /*ring_rna*/);

	//Calc C1* + C2* + C3* + C4* + C5* SASA for residues where SASA has been input...
	int const nres = pose.total_residue();
	for (int i=1; i<=nres; i++) {
		if (user_defined_sasa(i) == 0.0f)  continue;
		decoy_sasa(i) =
			atom_sasa( c1star,i) +
			atom_sasa( c2star,i) +
			atom_sasa( c3star,i) +
			atom_sasa( c4star,i) +
			atom_sasa( c5star,i);
	}
}

///////////////////////////////////////////////////////////////////////////////
void
eval_rna_sasa_score(
	pose_ns::Pose & pose,
	float & rna_sasa_score,
	bool verbose, /* false */
	bool fast_sasa /*true*/
	){
	using namespace rna_scoring_user_defined;

	//Initialize
	rna_sasa_score = 0.0;
	int const nres = pose.total_residue();
	read_user_defined_sasa( nres ); //returns if already read in.
	if (!user_defined_sasa_flag) return;

	//SASA calculation of C1* + C2* + C3* + C4* + C5*.
	FArray1D_float decoy_sasa( nres, 1.0);
	get_ring_sasa( pose, user_defined_sasa, decoy_sasa, fast_sasa);//, neighbor_radius);

	//Normalize
	float total_ring_sasa = 0.0;
	int total_data_points = 0;
	for (int i=1; i<= nres; i++){
		if (user_defined_sasa(i) == 0.0f) continue;
		total_ring_sasa += decoy_sasa(i);
		total_data_points++;
	}
	for (int i=1; i<=nres; i++) decoy_sasa(i) *= (total_data_points/total_ring_sasa);

	for (int i=1; i<=nres; i++) {
		if (verbose){
			std::string tag = fast_sasa ? "APPROX_RNA_SASA":" ACTUAL_RNA_SASA";
			std::cout << tag << " "  << i << " " << decoy_sasa(i) << " " << user_defined_sasa(i) << std::endl;
		}
		if (user_defined_sasa(i) == 0.0f) continue;
		float const deviation =  (decoy_sasa(i) - user_defined_sasa(i));
		rna_sasa_score += deviation * deviation;
	}

	if (verbose)
		std::cout << " RNA_SASA_SCORE " << rna_sasa_score << std::endl;

}


///////////////////////////////////////////////////////////////////////////////
numeric::xyzVector_float
get_base_centroid( int const res_type, FArray3Da_float full_coord )
{
  using namespace numeric;
  using namespace param_aa;
  using namespace aaproperties_pack;

  assert( is_RNA(res_type) );

  int const num_heavy_atoms = nheavyatoms( res_type, 1);
  int const rna_base_start = 13;

  full_coord.dimension(3, num_heavy_atoms, 1);

  xyzVector_float centroid( 0.0 );
  int numatoms = 0;

  for (int i = rna_base_start; i <= num_heavy_atoms; i++){
    numatoms++;
    xyzVector_float coord_heavy( &full_coord(1, i, 1) );
    centroid += coord_heavy;
  }
  assert(numatoms > 0);

  centroid /= static_cast<float>( numatoms );

  return centroid;
}

///////////////////////////////////////////////////////////////////////////////
kin::Stub
get_base_coordinate_system(
   int const res_type,
   FArray3Da_float full_coord,
   numeric::xyzVector_float const centroid
   )
{
  using namespace param_aa;
  using namespace numeric;
  using namespace aaproperties_pack;

  xyzVector_float x, y, z;

  assert( is_RNA(res_type) );

  int const num_heavy_atoms = nheavyatoms( res_type, 1);
  full_coord.dimension(3, num_heavy_atoms, 1);

  // Make an axis pointing from base centroid to Watson-Crick edge.
  int WC_atom = 0;
  if ( res_type == na_rad ) WC_atom = 13; //N1
  if ( res_type == na_rcy ) WC_atom = 16; //N3
  if ( res_type == na_rgu ) WC_atom = 13; //N1
  if ( res_type == na_ura ) WC_atom = 16; //N3
  assert( WC_atom > 0);

  xyzVector_float WC_coord( & full_coord( 1, WC_atom, 1) );
  x = WC_coord - centroid;
  x.normalize();

  // Make a perpendicular axis pointing from centroid towards
	// Hoogstein edge (e.g., major groove in a double helix).
  int H_atom = 0;
  if ( res_type == na_rad ) H_atom = 20; //N7
  if ( res_type == na_rcy ) H_atom = 19; //C5
  if ( res_type == na_rgu ) H_atom = 21; //N7
  if ( res_type == na_ura ) H_atom = 19; //C5
  assert( H_atom > 0);

  xyzVector_float H_coord( & full_coord( 1, H_atom, 1) );
  y = H_coord - centroid; //not orthonormal yet...
  z = cross(x, y);
  z.normalize(); // Should point roughly 5' to 3' if in a double helix.

  y = cross(z, x);
  y.normalize(); //not necessary but doesn't hurt.

  //  std::cout << "WC : " << WC_coord << "   H : " << H_coord << "    centroid: " << centroid << std::endl;

  return kin::Stub( xyzMatrix_double::cols( x, y, z ), centroid );
}

///////////////////////////////////////////////////////////////////////////////

kin::Stub
get_base_coordinate_system(
   int const res_type,
   FArray3Da_float full_coord
   )
{
  numeric::xyzVector_float centroid = get_base_centroid(res_type, full_coord); //inefficient, really just need to compute once.
  return get_base_coordinate_system( res_type, full_coord(1,1,1), centroid);
}

void
initialize_query_pose_rna(
	 pose_ns::Pose & pose,
	 bool use_fasta /* = true */
)
{

	if ( use_fasta ){
		// read sequence
		if ( !files_paths::query_defined ) {
			std::cout << "STOP:: need 3 args for initialize_query_pose" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		bool seq_undefined = true;
		read_aa( seq_undefined );
		if ( seq_undefined ) {
			std::cout << "STOP:: Need fasta for RNA -- use a,g,c,u for RNA bases." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}


	// setup start pose
	pose.simple_fold_tree( misc::total_residue );
	for (int i=1; i<= misc::total_residue; ++i ) {
		pose.set_res( i, misc::res(i) );
		pose.set_allow_bb_move( i, true ); //stupid.
		pose.set_allow_chi_move( i, true ); //stupid.
	}

	make_ideal_rna_fullcoord( pose ); // uniform torsion angles, sets up atom tree too.
}

///////////////////////////////////////////////////////////////////////////////
void get_random_pose_from_input( silent_io::Silent_file_data & decoys, pose_ns::Pose & pose  ){
	using namespace silent_io;
	using namespace pose_ns;

  std::vector< std::string > tag_list = decoys.tags();
	int const num_tags = tag_list.size();

	int const random_tag_num = static_cast<int> (  ran3() * num_tags ) + 1;
	int j = 1;
	std::vector<std::string>::const_iterator it;
	for ( it = tag_list.begin(); j < random_tag_num; ++it,++j);
	std::string const & tag( *it );

	Silent_structure const & decoy( decoys.get_structure( tag ) );
	decoy.fill_pose( pose, true );
}

///////////////////////////////////////////////////////////////////////////////
void super_monte_carlo_boltzmann( pose_ns::Pose & pose,
																	pose_ns::Monte_carlo & mc,
																	pose_ns::Monte_carlo & mc_super ){
	using namespace pose_ns;

	pose = mc.low_pose();

	//How are we doing with respect to the expensive score function?
	Score_weight_map const w_super = mc_super.weight_map();
	//	pose.new_score_pose();
	pose.score( w_super );
	mc_super.boltzmann( pose );

	//OK, little monte carlo, you start from scratch.
	Score_weight_map const w = mc.weight_map();
	//	pose.new_score_pose();
	pose.score( w );
	mc.reset( pose );
	//	mc.reset_counters();
}
///////////////////////////////////////////////////////////////////////////////
pose_ns::Score_weight_map setup_rna_weight_map(){

	Score_weight_map w;
	add_rna_lores_terms_weight_map( w );

	return w;
}

///////////////////////////////////////////////////////////////////////////////
void
add_rna_lores_terms_weight_map( pose_ns::Score_weight_map & w){

	w.set_weight( RNA_BP_W, realafteroption("rna_bp_w_weight",1.0) );
	w.set_weight( RNA_BP_H, realafteroption("rna_bp_h_weight",1.0) );
	w.set_weight( RNA_BP_S, realafteroption("rna_bp_s_weight",1.0) );
	w.set_weight( RNA_BS, realafteroption("rna_bs_weight",1.0) );
	w.set_weight( RNA_AXIS, realafteroption("rna_axis_weight",0.2) );
	w.set_weight( RNA_STAGGER, realafteroption("rna_stagger_weight",1.0) );
	w.set_weight( RNA_BULGE, realafteroption("rna_bulge_weight", 0.0f ) );
	w.set_weight( RNA_NONBASEBASE, realafteroption("rna_nonbasebase_weight", 1.0f ) );
	w.set_weight( RNA_O2STAR, realafteroption("rna_o2star_weight", 1.0f ) );
	w.set_weight( RNA_PHOSPHATE, realafteroption("rna_phosphate_weight", 1.0f ) );
	w.set_weight( VDW, realafteroption("vdw_weight",1.0) );
	w.set_weight( RNAONLY_SCOREFXN, 1.0 );
	w.set_weight( RG, realafteroption("rg_weight",1.0) );
	w.set_weight( CONTACT_MJ, 1.0 );
	w.set_weight( RNA_CONTACT, realafteroption("rna_contact_weight",1.0) );
	w.set_weight( RNA_LONG_RANGE_CONTACT, realafteroption("rna_long_range_contact_weight",1.0) );
	w.set_weight( RNA_SASA, realafteroption("rna_sasa_weight",1.0) );

	w.set_weight( BARCODE_ENERGY, 1.0 );

}


///////////////////////////////////////////////////////////////////////////////
void
RNA_move_trial( pose_ns::Pose & pose, pose_ns::Monte_carlo & mc, bool const smooth, std::string const trial_type, int const frag_size /* = 3 */ )
{
	// Always do a fragment move.
	random_fragment_trial( pose, mc, smooth, trial_type, frag_size );

	//Following returns early if there are no jumps.
	static float const jump_change_frequency =	realafteroption("jump_change_frequency", 0.1);
	if  (ran3() < jump_change_frequency) 	random_jump_trial( pose, mc, trial_type );

}


///////////////////////////////////////////////////////////////////////////////
void
update_frag_size( int const r, int const rounds, int & frag_size )
{

	static bool const do_2mers_in_last_half       = truefalseoption("do_2mers_in_last_half");
	static bool const do_1mers_in_last_round      = truefalseoption("do_1mers_in_last_round");

	static bool const do_2mers_after_first_third  = truefalseoption("do_2mers_after_first_third");
	static bool const do_1mers_in_last_third      = truefalseoption("do_1mers_in_last_third");

	if ( (r >= rounds / 2)   && do_2mers_in_last_half ){
		if (frag_size != 2) std::cout << "Changing fragment size from " << frag_size << " to 2." << std::endl;
		frag_size = 2;
	}

	if (r == rounds-1 && do_1mers_in_last_round ){
		if (frag_size != 1) std::cout << "Changing fragment size from " << frag_size << " to 1." << std::endl;
				frag_size = 1;
	}

	if ( (r >= rounds / 3)   && do_2mers_after_first_third ){
		if (frag_size != 2) std::cout << "Changing fragment size from " << frag_size << " to 2." << std::endl;
		frag_size = 2;
	}

	if (r >= 2 * (rounds/ 3) && do_1mers_in_last_third ){
		if (frag_size != 1) std::cout << "Changing fragment size from " << frag_size << " to 1." << std::endl;
				frag_size = 1;
	}

}

///////////////////////////////////////////////////////////////////////////////
void
put_the_final_touch_on_rna(
      pose_ns::Pose & pose,
			silent_io::Silent_out & out,
			std::string const tag,
			bool const minimize
			)
{

	//	pose.reset_extra_scores(); //why was this necessary? Oh yeah. Lots of junk.

	if (minimize) {

		//Save some info for later. Where did we start?
		calc_rms( pose );
		float const original_lowres_score = pose.get_0D_score( SCORE );
		float const original_lowres_rmsd  = pose.get_0D_score( RMSD );

		minimize_rna( pose );

		pose.set_extra_score( "ORIG_SCORE", original_lowres_score );
		pose.set_extra_score( "ORIG_RMS"  , original_lowres_rmsd  );
	}

	if (pose.native_pose_exists()) {

		calc_rms( pose );

		fill_native_base_pair_info( pose );

		float const allatom_rmsd_value = allatom_rmsd( pose, pose.native_pose() );
		pose.set_extra_score( "ALL_RMS", allatom_rmsd_value );
	}

	calc_gsolt( pose );
	std::cout << tag << " " << pose.show_scores() << std::endl;
	out.write(tag, pose );

	//	pose.dump_pdb( tag+".pdb" );

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
rna_stuff()
{
	using namespace param_aa;
	using namespace pose_ns;
	using namespace silent_io;

	bool const jumping = truefalseoption("pairing_file") || truefalseoption("jumps_from_barcode");

	files_paths::mode_title = "RNA ab initio"; //For boinc graphics output.

	// read pose
	Pose native_pose, start_pose;

	/////////////////////////////////////////
	// NATIVE
	/////////////////////////////////////////
	bool native_exists = false;
	if (truefalseoption("n")){
		native_exists = true;
		std::string native_file_name = stringafteroption("n","blah.pdb");
		bool success = pose_from_pdb( native_pose, native_file_name,
									 true, false, true );
		if (!success){
			std::cout << "Had trouble with native pdb " << native_file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		pose_to_native( native_pose );
		//		native_pose.dump_pdb( "native.pdb" );
	}


	/////////////////////////////////////////
	// STARTING STRUCTURE, FROM SCRATCH
	/////////////////////////////////////////
	//	std::string const start_pdb_name =  stringafteroption("start");
	std::string const start_pdb_name = "blah.pdb";
	initialize_query_pose_rna( start_pose );

	read_rna_fragments();


	/////////////////////////////////////////
	// STARTING STRUCTURE, USER INPUT
	/////////////////////////////////////////
	bool const user_input = truefalseoption("s");
	Silent_file_data decoys;
	if (user_input){
		std::string silent_file_name = stringafteroption("s");
		decoys.read_file( silent_file_name, true /* fullatom */);
	}


	/////////////////////////////////////////
	// WEIGHT MAP SETUP
	/////////////////////////////////////////
	Score_weight_map w = setup_rna_weight_map();

	float const rna_axis_weight_final =  w.get_weight( RNA_AXIS );
	float const rna_stagger_weight_final =  w.get_weight( RNA_STAGGER );

	w.set_weight( RNA_SASA, 0.0 ); // Turn on SASA in "super weight map" below.

	float chainbreak_weight = realafteroption("chainbreak_weight",1.0);
	float const chainbreak_weight_start = realafteroption("chainbreak_weight_start",chainbreak_weight);
	float const chainbreak_weight_final   = realafteroption("chainbreak_weight_final",chainbreak_weight);	w.set_weight( CHAINBREAK, chainbreak_weight_start );
	w.set_weight( CHAINBREAK_OVERLAP, 0 );


	////////////////////////////////////////////////////////
	// Double Monte Carlo for expensive SASA calculation?
	////////////////////////////////////////////////////////
	Score_weight_map w_super( w );
	bool const super_monte_carlo = truefalseoption( "sasa_monte_carlo" );
	int const super_monte_carlo_period = intafteroption( "sasa_monte_carlo_period", 40);
	float const rna_sasa_weight = realafteroption("rna_sasa_weight",1.0);
	w_super.set_weight( RNA_SASA, rna_sasa_weight );

	int nstruct, cycles, rounds;
	intafteroption("nstruct", 100, nstruct );
	intafteroption("cycles", 50000, cycles );
	intafteroption("rounds", 3, rounds );
	static bool const do_smooth_moves = truefalseoption("smooth_moves");
	static bool const minimize = truefalseoption("minimize_rna");

	//The most common name of an outfile....
	Silent_out out( files_paths_pdb_out_prefix_nochain()+".out" );

	//Record native info.
#ifndef BOINC
	if (native_exists && !truefalseoption("benchmark")) {
		if (out.start_decoy( "NATIVE")) {
			Pose native_pose_copy;
			native_pose_copy = native_pose;

			native_pose_copy.score( w_super );
			native_pose_copy.set_native_pose( native_pose );
			put_the_final_touch_on_rna( native_pose_copy, out, "NATIVE", minimize );
		}
	}
#endif

	barcode_initialize_start( start_pose.total_residue() );

	////////////////////////////////
	////////////////////////////////
	// MAIN LOOP Here we go!!!
	////////////////////////////////
	////////////////////////////////
	for (int n = 1; n <= nstruct; n++ ){
		std::string tag( "S_"+string_of(n) );
		if ( !out.start_decoy(tag) ) continue; // already done or started

		std::cout << " RNA_POSE Here we go ... " << tag << std::endl;

		barcode_initialize_decoy();

		Pose pose;
		if (user_input){
			get_random_pose_from_input( decoys, pose ); //better come with its own fold tree.
		} else { //start from scratch
			pose = start_pose;
			if (jumping) pose_from_random_base_pairings( pose );
		}
		atom_tree_set_allow_move( pose, true /* move bb*/, true /*move sc*/, true /*move jump*/ );

		// No chainbreak score applied across special jumps that connect i, i+1.
		// Not exactly the best heuristic.
		//		prepare_cut_weight_for_pose( pose, w );
		//		prepare_cut_weight_for_pose( pose, w_super );

		prepare_cut_weight_for_pose_from_user_pairings( pose, w );
		prepare_cut_weight_for_pose_from_user_pairings( pose, w_super );

		if (native_exists) pose.set_native_pose( native_pose );

		int frag_size = intafteroption( "frag_size", 3);

		/////////////////////////////////////////
		// Stage 0. heat the structure.
		/////////////////////////////////////////
		if (!user_input){
			std::cout << " RNA_POSE heating the structure ... " <<std::endl;
			int const HEAT_CYCLES = 5 * pose.total_residue();
			for (int i = 1; i <= HEAT_CYCLES; i++ ){
				random_fragment_insertions( pose, frag_size, 1 );
			}
		}

		// create Monte_carlo object that we will use throughout
		w.set_weight( RNA_AXIS, 0.0 );
		w.set_weight( RNA_STAGGER, 0.0 );
		w.set_weight( CHAINBREAK, chainbreak_weight_start );

		float const init_temp = realafteroption("temperature", 2.0);
		Monte_carlo mc( pose, w, init_temp );
		mc.set_autotemp( true, init_temp );

		//Following not actually used unless user specifies "-sasa_monte_carlo"
		// A "super" monte carlo with a more expensive score function (with SASA).
		Monte_carlo mc_super( pose, w_super, init_temp );
		mc_super.set_autotemp( true, init_temp );

		prof::reset();
		mc.reset_counters();
		mc_super.reset_counters();

		/////////////////////////////////////////////////////////////////
		// Stage 1 to end. Base pairing and stacking, bump check, Rg.
		//    Titrate up axis/stagger terms (which are for fine tuning).
		////////////////////////////////////////////////////////////////
		bool smooth( false );
		for (int r = 0; r < rounds; r++ ){
			std::string trial_type = "round"+string_of(r);

			w.set_weight( RNA_AXIS, r * rna_axis_weight_final/(rounds - 1) );
			w.set_weight( RNA_STAGGER, r * rna_stagger_weight_final/(rounds - 1) );
			chainbreak_weight = chainbreak_weight_start +
				(chainbreak_weight_final - chainbreak_weight_start)* ( r/(rounds-1));
			w.set_weight( CHAINBREAK, chainbreak_weight );
			mc.set_weight_map( w );
			pose = mc.low_pose();

			w_super = w;
			w_super.set_weight( RNA_SASA, rna_sasa_weight );
			mc_super.set_weight_map( w_super );

			//NOTE: smooth fragment insertions is not working presently, with new fragment class.
			if (r == rounds-1 && do_smooth_moves){
				smooth = true; //Smooth frags in last stage.
				trial_type +="-smooth";
				std::cout << "In the last round, kicking off smooth fragment insertions." << std::endl;
			}

			update_frag_size( r, rounds, frag_size );

			for (int i = 1; i  <= cycles/rounds; i++){
				if (mc.checkpoint( pose )) continue;
				RNA_move_trial( pose, mc, smooth, trial_type, frag_size );
				if (super_monte_carlo && mod(i,super_monte_carlo_period)==0)
					super_monte_carlo_boltzmann( pose, mc, mc_super );
			}
		}


		/////////////////////////////////////////////////////////////////
		// The end, for this decoy.
		////////////////////////////////////////////////////////////////
		prof::show();
		mc.show_counters();

		pose = mc.low_pose();
		if (super_monte_carlo) {
			super_monte_carlo_boltzmann( pose, mc, mc_super);
			pose = mc_super.low_pose();
			mc_super.show_counters();
		}

		pose.new_score_pose();
		pose.score( w_super ); //includes SASA term, if specified.

		pose.reset_extra_scores();  //remove MC_ACCEPTED, UNS_LR, other junk.
		static bool const store_accept_rates = truefalseoption( "store_accept_rates" );
		if (store_accept_rates) mc.store_counters_in_pose( pose ); //easy readout of accept rates.

		put_the_final_touch_on_rna( pose, out, tag, minimize);

		main_job_distributor::jd->reset_mc_checkpoint(); // reset monte carlo checkpoint counter

#ifdef BOINC
		out.append_to_list( tag ); // mark as done
		store_low_info(); // for trajectory plotting.
		clear_trajectory();
		counters::monte_carlo_ints::ntrials = 0;
		int farlx_stage = 0;
		bool ready_for_boinc_end = boinc_checkpoint_in_main_loop(n, n, nstruct, farlx_stage);
		if (ready_for_boinc_end)	return; // Go back to main, which will then go to BOINC_END and shut off BOINC.
#endif
	}

	std::cout << " Completed " << nstruct << " RNA decoys."<< std::endl;
	std::cerr << " Completed " << nstruct << " RNA decoys."<< std::endl;

}

///////////////////////////////////////////////////////////////////////////////
void
make_ideal_rna_fullcoord( pose_ns::Pose & pose ){
	using namespace misc;
	using namespace param;

	total_residue = pose.total_residue();

	if (!MAX_RES().initialized()) MAX_RES_assign_res( total_residue );
	if ( total_residue > MAX_RES() ) {
		MAX_RES_assign_res( total_residue);

		Eposition.dimension( 3, MAX_POS, MAX_RES() );
		phi.dimension( MAX_RES() );
		psi.dimension( MAX_RES() );
		omega.dimension( MAX_RES() );
	}

	for (int i = 1; i <= total_residue; i++) {
		res(i) = pose.res(i);
	}

	make_ideal_rna_fullcoord();
	pose_from_misc( pose, true /*fullatom*/, false /*ideal_pose*/, true /*coords_init*/);

	pose.set_allow_bb_move( true );
	pose.set_allow_chi_move( true );
}

///////////////////////////////////////////////////////////////////////////////
void
make_ideal_rna_fullcoord(){
	//
	// 1. assumes there is only RNA in misc.
	//
	// 2. There are a million ways to do this... hopefully this way isn't wrong.
	//
	// 3. Parameters for bond lengths and angles from x-plor,
	//  http://www.aist.go.jp/infobase/ndbserver/NDB/archives/proj/param_ndbx.dna
	//
	// 4. Still don't have aavariants -- in particular extra oxygen for 5' phosphate.
	//     or no phosphate at all at 5' terminus might be useful.
	//

	using namespace misc;
	using namespace aaproperties_pack;
	using namespace kin;
	using namespace numeric;
	using namespace rna_variables;
	using numeric::conversions::radians;

	// First residue -- copy from icoor.
	int  i = 1;
	int res_i = res(i);
	assert( param_aa::is_RNA( res_i) );

	int num_atoms = natoms( res_i, 1);
	for (int j = 1; j <= num_atoms; j++){
		for (int k = 1; k <= 3; k++){
			full_coord( k, j, i) =  icoor( k, j, res_i, 1);
		}
	}

	for (i = 2; i <= total_residue; i++){
		res_i = res(i);
		assert( param_aa::is_RNA( res_i) );

		// Build on the next P.
		Stub stub;
		xyzVector_double const center( &full_coord(1, o3star, i-1 ));
		xyzVector_double const a( &full_coord(1, o3star, i-1 ));
		xyzVector_double const b( &full_coord(1, c3star, i-1 ));
		xyzVector_double const c( &full_coord(1, c4star, i-1 ));
		stub.from_four_points( center, a, b, c);

		// stolen from kin_bonded_atom.cc
		double phi   = -152.0; // epsilon is trans in A-form RNA.
		double theta = 180.0 - 119.70;
		double d     = 1.607;
		stub.M *= X_rot_rad( radians( phi ) );
		Stub P_stub( stub.M * Z_rot_rad( radians( theta ) ), stub.v );
		P_stub.v += d * P_stub.M.col_x();
		for (int k = 1; k <= 3; k++) full_coord(k, p, i) = P_stub.v(k);

		//Time to add on O1P and O2P -- and that should be enough to figure
		// out how to build on the next residue.
		phi   = -74.0 + 120.0; // zeta is usually gauche- in A-form RNA.
		theta = 180.0 - 107.4;
		d     = 1.485;
		Stub O1P_stub( P_stub.M, P_stub.v );
		O1P_stub.M *= X_rot_rad( radians( phi ) );
		O1P_stub.M *= Z_rot_rad( radians( theta ) );
		O1P_stub.v += d * O1P_stub.M.col_x();
		for (int k = 1; k <= 3; k++) full_coord(k, o1p, i) = O1P_stub.v(k);

		phi   = -74.0 - 120.0; // this offset might not be totally correct, but will get fixed with icoor.
		theta = 180.0 - 108.3;
		d     = 1.485;
		Stub O2P_stub( P_stub.M, P_stub.v );
		O2P_stub.M *= X_rot_rad( radians( phi ) );
		O2P_stub.M *= Z_rot_rad( radians( theta ) );
		O2P_stub.v += d * O2P_stub.M.col_x();
		for (int k = 1; k <= 3; k++) full_coord(k, o2p, i) = O2P_stub.v(k);

		// What rotation/translation takes icoord P, O1P, O2P to current coords?
		FArray2D_float Mgl( 4, 4 );
		get_GL_matrix( icoor( 1, o1p, res_i, 1),  //where we are starting
									 icoor( 1, p  , res_i, 1),
									 icoor( 1, o2p, res_i, 1),
									 full_coord(1, o1p, i), // where we want to go
									 full_coord(1, p  , i),
									 full_coord(1, o2p, i),
									 Mgl);

		//OK, do it.
		num_atoms = natoms( res_i, 1);
		GL_rotate(num_atoms, 1, Mgl, icoor(1, 1, res_i, 1),
							full_coord(1, 1, i));
	}


}

///////////////////////////////////////////////////////////////////////////////
void
rna_from_scratch_test(){
	using namespace pose_ns;
	using namespace silent_io;

	Pose pose;
	pose_from_pdb( pose, stringafteroption("s"),
								 true, false, true );
	pose.dump_pdb( "start.pdb" );


	//Fold tree appropriate to 1j6s (RNA tetraplex)
	Fold_tree f;
	f.simple_tree( pose.total_residue() );

	//1j6s
	//	f.new_jump( 1, 14, 6);
	//	f.new_jump( 8, 13, 12);
	//	f.new_jump( 7, 20, 18);

	//tetraloop/receptor
	//	f.new_jump(  3, 14, 6 );
	//	f.new_jump( 12, 13, 12 );


	//	f.new_jump(  1, 3, 2 );
	f.new_jump(  3, 6, 4 );

	pose.set_fold_tree( f );

	// Fill full_coord with and RNA constructed with ideal bond angles
	// and bond distances.
	make_ideal_rna_fullcoord();

	Pose new_pose;
	pose_from_misc( new_pose, true, false, true );

	//Copy fold tree and jumps.
	new_pose.set_fold_tree( f );
	copy_jumps( new_pose, pose );

	Score_weight_map w;
	float const rna_axis_weight_final =  realafteroption("rna_axis_weight",0.2);
	float const rna_stagger_weight_final =  realafteroption("rna_stagger_weight",1.0);
	w.set_weight( RNA_BP_W, realafteroption("rna_bp_w_weight",0.5) );
	w.set_weight( RNA_BP_H, realafteroption("rna_bp_h_weight",1.0) );
	w.set_weight( RNA_BP_S, realafteroption("rna_bp_s_weight",4.0) );
	w.set_weight( RNA_BS, realafteroption("rna_bs_weight",1.0) );
	w.set_weight( RNA_AXIS, rna_axis_weight_final );
	w.set_weight( RNA_STAGGER, rna_stagger_weight_final );
	w.set_weight( VDW, realafteroption("vdw_weight",1.0) );
	w.set_weight( RNAONLY_SCOREFXN, 1.0 );
	w.set_weight( RG, realafteroption("rg_weight",1.0) );


	Silent_out out( "blah.out" );

	pose.score( w );
	//	out.write( "START", pose );
	new_pose.score( w );
	out.write( "NEW", new_pose );

	new_pose.dump_pdb( "final.pdb" );

}


///////////////////////////////////////////////////////////////////////////////
void
icoor_fix_base( FArray2Da_float goodcoor, int const restype,
								std::string const resname,
								int const baseatom1, int const baseatom2, int const baseatom3 ){
	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace rna_variables;

	FArray2D_float tempcoor( 3, 50, 0.0);
	goodcoor.dimension(3, 50);

	int const num_non_base_heavy_atoms = 12;
	int const num_non_base_hydrogen_atoms = 7;
	int num_base_heavy_atoms, num_base_hydrogen_atoms;
	int  num_heavy_atoms, num_atoms;

	num_heavy_atoms = nheavyatoms( restype, 1);
	num_atoms       = natoms( restype, 1);

	FArray2D_float Mgl( 4, 4 );

	num_base_heavy_atoms = num_heavy_atoms - num_non_base_heavy_atoms;
	num_base_hydrogen_atoms = num_atoms - num_heavy_atoms - num_non_base_hydrogen_atoms;

	//align new template coordinates to icoor, based on three sugar atoms.
	get_GL_matrix( goodcoor(1, c2star), // where we are starting
								 goodcoor(1, c1star),
								 goodcoor(1, o4star),
								 icoor( 1, c2star, restype, 1),  //where we want to go
								 icoor( 1, c1star, restype, 1),
								 icoor( 1, o4star, restype, 1),
								 Mgl);
	GL_rotate(num_heavy_atoms, 1, Mgl, goodcoor(1, 1),
						tempcoor(1, 1));

	//rotate icoor's base to match onto new template.
	get_GL_matrix( icoor( 1, baseatom2, restype, 1),  //where we are starting
								 icoor( 1, baseatom1, restype, 1),
								 icoor( 1, baseatom3, restype, 1),
								 tempcoor(1, baseatom2), // where we want to go
								 tempcoor(1, baseatom1),
								 tempcoor(1, baseatom3),
								 Mgl);

	//Base atoms in icoor can now be replaced.
	for (int i = num_non_base_heavy_atoms + 1; i<= num_heavy_atoms; i++)
		GL_rot_in_place(Mgl, icoor(1, i, restype, 1));
	for (int i = num_heavy_atoms + num_non_base_hydrogen_atoms + 1; i<= num_atoms; i++)
		GL_rot_in_place(Mgl, icoor(1, i, restype, 1));

	//Print out in useful format for fixing read_approperties.cc
	for (int i = 1; i<= num_atoms; i++){
		for (int k = 1; k<= 3; k++){
			std::cout << "  icoor(" << I(2, k) << "," << I(2,i) << ", "
								<< resname << ", 1) = " << F(8, 3, icoor( k, i, restype, 1) ) <<
				"; // " << atom_name(i, restype, 1) << std::endl;
		}
	}
	std::cout << std::endl;

}

///////////////////////////////////////////////////////////////////////////////

void
rna_icoor_fix(){
	// from 2ina.pdb,  a bunch of A-form helices that apparently used
	// NDB parameters...
	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace rna_variables;

	FArray2D_float goodcoor( 3, 50, 0.0);
	int l, restype;
	int baseatom1, baseatom2, baseatom3;
	std::string resname;

	//C
	restype = na_rcy;
	resname = "na_rcy";
	l = goodcoor.index(1,1);
	goodcoor[l++] = 47.118; goodcoor[l++] = -14.883; goodcoor[l++] =  -1.858;
	goodcoor[l++] = 45.897; goodcoor[l++] = -14.050; goodcoor[l++] =  -1.706;
	goodcoor[l++] = 47.244; goodcoor[l++] = -15.794; goodcoor[l++] =  -3.024;
	goodcoor[l++] = 48.390; goodcoor[l++] = -13.921; goodcoor[l++] =  -1.852;
	goodcoor[l++] = 48.308; goodcoor[l++] = -12.581; goodcoor[l++] =  -1.360;
	goodcoor[l++] = 49.213; goodcoor[l++] = -11.679; goodcoor[l++] =  -2.166;
	goodcoor[l++] = 50.596; goodcoor[l++] = -12.034; goodcoor[l++] =  -1.898;
	goodcoor[l++] = 49.083; goodcoor[l++] = -11.785; goodcoor[l++] =  -3.678;
	goodcoor[l++] = 48.014; goodcoor[l++] = -10.986; goodcoor[l++] =  -4.173;
	goodcoor[l++] = 50.451; goodcoor[l++] = -11.308; goodcoor[l++] =  -4.151;
	goodcoor[l++] = 50.575; goodcoor[l++] =  -9.901; goodcoor[l++] =  -4.050;
	goodcoor[l++] = 51.358; goodcoor[l++] = -11.925; goodcoor[l++] =  -3.088;
	goodcoor[l++] = 51.875; goodcoor[l++] = -13.267; goodcoor[l++] =  -3.411;
	goodcoor[l++] = 53.133; goodcoor[l++] = -13.386; goodcoor[l++] =  -4.011;
	goodcoor[l++] = 53.747; goodcoor[l++] = -12.358; goodcoor[l++] =  -4.333;
	goodcoor[l++] = 53.649; goodcoor[l++] = -14.618; goodcoor[l++] =  -4.224;
	goodcoor[l++] = 52.954; goodcoor[l++] = -15.702; goodcoor[l++] =  -3.874;
	goodcoor[l++] = 53.514; goodcoor[l++] = -16.893; goodcoor[l++] =  -4.081;
	goodcoor[l++] = 51.656; goodcoor[l++] = -15.611; goodcoor[l++] =  -3.293;
	goodcoor[l++] = 51.156; goodcoor[l++] = -14.385; goodcoor[l++] =  -3.091;
	baseatom1 = 13; baseatom2 = 20; baseatom3 = 14; //N1, C6, C2
	icoor_fix_base( goodcoor(1,1), restype, resname, baseatom1, baseatom2, baseatom3);

	//U
	restype = na_ura;
	resname = "na_ura";
	l = goodcoor.index(1,1);
	goodcoor[l++] = 47.224; goodcoor[l++] = -11.441; goodcoor[l++] =  -5.497;
	goodcoor[l++] = 46.221; goodcoor[l++] = -10.388; goodcoor[l++] =  -5.798;
	goodcoor[l++] = 46.782; goodcoor[l++] = -12.848; goodcoor[l++] =  -5.323;
	goodcoor[l++] = 48.337; goodcoor[l++] = -11.409; goodcoor[l++] =  -6.637;
	goodcoor[l++] = 49.028; goodcoor[l++] = -10.203; goodcoor[l++] =  -6.950;
	goodcoor[l++] = 50.042; goodcoor[l++] = -10.451; goodcoor[l++] =  -8.042;
	goodcoor[l++] = 51.194; goodcoor[l++] = -11.129; goodcoor[l++] =  -7.472;
	goodcoor[l++] = 49.599; goodcoor[l++] = -11.365; goodcoor[l++] =  -9.175;
	goodcoor[l++] = 48.832; goodcoor[l++] = -10.680; goodcoor[l++] = -10.160;
	goodcoor[l++] = 50.933; goodcoor[l++] = -11.874; goodcoor[l++] =  -9.704;
	goodcoor[l++] = 51.614; goodcoor[l++] = -10.878; goodcoor[l++] = -10.446;
	goodcoor[l++] = 51.706; goodcoor[l++] = -12.068; goodcoor[l++] =  -8.403;
	goodcoor[l++] = 51.583; goodcoor[l++] = -13.410; goodcoor[l++] =  -7.813;
	goodcoor[l++] = 52.541; goodcoor[l++] = -14.355; goodcoor[l++] =  -8.142;
	goodcoor[l++] = 53.432; goodcoor[l++] = -14.146; goodcoor[l++] =  -8.947;
	goodcoor[l++] = 52.418; goodcoor[l++] = -15.555; goodcoor[l++] =  -7.490;
	goodcoor[l++] = 51.450; goodcoor[l++] = -15.906; goodcoor[l++] =  -6.576;
	goodcoor[l++] = 51.489; goodcoor[l++] = -17.019; goodcoor[l++] =  -6.053;
	goodcoor[l++] = 50.481; goodcoor[l++] = -14.890; goodcoor[l++] =  -6.315;
	goodcoor[l++] = 50.568; goodcoor[l++] = -13.708; goodcoor[l++] =  -6.931;
	baseatom1 = 13; baseatom2 = 20; baseatom3 = 14;//N1, C6, C2
	icoor_fix_base( goodcoor(1,1), restype, resname, baseatom1, baseatom2, baseatom3);


	//G
	restype = na_rgu;
	resname = "na_rgu";
	l = goodcoor.index(1,1);
	goodcoor[l++] = 47.842; goodcoor[l++] = -11.511; goodcoor[l++] = -11.119;
	goodcoor[l++] = 47.083; goodcoor[l++] = -10.522; goodcoor[l++] = -11.926;
	goodcoor[l++] = 47.106; goodcoor[l++] = -12.506; goodcoor[l++] = -10.297;
	goodcoor[l++] = 48.828; goodcoor[l++] = -12.295; goodcoor[l++] = -12.093;
	goodcoor[l++] = 49.864; goodcoor[l++] = -11.603; goodcoor[l++] = -12.782;
	goodcoor[l++] = 50.664; goodcoor[l++] = -12.557; goodcoor[l++] = -13.638;
	goodcoor[l++] = 51.696; goodcoor[l++] = -13.180; goodcoor[l++] = -12.823;
	goodcoor[l++] = 49.893; goodcoor[l++] = -13.738; goodcoor[l++] = -14.206;
	goodcoor[l++] = 49.175; goodcoor[l++] = -13.390; goodcoor[l++] = -15.386;
	goodcoor[l++] = 51.003; goodcoor[l++] = -14.749; goodcoor[l++] = -14.465;
	goodcoor[l++] = 51.753; goodcoor[l++] = -14.417; goodcoor[l++] = -15.619;
	goodcoor[l++] = 51.911; goodcoor[l++] = -14.511; goodcoor[l++] = -13.262;
	goodcoor[l++] = 53.306; goodcoor[l++] = -18.889; goodcoor[l++] = -11.136;
	goodcoor[l++] = 53.672; goodcoor[l++] = -18.342; goodcoor[l++] = -12.339;
	goodcoor[l++] = 54.549; goodcoor[l++] = -19.039; goodcoor[l++] = -13.064;
	goodcoor[l++] = 53.208; goodcoor[l++] = -17.197; goodcoor[l++] = -12.803;
	goodcoor[l++] = 52.332; goodcoor[l++] = -16.626; goodcoor[l++] = -11.953;
	goodcoor[l++] = 51.893; goodcoor[l++] = -17.086; goodcoor[l++] = -10.730;
	goodcoor[l++] = 52.403; goodcoor[l++] = -18.323; goodcoor[l++] = -10.243;
	goodcoor[l++] = 52.147; goodcoor[l++] = -18.902; goodcoor[l++] =  -9.178;
	goodcoor[l++] = 50.979; goodcoor[l++] = -16.202; goodcoor[l++] = -10.170;
	goodcoor[l++] = 50.888; goodcoor[l++] = -15.238; goodcoor[l++] = -11.046;
	goodcoor[l++] = 51.700; goodcoor[l++] = -15.421; goodcoor[l++] = -12.139;
	baseatom1 = 23; baseatom2 = 22; baseatom3 = 17;//N9, C8, C4
	icoor_fix_base( goodcoor(1,1), restype, resname, baseatom1, baseatom2, baseatom3);

	//A
	restype = na_rad;
	resname = "na_rad";
	l = goodcoor.index(1,1);
	goodcoor[l++] = 44.758; goodcoor[l++] = -23.894; goodcoor[l++] = -19.806;
	goodcoor[l++] = 44.309; goodcoor[l++] = -24.292; goodcoor[l++] = -21.166;
	goodcoor[l++] = 44.078; goodcoor[l++] = -22.774; goodcoor[l++] = -19.108;
	goodcoor[l++] = 44.700; goodcoor[l++] = -25.181; goodcoor[l++] = -18.867;
	goodcoor[l++] = 45.504; goodcoor[l++] = -26.324; goodcoor[l++] = -19.148;
	goodcoor[l++] = 45.188; goodcoor[l++] = -27.442; goodcoor[l++] = -18.183;
	goodcoor[l++] = 45.919; goodcoor[l++] = -27.226; goodcoor[l++] = -16.944;
	goodcoor[l++] = 43.738; goodcoor[l++] = -27.539; goodcoor[l++] = -17.735;
	goodcoor[l++] = 42.929; goodcoor[l++] = -28.222; goodcoor[l++] = -18.687;
	goodcoor[l++] = 43.864; goodcoor[l++] = -28.285; goodcoor[l++] = -16.412;
	goodcoor[l++] = 44.124; goodcoor[l++] = -29.663; goodcoor[l++] = -16.613;
	goodcoor[l++] = 45.141; goodcoor[l++] = -27.672; goodcoor[l++] = -15.845;
	goodcoor[l++] = 44.561; goodcoor[l++] = -26.275; goodcoor[l++] = -10.939;
	goodcoor[l++] = 44.641; goodcoor[l++] = -27.473; goodcoor[l++] = -11.534;
	goodcoor[l++] = 44.759; goodcoor[l++] = -27.765; goodcoor[l++] = -12.827;
	goodcoor[l++] = 44.794; goodcoor[l++] = -26.642; goodcoor[l++] = -13.566;
	goodcoor[l++] = 44.719; goodcoor[l++] = -25.342; goodcoor[l++] = -13.102;
	goodcoor[l++] = 44.600; goodcoor[l++] = -25.167; goodcoor[l++] = -11.709;
	goodcoor[l++] = 44.529; goodcoor[l++] = -23.976; goodcoor[l++] = -11.112;
	goodcoor[l++] = 44.780; goodcoor[l++] = -24.441; goodcoor[l++] = -14.157;
	goodcoor[l++] = 44.896; goodcoor[l++] = -25.205; goodcoor[l++] = -15.218;
	goodcoor[l++] = 44.935; goodcoor[l++] = -26.547; goodcoor[l++] = -14.930;
	baseatom1 = 22; baseatom2 = 21; baseatom3 = 16;//N9, C8, C4
	icoor_fix_base( goodcoor(1,1), restype, resname, baseatom1, baseatom2, baseatom3);


}

///////////////////////////////////////////////////////////////////////////////
void
rna_frag_bestpair( pose_ns::Pose & pose, pose_ns::Pose & fragments_pose, std::ofstream & outstream){

	using namespace numeric;

	int nres_fragments = fragments_pose.total_residue();
	int nres = pose.total_residue();
	int const FRAG_SIZE = 2;

	FArray1D_bool good_residue( nres, true );
	float const chainbreak_cutoff ( 7.5 * 7.5 );
	//Check for chainbreaks
	for (int i = 1; i <= nres - FRAG_SIZE + 1; i++) {
		xyzVector_float P1 = pose.full_coord( rna_variables::c4star, i   );
		xyzVector_float P2 = pose.full_coord( rna_variables::c4star, i+1 );
		if ((P1-P2).length_squared() > chainbreak_cutoff){
			std::cout << " CHAINBREAK? " << i << " " << "  dist to next residue " << (P1-P2).length() << std::endl;
				good_residue( i ) = false;
		}
	}

	for (int i = 1; i <= nres - FRAG_SIZE + 1; i++) {
		if (!pose.is_RNA(i)) continue;
		if (!good_residue(i)) continue;

		float best_frag_rms_all   = 100.0;
		float best_frag_rms_YR    = 100.0;
		float best_frag_rms_exact = 100.0;

		for (int j = 1; j <= nres_fragments - FRAG_SIZE + 1; j++){
			if (i==j) continue;
			if (!fragments_pose.is_RNA(j)) continue;

			float const frag_rms = get_frag_rms( pose, fragments_pose, i, j, FRAG_SIZE);

			bool frag_match = true;
			if (frag_match && (frag_rms < best_frag_rms_all)) best_frag_rms_all = frag_rms;

			for (int k = 1; k <= FRAG_SIZE; k++ ){
				if (!pur_pyr_comp(fragments_pose.res(i + k - 1), fragments_pose.res(j + k - 1)) )
					frag_match = false;
			}
			if (frag_match && (frag_rms < best_frag_rms_YR)) best_frag_rms_YR = frag_rms;


			for (int k = 1; k <= FRAG_SIZE; k++ ){
				if (!(fragments_pose.res(i + k - 1) == fragments_pose.res(j + k - 1)))
					frag_match = false;
			}
			if (frag_match &&	(frag_rms < best_frag_rms_exact)) best_frag_rms_exact = frag_rms;

		}

		outstream << i << " ";
		for (int k = 1; k <= FRAG_SIZE; k++) outstream << pose.res(i + k - 1) << " ";
		outstream << best_frag_rms_all
							<< " " << best_frag_rms_YR
							<< " " << best_frag_rms_exact
							<< std::endl;

	}

}

///////////////////////////////////////////////////////////////////////////////
void
rna_frag_bestpair_test(){
	// read pose
	Pose fragments_pose, pose;

	pose_from_pdb( pose, stringafteroption("s"),
								 true, false, true );

	pose_from_pdb( fragments_pose, stringafteroption("fragments"),
								 true, false, true );

	std::ofstream outstream( "rna_frags_bestpair.txt" );
	rna_frag_bestpair(pose, fragments_pose,outstream );
	outstream.close();

}

///////////////////////////////////////////////////////////////////////////////
float
get_frag_rms( pose_ns::Pose const & pose1, pose_ns::Pose const & pose2,
							int const res1, int const res2, int const frag_size){

	using namespace rna_variables;

	// Just look at P, C4*, and base centroid.
	int const numatoms = 3;

	int total_residue = frag_size * numatoms; // kind of a misnomer
	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );

	int count = 0;
	int atom_index;
	for (int i = 1; i <= frag_size; i++){
		count++;
		atom_index = p;
		for ( int k = 1; k <= 3; ++k ) p1a(k,count) = pose1.full_coord( atom_index, res1+i-1 )(k);
		for ( int k = 1; k <= 3; ++k ) p2a(k,count) = pose2.full_coord( atom_index, res2+i-1 )(k);

		count++;
		atom_index = c4star;
		for ( int k = 1; k <= 3; ++k ) p1a(k,count) = pose1.full_coord( atom_index, res1+i-1 )(k);
		for ( int k = 1; k <= 3; ++k ) p2a(k,count) = pose2.full_coord( atom_index, res2+i-1 )(k);

		count++;
		for ( int k = 1; k <= 3; ++k ) p1a(k,count) = pose1.centroid( res1+i-1 )(k);
		for ( int k = 1; k <= 3; ++k ) p2a(k,count) = pose2.centroid( res2+i-1 )(k);
	}

	// calc rms
	FArray1D_double const ww( total_residue, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	findUU( p1a, p2a, ww, total_residue, uu, ctx );
	float fast_rms;
	calc_rms_fast( fast_rms, p1a, p2a, ww, total_residue, ctx );
	return fast_rms;
}

///////////////////////////////////////////////////////////////////////////////
float
get_frag_rms( pose_ns::Pose  & pose1, int const res1, int const res2, int const frag_size){
	return get_frag_rms( pose1, pose1, res1, res2, frag_size);
}

///////////////////////////////////////////////////////////////////////////////
void
rna_frag_allpairs( pose_ns::Pose & pose, std::ofstream & outstream ){

	using namespace numeric;

	int total_residue = pose.total_residue();
	int const FRAG_SIZE = 3;
	FArray1D_bool good_residue( total_residue, true );
	float const chainbreak_cutoff ( 7.5 * 7.5 );
	//Check for chainbreaks
	for (int i = 1; i <= total_residue - FRAG_SIZE + 1; i++) {
		xyzVector_float P1 = pose.full_coord( rna_variables::c4star, i   );
		xyzVector_float P2 = pose.full_coord( rna_variables::c4star, i+1 );
		if ((P1-P2).length_squared() > chainbreak_cutoff){
			std::cout << " CHAINBREAK? " << i << " " << "  dist to next residue " << (P1-P2).length() << std::endl;
				good_residue( i ) = false;
		}
	}


	for (int i = 1; i <= total_residue - FRAG_SIZE + 1; i++) {
		if (!pose.is_RNA(i)) continue;
		if (!good_residue(i) || !good_residue(i+1)) continue;
		for (int j = i+1; j <= total_residue - FRAG_SIZE + 1; j++){
			if (!pose.is_RNA(j)) continue;
			if (!good_residue(j) || !good_residue(j+1)) continue;

			float const frag_rms = get_frag_rms( pose, i, j, FRAG_SIZE);

			for (int k = 1; k <= FRAG_SIZE; k++) outstream << pose.res(i + k - 1) << " " ;
			for (int k = 1; k <= FRAG_SIZE; k++) outstream << pose.res(j + k - 1) << " " ;
			outstream << frag_rms << std::endl;

		}
	}
}



///////////////////////////////////////////////////////////////////////////////
void
rna_frag_allpairs_test(){
	// read pose
	Pose fragments_pose;

	pose_from_pdb( fragments_pose, stringafteroption("fragments"),
								 true, false, true );

	std::ofstream outstream( "rna_frags_allpairs.txt" );
	rna_frag_allpairs(fragments_pose, outstream);
	outstream.close();

}
///////////////////////////////////////////////////////////////////////////////
void
print_sasa( pose_ns::Pose & pose, FArray2D_float & atom_sasa, std::ofstream & outstream ){
	using namespace pose_ns;
	using namespace rna_variables;

	int const nres = pose.total_residue();

	outstream << A(5,"SASA") << A(9,"residue ") << A(9,"sugars") <<
		A(9,"C1*") <<
		A(9,"C2*") <<
		A(9,"C3*") <<
		A(9,"C4*") <<
		A(9,"C5*") << std::endl;


	for (int i=1; i<=nres; i++){
		float const total_SASA =
			atom_sasa(c1star,i) +
			atom_sasa(c2star,i) +
			atom_sasa(c3star,i) +
			atom_sasa(c4star,i) +
			atom_sasa(c5star,i);

		outstream << A(5,"SASA") << I(10,i) << " " << F(8,2,total_SASA) <<
			" " << F(8,2, atom_sasa(c1star,i) ) <<
			" " << F(8,2, atom_sasa(c2star,i) ) <<
			" " << F(8,2, atom_sasa(c3star,i) ) <<
			" " << F(8,2, atom_sasa(c4star,i) ) <<
			" " << F(8,2, atom_sasa(c5star,i) ) << std::endl;

	}

}

///////////////////////////////////////////////////////////////////////////////
void rna_sasa_test(){
	using namespace pose_ns;
	using namespace param;


	prof::reset();

	// read pose
	Pose pose;
	pose_from_pdb( pose, stringafteroption("s"),
								 true, false, true );

	//Yeah, check it out.
	FArray2D_float atom_sasa( MAX_ATOM(), MAX_RES()());
	FArray1D_float rsd_sasa( MAX_RES());
	float const probe_radius = 1.400;
	calc_per_atom_sasa( atom_sasa, rsd_sasa, probe_radius,
											false /*ignore hydrogens*/, false /*self_chain_only*/);

	std::ofstream outstream( "SASA.txt" );
	print_sasa( pose, atom_sasa, outstream );
	outstream.close();

	//Try out score...
	float rna_sasa_score;
	eval_rna_sasa_score( pose, rna_sasa_score, true /*verbose*/, true);

	eval_rna_sasa_score( pose, rna_sasa_score, true /*verbose*/, false);

	prof::show();

}


///////////////////////////////////////////////////////////////////////////////

void
rna_test()
{

	//Preliminary setup, specific to RNA-only modes.
	files_paths::skip_dun = true; //skip bbdep, avgE, all that protein crap.
	pose_param::slow_and_safe_atom_tree_copy = false; //speeds up atom tree stuff.

	if (truefalseoption("rna_read_jumps")){
		read_jump_templates_RNA();
	} else if (truefalseoption("rna_basepair_database")){
		make_rna_basepair_database();
	} else if (truefalseoption("rna_frag_allpairs")){
		rna_frag_allpairs_test();
	} else if (truefalseoption("rna_pdb_stats")){
		rna_pdb_stats_test();
	} else	if (truefalseoption("rna_from_scratch")){
		rna_from_scratch_test();
	} else if (truefalseoption("rna_icoor_fix")){
		rna_icoor_fix();
	} else if (truefalseoption("rna_sasa")){
		rna_sasa_test();
	} else if (truefalseoption("rna_fullatom_minimize")){
		rna_fullatom_minimize_test();
	} else if (truefalseoption("rna_score")){
		rna_score_test();
	} else if (truefalseoption("pick_rna_fragments")){
		pick_rna_fragments_main();
	} else if (truefalseoption("rna_centroid_information")){
		rna_centroid_information_test();
	} else if (truefalseoption("pairwise_rmsd")){
		pairwise_rmsd_test();
	} else if (truefalseoption("p5c_helix")){
		p5c_helix_test();
	} else if (truefalseoption("create_vall_torsions_file")){
		create_vall_torsions_file_test();
	} else if (truefalseoption("rna_fragments_classes_test")){
		rna_fragments_classes_test();
	} else if (truefalseoption("rna_featurizer")){
		rna_featurizer_test();
	} else if (truefalseoption("rna_csa")){
		rna_csa_test();
	} else if (truefalseoption("rna_o2star_repack")){
		rna_o2star_repack_test();
	} else if (truefalseoption("rna_closest_frag")){
		rna_closest_frag_test();
	} else {
		rna_stuff();
	}

	//	exit(0);

}

