// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: rhiju $


// Rosetta Headers
#include "pose_rna_jumping.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "dna.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "fullatom_energies.h"
#include "force_barcode.h"
#include "hbonds_ns.h"
#include "jumping_util.h"
#include "kin_stub.h"
#include "minimize.h"
#include "nblist.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_rna.h"
#include "pose_rna_ns.h"
#include "pose_constraints.h"
#include "pose_design.h"
#include "pose_rna_fragments.h"
#include "pose_vdw.h"
#include "prof.h"
#include "read_aaproperties.h"
#include "read_paths.h"
#include "random_numbers.h"
#include "silent_input.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/StaticIndexRange.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

using namespace pose_ns;

// Base pairing template could be expanded to copy
// torsion angles too, like Pairing_template in jumping_pairings.cc
class Pairing_template_RNA {
public:
	pose_ns::Jump jump;
	Pairing_template_RNA ( FArray2Da_float Epos1, FArray2Da_float Epos2 ):
		jump( Epos1, Epos2 ) // old-style jump. NOT APPROPRIATE FOR ATOM_TREE
	{
		//newstyle jump. There's actually no need for the 1,2,4 indexing anymore,
		// its just historical.
		numeric::xyzVector_double x1( & Epos1(1,1) );
		numeric::xyzVector_double y1( & Epos1(1,2) );
		numeric::xyzVector_double z1( & Epos1(1,4) );

		numeric::xyzVector_double x2( & Epos2(1,1) );
		numeric::xyzVector_double y2( & Epos2(1,2) );
		numeric::xyzVector_double z2( & Epos2(1,4) );

		// First  stub: upstream jump atom, its parent, and its parent's parent.
		kin::Stub     stub( z1, z1, y1, x1 );
		// Second stub: downstream jump atom, its first child, and its first child's first child
		kin::Stub new_stub( x2, x2, y2, z2 );
		jump.from_stubs( stub, new_stub);
	}
};

typedef std::vector< Pairing_template_RNA > Pairing_template_list_RNA;

class Base_pair_type{
public:
	char aa1;
	char aa2;
	char edge1;
	char edge2;
	char orientation;

	Base_pair_type( char const aa1_in, char const aa2_in,
									char const edge1_in, char const edge2_in,
									char const orientation_in){
		aa1 = aa1_in;		aa2 = aa2_in;
		edge1 = edge1_in;		edge2 = edge2_in;
		orientation = orientation_in;
	}

	friend
	bool operator < (Base_pair_type const & lhs, Base_pair_type const & rhs )
	{
		//There must be a more elegant way to do this...
		if( lhs.aa1 < rhs.aa1 )
			return true;
		else if ( lhs.aa1 == rhs.aa1 )
			if ( lhs.aa2 < rhs.aa2 )
				return true;
			else if ( lhs.aa2 == rhs.aa2 )
				if ( lhs.edge1 < rhs.edge1 )
					return true;
				else if ( lhs.edge1 == rhs.edge1 )
					if ( lhs.edge2 < rhs.edge2 )
						return true;
					else
						if ( lhs.edge2 == rhs.edge2)
							return ( lhs.orientation < rhs.orientation);
		return false;
	}

};

typedef std::map< Base_pair_type, Pairing_template_list_RNA > Pairing_template_map_RNA;

namespace Pairing_template_map_RNA_ns {
	Pairing_template_map_RNA pairing_template_map_RNA;
}

/////////////////////////////////////////////////////////////////////////////
inline
float
arccos(
	float const x
)
{
	return std::acos( numeric::sin_cos_range( x ) );
}


/////////////////////////////////////////////////////////////////////////////
// Following atoms are chosen so that they show up sequentially
// in atom trees, even when they are downstream of a jump.
// They're basically the atoms of the Watson-Crick edge...
//
// Note: In future versions, might be worth saving *all* base heavy atoms,
//    and to define jumps, set up a mini-pose with two bases, setup atom_tree,
//    then appropriate jump.
//
/////////////////////////////////////////////////////////////////////////////
void
rna_basepair_jump_atoms( int const res,
												 int & atom1, int & atom2, int & atom3){
	using namespace param_aa;
	assert( is_RNA( res ));

	if ( res == na_rad ){
		atom1 = 18; // C6
		atom2 = 13; // N1
		atom3 = 14; // C2
	}

	if ( res == na_rcy ){
		atom1 = 17; // C4
		atom2 = 16; // N3
		atom3 = 14; // C2
	}

	if ( res == na_rgu ){
		atom1 = 19; // C6
		atom2 = 13; // N1
		atom3 = 14; // C2
	}

	if ( res == na_ura ){
		atom1 = 17; // C4
		atom2 = 16; // N3
		atom3 = 14; // C2
	}
}

/////////////////////////////////////////////////////////////////////////////
char get_edge_from_num( int const num ) {
	using namespace rna_scoring;
	if (num == WATSON_CRICK) return 'W';
	if (num == HOOGSTEEN)    return 'H';
	if (num == SUGAR)        return 'S';
	return 'X';
}

/////////////////////////////////////////////////////////////////////////////
char get_orientation_from_num( int const num ) {
	using namespace rna_scoring;
	if (num == 1) return 'A';
	if (num == 2) return 'P';
	return 'X';
}

/////////////////////////////////////////////////////////////////////////////
void
get_rna_basepairs( pose_ns::Pose & pose){
	using namespace param_aa;
	using namespace numeric;
	using namespace kin;
	using namespace rna_scoring;
  using numeric::conversions::degrees;

	//base pair list ... useful for input to jumping.
	std::string prefix = stringafteroption("prefix","");
	std::ofstream dataout  ( (prefix+"basepairs.pdat").c_str() );

	int const total_residue = pose.total_residue();

	Energy_base_pair_list  scored_base_pair_list;
	Energy_base_stack_list scored_base_stack_list;

	//Dummy variables...
	FArray2D_bool scored_base_pair( total_residue, total_residue );
	float rna_bs_score,	rna_bp_w_score,	rna_bp_h_score,	rna_bp_s_score,	rna_axis_score;
	float rna_stagger_score, rna_bulge_score, rna_contact_score,	rna_long_range_contact_score;
	FArray2D_bool edge_is_base_pairing( total_residue, rna_scoring::NUM_EDGES, false );

	eval_rna_score( pose, rna_bs_score, rna_bp_w_score, rna_bp_h_score,
									rna_bp_s_score, rna_axis_score, rna_stagger_score, rna_bulge_score,
									rna_contact_score, rna_long_range_contact_score,
									scored_base_pair,
									edge_is_base_pairing,
									scored_base_pair_list, scored_base_stack_list);

	//Now cycle through list and spit out the jumps...
	FArray3D_float const & full_coord( pose.full_coord() );

	for ( Energy_base_pair_list::const_iterator it = scored_base_pair_list.begin();
				it != scored_base_pair_list.end(); ++it ){
		Base_pair const base_pair = it->second;

		int const i = base_pair.res1;
		int const k = base_pair.edge1;

		int const j = base_pair.res2;
		int const m = base_pair.edge2;

		int const res_i = pose.res( i );
		int const res_j = pose.res( j );

		int atom1_i( 0 ), atom2_i( 0 ), atom3_i( 0 );
		int atom1_j( 0 ), atom2_j( 0 ), atom3_j( 0 );
		rna_basepair_jump_atoms( res_i, atom1_i, atom2_i, atom3_i);
		rna_basepair_jump_atoms( res_j, atom1_j, atom2_j, atom3_j);

		//Almost there, still need the orientation.
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );
    Stub stub_i = get_base_coordinate_system(res_i, full_coord(1, 1, i), centroid_i );
    xyzMatrix_float const M_i( stub_i.M );
    xyzVector_float const z_i = M_i.col_z();

		xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );
		Stub stub_j = get_base_coordinate_system(res_j, full_coord(1, 1, j), centroid_j );
		xyzMatrix_float const M_j( stub_j.M );
		xyzVector_float const z_j = M_j.col_z();

		float const theta = degrees( arccos( dot_product( z_i, z_j ) ) );
		char const orientation = (theta > 90.0) ? 'A' : 'P';

		char const edge_i = get_edge_from_num( k );
		char const edge_j = get_edge_from_num( m );

		std::cout << "PAIR " <<
			I(5, i) << ' ' << edge_i << ' ' <<
			I(5, j) << ' ' << edge_j << "   " <<
			orientation << "   " <<
			aa_name1( res_i ) << ' ' << aa_name1( res_j ) << "    " <<
			full_coord(1,atom1_i,i) << ' ' << full_coord(2,atom1_i,i) << ' ' << full_coord(3,atom1_i,i) << ' ' <<
			full_coord(1,atom2_i,i) << ' ' << full_coord(2,atom2_i,i) << ' ' << full_coord(3,atom2_i,i) << ' ' <<
			full_coord(1,atom3_i,i) << ' ' << full_coord(2,atom3_i,i) << ' ' << full_coord(3,atom3_i,i) << ' ' <<
			full_coord(1,atom1_j,j) << ' ' << full_coord(2,atom1_j,j) << ' ' << full_coord(3,atom1_j,j) << ' ' <<
			full_coord(1,atom2_j,j) << ' ' << full_coord(2,atom2_j,j) << ' ' << full_coord(3,atom2_j,j) << ' ' <<
			full_coord(1,atom3_j,j) << ' ' << full_coord(2,atom3_j,j) << ' ' << full_coord(3,atom3_j,j) << ' ' <<
			std::endl;


		dataout <<
			I(5, i) << ' ' <<
			I(5, j) << ' ' <<
			"  0 " << ' ' <<
			edge_i << ' ' <<
			edge_j << "   " <<
			orientation <<
			std::endl;

	}

	dataout.close();

}

/////////////////////////////////////////////////////////////////////////////
void
make_rna_basepair_database(){
	using namespace pose_ns;

	Pose fragments_pose;

	pose_from_pdb( fragments_pose, stringafteroption("fragments"),
								 true, false, true );

	get_rna_basepairs( fragments_pose );

}

///////////////////////////////////////////////////////////////////////////////
bool is_canonical( int const res1, int const res2 )
{
	using namespace param_aa;
	return ( (res1 == na_rad && res2 == na_ura ) ||
					 (res2 == na_rad && res1 == na_ura ) ||
					 (res1 == na_rgu && res2 == na_rcy ) ||
					 (res2 == na_rgu && res1 == na_rcy ) ||
					 (res1 == na_rgu && res2 == na_ura ) ||
					 (res2 == na_rgu && res1 == na_ura ) ) ;
}


///////////////////////////////////////////////////////////////////////////////
void
save_in_pairing_template_map( FArray2D_float const & Epos1, FArray2D_float const & Epos2,
															int const reschar1, int const reschar2,
															char const edgechar1, char const edgechar2,
															char const orientation,
															Pairing_template_map_RNA & pairing_template_map_RNA )
{

	// fill in a new base-pairing template
	Pairing_template_RNA t( Epos1, Epos2);

	//	Base_pair_type base_pair_type( reschar1, reschar2, edgechar1, edgechar2, orientation);
	//	pairing_template_map_RNA[ base_pair_type ].push_back( t );

	// Save to template lists with wildcards ('X') too.
	for ( int i = 0; i <= 1; i++ ) {

		char const edgechar1_temp = i ? edgechar1:'X';

		for ( int j = 0; j <= 1; j++ ) {

			char const edgechar2_temp = j ? edgechar2:'X';

			for ( int k = 0; k <= 1; k++ ) {

				char const orientation_temp = k ? orientation:'X';

				Base_pair_type base_pair_type( reschar1, reschar2, edgechar1_temp, edgechar2_temp, orientation_temp);
				pairing_template_map_RNA[ base_pair_type ].push_back( t );

			}
		}
	}

}


///////////////////////////////////////////////////////////////////////////////
void
read_jump_templates_RNA()
{
	static bool init ( false );
	if ( init ) return;
	init = true;

	using namespace Pairing_template_map_RNA_ns;
	using namespace param_aa;

	utility::io::izstream & data( open_data_file( "jump_templates_RNA_basepairs_v2.dat" ) );

	std::string line;
	std::string tag,filename;
	int pos1,pos2;
	FArray2D_float Epos1(3,param::MAX_POS), Epos2(3,param::MAX_POS);
	char reschar1,reschar2,edgechar1,edgechar2,orientation;
	float mx_dist;

	while ( getline( data,line ) ) {
		std::istringstream is( line );
		//This is a little hacky; Phil's jump-defining routine assumes
		// that coordinates are in 1,2,4 positions like in proteins.
		is >> tag >>
			pos1 >> edgechar1 >> pos2 >> edgechar2 >>
			orientation >> reschar1 >> reschar2 >>
			Epos1(1,1) >> Epos1(2,1) >> Epos1(3,1) >>
			Epos1(1,2) >> Epos1(2,2) >> Epos1(3,2) >>
			Epos1(1,4) >> Epos1(2,4) >> Epos1(3,4) >>
			Epos2(1,1) >> Epos2(2,1) >> Epos2(3,1) >>
			Epos2(1,2) >> Epos2(2,2) >> Epos2(3,2) >>
			Epos2(1,4) >> Epos2(2,4) >> Epos2(3,4);
		if ( is.fail() || tag != "PAIR" ) continue;

		numeric::xyzVector_float p1( & Epos1(1,2) );
		numeric::xyzVector_float p2( & Epos2(1,2) );
		mx_dist = (p1 - p2).length();

		save_in_pairing_template_map( Epos1, Epos2,
																	reschar1, reschar2, edgechar1, edgechar2, orientation,
																	pairing_template_map_RNA );

		save_in_pairing_template_map( Epos2, Epos1,
																	reschar2, reschar1, edgechar2, edgechar1, orientation,
																	pairing_template_map_RNA );

		//std::cout << "new pairing: " << filename << ' ' << o_key << ' ' <<
		//	p_key << std::endl;

	}
}




/////////////////////////////////////////////////////////////////////////////
pose_ns::Jump
get_random_base_pair_jump(
		char const aa1,
		char const aa2,
		char const edge1,
		char const edge2,
		char const orientation,
		bool & success
){

	read_jump_templates_RNA();

	// key for looking up the template geometry:
	Base_pair_type key( aa1, aa2, edge1, edge2, orientation);

	const Pairing_template_list_RNA & templates
		( Pairing_template_map_RNA_ns::pairing_template_map_RNA.find( key )->second );

	const int ntemplates ( templates.size() );

	if (ntemplates < 1){

		std::cout << "Can't seem to find a pairing inside database with aa1: " <<  aa1 << " aa2: " << aa2 << " edge1: " << edge1 << " edge2: " << edge2 << " orientation: " << orientation << std::endl;

		std::cerr << "Can't seem to find a pairing inside database with aa1: " <<  aa1 << " aa2: " << aa2 << " edge1: " << edge1 << " edge2: " << edge2 << " orientation: " << orientation << std::endl;

		//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		success = false;
		return Jump(); //default garbage jump.

	}

	int const index( static_cast<int>( ran3() * ntemplates ) );
	const Pairing_template_RNA & t ( templates[ index ] );
	success = true;

	return t.jump;
}

/////////////////////////////////////////////////////////////////////////////
void
add_new_RNA_jump(
								 pose_ns::Pose & pose,
								 const pairing_RNA_ns::Pairing_list_RNA & RNA_pairings,
								 int const which_jump,
								 bool & success )
{
	using namespace param_aa;

	const Fold_tree & fold_tree( pose.fold_tree() );

	int const jump_pos1( fold_tree.upstream_jump_residue( which_jump ) );
	int const jump_pos2( fold_tree.downstream_jump_residue( which_jump ) );

	pairing_RNA_ns::Pairing_RNA pairing = RNA_pairings[which_jump-1];
	char e1('W') ,e2('W'), o('A');
	if (pairing.pos1 == jump_pos1){
		e1 = pairing.edge1;
		e2 = pairing.edge2;
	} else {
		assert( pairing.pos2 = jump_pos2 );
		e1 = pairing.edge2;
		e2 = pairing.edge1;
	}

	o  = pairing.orientation;

	Jump const new_jump = get_random_base_pair_jump( aa_name1(pose.res(jump_pos1)),
																									 aa_name1(pose.res(jump_pos2)),
																									 e1, e2, o, success );

	if (!success) return; //sshh... don't do anything.

	//		std::cout << " PUTTING IN NEW JUMP: " << new_jump << std::endl;
	pose.set_jump( which_jump, new_jump );
}

/////////////////////////////////////////////////////////////////////////////
void
insert_base_pair_jumps( pose_ns::Pose & pose, const pairing_RNA_ns::Pairing_list_RNA & RNA_pairings, bool & success )
{

	const Fold_tree & fold_tree( pose.fold_tree() );
	const int num_jump( fold_tree.get_num_jump() );

	for (int i = 1; i <= num_jump; i++ ){

		add_new_RNA_jump( pose, RNA_pairings, i, success );

		if (!success) return;

	}

}

///////////////////////////////////////////////////////////////////////////////
// File is currently assumed to have following form:
//
//  <jump pos1> <jump pos2> <cutpoint>
//
// In the future, shouldn't be hard to randomly choose cutpoint, e.g.
// based on an input loop fraction estimate.
//
const pairing_RNA_ns::Pairing_list_RNA &
read_pairing_list_RNA_from_file()
{
	static bool init ( false );
	static pairing_RNA_ns::Pairing_list_RNA pairings;

	if (!init ) {
		// read the pairing file
		init = true;

		int a,b,c;
		char e1,e2,o;
		std::string pairing_file;

		stringafteroption("pairing_file", "nonexistent_file", pairing_file);
		utility::io::izstream pairing_stream( pairing_file );
		if ( !pairing_stream ) {
			pairing_stream.close();
			return pairings; //Just an empty list
		}

		std::string line;
		while ( getline( pairing_stream, line ) ) {
			std::istringstream line_stream( line );
			line_stream >> a >> b ;
			if ( line_stream.fail() ) {
				std::cout << "Parse error!!!" << a << ' ' << b << std::endl;
			}

			line_stream >> c ;
			if ( line_stream.fail() ) {
				c = 0;
			}

			line_stream >> e1 >> e2 >> o;
			if ( line_stream.fail() ) {
				e1 = 'W';
				e2 = 'W';
				o = 'A';
			}

			pairing_RNA_ns::Pairing_RNA p;

			if ( a < b ) {
				p.pos1 = a;
				p.pos2 = b;
			} else {
				p.pos1 = b;
				p.pos2 = a;
			}

			if (a == b) {
				std::cout << "Can't base pair a residue with itself " << a << " " << b << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			p.cutpoint = c;

			p.edge1 = e1;
			p.edge2 = e2;
			p.orientation = o;

			pairings.push_back( p );
		}

		pairing_stream.close();
	} // if (!init )

	return pairings;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// Barcode stuff.
///////////////////////////////////////////////////////////////////////////////
namespace barcode_pairings_RNA{
	pairing_RNA_ns::Pairing_list_RNA barcode_pairing_list_RNA;
}

///////////////////////////////////////////////////////////////////////////////
const pairing_RNA_ns::Pairing_list_RNA & get_barcode_pairing_list_RNA() {
	using namespace barcode_pairings_RNA;
	return barcode_pairing_list_RNA;
}

///////////////////////////////////////////////////////////////////////////////
void
clear_barcode_pairing_list_RNA(){
	using namespace barcode_pairings_RNA;

	barcode_pairing_list_RNA.clear();

	// If there's a pairing_file specifed, might as well copy jumps from there.
	barcode_pairing_list_RNA = read_pairing_list_RNA_from_file();

}

///////////////////////////////////////////////////////////////////////////////
void
add_to_barcode_pairing_list_RNA( int const res1, int const res2, int const cutpoint,
																 char const edge1, char const edge2, char const orientation ){
	using namespace pairing_RNA_ns;
	using namespace barcode_pairings_RNA;

	Pairing_RNA pairing;

	// Yea, this might be a little more concise if I use a constructor.
	pairing.pos1 = res1;
	pairing.pos2 = res2;
	pairing.cutpoint = cutpoint;
	pairing.edge1 = edge1;
	pairing.edge2 = edge2;
	pairing.orientation = orientation;

	barcode_pairing_list_RNA.push_back( pairing );

}

///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
const pairing_RNA_ns::Pairing_list_RNA & current_pairing_list_RNA(){

	using namespace barcode_pairings_RNA; // This is in force_barcode.cc

	static const bool jumps_from_barcode = truefalseoption( "jumps_from_barcode" );

	// Two modes.
	//  (1) Pairings (and possible cutpoints) specified in a "pairing_file" by user.
	//  (2) Pairings specified by a barcode file.
	// In principle we don't need the first pathway, since its encompassed by the latter.
	// Maybe we can deprecate (1) over time...

	if ( jumps_from_barcode ){
		return get_barcode_pairing_list_RNA(); // lives in force_barcode.cc
	} else {
		const pairing_RNA_ns::Pairing_list_RNA & pairing_list = read_pairing_list_RNA_from_file();

		//		if (pairing_list.size() < 1 ){
		//			std::cout << "Had some trouble with RNA pairings file!!! " << std::endl;
		//			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		//		}

		return pairing_list;
	}

}

/////////////////////////////////////////////////////////////////////////////
void
pose_from_random_base_pairings( pose_ns::Pose & pose, bool & success ){
	using namespace pairing_RNA_ns;

	Fold_tree f;
	int const nres = pose.total_residue();
	f.simple_tree( nres );

	// Need this to be a global so that its easy to
	// change jumps during the run?
	const Pairing_list_RNA & RNA_pairings( current_pairing_list_RNA());
	int num_jumps( RNA_pairings.size() );
	FArray2D_int jump_points( 2, num_jumps);
	FArray1D_float cut_bias( nres, 1.0 );

	std::vector< int > obligate_cut_points;

	for (int n = 1; n <= num_jumps; n++ ){
		jump_points(1, n) = RNA_pairings[n-1].pos1;
		jump_points(2, n) = RNA_pairings[n-1].pos2;
		//		cutpoints( n ) = RNA_pairings[n-1].cutpoint;

		int const cut_point =  abs( RNA_pairings[n-1].cutpoint );
		if (cut_point != 0) obligate_cut_points.push_back( cut_point );
	}

	//	f.tree_from_jumps_and_cuts( nres, num_jumps, jump_points, cutpoints, true /* verbose */);
	f.random_tree_from_jump_points( nres, num_jumps, jump_points, obligate_cut_points, cut_bias );

	pose.set_fold_tree( f );

	insert_base_pair_jumps( pose, RNA_pairings, success );

	if (!success) return;
}


//////////////////////////////////////////////////////////////////////////////////
void
pose_from_random_base_pairings( pose_ns::Pose & pose ){

	static const bool jumps_from_barcode = truefalseoption( "jumps_from_barcode" );

	int MAX_TRIES( 1 );
	if (jumps_from_barcode) MAX_TRIES = 10000; // Try several times.

	bool success( false );
	int count( 0 );

	while( !success && count++ < MAX_TRIES) {
		pose_from_random_base_pairings( pose, success );
		if (!success) barcode_initialize_decoy(); //try another barcode.
	}

	if (!success){
		std::cout << "Tried to set up RNA jumps " << MAX_TRIES << " times, but failed!! " << std::endl;
		std::cerr << "Tried to set up RNA jumps " << MAX_TRIES << " times, but failed!! " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}

//////////////////////////////////////////////////////////////////////////////////
// A move, for use during ab initio Monte Carlo
bool
random_jump_change( pose_ns::Pose & pose ){
	using namespace pairing_RNA_ns;

	// Any jumps in here to tweak?
	int const num_jump = pose.fold_tree().get_num_jump();
	if (num_jump == 0) return false;

	const Pairing_list_RNA & RNA_pairings( current_pairing_list_RNA());

	// Randomly pick one.
	int const which_jump ( static_cast<int>( ran3() * num_jump ) + 1 );

	// Tweak it.
	bool success( false );
	add_new_RNA_jump( pose, RNA_pairings, which_jump, success );

	return success;
}

///////////////////////////////////////////////////////////////////////////////
void
random_jump_trial( pose_ns::Pose & pose, pose_ns::Monte_carlo & mc, std::string const trial_type ) {

	bool success;
	success = random_jump_change( pose );
	if (success) mc.boltzmann( pose, trial_type+"-jump" );

}



///////////////////////////////////////////////////////////////////////////////
// guesses the cutpoints to ignore from the jump pattern.
// not exacty the best idea.
void
prepare_cut_weight_for_pose(
														pose_ns::Pose & pose,
														pose_ns::Score_weight_map & rna_weight_map )
{
	pose_ns::Fold_tree const f ( pose.fold_tree() );
	int num_fold_tree_cutpoint;
	FArray1D_int const & fold_tree_cutpoint
		( f.get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );

	FArray1D_float cut_weight( num_fold_tree_cutpoint, 1.0f );

	for (int i = 1; i <= num_fold_tree_cutpoint; i++){
		if ( std::abs( f.upstream_jump_residue(i) -
									 f.downstream_jump_residue(i) ) == 1 ) {
			std::cout << "Turning off chainbreak score at cutpoint " << ":  " << f.upstream_jump_residue(i) <<  " " <<
				f.downstream_jump_residue(i) << std::endl;

			for (int j = 1; j <= num_fold_tree_cutpoint; j++){
				if (fold_tree_cutpoint(j) == min( f.upstream_jump_residue(i), f.downstream_jump_residue(i) ) ) {
					cut_weight( j ) = 0.0;
				}
			}
		}
	}

	rna_weight_map.set_1D_weight( pose_ns::CUT_WEIGHT, cut_weight );

}


///////////////////////////////////////////////////////////////////////////////
// uses info from user-specified jump file to define which cuts require
// a chainbreak score.
void
prepare_cut_weight_for_pose_from_user_pairings(
			pose_ns::Pose & pose,
			pose_ns::Score_weight_map & rna_weight_map )
{
	using namespace pairing_RNA_ns;

	//Cuts from fold tree?
	pose_ns::Fold_tree const f ( pose.fold_tree() );
	int num_fold_tree_cutpoint;
	FArray1D_int const & fold_tree_cutpoint
		( f.get_fold_tree_cutpoint( num_fold_tree_cutpoint ) );

	//Cuts from user specified...
	const Pairing_list_RNA & RNA_pairings( current_pairing_list_RNA());
	int num_jumps( RNA_pairings.size() );

	assert( num_jumps == num_fold_tree_cutpoint );

	FArray1D_float cut_weight( num_jumps, 1.0f );

	// If cutpoint is explicitly specified, don't define a chainbreak weight across it!!!
	// If its zero or negative
	for (int n = 1; n <= num_jumps; n++ ){
		int const cut_point =  RNA_pairings[n-1].cutpoint;

		if (cut_point > 0)	{
			//user specified that this is a real cutpoint:
			for (int j = 1; j <= num_fold_tree_cutpoint; j++){
				if (fold_tree_cutpoint(j) == cut_point ) {
					cut_weight( j ) = 0.0;
				}
			}
		}

	}

	rna_weight_map.set_1D_weight( pose_ns::CUT_WEIGHT, cut_weight );


}
