// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $



// Rosetta Headers
#include "after_opts.h"
#include "dna_ns.h"
#include "pose_rna_ns.h"
#include "param_aa.h"
#include "param.h"
#include "read_paths.h" //for rna_basepair_xy.dat

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.i.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

namespace rna_scoring {
	int const NUM_RNA_VDW_ATOMS = 9;
	FArray4D_float rna_vdw_parameter( NUM_RNA_VDW_ATOMS, NUM_RNA_VDW_ATOMS, 30, 30, 0.0f );
	FArray2D_int   rna_vdw_atom( NUM_RNA_VDW_ATOMS, 30, 0 );
	float rna_vdw_scale = 1.0;
	int num_rna_vdw_atoms_check = 9;

	int const NUM_EDGES = 3; //Watson-Crick, Hoogsteen, Sugar.

	FArray5D_float  rna_basepair_xy( 10, 10, 2, 4, 4, 0.0);
	float rna_basepair_radius_cutoff  = 8.0;
	float rna_basepair_stagger_cutoff = 3.0;

	//craziness! map orientation, base identity 1 (a/c/g/u), base identity 2 (a/c/g/u)--> potential.
	std::map< rnaBasepairKey, FArray3D_float > rna_basepair_rho_omega1_omega2;

	FArray1D_float rna_o2star_potential( 20, 0.0);
	FArray1D_float rna_o2star_weight( 6, 0.0 );

	FArray1D_float rna_phosphate_potential( 20, 0.0);
	FArray1D_float rna_phosphate_weight( 6, 0.0 );

	FArray1D_float rna_axis( 21, 0.0);
	FArray1D_float rna_stagger( 11, 0.0);

	int const num_RNA_backbone_oxygen_atoms( 6 );
	FArray1D_int RNA_backbone_oxygen_atom( 6, 0 );
	FArray4D_float rna_nonbasebase_xy( 16, 16, 4, 6, 0.0 );

	bool rna_fullatom = false; //Switch for count_pair.h and hbonds.cc
	bool count_pair_ignore_neighbor_backbone = false;

	float const o2star_torsion_offset = -30.0;

}

////////////////////////////////////////////////////////////////////////////////////////
void initialize_vdw_rna_parameter(){
	using namespace rna_scoring;

	std::string filename = stringafteroption( "rna_vdw_parameter","rna_vdw_parameter.dat");

	// open file
  utility::io::izstream & data_stream( open_data_file(filename) );

	if ( !data_stream ) {
		std::cerr << "Can't find specified basepair potential file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading basepair x-y potential file: " << filename << std::endl;;
	// read data
	int res1, res2, xbin, ybin;
	float vdw_parameter;;
	while ( data_stream >> xbin ) {
		data_stream >> ybin >> res1 >> res2 >> vdw_parameter >> skip ;
		rna_vdw_parameter( xbin, ybin, res1 , res2 ) = vdw_parameter;
	}

	//close file
	data_stream.close();
}

////////////////////////////////////////////////////////////////////////////////////////
void initialize_rna_basepair_xy(){
	using namespace rna_scoring;

	std::string filename = stringafteroption( "rna_base_pair_xy","rna_base_pair_xy.dat");

	// open file
  utility::io::izstream & data_stream( open_data_file(filename) );

	if ( !data_stream ) {
		std::cerr << "Can't find specified basepair potential file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading basepair x-y potential file: " << filename << std::endl;;
	// read data
	int res1, res2, xbin, ybin, direction;
	float potential;
	while ( data_stream >> xbin ) {
		data_stream >> ybin >> res1 >> res2 >> direction >> potential >> skip ;
		rna_basepair_xy( xbin, ybin, res1 , res2, direction) = potential;
	}

	//close file
	data_stream.close();
}

////////////////////////////////////////////////////////////////////////////////////////
void initialize_rna_basepair_rho_omega1_omega2(){
	using namespace rna_scoring;

	std::string filename = stringafteroption( "rna_base_pair_rho_omega1_omega2","rna_base_pair_rho_omega1_omega2.dat");

	// open file
  utility::io::izstream & data_stream( open_data_file(filename) );

	if ( !data_stream ) {
		std::cerr << "Can't find specified rho/omega1/omega2 potential file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading basepair rho/omega1/omega2 potential file: " << filename << std::endl;;
	// read data
	int res1, res2, r_bin, omega1_bin, omega2_bin, obin;
	float potential;
	while ( data_stream >> r_bin ) {
		data_stream >> omega1_bin >> omega2_bin >> obin >> res1 >> res2 >> potential >> skip ;

		rnaBasepairKey which_potential = std::make_pair( obin, std::make_pair(res1, res2 ) );

		if ( rna_basepair_rho_omega1_omega2.count( which_potential) == 0 ) {
			FArray3D_float * new_potential = new FArray3D_float(8, 36, 36, 0.0);
			rna_basepair_rho_omega1_omega2[ which_potential ] = *new_potential;
		}

		rna_basepair_rho_omega1_omega2[ which_potential ]( r_bin, omega1_bin, omega2_bin ) = potential;

	}

	//close file
	data_stream.close();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
initialize_RNA_backbone_oxygen_atoms(){

	using namespace rna_scoring;
	using namespace rna_variables;

	RNA_backbone_oxygen_atom( 1 ) =  o1p;
	RNA_backbone_oxygen_atom( 2 ) =  o2p;
	RNA_backbone_oxygen_atom( 3 ) =  o5star;
	RNA_backbone_oxygen_atom( 4 ) =  o4star;
	RNA_backbone_oxygen_atom( 5 ) =  o3star;
	RNA_backbone_oxygen_atom( 6 ) =  o2star;

}

////////////////////////////////////////////////////////////////////////////////////////
void  initialize_rna_nonbasebase_xy(){
	using namespace rna_scoring;

	std::string filename = stringafteroption( "rna_nonbasebase_xy","rna_nonbasebase_xy.dat");

	// open file
  utility::io::izstream & data_stream( open_data_file(filename) );

	if ( !data_stream ) {
		std::cerr << "Can't find specified non-base-base potential file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading non-base-base x-y potential file: " << filename << std::endl;;
	// read data
	int res1, xbin, ybin, atomindex;
	float potential;
	while ( data_stream >> xbin ) {
		data_stream >> ybin >> res1 >> atomindex >> potential >> skip ;
		rna_nonbasebase_xy( xbin, ybin, res1 , atomindex) = potential;
	}

	//close file
	data_stream.close();
}


void
initialize_rna_o2star_score()
{
	using namespace rna_scoring;

	std::string filename = stringafteroption( "rna_o2star_score","rna_o2star_score.dat");

	// open file
  utility::io::izstream & data_stream( open_data_file(filename) );

	if ( !data_stream ) {
		std::cerr << "Can't find specified o2star potential file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		return;
	}

	std::cout << "Reading o2star potential file: " << filename << std::endl;;
	// read data
	int rbin;
	float r1,r2,potential;
	while ( data_stream >> r1 ) {
		data_stream >> r2 >> rbin >> potential >> skip ;
		rna_o2star_potential( rbin ) = potential;
	}

	//close file
	data_stream.close();
}

////////////////////////////////////////////////////////////////////////////////////////
void initialize_rna_axis(){
	using namespace rna_scoring;
   rna_axis(  1 ) = -3.319;
   rna_axis(  2 ) = -0.938;
   rna_axis(  3 ) = 0.000;
   rna_axis(  4 ) = 0.000;
   rna_axis(  5 ) = 0.000;
   rna_axis(  6 ) = -0.309;
   rna_axis(  7 ) = -0.748;
   rna_axis(  8 ) = -1.559;
   rna_axis(  9 ) = -2.576;
   rna_axis( 10 ) = -3.478;
   rna_axis( 11 ) = -3.529;
}


////////////////////////////////////////////////////////////////////////////////////////
void initialize_rna_stagger(){
	using namespace rna_scoring;
   rna_stagger(  1 ) = 0.000;
   rna_stagger(  2 ) = 0.000;
   rna_stagger(  3 ) = -0.062;
   rna_stagger(  4 ) = -1.174;
   rna_stagger(  5 ) = -2.085;
   rna_stagger(  6 ) = -2.897;
   rna_stagger(  7 ) = -2.496;
   rna_stagger(  8 ) = -2.253;
   rna_stagger(  9 ) = -1.156;
   rna_stagger( 10 ) = -0.461;
   rna_stagger( 11 ) = -0.354;
}

////////////////////////////////////////////////////////////////////////////////////////
void
initialize_rna_vdw_atoms(){
  using namespace param_aa;
  using namespace rna_scoring;

  // Following set has four dummy atoms on backbone, and one on base.
  // The base atom is the furthest from the sugar, and is
  // the same as the "centroid" atom that Jim and others used for DNA.
  rna_vdw_atom(1, na_rad) =  1; // P
  rna_vdw_atom(2, na_rad) =  5; // C5*
  rna_vdw_atom(3, na_rad) = 12; // C1*
  rna_vdw_atom(4, na_rad) =  8; // C3*
  rna_vdw_atom(5, na_rad) = 19; // N6
  rna_vdw_atom(6, na_rad) = 16; // N3
  rna_vdw_atom(7, na_rad) = 21; // N7
  rna_vdw_atom(8, na_rad) = 13; // N1
  rna_vdw_atom(9, na_rad) = 11; // O2*

  rna_vdw_atom(1, na_rcy) =  1; // P
  rna_vdw_atom(2, na_rcy) =  5; // C5*
  rna_vdw_atom(3, na_rcy) = 12; // C1*
  rna_vdw_atom(4, na_rcy) =  8; // C3*
  rna_vdw_atom(5, na_rcy) = 18; // N4
  rna_vdw_atom(6, na_rcy) = 15; // O2
  rna_vdw_atom(7, na_rcy) = 20; // C6
  rna_vdw_atom(8, na_rcy) = 13; // N1
  rna_vdw_atom(9, na_rcy) = 11; // C2*

  rna_vdw_atom(1, na_rgu) =  1; // P
  rna_vdw_atom(2, na_rgu) =  5; // C5*
  rna_vdw_atom(3, na_rgu) = 12; // C1*
  rna_vdw_atom(4, na_rgu) =  8; // C3*
  rna_vdw_atom(5, na_rgu) = 20; // O6
  rna_vdw_atom(6, na_rgu) = 15; // N2
  rna_vdw_atom(7, na_rgu) = 21; // N7
  rna_vdw_atom(8, na_rgu) = 13; // N1
  rna_vdw_atom(9, na_rgu) = 11; // O2*

  rna_vdw_atom(1, na_ura) =  1; // P
  rna_vdw_atom(2, na_ura) =  5; // C5*
  rna_vdw_atom(3, na_ura) = 12; // C1*
  rna_vdw_atom(4, na_ura) =  8; // C3*
  rna_vdw_atom(5, na_ura) = 18; // O4
  rna_vdw_atom(6, na_ura) = 15; // O2
  rna_vdw_atom(7, na_ura) = 20; // C6
  rna_vdw_atom(8, na_ura) = 13; // N1
  rna_vdw_atom(9, na_ura) = 10; // C2*

  // Following set has two dummy atoms on backbone, and three on base.
//   rna_vdw_atom(1, na_rad) =  1; // P
//   rna_vdw_atom(2, na_rad) =  6; // C4*
//   rna_vdw_atom(3, na_rad) = 13; // N1
//   rna_vdw_atom(4, na_rad) = 15; // N3
//   rna_vdw_atom(5, na_rad) = 20; // N7

//   rna_vdw_atom(1, na_rcy) =  1; // P
//   rna_vdw_atom(2, na_rcy) =  6; // C4*
//   rna_vdw_atom(3, na_rcy) = 15; // O2
//   rna_vdw_atom(4, na_rcy) = 16; // N3
//   rna_vdw_atom(5, na_rcy) = 19; // C5

//   rna_vdw_atom(1, na_rgu) =  1; // Ph
//   rna_vdw_atom(2, na_rgu) =  6; // C4*
//   rna_vdw_atom(3, na_rgu) = 13; // N1
//   rna_vdw_atom(4, na_rgu) = 16; // N3
//   rna_vdw_atom(5, na_rgu) = 21; // N7

//   rna_vdw_atom(1, na_ura) =  1; // P
//   rna_vdw_atom(2, na_ura) =  6; // C4*
//   rna_vdw_atom(3, na_ura) = 15; // O2
//   rna_vdw_atom(4, na_ura) = 16; // N3
//   rna_vdw_atom(5, na_ura) = 19; // C5

  initialize_vdw_rna_parameter();
	intafteroption( "num_rna_vdw_atoms_check", 9,  num_rna_vdw_atoms_check);

  initialize_rna_axis();
  initialize_rna_stagger();
	realafteroption( "rna_vdw_scale", 1.0, rna_vdw_scale );

	realafteroption( "rna_basepair_radius_cutoff",  8.0, rna_basepair_radius_cutoff );
	realafteroption( "rna_basepair_stagger_cutoff", 3.0, rna_basepair_stagger_cutoff );
}

