// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_rna_pdbstats.h"
#include "pose_rna.h"
#include "pose_dna.h"
#include "pose_rna_ns.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_tree_routines.h"
#include "current_pose.h"
#include "dna.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "fragments_pose.h"
#include "force_barcode.h"
#include "fullatom_energies.h"
#include "fullatom_sasa.h"
#include "hbonds.h"
#include "kin_stub.h"
#include "minimize.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "pdbstats.h" //for output_pair_energies
#include "pose.h"
#include "pose_rna_fragments.h" //For fragment scan.
#include "pose_rna_fullatom.h"  //for new chainbreak term. Move it?
#include "pose_rna_jumping.h"   //to guess where to apply chainbreak score or not.
#include "pose_io.h"
#include "read_aaproperties.h"
#include "read_paths.h"
#include "random_numbers.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h"
#include "silent_input.h"

#include "prof.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/StaticIndexRange.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyz.io.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


///////////////////////////////////////////////////////////////////////////////

inline
float
arccos(
	float const x
)
{
	return std::acos( numeric::sin_cos_range( x ) );
}

///////////////////////////////////////////////////////////////////////////////
void
rna_pdb_stats( std::ofstream & outstream, std::string const tag = "" ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace numeric;
  using namespace kin;
  using numeric::conversions::degrees;

	static bool const save_good_doublets_only = truefalseoption("save_good_doublets_only");

	//  float const chainbreak_cutoff ( 7.5 * 7.5 );
  float const dist_cutoff ( 12.0 );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );
    Stub stub_i = get_base_coordinate_system(res_i, full_coord(1, 1, i), centroid_i );
    xyzMatrix_float const M_i( stub_i.M );
    assert( is_orthonormal( M_i, 1e-3) );
    xyzVector_float const x_i = M_i.col_x();
    xyzVector_float const y_i = M_i.col_y();
    xyzVector_float const z_i = M_i.col_z();

			for ( int j = 1; j <= total_residue; j++ ){

      if (i == j) continue;
      int const res_j = res(j);
      if (!is_RNA( res_j)) continue;

      xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );
      xyzVector_float d_ij = centroid_j - centroid_i;
      float const dist_ij = d_ij.length();

      if (dist_ij < dist_cutoff){
				//Actually no need to compute these over and over again. Oh well.
				Stub stub_j = get_base_coordinate_system(res_j, full_coord(1, 1, j), centroid_j );
				xyzMatrix_float const M_j( stub_j.M );
				assert( is_orthonormal( M_j, 1e-3) );
				xyzVector_float const x_j = M_j.col_x();
				xyzVector_float const y_j = M_j.col_y();
				xyzVector_float const z_j = M_j.col_z();

				float const dist_x = dot_product( d_ij, x_i );
				float const dist_y = dot_product( d_ij, y_i );
				float const dist_z = dot_product( d_ij, z_i );

				float projectontox, projectontoy;
				float const theta = degrees( arccos( dot_product( z_i, z_j ) ) );
				projectontox = dot_product( x_i, z_j );
				projectontoy = dot_product( y_i, z_j );
				float const phi   = degrees( std::atan2( projectontoy, projectontox ) );

				projectontox = dot_product( x_i, x_j );
				projectontoy = dot_product( y_i, x_j );
				float const alpha   = degrees( std::atan2( projectontoy, projectontox ) );

				//Perhaps a better set of easy parameters to define base pairing geometry.
				// omega_i, omega_j, d_ij completely define relative base orientations
				// if bases are completely coplanar [z = 0.0; phi = 0 or 180 degrees].
				// Also, omega_i and omega_j should be symmetric.
				xyzVector_float d_ji = -1.0f * d_ij;
				float const dist_x_i = dot_product( d_ij, x_i );
				float const dist_y_i = dot_product( d_ij, y_i );
				float const omega_i   = degrees( std::atan2( dist_y_i, dist_x_i ) );

				float const dist_x_j = dot_product( d_ji, x_j );
				float const dist_y_j = dot_product( d_ji, y_j );
				float const omega_j   = degrees( std::atan2( dist_y_j, dist_x_j ) );

				float const dist_z_j = dot_product( d_ji, z_j );

				float const avg_z = ( abs(dist_z) + abs(dist_z_j) ) / 2.0 ;
				float const rho = std::sqrt( dist_x*dist_x + dist_y*dist_y);

				if ( save_good_doublets_only &&
						 !( avg_z < 2.5 ) &&
						 !( avg_z < 6.0 && rho < 6.0 ) ) continue;

				//	std::cout << " CONTACT: " << aa_name3(res_i) << i << "-" << aa_name3(res_j) << j << "   : " << dist_ij << std::endl;
				outstream <<
					i << " " << j << " " <<
					res_i << " "  << res_j <<  " " <<
					dist_ij << " "  << " " <<
					dist_x << " " << dist_y << " " << dist_z << " " <<
					theta << " " << phi << " " << alpha << " " <<
					omega_i << " " << omega_j << " " << dist_z_j << " " <<
					tag << std::endl;

      }

    }
  }
}

///////////////////////////////////////////////////////////////////////////////
void
rna_nonbase_base_pdb_stats( std::ofstream & outstream, std::string const tag = "" ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace numeric;
  using namespace kin;
  using numeric::conversions::degrees;

	static bool const save_good_doublets_only = truefalseoption("save_good_doublets_only");

	//  float const chainbreak_cutoff ( 7.5 * 7.5 );
  float const dist_cutoff ( 12.0 );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );
    Stub stub_i = get_base_coordinate_system(res_i, full_coord(1, 1, i), centroid_i );
    xyzMatrix_float const M_i( stub_i.M );
    assert( is_orthonormal( M_i, 1e-3) );
    xyzVector_float const x_i = M_i.col_x();
    xyzVector_float const y_i = M_i.col_y();
    xyzVector_float const z_i = M_i.col_z();

		int const num_heavy_atoms = aaproperties_pack::nheavyatoms( res(i), 1);

		for ( int j = 1; j <= total_residue; j++ ){

      int const res_j = res(j);
      if (!is_RNA( res_j)) continue;

			static int const rna_base_start = 13;

			// Go over sugar and phosphate atoms!
			for (int atom_num_j = 1; atom_num_j < rna_base_start; atom_num_j++ ){
				//Go over sugar and phosphate heavy atoms in the second base.
				xyzVector_float const heavy_atom_j( & full_coord( 1, atom_num_j, j ) );

				xyzVector_float d_ij = heavy_atom_j - centroid_i;

				float const dist_ij = d_ij.length();

				if (dist_ij < dist_cutoff){

					float const dist_x = dot_product( d_ij, x_i );
					float const dist_y = dot_product( d_ij, y_i );
					float const dist_z = dot_product( d_ij, z_i );

					if (save_good_doublets_only) {

						if ( abs(dist_z > 3.0) ) continue; // Look for atoms in the base plane

						static float const atom_dist_cutoff = 4.0; //make sure we're in H-bonding distance of some base atom.

						bool found_a_neighbor = false;
						for (int atom_num_i = rna_base_start; atom_num_i <= num_heavy_atoms; atom_num_i++){
							xyzVector_float base_heavy_atom_i( &full_coord(1, atom_num_i, i) );
							if ( (heavy_atom_j - base_heavy_atom_i ).length() < atom_dist_cutoff){
								found_a_neighbor = true;
								break;
							}
						}

						if (!found_a_neighbor) continue;
					}



					outstream <<
						i << " " << j << " " <<
						res_i << " "  << res_j <<  " " <<
						atom_num_j <<  " " <<
						dist_ij << " "  << " " <<
						dist_x << " " << dist_y << " " << dist_z << " " <<
						tag << std::endl;

				}

			}
		}
	}

}

///////////////////////////////////////////////////////////////////////////////////////////
void
rna_backbone_hbond_pdb_stats( std::ofstream & outstream, std::string const tag = "" ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace numeric;
	using namespace rna_scoring;

	static bool init = {false};
	if (!init){
		initialize_RNA_backbone_oxygen_atoms();
		init = true;
	}

  float const dist_cutoff ( 12.0 );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;

		for (int k = 1; k <= num_RNA_backbone_oxygen_atoms; k++ ){

			int const atom_num_i = RNA_backbone_oxygen_atom( k );

			xyzVector_float const heavy_atom_i( & full_coord( 1, atom_num_i, i ) );

			for ( int j = 1; j <= total_residue; j++ ){

				if ( i == j ) continue;

				int const res_j = res(j);
				if (!is_RNA( res_j)) continue;

				for (int m = 1; m <= num_RNA_backbone_oxygen_atoms; m++ ){

					int const atom_num_j = RNA_backbone_oxygen_atom( m );

					xyzVector_float const heavy_atom_j( & full_coord( 1, atom_num_j, j ) );

					xyzVector_float d_ij = heavy_atom_j - heavy_atom_i;

					float const dist_ij = d_ij.length();

					if (dist_ij < dist_cutoff) {

						outstream <<
							i << " " << j << " " <<
							res_i << " "  << res_j <<  " " <<
							atom_num_i <<  " " << atom_num_j << " " <<
							dist_ij  << " " << tag << std::endl;
					}

				} // m
			} // j

		} // k
	} // i

}


///////////////////////////////////////////////////////////////////////////////////////////
void
rna_o2star_point_pdb_stats( std::ofstream & outstream, std::string const tag = "" ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace numeric;
	using namespace rna_variables;
	using namespace aaproperties_pack;

	//	static bool init = {false};
  float const dist_cutoff ( 4.0 ); //distance cutoff for a hydrogen bond.

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;

		int const atom_num_i = o2star;
		xyzVector_float const heavy_atom_i( & full_coord( 1, atom_num_i, i ) );
		xyzVector_float const heavy_atom_base_i ( & full_coord( 1, c2star, i ) );
		xyzVector_float const heavy_atom_base2_i( & full_coord( 1, c1star, i ) );

		for ( int j = 1; j <= total_residue; j++ ){

			if ( i == j ) continue;

			int const res_j = res(j);
			if (!is_RNA( res_j)) continue;

			int const num_heavy_atoms = aaproperties_pack::nheavyatoms( res(j), 1);

			for (int m = 1; m <= num_heavy_atoms; m++ ){

				xyzVector_float const heavy_atom_j( & full_coord( 1, m, j ) );

				xyzVector_float d_ij = heavy_atom_j - heavy_atom_i;

				float const dist_ij = d_ij.length();

				if (dist_ij < dist_cutoff) {
					bool is_donor( false ), is_acceptor( false );

					//Is the other atom a donor?
					int const n_H_polar = nH_polar( res_j, 1);
					for (int count = 1; count <= n_H_polar; count++ ){
						if ( atom_base( Hpos_polar(count, res_j, 1), res_j, 1) == m ){
							is_donor = true;
							break;
						}
					}

					//Is the other atom an acceptor?
					int const n_acceptors = nacceptors(res_j, 1);
					for (int count = 1; count <= n_acceptors; count++ ){
						if ( accpt_pos(count, res_j, 1) == m ){
							is_acceptor = true;
							break;
						}
					}

					//Figure out dihedral angle.
					float const o2star_dihedral (dihedral( heavy_atom_base2_i, heavy_atom_base_i, heavy_atom_i, heavy_atom_j ) );

					outstream <<
						i << " " << j << " " <<
						res_i << " "  << res_j <<  " " <<
						m << " " <<
						is_donor << " " <<
						is_acceptor << " " <<
						o2star_dihedral << " " <<
						dist_ij  << " " << tag << std::endl;

					}

				} // m
			} // j

	} // i

}

///////////////////////////////////////////////////////////////////////////////
bool
rna_count_pair( int const i, int const j, int const k, int const m ){

//Copied from count_pair.h, added in hydrogens
	if ( i == j + 1 &&
			 (m == 6 || m == 10 || m == 27 ) && // C4* or C2*
			 (k == 1 ) ) return false;
	if ( i == j + 1 &&
			 (m == 8 ) && // C3*
			 (k == 1 || k == 2 || k == 3 || k == 4) ) return false;
	if ( i == j + 1 &&
		 (m == 9 ) && // O3*
		 (k == 1 || k == 2 || k == 3 || k == 4 || k == 5) ) return false;
 if ( j == i + 1 &&
			(k == 6 || k == 10 || m == 27 ) && // C4* or C2*
			(m == 1 ) ) return false;
 if ( j == i + 1 &&
		 (k == 8 ) && // C3*
		 (m == 1 || m == 2 || m == 3 || m == 4) ) return false;
 if ( j == i + 1 &&
			(k == 9 ) && // O3*
		 (m == 1 || m == 2 || m == 3 || m == 4 || m == 5) ) return false;

 return true;
}

///////////////////////////////////////////////////////////////////////////////
void
rna_fullatom_pdb_stats( std::ofstream & outstream ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace aaproperties_pack;
  using namespace numeric;
  using namespace kin;
  using numeric::conversions::degrees;

	//  float const chainbreak_cutoff ( 7.5 * 7.5 );
  float const dist_cutoff ( 12.0 );
	float const atom_dist_cutoff( 7.5 );

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );

    for ( int j = 1; j <= total_residue; j++ ){

      if (i == j) continue;
      int const res_j = res(j);
      if (!is_RNA( res_j)) continue;

      xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );
      xyzVector_float d_ij = centroid_j - centroid_i;
      float const dist_ij = d_ij.length();

      if (dist_ij < dist_cutoff){

				//For now, just output atom-atom pairs.
				for ( int k = 1; k <= natoms( res_i, 1); k++ ) {

					xyzVector_float v1( & full_coord(1, k, i ) );

					for ( int m = 1; m <= natoms( res_j, 1); m++ ) {

						if (!rna_count_pair(i,j,k,m))  continue;

						xyzVector_float v2( & full_coord(1, m, j ) );

						float const dist = (v1-v2).length();

						//Later can output coordinates in base frame.
						if ( dist < atom_dist_cutoff ) {
							outstream << i << " " << j << " " << res_i << " " << res_j << " " << k << " " << m << " " <<
								fullatom_type( k, res_i, 1) << " " << fullatom_type( m, res_j, 1 ) << " " << dist << std::endl;
						}

					}
				}
      }

    }
  }
}



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
namespace rna_fullatom_pairterm_score_ns{
	FArray3D_float rna_fullatom_pairterm( 16 , 25, 25 );
}

///////////////////////////////////////////////////////////////////////////////
void
initialize_rna_fullatom_pairterm() {

	using namespace rna_fullatom_pairterm_score_ns;

	static bool init( false );
	if (init) return;
	init = true;

	std::string const filename = stringafteroption( "rna_fullatom_pairterm_file", "rna_fullatom_pairterm.pdat" );
	utility::io::izstream & file_in( open_data_file( filename ) );

	std::string line, tag;
	int i,j;
	float x, val;

	while (  getline( file_in, line) ){

		std::istringstream line_stream( line );

		line_stream >> i >> j >> x >> val;

		int const which_bin = 1 + static_cast<int>( 2*x );

		rna_fullatom_pairterm( which_bin, i, j ) = val;

	} // line_stream

	file_in.close();

}


///////////////////////////////////////////////////////////////////////////////
// Hmmm, probably shouldn't repeat code, see above rna_fullatom_pdbstats.
void
eval_rna_fullatom_pairterm_score( pose_ns::Pose & pose ){

  using namespace param_aa;
  using namespace aaproperties_pack;
  using namespace numeric;
	using namespace rna_fullatom_pairterm_score_ns;

	static bool const rna_fullatom_pairterm_file_defined = truefalseoption("rna_fullatom_pairterm_file");
	if (!rna_fullatom_pairterm_file_defined) return;

	initialize_rna_fullatom_pairterm();

	float rna_fullatom_pairterm_score = 0.0;

  float const dist_cutoff ( 12.0 );
	//	float const atom_dist_cutoff( 7.5 );

	FArray3D_float const & full_coord( pose.full_coord() );
	int const total_residue = pose.total_residue();


  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = pose.res(i);
    if (!is_RNA( res_i)) continue;
    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );

    for ( int j = 1; j <= total_residue; j++ ){

			float pair_score = 0.0;

      if (i == j) continue;
      int const res_j = pose.res(j);
      if (!is_RNA( res_j)) continue;

      xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );
      xyzVector_float d_ij = centroid_j - centroid_i;
      float const dist_ij = d_ij.length();

      if (dist_ij < dist_cutoff){

				//For now, just output atom-atom pairs.
				for ( int k = 1; k <= natoms( res_i, 1); k++ ) {

					xyzVector_float v1( & full_coord(1, k, i ) );

					for ( int m = 1; m <= natoms( res_j, 1); m++ ) {

						if (!rna_count_pair(i,j,k,m))  continue;

						xyzVector_float v2( & full_coord(1, m, j ) );

						float const dist = (v1-v2).length();

						if (dist < 7.5 ){
							int const which_bin = 1 + static_cast<int>( 2 * dist );
							float const contribution = rna_fullatom_pairterm( which_bin,
																																fullatom_type( k, res_i, 1),
																																fullatom_type( m, res_j, 1));
							pair_score += contribution;

							if (i == 4 && j == 9 ) {
								std::cout << "RNA_FA_PAIR_ATOM " << atom_name(k,res_i,1 ) << " - " << atom_name(m,res_j,1) << " : " <<
									contribution << std::endl;
							}
						}

					}
				}
      }

			std::cout << "RNA_FA_PAIR " << i << " " << j << " " << pair_score << std::endl;

			rna_fullatom_pairterm_score += pair_score;
    }
  }

	pose.set_extra_score( "RNA_FA_PAIR", rna_fullatom_pairterm_score );

}

///////////////////////////////////////////////////////////////////////////////
void
rna_vdw_pdb_stats( std::ofstream & outstream ){

  //Everything better be in misc.
  using namespace misc;
  using namespace param_aa;
  using namespace numeric;
  using namespace rna_scoring;

	float const dist_cutoff = 12.0;

	initialize_rna_vdw_atoms();

  for (int i = 1; i <= total_residue; i++ ){

    int const res_i = res(i);
    if (!is_RNA( res_i)) continue;

    xyzVector_float const centroid_i = get_base_centroid( res_i, full_coord(1, 1, i) );

    for ( int j = 1; j <= total_residue; j++ ){

      if (i == j) continue;
      int const res_j = res(j);
      if (!is_RNA( res_j)) continue;

      xyzVector_float const centroid_j = get_base_centroid( res_j, full_coord(1, 1, j) );
      xyzVector_float d_ij = centroid_j - centroid_i;
      float const dist_ij = d_ij.length();

      if (dist_ij < dist_cutoff){

				for (int m = 1; m <= NUM_RNA_VDW_ATOMS; m++) {
					for (int n = 1; n <= NUM_RNA_VDW_ATOMS; n++) {
						int const atomnum_i = rna_vdw_atom(m, res_i);
						int const atomnum_j = rna_vdw_atom(n, res_j);

						xyzVector_float const r_i ( & full_coord(1, atomnum_i, i));
						xyzVector_float const r_j ( & full_coord(1, atomnum_j, j));

						float const dist_mn = (r_i - r_j).length();

						if (dist_mn < 8.0){
							outstream <<
								i << " " << j << " " <<
								res_i << " " << res_j << " " <<
								m << " " <<
								n << " " <<
								dist_mn << std::endl;
						}
					} //n
				}//m

			}//dist_cutoff

    } //j
  } //i
}

///////////////////////////////////////////////////////////////////////////////
void
score_the_pose(
							 pose_ns::Pose & source_pose,
							 pose_ns::Score_weight_map & rna_weight_map
							 )
{
	hbonds::output_hb_stats = true;
	source_pose.score ( rna_weight_map );
	hbonds::output_hb_stats = false;

	//	float const sasa = calc_sasa();
	calc_gsolt( source_pose );

	if (source_pose.native_pose_exists() ){
		calc_rms( source_pose );
		fill_native_base_pair_info( source_pose );
	}

	//useful to save distances used for long range contact info.
	// Saved as "extra scores" inside pose.
	evaluate_long_range_contact_distances( source_pose );

	static bool const output_pair_energy = truefalseoption( "pair_energy_file" );
	if (output_pair_energy) output_pair_energies();

	//temporary -- remove!
	static bool const fullatom_stats = truefalseoption( "fullatom_stats" );
	if ( fullatom_stats )	eval_rna_fullatom_pairterm_score( source_pose );

}


///////////////////////////////////////////////////////////////////////////////
void
read_decoys_silent_or_not( silent_io::Silent_file_data * & decoys_p, std::vector< std::string> & files, bool & silent_input )
{
  using namespace silent_io;

	static bool const silent_input_local = truefalseoption( "silent" );
	silent_input = silent_input_local;

  if (silent_input){
  // read silent file
    decoys_p = new Silent_file_data( stringafteroption("s"), true /*fullatom*/ );
    files = decoys_p->tags();
  } else {
		if (truefalseoption("s")) {
			files.push_back( stringafteroption("s") );
		}

		if (truefalseoption("l")) {
			std::ifstream data( stringafteroption("l").c_str() );
			std::string line;
			while ( getline( data,line ) ) {
				files.push_back(line);
			}
			data.close();
		}

		decoys_p = NULL; //silent file data not used.
	}

}

///////////////////////////////////////////////////////////////////////////////
void
fill_decoy_silent_or_not( pose_ns::Pose & pose, silent_io::Silent_file_data * decoys_p,
													std::string const start_file, bool const silent_input) {

  using namespace silent_io;

	if ( silent_input ){
		Silent_structure const & decoy( decoys_p->get_structure( start_file ) );
		decoy.fill_pose( pose, true );
	} else {
		pose_from_pdb( pose, start_file, true /*fullatom*/, false /*ideal_pose*/, true /*read_all_chains*/ );
	}

}


///////////////////////////////////////////////////////////////////////////////
void
rna_score_test()
{
  using namespace pose_ns;
  using namespace param_aa;
  using namespace silent_io;


	Pose native_pose;
	bool native_pose_exists( false );
	if (truefalseoption("n")) {
		native_pose_exists = true;
		std::string native_file_name = stringafteroption("n","blah.pdb");
		bool success = pose_from_pdb( native_pose, native_file_name,
									 true, false, true );
		if (!success){
			std::cout << "Had trouble with native pdb " << native_file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}


  Silent_file_data * decoys_p;
	std::vector< std::string > files;
  bool silent_input;

	read_decoys_silent_or_not( decoys_p, files, silent_input );

	Score_weight_map rna_weight_map = setup_rna_weight_map();
	rna_weight_map.set_weight( CHAINBREAK, realafteroption("chainbreak_weight",1.0) );

	std::string const scorefilename = stringafteroption("scorefile","rescore.sc");
	Silent_out out( scorefilename );

	for ( std::vector< std::string >::const_iterator file=files.begin();
				file != files.end(); ++file ) {

		// read pose
		Pose pose;

		std::string const start_file( *file);
		std::cout << "Reading in ... " << start_file << std::endl;

		fill_decoy_silent_or_not( pose, decoys_p, start_file, silent_input );

		pose.copy_to_misc(); // need to set total_residue correctly, etc.

		// Rhiju sort-of "hack":
		//Don't want to always do this -- just a check of
		// new, more sophisticated chain closure term.
		//		setup_rna_chainbreak_constraints( pose );

		// Don't want chainbreak score to be calculated at end of helix.
		prepare_cut_weight_for_pose( pose, rna_weight_map );

		//Diagnostics.
		if (native_pose_exists){
			pose.set_native_pose( native_pose );
		}

		barcode_initialize_start( pose.total_residue() );
		barcode_initialize_decoy();

		score_the_pose( pose, rna_weight_map );

		out.write_scores( start_file, pose );

		//would be nice to check "chainbreak" score on the native too.
		if (native_pose_exists)		native_pose.set_fold_tree( pose.fold_tree() );
	}

	//icing on the cake
	//	setup_rna_chainbreak_constraints( native_pose );
	if (native_pose_exists){
		score_the_pose( native_pose, rna_weight_map );
		out.write_scores( "NATIVE", native_pose );
	}

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
pairwise_rmsd_test()
{
  using namespace pose_ns;
  using namespace param_aa;
  using namespace silent_io;

	std::vector< std::string > files;

	Pose native_pose;
	bool native_pose_exists( false );
	if (truefalseoption("n")) {
		native_pose_exists = true;
		std::string native_file_name = stringafteroption("n","blah.pdb");
		bool success = pose_from_pdb( native_pose, native_file_name,
									 true, false, true );
		if (!success){
			std::cout << "Had trouble with native pdb " << native_file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	std::ifstream data( stringafteroption("l").c_str() );
	std::string line;
	while ( getline( data,line ) ) {
		files.push_back(line);
	}
	data.close();

	Score_weight_map rna_weight_map = setup_rna_weight_map();

	std::vector < Pose* > pose_list;

	for ( std::vector< std::string >::const_iterator file=files.begin();
				file != files.end(); ++file ) {

		// read pose
		Pose * source_pose;
		source_pose  = new Pose;

		std::string const start_file( *file);
		std::cout << "Reading in ... " << start_file << std::endl;
		pose_from_pdb( *source_pose, start_file,
									 true, false, true );
		source_pose->copy_to_misc(); // need to set total_residue correctly, etc.

		//Diagnostics.
		if (native_pose_exists){
			source_pose->set_native_pose( native_pose );
			calc_rms( *source_pose );
		}
		pose_list.push_back( source_pose );

	}


	int count( 0 );
	float sum_pair_rms( 0.0 );
	float sum_rms_to_first( 0.0 );
	float sum_rms_to_native( 0.0 );

	int const num_poses = pose_list.size();

	for (int i = 0; i < num_poses; i++){
		Pose * pose1( pose_list[i] );

		sum_rms_to_native += pose1->get_0D_score( RMSD );

		for (int j = 0; j < num_poses; j++){
			if (i==j) continue;
			Pose * pose2( pose_list[j] );

			float const pair_rms = CA_rmsd( *pose1, *pose2 );
			std::cout << "Pairwise rmsd between " << i << " and " << j << " : " << pair_rms << std::endl;
			sum_pair_rms += pair_rms;

			if (i==0 && j>0) sum_rms_to_first += pair_rms;

			count++;

		}
	}

	float const mean_pair_rms = sum_pair_rms/ count;
	float const mean_rms_to_native = sum_rms_to_native / num_poses;
	float const mean_rms_to_first = sum_rms_to_first / (num_poses-1);

	std::cout << "mean_pair_rms: " << mean_pair_rms << std::endl;
	std::cout << "mean_rms_to_native: " << mean_rms_to_native << std::endl;
	std::cout << "mean_rms_to_first: " << mean_rms_to_first << std::endl;

}


///////////////////////////////////////////////////////////////////////////////
//For P4-P6 modeling, test whether P5c ands up in teh "front" or "back" of the structure.
void
p5c_helix_test()
{
  using namespace silent_io;
	using namespace numeric;

	Silent_file_data decoys;
	std::string silent_file_name = stringafteroption("s");
	decoys.read_file( silent_file_name, true /* fullatom */);

  std::vector< std::string > tag_list = decoys.tags();

	int const origin = 38;
	int const x_anchor = 49;
	int const y_anchor = 110;
	//	int const p5c_anchor = 69;

	std::vector<std::string>::const_iterator it;
	for ( it = tag_list.begin(); it !=tag_list.end(); ++it){

		std::string const & tag( *it );

		Silent_structure const & decoy( decoys.get_structure( tag ) );

		xyzVector_float origin_vec( & decoy.coords(1, origin) );
		xyzVector_float x_vec     ( & decoy.coords(1, x_anchor) );
		xyzVector_float y_vec     ( & decoy.coords(1, y_anchor) );

		xyzVector_float p5c_vec( 0.0 );
		int count = 0;
		for (int i = 63; i <= 73; i++ ){
			xyzVector_float p5c_vec_i   ( & decoy.coords(1, i) );
			p5c_vec += p5c_vec_i;
			count++;
		}
		p5c_vec /= count;

		x_vec -= origin_vec;
		y_vec -= origin_vec;
		p5c_vec -= origin_vec;

		xyzVector_float z_vec = cross(x_vec,y_vec).normalized();
		float const height = dot( p5c_vec, z_vec);
		std::cout << "HEIGHT " << tag << " " << height << std::endl;

	}

}

///////////////////////////////////////////////////////////////////////////////
void
rna_rms_pdb_stats( std::ofstream & rna_rms_stats_stream,
									 pose_ns::Pose & pose,
									 pose_ns::Pose & native_pose,
									 std::string tag = "" )
{

	pose.set_native_pose( native_pose );

	float const rms = calc_rms( pose );
	fill_native_base_pair_info( pose );

	rna_rms_stats_stream << rms << " " <<
		pose.get_extra_score( "N_WC" ) << " " <<
		pose.get_extra_score( "N_NWC" ) << " " <<
		pose.get_extra_score( "N_BUL" ) << " " <<
		pose.get_extra_score( "MISS_NBP" ) << " " <<
		tag << std::endl;

}

///////////////////////////////////////////////////////////////////////////////
void
rna_pdb_stats_test()
{
  using namespace pose_ns;
  using namespace param_aa;
	using namespace silent_io;

  Silent_file_data * decoys_p;
	std::vector< std::string > files;
  bool silent_input;

	read_decoys_silent_or_not( decoys_p, files, silent_input );

	Pose native_pose;
	bool native_pose_exists( false );
	if (truefalseoption("n")) {
		native_pose_exists = true;
		std::string native_file_name = stringafteroption("n","blah.pdb");
		bool success = pose_from_pdb( native_pose, native_file_name, true, false, true );
		if (!success){
			std::cout << "Had trouble with pdb " << native_file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	std::string const tag = stringafteroption("o","");

	std::ofstream rna_stats_stream( (tag+"rna_stats.txt").c_str() );
	std::ofstream rna_nonbase_base_stats_stream( (tag+"rna_sugar_phosphate_stats.txt").c_str() );
	std::ofstream rna_backbone_hbond_stats_stream( (tag+"rna_backbone_hbond_stats.txt").c_str() );
	std::ofstream rna_o2star_point_stats_stream( (tag+"rna_o2star_point_stats.txt").c_str() );
	std::ofstream rna_rms_stats_stream( (tag+"rna_rms_stats.txt").c_str() );

	//	std::ofstream rna_vdw_stats_stream( tag+"rna_vdw_stats.txt" );
	//	std::ofstream rna_fullatom_stats_stream( tag+"rna_fullatom_stats.txt" );

	int count( 0 );

	for ( std::vector< std::string >::const_iterator file=files.begin();
				file != files.end(); ++file ) {

		count++;

		// read pose
		Pose source_pose;

		std::string const start_file( *file);
		std::cout << "Reading in ... " << start_file << std::endl;
		fill_decoy_silent_or_not( source_pose, decoys_p, start_file, silent_input );

		source_pose.dump_pdb( "out.pdb" );
		//		exit( 0 );

		source_pose.copy_to_misc(); // need to set total_residue correctly, etc.


		rna_pdb_stats( rna_stats_stream, string_of( count ) );

		rna_nonbase_base_pdb_stats( rna_nonbase_base_stats_stream, string_of( count ) );

		rna_backbone_hbond_pdb_stats( rna_backbone_hbond_stats_stream, string_of( count ) );

		rna_o2star_point_pdb_stats( rna_o2star_point_stats_stream, string_of( count ) );

		//		rna_vdw_pdb_stats( rna_vdw_stats_stream );

		//		rna_fullatom_pdb_stats( rna_fullatom_stats_stream );

		if (native_pose_exists) rna_rms_pdb_stats( rna_rms_stats_stream, source_pose, native_pose, string_of( count) );

	}

	rna_stats_stream.close();
	rna_rms_stats_stream.close();
	//	rna_vdw_stats_stream.close();
	//	rna_fullatom_stats_stream.close();

}


////////////////////////////////////////////////////////////////////////////
void
get_rna_centroid_information(
														 pose_ns::Pose & pose, std::ofstream & dataout,
														 std::ofstream & mapout, int const count)
{
  using namespace pose_ns;

	static bool initResMap = { false };

	static const Score_weight_map w = setup_rna_weight_map();
	pose.score( w );

	static bool const just_even_residues = truefalseoption("just_even_residues");

	//Distance matrix.
	//	pose_update_cendist( pose ); //may not be necessary if computing VDW anyway.
	FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );

	int const total_residue = pose.total_residue();
	int scorecount( 0 );

	///////////////////////////////////////////
	// Allow one degree of freedom that will
	// allow the mean energy to go up and down.
	///////////////////////////////////////////
	dataout << count << " " << ++scorecount << " " <<
		pose.get_0D_score( RNA_SASA ) << std::endl;


	//Longer range contacts, not necessarily base pairs (might be better to base distance off a sugar atom, rather than base centroid).

	FArray3D_float const & full_coord( pose.full_coord() );
	int const atomindex( 6 ); //C4*

	float const dist_cutoff = realafteroption( "dist_cutoff", 16.0 );
	float DISTCUTOFF2( dist_cutoff * dist_cutoff );

	static float const C4_C4 = truefalseoption("C4_C4");

	for ( int ii=1; ii<= total_residue; ++ii ) {

		numeric::xyzVector_float r_i( & full_coord(1, atomindex, ii ) );

		for ( int jj=ii+1; jj<= total_residue; ++jj) {

			numeric::xyzVector_float r_j( & full_coord(1, atomindex, jj ) );

			scorecount++;

			float dist = cendist( ii, jj );
			if (C4_C4) dist = (r_i - r_j).length_squared();

			if (!initResMap) mapout << scorecount << " " << ii << " " << jj << " 16.0 CONTACT" << std::endl;

			if ( dist < DISTCUTOFF2 ){

				if (just_even_residues && !(ii % 2 == 0 && jj % 2 == 0))  continue;

				dataout << count << " "  << scorecount << " " << std::sqrt( cendist(ii,jj) ) << std::endl;

			}


		}
	}


	if (!initResMap) mapout.close();
	initResMap = true;
	return;

}


////////////////////////////////////////////////////////////////////////////
void
rna_centroid_information_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	std::string prefix = stringafteroption("prefix","");

	std::ofstream dataout  ( (prefix+"CentroidInformation.txt").c_str() );
	std::ofstream  mapout  ( (prefix+"ResMap.txt").c_str() );

	bool const fullatom = true;

	Pose pose;

	//If just doing one structure (e.g., native P4-P6 RNA), bypass all the silent file readin stuff.
	if (truefalseoption("start_structure")){
		std::string file_name = stringafteroption("start_structure","blah.pdb");
		bool success = pose_from_pdb( pose, file_name,
									 true, false, true );
		if (!success){
			std::cout << "Had trouble with pdb " << file_name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		get_rna_centroid_information( pose, dataout, mapout, 1 /* count */ );
		return;
	}


	std::string silent_file_name= stringafteroption("s");

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
	tag_list = decoys.tags();

	int count = 0;
	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		//Fresh pose for each decoy. (Necessary for RNA stuff, which sets up atom trees.)
		Pose pose;

    // fill the pose
    decoy.fill_pose( pose, true /*check_coords*/, true /*save_input_score*/ );

		//extract centroid information
		count++;
		if (mod(count,100) == 0) std::cout << "DECOY " << count << std::endl;

		get_rna_centroid_information( pose, dataout, mapout, count );

  }

	dataout.close();
	mapout.close();
}

