// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_sse.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_tree_routines.h"
#include "current_pose.h"
#include "files_paths.h"
#include "filters.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_setup.h"
#include "initialize.h"
#include "jumping_util.h" // debug
#include "jumping_loops.h"
#include "minimize.h"
#include "nblist.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "pdb.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_rotamer_trials.h"
#include "pose_rms.h"
#include "pose_vdw.h"//for cendist.
#include "random_numbers.h"
#include "read_aa_ss.h"
#include "refold.h"
#include "rotamer_trials.h"
#include "rotamer_trial_energies.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h"
#include "silent_input.h"
#include "vdw.h"

#include "prof.h"

#include "misc.h"
#include "orient_rms.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//
// A bunch of routines to get familiar with posing, process decoys, etc.
// Mostly to manipulate segments of structures -- much of the stuff below
// is redundant with other code in Rosetta (e.g. in docking, or within
// the pose helper functions), but its nice to have private code I
// could mess around with.
//
// Rhiju, 2007.
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

void
setup_fullatom_options(){
	set_rot_limit( 45, 24 ); //like in relax_structure.
	set_perc_limit( 0.9999, 0.9999 );
	get_packing_options(); // does it hurt to set this up again?

	param_rotamer_trie::use_rotamer_trie = true;
	rotamer_trial_energies::use_trials_trie_flag = true;
	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(true);
	minimize_set_vary_chi(true);
	minimize_set_vary_rb_angle(true);
	minimize_set_vary_rb_trans(true);

	bool no_minimize_rb = truefalseoption("no_minimize_rb");
	if (no_minimize_rb){
		minimize_set_vary_rb_angle(false);
		minimize_set_vary_rb_trans(false);
	}

	bool no_minimize_backbone = truefalseoption("no_minimize_backbone");
	if (no_minimize_backbone){
		minimize_set_vary_phipsi(false);
		minimize_set_vary_omega(false);
	}

	if (truefalseoption("rtmin") ) score_set_minimize_rot( true );

	if (truefalseoption("try_rotamers")) score_set_try_rotamers( true );
	//	float const energycut(0.01); // for rotamer_trials

}

pose_ns::Score_weight_map favorite_weight_map(){
	using namespace pose_ns;
	Score_weight_map weight_map( score12 );
	weight_map.set_weight( OMEGA_CST, 1.0 )  ; // the torsion/bond/angle tether
	weight_map.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
	weight_map.set_weight( KIN_3D_CST, 1.0 )  ; // the torsion/bond/angle tether
	weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
	return weight_map;
}

void
rotate_translate_pose( pose_ns::Pose & pose,
		       float const phi,
		       float const theta,
		       float const psi,
		       float const x,
		       float const y,
		       float const z,
		       int const total_residue1)
{
  using namespace pose_ns;
  using namespace numeric;
  using namespace numeric::conversions;

  //Define a coordinate system on first helix.
  //  How about:
  //    origin at center of mass.
  //    z parallel to helix axis,
  //    x approximately from origin to central residue.
  xyzVector_float origin,r_cm,xaxis,yaxis,zaxis;

  FArray3D_float Epos = pose.Eposition();
  int const jump_point1 = total_residue1/2 + 1;

  r_cm = 0.0;
  for (int i = 1; i <= total_residue1; i++){
    xyzVector_float temp_CA( & Epos(1, 2, i) );
    r_cm += temp_CA;
  }
  r_cm /= total_residue1;
  origin = r_cm;

  xyzVector_float helix_center_CA( &Epos(1,2,jump_point1) );
  xaxis =  helix_center_CA - r_cm;
  xyzVector_float helixN( & Epos(1, 1,jump_point1 - 1) );
  xyzVector_float helixC( & Epos(1, 4,jump_point1 + 2) );
  zaxis = ( helixC - helixN ).normalize();
  yaxis = cross(zaxis, xaxis).normalize();
  xaxis = cross(yaxis, zaxis).normalize();

  xyzMatrix_float M1 = rotation_matrix( zaxis, radians(phi) );
  xyzMatrix_float M2 = rotation_matrix( xaxis, radians(theta) );
  xyzMatrix_float M3 = rotation_matrix( zaxis, radians(psi) );
  xyzMatrix_float M = M3 * M2 * M1;
	//  std::cout << "ROTATION MATRIX: " << std::endl << M << std::endl;

  // Aha figured out the jump routine.
  Jump j = pose.get_jump( 1 );
  j.rotation_by_matrix( Epos( 1,1,jump_point1), origin, M);


  //  xyzVector_double t( j.get_translation() );
  //  t += xaxis*x + yaxis*y + zaxis*z;
  //  j.set_translation( t );
  j.translation_along_axis( Epos( 1,1,jump_point1 ),
			    xaxis*x + yaxis*y + zaxis*z,
			    std::sqrt( x*x + y*y + z*z) );


  pose.set_jump( 1, j );


  // Do the rotation manually:
//   xyzVector_float r_N ( & Epos(1,1,jump_point1) );
//   xyzVector_float r_CA( & Epos(1,2,jump_point1) );
//   xyzVector_float r_C ( & Epos(1,4,jump_point1) );
//   r_N = M * (r_N - origin) + origin;
//   r_CA= M * (r_CA- origin) + origin;
//   r_C = M * (r_C - origin) + origin;
//   FArray3D_float Epos_new(1, param::MAX_POS, 1);
//   for (int i = 1; i <= 3; i++){
//     Epos_new(i,1,1) = r_N (i);
//     Epos_new(i,2,1) = r_CA(i);
//     Epos_new(i,4,1) = r_C (i);
//   }

//   Jump newjump( Epos(1,1,jump_point1), Epos_new(1,1,1));
//   pose.set_jump( 1, newjump );

}
////////////////////////////////////////////////////////////////////////////////////////////////////////

void
rotate_translate_pose( pose_ns::Pose & pose,
		       float const phi,
		       float const theta,
		       float const psi,
					 numeric::xyzVector_float const displacement,
					 int const total_residue1)
{
	rotate_translate_pose( pose, phi, theta, psi,
												 displacement(1), displacement(2), displacement(3),
												 total_residue1);
}



void
rotate_twist_pose( pose_ns::Pose & pose,
									 float const phi,
									 int const jump_point1,
									 int const jump_point2
									 )

{
  using namespace pose_ns;
  using namespace numeric;
  using namespace numeric::conversions;

  FArray3D_float Epos = pose.Eposition();

	xyzVector_float const jump_position1( &Epos(1, 2, jump_point1));
	xyzVector_float const jump_position2( &Epos(1, 2, jump_point2));
  xyzVector_float const rotation_axis  = jump_position2 - jump_position1;;

  // Do the rotation manually:
	xyzVector_float r_N ( & Epos(1,1,jump_point2) );
	xyzVector_float r_CA( & Epos(1,2,jump_point2) );
	xyzVector_float r_C ( & Epos(1,4,jump_point2) );
	xyzVector_float const origin = r_CA;

	xyzMatrix_float M = rotation_matrix( rotation_axis, radians(phi) );

	r_N = M * (r_N - origin) + origin;
	r_CA= M * (r_CA- origin) + origin;
	r_C = M * (r_C - origin) + origin;

	FArray3D_float Epos_new(3, param::MAX_POS, 1, 0.0);
	for (int i = 1; i <= 3; i++){
		Epos_new(i,1,1) = r_N (i);
		Epos_new(i,2,1) = r_CA(i);
		Epos_new(i,4,1) = r_C (i);
	}

	Jump newjump( Epos(1,1,jump_point1), Epos_new(1,1,1));
	pose.set_jump( 1, newjump );

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_gridsearch()
{
  using namespace pose_ns;
  using namespace numeric::conversions;

	bool const fullatom = truefalseoption( "fullatom" );

  Pose pose1, pose2, pose, native_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);


	//Create a starting pose with appropriate fold tree.
  Fold_tree f( pose.fold_tree() );
	int jump_point1  = total_residue1/2 + 1;
	int jump_point2 = total_residue2/2 + 1 + total_residue1;


  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );


	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;

	if (!fullatom) pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		set_rot_limit( 45, 24 ); //like in relax_structure.
		set_perc_limit( 0.9999, 0.9999 );
		get_packing_options(); // does it hurt to set this up again?
		weight_map = fullatom_weight_map;
		//		pose.full_repack( true );
	}	else {
		weight_map = centroid_weight_map;
	}

	pose.score( weight_map );
	calc_rms( pose );
	std::cout <<"NATIVE NATIVE NATIVE NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//starting conformation -- make helices overlapping.
  FArray3D_float Epos = pose.Eposition();
  Jump jump( Epos(1, 1, jump_point1), Epos(1,1,jump_point1) );
  pose.set_jump( 1, jump );

  pose.dump_pdb( "new_jump.pdb" );

  //Euler angles
  float phi   = 90.0 ;
  float theta =  0.0 ;
  float psi   =  0.0 ;
  float x = 0.0;
  float y = 0.0;
  float z = 6.0;

	float costheta = 1.0;
	float vdw_score, centroid_score, rms_err;
	float rms_cutoff = realafteroption( "rms_cutoff", 2.0 );
	float vdw_cutoff = realafteroption( "vdw_cutoff", 2.0 );
	float centroid_cutoff = realafteroption( "centroid_cutoff", 10000.0 );

	float phi_start = realafteroption("phi_start",0.0);

	for ( phi = phi_start; phi <= 360.0; phi += 40.0 ){
		for ( costheta = -1.0; costheta <= 1.0; costheta += 0.2 ){
			for ( psi = 0.0; psi <= 360.0; psi += 40.0 ){
				for ( x = -10.0; x <= 10.0; x += 2.0 ){
					for ( y = -10.0; y <= 10.0; y += 2.0 ){
						for ( z = -10.1; z <= 10.1; z += 2.0 ){

							theta = degrees( std::acos( numeric::sin_cos_range( costheta ) ) );

							Pose shift_pose;
							shift_pose = pose;

							shift_pose.set_fullatom_flag( false );
							rotate_translate_pose( shift_pose,
																		 phi, theta, psi,
																		 x, y, z, total_residue1);

							//screen based on rms
							calc_rms( shift_pose );
							rms_err = shift_pose.get_0D_score( RMSD );
							if (rms_err < rms_cutoff){

								//screen based on low resolution vdw.
								shift_pose.score( centroid_weight_map );
								vdw_score = shift_pose.get_0D_score( VDW );
								centroid_score = shift_pose.get_0D_score( SCORE );
								if (vdw_score < vdw_cutoff && centroid_score < centroid_cutoff){
									//OK now do a repack.
									if (fullatom) {
										shift_pose.set_fullatom_flag( true, true /* repack */);
										shift_pose.score( fullatom_weight_map );
										calc_rms( shift_pose );
									}
									std::cout << phi << " " << theta << " " << psi <<
										" " << x << " " << y << " " << z << " " <<
										shift_pose.show_scores() << std::endl;
								}
								//	    shift_pose.dump_pdb( "rotate_pose_" + string_of(i) + ".pdb" );
							}
						}
					}
				}
			}
		}
	}

}



void
pose_sse_translate_search()
{
  using namespace pose_ns;
  using namespace numeric::conversions;
	using namespace silent_io;

	bool const fullatom = truefalseoption( "fullatom" );

  Pose pose1, pose2, pose, native_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);


	//setup pdb info.
	pose.copy_to_misc();
	Pdb_info pdb_info;
	pdb_info.set_pdb_info( total_residue );
	pose.set_pdb_information( pdb_info );

	//Create a starting pose with appropriate fold tree.
  Fold_tree f( pose.fold_tree() );
  int jump_point1  = total_residue1/2 + 1;
  int jump_point2 = total_residue2/2 + 1 + total_residue1;
  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );


	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;

	if (!fullatom) pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		select_rotamer_set( "large" );
		set_rot_limit( 45, 24 ); //like in relax_structure.
		set_perc_limit( 0.9999, 0.9999 );
		get_packing_options(); // does it hurt to set this up again?

		weight_map = fullatom_weight_map;
		//		pose.full_repack( true );
	}	else {
		weight_map = centroid_weight_map;
	}

	pose.score( weight_map );
	calc_rms( pose );
	std::cout <<"NATIVE NATIVE NATIVE NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//starting conformation -- make helices overlapping.
	//  FArray3D_float Epos = pose.Eposition();
	//  Jump jump( Epos(1, 1, jump_point1), Epos(1,1,jump_point1) );
	//  pose.set_jump( 1, jump );
	pose.dump_pdb( "new_jump.pdb" );

  //Euler angles
  float phi   =  0.0 ;
  float theta =  0.0 ;
  float psi   =  0.0 ;
  float x = 0.0;
  float y = 0.0;
  float z = -0.0001;

	float vdw_score;
	float const vdw_cutoff = realafteroption( "vdw_cutoff", 2.0);
	bool minimize = truefalseoption("minimize");
	bool no_minimize_backbone = truefalseoption("no_minimize_backbone");

	int count = 0;

	std::string slavetag;
	stringafteroption("slavetag","",slavetag);
	int const slavenum = intafteroption("slavetag",0);

	//Overwrite any previous outfile.
	std::string output_file_name( "helixhelix"+slavetag+".out" );
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );
	Silent_out out( output_file_name );

	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(true);
	minimize_set_vary_chi(true);
	minimize_set_vary_rb_angle(false);
	minimize_set_vary_rb_trans(false);

	if (no_minimize_backbone){
		minimize_set_vary_phipsi(false);
		minimize_set_vary_omega(false);
	}

	if (truefalseoption("minimize_vary_rb_angle")) minimize_set_vary_rb_angle( true );
	if (truefalseoption("minimize_vary_rb_trans")) minimize_set_vary_rb_trans( true );
	bool const repack = truefalseoption("repack");
	bool const repack_trp = truefalseoption("repack_trp");
	bool const use_input_rotamer = truefalseoption("use_input_rotamer");

	if (minimize) pose.main_minimize( weight_map, "dfpmin" );
	pose.set_extra_score("x",x);
	pose.set_extra_score("y",y);
	pose.set_extra_score("z",z);
	std::string tag = "S_"+lead_zero_string_of(count++, 3);
	calc_rms( pose );
	out.write( tag, pose );

	//Oh man, this is silly.
	param_rotamer_trie::use_rotamer_trie = true;
	rotamer_trial_energies::use_trials_trie_flag = true;

	float const spacing = realafteroption("spacing", 0.2);

	int const numnodes = intafteroption( "numnodes", 1);
	//	int const total_decoys = static_cast<int>( (20/spacing)*(20/spacing)*(20/spacing) );

	//Should probably distribute burden by modulo.
	//	int const mincount = slavenum * (total_decoys/ numnodes);
	//	int const maxcount = mincount + total_decoys/numnodes;


	for ( x = -10.0; x < 10.0; x += spacing ){
		for ( y = -10.0; y < 10.0; y += spacing ){
			for ( z = -9.9999; z < 10.0; z += spacing ){

				tag = "S_"+lead_zero_string_of(count++, 3);

				//				if (count <= mincount || count > maxcount) continue;
				if ( mod(count,numnodes) != slavenum ) continue;

				//if ( !out.start_decoy(tag) ) continue; // already done or started

				std::cout << "Doing " << tag << " " << x << " " << y << " " << z << std::endl;
				//				theta = degrees( std::acos( numeric::sin_cos_range( costheta ) ) );

		    Pose shift_pose;
				shift_pose = pose;

				shift_pose.set_fullatom_flag( false );
				rotate_translate_pose( shift_pose,
															 phi, theta, psi,
															 x, y, z, total_residue1);

				shift_pose.score( centroid_weight_map );
				vdw_score = shift_pose.get_0D_score( VDW );

				//OK now do a repack.
				if (vdw_score < vdw_cutoff) {
					if (repack) shift_pose.set_fullatom_flag( true, true /* repack */);
					//if (repack || repack_trp) shift_pose.set_fullatom_flag( true, false /* repack */);
					if (repack) shift_pose.full_repack( use_input_rotamer );
					//					if (repack) shift_pose.full_repack( false );

					if (repack_trp) {
						FArray1D_bool allow_repack( total_residue, false );
						allow_repack( 22 ) = true; //the tryptophan.
						shift_pose.repack( allow_repack, use_input_rotamer /*use input rotamer*/);
					}

					if (minimize) shift_pose.main_minimize( weight_map, "dfpmin" );
					if (fullatom) shift_pose.score( fullatom_weight_map );
					calc_rms( shift_pose );

					shift_pose.set_extra_score("x",x);
					shift_pose.set_extra_score("y",y);
					shift_pose.set_extra_score("z",z);

					//shift_pose.dump_pdb( tag +".pdb");

					out.write( tag, shift_pose );

				}

				if (count==10) prof::show();
			}
		}
	}
	prof::show();

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_translate_1D_search()
{
  using namespace pose_ns;
  using namespace numeric::conversions;
	using namespace silent_io;

	bool const fullatom = truefalseoption( "fullatom" );

  Pose pose1, pose2, pose, native_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);


	//Create a starting pose with appropriate fold tree.
  Fold_tree f( pose.fold_tree() );
  int jump_point1  = total_residue1/2 + 1;
  int jump_point2 = total_residue2/2 + 1 + total_residue1;
  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );


	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;
	pose_to_native(native_pose);

	if (!fullatom) pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		select_rotamer_set( "large" );
		set_rot_limit( 45, 24 ); //like in relax_structure.
		set_perc_limit( 0.9999, 0.9999 );
		get_packing_options(); // does it hurt to set this up again?

		weight_map = fullatom_weight_map;
		//		pose.full_repack( true );
	}	else {
		weight_map = centroid_weight_map;
	}

	pose.score( weight_map );
	calc_rms( pose );
	std::cout <<"NATIVE NATIVE NATIVE NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//starting conformation -- make helices overlapping.
	//  FArray3D_float Epos = pose.Eposition();
	//  Jump jump( Epos(1, 1, jump_point1), Epos(1,1,jump_point1) );
	//  pose.set_jump( 1, jump );
	pose.dump_pdb( "new_jump.pdb" );

  //Euler angles
  float phi   =  0.0 ;
  float theta =  0.0 ;
  float psi   =  0.0 ;
  float x = 0.0;
  float y = 0.0;
  float z = -0.0001;

	bool minimize = truefalseoption("minimize");
	int num_min_trials = intafteroption("num_min_trials",1);

	int count = 0;

	//Overwrite any previous outfile.
	std::string output_file_name( "helixhelix.out" );
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );
	Silent_out out( output_file_name );

	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(true);
	minimize_set_vary_chi(true);
	minimize_set_vary_rb_angle(false);
	minimize_set_vary_rb_trans(false);

	if (truefalseoption("rtmin") ) score_set_minimize_rot( true );

	if (truefalseoption("minimize_vary_rb_angle")) minimize_set_vary_rb_angle( true );
	if (truefalseoption("minimize_vary_rb_trans")) minimize_set_vary_rb_trans( true );
	bool const repack = truefalseoption("repack");
	bool const repack_trp = truefalseoption("repack_trp");
	bool const use_input_rotamer = truefalseoption("use_input_rotamer");

	if (minimize) pose.main_minimize( weight_map, "dfpmin" );
	pose.set_extra_score("x",x);
	pose.set_extra_score("y",y);
	pose.set_extra_score("z",z);
	std::string tag = "S_"+lead_zero_string_of(count++, 3);
	calc_rms( pose );
	out.write( tag, pose );

	//Oh man, this is silly.
	param_rotamer_trie::use_rotamer_trie = true;
	rotamer_trial_energies::use_trials_trie_flag = true;
	score_set_try_rotamers( true );

	Pose shift_pose;
	shift_pose = pose;

	float const shift_amount = realafteroption("shift_amount",0.1);
	float const s_min = realafteroption("s_min",-10.0001);
	float const s_max = realafteroption("s_max",10.0);

	//Starting configuration
	using namespace numeric;
	float const shiftx = realafteroption( "shiftx", 0.0 );
	float const shifty = realafteroption( "shifty", 0.0 );
	float const shiftz = realafteroption( "shiftz", 1.0 );

	xyzVector_float shift_vector( shiftx, shifty, shiftz ), displacement, displacement_prev;
	shift_vector.normalize();

	displacement = shift_vector * (s_max + shift_amount);
	rotate_translate_pose( shift_pose,
												 phi, theta, psi,
												 displacement, total_residue1);
	if (fullatom) shift_pose.full_repack( false );

	displacement_prev = displacement;
	for ( float s = s_max; s >= s_min; s -= shift_amount ){
				tag = "S_"+lead_zero_string_of(count++, 3);

				//if ( !out.start_decoy(tag) ) continue; // already done or started
				displacement = s * shift_vector;

				std::cout << "Doing " << tag << " " << displacement << std::endl;
				//				theta = degrees( std::acos( numeric::sin_cos_range( costheta ) ) );

				//				shift_pose.set_fullatom_flag( false );
				rotate_translate_pose( shift_pose,
															 phi, theta, psi,
															 displacement - displacement_prev /*shift_vector*/,
															 total_residue1);

				//OK now do a repack.
				if (repack || repack_trp) shift_pose.set_fullatom_flag( true, true /* repack */);
				//if (repack || repack_trp) shift_pose.set_fullatom_flag( true, false /* repack */);
				if (repack) shift_pose.full_repack( use_input_rotamer );
				//					if (repack) shift_pose.full_repack( false );

				if (repack_trp) {
					FArray1D_bool allow_repack( total_residue, false );
					allow_repack( 22 ) = true; //the tryptophan.
					shift_pose.repack( allow_repack, use_input_rotamer /*use input rotamer*/);
				}

				if (minimize) {
					for (int i = 1; i <= num_min_trials; i++)
						shift_pose.main_minimize( weight_map, "dfpmin" );
				}
				shift_pose.score( weight_map );
				calc_rms( shift_pose );

				shift_pose.set_extra_score("x",displacement(1));
				shift_pose.set_extra_score("y",displacement(2));
				shift_pose.set_extra_score("z",displacement(3));

				shift_pose.dump_pdb( tag +".pdb");

				out.write( tag, shift_pose );

				if (count==10) prof::show();

				displacement_prev = displacement;
	}

	prof::show();

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_translate_1D_boltzmann()
{
  using namespace pose_ns;
  using namespace numeric::conversions;
	using namespace silent_io;

	bool const fullatom = truefalseoption( "fullatom" );

  Pose pose1, pose2, pose, native_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);


	//Create a starting pose with appropriate fold tree.
  Fold_tree f( pose.fold_tree() );
  int jump_point1  = total_residue1/2 + 1;
  int jump_point2 = total_residue2/2 + 1 + total_residue1;
  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );


	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;

	if (!fullatom) pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		select_rotamer_set( "large" );
		set_rot_limit( 45, 24 ); //like in relax_structure.
		set_perc_limit( 0.9999, 0.9999 );
		get_packing_options(); // does it hurt to set this up again?

		weight_map = fullatom_weight_map;
		//		pose.full_repack( true );
	}	else {
		weight_map = centroid_weight_map;
	}

	pose.score( weight_map );
	calc_rms( pose );
	std::cout <<"NATIVE NATIVE NATIVE NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//starting conformation -- make helices overlapping.
	//  FArray3D_float Epos = pose.Eposition();
	//  Jump jump( Epos(1, 1, jump_point1), Epos(1,1,jump_point1) );
	//  pose.set_jump( 1, jump );
	pose.dump_pdb( "new_jump.pdb" );

  //Euler angles
  float phi   =  0.0 ;
  float theta =  0.0 ;
  float psi   =  0.0 ;
  float x = 0.0;
  float y = 0.0;
  float z = -0.0001;

	bool minimize = truefalseoption("minimize");
	int num_min_trials = intafteroption("num_min_trials",1);

	int count = 0;

	//Overwrite any previous outfile.
	std::string output_file_name = stringafteroption("o", "helixhelix.out" );
	//	std::string command = "rm -f "+output_file_name;
	//	std::system( command.c_str() );
	Silent_out out( output_file_name );

	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(true);
	minimize_set_vary_chi(true);
	minimize_set_vary_rb_angle(false);
	minimize_set_vary_rb_trans(false);

	if (truefalseoption("minimize_vary_rb_angle")) minimize_set_vary_rb_angle( true );
	if (truefalseoption("minimize_vary_rb_trans")) minimize_set_vary_rb_trans( true );
	bool const repack = truefalseoption("repack");
	int const repack_period = intafteroption("repack_period",1);
	bool const use_input_rotamer = truefalseoption("use_input_rotamer");

	if (minimize) pose.main_minimize( weight_map, "dfpmin" );
	pose.set_extra_score("x",x);
	pose.set_extra_score("y",y);
	pose.set_extra_score("z",z);
	std::string tag = "S_"+lead_zero_string_of(count++, 3);
	calc_rms( pose );
	//	out.write( tag, pose );

	//Oh man, this is silly.
	param_rotamer_trie::use_rotamer_trie = true;
	rotamer_trial_energies::use_trials_trie_flag = true;
	score_set_try_rotamers( true );

	Pose shift_pose, init_pose;
	init_pose = pose;

	float const shift_amount = realafteroption("shift_option",1.0);

	//Starting configuration
	x = 0.0;
	y = 0.0;
	z = -10.0;

	rotate_translate_pose( init_pose,
												 phi, theta, psi,
												 x, y, z, total_residue1);
	if (fullatom) init_pose.full_repack( false );


	static int const nstruct = intafteroption( "nstruct", 1);
	static int const cycles = intafteroption( "cycles", 100);

	for (count = 1; count <= nstruct; count++ ){
		tag = "S_"+lead_zero_string_of(count, 3);

		if ( !out.start_decoy(tag) ) continue; // already done or started
		std::cout << "Doing " << tag << " " << std::endl;

		float const init_temp ( 1.0 );
		Monte_carlo mc( init_pose, weight_map, init_temp );
		mc.set_autotemp( true, init_temp );

		shift_pose = init_pose;
		shift_pose.set_fullatom_flag( true, false /* repack */);
		z = -10.0;

		for (int k = 1; k <= cycles; k++){

			float random_shift = -100.0;
			while ( abs(z + random_shift) > 10.0 ){
				random_shift = shift_amount * gaussian();
			}
			z += random_shift; //Hmm, roundoff errors?

			rotate_translate_pose( shift_pose,
														 phi, theta, psi,
														 0, 0, random_shift, total_residue1);

			if ( repack && (mod(k,repack_period) == 0) ){
				//				std::cout << "Time for a repack ... " << std::endl;
				shift_pose.full_repack( use_input_rotamer );
			}
			if (minimize) {
				for (int i = 1; i <= num_min_trials; i++)
					shift_pose.main_minimize( weight_map, "dfpmin" );
			}

			shift_pose.score( weight_map );
			shift_pose.set_extra_score("x",x);
			shift_pose.set_extra_score("y",y);
			shift_pose.set_extra_score("z",z);
			calc_rms( shift_pose );

			//			std::cout << "Iteration " << k <<
			//				"   Energy " << shift_pose.get_0D_score( SCORE ) <<
			//				"   RMS " << shift_pose.get_0D_score( RMSD) <<
			//				"     z " << z << std::endl;

			mc.boltzmann( shift_pose );

			z = shift_pose.get_extra_score("z");

		}

		shift_pose = mc.low_pose();
		calc_rms(shift_pose);

		//		shift_pose.dump_pdb( tag +".pdb");
		out.write( tag, shift_pose );
	}

	prof::show();

}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_twist_1D_search()
{
  using namespace pose_ns;
  using namespace numeric::conversions;
	using namespace silent_io;

	bool const fullatom = truefalseoption( "fullatom" );

  Pose pose1, pose2, pose, native_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);


	//Create a starting pose with appropriate fold tree.
	int const  jump_point1 = intafteroption( "jump_point1", 8);
	int const  jump_point2 = intafteroption( "jump_point2", 22);

  Fold_tree f( pose.fold_tree() );
	//  int jump_point1  = total_residue1/2 + 1;
	//  int jump_point2 = total_residue2/2 + 1 + total_residue1;
  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );

	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;

	if (!fullatom) pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		select_rotamer_set( "large" );
		set_rot_limit( 45, 24 ); //like in relax_structure.
		set_perc_limit( 0.9999, 0.9999 );
		get_packing_options(); // does it hurt to set this up again?

		weight_map = fullatom_weight_map;
		//		pose.full_repack( true );
	}	else {
		weight_map = centroid_weight_map;
	}

	pose.score( weight_map );
	calc_rms( pose );
	std::cout <<"NATIVE NATIVE NATIVE NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//starting conformation -- make helices overlapping.
	//  FArray3D_float Epos = pose.Eposition();
	//  Jump jump( Epos(1, 1, jump_point1), Epos(1,1,jump_point1) );
	//  pose.set_jump( 1, jump );
	pose.dump_pdb( "new_jump.pdb" );

  //Euler angles
  float phi   =  0.0 ;

	bool minimize = truefalseoption("minimize");
	int num_min_trials = intafteroption("num_min_trials",1);
	bool no_minimize_backbone = truefalseoption("no_minimize_backbone");

	int count = 0;

	//Overwrite any previous outfile.
	std::string output_file_name( "helixhelix.out" );
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );
	Silent_out out( output_file_name );

	minimize_set_vary_phipsi(true);
	minimize_set_vary_omega(true);
	minimize_set_vary_chi(true);
	minimize_set_vary_rb_angle(false);
	minimize_set_vary_rb_trans(false);

	if (no_minimize_backbone){
		minimize_set_vary_phipsi(false);
		minimize_set_vary_omega(false);
	}

	if (truefalseoption("minimize_vary_rb_angle")) minimize_set_vary_rb_angle( true );
	if (truefalseoption("minimize_vary_rb_trans")) minimize_set_vary_rb_trans( true );
	bool const repack = truefalseoption("repack");
	bool const repack_trp = truefalseoption("repack_trp");
	bool const use_input_rotamer = truefalseoption("use_input_rotamer");

	if (minimize) pose.main_minimize( weight_map, "dfpmin" );
	pose.set_extra_score("phi",phi);
	std::string tag = "S_"+lead_zero_string_of(count++, 3);
	calc_rms( pose );
	out.write( tag, pose );

	//Oh man, this is silly.
	param_rotamer_trie::use_rotamer_trie = true;
	rotamer_trial_energies::use_trials_trie_flag = true;
	score_set_try_rotamers( true );


	float const shift_amount = realafteroption("shift_amount",10.0);
	float const s_min = realafteroption("s_min",-180.001);
	float const s_max = realafteroption("s_max",180.0);

	for ( float s = s_min; s <= s_max; s += shift_amount ){
				tag = "S_"+lead_zero_string_of(count++, 3);

				//if ( !out.start_decoy(tag) ) continue; // already done or started

				Pose shift_pose;
				shift_pose = pose;

				phi = s;

				std::cout << "Doing " << tag << " " << phi << std::endl;

				rotate_twist_pose( shift_pose, phi,
													 jump_point1, jump_point2);

				//OK now do a repack.
				if (repack || repack_trp) shift_pose.set_fullatom_flag( true, false /* repack */);
				if (repack) shift_pose.full_repack( use_input_rotamer );
				if (repack_trp) {
					FArray1D_bool allow_repack( total_residue, false );
					allow_repack( 22 ) = true; //the tryptophan.
					shift_pose.repack( allow_repack, use_input_rotamer /*use input rotamer*/);
				}

				if (minimize) {
					for (int i = 1; i <= num_min_trials; i++)
						shift_pose.main_minimize( weight_map, "dfpmin" );
				}
				shift_pose.score( weight_map );
				calc_rms( shift_pose );

				pose.set_extra_score("phi",phi);

				shift_pose.dump_pdb( tag +".pdb");

				out.write( tag, shift_pose );

				if (count==10) prof::show();
	}

	prof::show();

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_testjumps()
{
  using namespace pose_ns;

  Pose pose1, pose2, pose, native_pose, start_pose;

  pose_from_pdb( pose1, stringafteroption("s1"),
		 true, false, true );

  pose_from_pdb( pose2, stringafteroption("s2"),
		 true, false, true );

  int total_residue1( pose1.total_residue());
  int total_residue2( pose2.total_residue());
  int  total_residue = total_residue1 + total_residue2;

  pose.simple_fold_tree( total_residue );
  pose.set_fullatom_flag( true, false );
  pose.copy_segment( total_residue1, pose1, 1, 1, false);
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, false /*update jump*/);

	start_pose = pose;

	//Create a starting pose with appropriate fold tree.
  Fold_tree f( pose.fold_tree() );
  int jump_point1  = total_residue1/2 + 1;
  int jump_point2 = total_residue2/2 + 1 + total_residue1;
  int cutpoint = total_residue1;
  f.new_jump( jump_point1, jump_point2, cutpoint );
  pose.set_fold_tree( f );


	//native_pose
  pose.copy_segment( total_residue2, pose2, total_residue1+1, 1, true /*update jump*/);
  pose.dump_pdb( "native_pose.pdb" );
	native_pose = pose;

	pose.set_fullatom_flag( false );

	//score native.
  pose.set_native_pose( native_pose );
  start_pose.set_native_pose( native_pose );
  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	weight_map = centroid_weight_map;

	pose.score( weight_map );
	calc_rms( pose );
	std::cout << "NATIVE NATIVE NATIVE " <<
		pose.show_scores() << std::endl;


	//Read jump templates from an input file.
	std::string pairing_file = stringafteroption("pairing_file");
	utility::io::izstream pairing_stream( pairing_file );
	if ( !pairing_stream ) {
		std::cout << "can't open pairings file!!!" << pairing_file << std::endl;
		pairing_stream.close();
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string line,srcfile;
	int pos1,pos2,count(1);
	FArray2D_float Epos1(3,param::MAX_POS), Epos2(3,param::MAX_POS);

	while ( getline( pairing_stream, line ) ) {

		std::istringstream line_stream( line );
		line_stream >> srcfile >> pos1 >> pos2 >>
			Epos1(1,1) >> Epos1(2,1) >> Epos1(3,1) >>
			Epos1(1,2) >> Epos1(2,2) >> Epos1(3,2) >>
			Epos1(1,4) >> Epos1(2,4) >> Epos1(3,4) >>
			Epos2(1,1) >> Epos2(2,1) >> Epos2(3,1) >>
			Epos2(1,2) >> Epos2(2,2) >> Epos2(3,2) >>
			Epos2(1,4) >> Epos2(2,4) >> Epos2(3,4);

		Jump j( Epos1, Epos2 );


		// Ignore input position -- don't appear to be correct anyway.
		for (pos1 = 0; pos1 < total_residue1; pos1++){
			for (pos2 = 0; pos2 < total_residue2; pos2++){
				pose.simple_fold_tree( total_residue );
				Fold_tree f2( pose.fold_tree() );
				f2.new_jump( 1+pos1, total_residue1+1+pos2, cutpoint );
				pose.set_fold_tree( f2 );

				pose.set_jump( 1, j );
				std::cout << "JUMP " << j << std::endl;

				pose.score( weight_map );
				calc_rms( pose );

				std::cout << count << " " << pos1 << " " << pos2 << " " <<
					pose.show_scores() << std::endl;
			}
		}
		count++;

		//		pose.dump_pdb( "new_jump.pdb" );
	}

	pairing_stream.close();

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
save_pose_scores( FArray1Da_float score_array, pose_ns::Pose & pose, bool output_dun = false ){
	score_array.dimension( 12 );

	using namespace pose_ns;
	score_array( 1) = pose.get_0D_score( SCORE );
	score_array( 2) = pose.get_0D_score( FA_ATR );
	score_array( 3) = pose.get_0D_score( FA_REP );
	score_array( 4) = pose.get_0D_score( FA_SOL );
	score_array( 5) = pose.get_0D_score( FA_PAIR );
	score_array( 6) = pose.get_0D_score( FA_REF );
	score_array( 7) = pose.get_0D_score( FA_DUN );
	score_array( 8) = pose.get_0D_score( FA_PROB );
	score_array( 9) = pose.get_0D_score( HB_SRBB );
	score_array(10) = pose.get_0D_score( HB_LRBB );
	score_array(11) = pose.get_0D_score( HB_SC );
	score_array(12) = pose.get_0D_score( FA_INTRA );

	if (output_dun){
		int const total_residue = pose.total_residue();
		float dunenergy_total = 0.0;
		FArray1D_float dunenergy( pose.get_1D_score( DUNENERGY ) );

		for (int i = 1; i<= total_residue; i++ ){
			std::cout << "DUN " << i << " " << dunenergy(i) << std::endl;
			dunenergy_total += dunenergy(i);
		}
		std::cout << "DUNENERGY " << dunenergy_total << " " << pose.get_0D_score( FA_DUN ) << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_threebody( pose_ns::Pose & pose_input)
{

	using namespace pose_ns;

	Pose pose1, pose2, pose3;

	int const N = 3;
	FArray2D_int res_boundary( N, 2, 0 );
	FArray1D_int total_residue_segment( N, 0);

	res_boundary( 1, 1 ) =  2;
	res_boundary( 1, 2 ) = 23;

	res_boundary( 2, 1 ) = 29;
	res_boundary( 2, 2 ) = 51;

	res_boundary( 3, 1 ) = 53;
	res_boundary( 3, 2 ) = 63;

	for (int i = 1; i <= N; i++) {
		total_residue_segment( i ) = res_boundary( i, 2) - res_boundary( i, 1) + 1;
	}


	Score_weight_map weight_map( score12 );

	FArray2D_float singlet_score( 12,  3, 0.0);
	FArray3D_float doublet_score( 12, 3, 3, 0.0);
	FArray1D_float numeric_score( 12, 0.0);

	FArray2D_float singlet_score_repack( 12, 3, 0.0);
	FArray3D_float doublet_score_repack( 12, 3, 3, 0.0);
	FArray1D_float numeric_score_repack( 12, 0.0);

	//Singlets
	for (int i = 1; i <= N; i++) {
		Pose pose;

		pose.simple_fold_tree( total_residue_segment( i ) );
		pose.set_fullatom_flag( true, false );

		pose.copy_segment( total_residue_segment(i), pose_input,
											 1, res_boundary( i, 1 ), false );

		pose.score( weight_map );
		//		singlet_score( i ) = pose.get_0D_score( SCORE );
		save_pose_scores( singlet_score(1, i), pose );


		pose.full_repack( true );
		minimize_set_vary_phipsi(true);
		minimize_set_vary_chi(true);
		pose.main_minimize( weight_map, "dfpmin" );
		pose.score( weight_map );
		//		singlet_score_repack( i ) = pose.get_0D_score( SCORE );
		save_pose_scores( singlet_score_repack(1, i), pose );

		//		std::cout << i << " SELF " <<  pose.show_scores() << std::endl;
	}

	//Doublets
	for (int i = 1; i <= N; i++) {
		for (int j = i+1; j <= N; j++) {

			Pose pose;

			int total_residue = total_residue_segment( i )  +
				total_residue_segment( j );

			pose.simple_fold_tree( total_residue );
			pose.set_fullatom_flag( true, false );

			pose.copy_segment( total_residue_segment(i), pose_input,
												 1, res_boundary(i,1), false );
			pose.copy_segment( total_residue_segment(j), pose_input,
												 total_residue_segment(i) + 1,
												 res_boundary(j,1), false );

			Fold_tree f( pose.fold_tree() );
			int jump_point1  = total_residue_segment(i)/2 + 1;
			int jump_point2 = total_residue_segment(j)/2 + 1 + total_residue_segment(i);
			int cutpoint = total_residue_segment(i);
			f.new_jump( jump_point1, jump_point2, cutpoint );
			pose.set_fold_tree( f );

			pose.copy_segment( total_residue_segment(j), pose_input,
										 total_residue_segment(i) + 1, res_boundary(j,1), true );

			pose.score( weight_map );
			//			doublet_score( i, j ) = pose.get_0D_score( SCORE );
			save_pose_scores( doublet_score(1, i, j), pose );

			pose.full_repack( true );
			pose.score( weight_map );
			minimize_set_vary_phipsi(true);
			minimize_set_vary_chi(true);
			pose.main_minimize( weight_map, "dfpmin" );
			pose.score( weight_map );
			//			doublet_score_repack( i, j ) = pose.get_0D_score( SCORE );
			save_pose_scores( doublet_score_repack(1, i, j), pose );

			//			std::cout << i << " " << j << " " << pose.show_scores() << std::endl;
		}
	}

	//Numeric
	Pose pose;

	int total_residue = total_residue_segment( 1 )  +
		total_residue_segment( 2 ) +
		total_residue_segment( 3 );

	pose.simple_fold_tree( total_residue );
	pose.set_fullatom_flag( true, false );

	pose.copy_segment( total_residue_segment(1), pose_input,
										 1, res_boundary(1,1), false );
	pose.copy_segment( total_residue_segment(2), pose_input,
										 total_residue_segment(1) + 1,
										 res_boundary(2,1), false );
	pose.copy_segment( total_residue_segment(3), pose_input,
										 total_residue_segment(1) +
										 total_residue_segment(2) + 1,
										 res_boundary(3,1), false );

	Fold_tree f( pose.fold_tree() );
	int jump_point1  = total_residue_segment(1)/2 + 1;
	int jump_point2 = total_residue_segment(2)/2 + 1 + total_residue_segment(1);
	int jump_point3 = total_residue_segment(3)/2 + 1 + total_residue_segment(1) + total_residue_segment(2);
	int cutpoint  = total_residue_segment(1);
	int cutpoint2 = total_residue_segment(1) + total_residue_segment(2);
	f.new_jump( jump_point1, jump_point2, cutpoint  );
	f.new_jump( jump_point2, jump_point3, cutpoint2 );
	pose.set_fold_tree( f );

	pose.copy_segment( total_residue_segment(2), pose_input,
										 total_residue_segment(1) + 1,
										 res_boundary(2,1), true );
	pose.copy_segment( total_residue_segment(3), pose_input,
										 total_residue_segment(1) +
										 total_residue_segment(2) + 1,
										 res_boundary(3,1), true );


	pose.score( weight_map );
	//	numeric_score = pose.get_0D_score( SCORE );
	save_pose_scores( numeric_score(1), pose );
	//	std::cout << "NUMERIC NUMERIC " << pose.show_scores() << std::endl;

	pose.full_repack( true );
	pose.score( weight_map );
	minimize_set_vary_phipsi(true);
	minimize_set_vary_chi(true);
	pose.main_minimize( weight_map, "dfpmin" );
	//	numeric_score_repack = pose.get_0D_score( SCORE );
	save_pose_scores( numeric_score_repack(1), pose );


	pose.dump_pdb( "blah.pdb" );

	float three_body_interaction = numeric_score(1)
		- doublet_score(1,1,2) - doublet_score(1,1,3) - doublet_score(1,2,3)
		+ singlet_score(1,1) + singlet_score(1,2) + singlet_score(1,3);

	float three_body_interaction_repack = numeric_score_repack(1)
		- doublet_score_repack(1,1,2) - doublet_score_repack(1,1,3) - doublet_score_repack(1,2,3)
		+ singlet_score_repack(1,1) + singlet_score_repack(1,2) + singlet_score_repack(1,3);

	std::cout << "SUMMARY " <<
		singlet_score(1,1) << " " <<
		singlet_score(1,2) << " " <<
		singlet_score(1,3) << " " <<
		doublet_score(1,1,2) << " " <<
		doublet_score(1,1,3) << " " <<
		doublet_score(1,2,3) << " " <<
		numeric_score(1) << " " <<
		three_body_interaction << " " <<
		singlet_score_repack(1,1) << " " <<
		singlet_score_repack(1,2) << " " <<
		singlet_score_repack(1,3) << " " <<
		doublet_score_repack(1,1,2) << " " <<
		doublet_score_repack(1,1,3) << " " <<
		doublet_score_repack(1,2,3) << " " <<
		numeric_score_repack(1) << " " <<
		three_body_interaction_repack << " "  <<
		std::endl;


	//	three_body_interaction = numeric_score(7)
	//	- doublet_score(7,1,2) - doublet_score(7,1,3) - doublet_score(7,2,3)
	//		+ singlet_score(7,1) + singlet_score(7,2) + singlet_score(7,3);
	//
	//  three_body_interaction_repack = numeric_score_repack(7)
	//		- doublet_score_repack(7,1,2) - doublet_score_repack(7,1,3) - doublet_score_repack(7,2,3)
	//		+ singlet_score_repack(7,1) + singlet_score_repack(7,2) + singlet_score_repack(7,3);
	//
	//	std::cout << "DUNSUMMARY " <<
	//		singlet_score(7,1) << " " <<
	//		singlet_score(7,2) << " " <<
	//		singlet_score(7,3) << " " <<
	//		doublet_score(7,1,2) << " " <<
	//		doublet_score(7,1,3) << " " <<
	//		doublet_score(7,2,3) << " " <<
	//		numeric_score(7) << " " <<
	//		three_body_interaction << " " <<
	//		singlet_score_repack(7,1) << " " <<
	//		singlet_score_repack(7,2) << " " <<
	//		singlet_score_repack(7,3) << " " <<
	//		doublet_score_repack(7,1,2) << " " <<
	//		doublet_score_repack(7,1,3) << " " <<
	//		doublet_score_repack(7,2,3) << " " <<
	//		numeric_score_repack(7) << " " <<
	//		three_body_interaction_repack << " "  <<
	//		std::endl;

	std::cout << "THREEBODY " ;
	for (int k = 1; k <= 12; k++) {
		float temp = numeric_score(k)
		- doublet_score(k,1,2) - doublet_score(k,1,3) - doublet_score(k,2,3)
		+ singlet_score(k,1) + singlet_score(k,2) + singlet_score(k,3);
		std::cout << " " << temp;
	}
	std::cout << std::endl;

	for (int i = 1; i <= 3; i++){
		for (int j = i+1; j <= 3; j++){
			std::cout << "TWOBODY " << i << " " << j;
			for (int k = 1; k <= 12; k++) {
				float temp =
					doublet_score(k,i,j)
					- singlet_score(k,i) - singlet_score(k,j);
				std::cout << " " << temp;
			}
			std::cout << std::endl;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
pose_sse_threebody_test()
{

// 	using namespace pose_ns;
// 	Pose pose_input;
//   pose_from_pdb( pose_input, stringafteroption("s"),
// 								 true, false, true );
// 	pose_sse_threebody( pose_input );

  using namespace silent_io;
  using namespace pose_ns;

  const bool fullatom( truefalseoption("fa_input") );
	Pose pose;

	std::string silent_file_name= stringafteroption("s");

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
  if ( truefalseoption("t") ) {
    tag_list.push_back( stringafteroption("t") );
	} else if ( truefalseoption("all") ) {
		tag_list = decoys.tags();
  } else if ( truefalseoption("l") ) {
    std::ifstream data( stringafteroption("l").c_str() );
    if ( !data.good() ) {
      std::cout << "STOP:: cant open tags file: " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    std::string tag;
    while ( getline(data,tag) ) {
      tag_list.push_back(tag);
    }
		data.close();
  }

	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

    if ( ! decoys.has_key( tag ) ) {
      std::cout << "couldnt find tag in silent-file: " << tag << std::endl;
      continue;
    }

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		//Fresh pose for each decoy. (Necessary for RNA stuff, which sets up atom trees.)
		Pose pose_input;

    // fill the pose
    decoy.fill_pose( pose_input, true );

		pose_sse_threebody( pose_input );
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

typedef std::pair< int, int> Segment;
typedef std::vector< Segment > Segment_list;

////////////////////////////////////////////////////////////////////////////////////////////////////////

void
read_segments( Segment_list & segment_list_input ){
	std::ifstream data( stringafteroption("segments").c_str() );
	if ( !data.good() ) {
		std::cout << "STOP:: cant open segments file: " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string line;
	getline(data, line);
	std::istringstream line_stream( line );

	while( !line_stream.fail() ){
		int startsegment, endsegment;
		line_stream >> startsegment >> endsegment;
		if (!line_stream.fail()) segment_list_input.push_back( std::make_pair(startsegment, endsegment) );
	}

	data.close();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
void
extract_segment( pose_ns::Pose & pose_input, pose_ns::Pose & pose,
								 Segment_list & segment_list )
{
	using namespace pose_ns;

	//How many segments are there? Copy to FArray for historical reasons.
	int const N = segment_list.size();
	FArray2D_int res_boundary( N, 2, 0 );
	FArray1D_int total_residue_segment( N, 0);

	int count = 0;
	for ( std::vector< Segment >::const_iterator it=segment_list.begin(),
					it_end = segment_list.end(); it != it_end; ++it ) {
		count++;
		res_boundary( count, 1 ) =  it->first;
		res_boundary( count, 2 ) =  it->second;
	}

	int total_residue = 0;
	for (int i = 1; i <= N; i++) {
		total_residue_segment( i ) = res_boundary( i, 2) - res_boundary( i, 1) + 1;
		total_residue += total_residue_segment(i);
	}

	//Initialize pose.
	pose.simple_fold_tree( total_residue );
	pose.set_fullatom_flag( true, false );

	//Copy in coordinates,res,resv. Actually this doesn't handle coordinates correctly.
	// but at least the amino acid types get read in correctly.
	// Probably would be cleaner to write a pose.copy_res_resv( ... ) routine.
	int new_position;
	new_position = 1;
	for (int i = 1; i <= N; i++) {
		pose.copy_segment( total_residue_segment(i), pose_input,
											 new_position, res_boundary(i,1), false );
		new_position += total_residue_segment(i);
	}
	assert( new_position == total_residue + 1);

	//OK, need a fold tree. This is currently a stupid fold tree with jumps
	// defined between the centers of adjacent segments -- probably would be
	// smarter to look for segments in contact, and put jumps between points of
	// closest contacts. To do!
	Fold_tree f( pose.fold_tree() );

	new_position = 1;
	int jump_point1 = total_residue_segment(1)/2 + new_position;
	int jump_point2,cutpoint;
	for (int i = 1; i <= N-1; i++) {
		new_position += total_residue_segment(i);
		jump_point2 = total_residue_segment(i+1)/2 + new_position;
		cutpoint = new_position - 1;
		f.new_jump( jump_point1, jump_point2, cutpoint );

		jump_point1 = jump_point2;
	}

	pose.set_fold_tree( f );


	//Copy in coordinates for real. Now the fold tree has cutpoints so
	// that pose behaves properly when "update_jumps" is activated.
	new_position = 1;
	for (int i = 1; i <= N; i++) {
		if (i>1) {
			pose.copy_segment( total_residue_segment(i), pose_input,
												 new_position, res_boundary(i,1), true );
		}
		new_position += total_residue_segment(i);
	}
	assert( new_position == total_residue + 1);

	Score_weight_map weight_map( score12 );
	pose.score( weight_map );

}

void
extract_segment_test(){
  using namespace silent_io;
  using namespace pose_ns;

  const bool fullatom( truefalseoption("fa_input") );
	Pose pose;

	std::string silent_file_name= stringafteroption("s");

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
  if ( truefalseoption("t") ) {
    tag_list.push_back( stringafteroption("t") );
	} else if ( truefalseoption("all") ) {
		tag_list = decoys.tags();
  } else if ( truefalseoption("l") ) {
    std::ifstream data( stringafteroption("l").c_str() );
    if ( !data.good() ) {
      std::cout << "STOP:: cant open tags file: " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    std::string tag;
    while ( getline(data,tag) ) {
      tag_list.push_back(tag);
    }
		data.close();
  }

	// Silent output
	unsigned int n = silent_file_name.rfind(".out");
	if (n == std::string::npos) {
      std::cout << "STOP:: silent file needs to end in .out " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::string output_file_name( silent_file_name );
	output_file_name.replace(n,4,"_segment.out");

	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );

	Silent_out out( output_file_name );

	Segment_list segment_list_input;
	read_segments( segment_list_input );

	//	int const startres1 = intafteroption("startres1", 2);
	//	int const   endres1 = intafteroption("endres1",  23);
	//	int const startres2 = intafteroption("startres2",29);
	//	int const   endres2 = intafteroption("endres2",  51);
	//		int const startres2 = 53;
	//		int const   endres2 = 63;

	//Read in native.
	Pose native_pose, native_pose_segment;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
		extract_segment( native_pose, native_pose_segment, segment_list_input);
		int const total_residue = native_pose_segment.total_residue();
		native_mapping.dimension( total_residue );
		for (int i = 1; i<=total_residue; i++) native_mapping(i) = i;
	}

	int count = 0;
	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

    if ( ! decoys.has_key( tag ) ) {
      std::cout << "couldnt find tag in silent-file: " << tag << std::endl;
      continue;
    }

		//		if ( !out.start_decoy(tag) ) continue; // already done or started

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		//Fresh pose for each decoy.
		Pose pose_input, pose_output;

    // fill the pose
    decoy.fill_pose( pose_input, true );

	  extract_segment( pose_input, pose_output, segment_list_input);

		if (native_exists) calc_homolog_rms( pose_output, native_pose_segment, native_mapping );

		//		pose_output.dump_pdb( "blah.pdb" );
		out.write( tag, pose_output );

		count++;
		if (mod(count,100) == 0) std::cout << "DECOY " << count << std::endl;
  }
}




void
pose_score_test(){
  using namespace silent_io;
  using namespace pose_ns;

  const bool fullatom( truefalseoption("fa_input") );
	Pose pose;

	initialize_query(); // for sstype, etc.

	if (truefalseoption( "test_generic_rama" )){
		std::string protein_sstype = get_protein_sstype();
		std::cout << std::endl << "SSTYPE:   " << protein_sstype << std::endl;
		if ( protein_sstype != "a" ) {
			std::cout << "STOP:: Testing generic rama, and not an all-alpha protein." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	std::string silent_file_name= stringafteroption("s");

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
  if ( truefalseoption("t") ) {
    tag_list.push_back( stringafteroption("t") );
	} else if ( truefalseoption("all") ) {
		tag_list = decoys.tags();
  } else if ( truefalseoption("l") ) {
    std::ifstream data( stringafteroption("l").c_str() );
    if ( !data.good() ) {
      std::cout << "STOP:: cant open tags file: " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    std::string tag;
    while ( getline(data,tag) ) {
      tag_list.push_back(tag);
    }
		data.close();
  }

	// Silent output
	std::string output_file_name =  files_paths_pdb_out_prefix_nochain()+".sc" ;

	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );

 //The most common name of an outfile....
	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	FArray1D_int native_mapping;
	std::string const native_file_name = files_paths::protein_name+".pdb";
	bool const native_exists = pose_from_pdb( native_pose, native_file_name, fullatom, false );
	int const total_residue = native_pose.total_residue();
	native_mapping.dimension( total_residue );
	for (int i = 1; i<=total_residue; i++) native_mapping(i) = i;

	int count = 0;
	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

    if ( ! decoys.has_key( tag ) ) {
      std::cout << "couldnt find tag in silent-file: " << tag << std::endl;
      continue;
    }

		//		if ( !out.start_decoy(tag) ) continue; // already done or started

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		//Fresh pose for each decoy.
		Pose pose;

    // fill the pose
    decoy.fill_pose( pose, true );

		if (fullatom){
			pose.score( score12 );
		} else {
			pose.score( score4 );
		}
		if (native_exists) calc_homolog_rms( pose, native_pose, native_mapping );

		//		pose_output.dump_pdb( "blah.pdb" );
		out.write_scores( tag, pose );

		count++;
		if (mod(count,100) == 0) std::cout << "DECOY " << count << std::endl;
  }
}

//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////
void
get_moment_of_inertia( FArray3Da_float Epos, int const total_residue,
											 numeric::xyzVector_float & center_of_mass, numeric::xyzMatrix_float & M ){

	using namespace numeric;

	Epos.dimension( 3, param::MAX_POS, total_residue );

	FArray2D_float pos_recenter( 3, total_residue, 0.0);
	FArray2D_float m_moment( 3, 3 );

	for (int j = 1; j <= 3; j++){
		float pos_center = 0.0;
		for (int i = 1; i <= total_residue; i++) pos_center += Epos( j, 2, i );
		pos_center /= float( total_residue );

		center_of_mass(j) = pos_center;
		for (int i = 1; i <= total_residue; i++) pos_recenter(j, i) = Epos( j, 2, i ) - pos_center;
	}

	// Generate inertia tensor:
	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			float cross_term = 0.0;
			for ( int i = 1; i <= total_residue; ++i ) {
				cross_term -= pos_recenter(j, i) * pos_recenter(k, i);
			}
			m_moment(k,j) = cross_term;
		}
	}

	//	std::cout << "blah.. " << m_moment(1,1) << std::endl;

	float diagonal_term( 0.0 );
	for (int  i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			diagonal_term += pos_recenter(j, i) * pos_recenter(j, i);
		}
	}
	for ( int j = 1; j <= 3; ++j ) {
		m_moment(j,j) += diagonal_term;
	}


	M =  xyzMatrix_float::cols( &m_moment( 1,1 ) );

	M /= total_residue;

	//	std::cout << M << std::endl;
	//	std::cout << center_of_mass << std::endl;
}

void
set_rotation_magnitudes_and_axes(
		numeric::xyzMatrix_float const M,
		float const desired_rms,
		float & rot1, float & rot2, float & rot3,
		numeric::xyzVector_float & axis1,
		numeric::xyzVector_float & axis2,
		numeric::xyzVector_float & axis3
)
{
	using namespace numeric;

	xyzVector_float xyz_w_w;
	xyzMatrix_float xyz_eVec;

	float const tol( 0.000001) ;
	xyz_w_w = eigenvector_jacobi( M, tol, xyz_eVec );
	xyz_eVec.transpose();

	//	std::cout << "EIGENVALUES: " << xyz_w_w << std::endl;
	//	std::cout << "EIGENVECTORS: " << xyz_eVec << std::endl;

	rot1 = desired_rms / std::sqrt(xyz_w_w(1));
	rot2 = desired_rms / std::sqrt(xyz_w_w(2));
	rot3 = desired_rms / std::sqrt(xyz_w_w(3));

	//	std::cout << "ROT MAGNITUDES : " << rot1 << " " << rot2 << " " << rot3 << std::endl;

	axis1 = &xyz_eVec(1,1);
	axis2 = &xyz_eVec(2,1);
	axis3 = &xyz_eVec(3,1);

	//	std::cout << "AXIS1 " << axis1  << std::endl;
	//	std::cout << "AXIS2 " << axis2  << std::endl;
	//	std::cout << "AXIS3 " << axis3  << std::endl;

}

void
move_coord(
				FArray2Da_float Epos_new,
				numeric::xyzMatrix_float M,
				numeric::xyzVector_float translation,
				numeric::xyzVector_float center_of_mass
){
	using namespace numeric;

	xyzVector_float r ( & Epos_new(1,1) );

	r  -= center_of_mass;

	r = M * r ;
	r  += center_of_mass;
	r  += translation;

	for (int k = 1; k<=3; k++) Epos_new(k,1) = r(k);
}

void
rotate_and_translate(
				FArray2Da_float Epos_new,
				FArray2Da_float Epos,
				numeric::xyzMatrix_float M,
				numeric::xyzVector_float translation,
				numeric::xyzVector_float center_of_mass
){
	using namespace numeric;

	Epos.dimension( 3, param::MAX_POS);
	Epos_new.dimension( 3, param::MAX_POS);

  // Do the rotation manually:
	for (int j = 1; j <= param::MAX_POS; j++){
		for (int k = 1; k <= 3; k++){
			Epos_new(k,j) = Epos(k,j);
		}

		move_coord( Epos_new(1,j), M, translation, center_of_mass );
	}
}

void
rotate_and_translate(
				FArray2Da_float Epos_new,
				FArray2Da_float Epos,
				numeric::xyzVector_float omega,
				numeric::xyzVector_float translation,
				numeric::xyzVector_float center_of_mass
){
	using namespace numeric;

	xyzMatrix_float M =
		rotation_matrix(  omega , omega.length() );

	rotate_and_translate( Epos_new, Epos, M, translation, center_of_mass );
}

void
apply_rotation_around_center_of_mass( pose_ns::Pose & pose,
								numeric::xyzVector_float omega,
								numeric::xyzVector_float translation,
								numeric::xyzVector_float center_of_mass,
								int const whichjump = 1)
{

	using namespace numeric;
	using namespace pose_ns;

	int const jump_point1 = pose.fold_tree().upstream_jump_residue( whichjump );
	int const jump_point2 = pose.fold_tree().downstream_jump_residue( whichjump );

	FArray3D_float Epos = pose.Eposition();
	FArray3D_float Epos_new(3, param::MAX_POS, 1, 0.0);

	rotate_and_translate( Epos_new(1,1,1),
												Epos(1,1,jump_point2), omega, translation, center_of_mass );

	Jump newjump( Epos(1,1,jump_point1), Epos_new(1,1,1));
	pose.set_jump( whichjump, newjump );

}


void
move_segment( pose_ns::Pose & pose,
							int const startres,
							int const endres,
							numeric::xyzMatrix_float M,
							numeric::xyzVector_float translation,
							numeric::xyzVector_float center_of_mass)
{
	// This can be done faster, by just recalculating new Epositions for jump residues.
	using namespace numeric;
	using namespace pose_ns;
	using namespace aaproperties_pack;
	using namespace kin;

	Atom_tree const * atom_tree = pose.atom_tree();
	if ( atom_tree ){
		//Although the following was easy to write, it
		// is spectacularly inefficient -- doesn't remember
		// any prior computations of pair energies, etc.
		/////////////////
		//Atom tree
		/////////////////

		pose.copy_to_misc();

		//Move full_coord around
		for (int i = startres; i <= endres; i++){
			int const numatoms = natoms( misc::res(i), misc::res_variant(i) );
			for (int j = 1; j <= numatoms ; j++){
				move_coord( misc::full_coord(1, j, i), M, translation, center_of_mass );
			}
			for (int j = 1; j <= param::MAX_POS; j++){
				move_coord( misc::Eposition(1, j, i), M, translation, center_of_mass );
			}
		}

		pose_from_misc( pose, true /*fullatom*/, false /*ideal_pos*/,
										true /*coords_init*/ ); //retains constraints?
		pose.setup_atom_tree(); //necessary?

		return;
	}

	/////////////////
	//Usual fold tree
	/////////////////
	FArray3D_float Epos = pose.Eposition();
	//Scratch coordinates for determining new jumps.
	FArray3D_float Epos_new(3, param::MAX_POS, 2, 0.0);

	int const num_jumps = pose.num_jump();
	for (int n = 1; n <= num_jumps; n++){
		int const jump_point1 = pose.fold_tree().upstream_jump_residue( n );
		int const jump_point2 = pose.fold_tree().downstream_jump_residue( n );

		for (int j = 1; j <= param::MAX_POS; j++){
			for (int k = 1; k <= 3; k++){
				Epos_new(k,j,1) = Epos(k,j,jump_point1);
				Epos_new(k,j,2) = Epos(k,j,jump_point2);
			}
		}

		bool jump_changed = false;
		if (jump_point1 >= startres && jump_point1 <= endres){
			rotate_and_translate( Epos_new(1,1,1),
														Epos(1,1,jump_point1), M, translation, center_of_mass );
			jump_changed = true;
		}

		if (jump_point2 >= startres && jump_point2 <= endres){
			rotate_and_translate( Epos_new(1,1,2),
														Epos(1,1,jump_point2), M, translation, center_of_mass );
			jump_changed = true;
		}

		if (jump_changed){
			Jump newjump;

			newjump = Jump( Epos_new(1,1,1), Epos_new(1,1,2));

			// This was my early attempt to update atom_tree jumps in a clever way.
			// Unfortunately, doesn't really seem to work.
			//			if (atom_tree){
			//				//Note that following assumes a particular upstream anchor atom (C), and
			//				// downstream root atom (N). Only valid is fold_tree has these explicitly
			//				// set before setup_atom_tree.
			//				numeric::xyzVector_double x1( & Epos_new(1,1,1) );
			//				numeric::xyzVector_double y1( & Epos_new(1,2,1) );
			//				numeric::xyzVector_double z1( & Epos_new(1,4,1) );
			//
			//				numeric::xyzVector_double x2( & Epos_new(1,1,2) );
			//				numeric::xyzVector_double y2( & Epos_new(1,2,2) );
			//				numeric::xyzVector_double z2( & Epos_new(1,4,2) );
			//
			//				// First  stub: upstream jump atom, its parent, and its parent's parent.
			//				kin::Stub     stub( z1, z1, y1, x1 );
			//				// Second stub: downstream jump atom, its first child, and its first child's first child
			//				kin::Stub new_stub( x2, x2, y2, z2 );
			//				newjump.from_stubs( stub, new_stub);
			//
			//			}

			pose.set_jump( n, newjump );
		}
	}
		//	} // atom tree?

}



void
move_segment( pose_ns::Pose & pose,
							int const startres,
							int const endres,
							numeric::xyzVector_float omega,
							numeric::xyzVector_float translation,
							numeric::xyzVector_float center_of_mass)
{
	using namespace numeric;

	xyzMatrix_float M =
		rotation_matrix(  omega , omega.length() );

	move_segment( pose, startres, endres, M, translation, center_of_mass );
}

////////////////////////////////////////////////////////////////////////////////////
void
rotate_translate_segment( pose_ns::Pose & pose, int const num_segment,
													float const l, float const m, float const n,  //translation
													float const i, float const j, float const k   //rotation
){
	using namespace numeric;
	using namespace pose_ns;

	//Which residues need to be rotated -- need to be bracketed by cutpoints!
	int const nres = pose.total_residue();
	int numcuts = 0;
	int startres( 1 ), endres( nres );
	for (int q = 1; q <= nres; q ++){
		if ( pose.is_cutpoint(q)) {
			numcuts++;
			if (numcuts >= num_segment) {
				endres = q;
				break;
			}
			startres = q+1;
		}

	}

	//	std::cout << "SEGMENT TO ROTATE: " << startres << "-" << endres << std::endl;

	//Center of mass and moment of inertia of second object.
	xyzVector_float center_of_mass( 0.0 );
	xyzMatrix_float M( 0.0 );
	FArray3D_float const Epos( pose.Eposition() );
	get_moment_of_inertia( Epos(1,1,startres), endres - startres + 1, center_of_mass, M );

	// How much to rotate?
	static float const desired_rms = realafteroption( "desired_rms", 1.0);
	float rot1,rot2,rot3;
	xyzVector_float axis1,axis2,axis3;
	set_rotation_magnitudes_and_axes( M, desired_rms, rot1, rot2, rot3, axis1, axis2, axis3 );

	xyzVector_float const omega = rot1*i*axis1  + rot2*j*axis2  + rot3*k*axis3;
	xyzVector_float translation(  l, m, n);
	translation *= desired_rms;

	move_segment( pose, startres, endres, omega, translation,
								center_of_mass);

}


//////////////////////////////////////////////////////////////////////////////////
void rhiju_gridsearch( pose_ns::Pose & start_pose, pose_ns::Pose & native_pose,
											 silent_io::Silent_out & out, int const whichjump = 1){
	using namespace pose_ns;
	using namespace numeric;
	using namespace silent_io;


	set_default_atomvdw( "highres" );

	bool const fullatom = truefalseoption("fullatom");

  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		setup_fullatom_options();
		weight_map = fullatom_weight_map;
	}	else {
		weight_map = centroid_weight_map;
	}

	//Just do a configuration with two helices for now.

	//Create a starting pose with appropriate fold tree.
  Fold_tree f( start_pose.fold_tree() );
	int jump_point1  = f.upstream_jump_residue( whichjump );
	int jump_point2 = f.downstream_jump_residue( whichjump );
	//	int cutpoint = f.cutpoint_by_jump( whichjump );

	int const total_residue = start_pose.total_residue();

	//Center of mass and moment of inertia of second object.
	xyzVector_float center_of_mass( 0.0 );
	xyzMatrix_float M( 0.0 );
	int count = 0;

	FArray3D_float Epos = start_pose.Eposition();

	//What is the segment to rotate?
	int startres, endres;
	for (startres = jump_point2; startres > 1; startres--) {
		if (start_pose.is_cutpoint( startres-1 ) ) break;
	}
	for (endres = jump_point2; endres <= total_residue; endres++) {
		if (start_pose.is_cutpoint( endres ) ) break;
	}
	get_moment_of_inertia( Epos(1,1,startres), endres - startres + 1, center_of_mass, M );
	std::cout << "MOMENT OF INERTIA RESIDUES: " << startres << " " << endres <<
		"; JUMP POINTS " << jump_point1 << " " << jump_point2 << std::endl;

	// How much to rotate?
	float const desired_rms = realafteroption( "desired_rms", 1.0);
	float rot1,rot2,rot3;
	xyzVector_float axis1,axis2,axis3;
	set_rotation_magnitudes_and_axes( M, desired_rms, rot1, rot2, rot3, axis1, axis2, axis3 );

	count = 0;
	start_pose.dump_pdb( "START.pdb");

	if (!fullatom) {
		start_pose.set_fullatom_flag( false, false );
		native_pose.set_fullatom_flag( false, false );
	}


	std::string tag = "NATIVE";
	native_pose.set_native_pose( native_pose );
	native_pose.score( weight_map );
	calc_rms( native_pose );
	native_pose.dump_pdb( tag+".pdb" );
	out.write( tag, native_pose );

	tag = "S_"+lead_zero_string_of(count++,3);
	start_pose.set_native_pose( native_pose );
	start_pose.score( weight_map );
	calc_rms( start_pose );
	start_pose.dump_pdb( tag+".pdb" );
	out.write( tag, start_pose );

	for (float i = -1.001; i <= 1; i++) {
		for (float j = -1; j <= 1; j++) {
			for (float k = -1; k <= 1; k++) {
				for (float l = -1; l <= 1; l++) {
					for (float m = -1; m <= 1; m++) {
						for (float n = -1; n <= 1; n++) {
							Pose pose;
							pose = start_pose;
							if (fullatom)	pose.set_fullatom_flag( true, false );

							xyzVector_float const omega = rot1*i*axis1  + rot2*j*axis2  + rot3*k*axis3;
							xyzVector_float translation(  l, m, n);
							translation *= desired_rms;

							apply_rotation_around_center_of_mass( pose, omega, translation,
																										center_of_mass, whichjump);

// 							if (fullatom)	{
// 								pose.set_fullatom_flag( true, false );
// 								for (int q=1; q<=cycles; q++){
// 									bool use_input_rotamer = false;
// 									if (q > 1) use_input_rotamer = true;
// 									pose.full_repack( use_input_rotamer );
// 									pose.main_minimize( weight_map, "dfpmin" );
// 								}
// 							}

// 							pose.set_native_pose( native_pose );
// 							calc_rms( pose );
// 							pose.score( weight_map );
// 							std::cout << pose.show_scores() << std::endl;

 							tag = "S_"+lead_zero_string_of(count++,3);
							minimize_and_output( pose, native_pose, tag, out, false /*output start*/);
// 							pose.dump_pdb( tag+".pdb" );
// 							out.write( tag, pose );

						}
					}
				}
			}
		}
	}

}

//////////////////////////////////////////////////////////////////////////////////

void rhiju_relax( pose_ns::Pose & start_pose, pose_ns::Pose & native_pose,
									silent_io::Silent_out out, std::string const input_tag){
	using namespace pose_ns;
	using namespace numeric;
	using namespace silent_io;

	set_default_atomvdw( "highres" );

	bool const fullatom = true;

  Score_weight_map centroid_weight_map( score3 ), fullatom_weight_map( score12 ), weight_map;
	if (fullatom) {
		setup_fullatom_options();
		weight_map = fullatom_weight_map;
	}	else {
		weight_map = centroid_weight_map;
	}

	static bool init = false;
	std::string tag;
	if (!init){
// 		tag = "NATIVE";
// 		native_pose.set_native_pose( native_pose );
// 		native_pose.score( weight_map );
// 		calc_rms( native_pose );
// 		native_pose.dump_pdb( tag+".pdb" );
// 		out.write( tag, native_pose );
 		init = true;
	}

	tag = input_tag;
	start_pose.set_native_pose( native_pose );
	start_pose.score( weight_map );
	calc_rms( start_pose );
	start_pose.dump_pdb( tag+".pdb" );
	out.write( tag, start_pose );

	Pose pose;
	pose = start_pose;
	if (fullatom)	pose.set_fullatom_flag( true, false );

	int const cycles = intafteroption("cycles",10);
	for (int q=1; q <= cycles; q++) {
		bool use_input_rotamer = false;
		if (q > 1) use_input_rotamer = true;
		pose.full_repack( use_input_rotamer );
		pose.main_minimize( weight_map, "dfpmin" );
	}

	tag = input_tag+"_relax";
	pose.dump_pdb( tag+".pdb" );
	out.write( tag, pose );

}


//////////////////////////////////////////////////////////////////////////////////
void rhiju_relax_test( std::string mode ){
  using namespace silent_io;
  using namespace pose_ns;

	bool const fullatom = true;


	std::string silent_file_name= stringafteroption("s");

  // read silent file
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
	tag_list = decoys.tags();

	// Silent output
	std::string output_file_name( "blah_"+mode+".out" );
	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}

	int count = 0;
	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

		//		if (tag == "NATIVE") continue;

    if ( ! decoys.has_key( tag ) ) {
      std::cout << "couldnt find tag in silent-file: " << tag << std::endl;
      continue;
    }

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		Pose pose;

    // fill the pose
    decoy.fill_pose( pose, true );

		if (mode == "relax") {
			rhiju_relax( pose, native_pose, out, tag );
		}	else {
			rhiju_gridsearch( pose, native_pose, out );
		}
		count++;
		if (mod(count,100) == 0) std::cout << "DECOY " << count << std::endl;
  }


}

//////////////////////////////////////////////////////////////////////////////
// In this old version I looked for points of closest contact. But I don't think
// that's actually a good metric for a physically reasonable fold tree.
// Instead look for the pairs of rigid bodies making the *most* contacts.
// void
// figure_out_fold_tree( pose_ns::Pose & pose, bool const verbose = true ){
// 	using namespace pose_ns;
// 	using namespace numeric;

// 	int const total_residue = pose.total_residue();

// 	//Get rid of previous foldtree. Might want to be
// 	// more flexible in the future...
// 	Fold_tree f;
// 	f.simple_tree( total_residue );

// 	//Identify cutpoints
// 	FArray3D_float Epos = pose.Eposition();
// 	std::vector<int> cutpoint_list;
// 	float const CUTOFF = 4.0;
// 	for (int i = 1; i <= total_residue-1; i++ ){
// 		xyzVector_float Epos1( &Epos(1,2,i) );
// 		xyzVector_float Epos2( &Epos(1,2,i+1) );
// 		if ( (Epos1 - Epos2).length() > CUTOFF) {
// 			cutpoint_list.push_back( i );
// 			if (verbose) std::cout << " Found a cutpoint: " << i << std::endl;
// 		}
// 	}

// 	int const num_cutpoints = cutpoint_list.size();
// 	if (verbose) std::cout << " Total number of cutpoints: " << num_cutpoints << std::endl;

// 	if (num_cutpoints > 0){
// 		//Convert list of cutpoints to an FArray, easier to index.
// 		FArray1D_int cutpoints( num_cutpoints );
// 		int n = 1;
// 		for ( std::vector< int >::const_iterator it=cutpoint_list.begin(),
// 						it_end = cutpoint_list.end();
// 					it != it_end; it++,n++ ) cutpoints(n) = *it;

// 		//Define segments
// 		int const num_segments = num_cutpoints+1;
// 		FArray2D_int segment_boundary( num_segments, 2, 0 );
// 		for (int i = 1; i<= num_segments; i++ ){
// 			int startpoint = 1;
// 			if (i>1) startpoint = cutpoints(i-1)+1;
// 			int endpoint = total_residue;
// 			if (i < num_segments) endpoint = cutpoints(i);
// 			segment_boundary( i, 1) = startpoint;
// 			segment_boundary( i, 2) = endpoint;
// 			if (verbose) std::cout << "Segment boundary: " << segment_boundary(i,1) << "-" << segment_boundary(i,2) << std::endl;
// 		}

// 		//Find potential jump points.
// 		// Start by defining closest distances between segments,
// 		//  and points of contact.
// 		FArray2D_float contact_dist( num_segments, num_segments, 999999.9);
// 		FArray3D_int contact_points( num_segments, num_segments, 2, 0);
// 		for (int i = 1; i <= num_segments; i++ ){
// 			for (int j = i+1; j <= num_segments; j++ ){

// 				for (int pos1 = segment_boundary(i,1); pos1 <= segment_boundary(i,2); pos1++){
// 					xyzVector_float Epos1( &Epos(1,2,pos1) );
// 					for (int pos2 = segment_boundary(j,1); pos2 <= segment_boundary(j,2); pos2++){
// 						xyzVector_float Epos2( &Epos(1,2,pos2) );
// 						float const distance  = (Epos1-Epos2).length();
// 						if (distance  < contact_dist(i,j) ){
// 							contact_dist(i,j) = distance;
// 							contact_points(i,j,1) = pos1;
// 							contact_points(i,j,2) = pos2;
// 						}
// 					} // pos2
// 				} // pos1
// 					//Symmetrize matrix
// 				contact_dist(j,i) = contact_dist(i,j);
// 				contact_points(j,i,1) = contact_points(i,j,1);
// 				contact_points(j,i,2) = contact_points(i,j,2);
// 				if (verbose) std::cout << "Distance between segments " << i << " and " << j << " : " << contact_dist(i,j) <<std::endl;
// 			} // j
// 		} // i

// 		//Sort-of neighbor joining...
// 		FArray2D_int jumppoints( 2, num_cutpoints);
// 		FArray1D_int segments_so_far( num_segments, 0);
// 		FArray1D_bool segment_in_tree( num_segments, false);

// 		//To start the tree, pick the two segments with the closest contact.
// 		float distance = 9999999.9;
// 		int connection_point( 0 ), new_segment( 0 );
// 		for (int i = 1; i <= num_segments; i++ ){
// 			for (int j = i+1; j <= num_segments; j++ ){
// 				if (contact_dist(i,j) < distance){
// 					distance = contact_dist(i,j);
// 					connection_point = i;
// 					new_segment = j;
// 				}
// 			}
// 		}
// 		segments_so_far(1) = connection_point;
// 		segments_so_far(2) = new_segment;
// 		segment_in_tree( connection_point ) = true;
// 		segment_in_tree( new_segment ) = true;
// 		jumppoints( 1, 1) = contact_points( connection_point, new_segment, 1);
// 		jumppoints( 2, 1) = contact_points( connection_point, new_segment, 2);;

// 		//Now agglomerate the remaining segments into this tree, one-by-one
// 		for (int count = 2; count<=num_cutpoints; count++){
// 			float distance = 999999999.9;
// 			int connection_point = 0;
// 			int new_segment = 0;
// 			for (int i = 1; i <= count; i++){
// 				int const current_segment = segments_so_far(i);
// 				for (int j = 1; j <= num_segments; j++){
// 					if (segment_in_tree(j)) continue;
// 					if (contact_dist(i,j) < distance){
// 						distance = contact_dist(i,j);
// 						new_segment = j;
// 						connection_point = current_segment;
// 					}
// 				}
// 			}
// 			segment_in_tree( count+1 ) = new_segment;
// 			segment_in_tree( new_segment ) = true;
// 			jumppoints( 1, count ) = contact_points( connection_point, new_segment, 1);
// 			jumppoints( 2, count ) = contact_points( connection_point, new_segment, 2);;
// 		}

// 		if (truefalseoption("hack_tree")){
// 			jumppoints( 1, 1) = intafteroption( "jump1_upstream",contact_points( 1, 2, 1));
// 			jumppoints( 2, 1) = intafteroption( "jump1_downstream",contact_points( 1, 2, 2));

// 			jumppoints( 1, 2) = intafteroption( "jump2_upstream",contact_points( 2, 3, 1));
// 			jumppoints( 2, 2) = intafteroption( "jump2_downstream",contact_points( 2, 3, 2));
// 		}


// 		//Need to reorder jumppoints to be in sequential order, or tree_from_jumps_and_cuts
// 		// gets confused.
// 		for (int count = 1; count <= num_cutpoints; count++ ){
// 			if (jumppoints(1, count) > jumppoints(2, count) ) {
// 				int tmp = jumppoints(1, count);
// 				jumppoints(1, count) = jumppoints(2, count);
// 				jumppoints(2, count) = tmp;
// 			}
// 			if (verbose) std::cout << "Want a jump from " << jumppoints(1, count) << " to " << jumppoints(2, count) << std::endl;
// 		}

// 		//		f.set_allow_reorder( false );
// 		f.tree_from_jumps_and_cuts( total_residue, num_cutpoints, jumppoints, cutpoints );
// 	}

// 	pose.set_fold_tree( f );

// }

////////////////////////////////////////////////////////////////////////////
int
find_middle_of_vector( std::vector<int> v){
	//stupid C++.
	std::vector<int>::const_iterator it;
	std::sort( v.begin(), v.end() );
	int const numpoints = v.size();
	int const halfway = numpoints/2;
	int i = 1;
	for (it = v.begin();i<=halfway; it++,i++);
	return *it;
}

////////////////////////////////////////////////////////////////////////////
void
goofy_fold_tree( pose_ns::Pose & pose ){
	using namespace pose_ns;

	//Read in from file
	std::string pairing_file = stringafteroption( "goofy_fold_tree", "goofy_fold_tree.jumps" );

	utility::io::izstream pairing_stream( pairing_file );
	if ( !pairing_stream ) {
		std::cout << "can't open pairings file!!!" << pairing_file << std::endl;
		pairing_stream.close();
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int const total_residue = pose.total_residue();
	int const MAX_JUMPS = 100;
	FArray2D_int jump_points( 2, MAX_JUMPS);
	FArray1D_int cutpoints( MAX_JUMPS );

	std::string line;
	int pos1,pos2,cutpoint;
	int num_cutpoints( 0 );
	while ( getline( pairing_stream, line ) ) {
		std::istringstream line_stream( line );
		line_stream >> pos1 >> pos2 >> cutpoint;

		num_cutpoints++;
		jump_points(1,num_cutpoints) = pos1;
		jump_points(2,num_cutpoints) = pos2;
		cutpoints( num_cutpoints) = cutpoint;
	}

	Fold_tree f;
	f.simple_tree( total_residue );
	if (num_cutpoints > 0)
		f.tree_from_jumps_and_cuts( total_residue, num_cutpoints, jump_points, cutpoints );
	pose.set_fold_tree( f );

	pairing_stream.close();

}

////////////////////////////////////////////////////////////////////////////
void
figure_out_fold_tree( pose_ns::Pose & pose, bool const verbose = true ){
	using namespace pose_ns;
	using namespace numeric;

	if (truefalseoption("goofy_fold_tree")) {
		goofy_fold_tree( pose );
		return;
	}

	int const total_residue = pose.total_residue();

	//Get rid of previous foldtree. Might want to be
	// more flexible in the future...
	Fold_tree f;
	f.simple_tree( total_residue );

	//Identify cutpoints
	FArray3D_float Epos = pose.Eposition();
	std::vector<int> cutpoint_list;
	float const CUTOFF = 4.0;
	for (int i = 1; i <= total_residue-1; i++ ){
		xyzVector_float Epos1( &Epos(1,2,i) );
		xyzVector_float Epos2( &Epos(1,2,i+1) );
		if ( (Epos1 - Epos2).length() > CUTOFF) {
			cutpoint_list.push_back( i );
			if (verbose) std::cout << " Found a cutpoint: " << i << std::endl;
		}
	}

	int const num_cutpoints = cutpoint_list.size();
	if (verbose) std::cout << " Total number of cutpoints: " << num_cutpoints << std::endl;

	if (num_cutpoints > 0){
		//Convert list of cutpoints to an FArray, easier to index.
		FArray1D_int cutpoints( num_cutpoints );
		int n = 1;
		for ( std::vector< int >::const_iterator it=cutpoint_list.begin(),
						it_end = cutpoint_list.end();
					it != it_end; it++,n++ ) cutpoints(n) = *it;

		//Define segments
		int const num_segments = num_cutpoints+1;
		FArray2D_int segment_boundary( num_segments, 2, 0 );
		for (int i = 1; i<= num_segments; i++ ){
			int startpoint = 1;
			if (i>1) startpoint = cutpoints(i-1)+1;
			int endpoint = total_residue;
			if (i < num_segments) endpoint = cutpoints(i);
			segment_boundary( i, 1) = startpoint;
			segment_boundary( i, 2) = endpoint;
			if (verbose) std::cout << "Segment boundary: " << segment_boundary(i,1) << "-" << segment_boundary(i,2) << std::endl;
		}

		//Use centroid-centroid distances to determine contacts.
		pose_update_cendist( pose ); //may not be necessary if computing VDW anyway.
		FArray2D_float const & cendist( pose.get_2D_score( CENDIST ) );

		//Find potential jump points.
		//Start by defining residues making contacts.
		//  Pairs of segments with largest interfaces (most residues in contact)
		//   will be favored to be connected by jumps.
		//  Jumps will connect residues that are "in the middle" of each interface.
		FArray2D_int num_contact_pairs( num_segments, num_segments, 0);
		FArray3D_int potential_jump_points( 2, num_segments, num_segments, 0);
		float const CUTOFF_DISTANCE( 8.0*8.0 );

		for (int i = 1; i <= num_segments; i++ ){
			for (int j = i+1; j <= num_segments; j++ ){

				int temp_num_contact_pairs( 0 );
				std::vector< int > interface_points_i;
				std::vector< int > interface_points_j;

				//Just to avoid problems with extended structures, start out with 1 "contact pair"
				// between the middles of the two segments.
				temp_num_contact_pairs++;
				interface_points_i.push_back(  (segment_boundary(i,1) + segment_boundary(i,2))/2 );
				interface_points_j.push_back(  (segment_boundary(j,1) + segment_boundary(j,2))/2 );

				for (int pos1 = segment_boundary(i,1); pos1 <= segment_boundary(i,2); pos1++){
					for (int pos2 = segment_boundary(j,1); pos2 <= segment_boundary(j,2); pos2++){
						float const distance  = cendist(pos1, pos2);
						if (distance  < CUTOFF_DISTANCE ){
							temp_num_contact_pairs++;
							interface_points_i.push_back( pos1 );
							interface_points_j.push_back( pos2 );
						}
					} // pos2
				} // pos1

				//How we doing? Keep things symmetrized.
				num_contact_pairs( i, j) = temp_num_contact_pairs;
				num_contact_pairs( j, i) = temp_num_contact_pairs;

				potential_jump_points(1, i, j) = find_middle_of_vector( interface_points_i );
				potential_jump_points(2, i, j) = find_middle_of_vector( interface_points_j );
				//upstream jump point > downstream jump point.
				potential_jump_points(1, j, i) = find_middle_of_vector( interface_points_i );
				potential_jump_points(2, j, i) = find_middle_of_vector( interface_points_j );

				if (verbose) std::cout << "Number of contact pairs " << i << " and " << j << " : " << num_contact_pairs(i,j) <<std::endl;
			} // j
		} // i

		//Sort-of neighbor joining...
		FArray2D_int jump_points( 2, num_cutpoints);
		FArray1D_int segments_so_far( num_segments, 0);
		FArray1D_bool segment_in_tree( num_segments, false);

		//To start the tree, pick the two segments with the most extesnive interface.
		int num_contact_pairs_best = 0;
		int connection_point( 0 ), new_segment( 0 );
		for (int i = 1; i <= num_segments; i++ ){
			for (int j = i+1; j <= num_segments; j++ ){
				if (num_contact_pairs(i,j) > num_contact_pairs_best){
					num_contact_pairs_best = num_contact_pairs(i,j);
					connection_point = i;
					new_segment = j;
				}
			}
		}
		segments_so_far(1) = connection_point;
		segments_so_far(2) = new_segment;
		segment_in_tree( connection_point ) = true;
		segment_in_tree( new_segment ) = true;
		jump_points( 1, 1) = potential_jump_points( 1, connection_point, new_segment);
		jump_points( 2, 1) = potential_jump_points( 2, connection_point, new_segment);;

		//Now agglomerate the remaining segments into this tree, one-by-one
		for (int count = 2; count<=num_cutpoints; count++){
			int num_contact_pairs_best = 0;
			int connection_point = 0;
			int new_segment = 0;
			for (int i = 1; i <= count; i++){
				int const current_segment = segments_so_far(i);
				for (int j = 1; j <= num_segments; j++){
					if (segment_in_tree(j)) continue;
					if (num_contact_pairs(i,j) > num_contact_pairs_best){
						num_contact_pairs_best = num_contact_pairs(i,j);
						new_segment = j;
						connection_point = current_segment;
					}
				}
			}
			segments_so_far( count+1 ) = new_segment;
			segment_in_tree( new_segment ) = true;
			jump_points( 1, count ) = potential_jump_points( 1, connection_point, new_segment );
			jump_points( 2, count ) = potential_jump_points( 2, connection_point, new_segment );;
		}

		if (truefalseoption("hack_tree")){
			jump_points( 1, 1) = intafteroption( "jump1_upstream",potential_jump_points( 1, 1, 2));
			jump_points( 2, 1) = intafteroption( "jump1_downstream",potential_jump_points( 2, 1, 2));

			jump_points( 1, 2) = intafteroption( "jump2_upstream",potential_jump_points( 1, 2, 3));
			jump_points( 2, 2) = intafteroption( "jump2_downstream",potential_jump_points( 2, 2, 3));
		}


		//Need to reorder jump_points to be in sequential order, or tree_from_jumps_and_cuts
		// gets confused.
		for (int count = 1; count <= num_cutpoints; count++ ){
			if (jump_points(1, count) > jump_points(2, count) ) {
				int tmp = jump_points(1, count);
				jump_points(1, count) = jump_points(2, count);
				jump_points(2, count) = tmp;
			}
			if (verbose) std::cout << "Want a jump from " << jump_points(1, count) << " to " << jump_points(2, count) << std::endl;
		}

		//		f.set_allow_reorder( false );
		f.tree_from_jumps_and_cuts( total_residue, num_cutpoints, jump_points, cutpoints );
	}

	pose.set_fold_tree( f );

}


////////////////////////////////////////////////////////////////////////////
void
reorder_axes( numeric::xyzMatrix_float & xyz_eVec,
							numeric::xyzVector_float & xyz_w_w){

	using namespace numeric;

	int long_axis_num = 1;
	if (xyz_w_w(2) <  xyz_w_w(1)) {
		long_axis_num = 2;
		if (xyz_w_w(3) <  xyz_w_w(2)) {
			long_axis_num = 3;
		}
	}

	int short_axis_num = 1;
	if (xyz_w_w(2) >  xyz_w_w(1)) {
		short_axis_num = 2;
		if (xyz_w_w(3) >  xyz_w_w(2)) {
			short_axis_num = 3;
		}
	}

	assert( short_axis_num != long_axis_num );
	int const medium_axis_num = 6 - short_axis_num - long_axis_num;

	std::cout << "Axis assignments: " << long_axis_num << " " << medium_axis_num << " " <<short_axis_num << std::endl;


	xyzVector_float longvec   (& xyz_eVec( long_axis_num, 1) );
	xyzVector_float mediumvec (& xyz_eVec( medium_axis_num, 1));
	xyzVector_float shortvec  (& xyz_eVec( short_axis_num, 1));

  xyzMatrix_float temp = xyzMatrix_float::rows_constructor(
							longvec, mediumvec, shortvec);

	xyz_eVec = temp;

}

////////////////////////////////////////////////////////////////////////////
void
align_one_segment( pose_ns::Pose & pose, pose_ns::Pose & src_pose,
									 int const startres, int const endres ){

	using namespace pose_ns;
	using namespace numeric;

	int const num_rotate_residues = endres - startres + 1;

	//setup some poses to do scratchwork.
	Pose mini_pose, mini_src_pose;
	mini_pose.simple_fold_tree( num_rotate_residues);
	mini_pose.set_fullatom_flag( pose.fullatom(), false );
	mini_pose.copy_segment( num_rotate_residues, pose, 1, startres,
													false/*update jumps*/ );

	mini_src_pose.simple_fold_tree( num_rotate_residues);
	mini_src_pose.set_fullatom_flag( src_pose.fullatom(), false );
	mini_src_pose.copy_segment( num_rotate_residues, src_pose, 1, startres,
															false /*update jumps*/ );

	//find the orientation.
	mini_pose.CA_orient( mini_src_pose ) ;

	//OK, update the pose of interest.
	bool update_jumps = true;
	//	if (startres == 1) update_jumps = false;

	//	if (alpha < 1.0){
	//		pose.copy_segment( num_rotate_residues, mini_pose, startres, 1, update_jumps );
	//
	//		//Attempt to "morph" at intermediate values of alpha.
	//		xyzVector_float center_of_mass_init( 0.0 );
	//		xyzMatrix_float M_init( 0.0 );
	//		FArray3D_float const Epos_init( pose.Eposition() );
	//		get_moment_of_inertia( Epos_init(1,1,startres), endres - startres + 1, center_of_mass_init, M_init );
	//
	//		float const tol( 0.000001) ;
	//		xyzVector_float xyz_w_w_init;
	//		xyzMatrix_float xyz_eVec_init;
	//		xyz_w_w_init = eigenvector_jacobi( M_init, tol, xyz_eVec_init );
	//		xyz_eVec_init.transpose();
	//
	//		xyzVector_float center_of_mass_align( 0.0 );
	//		xyzMatrix_float M_align( 0.0 );
	//		FArray3D_float const Epos_align( align_pose.Eposition() );
	//		get_moment_of_inertia( Epos_align(1,1,startres), endres - startres + 1, center_of_mass_align, M_align );
	//
	//		xyzVector_float xyz_w_w_align;
	//		xyzMatrix_float xyz_eVec_align;
	//		xyz_w_w_align = eigenvector_jacobi( M_align, tol, xyz_eVec_align );
	//		xyz_eVec_align.transpose();
	//
	//
	//		//Smallest moment of inertia --> longest helix axis.
	//		//Reorder axes.
	//		reorder_axes( xyz_eVec_init, xyz_w_w_init);
	//		reorder_axes( xyz_eVec_align, xyz_w_w_align);
	//		int long_axis_num = 1;
	//
	//		xyzVector_float long_axis_init  (&  xyz_eVec_init(long_axis_num,1));
	//		xyzVector_float long_axis_align (& xyz_eVec_align(long_axis_num,1));
	//		long_axis_init.normalize();
	//		long_axis_align.normalize();
	//
	//		xyzVector_float long_axis_rotation_axis   = cross(long_axis_init, long_axis_align);
	//
	//		float const long_axis_rotation_amount = std::asin(  numeric::sin_cos_range(long_axis_rotation_axis.length()) );
	//		long_axis_rotation_axis.normalize();
	//
	//		xyzMatrix_float M1 = rotation_matrix( long_axis_rotation_axis, long_axis_rotation_amount );
	//
	//		xyz_eVec_init.transpose();
	//		xyz_eVec_align.transpose();
	//		xyzMatrix_float xyz_eVec_first_rotation = M1 * xyz_eVec_init;
	//
	//		std::cout << "LONG AXIS NUM: " << long_axis_num << std::endl;
	//		std::cout << "Initial axes:    " << std::endl << xyz_eVec_init << std::endl;
	//		std::cout << "Aligned axes:     " << std::endl << xyz_eVec_align << std::endl;
	//		std::cout << "New aligned axes: " << std::endl << xyz_eVec_first_rotation << std::endl;
	//
	//
	//		xyz_eVec_first_rotation.transpose();
	//		xyz_eVec_align.transpose();
	//		int const short_axis_num = 3;
	//		xyzVector_float short_axis_init  (& xyz_eVec_first_rotation(short_axis_num,1));
	//		xyzVector_float short_axis_align (& xyz_eVec_align(short_axis_num,1));
	//		short_axis_init.normalize();
	//		short_axis_align.normalize();
	//
	//		xyzVector_float short_axis_rotation_axis   = cross(short_axis_init, short_axis_align);
	//
	//		float const short_axis_rotation_amount = std::asin(  numeric::sin_cos_range(short_axis_rotation_axis.length()) );
	//		short_axis_rotation_axis.normalize();
	//
	//		xyzMatrix_float M2 = rotation_matrix( short_axis_rotation_axis, short_axis_rotation_amount );
	//
	//		xyzMatrix_float xyz_eVec_second_rotation = M2 * M1 * xyz_eVec_init;
	//		std::cout << "Final aligned axes: " << std::endl << xyz_eVec_second_rotation << std::endl;
	//
	//		xyzMatrix_float omega = M2* M1;
	//		xyzVector_float translation = center_of_mass_align - center_of_mass_init;
	//		move_segment( pose, startres, endres, omega, translation, center_of_mass_init );
	//	} else {
	pose.copy_segment( num_rotate_residues, mini_pose, startres, 1, update_jumps );
	//	}

	std::cout << "Aligned: " << startres << " - " << endres << std::endl;

}


////////////////////////////////////////////////////////////////////////////
void
align_segments( pose_ns::Pose & pose,
								pose_ns::Pose & src_pose,
								int const n = 0 )
{
	using namespace pose_ns;

	assert( pose.total_residue() == src_pose.total_residue() );

	int const total_residue = pose.total_residue();

	int startres( 1 );
	int count( 0 );
	for (int i=1; i <= total_residue; i++){

		//Figure out what the segments are.
		if (pose.is_cutpoint(i) || i==total_residue){
			int endres = i;
			count++ ;
			if ( n==0  || n==count ){
				align_one_segment( pose, src_pose, startres, endres );
			}
			startres = i+1;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////////////////////
void copy_chi_angles( pose_ns::Pose & pose, pose_ns::Pose const & src_pose ){
	using namespace pose_ns;
	int const total_residue = pose.total_residue();
	for (int i = 1; i <= total_residue; i++){
		int const nchi( aaproperties_pack::nchi( pose.res(i), pose.res_variant(i) ) );
		for ( int chino=1; chino<= nchi; ++chino ) {
			pose.set_chi( chino, i, src_pose.chi(chino, i) );
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////
float
get_rep_delta( float const start_rep_weight, float const end_rep_weight,
							  bool const geometric_ramp, int const lj_ramp_cycles ){
	float rep_delta ( 0.0 );
	if ( geometric_ramp ) {
		rep_delta =
			( std::exp( std::log( end_rep_weight / start_rep_weight ) /
									(lj_ramp_cycles) ) );
	} else {
		rep_delta = ( end_rep_weight - start_rep_weight ) / lj_ramp_cycles;
	}
	return rep_delta;
}


////////////////////////////////////////////////////////////////////////////////////////////////
void
really_set_rep_weight( float const rep_weight, float const default_Wrep,
											 pose_ns::Score_weight_map & weight_map, pose_ns::Pose & pose){
	using namespace pose_ns;
	using namespace param_pack;
	weight_map.set_weight( FA_REP, rep_weight );
	score_set_lj_rep_weight( rep_weight );
	pack_wts.set_Wrep( default_Wrep * rep_weight );
	//	std::cout << " SET REP WEIGHT: " << rep_weight << std::endl;
	pose.new_score_pose();
}

////////////////////////////////////////////////////////////////////////////////////////////////
void repack_minimize(
  pose_ns::Pose & pose,
	std::string tag = "",
	bool const dump_pdb = false
)
{
	using namespace pose_ns;

	// Load in options
	static bool const vary_bond_geometry = truefalseoption("vary_bond_geometry");
	static bool const vary_sidechain_bond_angles = truefalseoption("vary_sidechain_bond_angles");
	static bool const copy_native_chi_angles = truefalseoption("copy_native_chi_angles");
	static const bool staged_minimize = truefalseoption("staged_minimize");
	static bool setup_atom_tree = truefalseoption("setup_atom_tree");
	if (vary_bond_geometry) setup_atom_tree  = true;
	if (vary_sidechain_bond_angles) setup_atom_tree  = true;

	//Weight map.
	Score_weight_map weight_map = favorite_weight_map();
	if (vary_bond_geometry){
		weight_map.set_weight( KIN_1D_CST, 1.0 )  ; // the torsion/bond/angle tether
		weight_map.set_weight( KIN_3D_CST, 1.0 )  ; // the torsion/bond/angle tether
		weight_map.set_weight( ATOMPAIR_CST, 1.0 ); // the ring constraint tether
	}
	if (vary_sidechain_bond_angles){
		weight_map.set_weight( SIDECHAIN_BOND_ANGLE,
													 realafteroption("bond_angle_weight",50.0));
	}
	float const default_Wrep = param_pack::pack_wts.Wrep();

	//Output native pose.
	if (copy_native_chi_angles) copy_chi_angles(pose, pose.native_pose() );
	pose.score( weight_map );
	calc_rms( pose );

 	static int const cycles = intafteroption("cycles",1);
	//////////////////////////////////////////////////////////////////
	// atom tree setup, varying bond geometry.
	//////////////////////////////////////////////////////////////////
	if (setup_atom_tree && !pose.atom_tree()){
		//		Fold_tree f = pose.fold_tree();
		//		for (int n = 1; n <= pose.num_jump(); n++){
		//			f.set_jump_atoms( n,
		//												3 /* upstream anchor: C */,
		//												1 /* downsream root:  N */,
		//												true );
		//		}
		//		pose.set_fold_tree( f );
		pose.setup_atom_tree();
	}
	if ( (vary_bond_geometry || vary_sidechain_bond_angles)
			 && ! pose.get_vary_bond_geometry_flag()){
		if (vary_sidechain_bond_angles) set_sidechain_bond_angle_allow_move( pose );
		if (vary_bond_geometry) {
			set_all_bond_distance_allow_move( pose );
			set_all_bond_angle_allow_move( pose );
		}
	}
///////////////////////////////////////////////////////////////////
	// Main loop.
	//////////////////////////////////////////////////////////////////
	//	bool const geometric_ramp = truefalseoption("geometric_ramp");
	//	float const end_rep_weight( 1.0 );
	//	bool const use_nblist( true );
	std::string min_type = "dfpmin";

	//	float rep_delta = get_rep_delta( start_rep_weight, end_rep_weight,
	//																	 geometric_ramp, cycles );
	float const repack_rep_weight = realafteroption( "repack_rep_weight", 0.0 );
	float const minimize_rep_weight = realafteroption( "minimize_rep_weight", 1.0 );

	//	rep_weight = start_rep_weight;
	//	really_set_rep_weight(rep_weight, default_Wrep, weight_map);
	for ( int n = 1; n<= cycles; ++n ) {
		///////////////////////
		// start with a repack:
		files_paths::mode_title = tag+" Pack";
		really_set_rep_weight( repack_rep_weight, default_Wrep, weight_map, pose);
		if (!copy_native_chi_angles) pose.full_repack( true );

		if (n==1) {
			//			std::cout << " REPACK " << n <<" :  " << pose.show_scores() << std::endl;
			if (dump_pdb) pose.dump_pdb( tag+"_REPACK.pdb");
		}
		//Ramp up weights.
// 		if ( geometric_ramp ) {
// 			rep_weight *= rep_delta;
// 		} else {
// 			rep_weight += rep_delta;
// 		}
//		really_set_rep_weight(rep_weight, default_Wrep, weight_map);

		really_set_rep_weight( minimize_rep_weight, default_Wrep, weight_map, pose);
		pose.score( weight_map );
		///////////////
		// now minimize
		//		pose_set_use_nblist( false );
		if (staged_minimize) {
			files_paths::mode_title = tag+" Zero Min";
			atom_tree_set_allow_move( pose,  false /*backbone*/, false /*sidechain*/, false /*jump*/);
			pose.main_minimize( weight_map, min_type);
			std::cout << tag << " ZERO MIN" << pose.show_scores() << std::endl;

			files_paths::mode_title = tag+" First Min";
			atom_tree_set_allow_move( pose,  false /*backbone*/, true /*sidechain*/, false /*jump*/);
			pose.main_minimize( weight_map, min_type);
			std::cout << tag << " FIRST MIN" << pose.show_scores() << std::endl;

			files_paths::mode_title = tag+" Second Min";
			atom_tree_set_allow_move( pose,  true /*backbone*/, true /*sidechain*/, false /*jump*/);
						pose.main_minimize( weight_map, min_type);
			std::cout << tag << " SECOND MIN" << pose.show_scores() << std::endl;

			files_paths::mode_title = tag+" Third Min";
			atom_tree_set_allow_move( pose,  true /*backbone*/, true /*sidechain*/, true /*jump*/);
						pose.main_minimize( weight_map, min_type);
			std::cout << tag << " THIRD MIN" << pose.show_scores() << std::endl;

		} else {
			files_paths::mode_title = tag+" Min";
			pose.main_minimize( weight_map, min_type );
		}
		//		pose_set_use_nblist( use_nblist );

		calc_rms( pose );
		std::cout << tag << " CYCLE " << n << " : " << pose.show_scores() << std::endl;
		//pose.dump_pdb(tag+"_CYCLE"+string_of(n,3)+".pdb");
	} //ramp_outer_cycle


	//Following is to ensure that minimize (which needs to use weak
	// angle/length tethers to get any reasonable PDB-like deviations from ideality)
	// doesn't go too far.
	weight_map = favorite_weight_map(); //has strong bond length/angle tethers.
	pose.new_score_pose();
	pose.score( weight_map );
	calc_rms( pose );
	std::cout << tag << " FINAL " << " : " << pose.show_scores() << std::endl;

	//Clear atom tree?
	//pose.clear_atom_tree();

}

////////////////////////////////////////////////////////////////////////////
void repack_minimize(
  pose_ns::Pose & pose,
	pose_ns::Pose & native_pose,
	std::string tag,
	bool const dump_pdb //=true
)
{
	pose.set_native_pose( native_pose );
	repack_minimize( pose, tag, dump_pdb );
}

////////////////////////////////////////////////////////////////////////////
void minimize_and_output(
  pose_ns::Pose & start_pose,
	pose_ns::Pose & native_pose,
	std::string tag,
	silent_io::Silent_out out,
	bool const output_start, //=true
	bool const dump_pdb //=true
)
{
	using namespace pose_ns;

	Pose pose;
	pose = start_pose;

	//////////////////////////////////////////////////////////////////
	// Output start pose, if desired.
	//////////////////////////////////////////////////////////////////
	if (output_start) {
		if (dump_pdb) pose.dump_pdb( tag+".pdb");
		out.write( tag,  pose );
	}

	/////////////////////////////////////////////////////////////////
	// The good stuff.
	/////////////////////////////////////////////////////////////////
	repack_minimize( pose, native_pose, tag, dump_pdb);

	/////////////////////////////////////////////////////////////////
	// Output.
	/////////////////////////////////////////////////////////////////
 	pose.set_native_pose( native_pose );

	Score_weight_map weight_map = favorite_weight_map();
	pose.new_score_pose();
	pose.score( weight_map );
 	calc_rms( pose );
 	tag += "_RELAX";
	std::cout << tag << " " << pose.show_scores() << std::endl;
 	if (dump_pdb) pose.dump_pdb( tag+".pdb");
 	out.write( tag,  pose );

}
////////////////////////////////////////////////////////////////////////////
void three_body_relax_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	bool fullatom = true;
	setup_fullatom_options();

	// Silent output
	std::string output_file_name( "new.out" );
	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );
	pose_to_native( native_pose );

	std::string tag = "NATIVE";
	minimize_and_output( native_pose, native_pose, tag, out);

	//read in starting structure
	//	int count = 0;
	Pose start_pose, pose;
  pose_from_pdb( start_pose, stringafteroption("s"),
								 true /*false*/ /*fullatom*/, false, true );
	if (truefalseoption("copy_native_fold_tree")) {
		start_pose.set_fold_tree( native_pose.fold_tree() );
	} else {
		figure_out_fold_tree( start_pose );
	}
	pose = start_pose;

	//	tag = "S_"+string_of(count);
	tag = "START";
	minimize_and_output( start_pose, native_pose, tag, out);

	//Do a special case where starting structure's segments are aligned to native structure.
	Pose align_pose;
	//align_pose = start_pose;
	align_pose = pose;
	align_segments(align_pose, native_pose);
	if (truefalseoption("grid_search_from_align")) pose = align_pose;

	tag = "ALIGN";
	minimize_and_output( align_pose, native_pose, tag, out);

	//////////////////////////////////////////////////////////
	if (truefalseoption("three_body_gridsearch")){

		fullatom = truefalseoption("fullatom");
		if (!fullatom) pose.set_fullatom_flag( fullatom, false );
		int whichjump = 1;
// 		for (int i = 1; i<=pose.fold_tree().get_num_jump(); i++){
// 			if ( pose.fold_tree().upstream_jump_residue(i) <= 15 &&
// 					 pose.fold_tree().downstream_jump_residue(i) >= 29 ) whichjump = i;
// 		}
		if (truefalseoption("hack_tree")) whichjump = 2;
		std::cout << "Grid search around jump: " << whichjump << "  which connects " <<
			pose.fold_tree().upstream_jump_residue(whichjump) << " and " <<
			pose.fold_tree().downstream_jump_residue(whichjump)
							<< std::endl;
		rhiju_gridsearch( pose, native_pose, out, whichjump);
	}
}





////////////////////////////////////////////////////////////////////////////
void three_body_morph_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	bool fullatom = true;
	setup_fullatom_options();

	// Silent output
	std::string output_file_name( "new.out" );
	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name+" S_*.pdb ";
	std::system( command.c_str() );

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );

	std::string tag = "NATIVE";
	//	minimize_and_output( native_pose, native_pose, tag, out);

	//read in starting structure
	int count = 1;
	Pose start_pose;
  pose_from_pdb( start_pose, stringafteroption("s"),
								 true /*false*/ /*fullatom*/, false, true );
	if (truefalseoption("copy_native_fold_tree")) {
		start_pose.set_fold_tree( native_pose.fold_tree() );
	} else {
		figure_out_fold_tree( start_pose );
	}

	//	float const s_min = realafteroption( "s_min", 0.00);
	//	float const s_max = realafteroption( "s_max", 1.0);
	//	float const s_shift = realafteroption( "s_shift", 0.1);

	//	for (float x = s_min -0.0001; x <= s_max; x += s_shift) {

	int const num_jump = start_pose.num_jump();
	int const num_segments = num_jump + 1;
	int const num_combinations = static_cast<int>( std::pow( 2.0f, num_segments ) );

	for (int n = 0; n < num_combinations; n++){

		Pose pose;
		pose = start_pose;
		pose.set_native_pose( native_pose );

		align_one_segment( pose, native_pose, 1, pose.total_residue() ); //whole thing, to get in the same coordinate frame.

		int base2 = 1;
		for (int k = 1; k <= num_segments; k++){
			if ( n/base2 % 2 ) {
				align_segments(pose, native_pose, k);
			}
			base2 *= 2;
		}

		tag = "S_"+lead_zero_string_of(count++, 3);

		repack_minimize( pose, native_pose, tag,
										 false /*dump_pdb*/);
		pose.score( score12 );
		calc_rms( pose );

		base2 = 1;
		for (int k = 1; k <= num_segments; k++){
			pose.set_extra_score( "ALIGN_"+string_of( k ), (n/base2 % 2) );
			base2 *= 2;
		}
		out.write( tag, pose );
		pose.dump_pdb( tag+".pdb" );
	}

}


////////////////////////////////////////////////////////////////////////////
void
rigid_body_segment_moves( pose_ns::Pose & pose,
													int const num_moves,
													float const RMSD_per_move,
													bool const just_one_segment,
													pose_ns::Score_weight_map fast_weight_map){
	using namespace pose_ns;

  using namespace silent_io;
	Silent_out out( "I.out" );
	static int count( 1 );
	std::string const tag = "S_" + lead_zero_string_of(count++, 3);
	out.write(tag+"_start", pose);

	float const default_Wrep = param_pack::pack_wts.Wrep();

	float const init_temp ( 1.0 );
	//	Score_weight_map fast_weight_map( score12 );
	static float const fast_rep_weight = realafteroption( "fast_rep_weight", 0.5);
	really_set_rep_weight( fast_rep_weight, default_Wrep,
													 fast_weight_map, pose);
	Monte_carlo fast_mc( pose, fast_weight_map, init_temp);

	int const num_segments = pose.num_jump() + 1;
	int which_segment = static_cast<int> ( num_segments*ran3() ) + 1;

	//Rotamer trials?
	static float const try_rotamers = truefalseoption( "try_rotamers" );
	float const energycut( 0.01 );
	int const nres = pose.total_residue();
	if (try_rotamers)		score_set_try_rotamers( true );


	for (int r = 1; r <= num_moves; r++){

		if (!just_one_segment) which_segment = static_cast<int> ( num_segments*ran3() ) + 1;

		rigid_body_gaussian( pose, which_segment,  RMSD_per_move );

		if (try_rotamers){
			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
						fast_mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
		}

		if (num_moves > 1) fast_mc.boltzmann( pose );
	}

	really_set_rep_weight( 1.0, default_Wrep,
													 fast_weight_map, pose);

	out.write(tag+"_jiggle", pose);
	repack_minimize( pose );
	out.write(tag+"_relax", pose);
}

////////////////////////////////////////////////////////////////////////////
void
rigid_body_segment_moves( pose_ns::Pose & pose,
													int const num_moves,
													float const RMSD_per_move,
													bool const just_one_segment){
	pose_ns::Score_weight_map w( score12 );
	rigid_body_segment_moves( pose, num_moves, RMSD_per_move, just_one_segment, w );
}

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
void funky_relax( pose_ns::Pose & pose ){
	using namespace pose_ns;

	Score_weight_map  weight_map = favorite_weight_map();

	float start_score( 0.0 ), quarterway_score( 0.0 ), halfway_score( 0.0 );

	repack_minimize( pose );
	pose.score( weight_map );
	start_score = pose.get_0D_score( SCORE );

	calc_rms( pose );
	float const init_rmsd = pose.get_0D_score( RMSD );

	//Monte carlo object.
	float const init_temp ( 1.0 );
	Monte_carlo mc( pose, weight_map, init_temp );
	mc.set_autotemp( true, init_temp );

	int mcm_cycles = intafteroption( "mcm_cycles", 20 );
	for (int n = 1; n <= mcm_cycles; n++){
		// Move one segment, 1.0 A rms shift.
		//		files_paths::mode_title = "try one"+string_of(n);
		//		rigid_body_segment_moves( pose, 1 /*num_moves*/, 0.5 /*RMSD for each move*/, true /*just_one_segment*/);
		//		mc.boltzmann( pose, "try_one" );

		// Move one segment multiple times with fixed sidechain soft rep.
		//		files_paths::mode_title = "jiggle one"+string_of(n);
		//		rigid_body_segment_moves( pose, 500 /*num_moves*/, 0.1 /*RMSD for each move*/, true /*just_one_segment*/);
		//		mc.boltzmann( pose, "jiggle_one" );

		// Move multiple segments multiple times with fixed sidechain soft rep.
		files_paths::mode_title = "jiggle many "+string_of(n);
		rigid_body_segment_moves( pose, 500 /*num_moves*/, 0.1 /*RMSD for each move*/, false /*just_one_segment*/);
		mc.boltzmann( pose, "jiggle_many" );

		if (n == mcm_cycles/4) quarterway_score = pose.get_0D_score( SCORE );
		if (n == mcm_cycles/2) halfway_score = pose.get_0D_score( SCORE );
	}

	pose = mc.low_pose();

	calc_rms( pose );
	pose.set_extra_score( "STARTSCOR", start_score );
	pose.set_extra_score( "QUARTSCOR", quarterway_score );
	pose.set_extra_score( "HALFSCOR" , halfway_score );
	pose.set_extra_score( "IRMS", init_rmsd );
	mc.show_counters();

}

 ////////////////////////////////////////////////////////////////////////////
void funky_relax_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	bool fullatom = true;
	setup_fullatom_options();

	// Silent output
	std::string output_file_name( stringafteroption("o", "new.out") );
	//Overwrite any previous outfile.
	if (output_file_name == "new.out") {
		std::string command = "rm -f "+output_file_name+"  "+output_file_name+".list";
		std::system( command.c_str() );
	}

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );
	pose_to_native( native_pose );

	//read in starting structures
  // read silent file
	std::string silent_file_name= stringafteroption("s");
  Silent_file_data decoys( silent_file_name, fullatom );
	if ( !decoys.size() ) {
    std::cout << "STOP:: couldnt open silent-file!! " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // setup tag list
  std::vector< std::string > tag_list;
	tag_list = decoys.tags();

	prof::reset();

	//	int count = 0;
	// loop through the tag list
  for ( std::vector< std::string >::const_iterator it=tag_list.begin(),
					it_end = tag_list.end(); it != it_end; ++it ) {

    std::string const & tag( *it );

		if ( !out.start_decoy(tag) ) continue; // already done or started

    // get the data
    Silent_structure const & decoy( decoys.get_structure( tag ) );

		Pose pose;

    // fill the pose
    decoy.fill_pose( pose, true );

		//natural fold tree
		figure_out_fold_tree( pose );

		//set native
		pose.set_native_pose( native_pose );

		//do it!
		funky_relax( pose );

		out.write( tag, pose );
	}

	prof::show();
}




 ////////////////////////////////////////////////////////////////////////////
void funky_relax_onepdb_test()
{
  using namespace pose_ns;
  using namespace silent_io;

	bool fullatom = true;
	setup_fullatom_options();

	// Silent output
	std::string output_file_name( stringafteroption("o", "new.out") );
	//Overwrite any previous outfile.
	if (output_file_name == "new.out") {
		std::string command = "rm -f I.out I.out.list "+output_file_name+"  "+output_file_name+".list";
		std::system( command.c_str() );
	}

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );
	pose_to_native( native_pose );

	prof::reset();

	std::string const tag = stringafteroption("s");
	Pose pose;
  pose_from_pdb( pose, tag,
								 true /*false*/ /*fullatom*/, false, true );

	//natural fold tree
	figure_out_fold_tree( pose );

	//set native
	pose.set_native_pose( native_pose );

	//do it!
	funky_relax( pose );

	out.write( tag, pose );
	pose.dump_pdb("FINAL.pdb");

	prof::show();
}

////////////////////////////////////////////////////////////////////////////
void
rigid_body_gaussian( pose_ns::Pose & pose, int const which_segment, float const input_rms_shift /* = 1.0 */){
	using namespace numeric;

	float x,y,z,rot1,rot2,rot3;
	xyzVector_float axis1,axis2,axis3;

	static float  const rms_shift = realafteroption("rms_shift", input_rms_shift);
	x    = gaussian() * rms_shift/6.0;
	y    = gaussian() * rms_shift/6.0;
	z    = gaussian() * rms_shift/6.0;
	rot1 = gaussian() * rms_shift/6.0;
	rot2 = gaussian() * rms_shift/6.0;
	rot3 = gaussian() * rms_shift/6.0;

	rotate_translate_segment( pose, which_segment, x, y, z, rot1, rot2, rot3 );
}

////////////////////////////////////////////////////////////////////////////
void rhiju_mcm_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	//Full atom?
	bool const fullatom = truefalseoption( "fullatom" );
	if (fullatom) setup_fullatom_options();
	Score_weight_map weight_map( score4);
	if (fullatom)  weight_map = Score_weight_map( score12 );
	set_default_atomvdw( "highres" );

	// Silent output
	std::string output_file_name( "new.out" );
	//Overwrite any previous outfile.
	std::string command = "rm -f "+output_file_name;
	std::system( command.c_str() );

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );
	pose_to_native( native_pose );

	std::string tag = "NATIVE";
	//	if (fullatom) repack_minimize( native_pose, native_pose, tag, false);
	calc_rms( native_pose );
	//	native_pose.set_extra_score( "MC_ACCEPTED", 1.0);
	//	out.write( tag, native_pose );

	//read in starting structure
	int count = 1;
	Pose start_pose, pose;
  pose_from_pdb( start_pose, stringafteroption("s"),
								 true /*false*/ /*fullatom*/, false, true );
	if (truefalseoption("copy_native_fold_tree")) {
		start_pose.set_fold_tree( native_pose.fold_tree() );
	} else {
		figure_out_fold_tree( start_pose );
	}
	pose = start_pose;
	pose.set_native_pose( native_pose );

	tag = "START";
	if (fullatom) repack_minimize( pose, native_pose, tag, false /*dump_pdb*/);
	pose.score( weight_map );
	calc_rms( pose );
	pose.set_extra_score( "MC_ACCEPTED", 1.0);
	pose.set_extra_score( "FASTSCOR", pose.get_0D_score( SCORE ));
	out.write( tag, pose );

	//Monte carlo object.
	float const init_temp ( 1.0 );
	Monte_carlo mc( pose, weight_map, init_temp );
	mc.set_autotemp( true, init_temp );

	//OK, now do some trials
	int const mcm_cycles = intafteroption( "mcm_cycles", 20 );
	int const mcm_rounds = intafteroption( "mcm_rounds", 10 );

	bool const fast = truefalseoption( "fast" );
	bool const fast_mc_low_pose = truefalseoption( "fast_mc_low_pose" );
	bool const one_segment_per_round = truefalseoption( "one_segment_per_round" );
	float const default_Wrep = param_pack::pack_wts.Wrep();
	float const fast_rep_weight  = realafteroption("fast_rep_weight", 1.0);

	/////////////
	//Main loop
	/////////////
	for (int r = 1; r <= mcm_rounds; r++){

		//Do a mini monte carlo...
		pose.set_native_pose( native_pose );
		Score_weight_map fast_weight_map( weight_map );
		really_set_rep_weight( fast_rep_weight, default_Wrep,
													 fast_weight_map, pose);
		Monte_carlo fast_mc( pose, fast_weight_map, init_temp);

		//Choose a segment to move.
		int const num_segments = pose.num_jump() + 1;
		int which_segment = static_cast<int> ( num_segments*ran3() ) + 1;

		//Main loop of fast moves.
		for (int n = 1; n <= mcm_cycles; n++){
			tag = "POSE_" + lead_zero_string_of(count++, 3);
			files_paths::mode_title = tag;

			if (!one_segment_per_round) which_segment = static_cast<int> ( num_segments*ran3() ) + 1;

		// Rotate and translate it to give an rms of  about
		// 1 A from its current location
			rigid_body_gaussian( pose, which_segment );

			// Repack/minimize.
			if (fullatom){
				if (fast){
					; //Do something? Set rep low?
				} else {
					repack_minimize( pose, native_pose, tag,
													 false /*dump_pdb*/);
				}
			}
			calc_rms( pose );

			//Metropolis criterion
			fast_mc.boltzmann( pose, "rb" );

			//			if (pose.get_extra_score( "MC_ACCEPTED" )){
			//				out.write( tag, pose );
			//			}

		} //cycles

		if (fast_mc_low_pose)	 pose = fast_mc.low_pose();

		pose.set_extra_score( "FASTSCOR", pose.get_0D_score(SCORE) );
		really_set_rep_weight( 1.0, default_Wrep,
													 weight_map, pose);
		repack_minimize( pose, native_pose, tag, false );

		mc.boltzmann( pose, "repack" );

		if (pose.get_extra_score( "MC_ACCEPTED" )){
			out.write( tag+"_REPACK", pose );
		}

		pose = mc.low_pose();

	} //round



	calc_rms( pose );
	pose.dump_pdb("FINAL.pdb");
	out.write( "FINAL", pose );

}

////////////////////////////////////////////////////////////////////////////
void three_body_search_test()
{
  using namespace silent_io;
  using namespace pose_ns;

	//Full atom?
	bool const fullatom = truefalseoption( "fullatom" );
	if (fullatom) setup_fullatom_options();
	Score_weight_map weight_map( score4);
	if (fullatom)  weight_map = Score_weight_map( score12 );
	set_default_atomvdw( "highres" );

	// Silent output
	std::string output_file_name( stringafteroption( "o", "new.out" ));
	//Overwrite any previous outfile.
	if (output_file_name == "new.out") {
		std::string command = "rm -f "+output_file_name+"  "+output_file_name+".list";
		std::system( command.c_str() );
	}

	Silent_out out( output_file_name );

	//Read in native.
	Pose native_pose;
	bool const native_exists = truefalseoption("n");
	FArray1D_int native_mapping;
	if ( native_exists ) {
		pose_from_pdb( native_pose, stringafteroption("n"), fullatom, false );
	}
	figure_out_fold_tree( native_pose );
	pose_to_native( native_pose );


	std::string tag = "NATIVE";
	native_pose.score( weight_map );
	calc_rms( native_pose );
	if (out.start_decoy(tag)) out.write( tag, native_pose );

	//read in starting structure
	int count = 1;
	Pose start_pose, pose;
  pose_from_pdb( start_pose, stringafteroption("s"),
								 fullatom /*false*/ /*fullatom*/, false, true );
	if (truefalseoption("copy_native_fold_tree")) {
		start_pose.set_fold_tree( native_pose.fold_tree() );
	} else {
		figure_out_fold_tree( start_pose );
	}


	bool const dump_pdb = truefalseoption( "dump_pdb" );

	tag = "START";
	if (out.start_decoy(tag) ){
		pose = start_pose;
		pose.set_native_pose( native_pose );
		if (fullatom)	pose.set_fullatom_flag( true, false );
		if (fullatom)	 repack_minimize( pose, native_pose, tag, false );
		calc_rms( pose );
		out.write( tag, pose );
		if (dump_pdb) pose.dump_pdb( tag+".pdb" );
	}

	float const s_min = realafteroption( "s_min", -1.0);
	float const s_max = realafteroption( "s_max", 1.0);
	float const s_shift = realafteroption( "s_shift", 1.0);

	float const rms_cutoff = realafteroption("rms_cutoff", 2.0);
	//	float const centroid_score_cutoff = realafteroption("centroid_score_cutoff", 0.0);

	int const segment_num = intafteroption("segment_num", 1 );
	for (float x = s_min -0.0001; x <= s_max; x += s_shift) {
		for (float y = s_min; y <= s_max; y += s_shift) {
			for (float z = s_min; z <= s_max; z += s_shift) {
				for (float theta = s_min; theta <= s_max; theta += s_shift) {
					for (float phi = s_min; phi <= s_max; phi += s_shift) {
						for (float alpha = s_min; alpha <= s_max; alpha += s_shift) {

 							tag = "S_"+lead_zero_string_of(count++,3);
							if (!out.start_decoy( tag )) continue; //already done or started.

							pose = start_pose;
							pose.set_native_pose( native_pose );

							if (fullatom)	pose.set_fullatom_flag( true, false );

							rotate_translate_segment( pose, segment_num, x, y, z, theta, phi, alpha);

							pose.score( weight_map );
							calc_rms( pose );
							if (pose.get_0D_score( RMSD ) < rms_cutoff){
								if (fullatom)	 repack_minimize( pose, native_pose, tag, false );

								calc_rms( pose );
								pose.set_extra_score( "x", x );
								pose.set_extra_score( "y", y );
								pose.set_extra_score( "z", z );
								pose.set_extra_score( "theta", theta );
								pose.set_extra_score( "phi", phi );
								pose.set_extra_score( "alpha", alpha );

								out.write( tag, pose );
								if (dump_pdb) pose.dump_pdb( tag+".pdb" );
							}

						}
					}
				}
			}
		}
	}



}





void
pose_sse_test()
{

	files_paths::mode_title =  "Pose SSE";

 L100:
	if (truefalseoption("testjumps"))
		pose_sse_testjumps();
	else if (truefalseoption("threebody"))
		pose_sse_threebody_test();
	else if (truefalseoption("extract_segment"))
		extract_segment_test();
	else if (truefalseoption("translate_search"))
		pose_sse_translate_search();
	else if (truefalseoption("translate_1D_search"))
		pose_sse_translate_1D_search();
	else if (truefalseoption("twist_1D_search"))
		pose_sse_twist_1D_search();
	else if (truefalseoption("translate_1D_boltzmann"))
		pose_sse_translate_1D_boltzmann();
	else if (truefalseoption("rhiju_gridsearch"))
	  rhiju_relax_test( "gridsearch" );
	else if (truefalseoption("rhiju_relax"))
	  rhiju_relax_test( "relax" );
	else if (truefalseoption("pose_score"))
	  pose_score_test();
	else if (truefalseoption("three_body_relax"))
	  three_body_relax_test();
	else if (truefalseoption("funky_relax"))
	  funky_relax_test();
	else if (truefalseoption("funky_relax_onepdb"))
	  funky_relax_onepdb_test();
	else if (truefalseoption("rhiju_mcm"))
	  rhiju_mcm_test();
	else if (truefalseoption("three_body_search"))
	  three_body_search_test();
	else if (truefalseoption("three_body_morph"))
	  three_body_morph_test();
	else
		pose_sse_gridsearch();


	//ooh bad code.
	if (truefalseoption("repeat")) goto L100;

	exit(0);
}
