// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20656 $
//  $Date: 2008-02-26 21:06:09 -0500 (Tue, 26 Feb 2008) $
//  $Author: bblum $


// Rosetta Headers
#include "random_numbers.h"
#include "after_opts.h"
#include "dock_structure.h"
#include "repeat.h"

// Utility Headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/Time_Date.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <ctime>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <string>
// File to store the random number information to

// Namespaces
namespace random_save {
	bool seeded = { false };
	int seed_offset = { 0 };
	int inext;
  std::string rndnum_filename = "rosetta_random.txt";
	int inextp;
//	FArray1D_int ma( 55 );
	FArray1D_float ma( 55 );
}


// Numerical functions are taken from Numerical Recipes by Press et al.

// modified the Numerical Recipes code by separating the seeding
// operation from the generation operation and putting the 'idum'
// argument only in the seeding call, since it is irrelevant
// afterward.  JJG 6/02

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_random_numbers
///
/// @brief
/// generate a 'random' seed from the clock, unless the
/// command-line option constant_seed is set
/// local
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_random_numbers()
{
	using namespace std;
	using namespace random_save;
	int jran;
	FArray1D_int timearray( 3 );

	// yab: we add the following and change the code below to use /dev/urandom
	//      (/dev/random is better, but the delay due to lack of
	//      necessary entropy is sometimes undesirable)
	bool const use_time_as_seed = truefalseoption( "use_time_as_seed" );

	//js The seed offset flag exists primarily for submissions to large clusters.
	//js It allows for the command line specification of an offset to the internally
	//js generated clock seed.  Typically one would use a condor process id.
	int seed_offset;
	intafteroption("seed_offset",0,seed_offset);

	if ( truefalseoption("delay_at_start") ) {
		// jk put in a delay of ( 10 * seed_offset ) seconds
		time_t target_seconds = time(NULL) + ( 10 * seed_offset );
		time_t curr_seconds = time(NULL);
		while ( curr_seconds < target_seconds ) {
			curr_seconds = time(NULL);
		}
	}

	if ( truefalseoption("constant_seed") ) {

    // BOINC - from WCG code
		// If using a constant seed, attempt to restore the random number
	  // information from a previously known state.
    seeded = restore_random_numbers();
    if (seeded) return;

		intafteroption("jran",1111111,jran);
	} else {

#ifdef WIN32
		itime(timearray);
		jran = 1 + 20*(3600*timearray(1)+60*timearray(2)+timearray(3));
#else
		// yab: by default we seed from /dev/urandom
		//      (/dev/random is better, but the delay due to lack of
		//      necessary entropy is sometimes undesirable)
		string random_device_name( "/dev/urandom" );
		ifstream random_device( random_device_name.c_str(), ios::in | ios::binary );
		if ( random_device.fail() || use_time_as_seed ) { // fallback to time if failure to open file or requested by user
			if ( !use_time_as_seed ) {
				std::cout << "NOTICE: " << random_device_name << " was not found, taking seed from time" << std::endl;
			}
			random_device.close();

			itime(timearray);
			jran = 1 + 20*(3600*timearray(1)+60*timearray(2)+timearray(3));
		} else {
	        uint32_t seed;
	        random_device.read( reinterpret_cast< char * >( &seed ), sizeof( uint32_t ));
	        random_device.close();

	        jran = static_cast< int >( seed );
	        if ( jran < 0 ) { // enforce positive number so we don't have to change rest of code below
	        	jran = -jran;
	        }
		}
#endif
	}

	jran += seed_offset;

#ifdef BOINC
	std::cerr << "# random seed: " << jran << std::endl;
#endif

	std::cout << "# =====================================" << std::endl;
	std::cout << "# random seed: " << jran << std::endl;
	std::cout << "# =====================================" << std::endl;
	jran = -jran; // ran3 takes a negative seed
	ran3_seed( jran );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin ran3_seed
///
/// @brief
///
/// @detailed
///
/// @param  idum - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ran3_seed( int & idum )
{
	using namespace random_save;

	repeat_capture_ran3_seed(idum); // added by sheffler 11/5/04

	//int const MBIG = { 1000000000 };
	//int const MSEED = { 161803398 };
	//int const MZ = { 0 };
	float const MBIG = { 4000000. };
	float const MSEED = { 1618033 };
	float const MZ = { 0 };

	if ( idum > 0 ) error_stop("bad seed");
	seeded = true;
	//int mj = MSEED-std::abs(idum);
	float mj = std::abs(MSEED-std::abs(idum));
	mj = mod(mj,MBIG);
	ma(55) = mj;
	//int mk = 1;
	float mk = 1;
	for ( int i = 1; i <= 54; ++i ) {
		int ii = mod(21*i,55);
		ma(ii) = mk;
		mk = mj - mk;
		if ( mk < MZ ) mk += MBIG;
		mj = ma(ii);
	}
	for ( int k = 1; k <= 4; ++k ) {
		for ( int i = 1; i <= 55; ++i ) {
			ma(i) -= ma(1+mod(i+30,55));
			if ( ma(i) < MZ ) ma(i) += MBIG;
		}
	}
	inext = 0;
	inextp = 31;
	idum = 1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin ran3
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
ran3()
{
	repeat_count_ran3();  // added by sheffler 11/5/04

	using namespace random_save;

	float ran3; // Return value

// based on Numerical Recipes (2nd ed.) "ran3"
// seeding separated in a separate function
//	int const MBIG = { 1000000000 };
//	int const MZ = { 0 };
	float const MBIG = { 4000000. };
	float const MZ = { 0 };
	float const FAC = { 1.0f / MBIG };

	if ( !seeded ) initialize_random_numbers();
	++inext;
	if ( inext == 56 ) inext = 1;
	++inextp;
	if ( inextp == 56 ) inextp = 1;
	//int mj = ma(inext) - ma(inextp);
	float mj = ma(inext) - ma(inextp);
	if ( mj < MZ ) mj += MBIG;
	ma(inext) = mj;
	ran3 = mj * FAC;
	return ran3;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin gaussian
///
/// @brief
/// Returns a gaussian random number (normally distributed deviate with
/// zero mean and unit variance) using ran3 as a source of uniform deviates.
/// Always call with the same idum
///
/// @detailed
///
/// @return  gaussian
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
/// From Numerical Recipes, section 7.2, a.k.a. "GASDEV"
///
/// @authors
/// JJG 4/01
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
gaussian()
{
	float gaussian; // Return value

	static bool iset = { true };
	static float gset;
	float v1, v2, rsq, fac;

	if ( iset ) {
L1:
		v1 = 2.0f * ran3() - 1.0f;
		v2 = 2.0f * ran3() - 1.0f;
		rsq = ( v1 * v1 ) + ( v2 * v2 );
		if ( rsq >= 1.0 || rsq == 0.0 ) goto L1;
		fac = std::sqrt(-(2.0*std::log(rsq)/rsq));
		gset = v1*fac;
		gaussian = v2*fac;
		iset = false;
	} else {
		gaussian = gset;
		iset = true;
	}
//	std::cout << "NR Gaussian: " << gaussian << std::endl;
	return gaussian;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_range
///
/// @brief
/// Will return a random int in the range specified by two ints passed as
/// arguments.
///
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
///
/// @authors
/// XA
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
random_range(int low_num, int high_num)
{
	if(low_num > high_num){
		int temp;
		temp = low_num;
		low_num = high_num;
		high_num = temp;
	}

	int range = high_num - low_num + 1;
	return ( static_cast< int >( range * ran3() ) + low_num );
}

//-------------------------------------------------------------------
// Added for BOINC from World Community Grid.
// Saves all the random number information to rosetta_random.wcg.
// File format:
// inext inextp ma[0] ... ma[54]
//-------------------------------------------------------------------
bool
save_random_numbers()
{
	using namespace random_save;

	bool saved = false;

	utility::io::ozstream rnd_file( rndnum_filename );

	if ( rnd_file ) {
		// Store all floats in scientific notation with the proper amount of precision.
		rnd_file() << std::scientific << std::setprecision( 10 );

		rnd_file << inext << " " << inextp;

		for ( unsigned int i = 1; i <= ma.size(); ++i ) {
			rnd_file << " " << ma(i);
		}

		rnd_file.close();
		rnd_file.clear();

		saved = true;
	}

	return saved;
}


//-------------------------------------------------------------------
// Added for BOINC from World Community Grid
// Attempts to restore the random number information from a previously
// stored state in file rosetta_random.boinc.
// RETURN: true if it successfully restored the information, false if
//         not (e.g. the file doesn't exist).
//-------------------------------------------------------------------
bool
restore_random_numbers()
{
	using namespace random_save;

	bool restored = false;

	utility::io::izstream rnd_file( rndnum_filename );

	if ( rnd_file ) {
		rnd_file >> inext >> inextp;

		unsigned int i = 0;
		float f;
		for ( i = 1; i <= ma.size() && !rnd_file().eof(); ++i ) {
			rnd_file >> f;
			ma(i) = f;
		}

		rnd_file.close();
		rnd_file.clear();

		restored = ( ( i - 1 ) == ma.size() );
	}

	return restored;
}
