// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-

#include "rb_class.h"
#include "random_numbers.h"
#include "pose.h"
#include "loop_class.h"

// C++ Headers
#include <cassert>
#include <iostream>
#include <cassert>
#include <algorithm>
#include <list>
#include <map>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>
#include <utility/file/file_sys_util.hh>

namespace pose_ns {

  std::ostream & operator<<( std::ostream & os, const Rb & rb ) {
    os << rb.seg_begin_ << " " << rb.seg_end_ << " " << rb.anchor_pos_ << " "
       << std::endl;
    return os;
  }

	/////////////////////////////////////////////////////////

	void
	RbSegments::one_segment_fold_tree(
		pose_ns::Fold_tree & f,
		int const total_residue
							)
	{
		using namespace pose_ns;
		f.clear();

		if ( num_rb() > 1 ) {
			std::cout << "This is a single segment fold tree"
								<< "Multiple segments are present in the RbSegments object"
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		Rb one_segment_rb( rb_list.at( 0 ) ); //segment currently passed

		int seg_begin( one_segment_rb.seg_begin() );
		int seg_end( one_segment_rb.seg_end() );
		int anchor_pos( one_segment_rb.anchor_pos() );

		assert( anchor_pos - 1 != total_residue - anchor_pos );//unlikely
		//Is the anchor position close to n or c terminus.That will determine how the fold tree is set up
		bool const close_to_n ( ( anchor_pos - 1 < total_residue - anchor_pos ) ? true : false );
		int const flex_jump ( seg_begin + ( seg_end - seg_begin )/2 );
		int const fixed_jump( close_to_n ? seg_end + ( total_residue - seg_end )/2 : ( seg_begin - 1 )/2 );

		std::cout << seg_begin << " " << seg_end << " " << anchor_pos << " " <<  flex_jump << " " << fixed_jump << " " << close_to_n << std::endl;

		if ( close_to_n ) {

			f.add_edge( 1, anchor_pos, pose_param::PEPTIDE );//EDGES
			f.add_edge( anchor_pos, seg_begin - 1, pose_param::PEPTIDE );
			f.add_edge( seg_end + 1, fixed_jump, pose_param::PEPTIDE );
			f.add_edge( fixed_jump, total_residue, pose_param::PEPTIDE );
			f.add_edge( seg_begin, flex_jump, pose_param::PEPTIDE );
			f.add_edge( flex_jump, seg_end, pose_param::PEPTIDE );

			//set jump 1 as flexible
			f.add_edge( anchor_pos, flex_jump, 1 ); //JUMPS
			f.add_edge( anchor_pos, fixed_jump, 2 );

		} else {

			f.add_edge( 1, fixed_jump, pose_param::PEPTIDE ); //EDGES
			f.add_edge( fixed_jump, seg_begin - 1, pose_param::PEPTIDE );
			f.add_edge( seg_begin, flex_jump, pose_param::PEPTIDE );
			f.add_edge( flex_jump, seg_end, pose_param::PEPTIDE );
			f.add_edge( seg_end + 1, anchor_pos, pose_param::PEPTIDE );
			f.add_edge( anchor_pos, total_residue, pose_param::PEPTIDE );

			//set jump 1 as flexible
			f.add_edge( anchor_pos, flex_jump, 1 ); //JUMPS
			f.add_edge( anchor_pos, fixed_jump, 2 );
		}
		std::cout << "Fold tree " << f << std::endl;
		f.reorder( 1 );
		std::cout << "Fold tree reordereed " << f << std::endl;

	}


  ///////////////////////////////////////////////////////////
  void
  RbSegments::read_segments_from_file(
				std::string const filename
	      )
  {

    utility::io::izstream data( filename.c_str() );
    if( !data ) {
      std::cout << "Couldn't open rb file " << std::endl;
      utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    std::string line;
    while( getline( data, line ) ) {
      std::istringstream line_stream( line );
      int seg_begin, seg_end, anchor_pos,loop_stub_n, loop_stub_c; //the loop stub is the start position for loop modeling
      line_stream >> seg_begin >> seg_end >> anchor_pos >> loop_stub_n >> loop_stub_c;
//There are two loop stubs, one for loop n-term of segment and other c-term of segment
// The loops are stored as a loop object
// Note that the cut point is always the seg_begin or seg_end positions
      if ( !line_stream.fail() ) {
				add_segment( seg_begin, seg_end, anchor_pos, loop_stub_n, loop_stub_c );
			}
		}
    data.close();
    data.clear();

  }


  ///////////////////////////////////////////////////////////////////

  void
  RbSegments::add_segment(
			  int const seg_begin,
			  int const seg_end,
			  int const anchor_pos,
				int const loop_stub_n,
				int const loop_stub_c
			  )
  {

		bool segment_ok = verify_segment( seg_begin, seg_end, anchor_pos, loop_stub_n, loop_stub_c );
		if ( !segment_ok ) {
			std::cout << "Error adding segment"
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		rb_list.push_back( Rb( seg_begin, seg_end, anchor_pos, loop_stub_n, loop_stub_c ) );
	}

	//////////////////////////////////////////////////////////////////
	void
	RbSegments::add_segment(
			const RbSegments::iterator & it
				)
	{
		add_segment( it->seg_begin(), it->seg_end(), it->anchor_pos(), it->loop_stub_n(), it->loop_stub_c() );
	}


	///////////////////////////////////////////////////////////////////////

	void
	RbSegments::delete_segment(
				 int const seg_begin,
				 int const seg_end
				 )
	{

		assert( seg_begin < seg_end );
		bool segment_deleted( false );

		for ( iterator it=rb_list.begin(), it_end=rb_list.end(); it!=it_end; ++it ){
			if ( seg_begin == it->seg_begin() && seg_end == it->seg_end() ) {
				rb_list.erase( it );
				segment_deleted = true;
			}
		}
		if ( !segment_deleted ) {
			std::cout << "Segment not present in rb list "
								<< seg_begin << " " << seg_end
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	////////////////////////////////////////////////////////////////////
	void
	RbSegments::delete_segment(
					 const RbSegments::iterator & it
					 )
	{
		delete_segment( it->seg_begin(), it->seg_end() );
	}


	/////////////////////////////////////////////////////////////////////

	RbSegments::iterator
	RbSegments::one_random_segment()

	{
	int size = num_rb();
	assert( size > 0 );
	int index = 0;
	int const end = int ( ran3()*size );
	iterator it = rb_list.begin();
	while( index != end ) {
		++index;
		++it;
	}
	return it;

	}


	////////////////////////////////////////////////////////////////////
	/*
	void
	RbSegments::sequential_order()

	{

		std::vector < int > rb_begin_pos, rb_end_pos;
		std::vector < Rb > sorted_rb_segments;

		rb_begin_pos.clear();
		rb_end_pos.clear();

		int num_segments( num_rb() );
		assert( num_segments );

		for ( iterator it = rb_list.begin(), it_end = rb_list.end(); it != it_end; ++it ) {
			rb_begin_pos.push_back( it->seg_begin() );
			rb_end_pos.push_back( it->seg_end() );
		}

		std::sort( rb_begin_pos.begin(), rb_begin_pos.end() );
		std::sort ( rb_end_pos.begin(), rb_end_pos.end() );

		rb_list.clear();

		for ( int i = 0; i <= num_segments; ++i ) {
			if ( rb_begin_pos.at( i ) > rb_end_pos.at( i ) ) {
				std::cout << "Some error occurred in sequential reordering of segments"
									<< "seg_begin seg_end anchor_pos"
									<< rb_begin_pos.at( i ) << " " << rb_end_pos.at( i )
									<< std::endl;
			} else {
				int anchor( get_anchor_position( rb_begin_pos.at( i ), rb_end_pos.at( i ) ) );
				add_segment( rb_begin_pos.at( i ), rb_end_pos.at( i ), anchor );
			}
		}
		assert( int( rb_list.size() ) == num_segments );

	}
	*/

	///////////////////////////////////////////////////////////////////////


	bool
	RbSegments::verify_segment(
								 int const seg_begin,
								 int const seg_end,
								 int const anchor_pos,
								 int const loop_stub_n,
								 int const loop_stub_c
								 )
  {

		//need to put some check for jump anchor positions as well.
    if ( seg_begin < seg_end ) {
      for ( const_iterator it = rb_list.begin(), it_end = rb_list.end(); it != it_end; ++it ) {
				if ( seg_end < it->seg_begin() || seg_begin > it->seg_end() ) {
					//do nothing, just makes sure that there is no overlap between multiple segments
				} else {
					std::cout << "Rb segment error\n "
										<< "Overlapping regions\n"
										<< "Existing segment definition " << it->seg_begin() << " " << it->seg_end()
										<< "New segment added " << seg_begin << " " << seg_end
										<< std::endl;
					return false;

				}
      }
			if( seg_begin < anchor_pos && anchor_pos < seg_end ) {
				std::cout << "Anchor position cannot lie between seg_begin and seg_end "
									<< seg_begin << " " << seg_end << " " << anchor_pos
									<< std::endl;
				return false;
			} else if ( ( seg_begin < loop_stub_n && loop_stub_n < seg_end ) || ( seg_begin < loop_stub_c  && loop_stub_c < seg_end ) ) {
				std::cout << "Loop stub cannot lie between seg_begin and seg_end"
									<< seg_begin << " " << seg_end << " " << loop_stub_n << " " << loop_stub_c
									<< std::endl;
				return false;
			} else if ( ( anchor_pos >= loop_stub_n && anchor_pos <= seg_begin ) && ( anchor_pos >= seg_end ) && ( anchor_pos <= loop_stub_c ) ) {
				std::cout << "Anchor position has to be in a fixed regions of the protein "
									<< "Here the anchor position is the loop_region between the loop_stub and the segment tip "
									<< seg_begin << " " << seg_end << " " << anchor_pos << " " << loop_stub_n << " " << loop_stub_c
									<< std::endl;
				return false;
			}

    } else {
      std::cout << "Rb segment definition error\n"
								<< "seg_begin, seg_end, anchor_pos " << seg_begin << " " << seg_end << " " << anchor_pos
								<< std::endl;
			return false;
    }
		return true;
  }
	/////////////////////////////////////////////////////////////////////////////////

} //end namespace pose_ns
