// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 15454 $
//  $Date: 2007-06-14 02:31:14 -0400 (Thu, 14 Jun 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "read_aa_ss.h"
#include "docking_ns.h"
#include "dssp.h" //to run dssp if no secstruct info available
#include "files_paths.h"
#include "misc.h"
#include "param.h"
#include "recover.h"
#include "runlevel.h"
#include "termini.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
/// @begin read_aa
///
/// @brief
/// read sequence and angles from .dat file
///
/// @detailed
/// reads fasta file if .dat not found (note no angle or ss info)
/// sets total residue and sequence
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_aa( bool & fail )
{
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	int count;
	std::string filename;
	bool found;

	fail = true;
	filename = seq_path + protein_name_prefix + protein_name + ".dat";

	read_dat( filename, count, found );
	if ( !found ) {
		filename = ss_path + protein_name_prefix + protein_name + ".dat";
		read_dat( filename, count, found );
		if ( !found ) {
			std::cout << "WARNING!! .dat file not found!" << std::endl;
			filename = seq_path + protein_name_prefix + protein_name + protein_chain + ".fasta";
			read_fasta( filename, residue1, count, fail );
			if ( fail ) return;
		}
	}

	total_residue = count;
	total_domains = 1;
	domain_end(1) = total_residue;
	convert1_num(); // generate res for next call

	update_sequence();
	setup_termini();
	fail = false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_ss
///
/// @brief
///car define secondary structure information if not already defined
///car checks first for dssp, then ssa, and finally assigns based on
///car phipsi angles
///
/// @detailed
///car input argument 'paths' must be either 'native' or 'start' and
///car determines the paths which are checked.
///
///car option dependecies:
///car   require_ss:   must ss be defined?
///
/// @param  paths - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_ss( std::string const & paths )
{
	using namespace files_paths;
	using namespace misc;

	std::string filename1, filename2;
	bool defined;

	if ( paths == "native" ) {
		filename1 = ss_path + protein_name_prefix + protein_name + ".dssp";
		filename2 = ss_path + protein_name_prefix + protein_name + protein_chain + ".ssa";
	}
	if ( paths == "start" ) {
		filename1 = start_path + start_file + ".dssp";
		filename2 = start_path + start_file + ".ssa";
	}

	read_dssp( filename1, total_residue, residue1, protein_chain, secstruct,
	 defined );
	if ( defined ) return;

	read_ssa( filename2, total_residue, secstruct, defined );
	if ( defined ) return;

	if ( !require_ss ) return;

	std::cout << "calculating secondary structure from torsion angles" << std::endl;
	ss_from_phipsi( secstruct, phi, psi, total_residue );

	std::cout << "calculating secondary structure from internal DSSP calculation." << std::endl;
	ss_from_dssp( secstruct );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_dat
///
/// @brief
///
/// @detailed
///
/// @param  filename - [in/out]? -
/// @param  count - [in/out]? -
/// @param  found - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_dat(
	std::string const & filename,
	int & count,
	bool & found
)
{
	using namespace docking;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;

	count = 0;
	found = false;

	std::cout << "Searching for dat file: " << filename << std::endl;

  utility::io::izstream iunit( filename );

	if ( !iunit ) {
		iunit.close();
		iunit.clear();
		return;
	}

	found = true;

	// Find number of residues
	while ( ! iunit.eof() ) {
		iunit >> skip;
		if ( ! iunit.eof() ) ++count;
	}
	// For symmetrical pose we need to add extra residues
	int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
	int nres (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*count + num_simulated_monomers : count + num_simulated_monomers ) : count );
	MAX_RES_assign_res( nres ); // Set MAX_RES

	iunit.seek_beg(); // Reset to beginning of file

	// Read data
	int sectemp;
	for ( int i = 1; i <= count; ++i ) {
		iunit >> bite( 2, sectemp ) >> skip( 41 ) >>
		 bite( 8, phi(i) ) >> bite( 8, psi(i) ) >> bite( 8, omega(i) ) >>
		 skip( 15 ) >> bite( residue1(i) ) >> skip;
		if ( runlevel >= inform) std::cout << SS( sectemp ) << SS( phi(i) ) <<
		 SS( psi(i) ) << SS( omega(i) ) << ":" << SS( residue1(i) ) << ":" << std::endl;
		if ( sectemp == 22 ) secstruct(i) = 'H';
		if ( sectemp == 23 ) secstruct(i) = 'E';
		if ( sectemp == 24 ) secstruct(i) = 'L';
	}

	iunit.close();
	iunit.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_fasta
///
/// @brief
/// read fasta format file (can't handle spaces in lines!!)
///
/// @detailed
///
/// @param  filename - [in/out]? -
/// @param  array - [in/out]? - destination character array
/// @param  Ndim - [in/out]? - Ndim is max size of array
/// @param  count - [in/out]? - total residues placed in array
/// @param  fail - [out] - was sequence successfully obtained?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_fasta(
	std::string const & filename,
	FArray1D_char & array,
	int & count,
	bool & fail
)
{
	using namespace docking;
	using namespace param;

	count = 0;
	fail = true;

	std::cout << "Looking for fasta file: " << filename << std::endl;

	utility::io::izstream funit( filename );

	if ( !funit ) {
		std::cout << "fasta file not found!" << std::endl;
		return;
	}

	std::string linebuffer;
	std::string array_string( 1, ' ' ); // Blank space for 1-based indexing

L110:; // infinite loop
	funit >> bite( linebuffer ) >> skip;
	if ( funit.eof() ) {
		goto L200;
	} else if ( funit.fail() ) {
		goto L300;
	}
	ObjexxFCL::strip_whitespace( linebuffer ); // Remove leading and trailing whitespace
	if ( linebuffer.empty() ) goto L110;
	if ( linebuffer[0] == '>' ) goto L110; // skip comment lines
	count += linebuffer.length();
	array_string += linebuffer;
	goto L110;

L200:; // graceful end of file
	fail = false;
	goto L400;

L300:; // non-graceful read
	std::cout << "Error reading fasta file" << std::endl;

L400:;
	funit.close();
	funit.clear();
	// For symmetrial pose we need to add some extra residues
	int const num_simulated_monomers = docking_pose_symm_subsystem ? docking_pose_symm_subsystem_size : pose_symm_n_monomers;
	int nres (  docking_pose_symmetry ? (monomer_input ? num_simulated_monomers*count + num_simulated_monomers : count + num_simulated_monomers ) : count );
	MAX_RES_assign_res( nres ); // Set MAX_RES
	for ( int i = 1; i <= count; ++i ) {
		array( i ) = array_string[ i ];
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_ssa
///
/// @brief
///
/// @detailed
///
/// @param  filename - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  defined - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_ssa(
	std::string const & filename,
	int & total_residue,
	FArray1Da_char ss,
	bool & defined
)
{
	ss.dimension( total_residue );

//car local
	std::string linebuffer;

	defined = false;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ss(i) != 'L' && ss(i) != 'H' && ss(i) != 'E' ) goto L50;
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ss(i) == 'H' || ss(i) == 'E' ) goto L60;
	}
	goto L50; // ss is all loop

L60:
	std::cout << "secondary structure defined, not reading ssa file" << std::endl;
	defined = true;
	return;

L50:
	std::cout << "Looking for secondary structure assignment file: " <<
	 filename << std::endl;

  utility::io::izstream iunit( filename );

	if ( !iunit ) {
		std::cout << "ssa file not found" << std::endl;
		iunit.close();
		iunit.clear();
		return;
	}

	iunit >> bite( linebuffer ) >> skip;
	trim( linebuffer ); // Trim trailing space
	if ( static_cast< int >( linebuffer.length() ) != total_residue ) {
		std::cout << "Disagreement in chain length in ssa file" << std::endl;
		std::cout << "total residue " << total_residue << " ssa length " << linebuffer.length() <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	iunit.close();
	iunit.clear();

	for ( int i = 1; i <= total_residue; ++i ) {
		ss(i) = linebuffer[i-1];
		if ( ss(i) == 'C' ) ss(i) = 'L';
		if ( ss(i) != 'L' && ss(i) != 'H' && ss(i) != 'E' ) {
			iunit.close();
			iunit.clear();
			std::cout << "Unrecognized character in ssa file: " << ss(i) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	defined = true;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert1_num
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
convert1_num()
{
	using namespace misc;
//------------------------------------------------------------------------------
	for ( int j = 1; j <= total_residue; ++j ) {
		res(j) = 0;
		if ( residue1(j) == 'A' ) {
			res(j) = 1;
			residue3(j) = "ALA";
		}
		if ( residue1(j) == 'B' ) {  // B is ASP or ASN, I call it ASP
			res(j) = 3;
			residue1(j) = 'D';
			residue3(j) = "ASP";
		}
		if ( residue1(j) == 'C' ) {
			res(j) = 2;
			residue3(j) = "CYS";
		}
		if ( residue1(j) == 'D' ) {
			res(j) = 3;
			residue3(j) = "ASP";
		}
		if ( residue1(j) == 'E' ) {
			res(j) = 4;
			residue3(j) = "GLU";
		}
		if ( residue1(j) == 'F' ) {
			res(j) = 5;
			residue3(j) = "PHE";
		}
		if ( residue1(j) == 'G' ) {
			res(j) = 6;
			residue3(j) = "GLY";
		}
		if ( residue1(j) == 'H' ) {
			res(j) = 7;
			residue3(j) = "HIS";
		}
		if ( residue1(j) == 'I' ) {
			res(j) = 8;
			residue3(j) = "ILE";
		}
		if ( residue1(j) == 'K' ) {
			res(j) = 9;
			residue3(j) = "LYS";
		}
		if ( residue1(j) == 'L' ) {
			res(j) = 10;
			residue3(j) = "LEU";
		}
		if ( residue1(j) == 'M' ) {
			res(j) = 11;
			residue3(j) = "MET";
		}
		if ( residue1(j) == 'N' ) {
			res(j) = 12;
			residue3(j) = "ASN";
		}
		if ( residue1(j) == 'P' ) {
			res(j) = 13;
			residue3(j) = "PRO";
		}
		if ( residue1(j) == 'Q' ) {
			res(j) = 14;
			residue3(j) = "GLN";
		}
		if ( residue1(j) == 'R' ) {
			res(j) = 15;
			residue3(j) = "ARG";
		}
		if ( residue1(j) == 'S' ) {
			res(j) = 16;
			residue3(j) = "SER";
		}
		if ( residue1(j) == 'T' ) {
			res(j) = 17;
			residue3(j) = "THR";
		}
		if ( residue1(j) == 'V' ) {
			res(j) = 18;
			residue3(j) = "VAL";
		}
		if ( residue1(j) == 'W' ) {
			res(j) = 19;
			residue3(j) = "TRP";
		}
		if ( residue1(j) == 'Y' ) {
			res(j) = 20;
			residue3(j) = "TYR";
		}
		if ( residue1(j) == 'g' ) {
			res(j) = 22;
			residue3(j) = "  G";
		}
		if ( residue1(j) == 'a' ) {
			res(j) = 23;
			residue3(j) = "  A";
		}
		if ( residue1(j) == 'c' ) {
			res(j) = 24;
			residue3(j) = "  C";
		}
		if ( residue1(j) == 't' ) {
			res(j) = 25;
			residue3(j) = "  T";
		}
		if ( residue1(j) == 'g' ) {
			res(j) = 26;
			residue3(j) = " rG";
		}
		if ( residue1(j) == 'a' ) {
			res(j) = 27;
			residue3(j) = " rA";
		}
		if ( residue1(j) == 'c' ) {
			res(j) = 28;
			residue3(j) = " rC";
		}
		if ( residue1(j) == 'u' ) {
			res(j) = 29;
			residue3(j) = " rU";
		}
		if ( residue1(j) == 'Z' ) {  // B is GLU or GLN, I call it ASP
			res(j) = 4;
			residue1(j) = 'E';
			residue3(j) = "GLU";
		}
		if ( residue1(j) == 'O' ) {//KMa phospho_ser
			res(j) = 21;
			residue3(j) = "SEP";//KMa phospho_ser
		}
		if ( res(j) == 0 ) {
			std::cout << "unrecognized residue type at position " << j << " : " <<
			 residue1(j) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert3_num
///
/// @brief
///  THIS FUNCTION CONVERTS THE LETTER CODES INTO 1letter CODES
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
convert3_num()
{
//------------------------------------------------------------------------------
//
//   ONE LETTER CODE   THREE LETTER CODE   NUMBER CODE
//         A                   ALA              1
//         C                   CYS              2
//         D                   ASP              3
//         E                   GLU              4
//         F                   PHE              5
//         G                   GLY              6
//         H                   HIS              7
//         I                   ILE              8
//         K                   LYS              9
//         L                   LEU             10
//         M                   MET             11
//         N                   ASN             12
//         P                   PRO             13
//         Q                   GLN             14
//         R                   ARG             15
//         S                   SER             16
//         T                   THR             17
//         V                   VAL             18
//         W                   TRP             19
//         Y                   TYR             20
//         g                   GUA             21
//         a                   ADE             22
//         c                   CYT             23
//         t                   THY             24
//         g                   rGU             25
//         a                   rAD             26
//         c                   rCY             27
//         u                   rUR             28
//------------------------------------------------------------------------------

	using namespace misc;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( residue3(i) == "ALA" ) {
			residue1(i) = 'A';
			res(i) = 1;
			goto L100;
		}
		if ( residue3(i) == "CYS" ) {
			residue1(i) = 'C';
			res(i) = 2;
			goto L100;
		}
		if ( residue3(i) == "ASP" || residue3(i) == "ASX" ) {
			residue1(i) = 'D';
			res(i) = 3;
			goto L100;
		}
		if ( residue3(i) == "GLU" || residue3(i) == "GLX" ) {
			residue1(i) = 'E';
			res(i) = 4;
			goto L100;
		}
		if ( residue3(i) == "PHE" ) {
			residue1(i) = 'F';
			res(i) = 5;
			goto L100;
		}
		if ( residue3(i) == "GLY" ) {
			residue1(i) = 'G';
			res(i) = 6;
			goto L100;
		}
		if ( residue3(i) == "HIS" ) {
			residue1(i) = 'H';
			res(i) = 7;
			goto L100;
		}
		if ( residue3(i) == "ILE" ) {
			residue1(i) = 'I';
			res(i) = 8;
			goto L100;
		}
		if ( residue3(i) == "LYS" ) {
			residue1(i) = 'K';
			res(i) = 9;
			goto L100;
		}
		if ( residue3(i) == "LEU" ) {
			residue1(i) = 'L';
			res(i) = 10;
			goto L100;
		}
		if ( residue3(i) == "MET" ) {
			residue1(i) = 'M';
			res(i) = 11;
			goto L100;
		}
		if ( residue3(i) == "ASN" ) {
			residue1(i) = 'N';
			res(i) = 12;
			goto L100;
		}
		if ( residue3(i) == "PRO" ) {
			residue1(i) = 'P';
			res(i) = 13;
			goto L100;
		}
		if ( residue3(i) == "GLN" ) {
			residue1(i) = 'Q';
			res(i) = 14;
			goto L100;
		}
		if ( residue3(i) == "ARG" ) {
			residue1(i) = 'R';
			res(i) = 15;
			goto L100;
		}
		if ( residue3(i) == "SER" ) {
			residue1(i) = 'S';
			res(i) = 16;
			goto L100;
		}
		if ( residue3(i) == "THR" ) {
			residue1(i) = 'T';
			res(i) = 17;
			goto L100;
		}
		if ( residue3(i) == "VAL" ) {
			residue1(i) = 'V';
			res(i) = 18;
			goto L100;
		}
		if ( residue3(i) == "TRP" ) {
			residue1(i) = 'W';
			res(i) = 19;
			goto L100;
		}
		if ( residue3(i) == "TYR" ) {
			residue1(i) = 'Y';
			res(i) = 20;
			goto L100;
		}
		if ( residue3(i) == "  G" ) {
			residue1(i) = 'g';
			res(i) = 22;
			goto L100;
		}
		if ( residue3(i) == "  A" ) {
			residue1(i) = 'a';
			res(i) = 23;
			goto L100;
		}
		if ( residue3(i) == "  C" ) {
			residue1(i) = 'c';
			res(i) = 24;
			goto L100;
		}
		if ( residue3(i) == "  T" ) {
			residue1(i) = 't';
			res(i) = 25;
			goto L100;
		}
		if ( residue3(i) == " rG" ) {
			residue1(i) = 'g';
			res(i) = 26;
			goto L100;
		}
		if ( residue3(i) == " rA" ) {
			residue1(i) = 'a';
			res(i) = 27;
			goto L100;
		}
		if ( residue3(i) == " rC" ) {
			residue1(i) = 'c';
			res(i) = 28;
			goto L100;
		}
		if ( residue3(i) == " rU" ) {
			residue1(i) = 'u';
			res(i) = 29;
			goto L100;
		}
		if ( residue3(i) == "SEP" ) //KMa phospho_ser
		{
			residue1(i) = 'O';
			res(i) = 21;//KMa phospho_ser
			goto L100;
		}
		if ( residue3(i) != "   " ) {
			std::cout << "cannot handle " << residue3(i) << std::endl;
		}
		residue1(i) = ' ';
L100:;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_dssp
///
/// @brief
///
/// @detailed
///
/// @param  filename - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  residue1 - [in/out]? -
/// @param  chain - [in/out]? -
/// @param  ss - [in/out]? -
/// @param  defined - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_dssp(
	std::string const & filename,
	int & total_residue,
	FArray1Da_char residue1,
	char const chain,
	FArray1Da_char ss,
	bool & defined
)
{
	residue1.dimension( total_residue );
	ss.dimension( total_residue );

	defined = false;

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ss(i) != 'L' && ss(i) != 'H' && ss(i) != 'E' ) goto L50;
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( ss(i) == 'H' || ss(i) == 'E' ) goto L60;
	}
	goto L50; // ss is all loop

L60:
	std::cout << "secondary structure defined, not reading dssp file" << std::endl;
	defined = true;
	return;

L50:
	std::cout << "Looking for dssp file: " << filename << std::endl;
	utility::io::izstream iunit( filename );

	if ( !iunit ) {
		std::cout << "dssp file not found" << std::endl;
		iunit.close();
		iunit.clear();
		return;
	}


/// GET TO BEGINNING OF DATA AND READ IN ONLY AA AND SS

	std::string id;
	char fch, fres, fss;
	int tr = 0;
L601:
	while ( id != "RESIDUE AA STRUCTURE" ) {
		iunit >> skip( 5 ) >> bite( 20, id ) >> skip;
		if ( iunit.eof() ) {
			goto L701;
		} else if ( iunit.fail() ) {
			iunit.clear();
			iunit >> skip;
			goto L601;
		}
	}
	while ( true ) {
		iunit >> skip( 11 ) >> bite( fch ) >> skip( 1 ) >>
		 bite( fres ) >> skip( 2 ) >> bite( fss ) >> skip;
		if ( !iunit ) goto L701;
		if ( fres == '!' ) goto L300;
		if ( fch == ' ' ) fch = '_';
		if ( fres == 'a' || fres == 'b' || fres == 'c' ) fres = 'C';
		if ( fch == chain ) {
			++tr;
			if ( tr > total_residue ) {
				std::cout << "WARNING: dssp sequence is longer than pdb: dssp>= " <<
					tr << " total_residue= " << total_residue << std::endl;
				break;
			}
			if ( fres != residue1(tr) ) {
				std::cout << "WARNING: disagreement in sequence: residue " << tr <<
				 "fasta/dat: " << residue1(tr) << " dssp: " << fres << std::endl;
			}
			ss(tr) = fss;
		}
L300:;
	}
L701:
	iunit.close();
	iunit.clear();

//     DO SOME QUICK CHECKS
	if ( tr != total_residue ) {
		std::cout << "total residues in pdb file is: " << total_residue << std::endl;
		std::cout << "total residues in dssp file is: " << tr << std::endl;
		std::cout << "not using dssp file" << std::endl;
		return;
	}

//     NOW TRANSLATE SS ASSIGNMENTS
	for ( int i = 1; i <= tr; ++i ) {
		if ( ss(i) == 'E' ) {
			ss(i) = 'E'; // Sheet
		} else if ( ss(i) == 'H' || ss(i) == 'I' || ss(i) == 'G' ) {
			ss(i) = 'H'; // Helix alpha,310,but what is G
		} else {
			ss(i) = 'L'; // all other coil - B, ,S, and T
		}
	}

	defined = true;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin ss_from_phipsi
///
/// @brief
///bk determine secondary structure from phi and psi angles
///bk to be strand must be part of a string of 3 strand phipsi
///bk to be helix must be part of a string of 3 helix phipsi
///
/// @detailed
///
/// @param  sschar - [in/out]? - E = strand, H = helix, L = loop
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  nres - [in/out]? - number of residues
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ss_from_phipsi(
	FArray1Da_char sschar,
	FArray1Da_float phi,
	FArray1Da_float psi,
	int nres
)
{
	using namespace param;

	sschar.dimension( MAX_RES() );
	phi.dimension( MAX_RES() );
	psi.dimension( MAX_RES() );

//bk ss       :ss = 1 helix, ss = 2 sheet, ss = 3 other
	FArray1D_int ss( MAX_RES()() );
	FArray1D_int sstemp( DRange( -1, MAX_RES()()+2 ) );

//------------------------------------------------------------------------------

//bk make sure phi,psi angles are between -180.0 and 180.0
	for ( int i = 1; i <= nres; ++i ) {
		phi(i) = mod(phi(i)+180.0f,360.0f)-180.0f;
		psi(i) = mod(psi(i)+180.0f,360.0f)-180.0f;
	}

	sstemp(-1) = 3; // assign loop to fictious residues at ends of chain
	sstemp(0) = 3;
	sstemp(nres+1) = 3;
	sstemp(nres+2) = 3;

	for ( int i = 1; i <= nres; ++i ) {
		if ( phi(i) < -20.0 && psi(i) > -90.0 && psi(i) < -10.0 ) {
			sstemp(i) = 1;
		} else if ( phi(i) < -20.0 && (psi(i) > 20.0 || psi(i) < -170.0) ) {
			sstemp(i) = 2;
		} else {
			sstemp(i) = 3;
		}
	}

	for ( int i = 1; i <= nres; ++i ) {
		if ( sstemp(i) == 2 ) {
			if ( sstemp(i-1) == 2 && sstemp(i+1) == 2 ) {
				ss(i) = 2;
			} else if ( sstemp(i-1) == 2 && sstemp(i-2) == 2 ) {
				ss(i) = 2;
			} else if ( sstemp(i+1) == 2 && sstemp(i+2) == 2 ) {
				ss(i) = 2;
			} else {
				ss(i) = 3;
			}
		} else if ( sstemp(i) == 1 ) {
			if ( sstemp(i-1) == 1 && sstemp(i+1) == 1 ) {
				ss(i) = 1;
			} else if ( sstemp(i-1) == 1 && sstemp(i-2) == 1 ) {
				ss(i) = 1;
			} else if ( sstemp(i+1) == 1 && sstemp(i+2) == 1 ) {
				ss(i) = 1;
			} else {
				ss(i) = 3;
			}
		} else {
			ss(i) = 3;
		}
	}

	for ( int i = 1; i <= nres; ++i ) {
		if ( ss(i) == 1 ) {
			sschar(i) = 'H';
		} else if ( ss(i) == 2 ) {
			sschar(i) = 'E';
		} else {
			sschar(i) = 'L';
		}
	}


}

///////////////////////////////////////////////////////////
void
ss_from_dssp( FArray1DB_char & secstruct ){

	using namespace dssp_ns;

	//Assumes misc is ready to go.
	DSSP dssp;
	dssp.compute();
	dssp.dssp_reduced( secstruct );

}
