// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19866 $
//  $Date: 2008-01-23 14:54:04 -0500 (Wed, 23 Jan 2008) $
//  $Author: possu $

// Rosetta Headers
#include "aaproperties_pack.h"
#include "atom_descriptor.h"
#include "hbonds.h" // burial
#include "hbonds_geom.h" //hbond compute energy
#include "hbonds_ns.h" // hbond types
#include "fast_pairenergy.h"
#include "rotamer_descriptor.h"
#include "rotamer_trie_calc_energies.h"
#include "rotamer_trial_energies.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "pack_geom_inline.h"
#include "pdbstatistics_pack.h"
#include "template_pack.h"
#include "water.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3D.hh>

// Utility Headers
#include <utility/vector1.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <iostream>

/// local function declarations
bool atomtype_isbb( int atomtype );
bool atomtype_isheavyatom( int atomtype);
bool atomtype_isdonorH (int atomtype );
bool atomtype_isacceptor ( int atomtype );
bool atomtype_hbchk_bbhbg_sat (  int atomtype, FArray1DB_bool & bb_has_hydrogenbonds );
bool atomtype_is_bb_n (int atom_type );
bool atomtype_is_bb_h (int atom_type );
bool atomtype_is_bb_ca (int atom_type );
bool atomtype_is_bb_c (int atom_type );
bool atomtype_is_bb_o (int atom_type );
bool rotamer_atom_in_sidechain_proper(rotamer_descriptor const & rotamer, int atom_id_in_rotamer );

///


using namespace std;
using namespace param_aa;

bool trie_output_all_energy_calcs = false;

rotamer_descriptor::rotamer_descriptor() {}
rotamer_descriptor::~rotamer_descriptor() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_descriptor::rotamer_descriptor
///
/// @brief
/// copy constructor
///
/// @detailed
///
/// @authors apl
////////////////////////////////////////////////////////////////////////////////
rotamer_descriptor::rotamer_descriptor( const rotamer_descriptor & rhs )
{	using namespace param;
	num_atoms_ = rhs.num_atoms_;
	rotamer_id_ = rhs.rotamer_id_;
	aa_type_    = rhs.aa_type_;
	aa_variant_ = rhs.aa_variant_;
	for ( int ii = 0; ii < rotamer_trie::MAX_ATS_PER_RES; ++ii )
		atoms_[ii] = rhs.atoms_[ii];
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_descriptor::rotamer_descriptor
///
/// @brief
/// number-of-atoms constructor
///
/// @detailed
///
/// @param num_atoms - [in] - the number of atoms in this rotamer
///
/// @authors apl
////////////////////////////////////////////////////////////////////////////////
rotamer_descriptor::rotamer_descriptor(int num_atoms) :
	num_atoms_( num_atoms), rotamer_id_(0), aa_type_(0)
{}

//interface: index atoms by one, internal representation, index by 0;
void rotamer_descriptor::set_atom(int pos, const atom_descriptor & atom)
{	//std::assert( pos <= num_atoms_  && pos > 0);
	atoms_[pos - 1] = atom;
	return;
}

/// @ brief set rotamer identifier
void rotamer_descriptor::set_rotamer_id(int rotid)
{ rotamer_id_ = rotid; return;}

/// @ brief set aa type for the rotamer
void rotamer_descriptor::set_aa_type(int aa_type) {aa_type_ = aa_type; return;}

/// @ brief set aav type for the rotamer
void rotamer_descriptor::set_aa_variant(int aav) {aa_variant_ = aav; return;}

/// @ breif get number of atoms in the rotamer
int rotamer_descriptor::get_num_atoms() const
{ return num_atoms_;}

/// @ brief get the rotamer identifier for the rotamer
int rotamer_descriptor::get_rotamer_id() const
{	return rotamer_id_;}

/// @ breif get the aa type for the rotamer
int rotamer_descriptor::get_aa_type() const { return aa_type_;}

/// @ breif get the aav type for the rotamer
int rotamer_descriptor::get_aa_variant() const {return aa_variant_;}

/// @ breif get a particular atom
atom_descriptor rotamer_descriptor::get_atom(int pos) const
{
	//std::assert( pos <= num_atoms_ && pos > 0);
	return atoms_[pos - 1];
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_descriptor::operator <
///
/// @brief "lexographical" ordering for rotamers
///
/// @detailed
/// returns true when the first atom after a shared prefix for this rotamer
/// is less than that atom for the other rotamer
////////////////////////////////////////////////////////////////////////////////
bool rotamer_descriptor::operator < (const rotamer_descriptor & rhs) const
{	int fewer_atoms =
		(num_atoms_ < rhs.num_atoms_ ? num_atoms_ : rhs.num_atoms_);
	for ( int ii = 0; ii < fewer_atoms; ++ii )
	{
		if ( atoms_[ii] < rhs.atoms_[ii] )
			return true;
		else if ( ! (atoms_[ii] == rhs.atoms_[ii]) )
			return false;
	}
	return num_atoms_ < rhs.num_atoms_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_descriptor::count_atoms_in_common
///
/// @brief
/// returns the length of the shared prefix for two rotamers
///
/// @detailed
///
/// @param
/// rhs - [in] - the rotamer descriptor to compare this rotamer to
///
////////////////////////////////////////////////////////////////////////////////
int
rotamer_descriptor::count_atoms_in_common(
	const rotamer_descriptor & rhs
) const
{	int fewer_atoms = (num_atoms_ < rhs.num_atoms_ ?
		num_atoms_ : rhs.num_atoms_);
	for ( int ii = 0; ii < fewer_atoms; ++ii )
	{
		if ( ! (atoms_[ii] == rhs.atoms_[ii]))
			return ii;
	}
	return fewer_atoms;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin operator << (ostream & os, rotamer_descriptor)
///
/// @brief
/// output a rotamer to an output stream
///
/// @detailed
///
/// @param
/// os - [in/out] - the output stream to which to write
/// rd - [in] - the rotamer descriptor to output
///
////////////////////////////////////////////////////////////////////////////////
ostream & operator << (ostream & os, const rotamer_descriptor & rd)
{	os << "rotamer " << rd.rotamer_id_ << " ";
	for ( int ii = 0; ii < rd.num_atoms_; ++ii )
		os << rd.atoms_[ii] << ":";
	return os;
}

//---------------------------------------------trie_node-----------------------//

const unsigned short
trie_node::AA_BITS = 0xf800;  //WARNING: assumes aa can be specified with 5 bits.
const int
trie_node::AA_BITSHIFT = 11;  //AA_BITS >> AA_BITSHIFT = 0x1f
trie_node::trie_node() :
	flags_(0), flags2_(0), 	hybridization_(0), aa_variant_(0)
	,rotamers_in_subtree_(0), sibling_(0), radius_containing_subtree_(0.0f)
{}

trie_node::~trie_node() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::trie_node
///
/// @brief copy constructor
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
trie_node::trie_node( const trie_node & rhs)
	: ad_(rhs.ad_), flags_(rhs.flags_), flags2_(rhs.flags2_),
	hybridization_(rhs.hybridization_), aa_variant_(rhs.aa_variant_)
	, rotamers_in_subtree_(rhs.rotamers_in_subtree_), sibling_(rhs.sibling_),
	radius_containing_subtree_( rhs.radius_containing_subtree_)
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::trie_node
///
/// @brief
/// standard constructor
///
/// @detailed
/// initializes many internal flags on the basis of which atom type
/// this node is to represent, and from which amino acid this atom originated
///
/// @param
/// atom - [in] - the atom descriptor for this atom
/// hbchk - [in] - the two-element boolean array declaring whether the backbone
///   nitrogen and oxygen are already participating in backbone/backone
///   hydrogen bonds
/// aa_type - [in] - the amino acid this atom comes from; important only
///   so long as amino acid dependent hydrogen bond weights exist.
///
/// @remark
/// amino acid specific hydrogen bond weights put great strain on the
/// trie vs trie algorithm, which otherwise asserts that two atoms from
/// different amino acids, but with the same atom type, the same xyz coordinate
/// and the same orientation vector are the same atom.
/// With the atom ordering I've chosen, it will never happen that two atoms
/// like those I just described will ever be declared the same atom; however,
/// future non-native amino acids might cause this rule to be violated.
/// I would argue that it's better to take out amino-acid-specific hydrogen
/// bond weights than to bend the trie to accomodate them.
////////////////////////////////////////////////////////////////////////////////
trie_node::trie_node
(
	const atom_descriptor & atom,
	FArray1DB_bool & hbchk,
	int aa_type
) :
	ad_(atom),
	flags_(0), flags2_(0),
	hybridization_(0), aa_variant_(0), rotamers_in_subtree_(0),
	sibling_(0), radius_containing_subtree_(0.0f)

{	using namespace aaproperties_pack;

	unsigned char atomtype   = ad_.get_atom_type();

	assert(atomtype != 0);

	flags_         |= IS_H * (! atomtype_isheavyatom(atomtype) ? 1 : 0);
	flags_         |= IS_BB * (atomtype_isbb(atomtype) ? 1 : 0);
	flags_         |= IS_DONOR_H * (atomtype_isdonorH(atomtype) ? 1 : 0);
	flags_         |= IS_ACCEPTOR * (atomtype_isacceptor(atomtype) ? 1 : 0);
	flags_         |= IS_BBHBG_SAT *
		(atomtype_hbchk_bbhbg_sat(atomtype, hbchk) ? 1 : 0);
	flags2_        |= BB_N * ( atomtype_is_bb_n(atomtype) ? 1 : 0);
	flags2_        |= BB_H * ( atomtype_is_bb_h(atomtype) ? 1 : 0);
	flags2_        |= BB_CA * (atomtype_is_bb_ca(atomtype) ? 1 : 0);
	flags2_        |= BB_C * (atomtype_is_bb_c(atomtype) ? 1 : 0);
	flags2_        |= BB_O * (atomtype_is_bb_o(atomtype) ? 1 : 0);
	//assume non-cb; requires all CB's and HA's be later identified.
	flags2_        |= SC_OTHER * ( flags_ & IS_BB ? 0 : 1);
	hybridization_ = hybridization(atomtype);

	//currently save amino acid type only in donor/acceptor nodes.
	//store it only for the sidechain atoms.
	if ( (flags_ & (IS_DONOR_H | IS_ACCEPTOR)) && ! ( flags_ & IS_BB ))
		set_aatype(aa_type);
}

inline bool trie_node::is_rotamer_terminal() const
{return ROTAMER_TERMINAL & flags_;}

inline bool trie_node::is_hydrogen() const
{return IS_H & flags_;}

inline bool trie_node::is_backbone() const
{return IS_BB & flags_;}

inline bool trie_node::is_donor_h() const
{return IS_DONOR_H & flags_;}

inline bool trie_node::is_acceptor() const
{return IS_ACCEPTOR & flags_;}

inline bool trie_node::is_bbhbg_sat() const
{return IS_BBHBG_SAT & flags_;}

inline bool trie_node::has_sibling() const
{
	return sibling_ != 0;
}

void trie_node::set_is_bb_c()
{
	flags2_ &= ~SC_OTHER;
	flags_  |= IS_BB;
	flags2_ |= BB_C;
}

void trie_node::set_is_bb_ha() {
	flags2_ &= ~SC_OTHER;
	flags_  |= IS_BB;
	flags2_ |= BB_HA;
	return;
}

void trie_node::set_is_sc_cb() {
	flags2_ |= SC_CB;
	flags2_ &= ~SC_OTHER;
	return;
} //set SC_CB true; set SC_OTHER false

void trie_node::set_is_pro_cd(){flags2_ |= PRO_CD; return;}
void trie_node::set_is_pro_hd(){flags2_ |= PRO_HD; return;}

////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::get_res_i_equivalence_class()
///
/// @brief
/// accessor used to refactor semantics of count_pair()
////////////////////////////////////////////////////////////////////////////////
inline int trie_node::get_res_i_equivalence_class()
{  if (SC_OTHER        & flags2_) return 4;
   else if (BB_N_CB_HA & flags2_) return 1;
	else if (BB_CA_O    & flags2_) return 2;
	else if (BB_C       & flags2_) return 3;
	else return 5; //last case,  (BB_H       & flags2_)  is true.
}

atom_descriptor trie_node::get_atom() const {return ad_;}

////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::set_terminal(bool)
///
/// @brief
/// sets the flag stating that this atom is the last atom for a rotamer
////////////////////////////////////////////////////////////////////////////////
void trie_node::set_terminal(bool is_terminal) {
	 flags_ &= ~ROTAMER_TERMINAL;
	 flags_ |= ROTAMER_TERMINAL * (is_terminal ? 1 : 0);
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::set_sibling(int)
///
/// @brief
/// sets sibling to point at sibling node, or sets HAS_SIBLING flag to "true"
/// depending on what compiler macros are defined
////////////////////////////////////////////////////////////////////////////////
void trie_node::set_sibling(int sibling)
{
	sibling_ = sibling;
	return;
}

void trie_node::set_first_atom_in_branch()
{
	flags_ |= FIRST_ATOM_IN_BRANCH;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::set_aatype
///
/// @brief
/// set the amino acid type for a node
///
/// @detailed
/// needed only to manage amino-acid specific hydrogen bond weights
////////////////////////////////////////////////////////////////////////////////
void trie_node::set_aatype( int aa )
{
  assert(aa <= MAX_HBONDING_AA_TYPES && aa > 0);
	
  flags2_ &= ~AA_BITS;
  flags2_ |= aa << AA_BITSHIFT;
  return;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin trie_node::set_aavariant
///
/// @brief
/// a hydrogen bonding side chain atom must keep track of which amino acid
/// and amino acid variant it originated from for the sake of the atom-and-
/// amino-acid-specific hydrogen bonding weights defined in sc_hbW.  Should be
/// called exactly once.
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
void trie_node::set_aavariant(int aa_variant)
{
	assert(  flags_ & (IS_DONOR_H | IS_ACCEPTOR) && ! ( flags_ & IS_BB ) );
	assert(aa_variant_ == 0 && aa_variant > 0 && aa_variant < 255 );
	aa_variant_ = aa_variant;
}

inline int trie_node::get_aatype() const
{
	return (AA_BITS & flags2_) >> AA_BITSHIFT;
}

int trie_node::get_aavariant() const {return aa_variant_;}

////////////////////////////////////////////////////////////////////////////////
//----------------------------Rotamer Trie Class------------------------------//
////////////////////////////////////////////////////////////////////////////////


rotamer_trie::rotamer_trie()
{}
rotamer_trie::~rotamer_trie()
{
	for (int ii = 1; ii <= num_distinct_backbones_; ++ii)
	{
		if ( tries_(ii) != NULL)
		{
			delete [] tries_(ii);
			tries_(ii) = NULL;
		}
	}
}

rotamer_trie::rotamer_trie(const rotamer_trie & )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::rotamer_trie(int, std::vector< rotamer_descriptor> &
///
/// @brief
/// apl  constructor that takes sorted vector of rotamer_descriptors
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors - apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
rotamer_trie::rotamer_trie
(
	int resind,
	std::vector< rotamer_descriptor > & rotamers
) : res_id_(resind)
{
	utility::vector1< int > dummy_one_backbone_vector(1, 0);
	construct_rotamer_trie( resind, rotamers, dummy_one_backbone_vector );
}

/// @ breif constructor for generating a single trie object for multiple backbones;
/// internally represented as separate tries
rotamer_trie::rotamer_trie
(
	int resind,
	std::vector< rotamer_descriptor > & rotamers,
	utility::vector1< int > & rotamer_offsets_for_backbone
) : res_id_(resind)
{
	construct_rotamer_trie( resind, rotamers, rotamer_offsets_for_backbone );
}

/// @ breif constructor for a single-rotamer trie
rotamer_trie::rotamer_trie
(
	int resind,
	rotamer_descriptor & rotamer,
	bool build_sc_trie_only
)
	: res_id_( resind )
{
	if (! build_sc_trie_only )
	{
		utility::vector1< int > dummy_one_backbone_vector(1, 0);
		std::vector< rotamer_descriptor > rotamer_vect(1); //occupy 0th position
		rotamer_vect.push_back( rotamer );
		construct_rotamer_trie( resind, rotamer_vect, dummy_one_backbone_vector );
		set_backbone_focus( 1 );
	}
	else
	{
		construct_one_rotamer_trie( rotamer );
	}


}

/// @ brief initialization method called by the various constructors
void
rotamer_trie::construct_rotamer_trie
(
	int resind,
	std::vector< rotamer_descriptor > & rotamers,
	utility::vector1< int > & rotamer_offsets_for_backbone
)
{	using namespace hbonds;
	using namespace param_aa;
	using namespace template_pack;

	num_total_rotamers_ = rotamers.size() - 1;
	num_distinct_backbones_ = rotamer_offsets_for_backbone.size();
	total_rot_2_unique_preorder_.resize( num_distinct_backbones_ + 1);

	tries_.dimension( num_distinct_backbones_ );
	num_atoms_for_bb_.dimension( num_distinct_backbones_ );
	num_heavyatoms_for_bb_.dimension( num_distinct_backbones_ );
	num_unique_rotamers_for_bb_.dimension( num_distinct_backbones_ );
	num_total_rotamers_for_bb_.dimension( num_distinct_backbones_ );
	total_rotamer_offset_for_bb_.dimension( num_distinct_backbones_ );

	num_total_atoms_ = 0;

	for (int ii = 1; ii <= num_distinct_backbones_; ++ii)
	{
		set_backbone_focus(ii);
		int ii_rot_offset = rotamer_offsets_for_backbone[ii];

		total_rotamer_offset_for_bb_( ii ) = ii_rot_offset;

		if (ii == num_distinct_backbones_)
		{
			num_total_rotamers_for_bb_(ii)
				= num_total_rotamers_ - ii_rot_offset;
		}
		else
		{
			num_total_rotamers_for_bb_(ii)
				= rotamer_offsets_for_backbone[ii+1] - ii_rot_offset;
		}
		//std::cerr << "Residue: " << resind << ", backbone: " << ii << ", num_total_rotamers ";
		//std::cerr << num_total_rotamers_for_bb_(ii) << std::endl;

		int ii_num_total_rotamers = num_total_rotamers_for_bb_(ii);
		total_rot_2_unique_preorder_[ii].resize( ii_num_total_rotamers + 1);

		//apl should produce bounds check error
		//total_rot_2_unique_preorder_[ii][ ii_num_total_rotamers + 2] = 111;
		//total_rot_2_unique_preorder_[ii].at(ii_num_total_rotamers + 2) = 111;


		vector< int > node_stack(rotamer_trie::MAX_ATS_PER_RES + 1);

		//this should change, since the backbone is moving... in fact, the npd
		// hbond term concerning already-formed backbone-backbone hbonds will
		// not work in this case.
		FArray1Da_bool hbchk_this_res = hbond_set.hbchkPair(resind);
		num_heavyatoms_for_bb_(ii) = 0;

		// index ii represents the number of atoms rotamer ii has with ii-1
		vector< int >  num_shared_atoms( num_total_rotamers_for_bb_(ii)+1, 1);

		for ( int jj = 2; jj <= ii_num_total_rotamers; ++jj )
		{
			int jj_p_offset = ii_rot_offset + jj;
			num_shared_atoms[jj] = rotamers[jj_p_offset].
				count_atoms_in_common(rotamers[jj_p_offset-1]);
		}


		int count_num_shared_atoms = 0;
		int num_atoms_in_trie = rotamers[1].get_num_atoms();
		for ( int jj = 2; jj <= num_total_rotamers_for_bb_(ii); ++jj )
		{
			int jj_offset_for_bb = ii_rot_offset + jj;
			num_atoms_in_trie +=
				rotamers[jj_offset_for_bb].get_num_atoms() - num_shared_atoms[jj];
			count_num_shared_atoms += num_shared_atoms[jj];
		}

		//cerr << "Num atoms in trie "<<  resind << ": " << num_atoms_in_trie;
		//cerr << " down from " << num_atoms_in_trie + count_num_shared_atoms <<endl;
		//trie_.resize( num_atoms_in_trie + 1); //index by 1
		tries_(ii) = new trie_node[ num_atoms_in_trie + 1];
		num_atoms_for_bb_(ii) = num_atoms_in_trie;
		num_total_atoms_ += num_atoms_in_trie;

		//add the first rotamer
		//cerr << "add the first rotamer" << endl;
		max_heavyatom_depth_ = 0;
		int heavyatoms_at_depth[rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
		heavyatoms_at_depth[1] = 0;

		int first_rotamer = ii_rot_offset + 1;
		add_atom_to_trie( 1, 1, rotamers[first_rotamer].get_atom(1),
			hbchk_this_res,
			rotamers[first_rotamer].get_aa_type(),
			rotamers[first_rotamer].get_aa_variant());

		tries_(ii)[1].set_first_atom_in_branch();
		if (! tries_(ii)[1].is_hydrogen()) ++num_heavyatoms_for_bb_(ii);
		node_stack[1] = 1;

		for ( int jj = 2; jj <= rotamers[first_rotamer].get_num_atoms(); ++jj )
		{	//cerr << "atom : " << ii << endl;
			add_atom_to_trie( jj, jj, rotamers[first_rotamer].get_atom(jj),
				hbchk_this_res, rotamers[first_rotamer].get_aa_type(),
				rotamers[first_rotamer].get_aa_variant());
			if (! tries_(ii)[jj].is_hydrogen()) ++num_heavyatoms_for_bb_(ii);
			node_stack[jj] = jj;
		}
		int count_unique_rotamers = 1;
		tries_(ii)[ rotamers[first_rotamer].get_num_atoms()].set_terminal( true );

		//apl test
		assert(rotamers[first_rotamer].get_rotamer_id() - ii_rot_offset < ii_num_total_rotamers + 1);

		total_rot_2_unique_preorder_[ii][rotamers[first_rotamer]
			.get_rotamer_id() - ii_rot_offset] = count_unique_rotamers;

		int count_atoms_placed = rotamers[first_rotamer].get_num_atoms();
		//done adding first rotamer; add successive rotamers.

		for ( int jj = 2; jj <= num_total_rotamers_for_bb_(ii); ++jj )
		{	int jj_offset_for_bb = total_rotamer_offset_for_bb_(ii) + jj;
			int jj_num_shared_with_prev = num_shared_atoms[jj];
			int jj_num_atoms            = rotamers[jj_offset_for_bb].get_num_atoms();
			int jj_first_distinguished  = jj_num_shared_with_prev + 1;
			//cerr << "rotamer : " << jj << endl;

			//apl test
			assert(rotamers[jj_offset_for_bb].get_rotamer_id() - ii_rot_offset < ii_num_total_rotamers + 1
				&& rotamers[jj_offset_for_bb].get_rotamer_id() - ii_rot_offset > 0);


			if (jj_num_shared_with_prev == jj_num_atoms)
			{
				//duplicate rotamer

				total_rot_2_unique_preorder_[ii]
					[rotamers[jj_offset_for_bb].get_rotamer_id() - ii_rot_offset]
					= count_unique_rotamers;
				continue;
			}

			++count_atoms_placed;
			add_atom_to_trie( count_atoms_placed, jj_first_distinguished,
				rotamers[jj_offset_for_bb].get_atom(jj_first_distinguished),
				hbchk_this_res,
				rotamers[jj_offset_for_bb].get_aa_type(),
				rotamers[jj_offset_for_bb].get_aa_variant());
			tries_(ii)[count_atoms_placed].set_first_atom_in_branch();

			if (! tries_(ii)[count_atoms_placed].is_hydrogen() ) ++num_heavyatoms_for_bb_(ii);
			//apl test
			assert(  node_stack[jj_first_distinguished]  <= num_atoms_in_trie
				&& node_stack[jj_first_distinguished]  > 0);

			tries_(ii)[ node_stack[jj_first_distinguished] ]
				.set_sibling(count_atoms_placed);
			node_stack[jj_first_distinguished] = count_atoms_placed;

			for ( int kk = jj_num_shared_with_prev + 2; kk <= jj_num_atoms; ++kk )
			{	//cerr << "atom : " << kk << endl;
				++count_atoms_placed;
				add_atom_to_trie( count_atoms_placed, kk,
					rotamers[jj_offset_for_bb].get_atom(kk),
					hbchk_this_res,
					rotamers[jj_offset_for_bb].get_aa_type(),
					rotamers[jj_offset_for_bb].get_aa_variant());

				if (! tries_(ii)[count_atoms_placed].is_hydrogen()) ++num_heavyatoms_for_bb_(ii);
				node_stack[ kk ] = count_atoms_placed;
			}

			++count_unique_rotamers;
			total_rot_2_unique_preorder_[ii][rotamers[jj_offset_for_bb]
				.get_rotamer_id() - ii_rot_offset ] = count_unique_rotamers;

			tries_(ii)[ count_atoms_placed ].set_terminal( true );
		}

		num_unique_rotamers_for_bb_(ii) = count_unique_rotamers;
	}

	compute_max_stack_height();

	for (int ii = 1; ii <= num_distinct_backbones_; ++ii)
	{
		set_backbone_focus(ii);
		calculate_num_rotamers_in_subtree();
		calculate_subtree_containing_radii();
	}
	set_no_backbone_focused();

	num_neighbors_ = neighbors(res_id_);


	//cerr << "Constructor complete" << endl;
	return;
}

/// @ brief initialization method called by the single-rotamer trie constructor
void
rotamer_trie::construct_one_rotamer_trie(
	rotamer_descriptor & rotamer
)
{
	using namespace hbonds;
	using namespace param_aa;
	using namespace template_pack;

	num_total_rotamers_ = 1;
	num_distinct_backbones_ = 1;
	total_rot_2_unique_preorder_.resize( 2 );

	FArray1Da_bool hbchk_this_res = hbond_set.hbchkPair(res_id_);
	hbchk_this_res.dimension(2);

	tries_.dimension( num_distinct_backbones_ );
	num_atoms_for_bb_.dimension( num_distinct_backbones_ );
	num_heavyatoms_for_bb_.dimension( num_distinct_backbones_ );
	num_unique_rotamers_for_bb_.dimension( num_distinct_backbones_ );
	num_total_rotamers_for_bb_.dimension( num_distinct_backbones_ );
	total_rotamer_offset_for_bb_.dimension( num_distinct_backbones_ );

	set_backbone_focus( 1 );

	num_heavyatoms_for_bb_(1) = 0;
	num_unique_rotamers_for_bb_(1) = 1;
	num_total_rotamers_for_bb_(1) = 1;
	total_rotamer_offset_for_bb_(1) = 0;

	int num_sc_atoms = 0;
	for (int ii = 1; ii <= rotamer.get_num_atoms(); ++ii)
	{
		if ( rotamer_atom_in_sidechain_proper( rotamer, ii) )
		{
			++num_sc_atoms;
		}
	}
	num_atoms_for_bb_(1) = num_sc_atoms;
	num_total_atoms_ = num_sc_atoms;
	if ( num_sc_atoms == 0 )
	{
		tries_(1) = 0;
		return; //constructed empty trie
	}

	tries_(1) = new trie_node[ num_sc_atoms + 1];
	int num_atoms_inserted_into_trie = 0;
	for (int ii = 1; ii <= rotamer.get_num_atoms(); ++ii)
	{
		if ( rotamer_atom_in_sidechain_proper( rotamer, ii ) )
		{
			++num_atoms_inserted_into_trie;
			add_atom_to_trie( num_atoms_inserted_into_trie, ii,
			rotamer.get_atom( ii ), hbchk_this_res,
			rotamer.get_aa_type(), rotamer.get_aa_variant() );
			if (! tries_(1)[num_atoms_inserted_into_trie].is_hydrogen()) ++num_heavyatoms_for_bb_(1);
			if ( num_atoms_inserted_into_trie == 1 )
			{
				tries_(1)[num_atoms_inserted_into_trie].set_first_atom_in_branch();
			}
		}
	}
	tries_(1)[ num_atoms_inserted_into_trie].set_terminal( true );
	total_rot_2_unique_preorder_[1].resize( 2 );
	total_rot_2_unique_preorder_[1][1] = 1;

	compute_max_stack_height();
	calculate_num_rotamers_in_subtree();
	calculate_subtree_containing_radii();

	num_neighbors_ = neighbors(res_id_);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::add_atom_to_trie
///
/// @brief
/// handles details of adding an atom to the trie, used by constructor
///
/// @detailed
/// unfortunately contains some of the messy / dangerous semantics
/// for initializing trie_nodes.  Proline CD and HD1 & HD2 have specific
/// properties according to count_pair().  Glycine has no CB; both 1HA and 2HA
/// must be described as alpha-hydrogens for count_pair's purposes
/// 02/02/06 - adding terminal amino-acid-variant functionality.  I intend
/// to refactor this design.  This code is tightly coupled to
/// param_rotamer_trie.cc; that's not a good thing.
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::add_atom_to_trie(
	int trie_atom_id,
	int atom_id_in_rotamer,
	atom_descriptor const & ad,
	FArray1DB_bool & hbchk_this_res,
	int aa_type,
	int aa_variant)
{
	using namespace param_rotamer_trie;

	assert( current_backbone_ != 0 );

	tries_(current_backbone_)[ trie_atom_id ] = trie_node( ad, hbchk_this_res, aa_type );

	if ( aa_variant == trie_variant_for_c_terminus && atom_id_in_rotamer == 1 )
	{
		tries_(current_backbone_)[trie_atom_id].set_is_bb_c();
	}
	else if (aa_type == aa_pro )
	{
		if (aa_variant == 1)
		{
			if (atom_id_in_rotamer == 5) //pro HA
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			else if (atom_id_in_rotamer == 6) // pro CB
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			else if (atom_id_in_rotamer == 12) // pro CD;
				tries_(current_backbone_)[trie_atom_id].set_is_pro_cd();
			else if (atom_id_in_rotamer == 13 || atom_id_in_rotamer == 14)
				tries_(current_backbone_)[trie_atom_id].set_is_pro_hd();
		}
		else if ( aa_variant == trie_variant_for_n_terminus )
		{
			if (atom_id_in_rotamer == 7) //pro HA
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			else if (atom_id_in_rotamer == 8) // pro CB
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			else if (atom_id_in_rotamer == 14) // pro CD;
				tries_(current_backbone_)[trie_atom_id].set_is_pro_cd();
			else if (atom_id_in_rotamer == 15 || atom_id_in_rotamer == 16)
				tries_(current_backbone_)[trie_atom_id].set_is_pro_hd();
		}
		else if ( aa_variant == trie_variant_for_c_terminus )
		{
			if (atom_id_in_rotamer == 6) //pro HA
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			else if (atom_id_in_rotamer == 7) // pro CB
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			else if (atom_id_in_rotamer == 13) // pro CD;
				tries_(current_backbone_)[trie_atom_id].set_is_pro_cd();
			else if (atom_id_in_rotamer == 14 || atom_id_in_rotamer == 15)
				tries_(current_backbone_)[trie_atom_id].set_is_pro_hd();
		}

	}
	else if (aa_type == aa_gly)
	{
		if (aa_variant == 1)
		{
			if (atom_id_in_rotamer == 6 || atom_id_in_rotamer == 7)
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
		}
		else if ( aa_variant == trie_variant_for_n_terminus )
		{
			if (atom_id_in_rotamer == 8 || atom_id_in_rotamer == 9)
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
		}
		else if ( aa_variant == trie_variant_for_c_terminus )
		{
			if (atom_id_in_rotamer == 7 || atom_id_in_rotamer == 8)
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
		}

	}
	else
	{
		if (aa_variant == 1 || aa_variant == 2)
		{
			if (atom_id_in_rotamer == 6)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			}
			else if (atom_id_in_rotamer == 7)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			}
		}
		else if ( aa_variant == trie_variant_for_n_terminus )
		{
			if (atom_id_in_rotamer == 8)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			}
			else if (atom_id_in_rotamer == 9)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			}
		}
		else if ( aa_variant == trie_variant_for_c_terminus )
		{
			if (atom_id_in_rotamer == 7)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_bb_ha();
			}
			else if (atom_id_in_rotamer == 8)
			{
				tries_(current_backbone_)[trie_atom_id].set_is_sc_cb();
			}
		}

	}

	//must keep track of aa-variant type for atom's original rotamer
	//if that atom can hydrogen bond.
	if ( (tries_(current_backbone_)[ trie_atom_id ].is_donor_h() ||
		tries_(current_backbone_)[ trie_atom_id ].is_acceptor() ) &&
		! tries_(current_backbone_)[ trie_atom_id ].is_backbone() )
	{
		tries_(current_backbone_)[trie_atom_id].set_aavariant( aa_variant );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::get_num_atoms
///
/// @brief returns the (total) number of atoms in the trie
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////

int
rotamer_trie::get_num_atoms() const
{
	return num_total_atoms_;
}

int rotamer_trie::get_num_total_rotamers() const
{
	return num_total_rotamers_;
}

int rotamer_trie::get_resid() const
{
	return res_id_;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::count_num_heavyatoms
///
/// @brief
/// counts the number of non-hydrogen (heavy) atoms in the trie
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
// removing vestigal code
// int rotamer_trie::count_num_heavyatoms() const
// {	int count_heavy_seen = 0;
// 	for ( int ii = 1; ii <= num_atoms_in_trie_; ++ii )
// 	{
// 		if (! trie_[ii].is_hydrogen()) ++count_heavy_seen;
// 	}
// 	return count_heavy_seen;
// }


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_vs_trie
///
/// @brief
/// wrapper which decides between versions of the
/// iterative trie-vs-trie algorithm to call
///
/// @detailed
/// the trie vs trie algorithm needs a place to which to write the
/// rotamer-pair energies.  In addition, it needs a place
/// to write them temporarily.  Both proxy arrays it is handed
/// must be of dimension( rt.num_atoms, this->num_atoms ).
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::trie_vs_trie
(  rotamer_trie & rt,
	FArray2D_float & rotamer_rotamer_table,
	FArray2D_float & temp_table,
	bool same_fragment
)
{
	assert(res_id_ < rt.res_id_);

	//rotamer_rotamer_table.dimension(rt.num_total_rotamers_, num_total_rotamers_);
	//temp_table.dimension(rt.num_unique_rotamers_, num_unique_rotamers_);

	int rot_rotE_start_lindex = 0;

	int this_num_backbones = num_distinct_backbones_;
	int rt_num_backbones = rt.num_distinct_backbones_;

	if (same_fragment)
	{
		assert( this_num_backbones == rt_num_backbones);
		for (int ii = 1; ii <= this_num_backbones; ++ii)
		{
			set_backbone_focus( ii );
			rt.set_backbone_focus( ii );

			int this_total_rots = get_num_total_rotamers_for_curr_bb();
			int this_unique_rots = get_num_unique_rotamers_for_curr_bb();
			int rt_total_rots = rt.get_num_total_rotamers_for_curr_bb();
			int rt_unique_rots = rt.get_num_unique_rotamers_for_curr_bb();

			FArray2Da_float rot_rotE_proxy( rotamer_rotamer_table[rot_rotE_start_lindex],
				rt_total_rots, this_total_rots);
			FArray2Da_float scratch_space_proxy( temp_table(1,1),
				rt_unique_rots, this_unique_rots);

			rot_rotE_start_lindex += rt_total_rots * this_total_rots;

			if (rt.res_id_ - res_id_ == 1)
			{
				trie_v_trie_iter_i_ip1(rt, scratch_space_proxy);
			}
			else if (rt.res_id_ - res_id_ == 2)
			{
				trie_v_trie_iter_i_ip2(rt, scratch_space_proxy);
			}
			else
			{
				trie_v_trie_iter_i_ip3orMore(rt, scratch_space_proxy);
			}


			convert_preorder_rot_rot_table(rt, scratch_space_proxy, rot_rotE_proxy);

		}
		set_no_backbone_focused();
		rt.set_no_backbone_focused();
	}
	else
	{

		for (int ii = 1; ii <= this_num_backbones; ++ii)
		{

			set_backbone_focus(ii);
			int ii_total_rots = get_num_total_rotamers_for_curr_bb();
			int ii_unique_rots = get_num_unique_rotamers_for_curr_bb();
			for (int jj = 1; jj <= rt_num_backbones; ++jj)
			{
				rt.set_backbone_focus(jj);
				int jj_total_rots = rt.get_num_total_rotamers_for_curr_bb();
				int jj_unique_rots = rt.get_num_unique_rotamers_for_curr_bb();
				FArray2Da_float rot_rotE_proxy( rotamer_rotamer_table[ rot_rotE_start_lindex],
					jj_total_rots , ii_total_rots);
				FArray2Da_float scratch_space_proxy( temp_table(1,1),
					jj_unique_rots, ii_unique_rots);

				rot_rotE_start_lindex += jj_total_rots * ii_total_rots;

				if (rt.res_id_ - res_id_ == 1)
				{
					trie_v_trie_iter_i_ip1(rt, scratch_space_proxy);
				}
				else if (rt.res_id_ - res_id_ == 2)
				{
					trie_v_trie_iter_i_ip2(rt, scratch_space_proxy);
				}
				else
				{
					trie_v_trie_iter_i_ip3orMore(rt, scratch_space_proxy);
				}

				convert_preorder_rot_rot_table(rt, scratch_space_proxy, rot_rotE_proxy);

			}
		}
		set_no_backbone_focused();
		rt.set_no_backbone_focused();
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_vs_background
///
/// @brief
/// calculates the rotamer pair energy for one (large) rotamer set
/// with another rotamer set that contains exactly one rotamer.  Since
/// the residue for the trie with a single rotamer is fixed in place,
/// call it part of the background.
/// The output for the four components of the energy function are
/// separated instead of summed.
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::trie_vs_background
(	rotamer_trie & bg,
	FArray2Da_float trie_trials_componentE,
	FArray1Da_float trie_trials_sumE
)
{
	assert( num_distinct_backbones_ == 1 && bg.num_distinct_backbones_ == 1);

	trie_trials_componentE.dimension(4, num_total_rotamers_for_bb_(1) );
	trie_trials_sumE.dimension(num_total_rotamers_for_bb_(1));

	FArray2D_float trie_trials_componentE_preorder(4, num_unique_rotamers_for_bb_(1), 0. );
	FArray1D_float trie_trials_sumE_preorder( num_unique_rotamers_for_bb_(1), 0. );

	set_backbone_focus(1);
	bg.set_backbone_focus(1);

	int bbdist = std::abs(res_id_ - bg.res_id_);
	assert(bbdist != 0);
	if (bbdist < 3)
	{	trie_vs_background_i_ip1or2( bg, trie_trials_componentE_preorder,
			trie_trials_sumE_preorder );
	}
	else
	{	trie_vs_background_i_ip3orMore( bg, trie_trials_componentE_preorder,
			trie_trials_sumE_preorder );
	}

	convert_preorder_trials_data( trie_trials_componentE_preorder,
		trie_trials_componentE, trie_trials_sumE_preorder, trie_trials_sumE);

	return;
}

float
rotamer_trie::path_vs_path(
	rotamer_trie const & rt
) const
{
	if ( std::abs( res_id_ - rt.res_id_ ) > 2 )
	{
		return path_vs_path_i_ip3orMore( rt );
	}
	else
	{
		return path_vs_path_i_ip1or2( rt );
	}
}


float
rotamer_trie::path_vs_path_i_ip3orMore
(
	rotamer_trie const & rt
) const
{
	using namespace pdbstatistics_pack; //hydrogen_interaction_cutoff defined here.
	using namespace param_pack;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};

	float esum = 0;
	int s_heavy_depth = 0;

	int const this_num_atoms = get_num_atoms_for_curr_bb();
	int const rt_num_atoms = rt.get_num_atoms_for_curr_bb();
	int const rt_num_heavyatoms = rt.get_num_heavy_atoms_for_curr_bb();
	int const rt_num_heavyatoms_p_1 = rt_num_heavyatoms + 1;

	trie_node* rt_trie = rt.get_trie_for_curr_bb();
	trie_node* this_trie = get_trie_for_curr_bb();

	//FArray1D_bool parent_wi_h_dist( rt_num_heavyatoms, false );
	bool parent_wi_h_dist[ 40 ];
	//FArray1D_bool r_heavy_skip_s_subtree( rt_num_heavyatoms, false );
	int r_heavy_skip_s_subtree_depth = rt_num_heavyatoms_p_1;

	set_backbone_focus(1);
	rt.set_backbone_focus(1);

	for (int ii = 1; ii <= this_num_atoms; ++ii )
	{
		trie_node & r = this_trie[ii];
		s_heavy_depth = 0;
		if ( r.is_hydrogen() )
		{
			if (r_heavy_skip_s_subtree_depth != 1)
			{
			for (int jj = 1; jj <= rt_num_atoms; ++jj )
			{
				trie_node & s = rt_trie[jj];
				if ( s.is_hydrogen() )
				{
					if (parent_wi_h_dist[ s_heavy_depth ] )
					{
						esum +=  atom_atom_energy( r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond );
					}
				}
				else
				{
					++s_heavy_depth;

					if (r_heavy_skip_s_subtree_depth == s_heavy_depth) break;

					if ( parent_wi_h_dist[s_heavy_depth] )
					{
						esum += atom_atom_energy( r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond );
					}
				}
			}
			}
		}
		else
		{
			r_heavy_skip_s_subtree_depth = rt_num_heavyatoms_p_1;
			for (int jj = 1; jj <= rt_num_atoms; ++jj)
			{
				trie_node & s = rt_trie[jj];
				if ( s.is_hydrogen())
				{
					if (parent_wi_h_dist[ s_heavy_depth ] )
					{
						esum += atom_atom_energy(r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond);
					}
				}
				else
				{
					++s_heavy_depth;
					float d2;
					esum += heavyatom_heavyatom_energy(r, s, d2);

					parent_wi_h_dist[s_heavy_depth] =
						(d2 < hydrogen_interaction_cutoff);
					//const float subtree_heavy_square_sum =
					//	(( (fa_max_dis/2) + s.radius_containing_subtree_) *
					//	((fa_max_dis/2) + s.radius_containing_subtree_));
					if (d2 > s.radius_containing_subtree_)
					{
						r_heavy_skip_s_subtree_depth = s_heavy_depth;
						break;
					}
				}
			}
		}
	}

	return esum;
}

float
rotamer_trie::path_vs_path_i_ip1or2
(
	rotamer_trie const & rt
) const
{
	using namespace pdbstatistics_pack; //hydrogen_interaction_cutoff defined here.
	using namespace param_pack;

	//set count pair equivalence classes based on this.res_id_ - rt.res_id_
	unsigned short this_equivalence_class_bitmasks[6];
	//column 0: 0 equiv class, column 1: half equiv class.
	unsigned short rt_equivalence_class_bitmasks[6][2];

	prepare_count_pair_equivalence_class_masks( rt.res_id_, res_id_,
		this_equivalence_class_bitmasks, rt_equivalence_class_bitmasks);

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};
	float const count_pair_half_weight = { 0.2 };

	float esum = 0;
	int s_heavy_depth = 0;

	int const this_num_atoms = get_num_atoms_for_curr_bb();
	int const rt_num_atoms = rt.get_num_atoms_for_curr_bb();
	int const rt_num_heavyatoms = rt.get_num_heavy_atoms_for_curr_bb();
	int const rt_num_heavyatoms_p_1 = rt_num_heavyatoms + 1;

	trie_node* rt_trie = rt.get_trie_for_curr_bb();
	trie_node* this_trie = get_trie_for_curr_bb();

	//FArray1D_bool parent_wi_h_dist( rt_num_heavyatoms, false );
	bool  parent_wi_h_dist[ 40 ];
	//FArray1D_bool r_heavy_skip_s_subtree( rt_num_heavyatoms, false );
	int r_heavy_skip_s_subtree_depth = rt_num_heavyatoms_p_1;

	set_backbone_focus(1);
	rt.set_backbone_focus(1);

	for (int ii = 1; ii <= this_num_atoms; ++ii )
	{
		trie_node & r = this_trie[ii];
		if ( r.is_hydrogen() && r_heavy_skip_s_subtree_depth == 1)
		{
			continue;
		}

		unsigned short * r_class_masks = NULL;
		for (int eq_class = 0; eq_class < 6; ++eq_class)
		{	if (this_equivalence_class_bitmasks[eq_class] & r.flags2_)
			{  r_class_masks = rt_equivalence_class_bitmasks[eq_class];
				break;
			}
		}
		assert( r_class_masks );

		s_heavy_depth = 0;
		if ( r.is_hydrogen() )
		{
			for (int jj = 1; jj <= rt_num_atoms; ++jj )
			{
				trie_node & s = rt_trie[jj];
				if ( s.is_hydrogen() )
				{
					if (parent_wi_h_dist[ s_heavy_depth ]
						&& (! (s.flags2_ & r_class_masks[0]))  )
					{
						float tempE =  atom_atom_energy( r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond );
						if (s.flags2_ & r_class_masks[1])
						{
							tempE *= count_pair_half_weight;
						}
						if ( trie_output_all_energy_calcs ) std::cerr << tempE << std::endl;
						esum += tempE;
					}
				}
				else
				{
					++s_heavy_depth;

					if (r_heavy_skip_s_subtree_depth == s_heavy_depth) break;

					if ( parent_wi_h_dist[s_heavy_depth] &&
						(! (s.flags2_ & r_class_masks[0])) )
					{
						float tempE = atom_atom_energy( r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond );
						if (s.flags2_ & r_class_masks[1])
						{
							tempE *= count_pair_half_weight;
						}
						if ( trie_output_all_energy_calcs ) std::cerr << tempE << std::endl;
						esum += tempE;
					}
				}
			}
		}
		else
		{
			r_heavy_skip_s_subtree_depth = rt_num_heavyatoms_p_1;
			for (int jj = 1; jj <= rt_num_atoms; ++jj)
			{
				trie_node & s = rt_trie[jj];
				if ( s.is_hydrogen())
				{
					if (parent_wi_h_dist[ s_heavy_depth ]  &&
						(! (s.flags2_ & r_class_masks[0])) )
					{
						float tempE = atom_atom_energy(r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond);
						if (s.flags2_ & r_class_masks[1])
						{
							tempE *= count_pair_half_weight;
						}
						if ( trie_output_all_energy_calcs ) std::cerr << tempE << std::endl;
						esum += tempE;
					}
				}
				else
				{
					++s_heavy_depth;
					float d2;
					float tempE = heavyatom_heavyatom_energy(r, s, d2);
					if (s.flags2_ & r_class_masks[0])
					{
						tempE = 0;
					}
					else if (s.flags2_ & r_class_masks[1])
					{
						tempE *= count_pair_half_weight;
					}
					if ( trie_output_all_energy_calcs ) std::cerr << tempE << std::endl;
					esum += tempE;

					parent_wi_h_dist[s_heavy_depth] =
						(d2 < hydrogen_interaction_cutoff);
					//const float subtree_heavy_square_sum =
					//	(( (fa_max_dis/2) + s.radius_containing_subtree_) *
					//	((fa_max_dis/2) + s.radius_containing_subtree_));
					if (d2 > s.radius_containing_subtree_)
					{
						r_heavy_skip_s_subtree_depth = s_heavy_depth;
						break;
					}
				}
			}
		}
	}

	return esum;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_v_trie_iter_i_ip1
///
/// @brief
/// apl  iterative trie_vs_trie algorithm, specially adapted to handle
/// apl  interactions between residue i and i+1.  Semantics originally
/// apl  described in count_pair.h
///
/// @detailed
///  Backbone distance between rotamer tries R and S is 1, call them i,
///  i+1. R is i, S is i+1.  Special rules ("described" in count_pair.h)
///  apply to backbone/backbone interactions.
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::trie_v_trie_iter_i_ip1(
	rotamer_trie & rt,
	FArray2DB_float & rot_rot_table
)
{
	using namespace pdbstatistics_pack; //hydrogen_interaction_cutoff defined here.
	using namespace param_pack;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};

	// count pairs, in its current incarnation, defines five equivalence
	///classes for the interactions of the atoms of residue i in their
	// interactions with the atoms on residue i+1.
	//
	// Residue i equivalence clases:
	// I:		N, CB, HA
	// II:	CA, O
	// III:	C
	// IV:	SC (not CB)
	//	V:		H (HN)
	// Interactions with atoms on res i+1
	//		Class I:
	//		zero : N;				half: CA H PRO_CD			full: rest
	//    Class II:
	//		zero : N CA H PRO_HD	half: C O CB HA PRO_HD	full: SC
	//    Class III:
	//		zero : N CA C CB H HA PRO_CD PRO_HD	half: O	full: SC
	//		Class IV:
	//		full: ALL
	//		Class V:
	//		zero : none				half: N CA PRO_CD			full: rest


	//cerr << "begin trie_v_trie_iterative i, i+1" << endl;

	int rt_num_atoms = rt.get_num_atoms_for_curr_bb();
	int rt_num_heavyatoms = rt.get_num_heavy_atoms_for_curr_bb();
	int rt_num_unique_rotamers = rt.get_num_unique_rotamers_for_curr_bb();

	int this_num_atoms = get_num_atoms_for_curr_bb();
	//int this_num_heavyatoms = get_num_heavy_atoms_for_curr_bb();
	//int this_num_unique_rotamers = get_num_unique_rotamers_for_curr_bb();

	trie_node* rt_trie = rt.get_trie_for_curr_bb();
	trie_node* this_trie = get_trie_for_curr_bb();

	//rt.initialize_trie_for_trie_vs_trie();

	//cerr << "max_stack_height" << max_stack_height_ << endl;
	//cerr << "max_heavyatom_depth_" << max_heavyatom_depth_ << endl;
	FArray2D_float at_v_rot_stack
		( rt_num_unique_rotamers, max_stack_height_, 0.0);
	FArray2D_bool  parent_heavy_wi_hydrogen_cutoff
		(rt_num_heavyatoms, max_heavyatom_depth_, true);

	FArray2D_bool  r_heavy_skip_s_subtree
		(rt_num_heavyatoms, max_heavyatom_depth_, false);

	float energy_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];  //index by 1, + 1 for root_depth = 1.
	energy_stack[1] = 0;

	//Larger than needed; need be only as large as rt.max_heavyatom_depth_
	//but declared on the stack, and slightly more cache efficient
	bool parent_heavy_wi_hcut_stack[rotamer_trie::MAX_ATS_PER_RES + 1];

	//vector<int> s_sibling_stack(rotamer_trie::MAX_ATS_PER_RES + 1);
	int s_sibling_stack[rotamer_trie::MAX_ATS_PER_RES + 1];
	s_sibling_stack[1] = rt_num_atoms + 1;

	// int   r_heavy_depth_stack[ max_stack_height_ + 1];
	// int   r_tree_depth_stack [ max_stack_height_ + 1];
	// int   s_heavy_depth_stack[ rt.max_stack_height_ + 1];
	// int   s_tree_depth_stack[ rt.max_stack_height_ + 1];
	std::vector< int >   r_heavy_depth_stack( max_stack_height_ + 1   );
	//std::vector< int >   r_tree_depth_stack ( max_stack_height_ + 1   );
	std::vector< int >   s_heavy_depth_stack( rt.max_stack_height_ + 1);
	//std::vector< int >   s_tree_depth_stack ( rt.max_stack_height_ + 1);

	int r_curr_stack_top = 2;
	int s_curr_stack_top;
	int r_rotamers_seen = 0;
	int s_rotamers_seen;
	int s_heavyatoms_seen;
	r_heavy_depth_stack[1] = 0;
	//r_tree_depth_stack[1]  = 0;

	const float count_pair_half = 0.2f;
	//const float count_pair_full = 1.0f;

	unsigned short count_pair_ip1_zero_equiv_class = 0x0000;
	unsigned short count_pair_ip1_half_equiv_class = 0x0000;

	//cerr.precision(5);

	for ( int ii = 1; ii <= this_num_atoms; ++ii )
	{	//cerr << "ii : " << ii << " " << r_curr_stack_top << " " <<
		//r_heavy_depth_stack[ r_curr_stack_top ] ;
		energy_stack[1] = 0;
		trie_node & r = this_trie[ii];
		s_rotamers_seen = 0;
		s_heavyatoms_seen = 0;
		if (r.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{  --r_curr_stack_top;
		}


		//if (r._depth <= r_last_tree_depth)
		//{	//pop the stack;
		//	--r_curr_stack_top;
		//}
		//cerr << r._depth << " ";
		// r_last_tree_depth = r._depth;

		if (r.has_sibling() )
		{	//push - copy stack downwards

			++r_curr_stack_top;
			FArray1Da_float at_rot_array_d_proxy
				(at_v_rot_stack(1, r_curr_stack_top));
			FArray1Da_float at_rot_array_dminus1_proxy
				(at_v_rot_stack(1, r_curr_stack_top-1));
			at_rot_array_d_proxy.dimension(rt_num_unique_rotamers);
			at_rot_array_dminus1_proxy.dimension( rt_num_unique_rotamers );

			at_rot_array_d_proxy = at_rot_array_dminus1_proxy;
			r_heavy_depth_stack[ r_curr_stack_top ]
				= r_heavy_depth_stack[ r_curr_stack_top-1 ];
			//r_tree_depth_stack[ r_curr_stack_top ]
			//	= r_tree_depth_stack[ r_curr_stack_top-1 ];
		}

		//++r_tree_depth_stack[ r_curr_stack_top ];

		if (! r.is_hydrogen() )
		{
			++r_heavy_depth_stack[ r_curr_stack_top ];
			//assert(r._heavyatom_depth == r_heavy_depth_stack[ r_curr_stack_top]);
		}

		FArray1Da_float at_rot_array_proxy(at_v_rot_stack(1, r_curr_stack_top));
		at_rot_array_proxy.dimension( rt_num_unique_rotamers);

		FArray1Da_bool parent_wi_h_dist( parent_heavy_wi_hydrogen_cutoff
			( 1, r_heavy_depth_stack[ r_curr_stack_top ]) );
		parent_wi_h_dist.dimension(rt_num_heavyatoms);

		FArray1Da_bool r_heavy_skip_s_subtree_proxy( r_heavy_skip_s_subtree
			(1, r_heavy_depth_stack[ r_curr_stack_top ]));
		r_heavy_skip_s_subtree_proxy.dimension(rt_num_heavyatoms);

		int r_i_equiv_class = r.get_res_i_equivalence_class();
		switch (r_i_equiv_class )
		{
			case 1:
				count_pair_ip1_zero_equiv_class = trie_node::BB_N;
				count_pair_ip1_half_equiv_class = trie_node::BB_CA_H_PCD;
			break;
			case 2:
				count_pair_ip1_zero_equiv_class = trie_node::BB_N_CA_H_PCD;
				count_pair_ip1_half_equiv_class = trie_node::BB_C_O_CB_HA_PHD;
			break;
			case 3:
				count_pair_ip1_zero_equiv_class =
					trie_node::BB_N_CA_C_CB_H_HA_PCD_PHD;
				count_pair_ip1_half_equiv_class = trie_node::BB_O;
			break;
			case 4:
				//Nothing; sc/everything interactions counted at full strength.
			break;
			case 5:
				count_pair_ip1_zero_equiv_class = 0;
				count_pair_ip1_half_equiv_class = trie_node::BB_N_CA_PCD;
			break;
		}

		switch (r_i_equiv_class )
		{
			case 1: case 2: case 3: case 5:
				// the following code copy/pasted and modified to weight some pairs at half
				//cerr << "case 1: case 2: case 3: case 5:" << endl;
				if (r.is_hydrogen())
				{
					s_curr_stack_top = 2;
					s_heavy_depth_stack[1] = 0;
					//s_tree_depth_stack[1] = 0;

					for ( int jj = 1; jj <= rt_num_atoms; ++jj )
					{	//cerr << "ii is H, jj : " << jj << " " << s_curr_stack_top
						//<< " " << s_rotamers_seen << " " << s_heavyatoms_seen << endl;

						//cerr << "s_sibling_stack: [";
						//for ( int kk = 1; kk <= 14; ++kk )
						//	cerr << s_sibling_stack[kk] << " ";
						//cerr << " " <<  endl;

						trie_node & s = rt_trie[jj];
						if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
						{	--s_curr_stack_top;}
						// if (s._depth <= s_last_tree_depth)
						// {	--s_curr_stack_top;}
						// s_last_tree_depth = s._depth;

						if (s.has_sibling() )
						{	//push - copy stack downwards
							++s_curr_stack_top;
							energy_stack[s_curr_stack_top]
								= energy_stack[s_curr_stack_top - 1];
							s_heavy_depth_stack[ s_curr_stack_top]
								= s_heavy_depth_stack[ s_curr_stack_top - 1];
							//s_tree_depth_stack[ s_curr_stack_top]
							//	= s_tree_depth_stack[ s_curr_stack_top - 1];
							s_sibling_stack[s_curr_stack_top] = s.sibling_;
						}

						//++s_tree_depth_stack[s_curr_stack_top];

						if (!s.is_hydrogen()) {
							++s_heavyatoms_seen;
							++s_heavy_depth_stack[s_curr_stack_top];
							//assert(s._heavyatom_depth == s_heavy_depth_stack[ s_curr_stack_top]);
							parent_heavy_wi_hcut_stack
								[s_heavy_depth_stack[s_curr_stack_top]]
								= parent_wi_h_dist(s_heavyatoms_seen);


							if ( r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) )
							{
								if (energy_stack[s_curr_stack_top] != 0.0f )
								{
									for ( int kk = s_rotamers_seen + 1;
										kk <= s_rotamers_seen + s.rotamers_in_subtree_;
										++kk )
									{
										at_rot_array_proxy(kk)
											+= energy_stack[s_curr_stack_top];
									}
								}
								s_rotamers_seen   += s.rotamers_in_subtree_;
								jj = s_sibling_stack[s_curr_stack_top] - 1;
								continue;
							}
						}

						if ( parent_heavy_wi_hcut_stack
							[s_heavy_depth_stack[s_curr_stack_top]] &&
							! (count_pair_ip1_zero_equiv_class & s.flags2_ ) )
						{

							float this_energy = atom_atom_energy(r, s,
								res_id_, rt.res_id_,
								num_neighbors_, rt.num_neighbors_,
								Whbond);
							if ( count_pair_ip1_half_equiv_class & s.flags2_ )
							{
								this_energy *= count_pair_half;
							}
							energy_stack[ s_curr_stack_top ] += this_energy;
							//std::cerr << "this_energy: " << this_energy << std::endl; //TEMP DEBUG
						}

						if (s.is_rotamer_terminal() )
						{	++s_rotamers_seen;
							at_rot_array_proxy(s_rotamers_seen)
								+= energy_stack[ s_curr_stack_top ];
						}
					}
				}
				else //r is a heavy atom
				{
					s_curr_stack_top = 2;
					s_heavy_depth_stack[1] = 0;
					//s_tree_depth_stack[1] = 0;
					r_heavy_skip_s_subtree_proxy = false;


					for ( int jj = 1; jj <= rt_num_atoms; ++jj )
					{	//cerr << "! ii is H, jj : " << jj << " " <<
						//s_curr_stack_top << " " << s_rotamers_seen << " "
						//<< s_heavyatoms_seen << endl;
						trie_node & s = rt_trie[jj];

						//cerr << "s_sibling_stack [";
						//for ( int kk = 1; kk <= 14; ++kk )
						//	cerr << s_sibling_stack[kk] << " ";
						//cerr << " " <<  endl;
						if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
						{	--s_curr_stack_top;}


						// if (s._depth <= s_last_tree_depth)
						// {	--s_curr_stack_top;}
                  //
						// s_last_tree_depth = s._depth;

						if (s.has_sibling() )
						{	//push - copy stack downwards
							++s_curr_stack_top;
							energy_stack[s_curr_stack_top]
								= energy_stack[s_curr_stack_top - 1];
							s_heavy_depth_stack[ s_curr_stack_top]
								= s_heavy_depth_stack[ s_curr_stack_top - 1];
							//s_tree_depth_stack[ s_curr_stack_top]
							//	= s_tree_depth_stack[ s_curr_stack_top - 1];
							s_sibling_stack[s_curr_stack_top] = s.sibling_;
						}

						//++s_tree_depth_stack[s_curr_stack_top];

						if (s.is_hydrogen())
						{
							if (parent_heavy_wi_hcut_stack
								[s_heavy_depth_stack[ s_curr_stack_top]] &&
								! (count_pair_ip1_zero_equiv_class & s.flags2_ ) )
							{
								float this_energy = atom_atom_energy(r, s,
									res_id_, rt.res_id_,
									num_neighbors_, rt.num_neighbors_,
									Whbond);
								if (count_pair_ip1_half_equiv_class & s.flags2_)
								{
									this_energy *= count_pair_half;
								}
								energy_stack[ s_curr_stack_top ] += this_energy;
								//std::cerr << "this_energy: " << this_energy << std::endl; //TEMP DEBUG
							}

						}
						else
						{
							++s_heavyatoms_seen;
							++s_heavy_depth_stack[ s_curr_stack_top];

							float d2;

							float this_energy = heavyatom_heavyatom_energy(r, s, d2);
							if ( count_pair_ip1_zero_equiv_class & s.flags2_)
							{
								this_energy = 0;
							}
							else if (count_pair_ip1_half_equiv_class & s.flags2_)
							{
								this_energy *= count_pair_half;
							}
							energy_stack[s_curr_stack_top] += this_energy;

							parent_heavy_wi_hcut_stack[s_heavy_depth_stack
								[ s_curr_stack_top]] =
								parent_wi_h_dist(s_heavyatoms_seen) =
								(d2 < hydrogen_interaction_cutoff);

							if (d2 > s.radius_containing_subtree_)
							{
								r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) = true;
								if (energy_stack[s_curr_stack_top] != 0.0f )
								{
									for ( int kk = s_rotamers_seen + 1;
										kk <= s_rotamers_seen + s.rotamers_in_subtree_;
										++kk )
									{
										at_rot_array_proxy(kk)
											+= energy_stack[s_curr_stack_top];
									}
								}
								jj = s_sibling_stack[s_curr_stack_top] - 1;
								s_rotamers_seen += s.rotamers_in_subtree_;
								continue;

							}

						}
						if (s.is_rotamer_terminal() )
						{	++s_rotamers_seen;
							at_rot_array_proxy(s_rotamers_seen)
								+= energy_stack[s_curr_stack_top];
						}
					}
				}

			break; //break case 1,2,3 & 5
			case 4:
				//This code, copy and pasted and left in its original form;
				// no half/zero checks needed.
				//cerr << "case 4:" << endl;
				if (r.is_hydrogen())
				{
					s_curr_stack_top = 2;
					s_heavy_depth_stack[1] = 0;
					//s_tree_depth_stack[1] = 0;

					for ( int jj = 1; jj <= rt_num_atoms; ++jj )
					{	//cerr << "ii is H, jj : " << jj << " " << s_curr_stack_top
						//<< " " << s_rotamers_seen << " " << s_heavyatoms_seen
						//<< endl;

						//cerr << "s_sibling_stack: [";
						//for ( int kk = 1; kk <= 14; ++kk )
						//	cerr << s_sibling_stack[kk] << " ";
						//cerr << " " <<  endl;

						trie_node & s = rt_trie[jj];
						if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
						{	--s_curr_stack_top;}

						if (s.has_sibling() )
						{	//push - copy stack downwards
							++s_curr_stack_top;
							energy_stack[s_curr_stack_top]
								= energy_stack[s_curr_stack_top - 1];
							s_heavy_depth_stack[ s_curr_stack_top]
								= s_heavy_depth_stack[ s_curr_stack_top - 1];
							//s_tree_depth_stack[ s_curr_stack_top]
							//	= s_tree_depth_stack[ s_curr_stack_top - 1];
							s_sibling_stack[s_curr_stack_top] = s.sibling_;
						}

						//++s_tree_depth_stack[ s_curr_stack_top];

						if (!s.is_hydrogen()) {
							++s_heavyatoms_seen;
							++s_heavy_depth_stack[s_curr_stack_top];

							parent_heavy_wi_hcut_stack[s_heavy_depth_stack
								[s_curr_stack_top]] = parent_wi_h_dist(s_heavyatoms_seen);

							if ( r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) )
							{
								if (energy_stack[s_curr_stack_top] != 0.0f )
									for ( int kk = s_rotamers_seen + 1;
										kk <= s_rotamers_seen + s.rotamers_in_subtree_;
										++kk )
									{	at_rot_array_proxy(kk)
											+= energy_stack[s_curr_stack_top];
									}
								s_rotamers_seen += s.rotamers_in_subtree_;
								jj = s_sibling_stack[s_curr_stack_top] - 1;
								continue;
							}

						}

						if ( parent_heavy_wi_hcut_stack
							[s_heavy_depth_stack[s_curr_stack_top]] )
						{
							energy_stack[ s_curr_stack_top ] += atom_atom_energy(r, s,
								res_id_, rt.res_id_,
								num_neighbors_, rt.num_neighbors_,
								Whbond);
						}

						if (s.is_rotamer_terminal() )
						{	++s_rotamers_seen;
							at_rot_array_proxy(s_rotamers_seen)
								+= energy_stack[ s_curr_stack_top ];
						}
					}
				}
				else //r is a heavy atom
				{
					s_curr_stack_top = 2;
					s_heavy_depth_stack[1] = 0;
					//s_tree_depth_stack[1] = 0;
					r_heavy_skip_s_subtree_proxy = false;

					for ( int jj = 1; jj <= rt_num_atoms; ++jj )
					{	//cerr << "! ii is H, jj : " << jj << " " << s_curr_stack_top <<
						// " " << s_rotamers_seen << " " << s_heavyatoms_seen << endl;
						trie_node & s = rt_trie[jj];

						if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
						{	--s_curr_stack_top;}

						//cerr << "s_sibling_stack [";
						//for ( int kk = 1; kk <= 14; ++kk )
						//	cerr << s_sibling_stack[kk] << " ";
						//cerr << " " <<  endl;

						if (s.has_sibling() )
						{	//push - copy stack downwards
							++s_curr_stack_top;
							energy_stack[s_curr_stack_top]
								= energy_stack[s_curr_stack_top - 1];
							s_heavy_depth_stack[ s_curr_stack_top]
								= s_heavy_depth_stack[ s_curr_stack_top - 1];
							//s_tree_depth_stack[ s_curr_stack_top]
							//	= s_tree_depth_stack[ s_curr_stack_top - 1];
							s_sibling_stack[s_curr_stack_top] = s.sibling_;
						}

						//++s_tree_depth_stack[ s_curr_stack_top];

						if (s.is_hydrogen())
						{
							if (parent_heavy_wi_hcut_stack
								[s_heavy_depth_stack[ s_curr_stack_top]] )
							{
								energy_stack[ s_curr_stack_top ] += atom_atom_energy
									(r, s,
									res_id_, rt.res_id_,
									num_neighbors_, rt.num_neighbors_,
									Whbond);
							}

						}
						else
						{	++s_heavyatoms_seen;
							++s_heavy_depth_stack[ s_curr_stack_top];

							float d2;

							energy_stack[s_curr_stack_top]
								+= heavyatom_heavyatom_energy(r, s, d2);

							parent_heavy_wi_hcut_stack
								[s_heavy_depth_stack[ s_curr_stack_top]] =
								parent_wi_h_dist(s_heavyatoms_seen) =
								(d2 < hydrogen_interaction_cutoff);


							if (d2 > s.radius_containing_subtree_)
							{
								r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) = true;
								if (energy_stack[s_curr_stack_top] != 0.0f )
								{
									for ( int kk = s_rotamers_seen + 1;
										kk <= s_rotamers_seen + s.rotamers_in_subtree_;
										++kk )
									{
										at_rot_array_proxy(kk) +=
											energy_stack[s_curr_stack_top];
									}
								}
								jj = s_sibling_stack[s_curr_stack_top] - 1;
								s_rotamers_seen += s.rotamers_in_subtree_;
								continue;

							}

						}
						if (s.is_rotamer_terminal() )
						{	++s_rotamers_seen;
							at_rot_array_proxy(s_rotamers_seen)
								+= energy_stack[s_curr_stack_top];
						}
					}
				}


			break; //break case 4.
		}

		if (r.is_rotamer_terminal())
		{	++r_rotamers_seen;

			FArray1Da_float rot_rot_table_row( rot_rot_table(1, r_rotamers_seen));
			rot_rot_table_row.dimension(rt_num_unique_rotamers);

			rot_rot_table_row = at_rot_array_proxy;

			//cerr << "rotamer " << r.is_rotamer_terminal() << " energies ";

			//for ( int jj = 1, lrot_rot_array =
			//	rot_rot_table.index(1, r_rotamers_seen),
			//	lat_rot_array_proxy = at_rot_array_proxy.index(1);
			//	jj <= rt_num_unique_rotamers; ++jj,
			//	++lrot_rot_array, ++lat_rot_array_proxy )
			//{	rot_rot_table[ lrot_rot_array ]
			//			= at_rot_array_proxy[ lat_rot_array_proxy ];
			//   //cerr << " " << at_rot_array_proxy[ lat_rot_array_proxy ];
			//}
			//cerr << endl;
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_v_trie_iter_i_ip2
///
/// @brief
/// apl  iterative trie_vs_trie algorithm, specially adapted to handle
/// apl  interactions between residue i and i+2 (ip2).  Semantics originally
/// apl  described in count_pair.h
///
/// @detailed
/// apl  Backbone distance between rotamer tries R and S is 2, call them i,
/// apl  i+2. R is i, S is i+2.  Special rules ("described" in count_pair.h)
/// apl  apply to backbone/backbone interactions.  According to count_pair(),
/// apl  C(i) interacts with N(i+2) with "half" weight.
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::trie_v_trie_iter_i_ip2(
	rotamer_trie & rt,
	FArray2DB_float & rot_rot_table
)
{
	//hydrogen_interaction_cutoff defined in pdbstatistics_pack.
	using namespace pdbstatistics_pack;
	using namespace param_pack;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};

	//cerr << "begin trie_v_trie_iterative i, i+2" << endl;

	int rt_num_atoms = rt.get_num_atoms_for_curr_bb();
	int rt_num_heavyatoms = rt.get_num_heavy_atoms_for_curr_bb();
	int rt_num_unique_rotamers = rt.get_num_unique_rotamers_for_curr_bb();

	int this_num_atoms = get_num_atoms_for_curr_bb();
	//int this_num_heavyatoms = get_num_heavy_atoms_for_curr_bb();
	//int this_num_unique_rotamers = get_num_unique_rotamers_for_curr_bb();

	trie_node* rt_trie = rt.get_trie_for_curr_bb();
	trie_node* this_trie = get_trie_for_curr_bb();

	//rt.initialize_trie_for_trie_vs_trie();

	FArray2D_float at_v_rot_stack
		( rt_num_unique_rotamers, max_stack_height_, 0.0);
	FArray2D_bool  parent_heavy_wi_hydrogen_cutoff
		(rt_num_heavyatoms, max_heavyatom_depth_, true);

	FArray2D_bool  r_heavy_skip_s_subtree
		(rt_num_heavyatoms, max_heavyatom_depth_, false);

	//index by 1, + 1 for root_depth = 1.
	float energy_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	energy_stack[1] = 0;

	//Larger than needed; need be only as large as rt.max_heavyatom_depth_
	//but declared on the stack, and slightly more cache efficient
	bool parent_heavy_wi_hcut_stack[rotamer_trie::MAX_ATS_PER_RES + 1];


	int s_sibling_stack[rotamer_trie::MAX_ATS_PER_RES + 1];
	s_sibling_stack[1] = rt_num_atoms + 1;


	std::vector< int >   r_heavy_depth_stack( max_stack_height_ + 1   );
	//std::vector< int >   r_tree_depth_stack ( max_stack_height_ + 1   );
	std::vector< int >   s_heavy_depth_stack( rt.max_stack_height_ + 1);
	//std::vector< int >   s_tree_depth_stack ( rt.max_stack_height_ + 1);


	int r_curr_stack_top = 2;
	int s_curr_stack_top;
	int r_rotamers_seen = 0;
	int s_rotamers_seen;
	int s_heavyatoms_seen;

	r_heavy_depth_stack[1] = 0;
	//r_tree_depth_stack[1] = 0;

	const float count_pair_half = 0.2f;

	//cerr.precision(5);

	for ( int ii = 1; ii <= this_num_atoms; ++ii )
	{	//cerr << "ii : " << ii << " ";
		energy_stack[1] = 0;
		trie_node & r = this_trie[ii];
		s_rotamers_seen = 0;
		s_heavyatoms_seen = 0;

		if (r.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{  --r_curr_stack_top;
		}

		if (r.has_sibling() )
		{	//push - copy stack downwards

			++r_curr_stack_top;
			FArray1Da_float at_rot_array_d_proxy
				(at_v_rot_stack(1, r_curr_stack_top));
			FArray1Da_float at_rot_array_dminus1_proxy
				(at_v_rot_stack(1, r_curr_stack_top-1));
			at_rot_array_d_proxy.dimension
				(rt_num_unique_rotamers);
			at_rot_array_dminus1_proxy.dimension
				( rt_num_unique_rotamers );

			at_rot_array_d_proxy = at_rot_array_dminus1_proxy;
			r_heavy_depth_stack[ r_curr_stack_top ]
				= r_heavy_depth_stack[ r_curr_stack_top-1 ];
			//r_tree_depth_stack[ r_curr_stack_top ]
			//	= r_tree_depth_stack[ r_curr_stack_top-1 ];

		}

		//++r_tree_depth_stack[ r_curr_stack_top ];
		if (! r.is_hydrogen() )
		{
			++r_heavy_depth_stack[ r_curr_stack_top ];
		}


		FArray1Da_float at_rot_array_proxy(at_v_rot_stack(1, r_curr_stack_top));
		at_rot_array_proxy.dimension( rt_num_unique_rotamers);

		FArray1Da_bool parent_wi_h_dist
			( parent_heavy_wi_hydrogen_cutoff( 1,
			r_heavy_depth_stack[ r_curr_stack_top]) );
		parent_wi_h_dist.dimension(rt_num_heavyatoms);

		FArray1Da_bool r_heavy_skip_s_subtree_proxy
			( r_heavy_skip_s_subtree(1, r_heavy_depth_stack[ r_curr_stack_top]));
		r_heavy_skip_s_subtree_proxy.dimension(rt_num_heavyatoms);

		if (trie_node::BB_C & r.flags2_)
		{
		// the following code copy/pasted and modified to weight one pairs at half

		//else //r is a heavy atom
		//{
			s_curr_stack_top = 2;
			s_heavy_depth_stack[1] = 0;
			//s_tree_depth_stack[1] = 0;


			r_heavy_skip_s_subtree_proxy = false;


			for ( int jj = 1; jj <= rt_num_atoms; ++jj )
			{	//cerr << "! ii is H, jj : " << jj << " " << s_curr_stack_top <<
				//" " << s_rotamers_seen << " " << s_heavyatoms_seen << endl;
				trie_node & s = rt_trie[jj];
				if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
				{	--s_curr_stack_top;}


				//cerr << "s_sibling_stack [";
				//for ( int kk = 1; kk <= 14; ++kk )
				//	cerr << s_sibling_stack[kk] << " ";
				//cerr << " " <<  endl;

				if (s.has_sibling() )
				{	//push - copy stack downwards
					++s_curr_stack_top;
					energy_stack[s_curr_stack_top]
						= energy_stack[s_curr_stack_top - 1];
					s_heavy_depth_stack[ s_curr_stack_top]
						= s_heavy_depth_stack[ s_curr_stack_top - 1];
					//s_tree_depth_stack[ s_curr_stack_top]
					//	= s_tree_depth_stack[ s_curr_stack_top - 1];
					s_sibling_stack[s_curr_stack_top] = s.sibling_;
				}

				//++s_tree_depth_stack[ s_curr_stack_top];

				if (s.is_hydrogen())
				{
					//assert(s._heavyatom_depth == s_heavy_depth_stack[ s_curr_stack_top]);
					//if (true) //apl temp
					if (parent_heavy_wi_hcut_stack[s_heavy_depth_stack[ s_curr_stack_top]])
					{
						energy_stack[ s_curr_stack_top ] +=  atom_atom_energy(r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond);
					}

				}
				else
				{	++s_heavyatoms_seen;
					++s_heavy_depth_stack[s_curr_stack_top];
					//assert(s._heavyatom_depth ==
					//s_heavy_depth_stack[ s_curr_stack_top]);

					float d2;

					float this_energy = heavyatom_heavyatom_energy(r, s, d2);
					if ( trie_node::BB_N & s.flags2_)
					{
						this_energy *= count_pair_half;
					}
					energy_stack[s_curr_stack_top] += this_energy;

					parent_heavy_wi_hcut_stack
						[s_heavy_depth_stack[ s_curr_stack_top]] =
						parent_wi_h_dist(s_heavyatoms_seen) =
						(d2 < hydrogen_interaction_cutoff);

					if (d2 > s.radius_containing_subtree_)
					{
						r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) = true;
						if (energy_stack[s_curr_stack_top] != 0.0f )
						{
							for ( int kk = s_rotamers_seen + 1;
								kk <= s_rotamers_seen + s.rotamers_in_subtree_;
								++kk )
							{
								at_rot_array_proxy(kk) +=
									energy_stack[s_curr_stack_top];
							}
						}
						jj = s_sibling_stack[s_curr_stack_top] - 1;
						s_rotamers_seen += s.rotamers_in_subtree_;
						continue;

					}
				}
				if (s.is_rotamer_terminal() )
				{	++s_rotamers_seen;
					at_rot_array_proxy(s_rotamers_seen)
						+= energy_stack[s_curr_stack_top];
				}
			}
		}
		else //r is not bb_C
		{

			if (r.is_hydrogen())
			{
				s_curr_stack_top = 2;
				s_heavy_depth_stack[1] = 0;
				//s_tree_depth_stack[1] = 0;

				for ( int jj = 1; jj <= rt_num_atoms; ++jj )
				{	//cerr << "ii is H, jj : " << jj << " " << s_curr_stack_top << " "
					//	<< s_rotamers_seen << " " << s_heavyatoms_seen << endl;

					//cerr << "s_sibling_stack: [";
					//for ( int kk = 1; kk <= 14; ++kk )
					//	cerr << s_sibling_stack[kk] << " ";
					//cerr << " " <<  endl;

					trie_node & s = rt_trie[jj];
					if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
					{	--s_curr_stack_top;}


					// if (s._depth <= s_last_tree_depth)
					// {	--s_curr_stack_top;}
					// s_last_tree_depth = s._depth;

					if (s.has_sibling() )
					{	//push - copy stack downwards
						++s_curr_stack_top;
						energy_stack[s_curr_stack_top]
							= energy_stack[s_curr_stack_top - 1];
						s_heavy_depth_stack[ s_curr_stack_top]
							= s_heavy_depth_stack[ s_curr_stack_top - 1];
						//s_tree_depth_stack[ s_curr_stack_top]
						//	= s_tree_depth_stack[ s_curr_stack_top - 1];

						s_sibling_stack[s_curr_stack_top] = s.sibling_;
					}
					//++s_tree_depth_stack[ s_curr_stack_top];
					if (!s.is_hydrogen()) {
						++s_heavyatoms_seen;
						++s_heavy_depth_stack[s_curr_stack_top];
						parent_heavy_wi_hcut_stack
							[s_heavy_depth_stack[ s_curr_stack_top]] =
							parent_wi_h_dist(s_heavyatoms_seen);


						if (r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) )
						{
							if (energy_stack[s_curr_stack_top] != 0.0f )
							{
								for ( int kk = s_rotamers_seen + 1;
									kk <= s_rotamers_seen + s.rotamers_in_subtree_;
									++kk )
								{
									at_rot_array_proxy(kk) +=
										energy_stack[s_curr_stack_top];
								}
							}
							//s_heavyatoms_seen += s._heavyatoms_in_subtree;
							s_rotamers_seen   += s.rotamers_in_subtree_;
							jj = s_sibling_stack[s_curr_stack_top] - 1;
							continue;
						}
					}

					if ( parent_heavy_wi_hcut_stack
						[s_heavy_depth_stack[s_curr_stack_top]] )
					{
						energy_stack[ s_curr_stack_top ] +=
							atom_atom_energy(r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond);
					}

					if (s.is_rotamer_terminal() )
					{	++s_rotamers_seen;
						at_rot_array_proxy(s_rotamers_seen)
							+= energy_stack[ s_curr_stack_top ];
					}
				}
			}
			else //r is a heavy atom
			{
				s_curr_stack_top = 2;
				s_heavy_depth_stack[1] = 0;
				//s_tree_depth_stack[1] = 0;

				r_heavy_skip_s_subtree_proxy = false;

				for ( int jj = 1; jj <= rt_num_atoms; ++jj )
				{	//cerr << "! ii is H, jj : " << jj << " "
					//<< s_curr_stack_top << " " << s_rotamers_seen
					// << " " << s_heavyatoms_seen << endl;
					trie_node & s = rt_trie[jj];
					if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
					{	--s_curr_stack_top;}


					//cerr << "s_sibling_stack [";
					//for ( int kk = 1; kk <= 14; ++kk )
					//	cerr << s_sibling_stack[kk] << " ";
					//cerr << " " <<  endl;

					//if (s._depth <= s_last_tree_depth)
					//{	--s_curr_stack_top;}


					if (s.has_sibling() )
					{	//push - copy stack downwards
						++s_curr_stack_top;
						energy_stack[s_curr_stack_top]
							= energy_stack[s_curr_stack_top - 1];
						s_heavy_depth_stack[ s_curr_stack_top]
							= s_heavy_depth_stack[ s_curr_stack_top - 1];
						//s_tree_depth_stack[ s_curr_stack_top]
						//	= s_tree_depth_stack[ s_curr_stack_top - 1];
						s_sibling_stack[s_curr_stack_top] = s.sibling_;
					}
					//++s_tree_depth_stack[ s_curr_stack_top];

					if (s.is_hydrogen())
					{
						if (parent_heavy_wi_hcut_stack
							[s_heavy_depth_stack[s_curr_stack_top]] )
						{
							energy_stack[ s_curr_stack_top ] += atom_atom_energy(r, s,
								res_id_, rt.res_id_,
								num_neighbors_, rt.num_neighbors_,
								Whbond);
						}

					}
					else
					{	++s_heavyatoms_seen;
						++s_heavy_depth_stack[s_curr_stack_top];

						float d2;

						energy_stack[s_curr_stack_top]
							+= heavyatom_heavyatom_energy(r, s, d2);

						parent_heavy_wi_hcut_stack[s_heavy_depth_stack
							[s_curr_stack_top]] =
							parent_wi_h_dist(s_heavyatoms_seen) =
							(d2 < hydrogen_interaction_cutoff);

						if (d2 > s.radius_containing_subtree_)
						{
							r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) = true;
							if (energy_stack[s_curr_stack_top] != 0.0f )
							{
								for ( int kk = s_rotamers_seen + 1;
									kk <= s_rotamers_seen + s.rotamers_in_subtree_;
									++kk )
								{
									at_rot_array_proxy(kk)
										+= energy_stack[s_curr_stack_top];
								}
							}
							jj = s_sibling_stack[s_curr_stack_top] - 1;
							s_rotamers_seen += s.rotamers_in_subtree_;
							continue;

						}
					}
					if (s.is_rotamer_terminal() )
					{	++s_rotamers_seen;
						at_rot_array_proxy(s_rotamers_seen)
							+= energy_stack[s_curr_stack_top];
					}
				}
			}
		} // end else (r is bb_C)

		if (r.is_rotamer_terminal())
		{	++r_rotamers_seen;

			FArray1Da_float rot_rot_table_row
				( rot_rot_table(1, r_rotamers_seen));
			rot_rot_table_row.dimension
				(rt_num_unique_rotamers);

			rot_rot_table_row = at_rot_array_proxy;

		}
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_v_trie_iter_i_ip3orMore
///
/// @brief
/// apl  iterative trie_vs_trie algorithm, in its most pure form.  Count pair
/// apl  specifies that all atom pair interactions count between residues i
/// apl  and i+3 or greater
///
/// @detailed
/// The main structure, in
/// pseudocode:
/// Two rotamer tries, R and S.  (R is "this", S is rt)
/// for r \in R (in preorder traversal order)
///   for s \in S (in preorder traversal order)
///     calculate_atom_atom_energy(r, s)
///
/// @param
/// rt - [in] - the S rotamer trie (this is the R rotamer trie)
/// rot_rot_table - [out] - the table to which the output rotamer pair energies
///    should be written.
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::trie_v_trie_iter_i_ip3orMore(
	rotamer_trie & rt,
	FArray2DB_float & rot_rot_table
)
{
	using namespace pdbstatistics_pack; //hydrogen_interaction_cutoff defined here.
	using namespace param_pack;

	//_heavy_heavy_evals_count = 0;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};

	//cerr << "begin trie_v_trie_iterative i, i+3" << endl;

	int rt_num_atoms = rt.get_num_atoms_for_curr_bb();
	int rt_num_heavyatoms = rt.get_num_heavy_atoms_for_curr_bb();
	int rt_num_unique_rotamers = rt.get_num_unique_rotamers_for_curr_bb();

	int this_num_atoms = get_num_atoms_for_curr_bb();

	trie_node* rt_trie = rt.get_trie_for_curr_bb();
	trie_node* this_trie = get_trie_for_curr_bb();

	FArray2D_float at_v_rot_stack
		( rt_num_unique_rotamers, max_stack_height_, 0.0);
	FArray2D_bool  parent_heavy_wi_hydrogen_cutoff
		(rt_num_heavyatoms, max_heavyatom_depth_, true);

	FArray2D_bool r_heavy_skip_s_subtree
		(rt_num_heavyatoms, max_heavyatom_depth_, false);

	//index by 1, + 1 for root_depth = 1.
	float energy_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	energy_stack[1] = 0;
	energy_stack[0] = 0;

	std::vector< int >   r_heavy_depth_stack( max_stack_height_ + 1   );
	//std::vector< int >   r_tree_depth_stack ( max_stack_height_ + 1   );
	std::vector< int >   s_heavy_depth_stack( rt.max_stack_height_ + 1);
	//std::vector< int >   s_tree_depth_stack ( rt.max_stack_height_ + 1);

	//Larger than needed; need be only as large as rt.max_heavyatom_depth_
	//but declared on the stack, and slightly more cache efficient
	bool parent_heavy_wi_hcut_stack[rotamer_trie::MAX_ATS_PER_RES + 1];

	int s_sibling_stack[rotamer_trie::MAX_ATS_PER_RES + 1];
	s_sibling_stack[1] = rt_num_atoms + 1;
	for ( int ii = 2; ii <= MAX_ATS_PER_RES; ++ii )
		s_sibling_stack[ii] = 0;

	int r_curr_stack_top = 2; //immediately decremented to 1
	int s_curr_stack_top;
	int r_rotamers_seen = 0;
	int s_rotamers_seen;
	int s_heavyatoms_seen;

	r_heavy_depth_stack[1] = 0;
	//r_tree_depth_stack[1] = 0;

	for ( int ii = 1; ii <= this_num_atoms; ++ii )
	{	//cerr << "ii : " << ii << " ";
		energy_stack[1] = 0;
		trie_node & r = this_trie[ii];
		s_rotamers_seen = 0;
		s_heavyatoms_seen = 0;

		if (r.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{  --r_curr_stack_top;
		}

		if (r.has_sibling() )
		{	//push - copy stack downwards

			++r_curr_stack_top;

			FArray1Da_float at_rot_array_d_proxy
				(at_v_rot_stack(1, r_curr_stack_top));
			FArray1Da_float at_rot_array_dminus1_proxy
				(at_v_rot_stack(1, r_curr_stack_top-1));
			at_rot_array_d_proxy.dimension
				(rt_num_unique_rotamers);
			at_rot_array_dminus1_proxy.dimension
				( rt_num_unique_rotamers );

			at_rot_array_d_proxy = at_rot_array_dminus1_proxy;
			r_heavy_depth_stack[ r_curr_stack_top ]
				= r_heavy_depth_stack[ r_curr_stack_top-1 ];
			//r_tree_depth_stack[ r_curr_stack_top ]
			//	= r_tree_depth_stack[ r_curr_stack_top-1 ];
		}

		//++r_tree_depth_stack[ r_curr_stack_top ];

		if (! r.is_hydrogen() )
		{
			++r_heavy_depth_stack[ r_curr_stack_top ];
		}

		FArray1Da_float at_rot_array_proxy(at_v_rot_stack(1, r_curr_stack_top));
		at_rot_array_proxy.dimension( rt_num_unique_rotamers);

		FArray1Da_bool parent_wi_h_dist
			( parent_heavy_wi_hydrogen_cutoff( 1,
			r_heavy_depth_stack[ r_curr_stack_top ] ) );
		parent_wi_h_dist.dimension(rt_num_heavyatoms);

		FArray1Da_bool r_heavy_skip_s_subtree_proxy
			( r_heavy_skip_s_subtree(1, r_heavy_depth_stack[ r_curr_stack_top ] ));
		r_heavy_skip_s_subtree_proxy.dimension(rt_num_heavyatoms);

		if (r.is_hydrogen())
		{
			s_curr_stack_top = 2;
			s_heavy_depth_stack[1] = 0;
			//s_tree_depth_stack[1] = 0;

			for ( int jj = 1; jj <= rt_num_atoms; ++jj )
			{	//cerr << "ii is H, jj : " << jj << " " << s_curr_stack_top << " "
				//<< s_rotamers_seen << " " << s_heavyatoms_seen << endl;

				// cerr << "s_sibling_stack: [";
				// for ( int kk = 1; kk <= 14; ++kk )
				// 	cerr << s_sibling_stack[kk] << " ";
				// cerr << " " <<  endl;

				trie_node & s = rt_trie[jj];
				if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
				{	--s_curr_stack_top;}


				if (s.has_sibling() )
				{	//push - copy stack downwards
					++s_curr_stack_top;
					energy_stack[s_curr_stack_top]
						= energy_stack[s_curr_stack_top - 1];
					s_heavy_depth_stack[ s_curr_stack_top]
						= s_heavy_depth_stack[ s_curr_stack_top - 1];
					//s_tree_depth_stack[ s_curr_stack_top]
					//	= s_tree_depth_stack[ s_curr_stack_top - 1];
					s_sibling_stack[s_curr_stack_top] = s.sibling_;
				}
				//++s_tree_depth_stack[ s_curr_stack_top];

				if (!s.is_hydrogen()) {
					++s_heavyatoms_seen;
					++s_heavy_depth_stack[s_curr_stack_top];
					parent_heavy_wi_hcut_stack[s_heavy_depth_stack[s_curr_stack_top]]
						= parent_wi_h_dist(s_heavyatoms_seen);

					if ( r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) )
					{
						if (energy_stack[s_curr_stack_top] != 0.0f )
						{
							for ( int kk = s_rotamers_seen + 1;
								kk <= s_rotamers_seen + s.rotamers_in_subtree_;
								++kk )
							{
								at_rot_array_proxy(kk)
									+= energy_stack[s_curr_stack_top];
							}
						}
						s_rotamers_seen   += s.rotamers_in_subtree_;
						jj = s_sibling_stack[s_curr_stack_top] - 1;
						continue;
					}
				}

				if ( parent_heavy_wi_hcut_stack
					[s_heavy_depth_stack[s_curr_stack_top] ] )
				{
					energy_stack[ s_curr_stack_top ] += atom_atom_energy(r, s,
						res_id_, rt.res_id_,
						num_neighbors_, rt.num_neighbors_,
						Whbond);
				}

				if (s.is_rotamer_terminal() )
				{	++s_rotamers_seen;
					at_rot_array_proxy(s_rotamers_seen)
					+= energy_stack[ s_curr_stack_top ];
				}
			}
		}
		else //r is a heavy atom
		{
			s_curr_stack_top = 2;
			s_heavy_depth_stack[1] = 0;
			//s_tree_depth_stack[1] = 0;

			r_heavy_skip_s_subtree_proxy = false;

			for ( int jj = 1; jj <= rt_num_atoms; ++jj )
			{	//cerr << "! ii is H, jj : " << jj << " " << s_curr_stack_top <<
				//" " << s_rotamers_seen << " " << s_heavyatoms_seen << endl;
				trie_node & s = rt_trie[jj];

				// cerr << "s_sibling_stack [";
				// for ( int kk = 1; kk <= 14; ++kk )
				// 	cerr << s_sibling_stack[kk] << " ";
				// cerr << " " <<  endl;

				if (s.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
				{	--s_curr_stack_top;}

				//s_last_tree_depth = s._depth;

				if (s.has_sibling() )
				{	//push - copy stack downwards
					++s_curr_stack_top;
					energy_stack[s_curr_stack_top]
						= energy_stack[s_curr_stack_top - 1];
					s_heavy_depth_stack[ s_curr_stack_top]
						= s_heavy_depth_stack[ s_curr_stack_top - 1];
					//s_tree_depth_stack[ s_curr_stack_top]
					//	= s_tree_depth_stack[ s_curr_stack_top - 1];
					s_sibling_stack[s_curr_stack_top] = s.sibling_;
				}
				//++s_tree_depth_stack[ s_curr_stack_top];

				if (s.is_hydrogen())
				{
					if (parent_heavy_wi_hcut_stack[s_heavy_depth_stack[s_curr_stack_top]] )
					{
						energy_stack[ s_curr_stack_top ] += atom_atom_energy(r, s,
							res_id_, rt.res_id_,
							num_neighbors_, rt.num_neighbors_,
							Whbond);
					}

				}
				else
				{
					++s_heavyatoms_seen;
					++s_heavy_depth_stack[s_curr_stack_top];

					float d2;
				  	float e;
					e = heavyatom_heavyatom_energy(r, s, d2);
					energy_stack[s_curr_stack_top] += e;

					parent_heavy_wi_hcut_stack[s_heavy_depth_stack
						[s_curr_stack_top]] =
						parent_wi_h_dist(s_heavyatoms_seen) =
						(d2 < hydrogen_interaction_cutoff);

					if (d2 > s.radius_containing_subtree_)
					{
						r_heavy_skip_s_subtree_proxy(s_heavyatoms_seen) = true;
						if (energy_stack[s_curr_stack_top] != 0.0f )
						{
							for ( int kk = s_rotamers_seen + 1;
								kk <= s_rotamers_seen + s.rotamers_in_subtree_;
								++kk )
							{
								at_rot_array_proxy(kk) +=
									energy_stack[s_curr_stack_top];
							}
						}
						jj = s_sibling_stack[s_curr_stack_top] - 1;
						s_rotamers_seen += s.rotamers_in_subtree_;
						continue;
					}
				}
				if (s.is_rotamer_terminal() )
				{	++s_rotamers_seen;
					at_rot_array_proxy(s_rotamers_seen)
						+= energy_stack[s_curr_stack_top];
				}
			}
		}

		if (r.is_rotamer_terminal())
		{	++r_rotamers_seen;

			FArray1Da_float rot_rot_table_row( rot_rot_table(1, r_rotamers_seen));
			rot_rot_table_row.dimension(rt_num_unique_rotamers);

			rot_rot_table_row = at_rot_array_proxy;
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::prepare_count_pair_equivalence_class_masks
///
/// @brief
/// fills the input arrays with the details of count_pair()'s
/// behavior
///
/// @detailed
///
/// @param
/// bg_res - [in] - the residue number for the background rotamer
/// rot_trie_res - [in] - the residue number for the rotamer set that
///   contains many (more than 1) rotamers.
/// this_equivalence_class_bitmasks - [out] - the trie_node bitmasks
///   that decide which equivalence class atoms in the rotamer trie
///   belong to.  The meaning for equivalence class numbers are arbitrary
///   and merely answer the question of "which equivalence class is this atom"?
/// bg_equivalence_class_bitmasks - [out] - the trie_node bitmasks
///   that decide which equivalence class the background atom belongs to, once
///   the equivalence class of the rotamer-trie atom is known.
///   equivalence class "0" means "weight this atom pair energy by 0"
///   and the equivalence class "1" means "weight this atom pair energy by
///   *half*" (where *half* means 0.2.  I know, confusing, not my definition.)
///   If a background atom is niether in class 0 or class 1, then it is in class
///   two, which means "weight this atom pair energy at full strenght".
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::prepare_count_pair_equivalence_class_masks
(  int bg_res,
	int rot_trie_res,
	unsigned short this_equivalence_class_bitmasks[6],
	unsigned short bg_equivalence_class_bitmasks[6][2]
)
{
	int bbdist = bg_res - rot_trie_res;
	assert( (-2 <= bbdist) && (bbdist != 0) && (bbdist <= 2));
	switch (bbdist)
	{
		case -2: //bg.resid = i, this.resid = i + 2
			this_equivalence_class_bitmasks[0] = trie_node::NOT_BB_N;
			bg_equivalence_class_bitmasks[0][0] = 0;
			bg_equivalence_class_bitmasks[0][1] = 0;

			this_equivalence_class_bitmasks[1] = trie_node::BB_N;
			bg_equivalence_class_bitmasks[1][0] = 0;
			bg_equivalence_class_bitmasks[1][1] = trie_node::BB_C;

			this_equivalence_class_bitmasks[2] = 0;
			bg_equivalence_class_bitmasks[2][0] = 0;
			bg_equivalence_class_bitmasks[2][1] = 0;

			this_equivalence_class_bitmasks[3] = 0;
			bg_equivalence_class_bitmasks[3][0] = 0;
			bg_equivalence_class_bitmasks[3][1] = 0;

			this_equivalence_class_bitmasks[4] = 0;
			bg_equivalence_class_bitmasks[4][0] = 0;
			bg_equivalence_class_bitmasks[4][1] = 0;

			this_equivalence_class_bitmasks[5] = 0;
			bg_equivalence_class_bitmasks[5][0] = 0;
			bg_equivalence_class_bitmasks[5][1] = 0;

			break;
		case -1: //bg.resid = i, this.resid = i+1

			this_equivalence_class_bitmasks[0] = trie_node::BB_CA_PCD;
			bg_equivalence_class_bitmasks[0][0] = trie_node::BB_CA_C_O;
			bg_equivalence_class_bitmasks[0][1] = trie_node::BB_N_CB_H_HA;

			this_equivalence_class_bitmasks[1] = trie_node::BB_C_CB_HA_PHD;
			bg_equivalence_class_bitmasks[1][0] = trie_node::BB_C;
			bg_equivalence_class_bitmasks[1][1] = trie_node::BB_CA_O;

			this_equivalence_class_bitmasks[2] = trie_node::SC_OTHER;
			bg_equivalence_class_bitmasks[2][0] = 0;
			bg_equivalence_class_bitmasks[2][1] = 0;

			this_equivalence_class_bitmasks[3] = trie_node::BB_O;
			bg_equivalence_class_bitmasks[3][0] = 0;
			bg_equivalence_class_bitmasks[3][1] = trie_node::BB_CA_C_O;

			this_equivalence_class_bitmasks[4] = trie_node::BB_N;
			bg_equivalence_class_bitmasks[4][0] = trie_node::BB_N_CA_C_O_CB_HA;
			bg_equivalence_class_bitmasks[4][1] = trie_node::BB_H;

			this_equivalence_class_bitmasks[5] = trie_node::BB_H;
			bg_equivalence_class_bitmasks[5][0] = trie_node::BB_CA_C_O;
			bg_equivalence_class_bitmasks[5][1] = trie_node::BB_N_CB_HA;

			break;
		case 1: //bg.resid = i+1, this.resid = i
			this_equivalence_class_bitmasks[0] = trie_node::SC_OTHER;
			bg_equivalence_class_bitmasks[0][0] = 0;
			bg_equivalence_class_bitmasks[0][1] = 0;

			this_equivalence_class_bitmasks[1] = trie_node::BB_N_CB_HA;
			bg_equivalence_class_bitmasks[1][0] = trie_node::BB_N;
			bg_equivalence_class_bitmasks[1][1] = trie_node::BB_CA_H_PCD;

			this_equivalence_class_bitmasks[2] = trie_node::BB_CA_O;
			bg_equivalence_class_bitmasks[2][0] = trie_node::BB_N_CA_H_PCD;
			bg_equivalence_class_bitmasks[2][1] = trie_node::BB_C_O_CB_HA_PHD;

			this_equivalence_class_bitmasks[3] = trie_node::BB_C;
			bg_equivalence_class_bitmasks[3][0] =
				trie_node::BB_N_CA_C_CB_H_HA_PCD_PHD;
			bg_equivalence_class_bitmasks[3][1] = trie_node::BB_O;

			this_equivalence_class_bitmasks[4] = trie_node::BB_H;
			bg_equivalence_class_bitmasks[4][0] = 0;
			bg_equivalence_class_bitmasks[4][1] = trie_node::BB_N_CA_PCD;

			this_equivalence_class_bitmasks[5] = 0;
			bg_equivalence_class_bitmasks[5][0] = 0;
			bg_equivalence_class_bitmasks[5][1] = 0;

			break;
		case 2:
			this_equivalence_class_bitmasks[0] = trie_node::NOT_BB_C;
			bg_equivalence_class_bitmasks[0][0] = 0;
			bg_equivalence_class_bitmasks[0][1] = 0;

			this_equivalence_class_bitmasks[1] = trie_node::BB_C;
			bg_equivalence_class_bitmasks[1][0] = 0;
			bg_equivalence_class_bitmasks[1][1] = trie_node::BB_N;

			this_equivalence_class_bitmasks[2] = 0;
			bg_equivalence_class_bitmasks[2][0] = 0;
			bg_equivalence_class_bitmasks[2][1] = 0;

			this_equivalence_class_bitmasks[3] = 0;
			bg_equivalence_class_bitmasks[3][0] = 0;
			bg_equivalence_class_bitmasks[3][1] = 0;

			this_equivalence_class_bitmasks[4] = 0;
			bg_equivalence_class_bitmasks[4][0] = 0;
			bg_equivalence_class_bitmasks[4][1] = 0;

			this_equivalence_class_bitmasks[5] = 0;
			bg_equivalence_class_bitmasks[5][0] = 0;
			bg_equivalence_class_bitmasks[5][1] = 0;

			break;
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_vs_background_i_ip1or2
///
/// @brief
/// calculates component energy using special-case trie-vs-trie iterative
/// algorithm, where one of the rotamer tries is only a single rotamer
///
/// @detailed
/// count pair has special weighting rules for backbone/backbone interactions
/// when the two residues in question are fewer than 3 residues appart along
/// the backbone.  This subroutine handles the cases when the backbone
/// distance of the two tries is one or two.
///
/// @param
/// bg - [in] - the background rotamer trie, it consists of a single rotamer
/// trie_trials_componentE - [out] - the array to write component
///    rotamer pair energies to
/// trie_trials_sumE - [out] - the array to write the the sum of the four
///    components of the rotamer pair energies.
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::trie_vs_background_i_ip1or2
(	rotamer_trie & bg,
	FArray2DB_float & trie_trials_componentE,
	FArray1DB_float & trie_trials_sumE
)
{
	using namespace rotamer_trial_energies;
	using namespace pdbstatistics_pack;
	using namespace param_pack;
	//set count pair equivalence classes based on this.res_id_ - bg.res_id_
	unsigned short this_equivalence_class_bitmasks[6];
	//column 0: 0 equiv class, column 1: half equiv class.
	unsigned short bg_equivalence_class_bitmasks[6][2];

	prepare_count_pair_equivalence_class_masks( bg.res_id_, res_id_,
		this_equivalence_class_bitmasks, bg_equivalence_class_bitmasks);

	int bg_num_atoms = bg.get_num_atoms_for_curr_bb();
	int bg_num_heavyatoms = bg.get_num_heavy_atoms_for_curr_bb();

	int this_num_atoms = get_num_atoms_for_curr_bb();

	trie_node* this_trie = get_trie_for_curr_bb();
	trie_node* bg_trie = bg.get_trie_for_curr_bb();

	FArray2D_float component_energy_stack( 4, max_stack_height_, 0.0);
	vector< float > energy_sum(rotamer_trie::MAX_ATS_PER_RES + 1 + 1);
	energy_sum[1] = 0;
	energy_sum[0] = 0;
	FArray2D_bool parent_heavy_wi_hydrogen_cutoff
		( bg_num_heavyatoms, max_heavyatom_depth_, true);
	FArray2D_bool  r_heavy_skip_s_subtree
		(bg_num_heavyatoms, max_heavyatom_depth_, false);
	std::vector< int >   r_heavy_depth_stack( max_stack_height_ + 1   );
	//std::vector< int >   r_tree_depth_stack ( max_stack_height_ + 1   );
	r_heavy_depth_stack[1] = 0;
	//r_tree_depth_stack[1] = 0;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};
	int r_num_rotamers_seen = 0;
	int r_curr_stack_top = 2;
	int s_heavy_depth;
	float atrE, repE, solvE, hbE, sumE;
	float const count_pair_half_weight = { 0.2 };

	for (int ii = 1; ii <= this_num_atoms; ++ii )
	{
		trie_node & r = this_trie[ii];
		unsigned short * r_class_masks = NULL;
		for (int eq_class = 0; eq_class < 6; ++eq_class)
		{	if (this_equivalence_class_bitmasks[eq_class] & r.flags2_)
			{  r_class_masks = bg_equivalence_class_bitmasks[eq_class];
				break;
			}
		}
		assert( r_class_masks );


		if (r.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{  --r_curr_stack_top;
		}
		if (r.has_sibling() )
		{	//push - copy stack downwards

			++r_curr_stack_top;

			FArray1Da_float component_energy_proxy_top
				(component_energy_stack(1, r_curr_stack_top));
			FArray1Da_float component_energy_proxy_topminus1
				(component_energy_stack(1, r_curr_stack_top-1));
			component_energy_proxy_top.dimension( 4 );
			component_energy_proxy_topminus1.dimension( 4 );
			component_energy_proxy_top = component_energy_proxy_topminus1;
			energy_sum[ r_curr_stack_top ] = energy_sum[ r_curr_stack_top - 1];

			r_heavy_depth_stack[ r_curr_stack_top ] =
				r_heavy_depth_stack[ r_curr_stack_top-1 ];
			//r_tree_depth_stack[ r_curr_stack_top ] =
			//	r_tree_depth_stack[ r_curr_stack_top-1 ];
		}

		FArray1Da_float component_energy_proxy
			(component_energy_stack(1, r_curr_stack_top) );
		component_energy_proxy.dimension(4);

		//++r_tree_depth_stack[ r_curr_stack_top ];
		if (! r.is_hydrogen() ) ++r_heavy_depth_stack[ r_curr_stack_top ];

		FArray1Da_bool parent_wi_h_dist
			( parent_heavy_wi_hydrogen_cutoff( 1,
			r_heavy_depth_stack[ r_curr_stack_top ] ) );
		parent_wi_h_dist.dimension(bg_num_heavyatoms);

		FArray1Da_bool r_heavy_skip_s_subtree_proxy
			( r_heavy_skip_s_subtree(1, r_heavy_depth_stack[ r_curr_stack_top ] ));
		r_heavy_skip_s_subtree_proxy.dimension(bg_num_heavyatoms);

		s_heavy_depth = 0;

		if ( r.is_hydrogen() )
		{
			for (int jj = 1; jj <= bg_num_atoms; ++jj )
			{
				trie_node & s = bg_trie[jj];
				if ( s.is_hydrogen() )
				{
					if (parent_wi_h_dist( s_heavy_depth ) &&
						(! (s.flags2_ & r_class_masks[0])) )
					{	atom_atom_energy(r, s,
							res_id_, bg.res_id_, num_neighbors_, bg.num_neighbors_,
							Whbond, repE, hbE, sumE);
						if (s.flags2_ & r_class_masks[1])
						{
							repE *= count_pair_half_weight;
						}
						component_energy_proxy( componentE_ljrep_row ) += repE;
						energy_sum[ r_curr_stack_top ] += repE;
					}
				}
				else
				{
					++s_heavy_depth;

					if (r_heavy_skip_s_subtree_proxy(s_heavy_depth)) break;

					if ( parent_wi_h_dist(s_heavy_depth) &&
						(! (s.flags2_ & r_class_masks[0])) )
					{	atom_atom_energy(r, s,
							res_id_, bg.res_id_, num_neighbors_, bg.num_neighbors_,
							Whbond, repE, hbE, sumE);
						if (s.flags2_ & r_class_masks[1])
						{
							repE *= count_pair_half_weight;
							hbE  *= count_pair_half_weight;
						}
						component_energy_proxy( componentE_ljrep_row ) += repE;
						component_energy_proxy( componentE_hbond_row ) += hbE;
						energy_sum[ r_curr_stack_top ] += repE + hbE;
					}

				}
			}
		}
		else
		{	//else r is a heavy atom
			r_heavy_skip_s_subtree_proxy = false;
			for (int jj = 1; jj <= bg_num_atoms; ++jj )
			{
				trie_node & s = bg_trie[jj];
				if ( s.is_hydrogen())
				{
					if (parent_wi_h_dist( s_heavy_depth) &&
						(! (s.flags2_ & r_class_masks[0])) )
					{	atom_atom_energy(r, s,
							res_id_, bg.res_id_, num_neighbors_, bg.num_neighbors_,
							Whbond, repE, hbE, sumE);
						if (s.flags2_ & r_class_masks[1])
						{
							repE *= count_pair_half_weight;
							hbE  *= count_pair_half_weight;
						}
						component_energy_proxy( componentE_ljrep_row ) += repE;
						component_energy_proxy( componentE_hbond_row ) += hbE;
						energy_sum[ r_curr_stack_top ] += repE + hbE;
					}
				}
				else
				{	++s_heavy_depth;
					float d2;
					heavyatom_heavyatom_energy(r, s, d2, atrE, repE, solvE, sumE);
					if (s.flags2_ & r_class_masks[0])
					{
						atrE = repE = solvE = 0;
					}
					else if (s.flags2_ & r_class_masks[1])
					{
						atrE *= count_pair_half_weight;
						repE *= count_pair_half_weight;
						solvE*= count_pair_half_weight;
					}
					component_energy_proxy( componentE_ljatr_row ) += atrE;
					component_energy_proxy( componentE_ljrep_row ) += repE;
					component_energy_proxy( componentE_solvE_row ) += solvE;
					energy_sum[ r_curr_stack_top ] += atrE + repE + solvE;

					parent_wi_h_dist(s_heavy_depth) =
						(d2 < hydrogen_interaction_cutoff);
					//const float subtree_heavy_square_sum =
					//	(( (fa_max_dis/2) + s.radius_containing_subtree_) *
					//	((fa_max_dis/2) + s.radius_containing_subtree_));
					if (d2 > s.radius_containing_subtree_)
					{	r_heavy_skip_s_subtree_proxy(s_heavy_depth) = true;
						break;
					}
				}
			}
		}

		if ( r.is_rotamer_terminal() )
		{	//write component energies
			++r_num_rotamers_seen;
			FArray1Da_float trie_trials_componentE_slice
				(trie_trials_componentE(1, r_num_rotamers_seen));
			trie_trials_componentE_slice.dimension(4);
			trie_trials_componentE_slice = component_energy_proxy;
			trie_trials_sumE(r_num_rotamers_seen) = energy_sum[ r_curr_stack_top ];
		}
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::trie_vs_background_i_ip3orMore
///
/// @brief
/// computes component energy using a special case version of the iterative
/// trie-vs-trie algorithm where one trie is a single rotamer.
///
/// @detailed
/// handles the cases when the backbone distance between the two tries
/// is 3 or more.
///
/// @param
/// bg - [in] - the background rotamer trie, it consists of a single rotamer
/// trie_trials_componentE - [out] - the array to write component
///    rotamer pair energies to
/// trie_trials_sumE - [out] - the array to write the the sum of the four
///    components of the rotamer pair energies.
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::trie_vs_background_i_ip3orMore
(	rotamer_trie & bg,
	FArray2DB_float & trie_trials_componentE,
	FArray1DB_float & trie_trials_sumE
)
{
	using namespace rotamer_trial_energies;
	using namespace pdbstatistics_pack;
	using namespace param_pack;

	int bg_num_atoms = bg.get_num_atoms_for_curr_bb();
	int bg_num_heavyatoms = bg.get_num_heavy_atoms_for_curr_bb();

	int this_num_atoms = get_num_atoms_for_curr_bb();

	trie_node* this_trie = get_trie_for_curr_bb();
	trie_node* bg_trie = bg.get_trie_for_curr_bb();


	FArray2D_float component_energy_stack( 4, max_stack_height_, 0.0);
	std::vector< float > energy_sum( rotamer_trie::MAX_ATS_PER_RES + 1 + 1);
	energy_sum[1] = 0;
	energy_sum[0] = 0;
	FArray2D_bool parent_heavy_wi_hydrogen_cutoff
		( bg_num_heavyatoms, max_heavyatom_depth_, true);
	FArray2D_bool r_heavy_skip_s_subtree
		(bg_num_heavyatoms, max_heavyatom_depth_, false);
	std::vector< int >   r_heavy_depth_stack( max_stack_height_ + 1   );
	//std::vector< int >   r_tree_depth_stack ( max_stack_height_ + 1   );
	r_heavy_depth_stack[1] = 0;
	//r_tree_depth_stack[1] = 0;

	float Whbond[3] = {pack_wts.Whbond_bb(), pack_wts.Whbond_bb_sc(), pack_wts.Whbond_sc()};
	int r_num_rotamers_seen = 0;
	int r_curr_stack_top = 2;
	int s_heavy_depth;
	float atrE, repE, solvE, hbE, sumE;


	for (int ii = 1; ii <= this_num_atoms; ++ii )
	{
		trie_node & r = this_trie[ii];

		if (r.flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{  --r_curr_stack_top;
		}
		if (r.has_sibling() )
		{	//push - copy stack downwards

			++r_curr_stack_top;

			FArray1Da_float component_energy_proxy_top
				(component_energy_stack(1, r_curr_stack_top));
			FArray1Da_float component_energy_proxy_topminus1
				(component_energy_stack(1, r_curr_stack_top-1));
			component_energy_proxy_top.dimension( 4 );
			component_energy_proxy_topminus1.dimension( 4 );
			component_energy_proxy_top = component_energy_proxy_topminus1;
			energy_sum[ r_curr_stack_top ] = energy_sum[ r_curr_stack_top - 1];

			r_heavy_depth_stack[ r_curr_stack_top ] =
				r_heavy_depth_stack[ r_curr_stack_top-1 ];
			//r_tree_depth_stack[ r_curr_stack_top ] =
			//	r_tree_depth_stack[ r_curr_stack_top-1 ];
		}

		FArray1Da_float component_energy_proxy
			(component_energy_stack(1, r_curr_stack_top) );
		component_energy_proxy.dimension(4);

		//++r_tree_depth_stack[ r_curr_stack_top ];
		if (! r.is_hydrogen() ) ++r_heavy_depth_stack[ r_curr_stack_top ];

		FArray1Da_bool parent_wi_h_dist
			( parent_heavy_wi_hydrogen_cutoff( 1,
			r_heavy_depth_stack[ r_curr_stack_top ] ) );
		parent_wi_h_dist.dimension(bg_num_heavyatoms);

		FArray1Da_bool r_heavy_skip_s_subtree_proxy
			( r_heavy_skip_s_subtree(1, r_heavy_depth_stack[ r_curr_stack_top ] ));
		r_heavy_skip_s_subtree_proxy.dimension(bg_num_heavyatoms);

		s_heavy_depth = 0;

		if ( r.is_hydrogen() )
		{
			for (int jj = 1; jj <= bg_num_atoms; ++jj )
			{
				trie_node & s = bg_trie[jj];
				if ( s.is_hydrogen() )
				{
					if (parent_wi_h_dist( s_heavy_depth ) )
					{	atom_atom_energy(r, s,
							res_id_, bg.res_id_, num_neighbors_, bg.num_neighbors_,
							Whbond, repE, hbE, sumE);
						component_energy_proxy( componentE_ljrep_row ) += repE;
						energy_sum[ r_curr_stack_top ] += repE;
					}
				}
				else
				{
					++s_heavy_depth;

					if (r_heavy_skip_s_subtree_proxy(s_heavy_depth)) break;

					if ( parent_wi_h_dist(s_heavy_depth) )
					{	atom_atom_energy(r, s,
						 	res_id_, bg.res_id_, num_neighbors_, bg.num_neighbors_,
						 	Whbond, repE, hbE, sumE);
						component_energy_proxy( componentE_ljrep_row ) += repE;
						component_energy_proxy( componentE_hbond_row ) += hbE;
						energy_sum[ r_curr_stack_top ] += repE + hbE;
					}

				}
			}
		}
		else //else r is a heavy atom
		{
			r_heavy_skip_s_subtree_proxy = false;
			for (int jj = 1; jj <= bg_num_atoms; ++jj )
			{
				trie_node & s = bg_trie[jj];
				if ( s.is_hydrogen())
				{
					if (parent_wi_h_dist( s_heavy_depth) )
					{	atom_atom_energy(r, s,
							res_id_, bg.res_id_, num_neighbors_,
							bg.num_neighbors_, Whbond,
							repE, hbE, sumE);
						component_energy_proxy( componentE_ljrep_row ) += repE;
						component_energy_proxy( componentE_hbond_row ) += hbE;
						energy_sum[ r_curr_stack_top ] += repE + hbE;
					}
				}
				else
				{	++s_heavy_depth;
					float d2;
					heavyatom_heavyatom_energy(r, s, d2, atrE, repE, solvE, sumE);
					component_energy_proxy( componentE_ljatr_row ) += atrE;
					component_energy_proxy( componentE_ljrep_row ) += repE;
					component_energy_proxy( componentE_solvE_row ) += solvE;
					energy_sum[ r_curr_stack_top ] += atrE + repE + solvE;

					parent_wi_h_dist(s_heavy_depth) =
						(d2 < hydrogen_interaction_cutoff);
					//const float subtree_heavy_square_sum =
					//	(( (fa_max_dis/2) + s.radius_containing_subtree_) *
					//	((fa_max_dis/2) + s.radius_containing_subtree_));
					if (d2 > s.radius_containing_subtree_)
					{	r_heavy_skip_s_subtree_proxy(s_heavy_depth) = true;
						break;
					}
				}

			}
		}

		if ( r.is_rotamer_terminal() )
		{	//write component energies
			++r_num_rotamers_seen;
			FArray1Da_float trie_trials_componentE_slice
				(trie_trials_componentE(1, r_num_rotamers_seen));
			trie_trials_componentE_slice.dimension(4);
			trie_trials_componentE_slice = component_energy_proxy;
			trie_trials_sumE(r_num_rotamers_seen) = energy_sum[ r_curr_stack_top ];
		}
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::convert_preorder_rot_rot_table
///
/// @brief
/// converts rotamer_rotamer_energy table from the preorder traversal order
/// to the original rotamer order.  Rotamers are ordered within the
/// trie_vs_trie algorithm by their position in the preorder traversal of
/// their containing rotamer trie.  After trie_vs_trie completes, this
/// subroutine restores the rot_rot_E table to the original order.
///
/// @detailed
///
/// @param
/// rt - [in] - the S rotamer trie
/// rot_rot_preorder - [in] - the table holding rotamer pair energies
///   as calculated in the preorder traversal order.  This table should
///   have dimensions (rt.num_unique_rotamers x this.num_unique_rotamers)
/// rot_rot_original_order - [out] - the table into which the rotamer
///   pair energies are to be output; the ordering of rotamers in this
///   table is the original (arbitrary) ordering of rotamers. This table
////  should have dimensions (rt.num_total_rots x this.num_total_rots)
///
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::convert_preorder_rot_rot_table
(  rotamer_trie & rt,
	FArray2DB_float & rot_rot_preorder,
	FArray2DB_float & rot_rot_original_order
)
{	//cerr << "convert_preorder_rot_rot_table" << endl;
	std::vector< int > const & this_total_2_unique = get_total_2_unique_for_curr_bb();
	std::vector< int > const & rt_total_2_unique = rt.get_total_2_unique_for_curr_bb();

	int this_num_total_rotamers = get_num_total_rotamers_for_curr_bb();
	int rt_num_total_rotamers = rt.get_num_total_rotamers_for_curr_bb();

	for ( int ii = 1; ii <= this_num_total_rotamers; ++ii )
	{	//cerr << "(ii:" << ii << ",";
		int ii_preorder = this_total_2_unique[ii];
		//cerr << ii_original << ") ";
		for ( int jj = 1; jj <= rt_num_total_rotamers; ++jj )
		{	//cerr << "(jj:" << jj << ",";
			int jj_preorder = rt_total_2_unique[jj];
			//cerr << rot_rot_preorder(jj, ii) << " ";
			rot_rot_original_order(jj, ii) =
				rot_rot_preorder(jj_preorder, ii_preorder);
		}
		//cerr << endl;
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::convert_preorder_trials_data
///
/// @brief
/// converts the component energy and the component sum arrays from the
/// preorder traversal order that the trie-vs-trie algorithm produces into
/// the rotamer ordering originally given to the rotamers
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////

void
rotamer_trie::convert_preorder_trials_data
(	FArray2DB_float & trie_trials_componentE_preorder,
	FArray2DB_float & trie_trials_componentE,
	FArray1DB_float & trie_trials_sumE_preorder,
	FArray1DB_float & trie_trials_sumE
)
{
	std::vector< int > const & this_total_2_unique = get_total_2_unique_for_curr_bb();

	int this_num_total_rotamers = get_num_total_rotamers_for_curr_bb();

	for (int ii = 1; ii <= this_num_total_rotamers; ++ii )
	{
		int ii_preorder = this_total_2_unique[ii];
		FArray1Da_float trie_trials_componentE_preorder_proxy
			(trie_trials_componentE_preorder(1, ii_preorder));
		FArray1Da_float trie_trials_componentE_proxy
			(trie_trials_componentE(1, ii));
		trie_trials_componentE_preorder_proxy.dimension(4);
		trie_trials_componentE_proxy.dimension(4);
		trie_trials_componentE_proxy = trie_trials_componentE_preorder_proxy;

		trie_trials_sumE(ii) += trie_trials_sumE_preorder(ii_preorder);
	}
	return;
}

//apl I'm leaving in this commented-out code: it is the recursive (as opposed
//apl to the iterative) version of the trie_vs_trie algorithm.   It may prove
//apl useful for people wanting to understand the algorithm
// void  rotamer_trie::trie_v_trie(
// 	rotamer_trie & rt,
// 	FArray2D_float & rot_rot_table)
// {	using namespace param;
// 	const int FIRST_ATOM_DEPTH = 2;
// 	//cerr << "rt.num_unique_rotamers_ = " << rt.num_unique_rotamers_ <<
//		//	" rotamer_trie::MAX_ATS_PER_RES = " << rotamer_trie::MAX_ATS_PER_RES
//		//	<< endl;
// 	FArray2D_float at_v_rot_stack( rt.num_unique_rotamers_,
//			rotamer_trie::MAX_ATS_PER_RES + 1, 0.0);
// 	FArray2D_bool  parent_heavy_wi_h_cutoff(
//			rt.get_num_atoms, rotamer_trie::MAX_ATS_PER_RES+1, true);
// 	//cerr << "atom_v_trie(1, depth 1);" << endl;
// 	atom_v_trie( trie_[1], rt, FIRST_ATOM_DEPTH, at_v_rot_stack,
//			parent_heavy_wi_h_cutoff, rot_rot_table);
// 	return;
// }
//
// void  rotamer_trie::atom_v_trie
// (  trie_node & r,
// 	rotamer_trie & rt,
// 	int depth,
// 	FArray2D_float & at_v_rot_stack,
// 	FArray2D_bool  & parent_heavy_wi_h_cutoff,
// 	FArray2D_float & rot_rot_table
// )
// {
// 	//copy upper stack layer down
// 	for ( int ii = 1, ld = at_v_rot_stack.index(1, depth),
//			ldminus1 = at_v_rot_stack.index(1, depth-1);
// 		ii <= rt.num_unique_rotamers_; ++ii, ++ld, ++ldminus1 )
// 	{	at_v_rot_stack[ ld ] = at_v_rot_stack[ ldminus1 ];}
//
// 	FArray1Da_float at_v_rot(at_v_rot_stack(1, depth+1));
// 	at_v_rot.dimension( rt.num_unique_rotamers_);
// 	//cerr << "atom_v_atom(1);" << endl;
// 	atom_v_atom(r, rt, rt.trie_[1], at_v_rot, 0);
//
// 	if (r.is_rotamer_terminal() != 0)
// 		for ( int ii = 1, lat_v_rot = at_v_rot_stack.index(1, depth+1),
// 		lrot_rot_table = rot_rot_table.index(1, r.is_rotamer_terminal());
// 		ii <= rt.num_unique_rotamers_; ++ii, ++lat_v_rot, ++lrot_rot_table )
// 		{
// 			rot_rot_table[ lrot_rot_table ] = at_v_rot_stack[ lat_v_rot ];
// 		}
//
// 	if (r._child_pointer != 0)
// 	{	//cerr << "atom_v_trie(" << r._child_pointer << ", depth "
//			//<< depth + 1 << ");" << endl;
// 		atom_v_trie( trie_[r._child_pointer], rt, depth + 1,
//				at_v_rot_stack, rot_rot_table);
// 	}
// 	if (r._sibling_pointer != 0)
// 	{	//cerr << "atom_v_trie(" << r._sibling_pointer
//			//<< ", depth " << depth << ");" << endl;
// 		atom_v_trie( trie_[r._sibling_pointer], rt,
//				depth, at_v_rot_stack, rot_rot_table);
// 	}
//
// 	//cerr << "return atom_v_trie" << endl;
// 	return;
// }
//
// void  rotamer_trie::atom_v_atom(
// 	trie_node & r,
// 	rotamer_trie & rt,
// 	trie_node & s,
// 	FArray1Da_float & at_v_rot,
// 	float ancestral_e
// )
// {
//
// 	float e = atom_atom_energy( r, s, res_id_, rt.res_id_);
//
// 	if (s.is_rotamer_terminal() != 0)
// 		at_v_rot(s.is_rotamer_terminal()) += ancestral_e + e;
//
// 	if (s._child_pointer != 0)
// 	{	//cerr << "atom_v_atom(" << s._child_pointer << ");" << endl;
// 		atom_v_atom(r, rt, rt.trie_[s._child_pointer],
//				at_v_rot, ancestral_e + e);
// 	}
// 	if (s._sibling_pointer != 0)
// 	{	//cerr << "atom_v_atom(" << s._sibling_pointer << ");" << endl;
// 		atom_v_atom(r, rt, rt.trie_[s._sibling_pointer],
//				at_v_rot, ancestral_e);
// 	}
//
// 	return;
// }


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::atom_atom_energy
///
/// @brief
/// calcs energy between atoms r and s.  Either r, s, or r and s are
///  hydrogen atoms.
///
/// @detailed
/// apl  If the energy function changes, if additional weights are added, this
/// apl  code must be modified, too.  For now, hydrogen/other atom energies
/// apl  come from VDW repulsion and hydrogen bonds only.  This subroutine
/// apl  relies on Jack Snoeyink's version of hydrogen bond energy calculation:
/// apl  the orientation vector of donor and acceptor atoms have been precomputed
///
///
/// @param r - [in] - an atom belonging to trie R
/// @param s - [in] - an atom belonging to trie S
/// @param r_res - [in] - the residue index for trie R
/// @param s_res - [in] - the residue index for trie S
/// @param R_num_neighbors - [in] - the number of neighbors R has, used for hbond
/// @param S_num_neighbors - [in] - the number of neighbors S has, used for hbond
/// @param Whbond - [in] - vector of hydrogen bonding weights, the order is
///   Whbond[0] = Whbond_bb_bb
///   Whbond[1] = pack_wts.Whbond_bb_sc()
///   Whbond[2] = Whbond_sc_sc
///
////////////////////////////////////////////////////////////////////////////////

inline
float rotamer_trie::atom_atom_energy(
	trie_node & r,
	trie_node & s,
	int r_res,
	int s_res,
	int R_num_neighbors,
	int S_num_neighbors,
	float * Whbond
) const
{
	// assert((r.ad_.get_atom_type() >= 22) || (s.ad_.get_atom_type() >= 22));

	using namespace param_pack;
	float e;
	float hbE = 0;
	float repE = 0;
  float elecE = 0.0; //SJF electrostatic interactions (not taken into account). To take into account, change the dummy charges below to real parameters.

  float const dummy_chrg1 = 0.0, dummy_chrg2 = 0.0;
fast_pairenergy_hydrogens(r.ad_.get_xyz(), s.ad_.get_xyz(),
    r.ad_.get_atom_type(), s.ad_.get_atom_type(), dummy_chrg1, dummy_chrg2, repE, elecE, 1);
	e = repE * pack_wts.Wrep();

	if (r.is_hydrogen() &&
		decide_calc_hbe_for_pair(r.is_donor_h(), s.is_acceptor(),
		r.is_backbone(), s.is_backbone(),
		r.is_bbhbg_sat(), s.is_bbhbg_sat(),
		r_res, s_res ))
	{
		int hb_bb_sc_type = 0;
		if (!r.is_backbone() ) ++hb_bb_sc_type;
		if (!s.is_backbone() ) ++hb_bb_sc_type;

		get_atomdon_atomact_hbE(r.ad_, s.ad_, r_res, s_res,
			!r.is_backbone(), !s.is_backbone(), R_num_neighbors, S_num_neighbors,
			hbE);

		//if (hbE != 0.0f) cerr << "hbE: " << hbE << " hb_bb_sc_type: " <<
		//hb_bb_sc_type << " e: " << hbE * Whbond[hb_bb_sc_type] << endl;
		e += hbE * Whbond[hb_bb_sc_type];
	}
	else if ( decide_calc_hbe_for_pair(s.is_donor_h(), r.is_acceptor(),
		s.is_backbone(), r.is_backbone(),
		s.is_bbhbg_sat(), r.is_bbhbg_sat(),
		s_res, r_res) )
	{
		int hb_bb_sc_type = 0;
		if (!r.is_backbone() ) ++hb_bb_sc_type;
		if (!s.is_backbone() ) ++hb_bb_sc_type;
		get_atomdon_atomact_hbE(s.ad_, r.ad_, s_res, r_res,
                            !s.is_backbone(), !r.is_backbone(), S_num_neighbors, R_num_neighbors,
                            hbE);
		//if (hbE != 0.0f) cerr << "hbE: " << hbE << " hb_bb_sc_type: "
		//<< hb_bb_sc_type << " e: " << hbE * Whbond[hb_bb_sc_type] << endl;
		e += hbE * Whbond[hb_bb_sc_type];
	}
	//std::cerr << "aaE: " << r.ad_ << " " << s.ad_ << e << std::endl; //TEMP DEBUG

	if ( trie_output_all_energy_calcs ) std::cerr << r.get_atom() << " " << s.get_atom() << " atom/atom e: " << e << std::endl;

	return e;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::atom_atom_energy
///
/// @brief
/// calculates atom pair energy for a pair where at least one of the two atoms
/// is a hydrogen.  Outputs component energies
///
/// @detailed
///
///
/// @param r - [in] - an atom belonging to trie R
/// @param s - [in] - an atom belonging to trie S
/// @param r_res - [in] - the residue index for trie R
/// @param s_res - [in] - the residue index for trie S
/// @param R_num_neighbors - [in] - the number of neighbors R has, used for hbond
/// @param S_num_neighbors - [in] - the number of neighbors S has, used for hbond
/// @param Whbond - [in] - vector of hydrogen bonding weights, the order is
///   Whbond[0] = Whbond_bb_bb
///   Whbond[1] = pack_wts.Whbond_bb_sc()
///   Whbond[2] = Whbond_sc_sc
/// @param repE - [out] - the repulsive Lennard Jones term for the atom pair
/// @param hbE - [out] - the hydrogen bond energy for the atom pair
/// @param sumE - [out] - the sum of the component energies (in this case, repE + hbE)
////////////////////////////////////////////////////////////////////////////////

inline
void
rotamer_trie::atom_atom_energy
(	trie_node & r,
	trie_node & s,
	int r_res,
	int s_res,
	int R_num_neighbors,
	int S_num_neighbors,
	float * Whbond,
	float & repE,
	float & hbE,
	float & sumE
) const
{
	// assert((r.ad_.get_atom_type() >= 22) || (s.ad_.get_atom_type() >= 22));

float elecE=0.0; // SJF the electrostatic interaction. This term is not summed into sumE, and is not // returned by this function. If you want to incorporate it, add it as a parameter to this function. Also, change the dummy charges below to real parameters

	using namespace param_pack;
	hbE = 0;
	repE = 0;
  float const dummy_chrg1 = 0.0, dummy_chrg2 = 0.0;
  fast_pairenergy_hydrogens(r.ad_.get_xyz(), s.ad_.get_xyz(),
    r.ad_.get_atom_type(), s.ad_.get_atom_type(), dummy_chrg1, dummy_chrg2, repE, elecE, 1);

	repE *= pack_wts.Wrep();
	sumE = repE;

	if (r.is_hydrogen() &&
		decide_calc_hbe_for_pair(r.is_donor_h(), s.is_acceptor(),
		r.is_backbone(), s.is_backbone(),
		r.is_bbhbg_sat(), s.is_bbhbg_sat(),
		r_res, s_res ))
	{
		int hb_bb_sc_type = 0;
		if (!r.is_backbone() ) ++hb_bb_sc_type;
		if (!s.is_backbone() ) ++hb_bb_sc_type;

		get_atomdon_atomact_hbE(r.ad_, s.ad_, r_res, s_res,
                            !r.is_backbone(), !s.is_backbone(), R_num_neighbors, S_num_neighbors,
                            hbE);

		//if (hbE != 0.0f) cerr << "hbE: " << hbE << " hb_bb_sc_type: "
		//<< hb_bb_sc_type << " e: " << hbE * Whbond[hb_bb_sc_type] << endl;
		hbE *= Whbond[hb_bb_sc_type];
		sumE += hbE;
	}
	else if ( decide_calc_hbe_for_pair(s.is_donor_h(), r.is_acceptor(),
		s.is_backbone(), r.is_backbone(),
		s.is_bbhbg_sat(), r.is_bbhbg_sat(),
		s_res, r_res) )
	{
		int hb_bb_sc_type = 0;
		if (!r.is_backbone() ) ++hb_bb_sc_type;
		if (!s.is_backbone() ) ++hb_bb_sc_type;
		get_atomdon_atomact_hbE(s.ad_, r.ad_, s_res, r_res,
                            !s.is_backbone(), !r.is_backbone(), S_num_neighbors, R_num_neighbors,
                            hbE);
		//if (hbE != 0.0f) cerr << "hbE: " << hbE << " hb_bb_sc_type: "
		//<< hb_bb_sc_type << " e: " << hbE * Whbond[hb_bb_sc_type] << endl;
		hbE *= Whbond[hb_bb_sc_type];
		sumE += hbE;
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::heavyatom_heavyatom_energy
///
/// @brief
/// apl  calcs energy between atoms r and s.  Both r and s are heavy atoms.
///
/// @detailed
/// If the energy function changes, if additional weights are added, this
/// code must be modified, too.
///
/// @param
/// r - [in] - heavy atom from trie R
/// s - [in] - heavy atom from trie S
/// d2 - [out] - the square distance between r and s.
////////////////////////////////////////////////////////////////////////////////

inline
float rotamer_trie::heavyatom_heavyatom_energy(
	trie_node & r,
	trie_node & s,
	float & d2
) const
{
	using namespace param_pack;
	float e;
	//++_heavy_heavy_evals_count;
	//assert( r.ad_.get_atom_type() < 22 && s.ad_.get_atom_type() < 22 );

	float solE = 0;
	float atrE = 0;
	float repE = 0;

	fast_pairenergy(r.ad_.get_xyz(), s.ad_.get_xyz(),
		r.ad_.get_atom_type(), s.ad_.get_atom_type(),
		solE, atrE, repE, d2, 1.0);
	e = pack_wts.Wsol() * solE + pack_wts.Watr() * atrE + pack_wts.Wrep() * repE;

	//std::cerr << "aaE: " << r.ad_ << " " << s.ad_ << e << std::endl; //TEMP DEBUG

	if ( trie_output_all_energy_calcs ) std::cerr << r.get_atom() << " " << s.get_atom() << " heavy/heavy e: " << e << std::endl;
	return e;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::heavyatom_heavyatom_energy
///
/// @brief
/// outputs the component energy for a pair of heavy atoms
///
/// @detailed
///
/// @param
/// r - [in] - heavy atom from trie R
/// s - [in] - heavy atom from trie S
/// d2 - [out] - the square distance between r and s.
/// atrE - [out] - the Lennard Jones attractive energy for r and s
/// repE - [out] - the Lennard Jones repulsive energy
/// solE - [out] - the Lazaridis/Karplus solvation energy
/// sumE - [out] - the sum atrE + repE + solE
///
////////////////////////////////////////////////////////////////////////////////
inline
void rotamer_trie::heavyatom_heavyatom_energy
(  trie_node & r,
   trie_node & s,
	float & d2,
	float & atrE,
	float & repE,
	float & solE,
	float & sumE
) const
{
	using namespace param_pack;

	solE = 0;
	atrE = 0;
	repE = 0;

	fast_pairenergy(r.ad_.get_xyz(), s.ad_.get_xyz(),
		r.ad_.get_atom_type(), s.ad_.get_atom_type(),
		solE, atrE, repE, d2, 1.0);
	solE *= pack_wts.Wsol();
	atrE *= pack_wts.Watr();
	repE *= pack_wts.Wrep();
	sumE = solE + atrE + repE;

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::compute_max_stack_height()
///
/// @brief
/// With a preorder traversal of the trie, this function computes the maximum
/// stack depth potentially needed by trie_vs_trie algorithm.  This value
/// is at most 6.
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
void rotamer_trie::compute_max_stack_height()
{
	//cerr << "begin rotamer_trie::compute_max_stack_height()" << endl;
	max_heavyatom_depth_ = 0;
	max_stack_height_ = 2;
	for (int ii = 1; ii <= num_distinct_backbones_; ++ii )
	{
		int heavy_depth_stack[rotamer_trie::MAX_ATS_PER_RES + 1]; //safe upper bound
		heavy_depth_stack[1] = 0;

		int stack_top = 2;
		for ( int jj = 1; jj <= num_atoms_for_bb_(ii); ++jj )
		{	if ( tries_(ii)[jj].flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
				--stack_top;

			if ( tries_(ii)[jj].has_sibling() )
			{	++stack_top;
				heavy_depth_stack[stack_top] = heavy_depth_stack[stack_top-1];
			}

			if ( ! tries_(ii)[jj].is_hydrogen() )
				++heavy_depth_stack[stack_top];

			if (max_stack_height_ < stack_top )
			{
				max_stack_height_ = stack_top;
			}
			if ( max_heavyatom_depth_ < heavy_depth_stack[stack_top] )
			{
				max_heavyatom_depth_ = heavy_depth_stack[stack_top];
			}
		}
	}

	//cerr << "max_stack_height_: " << max_stack_height_ << endl;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::calculate_num_rotamers_in_subtree()
///
/// @brief
/// Subtree pruning mechanisms require each heavy atom know how many
/// rotamers terminate in their subtrees.  These quantities can be computed
/// in linear time.
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
void rotamer_trie::calculate_num_rotamers_in_subtree()
{
	trie_node* curr_trie = get_trie_for_curr_bb();
	int num_atoms_in_trie = get_num_atoms_for_curr_bb();

	//cerr << "calculate_subtree_heavyatoms_and_rotamers()" << endl;
	int heavyatom_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	for ( int ii = 0; ii < rotamer_trie::MAX_ATS_PER_RES + 1 + 1; ++ii )
	{
		heavyatom_stack[ii] = -1;
	}

	int rotamers_in_subtree_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	for ( int ii = 0; ii < rotamer_trie::MAX_ATS_PER_RES + 1 + 1; ++ii )
	{
		rotamers_in_subtree_stack[ii] = 0;
	}

	vector< int > heavy_depth_stack( max_stack_height_ + 1);
	heavy_depth_stack[2] = -1;
	heavy_depth_stack[1] = 0;
	int stack_top = 2;

	for ( int ii = 1; ii <= num_atoms_in_trie; ++ii )
	{
		if ( curr_trie[ii].flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{	for (int jj = heavy_depth_stack[stack_top-1] + 1;
				jj <= heavy_depth_stack[stack_top]; ++jj )
			{
				//apl test
				assert ( heavyatom_stack[jj] <= num_atoms_in_trie && heavyatom_stack[jj] > 0 );

				curr_trie[ heavyatom_stack[ jj ]].rotamers_in_subtree_ =
					rotamers_in_subtree_stack[jj];
				rotamers_in_subtree_stack[jj] = 0;
			}
			--stack_top;
		}

		if ( curr_trie[ii].has_sibling() )
		{	++stack_top;
			heavy_depth_stack[ stack_top ] = heavy_depth_stack[ stack_top - 1 ];
		}

		if (! curr_trie[ii].is_hydrogen() )
		{
			++heavy_depth_stack[stack_top];
			heavyatom_stack[ heavy_depth_stack[stack_top] ] = ii;
		}

		if ( curr_trie[ii].is_rotamer_terminal() )
		{
			for ( int jj = 1; jj <= heavy_depth_stack[stack_top]; ++jj )
				++rotamers_in_subtree_stack[jj];
		}
	}
	for ( int ii = 1; ii <= heavy_depth_stack[ stack_top]; ++ii )
	{
		//apl test
		assert ( heavyatom_stack[ii] <= num_atoms_in_trie && heavyatom_stack[ii] > 0 );

		curr_trie[ heavyatom_stack[ii] ].rotamers_in_subtree_ =
			rotamers_in_subtree_stack[ii];
	}

	//cerr << "done..." << endl;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::calculate_subtree_containing_radii()
///
/// @brief
/// Subtree pruning mechanisms require each heavy atom hold a bounding sphere
/// for all heavy atoms in their subtree.  The "subtree interaction sphere"
/// for heavy atom r is centered at r, with radius exactly:
/// [max_(i in r's subtree) dis(i,r) + pdbstatistics_pack::fa_max_dis / 2]
///
/// @detailed
/// linear time computation: O(max_heavy * num_heavyatoms) = O(num_heavyatoms)
/// computes subtree interaction sphere radii
///
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::calculate_subtree_containing_radii()
{	using namespace pdbstatistics_pack;

	trie_node* curr_trie = get_trie_for_curr_bb();
	int num_atoms_in_trie = get_num_atoms_for_curr_bb();

	//cerr << "calculate_subtree_containing_radii()" << endl;
	int heavyatom_stack[ rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	for ( int ii = 0; ii < rotamer_trie::MAX_ATS_PER_RES + 1 + 1; ++ii )
	{
		heavyatom_stack[ii] = -1;
	}

	//vector<int> heavyatom_stack( rotamer_trie::MAX_ATS_PER_RES + 1 + 1);

	FArray2D_float heavyatom_centers(3, rotamer_trie::MAX_ATS_PER_RES+1+1, 0.0);

	vector< int > heavy_depth_stack( max_stack_height_ + 1 );
	heavy_depth_stack[2] = -1;
	heavy_depth_stack[1] = 0;
	int stack_top = 2;


	float maxd2_in_subtree_stack[rotamer_trie::MAX_ATS_PER_RES + 1 + 1];
	for ( int ii = 0; ii < rotamer_trie::MAX_ATS_PER_RES + 1 + 1; ++ii )
		maxd2_in_subtree_stack[ii] = 0;

	//int heavyatom_stack_depth = 0;

	for ( int ii = 1; ii <= num_atoms_in_trie; ++ii )
	{
		if ( curr_trie[ii].flags_ & trie_node::FIRST_ATOM_IN_BRANCH)
		{	for (int jj = heavy_depth_stack[stack_top-1] + 1;
				jj <= heavy_depth_stack[stack_top]; ++jj )
			{
				float subtree_plus_interaction_diameter =
					std::sqrt(maxd2_in_subtree_stack[ jj ]) + fa_max_dis;

				curr_trie[ heavyatom_stack[ jj ]].radius_containing_subtree_ =
					subtree_plus_interaction_diameter * subtree_plus_interaction_diameter;
				maxd2_in_subtree_stack[jj] = 0;
			}
			--stack_top;
		}

		if ( curr_trie[ii].has_sibling() )
		{	++stack_top;
			heavy_depth_stack[ stack_top ] = heavy_depth_stack[ stack_top - 1 ];
		}

		if ( ! curr_trie[ii].is_hydrogen() )
		{
			++heavy_depth_stack[stack_top];
			heavyatom_stack[ heavy_depth_stack[stack_top] ] = ii;
			heavyatom_centers(1, heavy_depth_stack[stack_top]) =
				curr_trie[ii].ad_.get_xyz(0);
			heavyatom_centers(2, heavy_depth_stack[stack_top]) =
				curr_trie[ii].ad_.get_xyz(1);
			heavyatom_centers(3, heavy_depth_stack[stack_top]) =
				curr_trie[ii].ad_.get_xyz(2);

			for ( int jj = 1; jj <= heavy_depth_stack[stack_top] - 1; ++jj )
			{	float d2;
				distance2_bk(heavyatom_centers(1,jj),
					heavyatom_centers(1,heavy_depth_stack[stack_top]), d2);
				if (d2 > maxd2_in_subtree_stack[jj])
				{
					maxd2_in_subtree_stack[jj] = d2;
				}
			}
		}

	}
	for ( int ii = 1; ii <= heavy_depth_stack[ stack_top]; ++ii )
	{
		float subtree_plus_interaction_diameter =
			std::sqrt(maxd2_in_subtree_stack[ ii ]) + fa_max_dis;

		curr_trie[ heavyatom_stack[ii] ].radius_containing_subtree_ =
			subtree_plus_interaction_diameter * subtree_plus_interaction_diameter;
	}

	//cerr << "done..." << endl;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::decide_calc_hbe_for_pair()
///
/// @brief
/// complex logic to determine if a hydrogen / other atom pair could form a
/// hydrogen bond
///
/// @detailed
/// half the logic is simple, half complicated.  The simple part: at1 and
/// at2 must be a proper donor/acceptor pair (respectively).  The complex
/// part deals with the formation of sidechain/backbone hydrogen bonds.
/// If a backbone hydrogen bonding atom already participates in a
/// hydrogen bond with another backbone atom (e.g. as in an alpha helix)
/// then hydrogen bonds with side chain hydrogen bonding atoms will not
/// count.  The final condition prevents bb / bb and bb / sc hbonds between
/// residues that are fewer than 3 residues appart.  sc/sc hbond formation
/// is still allowed.
////////////////////////////////////////////////////////////////////////////////
inline
bool rotamer_trie::decide_calc_hbe_for_pair(
	bool at1_is_donorH,
	bool at2_is_acceptor,
	bool at1_isbb,
	bool at2_isbb,
	bool at1_bbhbg_sat,
	bool at2_bbhbg_sat,
	int at1_res,
	int at2_res
) const
{
	return ( (at1_is_donorH && at2_is_acceptor) &&
				(at1_isbb || ! at2_bbhbg_sat)      &&
				(at2_isbb || ! at1_bbhbg_sat) &&
				((std::abs(at1_res - at2_res) > 2) || (!at1_isbb && !at2_isbb)) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::get_atomdon_atomact_hbE
///
/// @brief
/// calculates the hydrogen bonding energy between two atoms
///
/// @detailed
///
///
/// @param donor - [in] - the atom descriptor for the donor hydrogen
/// @param acceptor - [in] - the atom descriptor for the acceptor heavy atom
/// @param don_res - [in] - the residue index for the donor
/// @param act_res - [in] - the residue index for the acceptor
/// @param don_is_sc - [in] - true when the donor atom is part of the side chain
/// @param acc_is_sc - [in] - true when the acceptor atom is part of the side chain
/// @param don_nb - [in] - the number of neighbors for the donor residue
/// @param acc_nb - [in] - the number of neighbors for the acceptor residue
/// @param acceptor_hybridization - [in] - the hybridization of the acceptor
/// @param hbE - [out] - the hydrogen bond energy
///
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::get_atomdon_atomact_hbE(
	atom_descriptor donor,
	atom_descriptor acceptor,
	int don_res,
	int act_res,
	bool don_is_sc, //JSS to remove these later
	bool acc_is_sc,
	int don_nb,
	int acc_nb,
	float & hbE
) const
{	using namespace hbonds;
	using namespace param_rotamer_trie;
	using namespace pdbstatistics_pack;

	hbE = 0;
	bool scbond = (don_is_sc || acc_is_sc);
	HBEvalType const hb_eval_type
		= hbond_evaluation_type_AH(donor.get_atom_type(), acceptor.get_atom_type(),
															 don_res, act_res);
	float energy;
	hb_energy_deriv(hb_eval_type, donor.get_xyz(), donor.get_orientation_vector(),
		acceptor.get_xyz(), acceptor.get_orientation_vector(), energy);

	if (energy < MAX_HB_ENERGY) {
		float weight;

		if ( scbond && use_W_hb_env_dep_tk ) {
			if ( smooth_env_dep_hb ) weight = hb_env_dep_burial_lin(acc_nb,don_nb);
			else weight = hb_env_dep_burial_tk(acc_nb,don_nb);
		} else {
			weight = 1.0; // default constant weight
		}
		hbE = weight*energy;
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_trie::update_coordinates
///
/// @brief
/// writes new coordinates into a one-rotamer trie
/// useful in rotamer_trials.
///
/// @detailed
/// recomputes subtree interaction sphere radii
////////////////////////////////////////////////////////////////////////////////
void
rotamer_trie::update_coordinates( FArray2Da_float xyz_new, int aa, int aav )
{
	assert(num_total_rotamers_ == 1);
	assert(num_distinct_backbones_ == 1);

	rotamer_descriptor rd;
	create_rotamer_descriptor(res_id_, rd, aa, aav, xyz_new, 1);
	for (int ii = 1; ii <= num_atoms_for_bb_(1); ++ii )
		tries_(1)[ii].ad_ = rd.get_atom( ii );
	set_backbone_focus( 1 );
	calculate_subtree_containing_radii();
	set_no_backbone_focused();
	return;
}

void rotamer_trie::set_backbone_focus(int current_backbone) const
{
	assert(current_backbone > 0 && current_backbone <= num_distinct_backbones_);
	current_backbone_ = current_backbone;
}

void rotamer_trie::set_no_backbone_focused() const
{
	current_backbone_ = 0;
	return;
}

int rotamer_trie::get_num_atoms_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return num_atoms_for_bb_( current_backbone_ );
}

int rotamer_trie::get_num_heavy_atoms_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return num_heavyatoms_for_bb_( current_backbone_ );
}

int rotamer_trie::get_num_unique_rotamers_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return num_unique_rotamers_for_bb_( current_backbone_);
}

int rotamer_trie::get_num_total_rotamers_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return num_total_rotamers_for_bb_( current_backbone_);
}

int rotamer_trie::get_rotamer_offset_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return total_rotamer_offset_for_bb_( current_backbone_ );
}
trie_node* rotamer_trie::get_trie_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return tries_( current_backbone_ );
}

vector< int > const & rotamer_trie::get_total_2_unique_for_curr_bb() const
{
	assert(current_backbone_ != 0);
	return total_rot_2_unique_preorder_[ current_backbone_ ];
}


////////////////////////////////////////////////////////////////////////////////
/// @begin operator << (ostream & os, const rotamer_trie & rt)
///
/// @brief
/// outputs the rotamer trie to the output stream
///
/// @detailed
////////////////////////////////////////////////////////////////////////////////
ostream & operator << (ostream & os, const rotamer_trie & rt)
{
	os << "rotamer_trie:  " << rt.res_id_ << std::endl;
	if ( rt.num_distinct_backbones_ > 1 )
	{
		os << "num backbones: " << rt.num_distinct_backbones_ << std::endl;
	}
	for (int ii = 1; ii <= rt.num_distinct_backbones_; ++ii)
	{

		if (rt.num_distinct_backbones_ > 1 )
		{
			os << "BACKBONE " << ii << std::endl;
		}

		os << "num atoms: " << rt.num_atoms_for_bb_(ii) << " num heavy atoms: " << rt.num_heavyatoms_for_bb_(ii) << std::endl;
		os << "max stack height: " << rt.max_stack_height_ << " max heavy atom depth: " << rt.max_heavyatom_depth_ << std::endl;

		os << "total rotamer ordering 2 unique preorder ordering: " << endl;
		int ii_num_total_rotamers = rt.num_total_rotamers_for_bb_(ii);
		for ( int jj = 1; jj <= ii_num_total_rotamers; ++jj )
			os << rt.total_rot_2_unique_preorder_[ii][jj] << " ";
		os << std::endl;

		int ii_num_atoms = rt.num_atoms_for_bb_(ii);
		int ii_rotamers_seen = 0;
		for ( int jj = 1; jj <= ii_num_atoms; ++jj )
		{
			os << "node : " << jj;
			os << rt.tries_(ii)[jj].get_atom() << " ";
			os << (int) rt.tries_(ii)[jj].sibling_ << " ";
			os << "h" << ( int ) rt.tries_(ii)[jj].hybridization_ << " ";
			os << "a" << rt.tries_(ii)[jj].get_aatype()  << " ";
			os << "v" << ( int ) rt.tries_(ii)[jj].aa_variant_ << " ";
			os << rt.tries_(ii)[jj].is_hydrogen() << " ";
			os << rt.tries_(ii)[jj].is_rotamer_terminal() << " ";
			os << (int) rt.tries_(ii)[jj].rotamers_in_subtree_ << " ";
			os << rt.tries_(ii)[jj].radius_containing_subtree_ << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::ROTAMER_TERMINAL) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::IS_H            ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::IS_BB           ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::IS_DONOR_H      ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::IS_ACCEPTOR     ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::IS_BBHBG_SAT    ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags_ & trie_node::HAS_SIBLING     ) << " ";

			os << "|";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_N    ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_H    ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_CA   ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_HA   ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_C    ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::BB_O    ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::SC_CB   ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::SC_OTHER) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::PRO_CD  ) << " ";
			os << (bool) (rt.tries_(ii)[jj].flags2_ & trie_node::PRO_HD  ) << " ";

			if ( rt.tries_(ii)[jj].flags_ & trie_node::ROTAMER_TERMINAL)
			{
				++ii_rotamers_seen;
				os << ii_rotamers_seen;
			}
			os << std::endl;
		}
	}
	os << "---" << std::endl;

	return os;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_isbb
///
/// @brief
/// returns true when an atom is a part of the backbone
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_isbb( int atomtype )
{	return (((atomtype >= 17) && (atomtype <= 20))
		|| (atomtype == 25) || (atomtype == 12) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_isheavyatom
///
/// @brief
/// returns true for all non-hydrogen atom types
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_isheavyatom( int atomtype)
{	return ((atomtype < 22) || (atomtype > 26));}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_isdonorH
///
/// @brief
/// returns true when an atom is a hydrogen and capable of forming an hbond
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_isdonorH (int atomtype )
{	return ((atomtype == 22) || (atomtype == 25));}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_isacceptor
///
/// @brief
/// returns true when an atom is a heavy atom, and can form a hydrogn bond
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_isacceptor ( int atomtype )
{	return (atomtype == 8 || atomtype == 13 || atomtype == 14 ||
		atomtype == 15 || atomtype == 20) ;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_hbchk_bbhbg_sat
///
/// @brief
/// returns true when an atom is either the backbone oygen or the backbone
/// amide hydrogen, and when this atom is already participating in a
/// backbone/backbone hydrogen bond
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_hbchk_bbhbg_sat (int atomtype, FArray1DB_bool & bb_has_hydrogenbonds)
{	//bb_has_hydrogenbonds.dimension(2);
	return ( (atomtype == 25  && bb_has_hydrogenbonds(1)) ||
		(atomtype == 20 && bb_has_hydrogenbonds(2)) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_is_bb_n
///
/// @brief
/// returns true for the two backbone nitrogen atom types
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_is_bb_n (int atomtype ) { return (atomtype == 17 || atomtype == 12);}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_is_bb_h
///
/// @brief
/// returns true for the backbone amino hydrogen
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_is_bb_h (int atomtype ) {return (atomtype == 25);}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_is_bb_ca
///
/// @brief
/// returns true for the backbone c-alpha carbon
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_is_bb_ca (int atomtype ) {return (atomtype == 18);}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_is_bb_c
///
/// @brief
/// returns true for the backbone carboxyl carbon
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_is_bb_c (int atomtype ) {return (atomtype == 19);}

////////////////////////////////////////////////////////////////////////////////
/// @begin atomtype_is_bb_o
///
/// @brief
/// returns true for the backbone carboxyl oxygen
///
/// @detailed
/// dangerously hard coded "magic numbers"
/// atom type enums would be good here
////////////////////////////////////////////////////////////////////////////////
bool
atomtype_is_bb_o (int atomtype ) {return (atomtype == 20);}

bool
rotamer_atom_in_sidechain_proper(
	rotamer_descriptor const & rotamer,
	int atom_id_in_rotamer
)
{
	using namespace param_rotamer_trie;
	if (rotamer.get_aa_type() == aa_gly )
	{
		return false;
	}
	else if (rotamer.get_aa_type() == aa_pro )
	{
		if (rotamer.get_aa_variant() == 1)
		{
			if (atom_id_in_rotamer <= 5) //5 == pro HA
				return false;
		}
		else if ( rotamer.get_aa_variant() == trie_variant_for_n_terminus )
		{
			if (atom_id_in_rotamer <= 7) //pro HA
				return false;
		}
		else if ( rotamer.get_aa_variant() == trie_variant_for_c_terminus )
		{
			if (atom_id_in_rotamer <= 6) //pro HA
				return false;
		}

	}
	else
	{
		if (rotamer.get_aa_variant() == 1 || rotamer.get_aa_variant() == 2)
		{
			if (atom_id_in_rotamer <= 6)
			{
				return false;
			}
		}
		else if ( rotamer.get_aa_variant() == trie_variant_for_n_terminus )
		{
			if (atom_id_in_rotamer <= 8)
			{
				return false;
			}
		}
		else if ( rotamer.get_aa_variant() == trie_variant_for_c_terminus )
		{
			if (atom_id_in_rotamer <= 7)
			{
				return false;
			}
		}
	}
	return true;

}


