// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14207 $
//  $Date: 2007-04-13 09:52:41 -0400 (Fri, 13 Apr 2007) $
//  $Author: leaverfa $

#ifndef INCLUDED_ROTAMER_DESCRIPTOR_H_
#define INCLUDED_ROTAMER_DESCRIPTOR_H_


// Rosetta Headers
#include "atom_descriptor.h"

// Utility Headers
#include <utility/vector1.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>

// C++ Headers
#include <iosfwd>
#include <vector>

extern bool trie_output_all_energy_calcs;

class rotamer_descriptor;
class trie_node;
class rotamer_trie;


class rotamer_descriptor
{
public:
	rotamer_descriptor();
	~rotamer_descriptor();
	rotamer_descriptor( const rotamer_descriptor & rhs );
	rotamer_descriptor(int);

	void set_atom(int, const atom_descriptor &);
	void set_rotamer_id(int);
	void set_aa_type(int);
	void set_aa_variant(int);
	int  get_num_atoms() const;
	int  get_rotamer_id() const;
	int  get_aa_type() const;
	int  get_aa_variant() const;

	atom_descriptor get_atom(int) const;

	bool operator < (const rotamer_descriptor & rhs) const;
	int  count_atoms_in_common( const rotamer_descriptor & rhs) const;

	friend std::ostream & operator << ( std::ostream & os, const rotamer_descriptor & rd);

private:
	int             num_atoms_;
	int 				 rotamer_id_;
	int				 aa_type_;
	int				 aa_variant_;
	atom_descriptor atoms_[40];

};

std::ostream & operator << ( std::ostream & os, const rotamer_descriptor & rd);

class trie_node
{
public:
	trie_node();
	~trie_node();
	trie_node( const trie_node & rhs);

	trie_node( const atom_descriptor & atom, FArray1DB_bool & hbchk, int aatype);

	inline bool is_rotamer_terminal() const;
	inline bool is_hydrogen()         const;
	inline bool is_backbone()         const;
	inline bool is_donor_h()          const;
	inline bool is_acceptor()         const;
	inline bool is_bbhbg_sat()        const;
	inline bool has_sibling()         const;
	atom_descriptor get_atom() const;

	void set_terminal( bool );
	// void set_child(int);
	void set_sibling( int );
	void set_first_atom_in_branch();
	void set_is_bb_c(); //necessary for c-terminus variants
	void set_is_bb_ha();
	void set_is_sc_cb();
	void set_is_pro_cd();
	void set_is_pro_hd();
	void set_is_gly_ha();

	void set_aatype( int );
	void set_aavariant( int );
	int get_aatype() const;
	int get_aavariant() const;

	inline int get_res_i_equivalence_class();

	friend class rotamer_trie;
	friend std::ostream & operator << ( std::ostream & os, const rotamer_trie & rt );

private:
	//40 bytes total -- allign to the 4 and 8 byte boundaries.

	//28 bytes
	atom_descriptor ad_; //two floats, one unsigned char
	unsigned char  flags_;
	unsigned short	flags2_;

	//2 bytes
	unsigned char hybridization_;
	unsigned char aa_variant_;
	//2 bytes
	unsigned short rotamers_in_subtree_;
	//4 bytes
	unsigned int   sibling_;
	//4 bytes
	float          radius_containing_subtree_;

	// What _flags replaced:
	//bool  _rotamer_terminal;
	//bool  _is_H;
	//bool  _is_bb;
	//bool  _is_donorH;
	//bool  _is_acceptor;
	//bool  _is_bbhbg_sat;
	// - sort of - bool _has_sibling; unsigned short _sibling;

	static const unsigned char ROTAMER_TERMINAL = 0x01;
	static const unsigned char IS_H             = 0x02;
	static const unsigned char IS_BB            = 0x04;
	static const unsigned char IS_DONOR_H       = 0x08;
	static const unsigned char IS_ACCEPTOR      = 0x10;
	static const unsigned char IS_BBHBG_SAT     = 0x20;
	static const unsigned char HAS_SIBLING      = 0x40;
	static const unsigned char FIRST_ATOM_IN_BRANCH = 0x80;

	static const unsigned short BB_N					= 0x0001;
	static const unsigned short BB_H					= 0x0002;
	static const unsigned short BB_CA				= 0x0004;
	static const unsigned short BB_HA            = 0x0008;
	static const unsigned short BB_C					= 0x0010;
	static const unsigned short BB_O					= 0x0020;
	static const unsigned short SC_CB            = 0x0040;
	static const unsigned short SC_OTHER         = 0x0080;
	static const unsigned short PRO_CD           = 0x0100;
	static const unsigned short PRO_HD           = 0x0200;

	static const unsigned short BB_CA_H_PCD		= BB_CA | BB_H | PRO_CD;
	static const unsigned short BB_N_CA_PCD		= BB_N | BB_CA | PRO_CD;
	static const unsigned short BB_N_CA_H_PCD		= BB_N | BB_CA | BB_H | PRO_CD;
	static const unsigned short BB_C_O_CB_HA_PHD = BB_C | BB_O | SC_CB | BB_HA | PRO_HD;
	static const unsigned short BB_N_CA_C_CB_H_HA_PCD_PHD = BB_N | BB_C | BB_CA | SC_CB | BB_H | BB_HA | PRO_CD | PRO_HD;

	static const unsigned short BB_N_CB_HA       = BB_N | SC_CB | BB_HA; //res i, equiv class I
	static const unsigned short BB_CA_O          = BB_CA | BB_O;			//res i, equiv class II

	static const unsigned short NOT_BB_N         = 0x03FF & (~BB_N);
	static const unsigned short NOT_BB_C         = 0x03FF & (~BB_C);
	static const unsigned short BB_CA_PCD			= BB_CA | PRO_CD;
	static const unsigned short BB_CA_C_O			= BB_CA | BB_C | BB_O;
	static const unsigned short BB_C_CB_HA_PHD	= BB_C | SC_CB | BB_HA | PRO_HD;
	static const unsigned short BB_N_CB_H_HA		= BB_N | SC_CB | BB_H | BB_HA;
	static const unsigned short BB_N_CA_C_O_CB_HA= BB_N | BB_CA | BB_C | BB_O | SC_CB | BB_HA;

	//Five equivalence classes for res i in the interaction with res i+1.
	//Equivalence classes originally defined in count_pair.h
	// I:		N, CB, HA
	// II:	CA, O
	// III:	C
	// IV:	SC (not CB)
	//	V:		H (HN)

	//5 bits for amino acid type
	//NOTE: The ONLY purpose of tracking the amino acid type for each h-bonding atom is to
	//support the atom-and-amino-acid-specific hydrogen bonding weights defined in
	//sc_hbW
	//sc_hbW is not amino-acid-variant friendly, so the weights it stores stop making sense
	//for alternate variants.  What should the weight be for proline's 1HD?  I ask because that's
	//where the proline's N terminus variant is going to look to find the weight for 2H.
	//Do we change sc_hbW to be variant friendly?  I suggest we remove sc_hbW.  Here's why.
	//With the inclusion of termini, we have new h-bonding side chain atoms OXT, 2H and 3H.
	//These atoms are shared between all rotamers and are optimally included early in the
	//trie ordering.  The single copy of OXT in the trie will hold the aa and aav it originated from
	//(This is almost always proline, though this detail will change
	//if the definition of "operator < ()" changes for class atom_descriptor)
	//When OXT forms an hbond - it will be weighted by the proline OXT weight.  This hbond
	//energy is shared for all rotamers including those from other amino acids.
	//It means that if you want weights to differ between the OXT on prolien and valine, the
	//trie will not give you the energies you ask for.  This is problem.
	//At the moment, there's a line of code that nullifies all my concerns: sc_hbW = 1;
	//All weights are equal.
	//Anyone is free to change that line of code and the consequence will be a divergence between
	//get energies and trie_vs_trie.
	static const int MAX_HBONDING_AA_TYPES = 32; //2^(num_bits_for_aatype);
	//if you want more h-bonding types, must use more bits in flags2
	static const unsigned short AA_BITS;
  static const int AA_BITSHIFT;
};

class rotamer_trie
{
public:
	rotamer_trie();
	~rotamer_trie();
	rotamer_trie(
		int,
		std::vector< rotamer_descriptor > & rotamers
	);

	rotamer_trie(
		int,
		std::vector< rotamer_descriptor > & rotamers,
		utility::vector1< int > & num_rotamers_per_frag
	);

	rotamer_trie(
		int resind,
		rotamer_descriptor & rotamer,
		bool build_sc_trie_only = false
	);


	int get_num_atoms() const;
	int get_num_total_rotamers() const;
	int get_resid() const;

	//wrapper function
	void
	trie_vs_trie
	(  rotamer_trie & rt,
		FArray2D_float & rotamer_rotamer_table,
		FArray2D_float & temp_table,
		bool same_fragment = false
	);

	float
	path_vs_path
	(
		rotamer_trie const & rt
	) const;

	float
	path_vs_path_gb
	(
		rotamer_trie &
	) { return 0;} //stubbed; coming soon

	//wrapper function
	void
	trie_vs_background
	(	rotamer_trie & bg,
		FArray2Da_float trie_trials_componentE,
		FArray1Da_float trie_trials_sumE
	);

	//the following function for single-rotamer tries only.
	void
	update_coordinates( FArray2Da_float xyz_new, int aa, int aav );

	friend class trie_node;
	friend std::ostream & operator << ( std::ostream & os, const rotamer_trie & rt );

	static const int MAX_ATS_PER_RES = 40; //replacement for late constant MAX_ATOM

private:

	void
	construct_rotamer_trie(
		int resind,
		std::vector< rotamer_descriptor > & rotamers,
		utility::vector1< int > & rotamer_offsets_for_backbone
	);

	void
	construct_one_rotamer_trie(
		rotamer_descriptor & rotamer
	);

	void add_atom_to_trie(
		int trie_atom_id,
		int atom_id_in_rotamer,
		atom_descriptor const & ad,
		FArray1DB_bool & hbchk_this_res,
		int aa_type,
		int aa_variant
	);

	rotamer_trie(const rotamer_trie & rhs); // no default copy constructor;

	float
	path_vs_path_i_ip3orMore
	(
		rotamer_trie const & rt
	) const;

	float
	path_vs_path_i_ip1or2
	(
		rotamer_trie const & rt
	) const;

	void  trie_v_trie_iter_i_ip1
   (	rotamer_trie & rt,
   	FArray2DB_float & rot_rot_table
   );

	void  trie_v_trie_iter_i_ip2
   (	rotamer_trie & rt,
   	FArray2DB_float & rot_rot_table
   );


	void  trie_v_trie_iter_i_ip3orMore
   (	rotamer_trie & rt,
   	FArray2DB_float & rot_rot_table
   );

	static
   void prepare_count_pair_equivalence_class_masks
   (  int bg_res,
   	int rot_trie_res,
   	unsigned short this_equivalence_class_bitmasks[6],
   	unsigned short bg_equivalence_class_bitmasks[6][2]
   );

	void trie_vs_background_i_ip1or2
	(	rotamer_trie & bg,
		FArray2DB_float & trie_trials_componentE,
		FArray1DB_float & trie_trials_sumE
	);

	void trie_vs_background_i_ip3orMore
	(	rotamer_trie & bg,
		FArray2DB_float & trie_trials_componentE,
		FArray1DB_float & trie_trials_sumE
	);

   void convert_preorder_rot_rot_table
   (  rotamer_trie & rt,
   	FArray2DB_float & rot_rot_preorder,
   	FArray2DB_float & rot_rot_original_order
	);

	void convert_preorder_trials_data
	(	FArray2DB_float & trie_trials_componentE_preorder,
		FArray2DB_float & trie_trials_componentE,
		FArray1DB_float & trie_trials_sumE_preorder,
		FArray1DB_float & trie_trials_sumE
	);


	float atom_atom_energy
	(	trie_node & r,
		trie_node & s,
		int r_res,
		int s_res,
		int R_num_neighbors,
		int S_num_neighbors,
		float * Whbond
	) const;

	void atom_atom_energy
	(	trie_node & r,
		trie_node & s,
		int r_res,
		int s_res,
		int R_num_neighbors,
		int S_num_neighbors,
		float * Whbond,
		float & repE,
		float & hbE,
		float & sumE
	) const;

	float heavyatom_heavyatom_energy
	(  trie_node & r,
	   trie_node & s,
	   float & d2
	) const;

	void heavyatom_heavyatom_energy
	(  trie_node & r,
	   trie_node & s,
	   float & d2,
		float & atrE,
		float & repE,
		float & solE,
		float & sumE
	) const;

	void
  get_atomdon_atomact_hbE(
    atom_descriptor donor,
    atom_descriptor acceptor,
    int don_res,
    int act_res,
    bool don_is_sc, //JSS to remove these later
    bool acc_is_sc,
    int don_nb,
    int acc_nb,
    float & hbE
    ) const;

	inline
	bool decide_calc_hbe_for_pair(
	   bool at1_is_donorH,
	   bool at2_is_acceptor,
	   bool at1_isbb,
	   bool at2_isbb,
	   bool at1_bbhbg_sat,
	   bool at2_bbhbg_sat,
		int at1_res,
		int at2_res
	) const;

	void compute_max_stack_height();

	void set_backbone_focus(int current_backbone) const;
	void set_no_backbone_focused() const;

	int get_num_atoms_for_curr_bb() const;
	int get_num_heavy_atoms_for_curr_bb() const;
	int get_num_unique_rotamers_for_curr_bb() const;
	int get_num_total_rotamers_for_curr_bb() const;
	int get_rotamer_offset_for_curr_bb() const;
	trie_node* get_trie_for_curr_bb() const;
	std::vector< int > const & get_total_2_unique_for_curr_bb() const;

	void calculate_num_rotamers_in_subtree();
	void calculate_subtree_containing_radii();

	int                 num_unique_rotamers_;
	int                 num_total_rotamers_;
	int					  num_total_atoms_;

	int                 res_id_;

	FArray1D< trie_node* > tries_;

	int                 num_distinct_backbones_;
	FArray1D_int        num_atoms_for_bb_;
	FArray1D_int		  num_heavyatoms_for_bb_;
	FArray1D_int        num_unique_rotamers_for_bb_;
	FArray1D_int        num_total_rotamers_for_bb_;
	FArray1D_int        total_rotamer_offset_for_bb_;

	std::vector< std::vector< int > >  total_rot_2_unique_preorder_;

	//which backbone am I currently focused on?
	mutable int			  current_backbone_;

	int                 max_heavyatom_depth_;
	int					  max_stack_height_;

	//long long			  _heavy_heavy_evals_count;

	int					num_neighbors_; //for environment dependent hydrogen bond energies.
};

std::ostream & operator << ( std::ostream & os, const rotamer_trie & rt);

bool
atomtype_isbb( int atomtype );

bool
atomtype_isheavyatom( int atomtype);

bool
atomtype_isdonorH (int atomtype );

bool
atomtype_isacceptor ( int atomtype );


bool
atomtype_is_bb_n (int atom_type );

bool
atomtype_is_bb_h (int atom_type );

bool
atomtype_is_bb_ca (int atom_type );

bool
atomtype_is_bb_c (int atom_type );

bool
atomtype_is_bb_o (int atom_type );

bool
rotamer_atom_in_sidechain_proper(
	rotamer_descriptor const & rotamer,
	int atom_id_in_rotamer
);

#endif
