// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18940 $
//  $Date: 2007-12-06 15:30:12 -0500 (Thu, 06 Dec 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "files_paths.h"
#include "options.h"
#include "score_ns.h"
#include "score_data.h"
#include "score.h" // setup weight maps

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <iostream>
#include <sstream>
#include <fstream>
#include <string>

//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80

namespace pose_ns {

	/////////////////////////////////////////////////////////
	// Stored_score
	//
	// constructor
	Stored_score::Stored_score( int dim_in, int size_in ) :
		dimension( dim_in ),
		state( BAD ),
		size( size_in ),
		data0( 0.0 ),
		data1( dimension == 1 ? new FArray1D_float( size ) : 0 ),
		data2( dimension == 2 ? new FArray2D_float( size, size ) : 0 ),
		data3( dimension == 3 ? new FArray3D_float( size, size, size ) : 0 )
	{}

	Stored_score::Stored_score( int dim_in, int size1, int size2, int size3 ) :
		dimension( dim_in ),
		state( BAD ),
		size( size1 ),
		data0( 0.0 ),
		data1( dimension == 1 ? new FArray1D_float( size1 ) : 0 ),
		data2( dimension == 2 ? new FArray2D_float( size1, size2 ) : 0 ),
		data3( dimension == 3 ? new FArray3D_float( size1, size2, size3 ) : 0 )
	{}

	// copy-constructor
	Stored_score::Stored_score ( Stored_score const & src ) :
		dimension( src.dimension ),
		state ( src.state ),
		size ( src.size ),
		data0( dimension == 0 ? src.data0 : 0.0 ),
		data1( dimension == 1 ? new FArray1D_float( *src.data1 ) : 0 ),
		data2( dimension == 2 ? new FArray2D_float( *src.data2 ) : 0 ),
		data3( dimension == 3 ? new FArray3D_float( *src.data3 ) : 0 )
	{}

	// destructor
	Stored_score::~Stored_score()
	{
		delete data1;
		delete data2;
		delete data3;
	}

	// operator=
	Stored_score &
	Stored_score::operator =( Stored_score const & src )
	{
		if ( dimension != src.dimension ) { //Objexx:SGM Is this more of an assert check?
			std::cout << "dimension mismatch in Stored_score::operator=" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		int const old_size( size );
		size = src.size;
		state = src.state;
		if ( dimension == 0 ) {
			data0 = src.data0;
		} else if ( dimension == 1 ) {
			if ( old_size != size ) data1->dimension( size );
			*data1 = *(src.data1);
		} else if ( dimension == 2 ) {
			if ( old_size != size ) data2->dimension( size, size );
			*data2 = *(src.data2);
		} else if ( dimension == 3 ) {
			if ( old_size != size ) data3->dimension( src.data3->size1(),
																								src.data3->size2(),
																								src.data3->size3());
			*data3 = *(src.data3);
		}
		return *this;
	}

	////////////////////////////////////////
	// Score_data
	bool Score_data::name2string_init( false );
	std::map< Score_name, std::string > Score_data::name2string_;

	void Score_data::show_scores( std::ostream & os, int const mode ) const {
		// mode == 0 : tag score
		// mode == 1 : score
		// mode == 2 : tag
		// mode == 3 : human-readable format
		initialize_name2string();
		if ( mode == 3 ) os << std::endl;

		for ( Score_map::const_iterator it=score_map.begin(), it_end = score_map.end();
		 it != it_end; ++it ) {
			if ( it->second->get_dimension() == 0 &&
					 it->second->get_state() == GOOD ) {
				std::map< Score_name, std::string >::const_iterator it2
					( name2string_.find( it->first ) );
				if ( it2 != name2string_.end() ) {
					if ( mode == 0 ) { // tag and score
						os << it2->second << ' ' << it->second->get0() <<' ';
					} else if ( mode == 1 ) { // score only
						os << F( 8, 2, it->second->get0()) << ' ';
					} else if ( mode == 3 ) { //ja easier to read
						os << A(15,it2->second) << ' ' << F(8,2,it->second->get0()) <<'\n';
					} else { // tag only
						os << A( 8, it2->second ) << ' ';
					}
				}
			}
		}
	}


	// constructor:
	Score_data::Score_data() {}

	// destructor
	Score_data::~Score_data() {
		// free all the pointers
		for ( Score_map::iterator it = score_map.begin(), it_end = score_map.end();
		 it != it_end; ++it ) {
			delete it->second;
		}
	}

	// operator=
	// only copies scores with state == GOOD
	Score_data & Score_data::operator =( const Score_data & source ) {
		const Score_map & src_map( source.score_map );

		// first set old data states to BAD
		// note that these may have the wrong dimension to begin with
		for ( Score_map::iterator it = score_map.begin(),
		 it_end = score_map.end(); it != it_end; ++it ) {
			it->second->set_state( BAD );
		}
		// now copy data from source:
		// NOTE we only copy the up-to-date scores!
		for ( Score_map::const_iterator it = src_map.begin(),
		 it_end = src_map.end(); it != it_end; ++it ) {
			if ( it->second->get_state() == GOOD ) {
				const Score_name name( it->first );
				Score_map::iterator it2( score_map.find( name ) );
				if ( it2 == score_map.end() ) {
					// Stored_score copy-constructor
					score_map.insert( std::make_pair( name, new Stored_score( *(it->second) ) ) );
				} else {
					// Stored_score operator=
					*(it2->second) = *(it->second);
				}
			}
		}
		return *this;
	}

	// when we're done scoring:
	void Score_data::after_scoring() {
		// anything that wasn't evaluated in this round goes to BAD
		// this because we are about to forget about the changes to
		// the coordinates; see Pose::after_scoring();
		switch_state( OK, BAD );
	}

	// after refolding
	void Score_data::after_refolding() {
		// things that were up-to-date are now only partially up-to-date
		switch_state( GOOD, OK );
	}

	/////////////////////////////////////////////////////////////////////////////
	// update the state of a particular score
	void
	Score_data::set_score_state_OK_if_GOOD(
		const Score_name name
	)
	{
		Score_map::iterator it( score_map.find( name ) );
		if ( it != score_map.end() ) {
			Stored_score & score( *(it->second));
			if ( score.get_state() == GOOD ) score.set_state( OK );
		}
	}

	// switch from src to dest
	void Score_data::switch_state( Score_state const src, Score_state const dest ) {
		for ( Score_map::iterator it = score_map.begin(), it_end = score_map.end();
		 it != it_end; ++it ) {
			if ( it->second->get_state() == src ) {
				it->second->set_state( dest );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// static function
	void
	Score_data::initialize_name2string()
	{
		if ( name2string_init ) return;
		name2string_init = true;
		name2string_.clear();
		name2string_[SCORE] = "SCORE";
		name2string_[VDW] = "VDW";
		name2string_[ENV] = "ENV";
		name2string_[PAIR] = "PAIR";
		name2string_[SSPAIR] = "SS";
		name2string_[RSIGMA] = "RSIGMA";
		name2string_[HS] = "HS";
		name2string_[SHEET] = "SHEET";
		name2string_[CB] = "CB";
		name2string_[RG] = "RG";
		name2string_[CO] = "CO";
		name2string_[DNA_BS] = "DNA_BS";
		name2string_[DNA_BP] = "DNA_BP";
		name2string_[RNA_BS] = "RNA_BS";
		name2string_[RNA_BP_W] = "RNA_BP_W";
		name2string_[RNA_BP_H] = "RNA_BP_H";
		name2string_[RNA_BP_S] = "RNA_BP_S";
		name2string_[RNA_AXIS] = "RNA_AXIS";
		name2string_[RNA_STAGGER] = "RNA_STAGGER";
		name2string_[RNA_BULGE] = "RNA_BULGE";
		name2string_[RNA_NONBASEBASE] = "RNA_NONBASEBASE";
		name2string_[RNA_O2STAR] = "RNA_O2STAR";
		name2string_[RNA_PHOSPHATE] = "RNA_PHOSPHATE";
		name2string_[RNA_CONTACT] = "RNA_CONTACT";
		name2string_[RNA_LONG_RANGE_CONTACT] = "RNA_LR";
		name2string_[RNA_SASA] = "RNA_SASA";
		name2string_[CONTACT_MJ] = "CONTACT";
		name2string_[RAMACHANDRAN] = "RAMACHANDRAN";
		name2string_[CHAINBREAK] = "CHAINBREAK";
		name2string_[DUMMY_MODEL] = "DUMMY_MODEL";
		name2string_[EDENSITY] = "EDENSITY";
		name2string_[NSD] = "NSD";
		name2string_[DNA_AA] = "DNA_AA";
		name2string_[DNA_R] = "DNA_R";
		name2string_[ATOMPAIR_CST] = "ATOMPAIR_CST";
		name2string_[COORD_CST] = "COORD_CST";
		name2string_[CHAINBREAK_CST] = "CHAINBREAK_CST";
		name2string_[PHIPSI] = "PHIPSI";
		name2string_[OMEGA] = "OMEGA";
		name2string_[PHIPSI_CST] = "PHIPSI_CST";
		name2string_[OMEGA_CST] = "OMEGA_CST";
		name2string_[CHI_CST] = "CHI_CST";
		name2string_[KIN_1D_CST] = "KIN_1D_CST";
		name2string_[KIN_3D_CST] = "KIN_3D_CST";
		name2string_[FA_ATR] = "FA_ATR";
		name2string_[FA_REP] = "FA_REP";
		name2string_[FA_SOL] = "FA_SOL";
    name2string_[FA_ELEC]= "FA_ELEC";
		name2string_[FA_PAIR] = "FA_PAIR";
		name2string_[FA_DUN] = "FA_DUN";
		name2string_[FA_PROB] = "FA_PROB";
		name2string_[FA_H2O] = "FA_H2O";
		name2string_[HB_SRBB] = "HB_SRBB";
		name2string_[HB_LRBB] = "HB_LRBB";
		name2string_[HB_SC] = "HB_SC";
		name2string_[FA_INTRA] = "FA_INTRA";
		name2string_[GB] = "GB";
		name2string_[FA_REF] = "FA_REF";
		name2string_[PLANE] = "PLANE";
		name2string_[GSOLT] = "GSOLT";
		name2string_[SASA] = "SASA";
		name2string_[SASA_PACK] = "SASA_PACK";
		name2string_[DELG_PACK] = "DELG_PACK";
		name2string_[ENERGY_PACK] = "E_PACK";
		name2string_[BOND_ANGLE] = "BOND_ANGLE";
		name2string_[RMSD] = "rms"; // sorry need this to be lowercase for boinc
		name2string_[CO] = "co"; // sorry need this to be lowercase for boinc
		name2string_[CONTACT_MJ] = "CONTACT";
		name2string_[SIDECHAIN_BOND_ANGLE] = "SIDECHAIN_BOND_ANGLE";
		name2string_[MAXSUB] = "MAXSUB";
		name2string_[RMSD_MIN] = "RMS_MIN";
		name2string_[RMSD_G] = "RMSD_G";
		name2string_[START_RMSD] = "START_RMSD";
		name2string_[BARCODE] = "BARCODE";
		name2string_[BARCODE_ENERGY] = "BAR_ENRG";
		name2string_[CST_SCORE] = "CST";
		name2string_[DOCK_ENV] = "DOCK_ENV"; // start docking
		name2string_[DOCK_PAIR] = "DOCK_PAIR";
		name2string_[DOCK_CONT] = "DOCK_CONT";
		name2string_[DOCK_VDW] = "DOCK_VDW";
		name2string_[DOCK_SITE_CST] = "DOCK_SITE_CST";
		name2string_[DOCK_FAB] = "DOCK_FAB";
		name2string_[DOCK_WSL_ELEC] = "DOCK_WSL_ELEC";
		name2string_[LOOP_CONTACT] = "LOOP_CONTACT"; // aroop
		name2string_[RNA_BASE_STACK_ARRAY] = "RNA_BASE_STACK_ARRAY";
		name2string_[RNA_BASE_STACK_AXIS_ARRAY] = "RNA_BASE_STACK_AXIS_ARRAY";
		name2string_[RNA_BASE_PAIR_ARRAY] = "RNA_BASE_PAIR_ARRAY";
		name2string_[RNA_BASE_AXIS_ARRAY] = "RNA_BASE_AXIS_ARRAY";
		name2string_[RNA_BASE_STAGGER_ARRAY] = "RNA_BASE_STAGGER_ARRAY";
	}

	/////////////////////////////////////////////////////////////////////////////
	// static function for getting score names
	std::string
	Score_data::name2string(
		Score_name name
	)
	{
		initialize_name2string();
		std::map< Score_name, std::string >::const_iterator it
			( name2string_.find( name ) );
		if ( it != name2string_.end() ) {
			return it->second;
		} else {
			return "NULL";
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// static function for getting score names
	bool
	Score_data::string2name(
			std::string key, Score_name & name
	)
	{
		initialize_name2string();

		bool found( false );
		for ( std::map< Score_name, std::string >::const_iterator
						it = name2string_.begin(); it != name2string_.end(); ++it ) {
			if ( it->second == key ) {
				name=it->first;
				found = true;
				break ;
			}
		}
		if ( !found ) {
			std::cout<<" key= "<<key <<" does not match with Score_name"<<std::endl;
		}
		return found ;
	}


	/////////////////////////////////////////////////////////////////////////////
	// Score_weight_map functions
	/////////////////////////////////////////////////////////////////////////////

	Score_weight_map::Score_weight_map()
	{
	}

	/////////////////////////////////////////////////////////////////////////////
	Score_weight_map::Score_weight_map(
		Scoring_Function score_fxn
	)
	{
		set_weights( score_fxn );
	}


	/////////////////////////////////////////////////////////////////////////////
	//lin specify score weight set from file
	//
	// file format:
	//
	// reading begins with "SCORE_WEIGHT_MAP::BEGIN" and end with "SCORE_WEIGHT_MAP::END"
	//
	// SCORE_WEIGHT_MAP::BEGIN
	// # specify rosetta score function, default is zero weight set
	//         scorefxn: score12
	// # assign value to each weight, whose name is defined in "score_name.h"
	// #                     weight_name  value
	//         score_weight: KIN_1D_CST   1.0
	//         score_weight: KIN_3D_CST   1.0
	//         score_weight: ATOMPAIR_CST 1.0
	// SCORE_WEIGHT_MAP::END
	void
	Score_weight_map::read_weight_file( std::string const & filename_in ) {

    using namespace files_paths;

    std::ifstream file;
    std::istringstream line_stream;
    std::string filename;

    if ( filename_in == "dummy" ) {
      std::cout << "Score_weight_map::read_weight_file - please specify the weights file" << std::endl;
			exit( EXIT_FAILURE );
		} else if( filename_in == "default" ) {
			filename = start_path + "score_weights_set";
    } else {
			filename = start_path + filename_in;
		}
    file.clear();
		file.open( filename.c_str() );

    if ( !file ) {
      std::cout << "Score_weight_map::read_weight_file - unable to open file " << filename  << std::endl;
			exit( EXIT_FAILURE );
    } else {
      std::cout << "Score_weight_map::read_weight_file "<< filename  << std::endl;
    }

		//weight_map.clear();
		scorefxns::fa_scorefxn=true;

    std::string line, key(""), tag(""), weight_tag, scorefxn_name;
		float  value;
    bool start( false);

    while(file) {
      key = "";
      getline(file, line);
      line_stream.clear();
      line_stream.str(line);
      line_stream.seekg( std::ios_base::beg );
      line_stream >> key;

      if( key == "SCORE_WEIGHT_MAP::BEGIN" ) {
				start=true;
      }
      if( key == "SCORE_WEIGHT_MAP::END" ) {
				start=false;
				return;
      }

      if( !start ) continue;

      //lin read template information
      if( key == "scorefxn:" ) {
				Scoring_Function score_fxn;
				line_stream >> scorefxn_name;
				score_fxn = score_function_from_string( true, scorefxn_name);
				setup_score_weight_map( *this, score_fxn );
			} else if( key == "score_weight:" ) {
				line_stream >> tag;
				Score_name name;
				if( Score_data::string2name( tag, name ) ){
					line_stream >> value ;
          std::cout<<"set "<<Score_data::name2string(name)<<" from "
									 <<get_weight( name )<<" to "<<value<<std::endl;
					set_weight( name, value );
				}
			}
		}
		return;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Score_weight_map::show() const
	{
		std::cout << "Score_weight_map::show()";
		for ( const_iterator it=weight_map.begin(), it_end=weight_map.end();
					it != it_end; ++it ) {
			Score_name const & name( it->first );
			float const setting( it->second );
			std::cout << ' ' << Score_data::name2string(name) << name << ' ' <<
				setting;
		}
		std::cout << std::endl;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Score_weight_map::set_weights(
		Scoring_Function score_fxn
	)
	{
		setup_score_weight_map( *this, score_fxn );
	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Score_weight_map::get_weight(
		const Score_name name,
		float const default_value // = 0.0
	) const {
		float weight;
		const_iterator it ( weight_map.find( name ) );
		if ( it == weight_map.end() ) { // not in map: set to 0.0
			weight = default_value;
		} else {
			weight = it->second;
		}
		return weight;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Score_weight_map::set_weight(
		const Score_name name,
		float const weight
	)
	{
		weight_map[name] = weight;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Score_weight_map::clear()
	{
		weight_map.clear();
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Score_weight_map::has_1D_weight(
		const Score_name name,
    int & size   // WARNING: function has side effect of setting size.
	) const
	{
		Weight_map_1D::const_iterator it ( weight_map_1D.find( name ) );
		if ( it == weight_map_1D.end() ) {
			size = 0;
			return false;
		} else {
			size = it->second.size1();
			return true;
		}
	}

	///////////////////////////////////////////////////////////////////
	const FArray1D_float &
	Score_weight_map::get_1D_weight(
		const Score_name name
	) const
	{
		Weight_map_1D::const_iterator it( weight_map_1D.find( name ) );
		if ( it == weight_map_1D.end() ) {
			std::cout << "Score_weight_map::get_weight_1D: weight does not exist "
								<< name << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return it->second;
	}

	///////////////////////////////////////////////////////////////////
	void
	Score_weight_map::set_1D_weight(
		const Score_name name,
		const FArray1D_float & weight_1D
	)
	{
		Weight_map_1D::const_iterator it( weight_map_1D.find( name ) );

		weight_map_1D.erase( name );
		weight_map_1D.insert( std::make_pair( name, weight_1D ) );

	}

} // namespace pose_ns
