// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13913 $
//  $Date: 2007-04-03 00:59:09 -0400 (Tue, 03 Apr 2007) $
//  $Author: pbradley $

#ifndef INCLUDED_score_data
#define INCLUDED_score_data


// Rosetta Headers
#include "score_name.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// C++ Headers
#include <cstdlib>
#include <cassert>
#include <map> // Score_map
#include <iostream> // std::cout
#include <utility> // make_pair
//#include <iosfwd>
#include <string>

typedef  float (*Scoring_Function)();

//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80
// these comments are currently out of date
//
// a class for storing score data about a structure, and keeping track of what
// data is up-to-date.
//
// currently, the data are 0,1,and 2D FArrays of floats (0d are just floats)
//
// the different data names are given in the enum "Score_name" below
//
// a Score_data object has the following interface:
//
// get_0D( score_name, bool & ok ), set_0D( score_name, bool & ok )
// get_1D( score_name, bool & ok ), set_1D( score_name, bool & ok )
// get_2D( score_name, bool & ok ), set_2D( score_name, bool & ok )
//
// after_refolding()
// after_scoring()
//
// the get_* methods return const references to the data, set_* methods
// return non-const references.
//
// internally, scores can have 4 states: NULL,BAD,OK,GOOD
//
// NULL means that no space has been allocated for that type of score
// BAD means that the score is not up-to-date
// OK means that the score is correct modulo pair_moved,res_moved
// GOOD means that the score is completely correct
//
// after we refold scores that were GOOD become OK
// during a score calculation, any score whose set*() method
// is called is moved from OK->GOOD
// (BAD stays BAD, NULL goes to BAD and space is allocated)
//
// after we score, scores that are just OK go to BAD b/c they
// were not updated. This is necessary because after a score
// calculation we forget about the changes just made
// pair_moved, etc become false.
//
// routines that access stored score_data should catch the
// value of "ok" after the get or set call and not depend on
// the data if ok == false. In practice this probably will not be
// happening, so maybe we can use assert statements... I could
// imagine this happening when we change score-functions. In this
// situation maybe we want to do a new_pose type thingy.
//
// somewhat contrary to the current rosetta logic: arrays are modified
// in place during each score calculation. When we reject a move we
// copy the old data from best. Some of this copying we would do
// next time around anyhow under the current system. But I have to
// make sure that this copying doesnt become a rate limiting.
//
// One thing to keep in mind: it's necessary that all the pair scores
// have the same dimension; some routines may be making this
// assumption in setting up FArray linear indexing. This should be
// guaranteed by set_nres()
//
// Added 3D arrays -- need to specify the three dimensions of the arrays,
//  which are *not* assumed to be nres, nres, nres.
//
////////////////////////////////////////////////////////////////////////////////80
////////////////////////////////////////////////////////////////////////////////80





// this is a bit of a hack, enum as ints are used as std::MAP keys


namespace pose_ns
{
//  	enum Score_name {
//  		// 0D scores
//  		SCORE,
//  		VDW,
//  		ENV,
//  		PAIR,
//  		CB,
//  		SHEET,
//  		SSPAIR, // avoid conflict with ObjexxFCL::SS
//  		HS,
//  		RSIGMA,
//  		RAMACHANDRAN,
//  		RG,
//  		CO,
//  		CHAINBREAK,
//  		FA_ATR, // fullatom scores...
//  		FA_REP,
//  		FA_SOL,
//  		FA_PAIR,
//  		FA_REF,
//  		FA_DUN,
//  		FA_PROB,
//  		HB_SRBB,
//  		HB_LRBB,
//  		HB_SC,
//  		FA_INTRA,
//  		// diagnostics
//  		RMSD,
//  		RMSD_G,
//  		// not really scores, but useful to set inside the
//  		// Weight_map:
//  		SS_LOWSTRAND,
//  		SS_CUTOFF,
//  		PC_MODE,
//  		CHAINBREAK_OVERLAP,
//  		FA_SCOREFXN,
//  		// 1D scores
//  		DUNENERGY,
//  		PROBENERGY,
//  		INTRARESENERGY,
//  		RESENERGY,
//  		// 2D scores
//  		ATR_PAIR,
//  		REP_PAIR,
//  		SOL_PAIR,
//  		CENDIST,
//  		VDW_PAIR // 2D
//  	};

	enum Score_state {
		BAD, OK, GOOD
	};

	class Stored_score {
	private:
		int const dimension; // is set in the constructor
		mutable Score_state state; // check_size(...)const; sets state to BAD
		int size; // ( = total_residue )
		float data0;
		FArray1D_float * data1;
		FArray2D_float * data2;
		FArray3D_float * data3;
	public:
		// constructor allocates memory, sets dimension
		Stored_score( int const dim_in, int const size_in );
		// constructor allocates memory, sets dimension for 3D arrays.
		Stored_score( int const dim_in, int const size1, int const size2, int const size3 );
		// copy-constructor
		Stored_score( Stored_score const & src );
		// operator=
		Stored_score & operator =( Stored_score const & src );
		// destructor
		~Stored_score();

		// get dimension
		int get_dimension() const { return dimension; };

		// get/set state
		inline Score_state const & get_state() const { return state; }
		//inline Score_state & set_state() { return state; }
		inline void set_state( Score_state const state_in ) const { state = state_in; }

		// resize to new_size (if necessary...will set state to BAD )
		inline void set_size( int const new_size );
		inline void set_size( int const new_size1, int const new_size2, int const new_size3 ); //For 3D arrays!?
		// if size != new_size set state to BAD:
		inline void check_size( int const new_size) const;
		// 0
		inline float get0() const;
		inline float & set0();
		// 1
		inline FArray1D_float const & get1() const;
		inline FArray1D_float & set1();
		// 2
		inline FArray2D_float const & get2() const;
		inline FArray2D_float & set2();
		// 3
		inline FArray3D_float const & get3() const;
		inline FArray3D_float & set3();
	};

	// define get,set for Stored_score
	// DANGER DANGER DANGER: calls to set?() change the STATE to good
	// these calls are made by pose.set_score_?D
	// 0
	inline float Stored_score::get0() const
	{ assert( dimension == 0 ); return data0; }
	inline float & Stored_score::set0()
	{ assert( dimension == 0 ); state = GOOD; return data0; }
	// 1
	inline FArray1D_float const & Stored_score::get1() const
	{ assert( dimension == 1 ); return *data1; }
	inline FArray1D_float & Stored_score::set1()
	{ assert( dimension == 1 ); state = GOOD; return *data1; }
	// 2
	inline FArray2D_float const & Stored_score::get2() const
	{ assert( dimension == 2 ); return *data2; }
	inline FArray2D_float & Stored_score::set2()
	{ assert( dimension == 2 ); state = GOOD; return *data2; }
	// 3
	inline FArray3D_float const & Stored_score::get3() const
	{ assert( dimension == 3 ); return *data3; }
	inline FArray3D_float & Stored_score::set3()
	{ assert( dimension == 3 ); state = GOOD; return *data3; }

	// check the size:
	inline void Stored_score::check_size( int const new_size ) const {
		if ( size != new_size ) {
			std::cout << "Stored_score::check_size: size_mismatch: " <<
				size << " , " << new_size << std::endl;
			state = BAD;
		}
	}
	// set the size:
	inline void Stored_score::set_size( int const new_size ) {
		if ( size != new_size ) {
			// have to redimension
			if ( dimension == 1 ) {
				data1->dimension( new_size );
			} else if ( dimension == 2 ) {
				data2->dimension( new_size, new_size );
			} else if ( dimension == 3 ) {
				data3->dimension( new_size, new_size, new_size );
			}
			state = BAD; // resizing kills the data
			size = new_size;
		}
	}

	inline void Stored_score::set_size( int const new_size1, int const new_size2, int const new_size3 ) {
		if ( size != new_size1 ) {
			// have to redimension
			if ( dimension == 1 ) {
				data1->dimension( new_size1 );
			} else if ( dimension == 2 ) {
				data2->dimension( new_size1, new_size2 );
			} else if ( dimension == 3 ) {
				data3->dimension( new_size1, new_size2, new_size3 );
			}
			state = BAD; // resizing kills the data
			size = new_size1;
		}
	}


	//////////////////////////////////////////////////////////////////////////
	typedef std::map< Score_name, Stored_score * > Score_map;
	class Score_data {
	private:
		Score_map score_map;
		// private copy-constructor ==> no pass-by-value
		Score_data( Score_data & ) {}
		// switch data from one state to another
		void switch_state( Score_state const src, Score_state const dest );

		// shared by all instances:
		static std::map< Score_name, std::string > name2string_;
		static bool name2string_init;
		static void initialize_name2string();

	public:
		Score_data();
		~Score_data();

		// check if we have a score name in the map:
		bool has_score( Score_name name ) const;
		// add score
		Score_map::iterator add_score( Score_name name,
																	 int const dimension,
																	 int const size );
		// add score
		Score_map::iterator add_score( Score_name name,
																	 int const dimension,
																	 int const size1,
																	 int const size2,
																	 int const size3 );
		// get ( is CONST method )
		Stored_score const & get_score( Score_name name,
																		int const dimension,
																		int const size ) const;
		// set
		Stored_score & set_score( Score_name name,
															int const dimension,
															int const size );

		// set
		Stored_score & set_score( Score_name name,
															int const dimension,
															int const size1,
															int const size2,
															int const size3 );

		// operator=
		Score_data & operator =( const Score_data & s );
		// called after refolding:
		void after_refolding();
		// called after scoring:
		void after_scoring();
		// score-output:
		void show_scores( std::ostream & os, int const mode ) const;
		// reset all data
		void clear()
		{
			for ( Score_map::iterator it = score_map.begin(), it_end = score_map.end(); it != it_end; ++it ) {
				delete it->second;
			}
			score_map.clear();
		}

		// hack to force updating, eg of cendist
		void
		set_score_state_OK_if_GOOD(
			const Score_name name
		);

		static
		std::string
		name2string(
			Score_name name
		);

		static
		bool  string2name(
			std::string key, Score_name & name
		);

	};

	//////////// inline definitions ////////////


	inline bool Score_data::has_score( Score_name name ) const
	{
		return score_map.find( name ) != score_map.end();
	}

	// add a new score to the map
	inline Score_map::iterator Score_data::add_score( Score_name name,
																										int const dimension,
																										int const size )
	{
		score_map.insert( std::make_pair( name, new Stored_score( dimension, size ) ) );
		Score_map::iterator it( score_map.find( name ) );
		assert( it != score_map.end() );
		return it;
	}

	// add a new score to the map
	inline Score_map::iterator Score_data::add_score( Score_name name,
																										int const dimension,
																										int const size1,
																										int const size2,
																										int const size3)
	{
		score_map.insert( std::make_pair( name, new Stored_score( dimension, size1, size2, size3 ) ) );
		Score_map::iterator it( score_map.find( name ) );
		assert( it != score_map.end() );
		return it;
	}

	// get, will DIE if name isnt in the map
	inline Stored_score const & Score_data::get_score( Score_name name,
																										 int const dimension,
																										 int const size ) const
	{
		Score_map::const_iterator it ( score_map.find( name ) );
		if ( it == score_map.end() ) {
			// DIE
			std::cout << "Score_data::get_score( " << name << " , " << dimension <<
				" , " << size << " ): score doesnt exist in map" << std::endl;
			assert( false );
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		// check_size() will set state->BAD in the event of a mismatch,
		// but it can't redimension since this is a CONST method
		it->second->check_size( size );
		return *(it->second);
	}
	// set
	inline Stored_score & Score_data::set_score( Score_name name,
																							 int const dimension,
																							 int const size ) {
		Score_map::iterator it ( score_map.find( name ) );
		if ( it == score_map.end() ) it = add_score ( name, dimension, size );
		it->second->set_size( size );
		return *(it->second);
	}

	inline Stored_score & Score_data::set_score( Score_name name,
																							 int const dimension,
																							 int const size1,
																							 int const size2,
																							 int const size3 ) {
		Score_map::iterator it ( score_map.find( name ) );
		if ( it == score_map.end() ) it = add_score ( name, dimension, size1, size2, size3 );
		it->second->set_size( size1, size2, size3 );
		return *(it->second);
	}

	///////////////// for setting score weights on the fly /////////
	// goes with the scoring function:
	//
	//
	class Score_weight_map {
	public:
		typedef std::map< Score_name, float > Weight_map;
		typedef std::map< Score_name, FArray1D_float > Weight_map_1D;
		typedef Weight_map::iterator iterator;
		typedef Weight_map::const_iterator const_iterator;
	private:
		Weight_map weight_map;
		Weight_map_1D weight_map_1D;
	public:
		// iterate over the weights
		inline
		Weight_map::const_iterator
		begin() const { return weight_map.begin(); }

		// iterate over the weights
		inline
		Weight_map::const_iterator
		end() const { return weight_map.end(); }

		// empty weights:
		Score_weight_map();

		// take weights from this Rosetta scoring function
		Score_weight_map( Scoring_Function score_fxn );

		// lin take weights from file
		void read_weight_file( std::string const & filename_in );

		// lin output weights
		void show() const;

		// set weights from rosetta scorefxn
		void
		set_weights( Scoring_Function score_fxn );

		float
		get_weight(
			const Score_name name,
			float const default_value = 0.0
		) const;

		void
		set_weight(
			const Score_name name,
			float const weight
		);

		void
		clear();

		bool
		has_1D_weight(
			const Score_name name,
			int & size
		) const;

		const FArray1D_float &
		get_1D_weight(
			const Score_name name
		) const;

		void
		set_1D_weight(
			const Score_name name,
			const FArray1D_float & weight_1D
		);

	};

}

#endif
