// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20369 $
//  $Date: 2008-02-13 23:38:22 -0500 (Wed, 13 Feb 2008) $
//  $Author: bblum $


// Rosetta Headers
#include "silent_input.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "barcode_classes.h"
#include "files_paths.h"
#include "force_barcode.h" // to output barcode
#include "jumping_util.h"
#include "orient_rms.h"
#include "param.h"
#include "param_aa.h"
#include "param_torsion.h"
#include "pack.h" // get_rot_coord
#include "pack_geom_inline.h"
#include "pdb.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rna.h" //to setup RNA atomtree.
#include "read_aaproperties.h"
#include "shotgun.h"// to output maxsub to server models.
#include "status.h" // to output filter

#include "enzyme.h" //For CAPRI hack
#include "docking.h" //For CAPRI hack

// Numeric Headers
#include <numeric/xyz.functions.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <utility>
#include <vector>
#include <list>
#include <string>
#include <map>
#include <sstream>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>
#include <utility/file/file_sys_util.hh>

//using namespace silent_io;

inline
bool
is_HN_or_OXT_atom(
									int const atomno,
									int const aa,
									int const aav
									)
{
	using namespace aaproperties_pack;

	int const HN_pos( HNpos(aa,aav) );
	bool const is_proline( aa == param_aa::aa_pro );

	if ( variant_type( aav_Nterm, aa, aav ) ) {
		int const num_of_HN( is_proline  ? 2 : 3 );
		if ( atomno >= HN_pos && atomno < HN_pos + num_of_HN ) return true;
	}

	if ( variant_type( aav_Cterm, aa, aav ) &&
			 atomno == LookupByName( aa, aav, " OXT") ) return true;

	return !is_proline && atomno == HN_pos;
}


// definitions for some class methods:
namespace silent_io {

	/////////////////////////////////////////////////////////////////////////////
	// Rotamer_template constructor
	Rotamer_template::Rotamer_template():
		aa_(0),
		aav_(0),
		natoms_(0)
	{}

	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////
	// Rotamer_template constructor
	Rotamer_template::Rotamer_template(
		int const aa,
		int const aav,
		FArray2Da_float xyz_in
	):
		aa_( aa ),
		aav_( aav )
	{
		using namespace aaproperties_pack;
		natoms_ = natoms(aa,aav);
		xyz_in.dimension( 3, natoms_ );

		xyz_.dimension( 3, natoms_ );

		// from pack.cc: get_rot_coord
		FArray2D_float mat( 3, 3 );
		FArray1D_float vec( 3 );
		int const chi_required_size1 = chi_required.size1();

		//////////////////////////////////////////
		// reorient to standard backbone location:
		// C-alpha at 0.0,0.0,0.0
		// N along 1,0,0
		numeric::xyzMatrix_double p, m;
		FArray1D_double v(3);

		get_ncac_full_coord(xyz_in,p);
		get_coordinate_system(p,m);

		for ( int j=1; j<= natoms_; ++j ) {
			numeric::xyzVector_double old_pos( &xyz_in(1,j) );
			numeric::xyzVector_double new_pos = m.transposed() * ( old_pos - p.col(2) ); //p.col(2) is CA
			for ( int k=1; k<=3; ++k ) xyz_(k,j) = new_pos(k);
		// 	for ( int k=1; k<= 3; ++k ) {
// 				v(k) = xyz_in(k,j) - xyz_in(k,2); // translate C-alpha to origin
// 			}
// 			for ( int k=1; k<= 3; ++k ) {
// 				// non-standard matrix indexing, ala CEMS
// 				xyz_(k,j) = v(1) * m(1,k) + v(2) * m(2,k) + v(3) * m(3,k);
// 			}
		}

		//////////////////////////
		// set all chi angles to 0
		int const nchi_aa( nchi(aa_,aav_) );
		float const target_chi( 0.0 );
		for ( int ch = 1; ch <= param::MAX_CHI; ++ch ) {
			//bk change chi angle if it exists
			if ( nchi_aa >= ch ) {
				int l = chi_atoms.index(1,ch,aa_,aav_);
				int c1 = chi_atoms[  l ]; // chi_atoms(1,ch,aan,aav);
				int c2 = chi_atoms[ ++l ]; // chi_atoms(2,ch,aan,aav);
				int c3 = chi_atoms[ ++l ]; // chi_atoms(3,ch,aan,aav);
				int c4 = chi_atoms[ ++l ]; // chi_atoms(4,ch,aan,aav);
				//** determine initial chi angle
				float ichi;
				dihedral_bk(xyz_(1,c1),xyz_(1,c2),xyz_(1,c3),xyz_(1,c4),ichi);
				//** rotate angle by new-initial degrees
				float rot = ( target_chi - ichi ) * (3.141592654/180.0);
				//**   generate rotation vector and matrix
				getrot_bk(xyz_(1,c2), xyz_(1,c3), rot, mat, vec);
				//** move atoms
				for ( int i = 1, lcr = chi_required.index(ch,i,aa_,aav_);
							i <= natoms_; ++i, lcr+=chi_required_size1 ) {
					if ( chi_required[ lcr ] ) { // chi_required(ch,i,aan,aav)
						move_bk(xyz_(1,i),mat,vec);
					}
				}
			}
		}
	} // constructor for Rotamer_template

	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////
	// taken from pack.cc get_rot_coord

	void
	Rotamer_template::insert(
		FArray2Da_float xyz_inout,
		FArray1Da_float chi,
		bool const rebuild_pro // = true
	) const
	{
		using namespace aaproperties_pack;
		xyz_inout.dimension(3,param::MAX_ATOM()());
		chi.dimension(param::MAX_CHI);

		// from pack.cc: get_rot_coord
		FArray2D_float mat( 3, 3 );
		FArray1D_float vec( 3 );
		int const chi_required_size1 = chi_required.size1();

		// copy our coords
		FArray2D_float coor( xyz_ );

		int & nchi_aa( nchi(aa_,aav_) );
		for ( int ch = 1; ch <= param::MAX_CHI; ++ch ) {
			//bk change chi angle if it exists
			if ( nchi_aa >= ch ) {
				float ichi;
				int l = chi_atoms.index(1,ch,aa_,aav_);
				int c1 = chi_atoms[  l ]; // chi_atoms(1,ch,aa_,aav_);
				int c2 = chi_atoms[ ++l ]; // chi_atoms(2,ch,aa_,aav_);
				int c3 = chi_atoms[ ++l ]; // chi_atoms(3,ch,aa_,aav_);
				int c4 = chi_atoms[ ++l ]; // chi_atoms(4,ch,aa_,aav_);
				//** determine initial chi angle
				dihedral_bk(coor(1,c1),coor(1,c2),coor(1,c3),coor(1,c4),ichi);
				//** rotate angle by new-initial degrees
				float rot = ( chi(ch) - ichi ) * (3.141592654/180.0);
				//**   generate rotation vector and matrix
				getrot_bk(coor(1,c2), coor(1,c3), rot, mat, vec);
				//** move atoms
				for ( int i = 1, lcr = chi_required.index(ch,i,aa_,aav_);
							i <= natoms_; ++i, lcr+=chi_required_size1 ) {
					if ( chi_required[ lcr ] ) { // chi_required(ch,i,aa_,aav_)
						move_bk(coor(1,i),mat,vec);
					}
				}
			}
		}

		//** superimpose the rotamer on to the backbone
		//**   find the matrix/vector that lines up the NH and CA atoms
		//**   and apply this operation to all the atoms in the rotamer

		lineup_bk(coor(1,2),coor(1,1),xyz_inout(1,2),xyz_inout(1,1),mat,vec);
		for ( int i = 1; i <= natoms_; ++i ) {
			move_bk(coor(1,i),mat,vec);
		}

		//**   find the matrix/vector that lines up the C = O carbons
		//**   and apply this operation to the rotamer
		align_bk(coor(1,1),coor(1,2),coor(1,3),xyz_inout(1,3),mat(1,1),vec(1));
		for ( int i = 3; i <= natoms_; ++i ) {
			move_bk(coor(1,i),mat,vec);
		}

		// fill in the coords in rescoord:
		//**  store coordinates in rotcoord
		//int const HNpos_aa = HNpos(aa_,aav_);
		for ( int j = 1; j <= natoms_; ++j ) {
			std::string const & atom_name_j( atom_name(j,aa_,aav_) );
			if ( j <= 4 ) {
				if ( j==2 ) {
					float ca_dev(0.0);
					for ( int k=1; k<= 3; ++k ) {
						ca_dev += std::abs( xyz_inout(k,j) - coor(k,j) );
					}
					if ( ca_dev > 0.001 ) {
						std::cout << "ca_dev= " << ca_dev << std::endl;
					}
				}
				continue;
			} else if ( is_HN_or_OXT_atom(j,aa_,aav_) ) {
				continue;
			} else if ( rebuild_pro && aa_ == 13 && ( atom_name_j == "2HD" ||
																								atom_name_j == "3HD" ||
																								atom_name_j == "2HG" ||
																								atom_name_j == "3HG" ) ) {
				// proline!
				// I guess the hydrogens must come after all the heavyatoms?
				place_atom(aa_,aav_,j,xyz_inout);
			} else if ( atom_name_j == "1WN " ||
									atom_name_j == "1WO " ||
									atom_name_j == "2WO " ) {
				std::cout << "water building not supported in silent-io yet" <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			} else {
				for ( int k = 1; k <= 3; ++k ) {
					xyz_inout(k,j) = coor(k,j);
				}
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////
	// Rotamer_template constructor from stream
	Rotamer_template::Rotamer_template(
		std::istream & is
	)
	{
		is >> aa_ >> aav_;
		natoms_ = aaproperties_pack::natoms(aa_,aav_);
		xyz_.dimension( 3, natoms_ );

		for ( int i=1; i<= natoms_; ++i ) {
			int ii;
			is >> ii;
			if ( ii != i ) {
				is.setstate( std::ios::failbit );
				break;
			}
			for ( int k=1; k<= 3; ++k ) {
				is >> xyz_(k,i);
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////
	// operator =
	Rotamer_template & Rotamer_template::operator=(
		Rotamer_template const & t2
	)
	{
		aa_ = t2.aa_;
		aav_ = t2.aav_;
		natoms_ = t2.natoms_;

		xyz_.dimension(3,natoms_);
		xyz_ = t2.xyz_;
		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
	float Rotamer_template::distance(
		Rotamer_template const & t2
	) const
	{
		assert( t2.aa() == aa_ && t2.aav() == aav_ );
		//int const HN_pos( aaproperties_pack::HNpos( aa_, aav_ ) );
		float dev(0.0);
		for ( int j=5; j<= natoms_; ++j ) {
			if ( is_HN_or_OXT_atom( j, aa_, aav_ ) ) continue;
			for ( int k=1; k<= 3; ++k  ){
				dev += std::abs( xyz_(k,j) - t2.xyz(k,j) );
			}
		}
		return dev;
	}

	/////////////////////////////////////////////////////////////////////////////
	//////////////////
	// stream inserter and extractor for rotamer_template
	std::ostream& operator <<(
		std::ostream& os,
		Rotamer_template const & t
	)
	{
		os << t.aa() << ' ' << t.aav() << ' ';
		for ( int j=1; j<= t.get_natoms(); ++j ) {
			os << j << ' ';
			for ( int k=1; k<= 3; ++k ) {
				os << t.xyz(k,j) << ' ';
			}
		}
		return os;
	}

//  	std::istream& operator>>(
//  		std::istream & is,
//  		Rotamer_template & t
//  	)
//  	{
//  		int aa,aav;
//  		is >> aa >> aav;
//  		int const natoms( aaproperties_pack::natoms(aa,aav));
//  		t.aa_ = aa;
//  		t.aav_ = aav;
//  		t.natoms_ = natoms;
//  		t.xyz_.dimension(3,natoms);

//  		for ( int j=1; j<= natoms << ++j ) {
//  			int jj;
//  			is >> jj;
//  			if ( jj != j ) {
//  				is.setstate( std::ios::failbit );
//  				break;
//  			}
//  			for ( int k=1; k<= 3; ++k ) {
//  				is >> t.xyz_(k,j);
//  			}
//  		}
//  		return is;
//  	}

	/////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////

	void
	Silent_structure::dimension(
		int const nres_in
	)
	{
		// does not handle bonds
		nres = nres_in;
		res.dimension( nres );
		res_variant.dimension( nres );
		res_variant = 1; // default
		secstruct.dimension( nres );
		phi.dimension( nres );
		psi.dimension( nres );
		omega.dimension( nres );
		coords.dimension( 3, nres );
		name.dimension( nres );
		name = "   "; //blank by default.
		if ( fullatom ) {
			chi.dimension( param::MAX_CHI, nres );
			torsion.dimension( param::MAX_TORSIONS, nres );
			rot_template.dimension( nres );
			rot_template = 0;
			full_coord.dimension( 3, param::MAX_ATOM(), nres );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Silent_structure::set_sequence(
		std::string const & sequence
	)
	{
		if ( int(sequence.size()) != nres ) {
			std::cout << "Silent_structure::set_sequence: sequence-length mismatch " <<
				sequence.size() << ' ' << nres << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		for ( int i=1; i<= nres; ++i ) {
			res(i) = aa2int( sequence[i-1] );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Silent_structure &
	Silent_structure::operator= (
		Silent_structure const & src
	)
	{
		fullatom = src.fullatom;
		dimension( src.nres );
		res = src.res;
		res_variant = src.res_variant;
		secstruct = src.secstruct;
		phi = src.phi;
		psi = src.psi;
		omega = src.omega;
		coords = src.coords;
		if ( fullatom ) {
			chi = src.chi;
			rot_template = src.rot_template;
		}
		if ( src.bonds.nres() ) {
			bonds = src.bonds;
		} else if ( bonds.nres() ) {
			bonds.clear();
		}

		fold_tree = src.fold_tree;
		int const num_jump( fold_tree.get_num_jump() );
		if ( num_jump > 0 ) {
			jump_transform.dimension( num_jump );
			jump_transform = src.jump_transform;
		}
		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
	// simple constructor
	Silent_file_data::Silent_file_data(
		std::string const filename,
		bool const fullatom // = false
		):
		symm_info(0)
	{
		read_file( filename, fullatom );
	}

	/////////////////////////////////////////////////////////////////////////////
	Silent_file_data::Silent_file_data(
		std::string const filename,
		std::vector< std::string > const & desired_tags,
		bool const fullatom // = false
		):
		symm_info(0)
	{
		read_file( filename, desired_tags, fullatom );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Silent_file_data::score_filter(
		float const score_fraction
	)
	{
		// make a list of the scores
		std::list< float > score_list;
		for ( const_iterator it=structure_map_.begin(),
						it_end = structure_map_.end(); it != it_end; ++it ){
			score_list.push_back( it->second->total_score );
		}

		score_list.sort();

		int const ndecoys( score_list.size() );
		std::list< float >::const_iterator list_it( score_list.begin() );
		int const n( static_cast< int >( score_fraction * ndecoys ) );
		float const min_score( *list_it );
		for ( int i=0; i<n; ++i, ++list_it ) {
			assert( min_score <= *list_it );
		}

		float const max_score( *list_it );
		std::cout << "Silent_file_data::score_filter( " << score_fraction <<
			" ) ndecoys= " << ndecoys << " ncut= " << n << " cutoff_score= " <<
			max_score << std::endl;


		// compile a list of bad tags:
		std::vector< std::string > bad_tags;
		for ( iterator it=structure_map_.begin(),
						it_end = structure_map_.end(); it != it_end; ++it ){
			if ( it->second->total_score > max_score ) {
				bad_tags.push_back( it->first );
			}
		}


		// now start deleting
		for ( std::vector< std::string >::const_iterator tag=bad_tags.begin();
						tag != bad_tags.end(); ++tag ) {
			iterator it = structure_map_.find( *tag );
			if ( it == structure_map_.end() ) {
				std::cout << "whoah! it2! " << *tag << std::endl;
				continue;
			}
			assert( it->second->total_score > max_score );
			delete it->second;
			structure_map_.erase( it );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// get all tags
	std::vector< std::string >
	Silent_file_data::tags() const
	{
		std::vector< std::string > tag_list;
		for ( const_iterator it=structure_map_.begin(),
						it_end = structure_map_.end(); it != it_end; ++it ){
			tag_list.push_back( it->first );
		}
		return tag_list;
	}

	/////////////////////////////////////////////////////////////////////////////
	bool
	Silent_file_data::has_key(
		const std::string & tag
	) const
	{
		return (  structure_map_.count( tag ) > 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	// add a new structure to the map
	void
	Silent_file_data::new_structure(
		const std::string & tag_in,
		Silent_structure * new_data
	)
	{
		// check for duplicate tag
		std::string tag( tag_in );
		if ( structure_map_.count( tag ) != 0 ) {
			int count(0);
			while ( structure_map_.count( tag ) ) {
				++count;
				tag = tag_in+"_r"+string_of(count);
			}
			std::cout << tag << std::endl;
			// error msg
// 			std::cout << "duplicate tag: " << tag <<
// 				" replacing first occurrence" << std::endl;
// 			// have to free the old data if we kill the pointer to it!
// 			delete structure_map_[ tag ];
		}
		new_data->parent = this;
		structure_map_[ tag ] = new_data;
	}

	/////////////////////////////////////////////////////////////////////////////
	// assumes that we have checked the tag!!
	Silent_structure const &
	Silent_file_data::get_structure(
		const std::string & tag
	) const
	{
		if ( structure_map_.count( tag ) == 0 ) {
			// err msg
			std::cout << "failed to find tag " << tag << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return *( structure_map_.find(tag)->second );
	}

	/////////////////////////////////////////////////////////////////////////////
	// clean up -- call delete on all pointers in structure_map!
	void
	Silent_file_data::clear_structure_map()
	{
		for ( std::map<std::string, Silent_structure*>::iterator
						it = structure_map_.begin(), it_end = structure_map_.end();
					it != it_end; it++) {
			delete it->second;
		}
		structure_map_.clear();
	}

	/////////////////////////////////////////////////////////////////////////////
	////////////////////////////
	// get nres from first decoy
	int
	Silent_file_data::nres() const
	{
		if ( structure_map_.size() == 0 ) {
			return 0;
		} else {
			return structure_map_.begin()->second->nres;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	////////////////////////////////
	// get sequence from first decoy
	std::string
	Silent_file_data::sequence() const
	{
		std::string sequence;
		if ( structure_map_.size() > 0 ) {
			Silent_structure const * const decoy( structure_map_.begin()->second );
			if ( symm_info ) {
				// symmetric case
				int const nres_monomer( symm_info->nres_monomer_bb() );
				int const N( symm_info->N_bb() );
				int const npseudo( symm_info->npseudo() );
				assert( nres_monomer == decoy->nres );
				for ( int j=1; j<= N; ++j ) {
					for ( int i=1; i<= nres_monomer; ++i ) {
						sequence = sequence + param_aa::aa_name1( decoy->res(i) );
					}
				}
				for ( int i=1; i<= npseudo; ++i ) {
					sequence = sequence + 'G';
				}
			} else {
				// non symmetric
				int const nres( decoy->nres );
				for ( int i=1; i<= nres; ++i ) {
					sequence = sequence + param_aa::aa_name1( decoy->res(i) );
				}
			}
		}
		return sequence;
	}

	/////////////////////////////////////////////////////////////////////////////
	////////////////////////////
	// handle non-ideal rotamers

	bool
	Silent_file_data::ideal_rotamers() const
	{
		return ( rotamer_template_map_.size() == 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	std::vector< Rotamer_template > const &
	Silent_file_data::rotamer_templates(
		int const seqpos
	) const
	{
		if (! rotamer_template_map_.count( seqpos ) ) {
			std::cout << "No entry in rotamer_template_map !! " << seqpos <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return rotamer_template_map_.find( seqpos )->second;
	}


	/////////////////////////////////////////////////////////////////////////////
	////////////////////////////////////
	//
	// file reader
	//
	bool
	Silent_file_data::read_file(
		std::string const filename,
		bool const fullatom, // = false
		bool const prepend_filename // = false
	)
	{
		std::vector< std::string > empty_tag_list;
		return read_file( filename, empty_tag_list, fullatom, prepend_filename );
	}


	// if desired_tags is empty -- read all tags
	bool
	Silent_file_data::read_file(
		std::string const filename,
		std::vector< std::string > const & desired_tags,
		bool const fullatom, // = false
		bool const prepend_filename // = false
	)
	{
		bool io_fail = true; // default value for the early return statements

		// clear old decoys
		//clear_structure_map();

		////////////////////////////////////////////////
		// check for a bonds file, if exists fill bonds_
		int nres_bonds(0);
		bool ideal( true );
		bonds_class::Bonds bonds;
		{
			utility::io::izstream data( (filename+".bonds").c_str() );
			if ( data.good() ) {
				ideal = false;
				bonds.read( data );
				nres_bonds = bonds.nres();
			}
			data.close();
		} // scope

		///////////////////////////////////////////////////
		// check for rotamer template file
		rotamer_template_map_.clear();
		{
			utility::io::izstream data( (filename+".rot_templates").c_str() );
			if ( data.good() ) {
				std::string line;
				while ( getline( data, line ) ) {
					std::istringstream line_stream( line );
					int seqpos, n;
					line_stream >> seqpos >> n;
					Rotamer_template t( line_stream );

					if ( line_stream.fail() ) {
						std::cout << "bad rot template line:  "<< line << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
					if ( !rotamer_template_map_.count( seqpos ) ){
						std::vector< Rotamer_template > v;
						rotamer_template_map_.insert( std::make_pair( seqpos, v ) );
					}

					if ( seqpos != int(rotamer_template_map_.size()) ) {
						std::cout << "bad format in rot template file: 1 " <<
							seqpos << ' ' <<  rotamer_template_map_.size() << ' ' <<
							line << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}

					std::vector< Rotamer_template > & v
						( rotamer_template_map_.find( seqpos )->second );

					v.push_back( t );
					// numbering for n starts at 0 (ie it's the actual index
					// into the std::vector at which the correct template is
					// located
					if ( n+1 != int( v.size() ) ) {
						std::cout << "bad format in rot template file: 2 " <<
							n << ' ' << v.size() << ' ' << line << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}
				}
			}
			data.close();
		}


		///////////////////////////////////////////////////////////
		// check for a full-pose file; if it exists, read full_pose
		// save torsions
		bool subset( false );
		Silent_structure decoy_full;
		int nres_full(0);
		std::string full_sequence;
		{
			std::string full_filename( filename+".full_torsions");
			utility::io::izstream data( full_filename.c_str() );
			if ( data.good() ) {
				subset = true;
				// sneaky: open another Silent_file_data object inside of this one!
				Silent_file_data full_data( full_filename, fullatom );
				if ( full_data.size() < 1 ) {
					std::cout << "failed to read full datafile: " <<
						full_filename << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				// make a copy of the decoy
				decoy_full = *(full_data.begin()->second);
				decoy_full.coords = 0.0; // flag non-subset positions
				nres_full = decoy_full.nres;
				full_sequence = full_data.sequence();
			}
			data.close();
 		} // scope

		////////////////////////////////////////////////////////////
		// open the silent file
		utility::io::izstream data( filename.c_str() );
		if ( !data ) {
			std::cout << "ERROR:: Unable to open silent_input file: " <<
				filename << std::endl;
			return io_fail;
		}

		// See if barcode flavors should be included
		bool read_barcode(false);
		bool has_flavor(false);
		int barcode_mode(0);
		std::string barcode_file;
		read_barcode = truefalseoption("barcode_input");
		if(read_barcode)
			stringafteroption("barcode_input","junk.txt",barcode_file);

		// read SEQUENCE,SCORE-header lines
		int nres(0);
		std::string sequence;
		{
			// SEQUENCE line:
			std::string line,tag;
			getline( data, line );
			std::istringstream line_stream( line );
			line_stream >> tag >> sequence;

			if ( line_stream.fail() || tag != "SEQUENCE:" ) {
				std::cout << "bad format in first two lines of silent file\n";
				std::cout << filename << std::endl;
				return io_fail;
			}

			line_stream >> tag;
			if ( !line_stream.fail() && tag == "SYMM_INFO" ) {
				// parse symm_info
				pose_ns::Symmetry_info s2;
				std::istringstream line_stream2( line );
				line_stream2 >> tag >> sequence >> s2;
				// symm_info is a member of Silent_file_data
				if ( symm_info ) {
					if ( *symm_info != s2 ) {
						std::cout << "Silent_file_data::read_file: Symm_info mismatch!"<<
							std::endl;
						std::exit( EXIT_FAILURE );
					}
				} else {
					symm_info = new pose_ns::Symmetry_info( s2 );
				}
			}

			// read SCORE
			getline(data,line);

			// debug file format:
			if ( line.substr(0,7) != "SCORE: " ) {
				std::cout << "bad format in first two lines of silent file\n";
				std::cout << filename << std::endl;
				return io_fail;
			}

			if(read_barcode) {
				std::istringstream line_stream3(line);
				std::list<std::string> fields;
				std::string field;
				line_stream3 >> field;
				while(!line_stream3.fail()) {
					fields.push_back(field);
					line_stream3 >> field;
				}
				std::list<std::string>::reverse_iterator it = fields.rbegin();
				it++; // second-to-last entry
				has_flavor = (*it == "FLAVOR");
				if(!has_flavor)
					std::cout << "No flavor field in .out file." << std::endl;
			}

			if ( this->size() && !symm_info && this->sequence() != sequence ) {
				std::cout << "Silent_file_data::read_file -- sequence mismatch in " <<
					"new file\nold_seq: " << this->sequence() << "\nnew_seq: " <<
					sequence << std::endl;
				std::exit( EXIT_FAILURE );
			}
			nres = sequence.size();
		}

		// debug nres
		if ( ideal && subset && nres > nres_full ||
				!ideal && ( subset && nres_bonds != nres_full ||
					!subset && nres_bonds != nres ) ) {
			std::cout << "nres mismatch: " << subset << ' ' << ideal << ' ' <<
				nres_bonds << ' ' << nres_full << ' ' << nres << std::endl;
			return io_fail;
		}

		// read barcode file
		barcode_classes::feature_set barcodes;
		if(read_barcode)
			barcodes.read_file(barcode_file, nres, barcode_mode, false);

		// start looping over the structures
		std::string line;
		//std::istringstream line_stream;
		int tag_counter = 0;

		Silent_structure* tmp_data(0);
		while ( getline(data,line) ) {

			if ( line.substr(0,7) != "SCORE: " ) continue;

			// DANGER DANGER DANGER -- call to "new"
			// who will call delete ??
			// if we add this to big_map, he is responsible, handled by his
			// cleanup routine
			// otherwise, we call delete below
			//  the structure we're reading in
			if ( tmp_data ) {
				delete tmp_data;
				tmp_data = 0;
			}

			assert( !symm_info ||
							symm_info->nres_monomer_chi() >= symm_info->nres_monomer_bb() );

			int const nres_torsions
				( symm_info ? symm_info->nres_monomer_chi() : nres );
			tmp_data = new Silent_structure ( nres_torsions, fullatom );
			if ( subset ) {
				*tmp_data = decoy_full;
			}

			///////////////////////////////////////////////////
			// get the decoy tag from the end of the score-line
			// there must be a better way to get the last field!
			//
			// also parse the score
			//
			std::string decoy_tag;
			float total_score(0.0);
			{
				std::string tag;
				{ // get total_score
					std::istringstream l( line );
					l >> tag >> total_score;
					if ( l.fail() ) {
						total_score = 0.0;
					}
				}
				{ // get decoy_tag
					std::istringstream l( line );
					while ( true ) {
						l >> tag;
						if ( l.fail() ) {
							break;
						}
						decoy_tag = tag;
					}
				}
			}

			//std::cout << decoy_tag << ' ' << total_score << std::endl;
			tmp_data->total_score = total_score;

			if(read_barcode) {
				if(has_flavor) {
					std::istringstream l(line);
					std::list<std::string> fields;
					std::string field;
					l >> field;
					while(!l.fail()) {
						fields.push_back(field);
						l >> field;
					}
					std::list<std::string>::reverse_iterator it = fields.rbegin();
					it++; // second-to-last entry
					std::string flavorstring = *it;
					std::istringstream lf(flavorstring);
					char comma;
					unsigned int flav;
					std::list<unsigned int> flavors;
					while(!lf.fail()) {
						lf >> flav >> comma;
						flavors.push_back(flav);
					}
					if(flavors.size() != barcodes.num_features()) {
						std::cout << "FLAVORS DON'T MATCH BARCODES.  SKIPPING.\n";
						has_flavor = false;
					} else {
						std::list<unsigned int>::iterator flavor_it = flavors.begin();
						for(std::map< std::string, barcode_classes::feature >::iterator it = barcodes.feature_begin();
								it != barcodes.feature_end() && flavor_it != flavors.end();
								it++, flavor_it++ )
							it->second.set_current_index(*flavor_it);
					}
				}
				if(!has_flavor)	{
					for(std::map< std::string, barcode_classes::feature >::iterator it = barcodes.feature_begin();
							it != barcodes.feature_end();
							it++) {
						it->second.set_current_index(0);
					}
				}

				tmp_data->barcodes = barcodes;
			}

			///////////////////////
			// read the first line:
			// might be torsions for residue 1
			// or it might be info on the fold_tree

			if ( !getline(data,line) ) break;// EOF

			/////////////////////////////////////////////////////////////////////////
			//// check for fold_tree data
			{
				std::istringstream l( line );
				std::string tag;
				l >> tag;
				if ( tag == "FOLD_TREE" ) {
					///////////////////////////////////////////////
					//// there is fold_tree data in the silent-file
					l.seekg(0,std::ios::beg);
					// convenience refs to tmp_data stuff:
					pose_ns::Fold_tree & f( tmp_data->fold_tree );
					FArray1D< pose_ns::Jump > & jump_transform(tmp_data->jump_transform);
					// read the fold_tree
					f.clear();
					l >> f;
					if ( l.fail() || !f.check_fold_tree() ) {
						std::cout << "couldnt read fold_tree info: " << line << std::endl;
						continue; // NEXT
					}
					if ( !subset && f.get_nres() != nres ||
							 subset && f.get_nres() != nres_full ) {
						std::cout << "fold_tree length mismatch: " <<
							subset << ' ' << f.get_nres() << ' ' <<
							nres << ' ' << nres_full << std::endl;
						continue; // NEXT
					}

					////////////////////////////////////////////////////////////
					//// if we have a fold_tree, we also need to know the jumps!
					int const num_jump
						( symm_info ? symm_info->num_indep_jump(f.get_num_jump())
							: f.get_num_jump() );

					if ( !getline(data,line) ) break;

					{ // parse jumps line
						std::istringstream l( line );
						int njump;
						l >> tag >> njump;
						if ( l.fail() || tag != "JUMPS" || num_jump != njump ) {
							std::cout << "cant parse JUMP info: " << line << std::endl;
							continue; // NEXT
						}

						if ( num_jump > int( jump_transform.size() ) ) {
							jump_transform.dimension( num_jump );
						}
						bool fail(false);
						for ( int i=1; i<= num_jump; ++i ) {
							pose_ns::Jump jump;
							l >> jump;
							if ( l.fail() ) {
								std::cout << "failed to read jump# " << i << std::endl;
								fail = true;
								break;
							}
							jump_transform(i) = jump;
						}
						if ( fail ) continue; // NEXT
					}

					// read the next line: should be start of torsions
					if ( !getline(data,line) ) break;
				}
			}


			bool fail( false );

			//Dumbest PDB reader ever:
			while ( line.substr(0,5) == "ATOM " ) {
				if (fullatom) {
					tmp_data->full_coord_exists = true;
					std::istringstream l( line );
					std::string atom_name, pdb_resname;
					int ires,iatom;
					float x,y,z;
					l >> skip( 12 ) >>
						bite( 4, atom_name ) >> skip( 1 )  >>
						bite( 3, pdb_resname ) >> skip( 2 ) >> bite( 4, ires ) >>
						skip( 4 ) >> bite( 8, x ) >> bite( 8, y ) >> bite( 8, z ) >> skip;
					int currentres = aa2int( sequence[ ires - 1], fail );
					if (fail) break;
					atom_num_from_atom_name(atom_name,currentres,1,iatom);
					if (iatom != 0){
						tmp_data->full_coord(1,iatom,ires) = x;
						tmp_data->full_coord(2,iatom,ires) = y;
						tmp_data->full_coord(3,iatom,ires) = z;
					}
				}
				getline(data,line);
			}

			/////////////////////////////////////////////////////////////////////////
			//// read the torsion lines
			FArray1D_bool bb_idl( nres_torsions, false );
			int count( 0 );
			for ( int i=1; i<= nres_torsions && !fail; ++i ) {
				int pos;

				std::istringstream l( line );

				// get sequence position
				l >> pos;

				// debug pos:
				if ( ( symm_info && !symm_info->chi_independent(pos) ) ||
						 ( !symm_info && ( ( !subset && pos != i ) ||
															 ( subset && ( pos < 1 || pos > nres_full ))))) {
					std::cout << "parse error: " << subset << ' ' << i << ' ' << pos <<
						' ' << nres_full << ' ' << line << std::endl;
					fail = true;
					break;
				}

				// fill sequence and/or double-check
				int const i_pos( symm_info ? i : pos );
				if ( !subset ) {
					tmp_data->res( i_pos ) = aa2int( sequence[pos-1] );
				} else {
					if ( sequence[i-1] != full_sequence[pos-1] ) {
						std::cout << "Silent_file_data::read_file: sequence mismatch in" <<
							" full_seq: " << i << ' ' << pos << ' ' << sequence[i-1] <<
							' ' << full_sequence[pos-1] << std::endl;
						fail = true;

						break;
					}
				}

				// parse ss,torsions,c-alpha coords
				/////////////////////////////////////////////////////
				// Protein
				/////////////////////////////////////////////////////
				count++;
				if (param_aa::is_protein( aa2int( sequence[pos - 1] ) ) ){
					// parse ss,torsions,c-alpha coords


					l >>
						tmp_data->secstruct(count) >>
						tmp_data->phi  (count) >>
						tmp_data->psi  (count) >>
						tmp_data->omega(count) >>
						tmp_data->coords(1,count) >>
						tmp_data->coords(2,count) >>
						tmp_data->coords(3,count);

					if ( fullatom ) {
						// parse chi angles
						for ( int chino=1; chino<= param::MAX_CHI; ++chino ) {
							l >> tmp_data->chi(chino,count);
						}
					}
				}
				/////////////////////////////////////////////////////
				// RNA
				/////////////////////////////////////////////////////
				if (param_aa::is_RNA( aa2int( sequence[pos - 1] ) ) ){
					assert( fullatom );

					// parse ss,torsions,c4* coords.
					char dummy;
					l.read( &dummy, 1  );
					l.read( &dummy, 1  );
					tmp_data->secstruct(count)   = dummy;

					for (int j = 1; j <= param_torsion::total_rna_torsion-1; j++ ){
						l >> tmp_data->torsion( j, pos );
					}

					float temp_float;
					l >> temp_float;

					l.read( &dummy, 1  );
					dummy = l.peek();

					// New style outfile includes an "X" to demarcate when coordinates begin!
					if ( dummy == 'X' ) {
						tmp_data->torsion( param_torsion::total_rna_torsion, pos ) = temp_float;
						l.read( &dummy, 1  ); // get past the X
						l >>
							tmp_data->coords(1,pos);
					} else {
						// Old style outfile. We were already into the coordinates!
						tmp_data->torsion( param_torsion::total_rna_torsion, pos ) = -9999.9; //placeholder
						tmp_data->coords(1,pos) = temp_float;
					}

					l >>
						tmp_data->coords(2,pos) >>
						tmp_data->coords(3,pos);

				}
				/////////////////////////////////////////////////////

				std::string tag;
				l >> tag;

				if ( tag == "X") { //Hmm, could it be the "name"?
					l >> tmp_data->name(i_pos);
					l >> tag;
				}


				// check for AAV -- comes after chi angles and ROT tag
				if ( tag.substr(0,3) == "AAV" ) {
					tmp_data->res_variant(i_pos) = int_of( tag.substr(3,100) );
					l >> tag;
				}

				// check if file contains alternate rotamer template information
				if ( tag != decoy_tag &&
						 tag == "BB-IDL" ) {
					bb_idl(i) = true;
					l >> tag;
				}

				if ( tag != decoy_tag && !fullatom ) {
					// maybe this is a full-atom silent-file
					float dummy_float;
					for ( int chino=2; chino<= param::MAX_CHI; ++chino ) {
										l >> dummy_float;
					}
					l >> tag;
				}

				// check for AAV, might come now if !fullatom and fullatom silent
				if ( tag.substr(0,3) == "AAV" ) {
					tmp_data->res_variant(i_pos) = int_of( tag.substr(3,100) );
					l >> tag;
				}

				// check if file contains alternate rotamer template information
				if ( tag != decoy_tag &&
						 tag.substr(0,3) == "ROT" ) {
					if ( fullatom ) {
						std::istringstream is(tag.substr(3,100));
						is >> tmp_data->rot_template(i_pos);
						if ( is.fail() ) {
							std::cout << "ROT parse error: " << tag << std::endl;
							fail = true;
							break;
						}
					}
					l >> tag;
				}

				// check for AAV, might come now if !fullatom and fullatom silent
				if ( tag.substr(0,3) == "AAV" ) {
					tmp_data->res_variant(i_pos) = int_of( tag.substr(3,100) );
					l >> tag;
				}

				// check if file contains alternate rotamer template information
				if ( tag != decoy_tag &&
						 tag == "BB-IDL" ) {
					bb_idl(i) = true;
					l >> tag;
				}

				if ( tag != decoy_tag || l.fail() ) {
					std::cout << "parse error2: " << tag << ' ' << decoy_tag <<
						' ' << l.fail() << ' ' << pos << ' ' << i_pos << ' ' << i << ' ' <<
						nres_torsions << std::endl;
					fail = true;
					break;
				}

				if ( i < nres_torsions ) {
					// read the next line
					// this is a bit contorted b/c we want to read the first line
					// outside this loop, in case there is fold_tree data in the
					// silent-file
					if ( !getline( data, line) ) {
						std::cout << "EOF?" << std::endl;
						fail = true;
						break; // EOF, probably
					}
				}

			}

			if ( fail ) continue;

			// add bond info
			if ( !ideal ) {
				tmp_data->bonds = bonds;
				for ( int i=1; i<= nres; ++i ) {
					if ( bb_idl(i) ) {
						tmp_data->bonds.idealize(i,i);
					}
				}
			}

			// should we keep this tag?
			if ( desired_tags.size() &&
					 std::find( desired_tags.begin(), desired_tags.end(), decoy_tag ) ==
					 desired_tags.end() ) {
				// not a tag we are looking for
				delete tmp_data;
				tmp_data = 0;
			} else {
				// save this structure
				if ( prepend_filename ) {
					// pb 6/5/06 -- add filename to tag
					decoy_tag = filename+"_tag="+decoy_tag;
				}
				new_structure( decoy_tag, tmp_data);
				tmp_data = 0; // signal that someone else is worrying about space
			}

			// show progress
			tag_counter++;
			if ( tag_counter%500 == 0) {
				std::cout << "read structure " << decoy_tag << SS(tag_counter) <<
					std::endl;
			}
		} // while( getline(data,line))

		if ( tmp_data ) {
			std::cout << "EOF error?" << std::endl;
			delete tmp_data;
			tmp_data = 0;
		}

		io_fail = false;
		return io_fail;

	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Silent_structure::fill_pose(
		pose_ns::Pose & pose,
		const bool check_coords, // = false
		const bool save_input_score // = false
	) const
	{

		if (save_input_score) pose.set_extra_score( "INPUT_SCORE", total_score );

		// symmetry check
		pose_ns::Symmetry_info * symm_info(0);
		if ( parent && parent->get_symm_info() ) {
			symm_info = parent->get_symm_info();
		}

		// first set the tree
		if ( fold_tree.size() < 1 ) {
			assert(!symm_info);
			pose.simple_fold_tree( nres );
		} else {
			pose.set_fold_tree( fold_tree );
		}

		if (fold_tree.get_nres() > 0 ){
			param::MAX_RES_assign_res( fold_tree.get_nres() );
		}

		if ( symm_info ) {
			assert( nres == symm_info->nres_monomer_chi() );
			pose.setup_symm_info( *symm_info );
			// initialize ss, torsions, res
			pose.set_res_all( param_aa::aa_gly );
			insert_init_frag( pose, 1, fold_tree.get_nres() );
		}

		// set the jumps
		int const num_jump( fold_tree.get_num_jump() );
		if ( !symm_info ) {
			for ( int i=1; i<= num_jump; ++i ) {
				pose.set_jump( i, jump_transform(i) );
			}
		} else {
			int const num_indep_jump( symm_info->num_indep_jump( num_jump ) );
			for ( int i=1; i<= num_indep_jump; ++i ) {
				pose.set_jump( symm_info->indep_jump(i,num_jump), jump_transform(i) );
			}
		}

		pose.set_fullatom_flag( false, false );

		for ( int seqpos=1, i=0; seqpos<= pose.total_residue_for_scoring();
					++seqpos ) {
			if ( symm_info && !( symm_info->chi_independent(seqpos) ) ) continue;
			++i;
			assert( i <= nres );
			pose.set_phi        ( seqpos, phi         (i) );
			pose.set_psi        ( seqpos, psi         (i) );
			pose.set_omega      ( seqpos, omega       (i) );
			pose.set_secstruct  ( seqpos, secstruct   (i) );
			pose.set_res        ( seqpos, res         (i) );
			pose.set_res_variant( seqpos, res_variant (i) );
		}

		// fill bonds info
		if ( bonds.nres() ) {
			if ( symm_info ) {
				std::cout << "WARNING! Not sure non-ideal bonds are compatible with " <<
					"symmetry yet..." << std::endl;
				std::cerr << "WARNING! Not sure non-ideal bonds are compatible with " <<
					"symmetry yet..." << std::endl;
				//std::exit( EXIT_FAILURE );
				std::cout << "Bonds file found: " << bonds.nres() << " " << nres << std::endl;
			}
			assert( bonds.nres() == nres );
			pose.set_bonds( bonds );
		}

		//Pdb info.
		Pdb_info pdb_info;
		pdb_info.figure_out_pdb_info( pose );
		pose.set_pdb_information( pdb_info );

		//////////////////////////////////
		// RNA stuff
		//////////////////////////////////
		// check for non-protein, trigger atom-tree setup
		bool all_protein( true ), all_RNA( true );
		for ( int i=1; i<= nres; ++i ) {
			if ( !param_aa::is_protein( res(i) )) all_protein = false;
			if ( !param_aa::is_RNA( res(i) ))     all_RNA = false;
		}

		if ( !all_protein && !all_RNA) {
			std::cout << "Hey, can't handle mixtures of protein and RNA in silent format (yet)." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if (all_RNA && !all_protein){
			std::cout << "All RNA atoms detected, using Atom_tree object" << std::endl;
			make_ideal_rna_fullcoord( pose ); //sets up an ideal RNA backbone, and appropriate atom tree.
			for ( int i=1; i<= num_jump; ++i ) {
				pose.set_jump( i, jump_transform(i) );
			}
			for (int i=1; i<= nres; ++i ){
				assert( param_aa::is_RNA( res(i) ));
				for (int j = 1; j <= param_torsion::total_rna_torsion; j++ ){
					//Torsions we can't set. There should be a more elegant way of doing this...
					if (i == 1    && j == 1) continue;
					if (i == nres && j == 5) continue;
					if (i == nres && j == 6) continue;

					if ( torsion(j, i ) > -180.0 ) { //In case a torsion was missing, there's a big negative number.
						pose.set_torsion_by_number(i, j, torsion(j, i) );
					}

				}
			}
			pose.copy_to_misc(); //triggers a refold.

			//For non-ideal coordinates.
			if (full_coord_exists) pose.set_segment_full_coord( nres, 1, full_coord(1,1,1));

			return; // early return!!
		}
		//////////////////////////////////
		//////////////////////////////////


		// build backbone, centroids
		if ( fullatom ) {
			// fill the chi angles
			// pose.refold will put in icoor rotamers
			pose.set_fullatom_flag( true, false ); // setting,repack_rotamers
			for ( int seqpos=1, i=0; seqpos<= pose.total_residue_for_scoring();
						++seqpos ) {
				if ( symm_info && !( symm_info->chi_independent(seqpos) ) ) continue;
				++i;
				assert( i <= nres );
 				for ( int chino=1; chino<= param::MAX_CHI; ++chino ) {
 					pose.set_chi( chino, seqpos, chi(chino,i) );
				}
			}

 			if (full_coord_exists){
// 				for ( int l = full_coord.index(1,1,1), l_end = full_coord.index(1,1,1) +
// 								nres * full_coord.size1() * full_coord.size2(); l < l_end; ++l ) {
// 					dest_fcoord[l] = full_coord[l];
// 				}
				pose.set_segment_full_coord( nres, 1, full_coord(1,1,1) );
 			} else {

				if ( parent != 0 && !(parent->ideal_rotamers()) ) {
					if ( symm_info ) {
						std::cout << "WARNING! Not sure non-ideal rotamers are compatible with " <<
							"symmetry yet..." << std::endl;
						std::cerr << "WARNING! Not sure non-ideal rotamers are compatible with " <<
							"symmetry yet..." << std::endl;
							//						std::exit( EXIT_FAILURE );
					}

					// have to go back and fill in non-ideal sidechain geometry
					// first make a copy of the full_coord array (triggers a refold)
					FArray3D_float full_coord( pose.full_coord() );

					for ( int i=1; i<= nres; ++i ) {
						// use a template from the .rot_templates file
						int const rot_temp( rot_template(i) );

						if ( rot_temp != 0 ) {
							// not the standard rosetta rotamer
							std::vector< Rotamer_template > const & v
								( parent->rotamer_templates(i) );
							//std::cout << "use rot_template: " << i << ' ' <<
							//	rot_temp << std::endl;
							if ( rot_temp < 0 || rot_temp >= int(v.size()) ) {
								std::cout << "bad rot_template: " << i << ' ' <<
									rot_temp << ' ' << v.size() << std::endl;
								utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
							}
							// this generates a rotamer with chi angles given by tmp_chi
							// and superimposes it onto the backbone of full_coord
							// and overwrites whatever is present in full_coord for the
							// sidechain atoms. Assumes HN already calculated.
							// this assumes that non-rosetta-template rotamers
							// have just changed through sc-minimization, which
							// currently (07/08/05) does not rebuild proline
							// hydrogens
							bool const rebuild_pro( false );
							// 9/1/05 should change this to true, as soon as I change the
							// proline rebuilding inside refold_new_chi
							//
							v[ rot_temp ].insert( full_coord(1,1,i), chi(1,i), rebuild_pro );

							pose.copy_sidechain( i, pose.res(i), pose.res_variant(i),
																	 full_coord(1,1,i) );
						}
					} // i
				} // non-ideal sidechains
			} //full_coord_exists
		} // fullatom

		// double check coords
		if ( check_coords ) {
			// setup a mapping from the 1->nres indexing of coords
			// to the 1->total_residue indexing of pose (if symmetric)

			FArray1D_int mapping( nres, 0 );
			for ( int seqpos=1, i=0; seqpos<= pose.total_residue_for_scoring();
						++seqpos ) {
				if ( symm_info && !( symm_info->chi_independent(seqpos) ) ) continue;
				++i;
				mapping(i) = seqpos;
			}

			const FArray3D_float & Epos( pose.Eposition() );
			FArray2D_double p1a( 3, nres );
			FArray2D_double p2a( 3, nres );
			const int atom_index ( 2 ); // CA
			int natoms(0);
			for ( int i = 1; i <= nres; ++i ) {
				if ( std::abs( coords(1,i) ) + std::abs( coords(2,i) ) +
						 std::abs( coords(3,i) ) < 0.1 ) continue;
				++natoms;
				for ( int k = 1; k <= 3; ++k ) {
					p1a(k,natoms) = coords( k, i );
					p2a(k,natoms) = Epos( k, atom_index, mapping(i) );
				}
			}

			if ( natoms >= 3 ) {
				// calc rms
				FArray1D_double const ww( natoms, 1.0 );
				FArray2D_double uu( 3, 3 );
				double ctx;
				findUU( p1a, p2a, ww, natoms, uu, ctx );
				float fast_rms;
				calc_rms_fast( fast_rms, p1a, p2a, ww, natoms, ctx );
				std::cout << "fill_pose:: double-check CArmsd= " << fast_rms <<
					std::endl;
				if ( fast_rms > 0.1 ) {
					std::cout << "DANGER DANGER DANGER!! possible coords mismatch in " <<
						"silent-file reader: CA-rmsdeviation= " << fast_rms << std::endl;
					pose.dump_pdb( "test_io.pdb" );
					assert( false ); // die if DNDEBUG
				}
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////
	// returns TRUE if OK to start decoy; FALSE if it already exists
	//
	bool
	Silent_out::start_decoy(
		const std::string & decoy_name
	) const
	{
		bool decoy_exists(false);
		std::string list_filename( filename+".list" );

		utility::io::izstream data( list_filename );

		if ( data ) {
			std::string line,name;
			while ( getline( data, line ) ) {
				std::istringstream line_stream ( line );
				line_stream >> name;

#ifdef BOINC
				// ignore first character S or F
				if ( name.substr(1) == decoy_name.substr(1) ) {
#else
				if ( name == decoy_name ) {
#endif
					decoy_exists = true;
					break;
				}
			}
		}
		data.close();
#ifndef BOINC
		// mark as started
		// BOINC jobs must mark when finished
		if ( !decoy_exists ) append_to_list( decoy_name );
#endif
		return !decoy_exists;
	}

	void
	Silent_out::append_to_list(
		const std::string & decoy_name
	) const
	{
		std::string list_filename( filename+".list" );
		utility::io::ozstream out;
		out.open_append( list_filename );
		out << decoy_name << '\n';
		out.close();
	}

	///////////////////////////////////////////////////////////////
	// write scores, coords and torsions to file
	//
	// shows chi torsions if pose.get_fullatom_flag()
	//
	// assumes that backbone geometry is constant for all decoys
	// in the file. If use_pose_bonds is true for the 1st decoy, it
	// will output a bonds file ".bonds"
	//
	// does not handle non-ideal sidechains: call pose_idealize
	// to fix the sidechains first
	//
	// if outputting only a subset of positions, it will output all the
	// data the first time through in a file called ".full"
	//
	// decoys will only show the torsions for the subset positions
	//

	/////////////////////////////////////////////////////////////////////////////
	void
	Silent_out::write(
		std::string const & decoy_name,
		pose_ns::Pose const & pose
	)
	{
		// setup(...) handles initialization of the silent file (writes header)
		// if use_pose_bonds() makes a .bonds file
		// if subset makes a .full file with one full decoy
		//  then assumes only subset torsions are varying between decoys
		//
		setup( pose );

		FArray1D_int rot_template( pose.total_residue(), 0 );
		if ( !ideal_rotamers() ) {
			get_rot_templates( pose, rot_template );
		}

		//////////////////////////////////////
		// rework this for BOINC compatibility

		std::string tmpfile;

		{ //////////////////////////
			// NAME THE TEMPORARY FILE
			tmpfile = "tmp" + decoy_name + ".pdb";

			// try to remove any funny characters
			std::replace( tmpfile.begin(), tmpfile.end(), '\\', '_' );
			std::replace( tmpfile.begin(), tmpfile.end(), '/', '_' );
			std::replace( tmpfile.begin(), tmpfile.end(), '.', '_' );
		}

		{ ////////////////////////////////
			// CREATE THE TEMPORARY FILE

			utility::io::ozstream out;
			out.open( tmpfile );

			// score output
			out << "SCORE: ";
			pose.show_score_values( out );

			if (get_maxsub_to_server_models_flag())
				maxsub_to_server_models_output( out );

			if (get_output_flavor())
				barcode_flavor_output( out );

			if (get_output_filter())
				out << A(14, get_filter_tag());

			out << ' ' << decoy_name << '\n';

			// fold_tree, jumps, C-alpha coords and torsions
			dump_compressed_pose( decoy_name, pose, output_rsd, rot_template,
														symm_info, out );

			// close tmpfile
			out.close();
		}


		{ ////////////////////////////////////////
			// NOW CAT TO THE END OF THE SILENT FILE

			// open silentfile for appending
			utility::io::ozstream z_ostream_app;
			z_ostream_app.open_append( filename );
			if ( !z_ostream_app ) {
				std::cout << "Open failed for file: " << z_ostream_app.filename() <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			// open tmpfile for reading
			utility::io::izstream z_istream( tmpfile );
			if ( !z_istream ) {
				std::cout << "Open failed for file: " << z_istream.filename() <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			// append!
			z_ostream_app << z_istream.rdbuf();
			if ( z_ostream_app.fail() ) {
				std::cout << "Append failed for file: " << z_ostream_app.filename() <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			// close files
			z_ostream_app.close();
			z_istream.close();
		}


		{ /////////////////////////
			// NOW DELETE THE TMPFILE
			utility::file::file_delete( tmpfile );
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////
	// copy non-ideal rotamers
	//
	void
	Silent_out::set_rotamer_templates(
		std::map< int, std::vector< Rotamer_template > > const & map_in
	)
	{
		rotamer_template_map_ = map_in;
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////
	// store non-ideal rotamers
	//
	void
	Silent_out::store_rotamers(
		pose_ns::Pose const & pose
	)
	{
		int const nres( pose.total_residue() );
		assert( pose.fullatom() );
		FArray3D_float const & full_coord( pose.full_coord() );

		for ( int seqpos=1; seqpos<= nres; ++seqpos ) {
			int const aa( pose.res(seqpos) );
			int const aav( pose.res_variant(seqpos) );
			if ( rotamer_template_map_.count(seqpos) ) {
				// assume we already put the standard rotamer on there
				std::vector< Rotamer_template > & v
					( rotamer_template_map_.find( seqpos )->second );
				v.push_back( Rotamer_template( aa, aav, full_coord( 1, 1, seqpos ) ) );
			} else {
				std::vector< Rotamer_template > v;
				v.push_back( Rotamer_template( aa, aav,
					aaproperties_pack::icoor( 1, 1, aa, aav ) ) );
				v.push_back( Rotamer_template( aa, aav, full_coord( 1, 1, seqpos ) ) );
				rotamer_template_map_.insert( std::make_pair( seqpos, v ) );
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////
	//
	bool
	Silent_out::ideal_rotamers() const
	{
		return ( rotamer_template_map_.size() == 0 );
	}

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////
	//
	//
	void
	Silent_out::get_rot_templates(
		pose_ns::Pose const & pose,
		FArray1D_int & rot_template
	) const
	{
		int const nres( pose.total_residue() );
		assert( pose.fullatom() );
		assert( int(rot_template.size1()) >= nres );
		FArray3D_float const & full_coord( pose.full_coord() );

		float total_dist(0.0);

		for ( int seqpos=1; seqpos<= nres; ++seqpos ) {
			if ( !pose.is_protein( seqpos ) ) continue;
			int const aa( pose.res(seqpos) );
			int const aav( pose.res_variant(seqpos) );
			if ( !rotamer_template_map_.count(seqpos) ) {
				std::cout << "cant find seqpos in rotamer map: " << seqpos <<
					std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			std::vector< Rotamer_template > const & v
				( rotamer_template_map_.find( seqpos )->second );

			Rotamer_template const current( aa, aav, full_coord(1,1,seqpos) );

			float min_dist(1000.0);
			int best_rot(-1);

			for ( int rot=0; rot< int(v.size()); ++rot ) {
				float dist( current.distance( v[rot] ) );
				if ( dist < min_dist ) {
					min_dist = dist;
					best_rot = rot;
				}
			}
			rot_template(seqpos) = best_rot;
			total_dist += min_dist;
// 			std::cout << "get_rot_templaces: " << seqpos << ' ' << min_dist <<
// 				std::endl;
		}
		std::cout << "get_rot_templates: total_dist= " << total_dist <<
			std::endl;
	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////
	// this gets called each time we write a decoy
	//
	// private method:
	// DOES NOT handle sequence-checking anymore!! except maybe the
	// first time through
	void
	Silent_out::setup(
		const pose_ns::Pose & pose
	)
	{
		// decide if we should use symmetry
		if ( pose.symmetric() && pose.ideal_backbone() && ideal_rotamers() ) {
			if ( symm_info ) {
				assert( *symm_info == pose.symmetry_info() );
			} else {
				symm_info = new pose_ns::Symmetry_info( pose.symmetry_info() );
			}
		} else {
			assert( symm_info == 0 );
		}

		if ( pose.symmetric() && !pose.ideal_backbone() ){
			std::cout << "Silent_out::setup: silent output with symmetry info not compatible with non-ideal bonds yet." << std::endl;
			std::cerr << "Silent_out::setup: silent output with symmetry info not compatible with non-ideal bonds yet." << std::endl;
		}

		int const nres
			( symm_info ? symm_info->nres_monomer_chi() : pose.total_residue() );

		// debug filename
		if ( filename.size() <= 0 ) {
			std::cout << "Silent_out::setup: no file opened yet!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// define output_rsd:
		if ( !subset ) {
			output_rsd.dimension( pose.total_residue() );
			output_rsd = true;
		}

		// check if file exists
		// WARNING: note that stuff below this return statement will
		// not necessarily get executed during a run, eg if a different
		// process creates the file!
		utility::io::izstream data ( filename.c_str() );
		if ( data ) {
			data.close();
			return;
		}

		// open file for writing:
		utility::io::ozstream out( filename );
		if ( ! out.good() ) {
			std::cout << "cant open silent file for writing: " << filename <<
				std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// do all the once-per-file initialization:
		// write header
		// create bonds file if necessary
		// create full_torsions file if necessary

		// initialize the silent-file
		out << "SEQUENCE: " << pose.sequence();
		if ( symm_info ) out << ' ' << *symm_info;
		out << "\nSCORE: ";
		pose.show_score_tags( out ); // same order as output by show_scores

		if (get_maxsub_to_server_models_flag())
			maxsub_to_server_models_output_header( out );

		if (get_output_flavor())
			out << A(60,"FLAVOR") ;

		if (get_output_filter())
			out << A(14,"filter");

		out << " description\n";
		out.close();
		out.clear();

		// make .bonds file if necessary:
		if ( !pose.ideal_backbone() ) {
			std::string const bonds_filename( filename+".bonds" );
			out.open( bonds_filename );
			pose.bonds().dump( out );
			out.close();
			out.clear();
		}

		// make rotamers file if necessary:
		if ( !ideal_rotamers() ) {
			std::string const rot_filename( filename+".rot_templates" );
			out.open( rot_filename.c_str() );
			for( int i=1; i<= nres; ++i ) {
				if ( !rotamer_template_map_.count(i) ) {
					std::cout << "i not in rotamer_template_map_: " << i << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				std::vector< Rotamer_template > const & rot_list
					( rotamer_template_map_.find( i )->second );

				for( int j=0,j_end=rot_list.size(); j<j_end; ++j ) {
					out << i << ' ' << j << ' ' << rot_list[j] << '\n';
				}
			}
			out.close();
			out.clear();
		}

		// show non-varying torsions, only once
		// initialize output_rsd if !subset
		if ( subset ) {
			// tricky: open a new silent-file object WITHOUT subset
			// write a single decoy
			Silent_out full_decoy( filename+".full_torsions" );
			if ( !ideal_rotamers() ) {
				// ensure that this outfile also has the rotamer information
				full_decoy.set_rotamer_templates( rotamer_template_map_ );
			}
			full_decoy.write( "full_pose", pose );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////
	// setup to only write out a subset of the torsion angles
	//
	void
	Silent_out::set_subset(
		int const nres,
		FArray1D_bool const & rsd
	)
	{
		subset = true;
		output_rsd.dimension( nres );
		for ( int i=1; i<= nres; ++i ) {
			output_rsd(i) = rsd(i);
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////
	// just write the scores
	void
	Silent_out::write_scores(
		const std::string & decoy_name,
		const pose_ns::Pose & pose
	)
	{
		// this routine writes the header if necessary:
		setup( pose );

		// open out-file for appending
		utility::io::ozstream out;
		out.open_append( filename.c_str() );
		out << "SCORE: ";
		pose.show_score_values( out );
		out << ' ' << decoy_name << '\n';
		out.close();
	}

	/////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////
	// declarations for the Decoy_out class

	bool
	Decoy_out::decoy_exists(
		std::string const & decoy_name
	)
	{
		setup();
		if ( !silent ) {
			std::ifstream data( pdb_filename( decoy_name ).c_str() );
			if ( data.good() ) {
				data.close();
				return true;
			} else {
				data.close();
				std::ofstream out( pdb_filename( decoy_name ).c_str() );
				out.close(); // create empty file
				return false;
			}
		} else {
			return !silent_out.start_decoy( decoy_name );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////
	void
	Decoy_out::write(
		std::string const & decoy_name,
		pose_ns::Pose const & pose
	)
	{
		setup();
		if (!silent ) {
			// open decoy file
			std::ofstream out( pdb_filename( decoy_name ).c_str() );

			// dump coords
			if ( pose.fullatom() ) {
				dump_full_coord( pose, out );
			} else {
				dump_Eposition( pose, out );
			}

			// output score-info
			out << "REMARK ";
			pose.show_scores( out );
			out << '\n';

			// all done
			out.close();
		} else {
			silent_out.write( decoy_name, pose );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////
	// private method
	std::string
	Decoy_out::pdb_filename(
		std::string const & decoy_name
	) const
	{
		return ( pdb_prefix + decoy_name + ".pdb" );
	}

	/////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////
	void
	Decoy_out::setup()
	{
		if ( init ) return;
		init = true;

		// output path:
		std::string prefix( files_paths::pdb_out_path + "/" );

		if ( files_paths::query_defined ) {
			prefix = prefix + files_paths::code +	files_paths::protein_name +
				files_paths::protein_chain;
		} else {
			prefix = "decoy";
		}

		// silent-file
		if ( silent ) {
			// setup name
			if ( !silent_filename.size() ) {
				silent_filename = prefix + ".out";
			}

			// pass name to silent_out object
			silent_out.open( silent_filename );
		} else {
			if ( !pdb_prefix.size() ) {
				pdb_prefix = prefix;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	// helper function
	// outputs
	void
	dump_compressed_pose(
		std::string const & decoy_name,
		pose_ns::Pose const & pose,
		FArray1D_bool const & output_rsd,
		FArray1D_int const & rot_template, // best template for this rsd
		pose_ns::Symmetry_info const * const symm_info,
		std::ostream & out
	)
	{

		int const nres( pose.total_residue_for_scoring() );

		// Hey, I want to see those pseudoresidues!
		//		int const nres( pose.total_residue() );

		//( symm_info ? symm_info->nres_monomer_chi() : pose.total_residue() );
		int num_jump
			( symm_info ? symm_info->num_indep_jump(pose.num_jump())
				: pose.num_jump());

		const bool fullatom ( pose.fullatom() );
		const FArray3D_float & Eposition ( pose.Eposition() );
		const pose_ns::Fold_tree & f     ( pose.fold_tree() );

		// tree and jumps if not a simple-tree
		if ( num_jump ) {
			// show fold_tree

			if ( get_enable_ligaa_flag() && get_docking_flag() ){
				//////////////////////////////////////////////////////////////////////////////////////////
				//
				//Ingemar and Rhiju terrible hack for ligand!!!!!
				// To be replaced in the near future with a proper ligand output.
				// But this is a hack we need for CAPRI. Just pretend ligand is not there!
				//

				FArray2D_int const & jump_points ( f.get_jump_point() );
				int num_fold_tree_cut_points;
				FArray1D_int const & cuts ( f.get_fold_tree_cutpoint(num_fold_tree_cut_points) );
				pose_ns::Fold_tree f_temp;
				// this builds the tree:
				f_temp.tree_from_jumps_and_cuts( nres-1, num_fold_tree_cut_points-1, jump_points, cuts );
				output_foldtree(f_temp, out);
				num_jump--;
				// END HACK
				//////////////////////////////////////////////////////////////////////////////////////////
			} else {
				output_foldtree(f, out);
			}

			out << ' ' << decoy_name << '\n'; // need decoy name output too.
			// show the jumps
			out << "JUMPS " << num_jump << ' ';
			for ( int i=1; i<= num_jump; ++i ) {
				int const jump_number
					( symm_info ? symm_info->indep_jump(i,pose.num_jump()) : i );

				pose_ns::Jump jump( pose.get_jump( jump_number ) );

				///////////////////////////
				// HACKHACKHACK!!
				if ( get_enable_ligaa_flag() && get_docking_flag() ){
					FArray3D_float const & Eposition = pose.Eposition();
					jump = pose_ns::Jump(
														 Eposition( 1, 1, f.upstream_jump_residue( i ) ),
														 Eposition( 1, 1, f.downstream_jump_residue( i ) ) );
				}
				///////////////////////////

				jump.fold_in_rb_deltas();
				out << jump;

			}
			out << ' ' << decoy_name << '\n';
		}

		//Full coordinates if bond distances/angle moved to unpredictable values
		if (pose.get_vary_bond_geometry_flag() && fullatom){
			dump_full_coord( pose, out, " "+decoy_name);
		}

		// bb,chi torsions and c-alpha's
		for ( int i=1; i<= nres; ++i ) {
			if ( output_rsd(i) ) {
				if ( symm_info && !(symm_info->chi_independent(i)) ) continue;

				/////////////////////////////////////////////////////
				// Protein
				/////////////////////////////////////////////////////
				if ( param_aa::is_protein( pose.res(i)) ){
					out << I( 4, i ) << ' ' << pose.secstruct(i) << ' ' <<
						F( 9, 3, pose.phi(i) ) << F( 9, 3, pose.psi(i) ) <<
						F( 9, 3, pose.omega(i) ) <<
						F( 9, 3, Eposition(1,2,i) ) <<
						F( 9, 3, Eposition(2,2,i) ) <<
						F( 9, 3, Eposition(3,2,i) );
					if ( fullatom ) {
						for ( int chino=1; chino<= param::MAX_CHI; ++chino ) {
							out << F( 9, 3, pose.chi(chino, i) );
						}
						if ( rot_template(i) != 0 ) {
							out << " ROT" << rot_template(i);
						}
					}

					if ( pose.res_variant(i) != 1 ) {
						out << " AAV" << pose.res_variant(i);
					}

					if ( !pose.ideal_backbone() ) {
						if ( pose.bonds().is_ideal( i ) ) out << " BB-IDL";
					}

					out << ' ' << decoy_name << '\n';
				} //protein

				/////////////////////////////////////////////////////
				// RNA
				/////////////////////////////////////////////////////
				if ( param_aa::is_RNA( pose.res(i)) ){
					out << I( 4, i ) << ' ' << pose.secstruct(i) << ' ';
					for (int j = 1; j <= param_torsion::total_rna_torsion; j++ ){
						out << F( 9, 3, pose.get_torsion_by_number(i , j)  );
					}
					out << " X"; //new, just to demarcate end of torsions.

					//Assume nucleotide has a C4*?
					numeric::xyzVector_float const full_coord_i = pose.full_coord( 6, i );
					out <<
						F( 9, 3, full_coord_i(1) ) <<
						F( 9, 3, full_coord_i(2) ) <<
						F( 9, 3, full_coord_i(3) );

					//new, maybe dangerous.
					out << ' ' << 'X';
					out << ' ' << pose.name( i );

					out << ' ' << decoy_name << '\n';
				} //RNA

			}
		}

	} // dump_compressed_pose

} // namespace silent_io

// helper fcn
std::string
files_paths_pdb_out_prefix()
{
	using namespace files_paths;
	std::string prefix( pdb_out_path + "/" );
	if ( query_defined ) {
		prefix += code + protein_name + protein_chain;
	}
	return prefix;
}

// helper fcn
std::string
files_paths_pdb_out_prefix_nochain()
{
	using namespace files_paths;
	std::string prefix( pdb_out_path + "/" );
	if ( query_defined ) {
		prefix += code + protein_name;
	}
	return prefix;
}

/////////////////////////////////////////////////////////////////////////////
// This seems silly, it would be preferable to use << foldtree, but it looks like
// that operator is always being used with an endline (rhiju).
void output_foldtree(pose_ns::Fold_tree const & t, std::ostream & os)
{
	os << "FOLD_TREE ";

	for ( pose_ns::Fold_tree::const_iterator it = t.begin(), it_end = t.end();
				it != it_end; ++it ) {
		os << *it;
	}
}

// 	/////////////////////////////////////////////////////////////////////////////
// 	void
// 	Silent_structure::store_tree_and_jumps(
// 		const pose_ns::Fold_tree & fold_tree_in,
// 		const FArray1D< pose_ns::Jump > & jump_transform_in
// 	)
// 	{
// 		using namespace pose_ns;
// 		fold_tree = fold_tree_in;
// 		// jumps
// 		const int num_jump( fold_tree.get_num_jump() );
// 		if ( num_jump > 0 ) {
// 			jump_transform.dimension( num_jump );
// 			for ( int i=1; i<= num_jump; ++i ) {
// 				jump_transform(i) = jump_transform_in(i);
// 			}
// 		}
// 	}

