// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 9595 $
//  $Date: 2006-08-08 19:20:23 -0400 (Tue, 08 Aug 2006) $
//  $Author: tex $


// The code below  is using linux specific socket libraries
#ifdef __linux__

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#include <stdlib.h>
#include <iostream>
#include <string>

#include "socketcom.h"

///////////////////////////////////////
///
/// generic socket & helper functions

// wait's by wasting CPU cycles -
// a proper idling function would be good here instead.
double sillywait(int n)
{
	double expsum = 1.1;
	for(int i=0;i<n;i++){
		for(int k=0;k<1000000;k++){
			expsum += exp(-expsum);
		}
	}
	return expsum;
}

template <class T, class U>
T min(const T a,const U b){
	return (a < b ? a : b);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sendDataBlock
///
/// @brief
///        sends a block of data to the designated previously opened socket
/// @detailed
///
/// @param sockd  - the communication socket
///        data   - pointer to the the data
///        length - length of data pointed to by data
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int sendDataBlock(int sockd,const char *data,unsigned int length)
{

	int nsent;
	int nsenttotal=0;

	if((nsent = send(sockd,&length,sizeof(int), 0 )) == -1){
		std::cerr << "Error sending " << std::endl;
		return -1;
	};

	while(nsenttotal < (int)length){
		if((nsent = send(sockd,&data[nsenttotal],
			min((unsigned)length-(unsigned)nsenttotal,(unsigned)200),0 )) <= -1){
				std::cerr<< "Error sending " << std::endl;
				return -1;
			}
			nsenttotal += nsent;
	}

	return 0;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin recvDataBlock
///
/// @brief
///        receives a block of data from a previously opened socket
///        it will allocate an apropriate amount of memory which
///        will have to be freed by the caller !
///
/// @detailed
///        returns -1 on failure or lenght of data received otherwise
/// @param sockd  - the communication socket
///        data   - pointer to data passed by pointer - will be set to data (or NULL in case of failure)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int recvDataBlock(int sockd, char **data)
{
	unsigned length=0;

	int nrecv;
	int nrecvtotal=0;
	int RECVDATABLOCKLIMIT = 1000000;   // just to protect against allocating huge amounts of memory;

	(*data) = NULL;
	if ((nrecv=recv(sockd, &length, sizeof(int), 0)) == -1) {
		std::cerr << "Error during receiving data" << std::endl;
		std::cerr << strerror(errno) << std::endl;
		return -1;
	}

	if(length > (unsigned)RECVDATABLOCKLIMIT) length = RECVDATABLOCKLIMIT;
	(*data) = new char [length];

	while((unsigned)nrecvtotal < length){
		if((nrecv = recv(sockd,&((*data)[nrecvtotal]),length-nrecvtotal,0 )) <= -1){
			std::cerr << "Error receiving " << std::endl;
			std::cerr << strerror(errno) << std::endl;
			return -1;
		}
		nrecvtotal += nrecv;
	};

	return length;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin recvDataBlock
///
/// @brief
///        receives a block of data from a previously opened socket
///        expects a preallocated memory block to which the data will be written
///        will not write more than maxlength data - if more than maxdata is received
///        data will be truncated
/// @detailed
///        returns -1 on failure or lenght of data received otherwise
///
/// @param sockd  - the communication socket
///        data   - pointer to data
///        maxlength - max capacity of the space pointed to by data
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int recvDataBlock_fastmem(int sockd, char *data, int maxlength)
{
	unsigned length=0;

	int nrecv;
	int nrecvtotal=0;

	if ((nrecv=recv(sockd, &length, sizeof(int), 0)) == -1) {
		std::cerr << "Error during receiving data" << std::endl;
		std::cerr << strerror(errno) << std::endl;
		return -1;
	}

	if(length > (unsigned)maxlength) length = maxlength;

	while((unsigned)nrecvtotal < length){
		if((nrecv = recv(sockd,&((data)[nrecvtotal]),length-nrecvtotal,0 )) <= -1){
			std::cerr << "Error receiving " << std::endl;
			std::cerr << strerror(errno) << std::endl;
			return -1;
		}
		nrecvtotal += nrecv;
	};

	return length;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin setup_listening_port
///
/// @brief
///        sets up a port for incoming client requests
/// @detailed
///        returns socked handle as int or -1 in case of failure
/// @param
///        port - port number where the socket is to be set
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////


int setup_listening_port(unsigned short int port)
{
	int    sockfd=0;

	int    error;
	struct sockaddr_in netaddr;  // local address
	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	int    yes=1;

	if(sockfd < 0){
		std::cout << "Error during socket()" << std::endl;
		std::cerr << strerror(errno) << std::endl;
		return -1;
	}

	netaddr.sin_family  = AF_INET;
	netaddr.sin_port    = htons(port);
	netaddr.sin_addr.s_addr = INADDR_ANY;
	memset(&(netaddr.sin_zero),'\0',8);

	if((error = setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int)))!=0){
		std::cerr << "Error during socket()" << std::endl;
		std::cerr << strerror(errno) << std::endl;
		return -1;
	}


	if((error=bind(sockfd, (struct sockaddr *)&netaddr, sizeof(struct sockaddr)))!=0){
		std::cerr << "Error during bind()" << std::endl;
		std::cerr << strerror(errno) << std::endl;
		return -1;
	}

	return sockfd;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin setup_listening_port
///
/// @brief
///        sets up a connection to a remote server
/// @detailed
///        returns socked handle as int or -1 in case of failure
/// @param
///        port - port number of remove server
///        srvaddress - C type string (null terminated) to the ip adress of the server
///        nattempts - number of tries before failure
///        waitbetween - amount of time to be wasted between trials
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
setup_connection_to_server( unsigned short int port,
													 const char *srvaddress,
													 int nattempts,
													 int waitbetween)
{
	int sockfd;
	int i, error;
	struct sockaddr_in netaddr;


	for(i=0;i<nattempts;i++){
		sockfd = socket(AF_INET, SOCK_STREAM, 0);
		if(sockfd < 0){
			std::cout << "Attempt " << i << " Error during socket()" << std::endl;
			sillywait(waitbetween);
			continue;
		}
		break;
	}
	if(i>=nattempts) return -1;  // quit for loop because of timeout

	netaddr.sin_family  = AF_INET;
	netaddr.sin_port    = htons(port);
	netaddr.sin_addr.s_addr = inet_addr(srvaddress);
	memset(&(netaddr.sin_zero),'\0',8);

	std::cout << "CLNT: Waiting for connection (" << srvaddress << ":" << port << ") ..";
	for(i=0;i<nattempts;i++){
		error = connect(sockfd, (struct sockaddr *)&netaddr, sizeof(struct sockaddr));
		if(error != 0){
			sillywait(waitbetween);
			std::cout<<"."<<std::endl;
			std::cout.flush();
			continue;
		}
		std::cout << "connected" << std::endl;
		return sockfd;
	}

	std::cout << "timeout" << std::endl;

	return -1;
}


////////////////////////////////////////////////////////////////////////////////////////////
///
///   CSA protocol specific functions


////////////////////////////////////////////////////////////////////////////////
/// @begin request_new_id
///
/// @brief
///        requests a new id for the client
/// @detailed
///        returns id obtained
/// @param
///        sockfd - socket handle
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
request_new_id(int sockfd)
{
	int newid = -1;
	int error,length;
	char *recvbuffer;

	char command = RDIST_CLNT_GETID;

	error = sendDataBlock(sockfd, &command, 1);  // request the ID
	if(error!=0){
		std::cerr << "Error requesting of ID" << std::endl;
		return -1;
	}

	length = recvDataBlock(sockfd, &recvbuffer);

	if(length < 4){
		std::cerr << "Error receving ID" << std::endl;
		return -1;
	}
	newid = *((int*)recvbuffer);

	delete [] recvbuffer;

	return newid;
}




////////////////////////////////////////////////////////////////////////////////
/// @begin request_logoff
///
/// @brief
///        tells server that client intends to shut down
/// @detailed
///        returns -1 on failure
/// @param
///        sockfd - socket handle
///        myid   - client id previously obtained (so that server knows who we are)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
request_logoff(int sockfd, int myid)
{

	int error,length;
	char *recvbuffer;

	char cmd[5];
	cmd[0] =  RDIST_CLNT_LOGOFF;
	*((int*)(&cmd[1])) = myid;

	std::cout << "Logging off with ID " << myid << std::endl;

	error = sendDataBlock(sockfd,(char*) &cmd[0], 5 );
	if(error!=0){
		std::cerr << "Error logging off" << std::endl;
		return -1;
	}

	length = recvDataBlock(sockfd, &recvbuffer);
	if(length < 1){
		std::cerr << "Error receving logoff ack" << std::endl;
		return -1;
	}

	error = 0;
	if( recvbuffer[0] != RDIST_CLNT_LOGOFF ) error = -1;   // server didn't get it

	delete [] recvbuffer;
	return error;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin request_new_structure
///
/// @brief
///        tries to obtain a new strating structure & possibly handing back a
///        finished one
/// @detailed
///        returns -1 on failure
/// @param
///        sockfd - socket handle
///        finished pdb - string containing the PDB file of a finished structure
///                       (i.e. from the previous run) or "" if no finished structure is
///                       available (e.g. at the beginning of a run)
///        structure id - id of the parrent structure  (or -1 of structure failed filters)
///        controlstruct - will contain the 64 control structure sent accomanying the
///                       new starting structure
///        newpdbfile -   will contain new starting structure (or "" if no such structure is available)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

int
request_new_structure(int sockfd,
											const std::string &finishedpdb,
											int structureid,
											std::string &controlstruct,
											std::string &newpdbfile
											)
{

	int   length,error;
	char *recvbuffer;

	std::string sendbuffer_stl;
	int   errorstatus=0;

	// send back the finished PDB file name to the server (or just a 0 byte)
	if(finishedpdb.empty()){
		sendbuffer_stl = "";
		sendbuffer_stl.push_back(RDIST_CLNT_EXCHANGEPDBNAME);
		sendbuffer_stl.push_back(0);
		sendbuffer_stl.push_back(0);
		sendbuffer_stl.push_back(0);
		sendbuffer_stl.push_back(0);
		sendbuffer_stl.push_back(0);
		error = sendDataBlock(sockfd, sendbuffer_stl.c_str(), sendbuffer_stl.size());

	}else{
		sendbuffer_stl = "";
		sendbuffer_stl.push_back(RDIST_CLNT_EXCHANGEPDBNAME);
		sendbuffer_stl.push_back(((char*)&structureid)[0]);
		sendbuffer_stl.push_back(((char*)&structureid)[1]);
		sendbuffer_stl.push_back(((char*)&structureid)[2]);
		sendbuffer_stl.push_back(((char*)&structureid)[3]);

		sendbuffer_stl += finishedpdb;
		error = sendDataBlock(sockfd, sendbuffer_stl.c_str(), sendbuffer_stl.size());
	}

	if(error!=0){
		std::cout << "Error during structure request" << std::endl;
		return -1;
	}

	// now receive the responce and maybe a new file name to work on
	length = recvDataBlock(sockfd,&recvbuffer);
	std::string recvbuffer_stl(recvbuffer,length);
	delete [] recvbuffer;

	if(length<1){
		std::cout << "Error during new structure responce/receipt of new structure" << std::endl;
		return -1;
	}
	std::cout << "Received data block of length " << length << std::endl;
	switch((unsigned char)recvbuffer_stl[0]){
		 case RDIST_CLNT_EXCHANGEPDBNAME:

			 if(length <= 64){
				 std::cout << "Error: Data did not contain control structure" << std::endl;
				 errorstatus = -1;
				 break;
			 }
			 if(length <= 65){
				 std::cout << "Error: Data did not contain filename" << std::endl;
				 errorstatus = -1;
				 break;
			 }

			 controlstruct = recvbuffer_stl.substr(1,64);
			 newpdbfile = recvbuffer_stl.substr(65);
			 errorstatus=0;
			 break;
		 case RDIST_SRV_STARTEXTENDED:
			 errorstatus = 2;  // mark that we are instructed to start with an extended structure command
			 if(length < 65){
				 std::cout << "Error: Data did not contain control structure" << std::endl;
				 errorstatus = -1;
				 break;
			 }
			 newpdbfile = "";
			 controlstruct = recvbuffer_stl.substr(1,64);
			 std::cout << "Received RDIST_SRV_STARTEXTENDED" << std::endl;
			 break;
		 case RDIST_SRV_IDLE:
			 errorstatus = 3; // mark that an idle command has been received
			 std::cout << "Received IDLE command .. waiting for a while " << std::endl;
			 break;
		 case RDIST_SRV_FINISH:
			 errorstatus = 1;  // mark that we received a FINISH command
			 std::cout << "Received FINISH command" << std::endl;
			 break;
		 default:
			 std::cout << "Received unknown command, code: " << int(recvbuffer[0]) << std::endl;
			 errorstatus = -1;
			 break;
	}

	return errorstatus;
}

#endif

