// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/xyz.functions.hh
/// @brief  xyzVector and xyzMatrix functions
/// @author Frank M. D'Ippolito (Objexx@objexx.com)
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)

#ifndef INCLUDED_numeric_xyz_functions_HH
#define INCLUDED_numeric_xyz_functions_HH


// Package headers
#include <numeric/conversions.hh>
#include <numeric/NumericTraits.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// C++ headers
#include <cmath>
#include <cstdlib>


namespace numeric {


/// @brief xyzMatrix * xyzVector
/// @note  Same as product( xyzMatrix, xyzVector )
template< typename T >
inline
xyzVector< T >
operator *( xyzMatrix< T > const & m, xyzVector< T > const & v )
{
	return xyzVector< T >(
		m.xx_ * v.x_ + m.xy_ * v.y_ + m.xz_ * v.z_,
		m.yx_ * v.x_ + m.yy_ * v.y_ + m.yz_ * v.z_,
		m.zx_ * v.x_ + m.zy_ * v.y_ + m.zz_ * v.z_
	);
}


/// @brief xyzMatrix * xyzVector product
/// @note  Same as xyzMatrix * xyzVector
template< typename T >
inline
xyzVector< T >
product( xyzMatrix< T > const & m, xyzVector< T > const & v )
{
	return xyzVector< T >(
		m.xx_ * v.x_ + m.xy_ * v.y_ + m.xz_ * v.z_,
		m.yx_ * v.x_ + m.yy_ * v.y_ + m.yz_ * v.z_,
		m.zx_ * v.x_ + m.zy_ * v.y_ + m.zz_ * v.z_
	);
}


/// @brief xyzMatrix * xyzVector in-place product
/// @note  Input xyzVector is modified
template< typename T >
inline
xyzVector< T > &
inplace_product( xyzMatrix< T > const & m, xyzVector< T > & v )
{
	T const x = m.xx_ * v.x_ + m.xy_ * v.y_ + m.xz_ * v.z_;
	T const y = m.yx_ * v.x_ + m.yy_ * v.y_ + m.yz_ * v.z_;
	v.z_      = m.zx_ * v.x_ + m.zy_ * v.y_ + m.zz_ * v.z_;
	v.x_ = x;
	v.y_ = y;
	return v;
}


/// @brief xyzMatrix^T * xyzVector product
template< typename T >
inline
xyzVector< T >
transpose_product( xyzMatrix< T > const & m, xyzVector< T > const & v )
{
	return xyzVector< T >(
		m.xx_ * v.x_ + m.yx_ * v.y_ + m.zx_ * v.z_,
		m.xy_ * v.x_ + m.yy_ * v.y_ + m.zy_ * v.z_,
		m.xz_ * v.x_ + m.yz_ * v.y_ + m.zz_ * v.z_
	);
}


/// @brief xyzMatrix^T * xyzVector in-place transpose product
/// @note  Input xyzVector is modified
template< typename T >
inline
xyzVector< T > &
inplace_transpose_product( xyzMatrix< T > const & m, xyzVector< T > & v )
{
	T const x = m.xx_ * v.x_ + m.yx_ * v.y_ + m.zx_ * v.z_;
	T const y = m.xy_ * v.x_ + m.yy_ * v.y_ + m.zy_ * v.z_;
	v.z_      = m.xz_ * v.x_ + m.yz_ * v.y_ + m.zz_ * v.z_;
	v.x_ = x;
	v.y_ = y;
	return v;
}


/// @brief xyzVector xyzVector outer product
template< typename T >
inline
xyzMatrix< T >
outer_product( xyzVector< T > const & a, xyzVector< T > const & b )
{
	return xyzMatrix< T >(
		a.x_ * b.x_, a.x_ * b.y_, a.x_ * b.z_,
		a.y_ * b.x_, a.y_ * b.y_, a.y_ * b.z_,
		a.z_ * b.x_, a.z_ * b.y_, a.z_ * b.z_
	);
}


/// @brief Projection matrix onto the line through a vector
template< typename T >
inline
xyzMatrix< T >
projection_matrix( xyzVector< T > const & v )
{
	return ( xyzMatrix< T >(
	 v.x_ * v.x_, v.x_ * v.y_, v.x_ * v.z_,
	 v.y_ * v.x_, v.y_ * v.y_, v.y_ * v.z_,
	 v.z_ * v.x_, v.z_ * v.y_, v.z_ * v.z_
	) /= v.length_squared() );
}


// Angles and Rotations


/// @brief Dihedral (torsion) angle in radians: angle value passed
/// @note  Given four positions in a chain ( p1, p2, p3, p4 ), calculates the dihedral
///   (torsion) angle in radians between the vectors p2->p1 and p3->p4 while sighting
///   along the axis defined by the vector p2->p3 (positive indicates right handed twist)
/// @note  Angle returned is on [ -pi, pi ]
/// @note  Degenerate cases are handled and assigned a zero angle but assumed rare
///   (wrt performance tuning)
/// @note For a reference on the determination of the dihedral angle formula see:
///   http://www.math.fsu.edu/~quine/IntroMathBio_04/torsion_pdb/torsion_pdb.pdf
template< typename T >
inline
void
dihedral_angle(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3,
	xyzVector< T > const & p4,
	T & angle
)
{
	static T const ZERO( 0 );

	// Three normalized directional vectors hold the relative directions of
	// the points (the dihedral angle formula used below is only valid for
	// normalized vectors)
	xyzVector< T > const a( ( p2 - p1 ).normalize_or_zero() );
	xyzVector< T > const b( ( p3 - p2 ).normalize_or_zero() );
	xyzVector< T > const c( ( p4 - p3 ).normalize_or_zero() );

	// Compute the dihedral angle: Degenerate cases are assigned a zero angle by definition
	// Degenerate cases: Coincident adjacent points or parallel adjacent directional vectors
	if ( ( b == ZERO ) ) { // p2 == p3: Handle specially: atan2( 0, -dot( a, c ) ) == 0 or pi
		angle = ZERO;
	} else { // Use the formula
		// Degenerate cases a == 0, c == 0, a||b, and b||c give x == y == 0
		T const x = -dot( a, c ) + ( dot( a, b ) * dot( b, c ) );
		T const y = dot( a, cross( b, c ) );
		// Angle in [ -pi, pi ]
		angle = ( ( y != ZERO ) || ( x != ZERO ) ? std::atan2( y, x ) : ZERO );
	}
}


/// @brief Dihedral (torsion) angle in radians: angle value returned
/// @note  Given four positions in a chain ( p1, p2, p3, p4 ), calculates the dihedral
///   (torsion) angle in radians between the vectors p2->p1 and p3->p4 while sighting
///   along the axis defined by the vector p2->p3 (positive indicates right handed twist)
/// @note  Angle returned is on [ -pi, pi ]
/// @note  Degenerate cases are handled and assigned a zero angle but assumed rare
///   (wrt performance tuning)
/// @note For a reference on the determination of the dihedral angle formula see:
///   http://www.math.fsu.edu/~quine/IntroMathBio_04/torsion_pdb/torsion_pdb.pdf
template< typename T >
inline
T
dihedral_angle(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3,
	xyzVector< T > const & p4
)
{
	T angle( 0 );
	dihedral_angle( p1, p2, p3, p4, angle );
	return angle;
}


/// @brief Dihedral (torsion) angle in degrees: angle value passed
/// @note  This is a Rosetta2 compatibility version that operates in degrees
/// @note  Given four positions in a chain ( p1, p2, p3, p4 ), calculates the dihedral
///   (torsion) angle in degrees between the vectors p2->p1 and p3->p4 while sighting
///   along the axis defined by the vector p2->p3 (positive indicates right handed twist)
/// @note  Angle returned is on [ -180, 180 ]
/// @note  Degenerate cases are handled and assigned a zero angle but assumed rare
///   (wrt performance tuning)
/// @note For a reference on the determination of the dihedral angle formula see:
///   http://www.math.fsu.edu/~quine/IntroMathBio_04/torsion_pdb/torsion_pdb.pdf
template< typename T >
inline
void
dihedral(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3,
	xyzVector< T > const & p4,
	T & angle
)
{
	dihedral_angle( p1, p2, p3, p4, angle );
	angle = conversions::degrees( angle );
}


/// @brief Dihedral (torsion) angle in degrees: angle value returned
/// @note  This is a Rosetta2 compatibility version that operates in degrees
/// @note  Given four positions in a chain ( p1, p2, p3, p4 ), calculates the dihedral
///   (torsion) angle in degrees between the vectors p2->p1 and p3->p4 while sighting
///   along the axis defined by the vector p2->p3 (positive indicates right handed twist)
/// @note  Angle returned is on [ -180, 180 ]
/// @note  Degenerate cases are handled and assigned a zero angle but assumed rare
///   (wrt performance tuning)
/// @note For a reference on the determination of the dihedral angle formula see:
///   http://www.math.fsu.edu/~quine/IntroMathBio_04/torsion_pdb/torsion_pdb.pdf
template< typename T >
inline
T
dihedral(
	xyzVector< T > const & p1,
	xyzVector< T > const & p2,
	xyzVector< T > const & p3,
	xyzVector< T > const & p4
)
{
	return conversions::degrees( dihedral_angle( p1, p2, p3, p4 ) );
}


/// @brief Rotation matrix about a helical axis through the origin through an angle
template< typename T >
inline
xyzMatrix< T >
rotation_matrix( xyzVector< T > const & axis, T const & theta )
{
	xyzVector< T > const n( axis.normalized() );

	T const sin_theta( std::sin( theta ) );
	T const cos_theta( std::cos( theta ) );

	xyzMatrix< T > R( projection_matrix( n ) *= ( T( 1 ) - cos_theta ) );

	R.xx_ += cos_theta;        R.xy_ -= sin_theta * n.z_; R.xz_ += sin_theta * n.y_;
	R.yx_ += sin_theta * n.z_; R.yy_ += cos_theta;        R.yz_ -= sin_theta * n.x_;
	R.zx_ -= sin_theta * n.y_; R.zy_ += sin_theta * n.x_; R.zz_ += cos_theta;

	return R;
}


/// @brief Transformation from rotation matrix to helical axis of rotation
/// @note  Input matrix must be orthogonal
/// @note  Angle of rotation is also returned
/// @note  Orientation of axis chosen so that the angle of rotation is non-negative [0,pi]
template< typename T >
inline
xyzVector< T >
rotation_axis( xyzMatrix< T > const & R, T & theta )
{
	using std::abs;
	using std::acos;
	using std::sqrt;

	// This would be good here but slow and unclear what tolerance to use
	// assert( rm.orthogonal() );

	static T const ZERO( 0 );
	static T const ONE( 1 );
	static T const TWO( 2 );

	T const cos_theta = sin_cos_range( ( R.trace() - ONE ) / TWO );

	T const tolerance = NumericTraits< T >::sin_cos_tolerance();
	if ( cos_theta > -ONE + tolerance && cos_theta < ONE - tolerance ) {
		// Compute sign and absolute value of axis vector elements from matrix elements
		// Sign of axis vector is chosen to correspond to a positive sin_theta value
		T x = ( R.zy_ > R.yz_ ? ONE : -ONE ) *
		 sqrt( ( R.xx_ - cos_theta ) / ( ONE - cos_theta ) );
		T y = ( R.xz_ > R.zx_ ? ONE : -ONE ) *
		 sqrt( ( R.yy_ - cos_theta ) / ( ONE - cos_theta ) );
		T z = ( R.yx_ > R.xy_ ? ONE : -ONE ) *
		 sqrt( ( R.zz_ - cos_theta ) / ( ONE - cos_theta ) );
		// Above method appears to cover a greater range of cases than the original method:
		//
		// return ( xyzVector< T >( R.zy_ - R.yz_, R.xz_ - R.zx_, R.yx_ - R.xy_ )
		//          /= ( TWO * sine_theta ) );
		//
		// and is more stable for small sine_theta

		theta = acos( cos_theta );
		assert( abs( x*x + y*y + z*z - 1 ) <= T( 0.01 ) );
		return xyzVector< T >( x, y, z );
	} else if ( cos_theta >= ONE - tolerance ) {
		// R is the identity matrix, return an arbitrary axis of rotation
		theta = ZERO;
		return xyzVector< T >( ONE, ZERO, ZERO );
	} else { // cos_theta <= -ONE + tolerance
		// R is of the form 2*n*n^T - I, theta == pi
		xyzMatrix< T > const nnT( R.clone().add_diagonal( ONE, ONE, ONE ) /= TWO );
		T x, y, z;
		// Since theta = pi, both n and -n are equally valid axis vectors
		// Use convention to take first non-zero coordinate positive
		if ( nnT.xx_ > ZERO + tolerance ) {
			// Note: x is positive, but negative would also work
			x = sqrt( nnT.xx_ );
			y = nnT.yx_ / x;
			z = nnT.zx_ / x;
		} else if ( nnT.yy_ > ZERO + tolerance ) {
			// nnT.xx_ = n.x_ == ZERO, in this case
			x = ZERO;
			// Note: y is positive, but negative would also work
			y = sqrt( nnT.yy_ );
			z = nnT.zy_ / y;
		} else { // nnT.yy_ = n.y_ == ZERO, also, in this case
			// If we get here nnT.zz_ must be positive!
			assert( nnT.zz_ > ZERO + tolerance );
			x = ZERO;
			y = ZERO;
			// Note: z is positive, but negative would also work
			z = sqrt( nnT.zz_ );
		}
		theta = NumericTraits< T >::pi(); // theta == pi
		// For a valid orthogonal matrix R, axis should be a normal vector
		assert( abs( x*x + y*y + z*z - 1 ) <= T( 0.01 ) );
		return xyzVector< T >( x, y, z );
	}
}


// Eigenvalues and Eigenvectors


/// @brief Classic Jacobi algorithm for the eigenvalues of a real symmetric matrix
/// @note  Use eigenvector_jacobi if eigenvectors are also desired
template< typename T >
inline
xyzVector< T >
eigenvalue_jacobi( xyzMatrix< T > const & a, T const & tol )
{
	using std::abs;

	// May need a tolerance based test here
	assert( ( a.xy_ == a.yx_ ) && ( a.xz_ == a.zx_ ) && ( a.yz_ == a.zy_ ) );

	// Copy matrix as it will be modified by the algorithm
	xyzMatrix< T > m( a );

	// Initialize the off-diagonal, upper-triangular sum of squares
	T off = ( m.xy_ * m.xy_ ) + ( m.xz_ * m.xz_ ) + ( m.yz_ * m.yz_ );

	int i, j, n_iterations = 0;
	xyzMatrix< T > r;
	while ( off > tol ) {
		++n_iterations;

		// Ensure number of iterations does not exceed 50:
		// otherwise, re-evaluate stopping criterion
		assert( n_iterations <= 50 );

		// Determine index of upper-triangular element that will be zeroed out
		if ( abs( m.xy_ ) >= abs( m.xz_ ) ) {
			if ( abs( m.xy_ ) >= abs( m.yz_ ) ) {
				i = 1; j = 2;
			} else {
				i = 2; j = 3;
			}
		} else if ( abs( m.xz_ ) >= abs( m.yz_ ) ) {
			i = 1; j = 3;
		} else {
			i = 2; j = 3;
		}

		// After four iterations, skip the rotation if the off-diagonal element is small
		T const ij_scaled = abs( T( 100 ) * m(i,j) );
		if ( ( n_iterations > 4 )
		 && abs( m(i,i) ) + ij_scaled == abs( m(i,i) )
		 && abs( m(j,j) ) + ij_scaled == abs( m(j,j) ) ) {
			m(i,j) = m(j,i) = T( 0 );
		} else {
			// Compute the rotation matrix
			jacobi_rotation( m, i, j, r );

			// Zero out the i,j and j,i elements
			m.right_multiply_by( r );
			m.left_multiply_by_transpose( r );
		}

		// Recalculate the off-diagonal, upper-triangular sum of squares
		off = ( m.xy_ * m.xy_ ) + ( m.xz_ * m.xz_ ) + ( m.yz_ * m.yz_ );
	}

	return xyzVector< T >( m.xx_, m.yy_, m.zz_ );
}


/// @brief Classic Jacobi algorithm for the eigenvalues and eigenvectors of a
///   real symmetric matrix
/// @note  Use eigenvalue_jacobi if eigenvectors are not desired
template< typename T >
inline
xyzVector< T >
eigenvector_jacobi( xyzMatrix< T > const & a, T const & tol, xyzMatrix< T > & J )
{
	using std::abs;

	// May need a tolerance based test here
	assert( ( a.xy_ == a.yx_ ) && ( a.xz_ == a.zx_ ) && ( a.yz_ == a.zy_ ) );

	// Copy matrix as it will be modified by the algorithm
	xyzMatrix< T > m( a );

	// Initialize the off-diagonal, upper-triangular sum of squares
	T off = ( m.xy_ * m.xy_ ) + ( m.xz_ * m.xz_ ) + ( m.yz_ * m.yz_ );

	J.to_identity();
	int i, j, n_iterations = 0;
	xyzMatrix< T > r;
	while ( off > tol ) {
		++n_iterations;

		// Ensure number of iterations does not exceed 50:
		// otherwise, re-evaluate stopping criterion
		assert( n_iterations <= 50 );

		// Determine index of upper-triangular element that will be zeroed out
		if ( abs( m.xy_ ) >= abs( m.xz_ ) ) {
			if ( abs( m.xy_ ) >= abs( m.yz_ ) ) {
				i = 1; j = 2;
			} else {
				i = 2; j = 3;
			}
		} else if ( abs( m.xz_ ) >= abs( m.yz_ ) ) {
			i = 1; j = 3;
		} else {
			i = 2; j = 3;
		}

		// After four iterations, skip the rotation if the off-diagonal element is small
		T const ij_scaled = abs( T( 100 ) * m(i,j) );
		if ( ( n_iterations > 4 )
		 && abs( m(i,i) ) + ij_scaled == abs( m(i,i) )
		 && abs( m(j,j) ) + ij_scaled == abs( m(j,j) ) ) {
			m(i,j) = m(j,i) = T( 0 );
		} else {
			// Compute the rotation matrix
			jacobi_rotation( m, i, j, r );

			// Zero out the i,j and j,i elements
			m.right_multiply_by( r );
			m.left_multiply_by_transpose( r );

			// Accumulate the rotation transformations to form the matrix of eigenvectors
			J *= r;
		}

		// Recalculate the off-diagonal, upper-triangular sum of squares
		off = ( m.xy_ * m.xy_ ) + ( m.xz_ * m.xz_ ) + ( m.yz_ * m.yz_ );
	}

	return xyzVector< T >( m.xx_, m.yy_, m.zz_ );
}


/// @brief Jacobi rotation
/// @note  Compute the orthogonal transformation used to zero out a pair of
///        off-diagonal elements
template< typename T >
inline
void
jacobi_rotation( xyzMatrix< T > const & m, int const i, int const j, xyzMatrix< T > & r )
{
	using std::abs;
	using std::sqrt;

	assert( ( i > 0 ) && ( i <= 3 ) );
	assert( ( j > 0 ) && ( j <= 3 ) );
	assert( i != j );

	static T const ZERO( 0 );
	static T const ONE( 1 );

	T const tau = ( m(i,i) - m(j,j) ) / ( 2 * m(i,j) );
	T const t = ( tau < ZERO ? -ONE : ONE ) / ( abs( tau ) + sqrt( ONE + ( tau * tau ) ) );

	T const c = ONE / sqrt( ONE + ( t * t ) );
	T const s = c * t;

	r.to_identity();
	r(i,i) = c; r(i,j) = -s;
	r(j,i) = s; r(j,j) = c;
}


} // namespace numeric


#endif // INCLUDED_numeric_xyz_functions_HH
