// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2005 University of Washington
// (C) 199x-2005 University of California Santa Cruz
// (C) 199x-2005 University of California San Francisco
// (C) 199x-2005 Johns Hopkins University
// (C) 199x-2005 University of North Carolina, Chapel Hill
// (C) 199x-2005 Vanderbilt University

/// @file   testing/unittest/TestSuite.hh
///
/// @brief  Unit test test suites.
/// @author Ion Yannopoulos
/// @note   Currently a wrapper for Boost.Test.
///         http://www.boost.org/libs/test/doc/components/utf/components/test_suite/index.html


#ifndef INCLUDED_testing_unittest_TestSuite_HH
#define INCLUDED_testing_unittest_TestSuite_HH


// Project forward headers
//- #include <testing/unittest/TestSuite.fwd.hh>
 
 
// Project headers
#include <testing/unittest/TestCase.hh>

// External library headers
#include <boost/config.hpp>			// BOOST_JOIN
#include <boost/test/unit_test_suite.hpp>



// Macros

// --- Default test suites

/// @brief The most common test suite is a free function.
# define TEST_SUITE(SUITE)											\
  TEST_SUITE_CLASS(SUITE)


/// @brief Open a test suite.
#define TEST_SUITE_BEGIN(SUITE)									\
  TEST_SUITE_CLASS_BEGIN(SUITE)

/// @brief Close off a test suite.
#define TEST_SUITE_END													\
	TEST_SUITE_CLASS_END


// --- Free functions as test suites


/// @brief Declare a test suite as a function or methiod
# define TEST_SUITE_FUNCTION(FUNCTION)					\
  testing::unittest::TestSuite * FUNCTION();


#define TEST_SUITE_FUNCTION_BEGIN(FUNCTION)			\
  testing::unittest::TestSuite * FUNCTION()								\
  {																							\
		testing::unittest::TestSuite * suite =								\
	    BOOST_TEST_SUITE(#FUNCTION);


#define TEST_SUITE_FUNCTION_END									\
    return suite;																\
	}



/*
// --- Member functions as test suites

/// @brief Declare a test suite as method
# define TEST_SUITE_METHOD(CLASS, METHOD)				\
  void METHOD();

/// @brief Define the beginning of member function as test suite.
#define TEST_SUITE_METHOD_BEGIN(CLASS, METHOD)	\
	testing::unittest::TestSuite * CLASS::METHOD()          \
	{																							\
		TestSuite * suite =													\
	    BOOST_TEST_SUITE(#CLASS ## #METHOD);

/// @brief Define the end of member function as test suite.
#define TEST_SUITE_METHOD_END										\
    return suite;																\
	}
*/


// --- Classes as test suites


#define TEST_SUITE_CLASS(SUITE)									\
	class SUITE : public testing::unittest::TestSuite				\
	{																							\
	public:																				\
	  SUITE()																			\
	    : testing::unittest::TestSuite(#SUITE)							\
	  {																						\
      init();																		\
    }																						\
    ~SUITE()																		\
	  {																						\
	  }																						\
  private:																			\
    SUITE(SUITE const &);												\
		SUITE & operator=(SUITE const &);						\
    void init();																\
  };


#define TEST_SUITE_CLASS_BEGIN(SUITE)						\
  void																					\
  SUITE::init()																	\
  {																							\
    testing::unittest::TestSuite * suite = this;


#define TEST_SUITE_CLASS_END										\
	}


// --- Adding cases and suites to test suites


/// @brief Default for cases is functions
#define TEST_SUITE_USES_CASE(CASE)							\
	TEST_SUITE_USES_FUNCTION_CASE(CASE)


#define TEST_SUITE_USES_FUNCTION_CASE(FUNCTION)	\
  testing::unittest::TestCase * FUNCTION##_case =					\
    BOOST_TEST_CASE(&FUNCTION);									\
  suite->add(FUNCTION##_case);

/*
// XXX: I'd like to use BOOST_CLASS_TEST_CASE -- but it just doesn't work.
// XXX: The compiler won't call the class version of make_test_case,
// XXX: always selecting the overload for plain functions.
#define TEST_SUITE_USES_METHOD_CASE(CLASS, METHOD)	\
  CLASS CLASS##_##METHOD##_instance; \
  testing::unittest::TestCase * CLASS##_##METHOD##_case = \
    BOOST_CLASS_TEST_CASE(&CLASS::METHOD, &CLASS##_##METHOD##_instance);	\
  suite->add(CLASS##_##METHOD##_case);
*/


#define TEST_SUITE_USES_METHOD_CASE(CLASS, METHOD)	\
  CLASS CLASS##_##METHOD##_instance;								\
  testing::unittest::TestCase * CLASS##_##METHOD##_case =			\
		boost::unit_test::make_test_case<CLASS>(				\
      &CLASS::METHOD,																\
      BOOST_STRINGIZE(CLASS::METHOD),								\
     	boost::shared_ptr<CLASS>(											\
			  &CLASS##_##METHOD##_instance								\
      ) 																						\
    );																							\
  suite->add(CLASS##_##METHOD##_case);


/// @brief Default for suite is classes
#define TEST_SUITE_USES_SUITE(NAMESPACE, SUITE)	\
  TEST_SUITE_USES_CLASS_SUITE(NAMESPACE, SUITE)


#define TEST_SUITE_USES_FUNCTION_SUITE(NAMESPACE, FUNCTION) \
  testing::unittest::TestSuite * FUNCTION##_suite =			\
    BOOST_TEST_SUITE(																				\
      BOOST_STRINGIZE(NAMESPACE::FUNCTION)									\
    );																											\
  suite->add(FUNCTION##_suite);


#define TEST_SUITE_USES_METHOD_SUITE(NAMESPACE, CLASS, METHOD)	\
  testing::unittest::TestSuite * CLASS##_##METHOD##_suite =	\
    BOOST_TEST_SUITE(																						\
      BOOST_STRINGIZE(NAMESPACE::CLASS::METHOD);								\
	  );																													\
  suite->add(CLASS##_##METHOD##_suite);


#define TEST_SUITE_USES_CLASS_SUITE(NAMESPACE, CLASS) \
	NAMESPACE::CLASS * CLASS##_suite =		\
    new NAMESPACE::CLASS();														\
  suite->add(CLASS##_suite);  


/// @brief Set up a dependency between two test cases
/// @note DEPENDS has to be called AFTER all cases have been defined with USES_*.
#define TEST_SUITE_DEPENDS(CASE1, CASE2)				\
	CASE1##_case->depends_on(CASE2##_case);


namespace testing {
namespace unittest {


typedef boost::unit_test::test_suite TestSuite;


} // namespace unittest
} // namespace testing



#endif // INCLUDED_testing_unittest_TestSuite_HH
