// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/Bound.hh
/// @brief  Bound value class
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_utility_Bound_HH
#define INCLUDED_utility_Bound_HH


// Unit headers
#include <utility/Bound.fwd.hh>

// C++ headers
#include <cassert>


namespace utility {


/// @brief Bound value class
template< typename T >
class Bound
{


public: // Types


	// Project style
	typedef  T  Value;

	// STL/Boost style
	typedef  T  value_type;


public: // Creation


	/// @brief Default constructor
	inline
	Bound() :
		active_( false ),
		value_( Value() ),
		strict_( false )
	{}


	// Value constructor
	inline
	explicit
	Bound(
		Value const & value_a,
		bool const strict_a = false
	) :
		active_( true ),
		value_( value_a ),
		strict_( strict_a )
	{}


	// Strict named constructor
	inline
	static
	Bound
	Strict( Value const & value_a )
	{
		return Bound( value_a, true );
	}


	/// @brief Destructor
	inline
	~Bound()
	{}


public: // Methods


	/// @brief Value assignment
	inline
	Bound &
	value(
		Value const & value_a,
		bool const strict_a = false
	)
	{
		active_ = true;
		value_ = value_a;
		strict_ = strict_a;
		return *this;
	}


	/// @brief Value assignment
	inline
	Bound &
	operator ()(
		Value const & value_a,
		bool const strict_a = false
	)
	{
		active_ = true;
		value_ = value_a;
		strict_ = strict_a;
		return *this;
	}


	/// @brief Activate
	inline
	Bound &
	activate()
	{
		active_ = true;
		return *this;
	}


	/// @brief Deactivate
	inline
	Bound &
	deactivate()
	{
		active_ = false;
		return *this;
	}


	/// @brief Clear
	inline
	Bound &
	clear()
	{
		active_ = false;
		value_ = Value();
		strict_ = false;
		return *this;
	}


public: // Properties


	/// @brief Active bound?
	inline
	bool
	active() const
	{
		return active_;
	}


	/// @brief Inactive bound?
	inline
	bool
	inactive() const
	{
		return ( ! active_ );
	}


	/// @brief Bound value
	inline
	Value const &
	operator ()() const
	{
		assert( active_ );
		return value_;
	}


	/// @brief Bound value
	inline
	Value const &
	value() const
	{
		assert( active_ );
		return value_;
	}


	/// @brief Strict inequality (< or >) bound?
	inline
	bool
	strict() const
	{
		return strict_;
	}


private: // Fields


	/// @brief Active bound?
	bool active_;

	/// @brief Bound value
	Value value_;

	/// @brief Strict inequality (< or >) bound?
	bool strict_;


}; // Bound


} // namespace utility


#endif // INCLUDED_utility_Bound_HH
