// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/file/file_sys_util.cc
/// @brief  Platform independent operations on files (except I/O)
/// @author David Kim (dekim@u.washington.edu)
/// @author Ion Yannopoulos (ion@rosettacommons.org)
/// @todo   Break out platform-specific code.
/// @todo   trytry_* functions create mutual dependency between
///         utility/file and utility/io.  Resolve this.


// Unit headers
#include <utility/file/file_sys_util.hh>

// Project headers
#include <utility/basic_sys_util.hh>
#include <utility/exit.hh>
#include <utility/random.functions.hh>
#include <utility/io/ozstream.hh>

// C++ headers
#include <iostream>

// Platforms headers
#include <errno.h>
#include <sys/stat.h>

// Platform headers - Win32
#ifndef _WIN32
// #include <unistd.h>
#endif // _WIN32

// Boinc headers
#ifdef BOINC
#include <utility/boinc/boinc_util.hh>
#endif // BOINC


namespace utility {
namespace file {


/// @brief Does File Exist?
bool
file_exists( std::string const & path )
{
	struct stat buf;
	return !stat( path.c_str(), &buf ); // stat() returns zero on success
}


/// @brief Delete File
int
file_delete( std::string const & path )
{
	using utility::drand;
	using utility::sys_sleep;

	if ( !file_exists( path ) ) {
		return 0;
	}

	int retval;
#ifdef _WIN32
	for ( int i = 0; i < 5; ++i ) {
		retval = remove( path.c_str() );
		if ( !retval ) break;
		sys_sleep( drand() ); // avoid lockstep
	}
#else
	retval = remove( path.c_str() );
#endif // _WIN32
	return retval;
}


/// @brief Extension of a File Name
std::string
file_extension( std::string const & filename )
{
	using std::string;

	string::size_type const iver = filename.find_last_of( '~' );
	if ( iver == string::npos ) { // No version
		string::size_type const iext = filename.find_last_of( '.' );
		if ( iext == string::npos ) { // No extension
			return string();
		} else { // Return the extension (without the '.')
			return filename.substr( iext + 1 );
		}
	} else { // Find extension before version
		string::size_type const iext = filename.find_last_of( '.', iver );
		if ( iext == string::npos ) { // No extension
			return string();
		} else { // Return the extension (without the '.')
			return filename.substr( iext + 1, iver - iext - 1 );
		}
	}
}


/// @brief Platform independent way of getting file size
long
file_size( std::string const & filename )
{
	using std::cerr;
	using std::endl;
	using std::ifstream;
	using std::ios_base;

	// Skip if file does not exist
	if ( !file_exists( filename ) ) return -1;
	long l, m;
	ifstream file_;
	if ( !trytry_ifstream_open( file_, filename, ios_base::in|ios_base::binary ) ) {
		cerr << "WARNING! cannot get file size for " << filename << ": could not open file." << endl;
		return -1;
	}
	l = file_.tellg();
	file_.seekg( 0, ios_base::end );
	m = file_.tellg();
	file_.close();
	return m - l;
}


/// @brief Create a blank file if it doesn't already exist
bool
create_blank_file( std::string const & blank_file )
{
	utility::io::ozstream blank_out_stream;
	blank_out_stream.open( blank_file );
	if ( !blank_out_stream ) {
		std::cout << "Open failed for file: " << blank_file << std::endl;
		std::cerr << "Open failed for file: " << blank_file << std::endl;
		utility_exit();
	}
	blank_out_stream << std::endl;
	blank_out_stream.close();
	blank_out_stream.clear();
	return true;
}


/// @brief Try to open file a few times just in case it is locked (from BOINC LIB)
bool
trytry_ifstream_open(
	std::ifstream & ifstream_,
	std::string const & name,
	std::ios_base::openmode open_mode
)
{
	using std::ios;
	using std::ios_base;
	using std::string;
	using utility::drand;
	using utility::sys_sleep;

	ifstream_.close();
	ifstream_.clear();

	string resolved_name( name );

#ifdef BOINC
	// Files that are not temporary need to have resolved names.
	// Resolve them here since all file input should use this function.
	// Be sure to resolve file names used in other functions
	boinc::resolve_filename( resolved_name );
#endif // BOINC

	// If opening for read, and file isn't there,
	// leave now (avoid 5-second delay!!)
	if ( open_mode & ios::in ) {
		if ( !file_exists( resolved_name ) ) {
			ifstream_.setstate( ios_base::failbit ); // set ios state to failbit
			return false;
		}
	}

	ifstream_.open( resolved_name.c_str(), open_mode );
	if ( ifstream_ && ifstream_.is_open() ) return true;

#ifdef _WIN32
	// On Windows: if open fails, try again for 5 seconds
	// (since the file might be open by FastFind, Diskeeper etc.)
	if ( !ifstream_ ) {
		for ( int i = 0; i < 5; ++i ) {
			utility::sys_sleep( utility::drand() );
			ifstream_.open( resolved_name.c_str(), open_mode );
			if ( ifstream_ && ifstream_.is_open() ) return true;
		}
	}
#else
	// Unix - if call was interrupted, retry a few times
	if ( !ifstream_ ) {
		for ( int i = 0; i < 5; ++i ) {
			if ( errno != EINTR ) break;
			ifstream_.open( resolved_name.c_str(), open_mode );
			if ( ifstream_ && ifstream_.is_open() ) return true;
		}
	}
#endif // _WIN32
	ifstream_.setstate( ios_base::failbit ); // set ios state to failbit
	return false;
}


/// @brief Try to open file a few times just in case it is locked (from BOINC LIB)
bool
trytry_ofstream_open(
	std::ofstream & ofstream_,
	std::string const & name,
	std::ios_base::openmode open_mode
)
{
	using std::ios;
	using std::ios_base;
	using std::string;
	using utility::drand;
	using utility::sys_sleep;

	ofstream_.close();
	ofstream_.clear();

	string resolved_name( name );

#ifdef BOINC
	// Files that are not temporary need to have resolved names.
	// Resolve them here since all file output should use this function.
	// Be sure to resolve file names used in other functions
	boinc::resolve_filename( resolved_name );
#endif // BOINC

	// If opening for read, and file isn't there,
	// leave now (avoid 5-second delay!!)
	if ( open_mode & ios::in ) {
		if ( !file_exists( resolved_name ) ) {
			ofstream_.setstate( ios_base::failbit ); // set ios state to failbit
			return false;
		}
	}

	ofstream_.open( resolved_name.c_str(), open_mode );
	if ( ofstream_ && ofstream_.is_open() ) return true;

#ifdef _WIN32
	// On Windows: if open fails, try again for 5 seconds
	// (since the file might be open by FastFind, Diskeeper etc.)
	if ( !ofstream_ ) {
		for ( int i = 0; i < 5; ++i ) {
			utility::sys_sleep( utility::drand() );
			ofstream_.open( resolved_name.c_str(), open_mode );
			if ( ofstream_ && ofstream_.is_open() ) return true;
		}
	}
#else
	// Unix - if call was interrupted, retry a few times
	if ( !ofstream_ ) {
		for ( int i = 0; i < 5; ++i ) {
			if ( errno != EINTR ) break;
			ofstream_.open( resolved_name.c_str(), open_mode );
			if ( ofstream_ && ofstream_.is_open() ) return true;
		}
	}
#endif // _WIN32
	ofstream_.setstate( ios_base::failbit ); // set ios state to failbit
	return false;
}


} // namespace file
} // namespace utility
