// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/io/ozstream.hh
/// @brief  Output file stream wrapper for uncompressed and compressed files
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author David Kim (dekim@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_utility_io_ozstream_HH
#define INCLUDED_utility_io_ozstream_HH


// Unit headers
#include <utility/io/ozstream.fwd.hh>

// Package headers
#include <utility/io/orstream.hh>

// Project headers
#include <utility/file/gzip_util.hh>

// Utility headers
#include <utility/io/zipstream.hpp>

// C++ headers
#include <fstream>


namespace utility {
namespace io {


/// @brief ozstream: Output file stream wrapper for uncompressed and compressed files
class ozstream :
	public orstream
{


private: // Friends


	friend long utility::file::gzip( std::string const & uncompressedfile, bool overwrite );
	friend long utility::file::gunzip( std::string const & compressedfile, bool overwrite );


private: // Types


	enum Compression { NONE, UNCOMPRESSED, GZIP };


public: // Creation


	/// @brief Default constructor
	inline
	ozstream() :
		compression_( NONE ),
		zip_stream_p_( 0 )
	{}


	/// @brief Filename constructor
	inline
	explicit
	ozstream(
		std::string const & filename_a,
		std::ios_base::openmode open_mode = std::ios_base::out // Ignored for gzip files
	) :
		compression_( NONE ),
		zip_stream_p_( 0 )
	{
		open( filename_a, open_mode );
	}


	/// @brief Destructor
	inline
	virtual
	~ozstream()
	{
		if ( zip_stream_p_ ) {
			zip_stream_p_->zflush_finalize();
			delete zip_stream_p_;
		}
		of_stream_.close(); // must close the stream or gzip will be incomplete
		of_stream_.clear();
	}


public: // Methods: conversion


	/// @brief bool conversion
	inline
	operator bool() const
	{
		return ( zip_stream_p_ ? zip_stream_p_->good() : of_stream_.good() );
	}


	/// @brief Stream conversion
	inline
	operator std::ostream const &() const
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream const & >( *zip_stream_p_ )
		 : static_cast< std::ostream const & >( of_stream_ ) );
	}


	/// @brief Stream conversion
	inline
	operator std::ostream &()
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream & >( *zip_stream_p_ )
		 : static_cast< std::ostream & >( of_stream_ ) );
	}


public: // Methods: formatting


	/// @brief Stream output: override to preserve type of return value
	template< typename T >
	inline
	ozstream &
	operator <<( T const & t )
	{
		if ( zip_stream_p_ ) {
			(*zip_stream_p_) << t;
		} else {
			of_stream_ << t;
		}
		return *this;
	}


	/// @brief Stream output overload to replace std::endl with \n to avoid flushing
	/// @brief and call ozstream::flush() when passed std::flush
	inline
	ozstream &
	operator <<( manipulator m )
	{
		static manipulator const std_endl = std::endl;
		static manipulator const std_flush = std::flush;
		if ( ( m == std_endl ) && ( zip_stream_p_ ) ) { // Output newline instead
			(*zip_stream_p_) << '\n';
		} else if ( ( m == std_flush ) && ( zip_stream_p_ ) ) {
			flush(); // ozstream::flush()
		} else {
			of_stream_ << m;
		}
		return *this;
	}


public: // Methods: i/o


	/// @brief Open a file
	void
	open(
		std::string const & filename_a,
		std::ios_base::openmode open_mode = std::ios_base::out
	);


	/// @brief Open a text file for appending
	void
	open_append( std::string const & filename_a );


	/// @brief Write a char
	inline
	ozstream &
	put( char const c )
	{
		if ( zip_stream_p_ ) {
			zip_stream_p_->put( c );
		} else {
			of_stream_.put( c );
		}
		return *this;
	}


	/// @brief Write a string
	inline
	ozstream &
	write( char const * str, std::streamsize const count )
	{
		if ( zip_stream_p_ ) {
			zip_stream_p_->write( str, count );
		} else {
			of_stream_.write( str, count );
		}
		return *this;
	}


	/// @brief Write a string
	inline
	ozstream &
	write( std::string const & str, std::streamsize const count )
	{
		stream().write( str.c_str(), count );
		return *this;
	}


	/// @brief Flush the stream -- currently alias to flush_finalize()
	/// @details Instead doing a regular flush, we currently force a
	/// @details completion of the zip stream.  We do this to pre-empt
	/// @details against several things: (1) the expensive operation
	/// @details of closing and re-opening a stream (2) confusion and
	/// @details inconvenience that may result from users calling flushes
	/// @details and ending upon with broken or corrupted files if a
	/// @details job is somehow interrupted (e.g. on a cluster).
	/// @details Please note that calling flush_finalize() too often
	/// @details can seriously degrade compression.  zlib documentation
	/// @details appears to imply that every 1MB or so is a reasonable
	/// @details rule of thumb.
	inline
	ozstream &
	flush()
	{
		// comment out the zflush_finalize() containing line and uncomment
		// the flush() containing line to switch to "regular" flush behavior
		if ( zip_stream_p_ ) zip_stream_p_->zflush_finalize();
//		if ( zip_stream_p_ ) zip_stream_p_->zflush();
		of_stream_.flush();
		return *this;
	}


	/// @brief Flush the streams and finalize the zip stream
	/// @details Calling this will complete the zip stream.
	/// @details Upon the next write, a new zip stream will be started.
	/// @details Please note that calling flush_finalize() too often
	/// @details can seriously degrade compression.  zlib documentation
	/// @details appears to imply that every 1MB or so is a reasonable
	/// @details rule of thumb.
	inline
	ozstream &
	flush_finalize()
	{
		if ( zip_stream_p_ ) zip_stream_p_->zflush_finalize();
		of_stream_.flush();
		return *this;
	}


	/// @brief Flush the zip_ostream
	/// @details this will flush but *not* finalize the zip stream
	inline
	void
	zflush()
	{
		if ( zip_stream_p_ ) zip_stream_p_->zflush();
	}


	/// @brief Flush and finalize the zip_stream
	/// @details Calling this will complete the zip stream.
	/// @details Upon the next write, a new zip stream will be started
	/// @details Please note that calling zflush_finalize() too often
	/// @details can seriously degrade compression.  zlib documentation
	/// @details appears to imply that every 1MB or so is a reasonable
	/// @details rule of thumb.
	inline
	void
	zflush_finalize()
	{
		if ( zip_stream_p_ ) zip_stream_p_->zflush_finalize();
	}

	/// @brief Clear the stream
	inline
	void
	clear()
	{
		of_stream_.clear();
		if ( zip_stream_p_ ) zip_stream_p_->clear();
	}


	/// @brief Close the ofstream and reset the state
	inline
	void
	close()
	{
		if ( zip_stream_p_ ) {
			zip_stream_p_->zflush_finalize();
			delete zip_stream_p_; zip_stream_p_ = 0;
		}
		compression_ = NONE;
		of_stream_.close();
		of_stream_.clear();
		filename_.clear();
	}


public: // Properties


	/// @brief Stream access
	inline
	std::ostream const &
	operator ()() const
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream const & >( *zip_stream_p_ )
		 : static_cast< std::ostream const & >( of_stream_ ) );
	}


	/// @brief Stream access
	inline
	std::ostream &
	operator ()()
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream & >( *zip_stream_p_ )
		 : static_cast< std::ostream & >( of_stream_ ) );
	}


	/// @brief Stream access
	inline
	std::ostream const &
	stream() const
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream const & >( *zip_stream_p_ )
		 : static_cast< std::ostream const & >( of_stream_ ) );
	}


	/// @brief Stream access
	inline
	std::ostream &
	stream()
	{
		return ( zip_stream_p_
		 ? static_cast< std::ostream & >( *zip_stream_p_ )
		 : static_cast< std::ostream & >( of_stream_ ) );
	}


	/// @brief Pointer to the stream buffer
	inline
	std::streambuf *
	rdbuf() const
	{
		return stream().rdbuf();
	}


	/// @brief File name
	inline
	std::string const &
	filename() const
	{
		return filename_;
	}


public: // Properties: predicate


	/// @brief Good?
	inline
	bool
	good() const
	{
		return stream().good();
	}


	/// @brief End of file?
	inline
	bool
	eof() const
	{
		return stream().eof();
	}


	/// @brief Fail?
	inline
	bool
	fail() const
	{
		return stream().fail();
	}


	/// @brief Bad?
	inline
	bool
	bad() const
	{
		return stream().bad();
	}


	/// @brief Compressed?
	inline
	bool
	compressed() const
	{
		return ( compression_ == GZIP );
	}


	/// @brief Uncompressed?
	inline
	bool
	uncompressed() const
	{
		return ( compression_ == UNCOMPRESSED );
	}


	/// @brief gzipped?
	inline
	bool
	gzipped() const
	{
		return ( compression_ == GZIP );
	}


private: // Properties: internal zip stream predicates


	/// @brief Is stream attached to a gzip file?
	inline
	bool
	is_gzip() const
	{
		return ( zip_stream_p_ ? zip_stream_p_->is_gzip() : false );
	}


	/// @brief CRC of the uncompressed data (see zipstream documentation)
	inline
	long
	get_crc() const
	{
		return ( zip_stream_p_ ? zip_stream_p_->get_crc() : 0 );
	}


	/// @brief Uncompressed data size
	inline
	long
	get_in_size() const
	{
		return ( zip_stream_p_ ? zip_stream_p_->get_in_size() : 0 );
	}


	/// @brief Compressed data size
	inline
	long
	get_out_size() const
	{
		return ( zip_stream_p_ ? zip_stream_p_->get_out_size() : 0 );
	}


private: // Fields


	/// @brief Compression state
	Compression compression_;

	/// @brief File stream
	std::ofstream of_stream_;

	/// @brief File name
	std::string filename_;

	/// @brief Zip file stream pointer (owning)
	zlib_stream::zip_ostream * zip_stream_p_;


}; // ozstream


} // namespace io
} // namespace utility


#endif // INCLUDED_utility_io_ozstream_HH
