// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/options/OptionCollection.cc
/// @brief  Program options collection
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Unit headers
#include <utility/options/OptionCollection.hh>

// Package headers
#include <utility/options/keys/OptionKeys.hh>

// ObjexxFCL headers
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/vector0.hh>

// C++ headers
#include <algorithm>
#include <cassert>
#include <cstddef>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <utility>


namespace utility {
namespace options {


	/// @brief Add the built-in options
	void
	OptionCollection::add_built_in_options()
	{
		using namespace OptionKeys;

		{ // Help options
			add( help, "Generate help message and exit" );
		}

		{ // Option display options
			add( options::options, "Option display option group" ).legal( true ).def( true );
			add( options::user, "Show user-specified options" );
			add( options::all, "Show all options" );
			add( options::table::table, "Option table display option group" );
			add( options::table::text, "Generate the option definitions table in text format" );
			add( options::table::Wiki, "Generate the option definitions table in Wiki format" );
			add( options::exit, "Exit after displaying the options" );
		}

		// Check for problems in the option specifications
		check_specs();
	}


//	/// @brief Specify mutually exclusive options
//	void
//	OptionCollection::exclusive(
//		OptionKey const & key1,
//		OptionKey const & key2
//	)
//	{
//		if ( ( (*this)[ key1 ].user() ) && ( (*this)[ key2 ].user() ) ) {
//			std::cerr << "ERROR: Mutually exclusive options specified: " << key1.id() << " and " << key2.id() << std::endl;
//			std::exit( EXIT_FAILURE );
//		}
//	}


//	/// @brief Specify an option that requires a second option to also be specified
//	void
//	OptionCollection::requires(
//		OptionKey const & key1,
//		OptionKey const & key2
//	)
//	{
//		if ( ( (*this)[ key1 ].user() ) && ( ! (*this)[ key2 ].user() ) ) {
//			std::cerr << "ERROR: Option " << key1.id() << " specified requires option " << key2.id() << std::endl;
//			std::exit( EXIT_FAILURE );
//		}
//	}


	/// @brief Check for problems in the option specifications
	void
	OptionCollection::check_specs() const
	{
		// Initializations
		bool error( false );

		// Check specs
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option: Check it
				if ( ! option( key ).legal_specs_report() ) error = true;
			}
		}

		// Exit if an error was detected
		if ( error ) std::exit( EXIT_FAILURE );
	}


	/// @brief Load the user-specified option values
	void
	OptionCollection::load(
		int const argc,
		char * const argv[],
		bool const free_args // Support free argument (without - prefix)?
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		// Put the arguments strings in a list
		ValueStrings arg_strings;
		for ( int iarg = 1; iarg < argc; ++iarg ) {
			arg_strings.push_back( argv[ iarg ] );
		}

		// Load the options
		string cid; // Context option id
		while ( ! arg_strings.empty() ) { // Process the next option
			string const arg_string( arg_strings.front() ); // Lead argument string
			arg_strings.pop_front(); // Remove lead argument
			char const arg_first( arg_string[ 0 ] );
			if ( ( arg_first == '-' ) && ( ! ObjexxFCL::is_double( arg_string ) ) ) { // - prefix: Treat as an option

				// Load the option
				load_option_cl( arg_string, arg_strings, cid );

			} else if ( arg_first == '@' ) { // @ prefix: Treat as an option file

				// Parse argument to get file specification string
				size_type const fb( arg_string.find_first_not_of( "@\"" ) );
				if ( fb == string::npos ) { // -...-
					std::cerr << "ERROR: Unsupported option file specification: " << arg_string << std::endl;
					std::exit( EXIT_FAILURE );
				}
				size_type const fe( arg_string.find_last_not_of( '"' ) );
				string const file_string( fb <= fe ? arg_string.substr( fb, fe - fb + 1 ) : string() );

				// Open the option file
				std::ifstream stream( file_string.c_str() );
				if ( ! stream ) {
					std::cerr << "ERROR: Option file open failed for: " << file_string << std::endl;
					std::exit( EXIT_FAILURE );
				}

				// Read and load the options from the file
				using std::make_pair;
				string key_id( cid ); // Local context id
				string line;
				char const SPACE( ' ' );
				char const TAB( '\t' );
				string const QUOTES( "\"'" );
				string const WHITESPACE( " \t" );
				string const FIRST_SEP( "= \t#" );
				string const FIRST_SEP_NC( "= \t" );
				string const AFTER_SEP( " \t#" );
				enum { UNKNOWN_INDENT, SPACE_INDENT, TAB_INDENT } indent_type( UNKNOWN_INDENT );
				typedef  std::pair< int, std::string >  IndentContext;
				typedef  utility::vector0< IndentContext >  ContextStack;
				ContextStack context_stack( 1, make_pair( 0, cid ) );
				while ( stream ) {
					using ObjexxFCL::has;
					using ObjexxFCL::is_any_of;
					using ObjexxFCL::len_trim_whitespace;
					using ObjexxFCL::strip_whitespace;
					using ObjexxFCL::stripped_whitespace;
					std::getline( stream, line );
					if ( ( stream ) && ( len_trim_whitespace( line ) > 0 ) &&
					 ( stripped_whitespace( line )[ 0 ] == '-' ) ) { // Process option on line
						string const indent( line.substr( 0, line.find_first_not_of( WHITESPACE ) ) ); // Line indentation
						strip_whitespace( line );
						size_type const line_length( line.length() );

						// Tokenize the line into option identifier and values
						size_type vb( line.find_first_of( FIRST_SEP ) ), ve; // Begin/end token indexes
						string const opt_string( vb == string::npos ? line : line.substr( 0, vb ) );
						if ( vb != string::npos ) vb = line.find_first_not_of( FIRST_SEP_NC, vb );
						if ( ( vb != string::npos ) && ( line[ vb ] == '#' ) ) { // Rest is comment
							vb = string::npos;
						}
						ValueStrings val_strings;
						while ( ( vb != string::npos ) && ( line[ vb ] != '#' ) ) { // Tokenize the values
							if ( is_any_of( line[ vb ], QUOTES ) ) { // Quoted value
								char const quote( line[ vb ] ); // Quote character being used
								vb = line.find_first_not_of( QUOTES, vb ); // Skip all the quotes
								if ( vb != string::npos ) {
									ve = line.find( quote, vb ); // Find matching quote
									if ( ve == string::npos ) ve = line_length; // Take rest of line if unclosed quote
									if ( vb < ve ) { // Non-empty value string
										val_strings.push_back( line.substr( vb, ve - vb ) );
									}
									if ( ve < line_length ) {
										ve = line.find_first_not_of( QUOTES, ve ); // Skip past quotes
										if ( ve == string::npos ) ve = line_length; // Make sure we stop looking
									}
								}
							} else { // Non-quoted value
								ve = line.find_first_of( AFTER_SEP, vb );
								if ( ve == string::npos ) ve = line_length; // Take rest of line if no more whitespace
								if ( vb < ve ) { // Non-empty value string
									val_strings.push_back( line.substr( vb, ve - vb ) );
								}
							}
							vb = ( ( ve < line_length ) && ( line[ ve ] != '#' ) ?
							 line.find_first_not_of( WHITESPACE, ve ) :
							 string::npos ); // Start of next token
						}

						// Load the option
						if ( indent.empty() ) { // Indent level == 0: Use command line context
							key_id = cid;
							load_option_file( opt_string, val_strings, key_id, true );
							context_stack.resize( 1 );
							context_stack.push_back( make_pair( 0, key_id ) );
						} else { // Find the indent level and load the option
							if ( indent_type == UNKNOWN_INDENT ) { // Set the indent type
								if ( has( indent, SPACE ) ) {
									indent_type = SPACE_INDENT;
								} else {
									indent_type = TAB_INDENT;
								}
							}
							if ( ( ( indent_type == SPACE_INDENT ) && ( has( indent, TAB ) ) ) ||
							 ( ( indent_type == TAB_INDENT ) && ( has( indent, SPACE ) ) ) ) {
								std::cerr << "ERROR: Option file has mixed space and tab indent characters: "
								 << file_string << std::endl;
								std::exit( EXIT_FAILURE );
							}
							int const n_indent( indent.size() ); // Number of indent characters
							ContextStack::size_type j( context_stack.size() );
							while ( ( j > 0 ) && ( context_stack[ j - 1 ].first >= n_indent ) ) {
								--j;
							}
							if ( j == 0 ) { // Use command line context
								key_id = cid;
								load_option_file( opt_string, val_strings, key_id, true );
							} else { // Use indented @file context
								key_id = context_stack[ j - 1 ].second;
								load_option_file( opt_string, val_strings, key_id );
							}
							if ( j < context_stack.size() ) context_stack.resize( j ); // Remove obs. part of context
							context_stack.push_back( make_pair( n_indent, key_id ) );
						}
					}
				}
			} else if ( ! free_args ) { // Warn about free argument
				std::cerr << "WARNING: Unused \"free\" argument specified: " << arg_string << std::endl;
			}
		}

		// Check for problems in the option values
		check_values();

		{ // Generate any requested option outputs
			using namespace utility::options::OptionKeys;

			// Help
			if ( option( help ) ) { // Display help and exit
				std::cout << "\nUsage:\n\n" << argv[ 0 ] << " [options]\n";
				show_help_hier( std::cout );
				std::exit( EXIT_SUCCESS );
			}

			// Options displays
			bool showed( false );
			if ( option( options::user ) ) { // Show the specified options
				show_user( std::cout );
				showed = true;
			}
			if ( option( options::all ) ) { // Show all the options
				show_all_hier( std::cout );
				showed = true;
			}
			if ( option( options::table::text ) ) { // Show the options definitions table in text format
				show_table_text( std::cout );
				showed = true;
			}
			if ( option( options::table::Wiki ) ) { // Show the options definitions table in Wiki format
				show_table_Wiki( std::cout );
				showed = true;
			}
			if ( ( option( options::table::table ).user() ) &&
			 ( ! option( options::table::text ) ) &&
			 ( ! option( options::table::Wiki ) ) ) { // User specified -options:table : Default to -options:table:text
				show_table_text( std::cout );
				showed = true;
			}
			if ( ( ! showed ) && ( option( options::options ).user() ) ) { // User specified -options : Default to -options:user
				show_user( std::cout );
				showed = true;
			}
			if ( ( showed ) && ( option( options::exit ) ) ) { // Show all the options
				std::exit( EXIT_SUCCESS );
			}

		}

	} // load


	/// @brief Check for problems in the option values
	void
	OptionCollection::check_values() const
	{
		// Initializations
		bool error( false );

		// Check values are legal
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option: Check it
				if ( ! option( key ).legal_report() ) error = true;
			}
		}

		// Exit if an error was detected
		if ( error ) std::exit( EXIT_FAILURE );
	}


	/// @brief Show all the options and their descriptions
	void
	OptionCollection::show_help( std::ostream & stream ) const
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		stream << "\nOptions:   [Specify on command line or in @file]\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					stream << '\n'; // Spacer line between groups
					group = opt_group;
				}
				size_type const d( 2 + opt.id().length() + 3 ); // Description indent spaces
				stream
				 << wrapped(
					" -" + opt.id() + "   " +
					opt.description() +
					space_prefixed( opt.legal_string(), 2 ) +
					space_prefixed( opt.default_string(), 2 ),
					std::min( d, size_type( 20 ) ) )
				 << '\n';
			}
		}
	}


	/// @brief Show all the options and their descriptions in a hierarchy format
	void
	OptionCollection::show_help_hier( std::ostream & stream ) const
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		stream << "\nOptions:   [Specify on command line or in @file]\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					size_type const d( 2 + opt.id().length() + 3 ); // Description indent spaces
					stream << '\n' // Spacer line between groups
					 << wrapped(
						" -" + opt.id() + "   " +
						opt.description() +
						space_prefixed( opt.legal_string(), 2 ) +
						space_prefixed( opt.default_string(), 2 ),
						std::min( d, size_type( 20 ) ) )
					 << '\n';
					group = opt_group;
				} else { // Indent and remove prefix in common with previous id
					size_type const l( n_part( opt.id() ) - 1 ); // Indent level
					size_type const d( l + 2 + suffix( opt.id() ).length() + 3 ); // Description indent spaces
					stream
					 << wrapped(
						string( l, ' ' ) + " -" + suffix( opt.id() ) + "   " +
						opt.description() +
						space_prefixed( opt.legal_string(), 2 ) +
						space_prefixed( opt.default_string(), 2 ),
						std::min( d, size_type( 20 ) ) )
					 << '\n';
				}
			}
		}
	}


	/// @brief Show the user-specified options and their values
	void
	OptionCollection::show_user( std::ostream & stream ) const
	{
		using std::string;

		stream << "\nSpecified Options:\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				if ( opt.user() ) {
					string const opt_group( prefix( opt.id() ) );
					if ( opt_group != group ) { // New group
						stream << '\n'; // Spacer line between groups
						group = opt_group;
					}
					stream << " -" << opt.id() << opt.equals_string() << '\n';
				}
			}
		}
		stream << '\n';
	}


	/// @brief Show all the options and their values
	void
	OptionCollection::show_all( std::ostream & stream ) const
	{
		using std::string;

		stream << "\nOptions:   [Specify on command line or in @file]\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					stream << '\n'; // Spacer line between groups
					group = opt_group;
				}
				stream << " -" << opt.id() << opt.equals_string() << '\n';
			}
		}
		stream << '\n';
	}


	/// @brief Show all the options and their values in a hierarchy format
	void
	OptionCollection::show_all_hier( std::ostream & stream ) const
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		stream << "\nOptions:   [Specify on command line or in @file]\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					stream << '\n'; // Spacer line between groups
					stream << " -" << opt.id() << opt.equals_string() << '\n';
					group = opt_group;
				} else { // Indent and remove prefix in common with previous id
					size_type const l( n_part( opt.id() ) - 1 ); // Indent level
					stream << string( l, ' ' ) << " -" << suffix( opt.id() ) << opt.equals_string() << '\n';
				}
			}
		}
		stream << '\n';
	}


	/// @brief Show the options definitions table in text format
	void
	OptionCollection::show_table_text( std::ostream & stream ) const
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		stream << "\nOption Definitions Table\n";
		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					stream << '\n'; // Spacer line between groups
					stream << " -" << opt.id() << "   " << opt.description() << '\n';
					group = opt_group;
				} else { // Indent and remove prefix in common with previous id
					size_type const l( n_part( opt.id() ) - 1 ); // Indent level
					stream << string( l, ' ' ) << " -" << suffix( opt.id() ) << '\t' << opt.description() << '\t'
					 << opt.type_string() << '\t'
					 << opt.legal_string() << '\t'
					 << opt.default_string()
					 << '\n';
				}
			}
		}
		stream << '\n';
	}


	/// @brief Show the options definitions table in Wiki format
	/// @note  Based on Sergey Lyskov's Python Wiki table generator code
	void
	OptionCollection::show_table_Wiki( std::ostream & stream ) const
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		string group; // Previous option group name
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( has( key ) ) { // Active option
				Option const & opt( option( key ) );
				string const opt_group( prefix( opt.id() ) );
				if ( opt_group != group ) { // New group
					if ( ! group.empty() ) stream << " |}\n"; // Closing previous table if any
					stream << "{| border=\"1\" cellpadding=\"10\" width=\"100%\"\n |+ '''" << prefix( opt.id() ) << " Option Group'''\n";
					stream << " ! Option name\n";
					stream << " ! Description\n";
					stream << " ! Range\n";
					stream << " ! Default\n";
					stream << " ! Old name\n";
					stream << " |-\n";
					group = opt_group;
				}
				stream << " |-\n";
				stream << " | -" << suffix( opt.id() ) << " <" << opt.type_string() << ">\n";
				stream << " | " << opt.description() << '\n';
				stream << " | " << opt.legal_string() << '\n';
				stream << " | " << opt.default_string() << '\n';
				stream << " | \n"; // Don't have the oldName in the C++ option
				stream << " |-\n";
			}
		}
		stream << " |}\n";
	}


	/// @brief Output to stream
	std::ostream &
	operator <<( std::ostream & stream, OptionCollection const & options )
	{
		for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
			OptionKey const & key( *i );
			if ( options.has( key ) ) { // Active option
				Option const & opt( options.option( key ) );
				if ( opt.user() ) stream << " -" << opt.id() << opt.equals_string();
			}
		}
		return stream;
	}


	/// @brief Load a user-specified option argument from a command line
	void
	OptionCollection::load_option_cl(
		std::string const & arg_string, // Lead argument string
		ValueStrings & arg_strings, // Argument strings: Value string(s) in front
		std::string & cid // Context option id
	)
	{
		using std::string;
		using ObjexxFCL::stripped_whitespace;
		typedef  std::string::size_type  size_type;

		// Parse argument into key and value strings
		bool top( false ); // Top-level context?
		size_type kb( arg_string.find_first_not_of( '-' ) );
		char const COLON( ':' );
		if ( ( kb != string::npos ) && ( arg_string[ kb ] == COLON ) ) { // Top level context
			kb = arg_string.find_first_not_of( COLON, kb );
			top = true;
		}
		if ( kb == string::npos ) { // -...-
			std::cerr << "ERROR: Unsupported option specified: " << arg_string << std::endl;
			std::exit( EXIT_FAILURE );
		}
		size_type const ke( arg_string.find_first_of( "= \t" ) );
		bool const ends( ke != string::npos );
		string const key_string( cleaned( ends ? arg_string.substr( kb, ke - kb ) : arg_string.substr( kb ) ) );
		string const val_string( ( ends ) && ( ke + 1 < arg_string.length() ) ?
		 stripped_whitespace( arg_string.substr( ke + 1 ) ) : string() );
		if ( ! val_string.empty() ) arg_strings.push_front( val_string ); // Put in front of value strings

		// Find the option and set its value
		string const key_id( find_key_cl( key_string, cid, top ) );
		set_option_value_cl( key_id, arg_strings );

		// Update the context id
		cid = key_id;
	}


	/// @brief Load a user-specified option argument from an @file
	void
	OptionCollection::load_option_file(
		std::string const & arg_string, // Argument string
		ValueStrings & val_strings, // Value strings
		std::string & cid, // Context option id
		bool const cl_context // Use command line key context?
	)
	{
		using std::string;
		using ObjexxFCL::stripped_whitespace;
		typedef  std::string::size_type  size_type;

		// Parse argument into key and value strings
		bool top( false ); // Top-level context?
		size_type kb( arg_string.find_first_not_of( '-' ) );
		char const COLON( ':' );
		if ( ( kb != string::npos ) && ( arg_string[ kb ] == COLON ) ) { // Top level context
			kb = arg_string.find_first_not_of( COLON, kb );
			top = true;
		}
		if ( kb == string::npos ) { // -...-
			std::cerr << "ERROR: Unsupported option specified: " << arg_string << std::endl;
			std::exit( EXIT_FAILURE );
		}
		size_type const ke( arg_string.find_first_of( "= \t" ) );
		bool const ends( ke != string::npos );
		string const key_string( cleaned( ends ? arg_string.substr( kb, ke - kb ) : arg_string.substr( kb ) ) );
		string const val_string( ( ends ) && ( ke + 1 < arg_string.length() ) ?
		 stripped_whitespace( arg_string.substr( ke + 1 ) ) : string() );
		if ( ! val_string.empty() ) val_strings.push_front( val_string ); // Put in front of value strings

		// Find the option and set its value
		string const key_id( cl_context ? find_key_cl( key_string, cid, top ) : find_key_file( key_string, cid, top ) );
		set_option_value_file( key_id, val_strings );

		// Update the context id
		cid = key_id;
	}


	/// @brief Set a user-specified option value from a command line
	/// note Decides whether to use value strings and erases the ones used
	void
	OptionCollection::set_option_value_cl(
		std::string const & key_id, // Option key id
		ValueStrings & arg_strings // Argument strings: Value string(s) in front
	)
	{
		using std::string;
		using ObjexxFCL::is_any_of;

		// Set the option value
		assert( OptionKeys::has( key_id ) ); // Precondition
		OptionKey const & key( OptionKeys::key( key_id ) );
		if( ! has( key ) ) {
			std::cerr << "ERROR: No option exists for the valid option key -" << key.id() << std::endl;
			std::exit( EXIT_FAILURE );
		}
		Option & opt( option( key ) );
		if ( key.scalar() ) { // Scalar option key
			if ( ( arg_strings.empty() ) || ( ! opt.is_cl_value( arg_strings.front() ) ) )
			{ // Pass empty string: Valid for some options and others will give error
				opt.cl_value( string() );
			} else {
				opt.cl_value( arg_strings.front() ); // Use the argument
				arg_strings.pop_front(); // Remove the argument
			}
		} else { // Vector option key
			assert( key.vector() );
			if ( arg_strings.empty() ) { // No values
				VectorOption & vopt( option< VectorOption >( key ) );
				if ( ( vopt.n() > 0 ) || ( vopt.n_lower() > 0 ) ) {
					std::cerr << "ERROR: No values specified for multi-valued option -"
					 << key.id() << " requiring one or more values" << std::endl;
					std::exit( EXIT_FAILURE );
				}
			} else if ( ! opt.is_cl_value( arg_strings.front() ) ) { // No values of correct type
				std::cerr << "ERROR: No values of the appropriate type "
				 << "specified for multi-valued option -" << key.id() << std::endl;
				std::exit( EXIT_FAILURE );
			} else { // Take value(s)
				// This takes the first value even if the vector is full to trigger an error
				opt.cl_value( arg_strings.front() ); // Use the first argument
				arg_strings.pop_front(); // Remove the first argument
				while ( ( ! arg_strings.empty() ) &&  ( opt.can_hold_another() ) &&
				 ( opt.is_cl_value( arg_strings.front() ) ) ) { // Take another value
					opt.cl_value( arg_strings.front() ); // Use the argument
					arg_strings.pop_front(); // Remove the argument
				}
			}
		}
	}


	/// @brief Set a user-specified option value from an @file
	/// note Uses all value strings and doesn't erase them
	void
	OptionCollection::set_option_value_file(
		std::string const & key_id, // Option key id
		ValueStrings & val_strings // Value strings
	)
	{
		using std::string;

		// Set the option value
		assert( OptionKeys::has( key_id ) ); // Precondition
		OptionKey const & key( OptionKeys::key( key_id ) );
		if( ! has( key ) ) {
			std::cerr << "ERROR: No option exists for the valid option key -" << key.id() << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( key.scalar() ) { // Scalar option key
			if ( val_strings.size() > 1 ) { // Multiple values for a scalar option
				std::cerr << "ERROR: Multiple values specified for option -" << key.id() << std::endl;
				std::exit( EXIT_FAILURE );
			}
			option( key ).cl_value( val_strings.empty() ? string() : *val_strings.begin() );
		} else { // Vector option key
			assert( key.vector() );
			if ( val_strings.empty() ) { // No values for a vector option
				VectorOption & vopt( option< VectorOption >( key ) );
				if ( ( vopt.n() > 0 ) || ( vopt.n_lower() > 0 ) ) {
					std::cerr << "ERROR: No values specified for multi-valued option -"
					 << key.id() << " requiring one or more values" << std::endl;
					std::exit( EXIT_FAILURE );
				}
			}
			for ( ValueStrings::const_iterator i = val_strings.begin(), e = val_strings.end(); i != e; ++i ) {
				option( key ).cl_value( *i );
			}
		}
	}


	/// @brief Check that a key's identifiers are legal
	void
	OptionCollection::check_key( OptionKey const & key )
	{
		using ObjexxFCL::is_double;

		bool error( false );
		if ( is_double( suffix( key.id() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << key.id() << std::endl;
			error = true;
		}
		if ( is_double( suffix( key.identifier() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << key.id() << std::endl;
			error = true;
		}
		if ( is_double( suffix( key.code() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << key.id() << std::endl;
			error = true;
		}
		if ( error ) std::exit( EXIT_FAILURE );
	}


	/// @brief Check that an option's identifiers are legal
	void
	OptionCollection::check_key( Option const & option )
	{
		using ObjexxFCL::is_double;

		bool error( false );
		if ( is_double( suffix( option.id() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << option.id() << std::endl;
			error = true;
		}
		if ( is_double( suffix( option.identifier() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << option.id() << std::endl;
			error = true;
		}
		if ( is_double( suffix( option.code() ) ) ) {
			std::cerr << "ERROR: Options with numeric identifiers are not allowed: -" << option.id() << std::endl;
			error = true;
		}
		if ( error ) std::exit( EXIT_FAILURE );
	}


	/// @brief Find a user-specified option key in a command line context
	/// @note  Searches up the context to find a match
	std::string
	OptionCollection::find_key_cl(
		std::string const & key_string, // Option key string entered
		std::string const & cid, // Context option id
		bool const top // Top-level context?
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		// Find the option key
		string kid; // Matched key name
		if ( ( cid.empty() ) || ( top ) ) { // No context: Search for key with specified key string
			if ( OptionKeys::has( key_string ) ) {
				kid = key_string;
			}
		} else { // Search upwards in context
			string pid( cid ); // Context id prefix being tried
			bool done( false );
			while ( ! done ) {
				string const tid( merged( pid, key_string ) );
				if ( OptionKeys::has( tid ) ) { // Valid option identifier
					if ( ! kid.empty() ) { // Already found a match
						if ( tid != kid ) { // Different id match
							std::cerr << "WARNING: Specified option -" << key_string
							 << " resolved to option -" << kid
							 << " not option -" << tid
							 << " in command line context -" << cid << std::endl;
						}
					} else { // Assign the matched id
						kid = tid;
					}
				}
				if ( pid.empty() ) {
					done = true;
				} else {
					trim( pid ); // Remove last part of prefix for next pass
				}
			}
		}

		if ( kid.empty() ) { // Look for unique best suffix match wrt the context
			size_type const k_part( n_part( key_string ) ); // Number of parts in key string entered
			size_type m_part( 0 ); // Number of prefix parts matching context
			string bid; // Best id match so far
			int n_best( 0 );
			for ( OptionKey::Lookup::ConstIterator i = OptionKeys::begin(), e = OptionKeys::end(); i != e; ++i ) {
				OptionKey const & key( *i );
				string sid;
				if ( suffix( key.id(), k_part ) == key_string ) {
					sid = key.id();
				} else if ( suffix( key.identifier(), k_part ) == key_string ) {
					sid = key.identifier();
				} else if ( suffix( key.code(), k_part ) == key_string ) {
					sid = key.code();
				}
				if ( ! sid.empty() ) {
					size_type const p_part( n_part_prefix_match( cid, sid ) );
					if ( p_part > m_part ) { // New best prefix match level
						m_part = p_part;
						n_best = 0;
						bid = sid;
					} else if ( p_part == m_part ) { // Another match at this prefix match level
						++n_best;
						if ( bid.empty() ) bid = sid; // First match at zero level
					}
				}
			}
			if ( n_best == 1 ) { // Unique best match found
				kid = bid;
			} else if ( n_best > 1 ) { // Nonunique matches found
				std::cerr << "ERROR: Unique best command line context option match not found for -"
				 << key_string << std::endl;
				std::exit( EXIT_FAILURE );
			}
		}

		if ( kid.empty() ) { // No such option
			std::cerr << "ERROR: Option matching -" << key_string;
			if ( cid.empty() ) {
				std::cerr << " not found in command line top-level context" << std::endl;
			} else {
				std::cerr << " not found in command line context -" << cid << std::endl;
			}
			std::exit( EXIT_FAILURE );
		}

		return kid;
	}


	/// @brief Find a user-specified option key in an indented @file context
	/// @note  Looks in the context to find a match
	std::string
	OptionCollection::find_key_file(
		std::string const & key_string, // Option key string entered
		std::string const & cid, // Context option id
		bool const top // Top-level context?
	)
	{
		using std::string;

		// Find the option key
		string kid; // Matched key name
		if ( ( cid.empty() ) || ( top ) ) { // No context: Search for key with specified key string
			if ( OptionKeys::has( key_string ) ) {
				kid = key_string;
			}
		} else { // Look for key in context
			string const tid( merged( cid, key_string ) );
			if ( OptionKeys::has( tid ) ) { // Valid option identifier
				kid = tid;
			}
		}

		// Check if key matched
		if ( kid.empty() ) { // No such option
			std::cerr << "ERROR: Option matching -" << key_string
			 << " not found in indented @file context -" << cid << std::endl;
			std::exit( EXIT_FAILURE );
		}

		return kid;
	}


	/// @brief Number of parts in an option id
	std::string::size_type
	OptionCollection::n_part( std::string const & s )
	{
		if ( s.empty() ) {
			return 1u;
		} else { // Scan the string: Count each transition to : so we can accept -a::b as well as -a:b
			typedef  std::string::size_type  size_type;
			char const COLON( ':' );
			size_type n_part_( s[ 0 ] == COLON ? 2u : 1u );
			for ( size_type i = 1, e = s.size(); i < e; ++i ) {
				if ( ( s[ i ] == COLON ) && ( s[ i - 1 ] != COLON ) ) ++n_part_;
			}
			return n_part_;
		}
	}


	/// @brief Number of prefix parts of two ids that match
	std::string::size_type
	OptionCollection::n_part_prefix_match(
		std::string const & s,
		std::string const & t
	)
	{
		if ( ( s.empty() ) || ( t.empty() ) ) {
			return 0u;
		} else {
			typedef  std::string::size_type  size_type;
			for ( size_type n = std::min( n_part( s ), n_part( t ) ); n > 0; --n ) {
				if ( prefix( s, n ) == prefix( t, n ) ) return n;
			}
			return 0u;
		}
	}


	/// @brief Prefix of an option id with a specified number of parts
	std::string
	OptionCollection::prefix(
		std::string const & s, // String
		int const n // Number of prefix parts desired
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		size_type const n_s( n_part( s ) );
		if ( ( n <= 0 ) || ( s.empty() ) ) { // Nothing
			return string();
		} else if ( size_type( n ) >= n_s ) { // Whole string
			return s;
		} else {
			char const COLON( ':' );
			size_type idx( 0 );
			for ( int i = 1; i <= n; ++i ) {
				idx = s.find( COLON, idx );
				assert( idx != string::npos );
				if ( i < n ) idx = s.find_first_not_of( COLON, idx );
				assert( idx != string::npos );
			}
			return s.substr( 0, idx );
		}
	}


	/// @brief Suffix of an option id with a specified number of parts
	std::string
	OptionCollection::suffix(
		std::string const & s, // String
		int const n // Number of suffix parts desired
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		size_type const n_s( n_part( s ) );

		if ( ( n <= 0 ) || ( s.empty() ) ) { // Nothing
			return string();
		} else if ( size_type( n ) >= n_s ) { // Whole string
			return s;
		} else {
			char const COLON( ':' );
			size_type idx( s.length() - 1 );
			for ( int i = 1; i <= n; ++i ) {
				idx = s.find_last_of( COLON, idx );
				assert( idx != string::npos );
				if ( i < n ) idx = s.find_last_not_of( COLON, idx );
				assert( idx != string::npos );
			}
			return s.substr( idx + 1 );
		}
	}


	/// @brief Trim a specified number of parts from the suffix of an option id
	std::string &
	OptionCollection::trim(
		std::string & s, // String
		int const n // Number of suffix parts to trim
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		size_type const n_s( n_part( s ) );

		if ( ( n <= 0 ) || ( s.empty() ) ) { // Nothing
			// Do nothing
		} else if ( size_type( n ) >= n_s ) { // Trim whole string
			s.clear();
		} else {
			char const COLON( ':' );
			size_type idx( s.length() - 1 );
			for ( int i = 1; i <= n; ++i ) {
				assert( idx != string::npos );
				idx = s.find_last_of( COLON, idx );
				assert( idx != string::npos );
				idx = s.find_last_not_of( COLON, idx ); // None is OK since npos+1==0
			}
			return s.erase( idx + 1 );
		}
		return s;
	}


	/// @brief Prefix of an option id with a specified number of suffix parts removed
	std::string
	OptionCollection::trimmed(
		std::string const & s, // String
		int const n // Number of suffix parts to trim
	)
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		size_type const n_s( n_part( s ) );

		if ( ( n <= 0 ) || ( s.empty() ) ) { // Nothing
			return s;
		} else if ( size_type( n ) >= n_s ) { // Trim whole string
			return string();
		} else {
			char const COLON( ':' );
			size_type idx( s.length() - 1 );
			for ( int i = 1; i <= n; ++i ) {
				assert( idx != string::npos );
				idx = s.find_last_of( COLON, idx );
				assert( idx != string::npos );
				idx = s.find_last_not_of( COLON, idx ); // None is OK since npos+1==0
			}
			return s.substr( 0, idx + 1 );
		}
	}


	/// @brief Cleaned option id with repeat colons condensed
	std::string
	OptionCollection::cleaned( std::string const & s )
	{
		using std::string;
		typedef  std::string::size_type  size_type;

		char const COLON( ':' );
		string t;
		for ( size_type i = 0, e = s.size(); i < e; ++i ) {
			if ( ( i == 0 ) || ( s[ i ] != COLON ) || ( s[ i - 1 ] != COLON ) ) { // Keep it
				t += s[ i ];
			}
		}
		return t;
	}


	/// @brief Merged option ids with the minimal suffix-prefix overlap, if any, removed
	std::string
	OptionCollection::merged(
		std::string const & s, // Lead id
		std::string const & t // Tail id
	)
	{
		typedef  std::string::size_type  size_type;

		// Handle either id empty
		if ( ( s.empty() ) || ( t.empty() ) ) return t;

		// Part counts
		size_type const n_s( n_part( s ) );
		size_type const n_t( n_part( t ) );
		size_type const max_overlap( std::min( n_s, n_t ) );

		// Look for an overlap match
		size_type o( 1 ); // Overlap being tried
		while ( o <= max_overlap ) {
			if ( cleaned( suffix( s, o ) ) == cleaned( prefix( t, o ) ) ) { // Match found
				if ( o == n_s ) {
					return t;
				} else {
					return cleaned( trimmed( s, o ) ) + ':' += t;
				}
			}
			++o;
		}

		// No overlap match found
		return cleaned( s ) + ':' += cleaned( t ); // No overlap match found: Concatenate
	}


	/// @brief String wrapped and indented
	std::string
	OptionCollection::wrapped(
		std::string const & s, // String to wrap
		std::string::size_type const indent, // Width to indent continuation lines
		std::string::size_type const width // Column width to wrap at [80]
	)
	{
		using std::string;
		using ObjexxFCL::is_any_of;
		typedef  std::string::size_type  size_type;

		// check arguments
		assert( indent + 1 < width );

		// Handle empty or 1-line string
		if ( ( s.empty() ) || ( s.length() <= width ) ) return s;

		// Create the wrapped string
		string w;
		size_type l( 0 ); // Line length
		char const nl( '\n' ); // Newline
		string const ws( " \t" ); // Whitespace
		for ( size_type i = 0, e = s.length(); i < e; ++i ) {
			if ( l + 1 >= width ) { // Wrap
				while ( ( i < e ) && ( is_any_of( s[ i ], ws ) ) ) ++i;
				if ( i < e ) { // Indent and add next non-whitespace character
					w += nl + string( indent, ' ' ) + s[ i ];
					l = indent + 1;
				} else { // Nothing left
					l = 0;
				}
			} else if ( s[ i ] == nl ) { // Embedded line terminator: Wrap
				if ( i + 1 >= e ) { // Last character: Add the newline
					w += nl;
					l = 0;
				} else if ( s[ i + 1 ] == nl ) { // Next character a newline: Don't indent
					w += nl;
					l = 0;
				} else { // Next character not a newline: Indent
					while ( ( ++i < e ) && ( is_any_of( s[ i ], ws ) ) ) ++i;
					if ( i < e ) { // Indent and add next non-whitespace character
						w += nl + string( indent, ' ' ) + s[ i ];
						l = indent + 1;
					} else { // Nothing left: Add the newline
						w += nl;
						l = 0;
					}
				}
			} else { // Add next character to line
				size_type const b( std::min( s.find_first_of( ws, i ), s.length() ) ); // Next whitespace break
				if ( ( l + 1 + b - i >= width ) && ( b - i < width - indent ) ) { // Token won't fit: Wrap
					w += nl + string( indent, ' ' ) + s[ i ];
					l = indent + 1;
				} else { // Add the character
					w += s[ i ];
					++l;
				}
			}
		}
		return w;
	}


} // namespace options
} // namespace utility
