// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/options/VariantOption.hh
/// @brief  Variant option class
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_utility_options_VariantOption_HH
#define INCLUDED_utility_options_VariantOption_HH


// Unit headers
#include <utility/options/VariantOption.fwd.hh>

// C++ headers
#include <cassert>
#include <cstddef>
#include <string>


namespace utility {
namespace options {


/// @brief Variant option class
template< typename O >
class VariantOption
{


public: // Types


	// STL/boost style
	typedef  O  option_type;
	typedef  typename O::key_type  key_type;
	typedef  std::size_t  size_type;

	// Project style
	typedef  O  Option;
	typedef  typename O::Key  Key;
	typedef  std::size_t  Size;


public: // Creation


	/// @brief Default constructor
	inline
	VariantOption() :
		option_p_( 0 )
	{}


	/// @brief Copy constructor
	inline
	VariantOption( VariantOption const & var ) :
		option_p_( var.option_p_ ? var.option_p_->clone() : 0 )
	{}


	/// @brief Option constructor
	inline
	VariantOption( Option const & option_a ) :
		option_p_( option_a.clone() )
	{}


	/// @brief Destructor
	inline
	~VariantOption()
	{
		delete option_p_;
	}


public: // Assignment


	/// @brief Copy assignment
	inline
	VariantOption &
	operator =( VariantOption const & var )
	{
		if ( this != &var ) {
			delete option_p_; option_p_ = ( var.option_p_ ? var.option_p_->clone() : 0 );
		}
		return *this;
	}


public: // Conversion


	/// @brief Option conversion
	inline
	operator Option const &() const
	{
		assert( option_p_ );
		return *option_p_;
	}


	/// @brief Option conversion
	inline
	operator Option &()
	{
		assert( option_p_ );
		return *option_p_;
	}


public: // Methods


	/// @brief Activate
	inline
	VariantOption &
	activate()
	{
		assert( option_p_ );
		option_p_->activate();
		return *this;
	}


	/// @brief Deactivate
	inline
	VariantOption &
	deactivate()
	{
		assert( option_p_ );
		option_p_->deactivate();
		return *this;
	}


	/// @brief Clear
	inline
	VariantOption &
	clear()
	{
		assert( option_p_ );
		option_p_->clear();
		return *this;
	}


	/// @brief Value assignment from a command line string
	inline
	VariantOption &
	cl_value( std::string const & value_str )
	{
		assert( option_p_ );
		option_p_->cl_value( value_str );
		return *this;
	}


public: // Properties


	/// @brief Key
	inline
	Key const &
	key() const
	{
		assert( option_p_ );
		return option_p_->key();
	}


	/// @brief ID
	inline
	std::string const &
	id() const
	{
		assert( option_p_ );
		return option_p_->id();
	}


	/// @brief Identifier
	inline
	std::string const &
	identifier() const
	{
		assert( option_p_ );
		return option_p_->identifier();
	}


	/// @brief Code
	inline
	std::string const &
	code() const
	{
		assert( option_p_ );
		return option_p_->code();
	}


	/// @brief Name
	inline
	std::string const &
	name() const
	{
		assert( option_p_ );
		return option_p_->name();
	}


	/// @brief Description
	inline
	std::string const &
	description() const
	{
		assert( option_p_ );
		return option_p_->description();
	}


	/// @brief Active?
	inline
	bool
	active() const
	{
		assert( option_p_ );
		return option_p_->active();
	}


	/// @brief Inactive?
	inline
	bool
	inactive() const
	{
		assert( option_p_ );
		return option_p_->inactive();
	}


	/// @brief Defaulted?
	inline
	bool
	defaulted() const
	{
		assert( option_p_ );
		return option_p_->defaulted();
	}


	/// @brief User-specified?
	inline
	bool
	user() const
	{
		assert( option_p_ );
		return option_p_->user();
	}


	/// @brief Specified (user-specified)?
	inline
	bool
	specified() const
	{
		assert( option_p_ );
		return option_p_->specified();
	}


	/// @brief Present (user-specified)?
	inline
	bool
	present() const
	{
		assert( option_p_ );
		return option_p_->present();
	}


	/// @brief Default size (number of default values)
	inline
	Size
	default_size() const
	{
		assert( option_p_ );
		return option_p_->default_size();
	}


	/// @brief Number of default values (default size)
	inline
	Size
	n_default_value() const
	{
		assert( option_p_ );
		return option_p_->n_default_value();
	}


	/// @brief Size (number of values)
	inline
	Size
	size() const
	{
		assert( option_p_ );
		return option_p_->size();
	}


	/// @brief Number of values (size)
	inline
	Size
	n_value() const
	{
		assert( option_p_ );
		return option_p_->n_value();
	}


	/// @brief Option
	inline
	Option const &
	operator ()() const
	{
		assert( option_p_ );
		return *option_p_;
	}


	/// @brief Option
	inline
	Option &
	operator ()()
	{
		assert( option_p_ );
		return *option_p_;
	}


public: // Comparison


	/// @brief VariantOption < VariantOption
	friend
	inline
	bool
	operator <( VariantOption const & a, VariantOption const & b )
	{
		assert( a.option_p_ );
		assert( b.option_p_ );
		return ( *a.option_p_ < *b.option_p_ );
	}


private: // Fields


	/// @brief Pointer to option
	Option * option_p_;


}; // VariantOption


} // namespace options
} // namespace utility


#endif // INCLUDED_utility_options_VariantOption_HH
