# This file is made available under the Rosetta Commons license.
# See http://www.rosettacommons.org/license
# (C) 199x-2007 University of Washington
# (C) 199x-2007 University of California Santa Cruz
# (C) 199x-2007 University of California San Francisco
# (C) 199x-2007 Johns Hopkins University
# (C) 199x-2007 University of North Carolina, Chapel Hill
# (C) 199x-2007 Vanderbilt University

## @file   utility/options/options_class.py
## @brief  Program options generation script classes
## @author Sergey Lyskov (Sergey.Lyskov@jhu.edu)


import sys

KnownTypes=['Boolean', 'Integer', 'Real', 'String', 'File', 'Path', 'BooleanVector', 'IntegerVector', 'RealVector', 'StringVector', 'FileVector', 'PathVector']

# check if given object can be int/bool/real - if not: wrap "" around it.
def wrapCStrig(s):
    if s:
        if len(s) < 2 :
            return '"' + s + '"'
        else:
            if s[0] == '"' and s[-1] == '"': return s
            else: return '"' + s + '"'
    return s


class Option:
    def __init__(self, name=None, ctype=None, group=None, desc="No description", oldName="-", lower=None, upper=None, default=None, legal=None, n=None, n_lower=None, n_upper=None):
        if ctype not in KnownTypes:
            print 'Unknow type:%s!!!' % ctype
            sys.exit()
        
        self.ctype = ctype;  self.name = name;  self.group = group
        self.desc = desc;  self.oldName = oldName
        self.lower = lower;  self.upper = upper;  self.default = default;  self.legal=legal
        self.n = n;  self.n_lower = n_lower;  self.n_upper = n_upper
        # Wraping c-strings in "
        if ctype == 'String' or ctype == 'Path' or ctype == 'File' or \
           ctype == 'StringVector' or ctype == 'PathVector' or ctype == 'FileVector':
            self.default = wrapCStrig( self.default )
            self.lower = wrapCStrig( self.lower )
            self.upper = wrapCStrig( self.upper )
            if type( self.legal ) == type([]):
                for i in range(0, len(self.legal) ):
                    self.legal[i] = wrapCStrig( self.legal[i] )
            else: self.legal = wrapCStrig( self.legal )

    # return C++ name of option object
    def getCName(self):
        if str( self.name[:1] ).isdigit() : return 'n' + self.name
        else: return self.name


    def getOptionCC(self):
        s = ''
        s += 'option.add( '
        if self.group:
            s += self.group.replace(':','::')+'::'
        s += self.getCName()+', "'+self.desc+'" )'
        if self.lower : s+= '.lower(' + self.lower + ')'
        if self.upper : s+= '.upper(' + self.upper + ')'
        if self.n: s+= '.n(' + self.n + ')'
        if self.n_lower: s+= '.n_lower(' + self.n_lower + ')'
        if self.n_upper: s+= '.n_upper(' + self.n_upper + ')'
        if self.legal :
            if type(self.legal) == type(''):
                s+= '.legal(' + self.legal + ')'
            else:
                for l in self.legal:
                    s+= '.legal(' + l + ')'
        if self.default :
            if type(self.default) == type(''): s+= '.def(' + self.default + ')'
            else:
                for d in self.default:
                    s+= '.def(' + d + ')'

        return s + ';\n'


    def getOptionKeysHH(self):
        s = '';  se = ''
        if self.group:
            for ns in self.group.split(':'):
                s += 'namespace ' + ns + ' { '
                se += ' }'
        s += 'extern '+ self.ctype+'OptionKey const '+self.getCName()+';'
        return s + se + '\n'


    def getOptionKeysCC(self):
        s = '';  se = ''
        if self.group:
            for ns in self.group.split(':'):
                s += 'namespace ' + ns + ' { '
                se += ' } '

        s += self.ctype+'OptionKey const '+self.getCName()+'( "'

        if self.group:
            s += self.group
            if self.name != self.group.split(':')[-1] : s += ':' + self.name
        else: s += self.name
        s+= '" ); ' + se + '\n'
        return s


    def getWikiTableRow(self):
        def smStr(s): return s or ''

        s =  ' |-\n'
        s += ' | -%(name)s <%(ctype)s>\n' % {'name':self.name, 'ctype':self.ctype}
        s += ' | ' + self.desc + '\n'
        s += ' | ' + smStr(self.lower) + '-' + smStr(self.upper) + '\n'
        if self.legal=='true' and self.default=='true': s += ' |\n'
        else:
            if type(self.default) == type( [] ):
                s += ' | ' + str( self.default ) + '\n'
            else: s += ' | ' + smStr(self.default) + '\n'
        s += ' | -' + self.oldName + '\n'
        s += ' |-\n'
        return s


def Option_Group(group, *args):
    res = []

    for o in args:  # first concat all lists
        if type(o) == type([]): res += o
        else: res.append( o )

    for o in res:
        if group:
            if o.group: o.group = group + ':' + o.group
            else: o.group = group
    return res


def writeToFile(opt_list, fileName, mapFunction):
    l = map(mapFunction, opt_list)
    f = file(fileName, 'wb');  f.write( "".join(l) );  f.close()


def printWikiTable(opt_list):
    s = ""
    prevGroup = None
    for o in opt_list:
        if prevGroup != o.group:  # Generating new table
            if prevGroup : s += ' |}\n' # Closing previos table if any
            s += """{| border="1" cellpadding="10" width="100%"\n |+ '''""" + (o.group or '')
            s += " Option Group'''\n"
            s += ' ! Option name\n'                    # width="10%" |
            s += ' ! Description\n'
            s += ' ! Range\n'
            s += ' ! Default\n'
            s += ' ! Old name\n'
            s += ' |-\n'
        s += o.getWikiTableRow()
        prevGroup = o.group
    return s + ' |}\n'
