// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19092 $
//  $Date: 2007-12-12 20:49:43 -0500 (Wed, 12 Dec 2007) $
//  $Author: johnk $

#ifndef INCLUDED_structure
#define INCLUDED_structure


// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>


namespace structure {

namespace functions {
	void
	radius_gyration(
		float & rg,
		int total_residue
	);
}

////////////////
/// Boundary ///
////////////////
// yab:
// functions here act as an interface between the classic monolithic
// Rosetta world and the new object world until misc removal is
// complete and Rosetta's computational pipeline is detangled.
namespace BOUNDARY {

void
init_module();

bool
init_module_called_once();

void
remove_module();

void
identify_ss();

void
evaluate_ss(
int & lowstrand,
int & cutoff,
float & hs_score,
float & ss_score,
float & rsigma_score,
float & sheet_score
);

void
evaluate_envpair(
	float & env_score,
	float & pair_score,
	float & cb_score,
	float & centroid_pack_score
);

bool
get_separate_centroidpackscore_flag();

void
setup_SS_default_weight();

void
setup_SS_weight();

void
setup_rsd_weight();

void
reset_rsd_wt();

void
read_homolog_env_stats_file();

void
read_homolog_pair_stats_file();

void
evaluate_homolog_env(
	float & env_score
);

void
evaluate_homolog_pair(
	float & pair_score
);

void
SSpair_score(
	int lowstrand,
	int cutoff,
	float & ss_score,
	float & rsigma_score
);

int
get_total_strands();

FArray2D_int const &
get_SS_strand_end();

FArray2D_float const &
get_strand_strand_score();

int
get_total_SS_dimer();

FArray1D_int const &
get_SS_resnum();

FArray1D_int const &
get_SS_strand();

FArray1D_float const &
get_pdb_cen10_prob();

FArray2D_float const &
get_pdb_pair_prob();

}

class HomologStats; // forward declaration

class SecondaryStructureEval {
	public: // Creation/Destruction
		SecondaryStructureEval(
			FArray2D_float const * position,
			FArray3Dp_float const * Eposition,
			FArray1D_int const * res,
			FArray1D_char const * secstruct,
			int const * total_residue
		);

		SecondaryStructureEval();
		~SecondaryStructureEval();

	public: // load and cache required external data
		static
		void
		load_and_cache_phi_theta_bins_from_file();

	public: // load external data for HomologStats instance
		void
		read_homolog_env_stats_file(
			std::string homolog_env_filename
		);

		void
		read_homolog_pair_stats_file(
			std::string homolog_pair_filename
		);

	public: // scoring for HomologStats instance
		void
		evaluate_homolog_env(
			float & env_score
		);

		void
		evaluate_homolog_pair(
			float & pair_score
		);

	public:
		FArray1D_float const &
		get_pdb_cen10_prob();

		FArray2D_float const &
		get_pdb_pair_prob();

	public: // structure scoring
		void
		identify_ss();

		void
		SSpair_score(
			int lowstrand,
			int cutoff,
			float & ss_score,
			float & rsigma_score
		);

		void
		evaluate_ss(
			int & lowstrand,
			int & cutoff,
			float & hs_score,
			float & ss_score,
			float & rsigma_score,
			float & sheet_score
		);

		void
		evaluate_envpair(
			float & env_score,
			float & pair_score,
			float & cb_score,
			float & centroid_pack_score
		);

	public: // accessors
		void
		set_localstrandpair_penalty(float val);

		float
		get_localstrandpair_penalty();

		void
		set_handedness_score_flag(bool flag);

		bool
		get_handedness_score_flag();

		void
		set_ignore_sspair_barcode_in_score(bool flag);

		bool
		get_ignore_sspair_barcode_in_score();

		void
		set_max_strand_dist_cutoff(float val);

		float
		get_max_strand_dist_cutoff();

		void
		set_seq_sep_scale(float val);

		float
		get_seq_sep_scale();

		void
		set_stretch_strand_dist_cutoff(bool flag);

		bool
		get_stretch_strand_dist_cutoff();

		void
		set_strand_dist_cutoff(float val);

		float
		get_strand_dist_cutoff();

		void
		set_cb_weight(float val);

		float
		get_cb_weight();

		void
		set_separate_centroidpackscore(bool flag);

		bool
		get_separate_centroidpackscore();

		void
		set_centroid_packing_weight(float val);

		float
		get_centroid_packing_weight();

		void
		set_new_centroid_packing(bool flag);

		bool
		get_new_centroid_packing();

		void
		set_homolog_stats(HomologStats * hs);

		HomologStats *
		get_homolog_stats();

	public: // accessors, get only, inner classes
		int
		get_total_strands();

		FArray2D_int const &
		get_SS_strand_end();

		FArray2D_float const &
		get_strand_strand_score();

		int
		get_total_SS_dimer();

		FArray1D_int const &
		get_SS_resnum();

		FArray1D_int const &
		get_SS_strand();


	public: // ss weight
		void
		reset_SS_weight();

		void
		setup_SS_weight(
			bool use_ss_weight,
			bool randomize_parallel_antiparallel_weight
		);

		void
		setup_SS_default_weight();

		float
		get_HS_wt();

		float
		get_SS_wt();

		float
		get_rsigma_wt();

		float
		get_sheet_wt();

		float
		get_parallel_wt();

		float
		get_antiparallel_wt();

		float
		get_default_parallel_wt();

		float
		get_default_antiparallel_wt();

		void
		set_default_parallel_wt(float val);

		void
		set_default_antiparallel_wt(float val);

	public: // rsd weight
		void
		reset_rsd_weight();

		void
		setup_rsd_weight(
			bool use_rsd_weight
		);

		void
		setup_rsd_default_weight();

		FArray1D_float const &
		get_rsd_wt();

		void
		set_rsd_wt_helix(float val);

		void
		set_rsd_wt_loop(float val);

		void
		set_rsd_wt_strand(float val);

		float
		get_rsd_wt_helix();

		float
		get_rsd_wt_loop();

		float
		get_rsd_wt_strand();

	public: // accessor for mirrors; these functions are temporary and will go away!  functionality is here only for HomologStats nested class
		int const &
		get_total_residue()
		const;

		FArray1D_int const &
		get_res()
		const;

	private: // rsd weight
		void
		reweight_rsd_wt_for_secstruct(
			FArray1D_float & rsd_wt,
			char secstruct_to_reweight,
			float reweightfactor
		);

	private: // structure scoring
		void
		sheets_from_dimers(
			float & sheet_score
		);

		void
		HSpair_score(
			float & hs_score
		);

	private: // structure geometry
		void
		helix_end(
			int pos1,
			FArray1DB_float & p1,
			FArray1DB_float & p2
		);

		void
		pair_dp(
			int ss1,
			int ss2,
			float & dp,
			FArray1DB_float const & vdist,
			int & sign1,
			int & sign2
		);

	private: // math
		static
		void
		dist_pair(
			FArray1DB_float const & a1,
			FArray1DB_float const & a2,
			FArray1DB_float const & a3,
			FArray1DB_float const & a4,
			float & dist, // length of v21
			FArray1DB_float & cen1,
			FArray1DB_float & cen2,
			FArray1DB_float & v21 // vector connecting midpoints
		);

		static
		void
		sigma(
			FArray1DB_float const & a2, // endpoints of v1
			FArray1DB_float const & cen1,
			FArray1DB_float const & v21,
			float & sig // sigma
		);

		static
		void
		spherical(
			FArray1DB_float const & a2,
			FArray1DB_float const & a4,
			float & phi,
			float & theta,
			FArray1DB_float const & cen1,
			FArray1DB_float const & cen2,
			FArray1DB_float const & v21 // vector between midpoints
		);

	private: // methods, data initializers
		static
		void
		SS_penalty_initializer(
			FArray1D_float & SS_penalty
		);

		static
		void
		m_term_initializer(
			FArray1D_float & m_term
		);

		//struct_catalog
		static
		void
		idsn_initializer(
			FArray1D_int & idsn
		);

		static
		void
		ids_initializer(
			FArray1D_int & ids
		);

		static
		void
		atom_vdw_tables_initializer(
			FArray3D_float & atom_vdw_tables_initializer
		);

		static
		void
		ssdist_initializer(
			FArray2D_float & ssdist
		);

		static
		void
		cbeta_den6_initializer(
			FArray1D_float & cbeta_den6
		);

		static
		void
		cbeta_den12_initializer(
			FArray1D_float & cbeta_den12
		);

		static
		void
		hs_dp_initializer(
			FArray1D_float & hs_dp
		);

		static
		void
		rsigma_dot_initializer(
			FArray4D_float & rsigma_dot
		);

		static
		void
		centroid_pack_correction_initializer(
			FArray1D_float & centroid_pack_correction
		);

		// seq_struct
		static
		void
		env_log_initializer_master(
			FArray2D_float & env_log
		);

		static
		void
		env_log_initializer_section1(
			FArray2D_float & env_log
		);

		static
		void
		env_log_initializer_section2(
			FArray2D_float & env_log
		);

		static
		void
		env_log_initializer_section3(
			FArray2D_float & env_log
		);

		static
		void
		pair_log_initializer_master(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section1a(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section1b(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section2a(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section2b(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section3a(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section3b(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section4a(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section4b(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section5a(
			FArray3D_float & pair_log
		);

		static
		void
		pair_log_initializer_section5b(
			FArray3D_float & pair_log
		);

	private: // Methods, setup
		void
		initialize();

	private: // Inner Classes, these are not meant to be used outside! hence public data
		class Strands {
			public:
				int total_SS_dimer; // number of strand dimers
				FArray1D_int SS_resnum; // residue number of strand dimer i
				int total_strands; // number of strands
				FArray1D_int SS_strand; // strand number containing SS_dimer i
				FArray2D_int SS_strand_end; // residue number of first non-E res
				FArray2D_int dimer_neighbor;
				// two neighbors, used for determing sheets
				// strand-strand score array, resized for each calculation:
				FArray2D_float strand_strand_score; // dimensioned to MAX_STRANDS

				FArray2D_int ss_orient; // does dimer pair i,j interact? 1=yes

			public: // constructor
				Strands();
				~Strands();
		};

		class Helices {
			public:
				int total_HH_dimer;
				FArray1D_int HH_resnum;
				FArray2D_int HH_helix_end;

			public: // constructor
				Helices();
				~Helices();
		};

	private: // pointers, mirrors
		FArray2D_float const * position_;
		FArray3Dp_float const * Eposition_;
		FArray1D_int const * res_;
		FArray1D_char const * secstruct_;
		int const * total_residue_;

	private: // "cached" data
		static bool cache_initialized; // initially false (see .cc), afterwards only set in load_and_cache_phi_theta_bins_from_file
		// struct_catalog
		static FArray1D_int cached_iptsn;
		static FArray4D_float cached_pts;
		static FArray1D_float cached_ds;

	private: // constant data
		// struct_catalog
		static FArray1D_int const idsn;
		static FArray1D_int const ids;
		static FArray2D_float const ssdist;
		static FArray1D_float const cbeta_den6;
		static FArray1D_float const cbeta_den12;
		static FArray1D_float const hs_dp; // not in use?
		static FArray4D_float const rsigma_dot; // lookup for new rsigma score
		static FArray1D_float const centroid_pack_correction;

		// seq_struct
		static FArray2D_float const env_log;
		static FArray3D_float const pair_log;

	private: // ss weights
		bool use_ss_wt;
		float SS_wt;
		float HS_wt;
		float sheet_wt;
		float rsigma_wt;

		bool random_parallel_antiparallel;
		float parallel_weight;
		float antiparallel_weight;
		float default_parallel_weight;
		float default_antiparallel_weight;

	private: // rsd weights
		bool use_rsd_wt;
		float rsd_wt_helix;
		float rsd_wt_loop;
		float rsd_wt_strand;
		FArray1D_float rsd_wt;

	private: // additional settings
		float cb_weight;

		bool new_centroid_packing;
		float centroid_packing_weight;

		float localstrandpair_penalty;

		float seq_sep_scale;

		bool stretch_strand_dist_cutoff;
		float max_strand_dist_cutoff;
		float strand_dist_cutoff;

		bool handedness_score_flag;

		bool ignore_sspair_barcode_in_score;

		bool separate_centroidpackscore;

	private: // data
		// struct_catalog
		FArray1D_int iptsn;
		FArray4D_float pts;
		FArray1D_float ds;

		// secstruct
		Strands strands;
		Helices helices;
		HomologStats * homolog_stats;

};

/**********************/
/*** SEPARATE CLASS ***/
/**********************/

class HomologStats {

// yab:
// This was originally an INNER class!  Which is why everything is public!
// This will be changed!  It was dragged out of the SecondaryStructureEval
// due to weird design issues.
// For now, please DO NOT use this outside of SecondaryStructureEval!

	public:
		static FArray1D_float const pdb_cen10_prob;
		static FArray2D_float const pdb_pair_prob;
		FArray2D_float homolog_cen10_log;
		FArray3D_float homolog_pair_log;

	public: // constructors
		HomologStats(SecondaryStructureEval * sse);
		HomologStats();
		~HomologStats();

	public: // initialize
		void
		initialize(); // public to allow re-initialization

	public: // input from file
		void
		read_homolog_env_stats_file(
			std::string homolog_env_filename
		);

		void
		read_homolog_pair_stats_file(
			std::string homolog_pair_filename
		);

	public: // accessors
		void
		set_parent_sse_object(
			SecondaryStructureEval * sse
		);

		SecondaryStructureEval *
		get_parent_sse_object();

	public: // evaluation
		void
		evaluate_homolog_env(
			float & env_score
		);

		void
		evaluate_homolog_pair(
			float & pair_score
		);

	private: // data initializers
		static
		void
		homolog_env_initializer(
			FArray1D_float & pdb_cen10_prob
		);

		static
		void
		homolog_pair_initializer(
			FArray2D_float & pdb_pair_prob
		);

	private: // data
		SecondaryStructureEval * parent_sse_object_;
};

}

//////////////////
/// Historical ///
//////////////////

	// spherical coordinates?
//car 1=N,2=CA,3=CB,4=C,5=O,6-33=centroids (restype+5)
//car types for all atoms can be looked up in Eatom_type(atom,residue)
//car                                     and atom_type_cen(residue)

	// key: 1-HS terms, 2-SS terms
	// indices: (bins, key) key: 1=HS, 2=SS
	// bins are presumably distance bins?

	/////////////////////////////////////////////////////////////////////////////
	// Inactive DATA initializers from original Fortran:
	//
	//      data iptsn/191,500,618,500,191/
	//
	//      data ipts_HS/  40, 271, 435, 364, 166,
	//     #               10,  11,  21, 192, 145,
	//     #               37, 255, 440, 370, 168,
	//     #               31,  61,  10,   2,  79,
	//     #               13,  92, 210,1652,4188,
	//     #               70, 175, 526,2964,5784,
	//     #               86, 178, 295,1671,4364,
	//     #               39,  97, 217,1213,3174,
	//     #             1290, 762, 515,1497,3240,
	//     #             1455,1927,2231,6728,7121,
	//     #             1871,1174, 659,1622,3411,
	//     #             3788,5611,2541,2099,2536/
	//
	//      data ipts_SS/   0,   0,   0,   0,   0,
	//     #                0,   0,   0,   0,   0,
	//     #                0,   0,   0,   3,   0,
	//     #                0,   0,   0,   0,   0,
	//     #                6,   0,  24,  20, 551,
	//     #               12,   7, 166,2155,3685,
	//     #               11,   0,   1,  38, 752,
	//     #               12,   6,   5,  27, 302,
	//     #              165,  18,  17,  22, 475,
	//     #               86,  25, 101,1557,2635,
	//     #             1000,  40,   7,  35, 581,
	//     #             2798,1103,  67,  19, 306/
	/////////////////////////////////////////////////////////////////////////////

#endif
