// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 9836 $
//  $Date: 2006-08-16 15:36:45 -0700 (Wed, 16 Aug 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "symmetry_info.h"
//#include "angles.h"
//#include "jumping_util.h"
#include "param_torsion.h"
//#include "pose.h"
//#include "random_numbers.h"
//#include "util_vector.h" // Ddotprod, etc
//#include "FArray_xyz_functions.h"

// ObjexxFCL Headers
//#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/FArray1D.hh>
//#include <ObjexxFCL/FArray2D.hh>

// Numeric Headers
//#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
//#include <numeric/xyzVector.hh>
//#include <numeric/xyzMatrix.hh>
//#include <numeric/xyz.functions.hh>
//#include <numeric/xyzVector.io.hh>
//#include <numeric/xyzMatrix.io.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>


namespace pose_ns {

	/////////////////////////////////////////////////////////////////////////////
	//
	bool
	operator== ( Symmetry_info const & a, Symmetry_info const & b ) {
		return ( a.npseudo_ == b.npseudo_ &&
						 a.bb_clones_ == b.bb_clones_ &&
						 a.chi_clones_ == b.chi_clones_ &&
						 a.jump_clones_ == b.jump_clones_ );
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	bool
	operator!= ( Symmetry_info const & a, Symmetry_info const & b ) {
		return !( a == b );
	}



	/////////////////////////////////////////////////////////////////////////////
	float
	Symmetry_info::deriv_scale_factor(
		int const seqpos,
		int const torsion
	) const
	{
		using namespace param_torsion;
		if ( torsion_type_is_bb( torsion ) ) {
			assert( bb_independent( seqpos ) &&
							int(bb_clones(seqpos).size()) == num_bb_clones() );
			return 1 + num_bb_clones();
		} else if ( torsion_type_is_chi( torsion ) ) {
			assert( chi_independent( seqpos ) &&
							int(chi_clones(seqpos).size()) == num_chi_clones() );
			return 1 + num_chi_clones();
		} else if ( torsion_type_is_rb( torsion ) ) {
			// seqpos is actually the jump number
			assert( jump_clones_.count( seqpos ) );
			return 1 + jump_clones( seqpos ).size();
		} else {
			std::cout << "Unrecognized torsion type! " << torsion << ' ' <<
				seqpos << std::endl;
			std::exit( EXIT_FAILURE );
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	//
	Symmetry_info::Symmetry_info(
		int const nres_monomer,
		int const njump_monomer,
		int const N,
		std::string const & type
	)
	{
		exclude_pseudo_from_total_residue = false;

		//space_group_ = 0;
		//nres_monomer_ = nres_monomer;
		//N_ = N;

		//////////////////////
		// setup bb,chi clones
		bb_clones_.clear();
		chi_clones_.clear();
		jump_clones_.clear();

		//special case of no symmetry
		if ( type == "c1" ) {
			type_ = type;
   		npseudo_ = N;
			//we need to map to an empty array in order for
			//num_xx_clones to return 0.
			for ( int i=1; i<= nres_monomer; ++i ) {
				std::vector< int > clones;
				clones.clear();
      	bb_clones_.insert( std::make_pair( i, clones ) );
				chi_clones_.insert( std::make_pair( i, clones ) );
    		}
				for ( int i=1; i<= njump_monomer; ++i ) {
					std::vector< int > clones;
        	clones.clear();
					jump_clones_.insert( std::make_pair( i, clones ) );
				}
			return;
		}//end c1 symmetry

		for ( int i=1; i<= nres_monomer; ++i ) {
			std::vector< int > clones;
			for ( int k=1; k<N; ++k ) {
				clones.push_back( i + k * nres_monomer );
			}
			bb_clones_.insert( std::make_pair( i, clones ) );
			chi_clones_.insert( std::make_pair( i, clones ) );
		}


		// the N*njump_monomer internal jumps
		for ( int i=1; i<= njump_monomer; ++i ) {
			std::vector< int > clones;
			for ( int k=1; k<N; ++k ) {
				clones.push_back( i + k * njump_monomer );
			}
			jump_clones_.insert( std::make_pair( i, clones ) );
		}

		if ( type == "no_pseudo" ) {
			npseudo_ = 0;

		} else if ( type == "simple" ) {
			// 1                 --> N*njump_monomer  : the internal jumps
			// N*njump_monomer+1 --> N*njump_monomer+N: the pseudo-rsd--monomer jumps
			// last N-1 jumps                         : jumps between pseudo-rsds

			npseudo_ = N;

			// the N jumps from pseudo-residues to monomers
			{
				int const base_jump( N*njump_monomer + 1 );
				std::vector< int > clones;
				for ( int k=1; k<N; ++k ) {
					clones.push_back( base_jump + k );
				}
				jump_clones_.insert( std::make_pair( base_jump, clones ) );
			}


			// the N-1 jumps between pseudo-residues
			{
				int const base_jump( N*njump_monomer + N + 1 );
				std::vector< int > clones;
				for ( int k=1; k<N-1; ++k ) {
					clones.push_back( base_jump + k );
				}
				jump_clones_.insert( std::make_pair( base_jump, clones ) );
			}
		} else {
			std::cout << "unrecognized type: " << type << std::endl;
			std::exit( EXIT_FAILURE );
		}

		setup_clone_follows();
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	Symmetry_info::Symmetry_info(
		int const nres_monomer,
		int const njump_monomer,
		int const N,
		int score_monomer,
		std::string const & type
	)
	{
		exclude_pseudo_from_total_residue = false;

		//space_group_ = 0;
		//nres_monomer_ = nres_monomer;
		//N_ = N;

		//////////////////////
		// setup bb,chi clones
		bb_clones_.clear();
		chi_clones_.clear();

		if ( score_monomer > N || score_monomer < 1 ) {
      std::cout << "score_monomer must be in the range 1-N" << std::endl;
      std::exit( EXIT_FAILURE );
    }

	 for ( int i=1; i<= nres_monomer; ++i ) {
      std::vector< int > clones;
			int base ( i + ( score_monomer - 1 ) * nres_monomer );
      for ( int k=0; k<N; ++k ) {
				if ( k+1 != score_monomer ) {
					clones.push_back( i + k * nres_monomer );
				}
      }
      bb_clones_.insert( std::make_pair( base, clones ) );
      chi_clones_.insert( std::make_pair( base, clones ) );
    }

	jump_clones_.clear();

		// the N*njump_monomer internal jumps
		for ( int i=1; i<= njump_monomer; ++i ) {
			std::vector< int > clones;
			for ( int k=1; k<N; ++k ) {
				clones.push_back( i + k * njump_monomer );
			}
			jump_clones_.insert( std::make_pair( i, clones ) );
		}

		if ( type == "no_pseudo" ) {
			npseudo_ = 0;

		} else if ( type == "simple" | type == "helix" ) {
			// 1                 --> N*njump_monomer  : the internal jumps
			// N*njump_monomer+1 --> N*njump_monomer+N: the pseudo-rsd--monomer jumps
			// last N-1 jumps                         : jumps between pseudo-rsds
			type_ = type;
			npseudo_ = N;

			// the N jumps from pseudo-residues to monomers
			{
				int const base_jump( N*njump_monomer + 1 );
				std::vector< int > clones;
				for ( int k=1; k<N; ++k ) {
					clones.push_back( base_jump + k );
				}
				jump_clones_.insert( std::make_pair( base_jump, clones ) );
			}


			// the N-1 jumps between pseudo-residues
			{
				int const base_jump( N*njump_monomer + N + 1 );
				std::vector< int > clones;
				for ( int k=1; k<N-1; ++k ) {
					clones.push_back( base_jump + k );
				}
				jump_clones_.insert( std::make_pair( base_jump, clones ) );
			}
		} else if ( type == "cn" ) {
        type_ = type;
        npseudo_ = N;

        // the N jumps from pseudo-residues to monomers
        {
          int const base_jump( N*njump_monomer + 1 );
          std::vector< int > clones;
          for ( int k=1; k<N; ++k ) {
            clones.push_back( base_jump + k );
          }
          jump_clones_.insert( std::make_pair( base_jump, clones ) );
        }
	} else if ( type == "dn" ) {
			// 1                 --> N*njump_monomer  : the internal jumps
			// N*njump_monomer+1 --> N*njump_monomer+N: the pseudo-rsd--monomer jumps
			// last N-1 jumps                         : jumps between pseudo-rsds
			type_ = type;
			npseudo_ = N;

			// the N jumps from pseudo-residues to monomers
			{
				int const base_jump( N*njump_monomer + 1 );
				std::vector< int > clones;
				for ( int k=1; k<N; ++k ) {
					clones.push_back( base_jump + k );
				}
				jump_clones_.insert( std::make_pair( base_jump, clones ) );
			}

    } else if ( type == "virus" ) {
      // Jumps from each pseudo residue at a face to a pseudo residue at the
      // center of each face
      type_ = type;
      npseudo_ = 2*N;
        {
          int const base_jump( N*njump_monomer + 1 );
          std::vector< int > clones;
          for ( int k=1; k<N; ++k ) {
            clones.push_back( base_jump + k );
          }
          jump_clones_.insert( std::make_pair( base_jump, clones ) );
        }

      {
        int const base_jump( N*njump_monomer + N + 1 );
        std::vector< int > clones;
        for ( int k=1; k<N; ++k ) {
          clones.push_back( base_jump + k );
        }
        jump_clones_.insert( std::make_pair( base_jump, clones ) );
      }
		} else {
			std::cout << "unrecognized type: " << type << std::endl;
			std::exit( EXIT_FAILURE );
		}

		setup_clone_follows();
	}


	/////////////////////////////////////////////////////////////////////////////
// 	void
// 	Symmetry_info::setup(
// 		int const npseudo,
// 		std::map< int, std::vector< int > > const & bclones,
// 		std::map< int, std::vector< int > > const & cclones,
// 		std::map< int, std::vector< int > > const & jclones
// 	)
// 	{
// 		npseudo_ = npseudo;
// 		bb_clones_   = bclones;
// 		chi_clones_  = cclones;
// 		jump_clones_ = jclones;
// 	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Symmetry_info::add_bb_clone(
															int const base_pos,
															int const clone_pos
															)
	{
		if ( bb_follows_[ base_pos ] != 0 ) {
			std::cout << "Error: add_bb_clone: base_pos is not independent: " <<
				base_pos << ' ' << bb_follows_[ base_pos ] << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( bb_follows_[ clone_pos ] != 0 &&
				 bb_follows_[ clone_pos ] != base_pos ) {
			std::cout << "Error: add_bb_clone: clone_pos already a follower: " <<
				clone_pos << ' ' << bb_follows_[ clone_pos ] << ' ' << base_pos <<
				std::endl;
			std::exit( EXIT_FAILURE );
		}

		bb_follows_[ clone_pos ] = base_pos;
		bb_clones_[ base_pos ].push_back( clone_pos );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Symmetry_info::add_chi_clone(
															int const base_pos,
															int const clone_pos
															)
	{
		if ( chi_follows_[ base_pos ] != 0 ) {
			std::cout << "Error: add_chi_clone: base_pos is not independent: " <<
				base_pos << ' ' << chi_follows_[ base_pos ] << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( chi_follows_[ clone_pos ] != 0 &&
				 chi_follows_[ clone_pos ] != base_pos ) {
			std::cout << "Error: add_chi_clone: clone_pos already a follower: " <<
				clone_pos << ' ' << chi_follows_[ clone_pos ] << ' ' << base_pos <<
				std::endl;
			std::exit( EXIT_FAILURE );
		}

		chi_follows_[ clone_pos ] = base_pos;
		chi_clones_[ base_pos ].push_back( clone_pos );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Symmetry_info::add_jump_clone(
															int const base_pos,
															int const clone_pos
															)
	{
		if ( jump_follows_[ base_pos ] != 0 ) {
			std::cout << "Error: add_jump_clone: base_pos is not independent: " <<
				base_pos << ' ' << jump_follows_[ base_pos ] << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( jump_follows_[ clone_pos ] != 0 &&
				 jump_follows_[ clone_pos ] != base_pos ) {
			std::cout << "Error: add_jump_clone: clone_pos already a follower: " <<
				clone_pos << ' ' << jump_follows_[ clone_pos ] << ' ' << base_pos <<
				std::endl;
			std::exit( EXIT_FAILURE );
		}

		jump_follows_[ clone_pos ] = base_pos;
		jump_clones_[ base_pos ].push_back( clone_pos );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Symmetry_info::setup_jump_maps(
		int const num_jump,
		int & num_indep_jumps,
		FArray1D_int & indep_jumps
	) const
	{
		FArray1D_bool indep( num_jump, true );
		for ( std::map< int, std::vector< int > >::const_iterator
						it = jump_clones_.begin(); it != jump_clones_.end(); ++it ) {
			std::vector< int > const & l( it->second );
			for ( int i=0; i< int(l.size()); ++i ) {
				indep( l[i] ) = false;
			}
		}

		num_indep_jumps = 0;
		for ( int i=1; i<= num_jump; ++i ) {
			if ( indep(i) ) {
				indep_jumps( ++num_indep_jumps ) = i;
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	// SLOW!!

	int
	Symmetry_info::num_indep_jump( int const num_jump ) const
	{
		int num_indep_jumps(0);
		FArray1D_int indep_jumps( num_jump );
		setup_jump_maps( num_jump, num_indep_jumps, indep_jumps );
		return num_indep_jumps;
	}


	/////////////////////////////////////////////////////////////////////////////
	// SLOW!!

	int
	Symmetry_info::indep_jump( int const i, int const num_jump ) const
	{
		assert( i <= num_jump );
		int num_indep_jumps(0);
		FArray1D_int indep_jumps( num_jump );
		setup_jump_maps( num_jump, num_indep_jumps, indep_jumps );
		return indep_jumps(i);
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Symmetry_info::setup_clone_follows()
	{
		// bb, chi, jump
		for ( int i=1; i<= 3; ++i ) {

			std::map< int, std::vector< int > > const & clone_list
				( i == 1 ? bb_clones_ : ( i == 2 ? chi_clones_ : jump_clones_ ) );

			std::map< int, int > & clone_follows
				( i == 1 ? bb_follows_ : ( i == 2 ? chi_follows_ : jump_follows_ ) );

			clone_follows.clear();
			for ( std::map< int, std::vector< int > >::const_iterator
							it = clone_list.begin(); it != clone_list.end(); ++it ) {
				int const base( it->first );
				std::vector< int > const & clones( it->second );

				for ( std::vector< int >::const_iterator clone= clones.begin();
							clone != clones.end(); ++clone ) {
					assert( clone_follows[ *clone ] == 0 &&
									!clone_list.count( *clone ) );
					clone_follows[ *clone ] = base;
				}
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	comma_strings_to_vector_map(
		std::istream & is,
		int const nbase,
		std::map< int, std::vector< int > > & clones
	)
	{
		bool fail( false );
		for ( int i=1; !fail && i<= nbase; ++i ) {
			std::string jump_string;
			is >> jump_string;
			if ( is.fail() ) {
				fail = true;
				break;
			}
			std::replace( jump_string.begin(), jump_string.end(), ',', ' ' );
			std::istringstream l( jump_string );
			int base_jump;
			l >> base_jump;
			if ( l.fail() ) {
				fail = true;
				break;
			}
			while ( true ) {
				int j;
				l >> j;
				if ( l.fail() ) break;
				clones[ base_jump ].push_back( j );
			}
			if ( clones[ base_jump ].size() < 1 ) {
				fail = true;
				break;
			}
		}
		if ( fail ) {
			is.setstate( std::ios_base::failbit );
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	vector_map_to_comma_strings(
		std::ostream & out,
		std::map< int, std::vector< int > > clones
	)
	{

		for ( std::map< int ,std::vector< int > >::const_iterator
						it = clones.begin(); it != clones.end(); ++it ) {
			out << ' ' << it->first;
			std::vector< int > const & l( it->second );
			for ( int i=0; i< int(l.size()); ++i ) {
				out << ',' << l[i];
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	std::istream & operator >>( std::istream & is, Symmetry_info & s ) {
		bool fail( false );

		std::string tag;
		int num_bb_indep, num_chi_indep, num_jump_indep;
		is >> tag >> s.npseudo_ >> num_bb_indep >> num_chi_indep >>
			num_jump_indep >> s.type_;

		if ( tag != "SYMM_INFO" || is.fail() ) fail = true;

		comma_strings_to_vector_map( is,   num_bb_indep,   s.bb_clones_ );
		comma_strings_to_vector_map( is,  num_chi_indep,  s.chi_clones_ );
		comma_strings_to_vector_map( is, num_jump_indep, s.jump_clones_ );

		s.setup_clone_follows();

		if ( fail || is.fail() ) {
			std::cout << "Symmetry_info operator>>: Input failed" << std::endl;
			is.setstate( std::ios_base::failbit );
			return is;
		}


		return is;
	}



	/////////////////////////////////////////////////////////////////////////////
	std::ostream & operator <<( std::ostream & out, Symmetry_info const & s ) {
		out << "SYMM_INFO " << s.npseudo_ << ' ' <<
			s.bb_clones_.size() << ' ' <<
			s.chi_clones_.size() << ' ' <<
			s.jump_clones_.size() << ' ' <<
			s.symm_type();

		vector_map_to_comma_strings( out, s.bb_clones_ );
		vector_map_to_comma_strings( out, s.chi_clones_ );
		vector_map_to_comma_strings( out, s.jump_clones_ );

		return out;
	}
}
