// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/Quaterion.test.hh
/// @brief  test suite for numeric::Quaternion
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Test headers
#include <numeric/Quaternion.test.hh>

// Unit headers
#include <numeric/Quaternion.hh>
#include <numeric/Quaternion.io.hh>


namespace test {
namespace numeric {


typedef ::numeric::Quaternion_double QD;

// --------------- Fixtures --------------- //

// --- define a test fixture (some initial state that several tests share)
TEST_FIXTURE_BEGIN( fixture_Quaternion )

// Shared initialization goes here.  (invoked when TEST_FIXTURE_USE is called)
TEST_FIXTURE_SETUP_BEGIN( fixture_Quaternion )
{
	delta_percent = 0.0001;
}
TEST_FIXTURE_SETUP_END

// Shared finalization goes here.  (invoked when a test case that uses the fixture ends)
TEST_FIXTURE_TEARDOWN_BEGIN( fixture_Quaternion )
{
}
TEST_FIXTURE_TEARDOWN_END

// Shared data elements go here.  (They will be declared public.)

double delta_percent; // percentage difference for floating-point comparisons in TEST_CHECK_CLOSE

TEST_FIXTURE_END  // fixture_Quaternion


// --------------- Test Suites --------------- //

TEST_SUITE_BEGIN(Tests_Quaternion)
	TEST_SUITE_USES_CASE( test_Quaternion_constructors );
	TEST_SUITE_USES_CASE( test_Quaternion_normalize );
	TEST_SUITE_USES_CASE( test_Quaternion_conjugate );
TEST_SUITE_END


// --------------- Test Cases --------------- //

/// @brief Constructors
TEST_CASE_BEGIN(test_Quaternion_constructors)
{
	QD i( QD::identity() );
	QD q( 1.0, 0.0, 0.0, 0.0 );
	
	// test various identity quaternions
	TEST_CHECK_EQUAL( q, i );
	TEST_CHECK_EQUAL( q, QD::I() );
	TEST_CHECK_EQUAL( q.magnitude(), 1.0 );
	TEST_CHECK_EQUAL( q.w(), 1.0 );
	TEST_CHECK_EQUAL( q.x(), 0.0 );
	TEST_CHECK_EQUAL( q.y(), 0.0 );
	TEST_CHECK_EQUAL( q.z(), 0.0 );
	TEST_CHECK_EQUAL( q.w_squared(), 1.0 );
	TEST_CHECK_EQUAL( q.x_squared(), 0.0 );
	TEST_CHECK_EQUAL( q.y_squared(), 0.0 );
	TEST_CHECK_EQUAL( q.z_squared(), 0.0 );
	
	// applying identity to itself should stay identity
	q.apply( q );
	TEST_CHECK_EQUAL( q, i );
	TEST_CHECK_EQUAL( q, QD::I() );
	TEST_CHECK_EQUAL( q.magnitude(), 1.0 );
}
TEST_CASE_END

/// @brief test normalization
TEST_CASE_BEGIN( test_Quaternion_normalize )
{
	TEST_FIXTURE_USE( fixture_Quaternion, fix );
	
	QD q( QD::identity() );
	const_cast< double & >( q.x() ) = .02; // Intentionally make non-unit
	q.normalize_if_needed( .1 ); // Shouldn't do anything
	TEST_CHECK( q.magnitude_squared() >= 1.000399 );
	q.normalize_if_needed( .0001 ); // Should normalize
	TEST_CHECK( q.magnitude_squared_error() < 1.0E-15 );
	TEST_CHECK_CLOSE( q.magnitude(), 1.0, fix.delta_percent );
}
TEST_CASE_END

/// @brief test conjugates
TEST_CASE_BEGIN( test_Quaternion_conjugate )
{
	QD q1( 0.0, 1.0, 0.0, 0.0 );
	QD q2( q1.conjugated() );
	TEST_CHECK_EQUAL( q1 * q2, QD::I() );
	q2.conjugate();
	TEST_CHECK_EQUAL( q2, q1 );
}
TEST_CASE_END


} // namespace numeric
} // namespace test
