// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/xyzMatrix.test.hh
/// @brief  test suite for numeric::xyzMatrix
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Test headers
#include <numeric/xyzMatrix.test.hh>

// Package headers
#include <numeric/xyzVector.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzMatrix.io.hh>
//#include <numeric/xyz.functions.hh>
//#include <numeric/constants.hh>


namespace test {
namespace numeric {


// --------------- Fixtures --------------- //

// --- define a test fixture (some initial state that several tests share)
TEST_FIXTURE_BEGIN( fixture_xyzMatrix )

// Shared initialization goes here.  (invoked when TEST_FIXTURE_USE is called)
TEST_FIXTURE_SETUP_BEGIN( fixture_xyzMatrix )
{
	using ::numeric::xyzVector_float;
	using ::numeric::xyzMatrix_float;

	// set tolerance for floating-point comparisons
	delta_percent = 0.0001;
	
	// initialize vectors
	v1 = xyzVector_float( 1.0, 2.0, 3.0 );
	v2 = xyzVector_float( 4.0, 5.0, 6.0 );
	v3 = xyzVector_float( 7.0, 8.0, 9.0 );
	
	// initialize matrices
	m1 = xyzMatrix_float::rows( v1, v2, v3 );  // m1 has row-ordered values consecutive
	m2 = xyzMatrix_float::cols( v1, v2, v3 );  // m2 has column-ordered values consecutive
}
TEST_FIXTURE_SETUP_END

// Shared finalization goes here.  (invoked when a test case that uses the fixture ends)
TEST_FIXTURE_TEARDOWN_BEGIN( fixture_xyzMatrix )
{
}
TEST_FIXTURE_TEARDOWN_END

// Shared data elements go here.  Everything will be declared public.

// tolerance for floating-point comparisons (expressed as a percentage difference)
double delta_percent;

// three vectors used for initialization
::numeric::xyzVector_float v1;
::numeric::xyzVector_float v2;
::numeric::xyzVector_float v3;

// two matrices that get filled with values 1-9
::numeric::xyzMatrix_float m1;  // filled by rows
::numeric::xyzMatrix_float m2;  // filled by columns

TEST_FIXTURE_END  // fixture_xyzMatrix


// --------------- Test Suites --------------- //

// --- set up the top-level test suite
TEST_SUITE_BEGIN( Tests_xyzMatrix )
	TEST_SUITE_USES_SUITE( test::numeric, Tests_xyzMatrix_Construct );
	TEST_SUITE_USES_SUITE( test::numeric, Tests_xyzMatrix_Assign );

	TEST_SUITE_USES_CASE( test_xyzMatrix_Access );
	TEST_SUITE_USES_CASE( test_xyzMatrix_CalculatedProperties );
	TEST_SUITE_USES_CASE( test_xyzMatrix_Operations );
//	TEST_SUITE_USES_CASE( test_xyzMatrix_Rotation );  // might delete this one
TEST_SUITE_END


/// --- set up a suite for constructor tests
TEST_SUITE_BEGIN( Tests_xyzMatrix_Construct )
	TEST_SUITE_USES_CASE( test_xyzMatrix_ConstructIdentity );
	TEST_SUITE_USES_CASE( test_xyzMatrix_ConstructFromVectors );
	TEST_SUITE_USES_CASE( test_xyzMatrix_ConstructFromValues );
	TEST_SUITE_USES_CASE( test_xyzMatrix_ConstructFromContiguous );
TEST_SUITE_END

/// -- set up a suite for assignment tests
TEST_SUITE_BEGIN( Tests_xyzMatrix_Assign )
	TEST_SUITE_USES_CASE( test_xyzMatrix_AssignFromSingleValue );
	TEST_SUITE_USES_CASE( test_xyzMatrix_AssignFromContiguous );
	TEST_SUITE_USES_CASE( test_xyzMatrix_AssignFromRowsCols );
TEST_SUITE_END
	

// --------------- Test Cases --------------- //

/// @brief Identity matrix constructor
TEST_CASE_BEGIN( test_xyzMatrix_ConstructIdentity )
{
	using ::numeric::xyzMatrix_float;

	xyzMatrix_float m = xyzMatrix_float::identity();
	TEST_CHECK_EQUAL( m.xx(), 1.0 );
	TEST_CHECK_EQUAL( m.xy(), 0.0 );
	TEST_CHECK_EQUAL( m.xz(), 0.0 );
	
	TEST_CHECK_EQUAL( m.yx(), 0.0 );
	TEST_CHECK_EQUAL( m.yy(), 1.0 );
	TEST_CHECK_EQUAL( m.yz(), 0.0 );

	TEST_CHECK_EQUAL( m.zx(), 0.0 );
	TEST_CHECK_EQUAL( m.zy(), 0.0 );
	TEST_CHECK_EQUAL( m.zz(), 1.0 );
}
TEST_CASE_END


/// @brief Test construction from rows/cols
TEST_CASE_BEGIN( test_xyzMatrix_ConstructFromVectors )
{
	using ::numeric::xyzMatrix_float;
	using ::numeric::xyzVector_float;

	// invoke the test fixture, which will return an object named "fix"
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	// build a matrix from row vectors
	xyzMatrix_float m1( xyzMatrix_float::rows( fix.v1, fix.v2, fix.v3 ) );
	TEST_CHECK_EQUAL( m1.xx(), 1.0 );
	TEST_CHECK_EQUAL( m1.xy(), 2.0 );
	TEST_CHECK_EQUAL( m1.xz(), 3.0 );	
	TEST_CHECK_EQUAL( m1.yx(), 4.0 );
	TEST_CHECK_EQUAL( m1.yy(), 5.0 );
	TEST_CHECK_EQUAL( m1.yz(), 6.0 );
	TEST_CHECK_EQUAL( m1.zx(), 7.0 );
	TEST_CHECK_EQUAL( m1.zy(), 8.0 );
	TEST_CHECK_EQUAL( m1.zz(), 9.0 );

	// build a matrix from column vectors
	xyzMatrix_float m2( xyzMatrix_float::cols( fix.v1, fix.v2, fix.v3 ) );
	TEST_CHECK_EQUAL( m2.xx(), 1.0 );
	TEST_CHECK_EQUAL( m2.xy(), 4.0 );
	TEST_CHECK_EQUAL( m2.xz(), 7.0 );	
	TEST_CHECK_EQUAL( m2.yx(), 2.0 );
	TEST_CHECK_EQUAL( m2.yy(), 5.0 );
	TEST_CHECK_EQUAL( m2.yz(), 8.0 );
	TEST_CHECK_EQUAL( m2.zx(), 3.0 );
	TEST_CHECK_EQUAL( m2.zy(), 6.0 );
	TEST_CHECK_EQUAL( m2.zz(), 9.0 );

	// use the named row vector constructor
	xyzMatrix_float m3( xyzMatrix_float::rows_constructor( fix.v1, fix.v2, fix.v3 ) );
	TEST_CHECK_EQUAL( m3, fix.m1 );
	
	// use the named column vector constructor
	xyzMatrix_float m4( xyzMatrix_float::cols_constructor( fix.v1, fix.v2, fix.v3 ) );
	TEST_CHECK_EQUAL( m4, fix.m2 );
}
TEST_CASE_END


/// @brief Test construction from a list of values
TEST_CASE_BEGIN( test_xyzMatrix_ConstructFromValues )
{
	using ::numeric::xyzMatrix_float;

	// invoke the test fixture, which will return an object named "fix"
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	// put list of values in by rows
	xyzMatrix_float m1 = xyzMatrix_float::rows( 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0 );
	TEST_CHECK_EQUAL( m1, fix.m1 );

	// put list of values in by columns
	xyzMatrix_float m2 = xyzMatrix_float::cols( 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0 );
	TEST_CHECK_EQUAL( m2, fix.m2 );
}
TEST_CASE_END


/// @brief Test construction from a vector of contiguous values
TEST_CASE_BEGIN( test_xyzMatrix_ConstructFromContiguous )
{
	using ::numeric::xyzMatrix_float;

	// invoke the test fixture, which will return an object named "fix"
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	float v[ 10 ];
	for ( int i = 0; i <= 9; ++i ) {
		v[ i ] = i;
	}

	// test pointer to contiuguous values (row-ordered)
	xyzMatrix_float m1( xyzMatrix_float::rows( &v[ 1 ] ) );
	TEST_CHECK_EQUAL( m1, fix.m1 );

	// test pointer to contiuguous values (row-ordered)
	xyzMatrix_float m2( xyzMatrix_float::cols( &v[ 1 ] ) );
	TEST_CHECK_EQUAL( m2, fix.m2 );
	
	// test pointer to contiuguous row values
	xyzMatrix_float m3( xyzMatrix_float::rows( &v[ 1 ], &v[ 4 ], &v[ 7 ] ) );
	TEST_CHECK_EQUAL( m3, fix.m1 );

	// test pointer to contiuguous column values
	xyzMatrix_float m4( xyzMatrix_float::cols( &v[ 1 ], &v[ 4 ], &v[ 7 ] ) );
	TEST_CHECK_EQUAL( m4, fix.m2 );
}
TEST_CASE_END


/// @brief assignment using a single value
TEST_CASE_BEGIN( test_xyzMatrix_AssignFromSingleValue )
{
	using ::numeric::xyzMatrix_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	xyzMatrix_float m1;
	m1 = 1.0f;
	TEST_CHECK_EQUAL( m1.xx(), 1.0 );
	TEST_CHECK_EQUAL( m1.xy(), 1.0 );
	TEST_CHECK_EQUAL( m1.xz(), 1.0 );	
	TEST_CHECK_EQUAL( m1.yx(), 1.0 );
	TEST_CHECK_EQUAL( m1.yy(), 1.0 );
	TEST_CHECK_EQUAL( m1.yz(), 1.0 );
	TEST_CHECK_EQUAL( m1.zx(), 1.0 );
	TEST_CHECK_EQUAL( m1.zy(), 1.0 );
	TEST_CHECK_EQUAL( m1.zz(), 1.0 );
}
TEST_CASE_END


/// @brief assignment using contiguous values
TEST_CASE_BEGIN( test_xyzMatrix_AssignFromContiguous )
{
	using ::numeric::xyzMatrix_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	float v[ 10 ];
	for ( int i = 0; i <= 9; ++i ) {
		v[ i ] = i;
	}

	// row-ordered assignment from contiguous values
	xyzMatrix_float m1;
	m1 = xyzMatrix_float::rows( &v[ 1 ] );
	TEST_CHECK_EQUAL( m1, fix.m1 );

	// column-ordered assignment from contiguous values
	xyzMatrix_float m2;
	m2 = xyzMatrix_float::cols( &v[ 1 ] );
	TEST_CHECK_EQUAL( m2, fix.m2 );
}
TEST_CASE_END


/// @brief assignment using rows/cols
TEST_CASE_BEGIN( test_xyzMatrix_AssignFromRowsCols )
{
	using ::numeric::xyzMatrix_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	// Assign individual rows using row_*
	xyzMatrix_float m1;
	m1.row_x( fix.v1 );
	m1.row_y( fix.v2 );
	m1.row_z( fix.v3 );
	TEST_CHECK_EQUAL( m1, fix.m1 );

	// Assign individual columns using col_*
	xyzMatrix_float m2;
	m2.col_x( fix.v1 );
	m2.col_y( fix.v2 );
	m2.col_z( fix.v3 );
	TEST_CHECK_EQUAL( m2, fix.m2 );

	// assign individual rows using row index
	xyzMatrix_float m3;
	m3.row( 1, fix.v1 );
	m3.row( 2, fix.v2 );
	m3.row( 3, fix.v3 );
	TEST_CHECK_EQUAL( m3, fix.m1 );

	// assign individual columsn using column index
	xyzMatrix_float m4;
	m4.col( 1, fix.v1 );
	m4.col( 2, fix.v2 );
	m4.col( 3, fix.v3 );
	TEST_CHECK_EQUAL( m4, fix.m2 );
}
TEST_CASE_END


/// @brief assignment using rows/cols
TEST_CASE_BEGIN( test_xyzMatrix_Access )
{
	using ::numeric::xyzMatrix_float;
	using ::numeric::xyzVector_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	// access rows using row_*
	xyzVector_float rx = fix.m1.row_x();
	xyzVector_float ry = fix.m1.row_y();
	xyzVector_float rz = fix.m1.row_z();
	TEST_CHECK_EQUAL( rx, fix.v1 );
	TEST_CHECK_EQUAL( ry, fix.v2 );
	TEST_CHECK_EQUAL( rz, fix.v3 );

	// access rows using row index
	xyzVector_float r1 = fix.m1.row( 1 );
	xyzVector_float r2 = fix.m1.row( 2 );
	xyzVector_float r3 = fix.m1.row( 3 );
	TEST_CHECK_EQUAL( r1, fix.v1 );
	TEST_CHECK_EQUAL( r2, fix.v2 );
	TEST_CHECK_EQUAL( r3, fix.v3 );

	// access columns using col_*
	xyzVector_float cx = fix.m2.col_x();
	xyzVector_float cy = fix.m2.col_y();
	xyzVector_float cz = fix.m2.col_z();
	TEST_CHECK_EQUAL( cx, fix.v1 );
	TEST_CHECK_EQUAL( cy, fix.v2 );
	TEST_CHECK_EQUAL( cz, fix.v3 );

	// access columns using column index
	xyzVector_float c1 = fix.m2.col( 1 );
	xyzVector_float c2 = fix.m2.col( 2 );
	xyzVector_float c3 = fix.m2.col( 3 );
	TEST_CHECK_EQUAL( c1, fix.v1 );
	TEST_CHECK_EQUAL( c2, fix.v2 );
	TEST_CHECK_EQUAL( c3, fix.v3 );
}
TEST_CASE_END


/// @brief test calculated properties
TEST_CASE_BEGIN( test_xyzMatrix_CalculatedProperties )
{
	using ::numeric::xyzMatrix_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	xyzMatrix_float i = xyzMatrix_float::identity();

	// compute some determinants
	TEST_CHECK_CLOSE( i.det(), 1.0f, fix.delta_percent );	
	TEST_CHECK_CLOSE( fix.m1.det(), 0.0f, fix.delta_percent );
	
	// compute some traces
	TEST_CHECK_CLOSE( i.trace(), 3.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( fix.m1.trace(), 15.0f, fix.delta_percent );
}
TEST_CASE_END


/// @brief operations tests
TEST_CASE_BEGIN( test_xyzMatrix_Operations )
{
	using ::numeric::xyzMatrix_float;
	using ::numeric::xyzVector_float;
	TEST_FIXTURE_USE( fixture_xyzMatrix, fix );

	// multiply by a scalar
	xyzMatrix_float m1_doubled = xyzMatrix_float::rows( 2.0, 4.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 18.0 );
	TEST_CHECK_EQUAL( 2 * fix.m1, m1_doubled );
	TEST_CHECK_EQUAL( fix.m1 * 2, m1_doubled );

	// matrix/matrix product
	xyzMatrix_float m1 = fix.m1;
	xyzMatrix_float m3 = xyzMatrix_float::rows( 1.0, 5.0, 2.0, 6.0, 8.0, 9.0, 4.0, 7.0, 3.0 );
	m1.left_multiply_by( m3 );
	TEST_CHECK_CLOSE( m1.xx(),  35.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.xy(),  43.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.xz(),  51.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.yx(), 101.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.yy(), 124.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.yz(), 147.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.zx(),  53.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.zy(),  67.0f, fix.delta_percent );
	TEST_CHECK_CLOSE( m1.zz(),  81.0f, fix.delta_percent );
}
TEST_CASE_END


} // namespace numeric
} // namespace test
