// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/factory/Factory.test.cc
/// @brief  utility::factory::Factory test suite
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Package headers
#include <utility/factory/Factory.test.hh>
#include <utility/factory/Factory.hh>
#include <utility/factory/Registrant.hh>

// Project headers
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <string>


namespace test {
namespace utility {
namespace factory {


/// @brief Product interface class
class Product :
	public ::utility::pointer::ReferenceCount
{


public: // Types


	typedef  ::utility::pointer::owning_ptr< Product >  ProductOP;

	typedef  ::utility::factory::Factory< Product >  Factory;
	typedef  std::string const  FactoryKey;
	typedef  std::string const *  FactoryKeyP;
	typedef  FactoryKey & (*FactoryKeyFxn)(); // Key function
	typedef  ProductOP  FactoryProductP;
	typedef  FactoryProductP (*FactoryCreate)(); // Creation function: 0 arg case
	typedef  int  FactoryArg; // Not used here


protected: // Creation


	/// @brief Constructor
	inline
	Product()
	{}


public: // Creation


	/// @brief Destructor
	inline
	virtual
	~Product()
	{}


public: // Methods


	/// @brief Name
	virtual
	std::string const &
	name() const = 0;


}; // Product


/// @brief Concrete Product A
class ProductA : public Product
{


public: // Types


	typedef  Product  Super;


public: // Creation


	/// @brief Default constructor
	inline
	ProductA()
	{}


	/// @brief Copy constructor
	inline
	ProductA( ProductA const & a ) :
		Super( a )
	{}


	/// @brief Factory creation
	inline
	static
	ProductOP
	factory_create()
	{
		return new ProductA();
	}


	/// @brief Destructor
	inline
	virtual
	~ProductA()
	{}


public: // Methods


	inline
	std::string const &
	name() const
	{
		static std::string const name_( "ProductA" );
		return name_;
	}


private: // Static fields


	/// @brief Factory registrant: Registers this class with Factory
	static ::utility::factory::Registrant< Product > const factory_registrant_;


}; // ProductA

// Static field definitions
::utility::factory::Registrant< Product > const ProductA::factory_registrant_( "ProductA", ProductA::factory_create );


/// @brief Concrete Product B
class ProductB : public Product
{


public: // Types


	typedef  Product  Super;


public: // Creation


	/// @brief Default constructor
	inline
	ProductB()
	{}


	/// @brief Copy constructor
	inline
	ProductB( ProductB const & b ) :
		Super( b )
	{}


	/// @brief Factory creation
	inline
	static
	ProductOP
	factory_create()
	{
		return new ProductB();
	}


	/// @brief Destructor
	inline
	virtual
	~ProductB()
	{}


public: // Methods


	inline
	std::string const &
	name() const
	{
		static std::string const name_( "ProductB" );
		return name_;
	}


private: // Static fields


	/// @brief Factory registrant: Registers this class with Factory
	static ::utility::factory::Registrant< Product > const factory_registrant_;


}; // ProductB

// Static field definitions
::utility::factory::Registrant< Product > const ProductB::factory_registrant_( "ProductB", ProductB::factory_create );


// --- set up the test suite
TEST_SUITE_BEGIN(Factory_Tests)
	TEST_SUITE_USES_CASE(test_Factory_general)
TEST_SUITE_END


// --- set up the individual test cases

/// @brief General tests
TEST_CASE_BEGIN(test_Factory_general)
{

	{ // ProductA
		TEST_CHECK( Product::Factory::create( "ProductA" )->name() == "ProductA" );
	}

	{ // ProductB
		TEST_CHECK( Product::Factory::create( "ProductB" )->name() == "ProductB" );
	}

}
TEST_CASE_END


} // namespace factory
} // namespace utility
} // namespace test
