// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/vector1_bool.test.cc
/// @brief  vector1_bool.test: test suite for utility::vector1_bool
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Package headers
#include <utility/vector1_bool.test.hh>
#include <utility/vector1_bool.hh>

// C++ headers
#include <vector>


namespace test {
namespace utility {


// --- Types
typedef  std::vector< bool >  std_vector_bool;

// --- set up the test suite
TEST_SUITE_BEGIN(Vector1_Bool_Tests)
	TEST_SUITE_USES_CASE(test_vector1_bool_constructor)
	TEST_SUITE_USES_CASE(test_vector1_bool_assignment)
	TEST_SUITE_USES_CASE(test_vector1_bool_copy)
	TEST_SUITE_USES_CASE(test_vector1_bool_compare_to_std)
	TEST_SUITE_USES_CASE(test_vector1_bool_swap)
	TEST_SUITE_USES_CASE(test_vector1_bool_flip)
//	TEST_SUITE_USES_CASE(test_vector1_bool_bounds_check)  // not used yet
TEST_SUITE_END

// --- set up the individual test cases

/// @brief Size + Value Constructor Test
TEST_CASE_BEGIN(test_vector1_bool_constructor)
{
	unsigned int vector_size = 5;
	::utility::vector1_bool v( vector_size, false );

	TEST_CHECK_EQUAL(v.size(), vector_size);
	TEST_CHECK_EQUAL(v[1], false);
	TEST_CHECK_EQUAL(v[2], false);
	TEST_CHECK_EQUAL(v[3], false);
	TEST_CHECK_EQUAL(v[4], false);
	TEST_CHECK_EQUAL(v[5], false);
}
TEST_CASE_END


/// @brief Assignment test
/// VERIFY THIS ONE!
TEST_CASE_BEGIN(test_vector1_bool_assignment)
{
	::utility::vector1_bool v( 3, false );
	::utility::vector1_bool w( 3, true );

	v = w;
	TEST_CHECK_EQUAL( v[1], true );
	TEST_CHECK_EQUAL( v[2], true );
	TEST_CHECK_EQUAL( v[3], true );
}
TEST_CASE_END


/// @brief Copy constructor test
/// VERIFY THIS ONE!
TEST_CASE_BEGIN(test_vector1_bool_copy)
{
	::utility::vector1_bool v( 3, false );
	::utility::vector1_bool w( v );

	TEST_CHECK_EQUAL( w[1], false );
	TEST_CHECK_EQUAL( w[2], false );
	TEST_CHECK_EQUAL( w[3], false );
}
TEST_CASE_END


/// @brief Test comparison to std::vector
TEST_CASE_BEGIN(test_vector1_bool_compare_to_std)
{
	::utility::vector1_bool v( 3, false );
	std_vector_bool  w( 3, false );

	TEST_CHECK_EQUAL_COLLECTION(v, w);
	TEST_CHECK( v == w );
	TEST_CHECK( !( v != w ) );
	TEST_CHECK( !( v < w ) );
	TEST_CHECK( v <= w );
	TEST_CHECK( !( v > w ) );
	TEST_CHECK( v >= w );
	TEST_CHECK( w == v );
	TEST_CHECK( !( w != v ) );
	TEST_CHECK( !( w < v ) );
	TEST_CHECK( w <= v );
	TEST_CHECK( !( w > v ) );
	TEST_CHECK( w >= v );
}
TEST_CASE_END


/// @brief Swap test
TEST_CASE_BEGIN(test_vector1_bool_swap)
{
	// set up first vector
	::utility::vector1_bool v( 3 );  v[1] = true; v[2] = false; v[3] = true;
	TEST_CHECK_EQUAL(v[1], true);
	TEST_CHECK_EQUAL(v[2], false);
	TEST_CHECK_EQUAL(v[3], true);

	// set up second vector and verify equal
	::utility::vector1_bool w( v );
	TEST_CHECK_EQUAL_COLLECTION(v, w);

	// alter second vector and verify not equal
	w[2] = true;
	TEST_CHECK_EQUAL(w[1], true);
	TEST_CHECK_EQUAL(w[2], true);
	TEST_CHECK_EQUAL(w[3], true);
	TEST_CHECK( v != w );
	TEST_CHECK( w != v );

	// make fixed copies of each (for reference)
	::utility::vector1_bool const V( v );
	::utility::vector1_bool const W( w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);

	// verify our swap
	::utility::swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, W);
	TEST_CHECK_EQUAL_COLLECTION(w, V);

	// verify std::swap
	std::swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);

	// verify swap as method call
	v.swap( w );
	TEST_CHECK_EQUAL_COLLECTION(v, W);
	TEST_CHECK_EQUAL_COLLECTION(w, V);

	// Let C++ pick best swap match from std or utility
	// (This one might not actually test Koenig lookup, according to Ion?)
	using namespace std;
	swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);
}
TEST_CASE_END


/// @brief Flip test (specific to bool vectors)
TEST_CASE_BEGIN(test_vector1_bool_flip)
{
	::utility::vector1_bool v( 3 );  v[1] = true; v[2] = false; v[3] = true;

	// check starting condition
	TEST_CHECK_EQUAL(v[1], true);
	TEST_CHECK_EQUAL(v[2], false);
	TEST_CHECK_EQUAL(v[3], true);

	// flip the whole vector
	v.flip();
	TEST_CHECK_EQUAL(v[1], false);
	TEST_CHECK_EQUAL(v[2], true);
	TEST_CHECK_EQUAL(v[3], false);

	// flip a single element
	v[1].flip();
	TEST_CHECK_EQUAL(v[1], true);
	TEST_CHECK_EQUAL(v[2], true);
	TEST_CHECK_EQUAL(v[3], false);

	// assign from an element
	v[3] = v[2];
	TEST_CHECK_EQUAL(v[1], true);
	TEST_CHECK_EQUAL(v[2], true);
	TEST_CHECK_EQUAL(v[3], true);
}
TEST_CASE_END


/// @brief Test of bounds checking
/// @note  This test is not used yet!  With the existing testing system,
///        it would cause an assert() to fail and abort testing.  But perhaps
///        it could be included in a future version that can catch runtime failures.
TEST_CASE_BEGIN(test_vector1_bool_bounds_check)
{
	::utility::vector1_bool v( 5, false );
	TEST_CHECK_EQUAL(v[1000], false); // should fail -- index out of bounds
}
TEST_CASE_END


} // namespace utility
} // namespace test
