// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/vectorL.test.cc
/// @brief  vectorL.test: test suite for utility::vectorL
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Package headers
#include <utility/vectorL.test.hh>
#include <utility/vectorL.hh>

// C++ headers
#include <vector>


namespace test {
namespace utility {


// --- Types
typedef  std::vector< int >  std_vector_int;

// --- set up the test suite
TEST_SUITE_BEGIN(VectorL_Tests)
	TEST_SUITE_USES_CASE(test_vectorL_constructor_minus2)
	TEST_SUITE_USES_CASE(test_vectorL_constructor_plus2)
	TEST_SUITE_USES_CASE(test_vectorL_has_index)
	TEST_SUITE_USES_CASE(test_vectorL_assignment)
	TEST_SUITE_USES_CASE(test_vectorL_copy)
	TEST_SUITE_USES_CASE(test_vectorL_compare_to_std)
	TEST_SUITE_USES_CASE(test_vectorL_swap)
	TEST_SUITE_USES_CASE(test_vectorL_shrink)
//	TEST_SUITE_USES_CASE(test_vectorL_bounds_check)  // not used yet
TEST_SUITE_END

// --- set up the individual test cases

/// @brief Size + Value Constructor Test (starting at -2)
TEST_CASE_BEGIN(test_vectorL_constructor_minus2)
{
	unsigned int vector_size = 5;
	::utility::vectorL< -2, int > v( vector_size, 99 );

	TEST_CHECK_EQUAL(v.size(), vector_size);
	TEST_CHECK_EQUAL(v[-2], 99);
	TEST_CHECK_EQUAL(v[-1], 99);
	TEST_CHECK_EQUAL(v[0], 99);
	TEST_CHECK_EQUAL(v[1], 99);
	TEST_CHECK_EQUAL(v[2], 99);
}
TEST_CASE_END

/// @brief Size + Value Constructor Test (starting at +2)
TEST_CASE_BEGIN(test_vectorL_constructor_plus2)
{
	unsigned int vector_size = 5;
	::utility::vectorL< 2, int > v( vector_size, 99 );

	TEST_CHECK_EQUAL(v.size(), vector_size);
	TEST_CHECK_EQUAL(v[2], 99);
	TEST_CHECK_EQUAL(v[3], 99);
	TEST_CHECK_EQUAL(v[4], 99);
	TEST_CHECK_EQUAL(v[5], 99);
	TEST_CHECK_EQUAL(v[6], 99);
}
TEST_CASE_END


/// @brief Test the has() method
TEST_CASE_BEGIN(test_vectorL_has_index)
{
	unsigned int vector_size = 5;

	// test with bounds from -2 to 2
	::utility::vectorL< -2, int >  v( vector_size, 99 );
	TEST_CHECK( v.has( -2 ) );
	TEST_CHECK( v.has( 2 ) );
	TEST_CHECK( ! v.has( -3 ) );
	TEST_CHECK( ! v.has( 3 ) );

	// test with bounds from +2 to +6
	::utility::vectorL<  2, int >  w( vector_size, 99 );
	TEST_CHECK( w.has( 2 ) );
	TEST_CHECK( w.has( 6 ) );
	TEST_CHECK( ! w.has( 1 ) );
	TEST_CHECK( ! w.has( 7 ) );
}
TEST_CASE_END


/// @brief Assignment test
/// VERIFY THIS ONE!
TEST_CASE_BEGIN(test_vectorL_assignment)
{
	typedef  ::utility::vectorL< 0, int >  Vector;
	Vector v( 3, 99 );
	Vector w( 3, 88 );

	v = w;
	TEST_CHECK_EQUAL( v[0], 88 );
	TEST_CHECK_EQUAL( v[1], 88 );
	TEST_CHECK_EQUAL( v[2], 88 );
}
TEST_CASE_END


/// @brief Copy constructor test
/// VERIFY THIS ONE!
TEST_CASE_BEGIN(test_vectorL_copy)
{
	typedef  ::utility::vectorL< 0, int >  Vector;
	Vector v( 3, 99 );
	Vector w( v );

	TEST_CHECK_EQUAL( w[0], 99 );
	TEST_CHECK_EQUAL( w[1], 99 );
	TEST_CHECK_EQUAL( w[2], 99 );
}
TEST_CASE_END


/// @brief Test comparison to std::vector
TEST_CASE_BEGIN(test_vectorL_compare_to_std)
{
	typedef  ::utility::vectorL< 0, int >  Vector;
	Vector v( 3, 99 );       // our version
	std_vector_int  w( 3, 99 );  // the std::vector version
	TEST_CHECK_EQUAL_COLLECTION(v, w);
	TEST_CHECK( v == w );
	TEST_CHECK( !( v != w ) );
	TEST_CHECK( !( v < w ) );
	TEST_CHECK( v <= w );
	TEST_CHECK( !( v > w ) );
	TEST_CHECK( v >= w );
	TEST_CHECK( w == v );
	TEST_CHECK( !( w != v ) );
	TEST_CHECK( !( w < v ) );
	TEST_CHECK( w <= v );
	TEST_CHECK( !( w > v ) );
	TEST_CHECK( w >= v );
}
TEST_CASE_END


/// @brief Swap test
TEST_CASE_BEGIN(test_vectorL_swap)
{
	// set up first vector
	typedef  ::utility::vectorL< 0, int >  Vector;
	Vector v( 3 );  v[0] = 1; v[1] = 2; v[2] = 3;
	TEST_CHECK_EQUAL(v[0], 1);
	TEST_CHECK_EQUAL(v[1], 2);
	TEST_CHECK_EQUAL(v[2], 3);

	// set up second vector and verify equal
	Vector w( v );
	TEST_CHECK_EQUAL_COLLECTION(v, w);

	// alter second vector and verify not equal
	w[0] = 5;
	TEST_CHECK_EQUAL(w[0], 5);
	TEST_CHECK_EQUAL(w[1], 2);
	TEST_CHECK_EQUAL(w[2], 3);
	TEST_CHECK( v != w );
	TEST_CHECK( w != v );

	// make fixed copies of each (for reference)
	Vector const V( v );
	Vector const W( w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);

	// verify our swap
	::utility::swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, W);
	TEST_CHECK_EQUAL_COLLECTION(w, V);

	// verify std::swap
	std::swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);

	// verify swap as method call
	v.swap( w );
	TEST_CHECK_EQUAL_COLLECTION(v, W);
	TEST_CHECK_EQUAL_COLLECTION(w, V);

	// Let C++ pick best swap match from std or utility
	// (This one might not actually test Koenig lookup, according to Ion?)
	using namespace std;
	swap( v, w );
	TEST_CHECK_EQUAL_COLLECTION(v, V);
	TEST_CHECK_EQUAL_COLLECTION(w, W);
}
TEST_CASE_END


/// @brief Shrink test
TEST_CASE_BEGIN(test_vectorL_shrink)
{
	typedef  ::utility::vectorL< 0, int >  Vector;
	Vector v( 222, 33 );
	v.push_back( 44 ); // Generally this will cause capacity > size
	v.shrink();
	TEST_CHECK_EQUAL( v.size(), v.capacity() );
}
TEST_CASE_END


/// @brief Test of bounds checking
/// @note  This test is not used yet!  With the existing testing system,
///        it would cause an assert() to fail and abort testing.  But perhaps
///        it could be included in a future version that can catch runtime failures.
TEST_CASE_BEGIN(test_vectorL_bounds_check)
{
	typedef ::utility::vectorL< 0, int > Vector;
	Vector v( 5, 99 );
	TEST_CHECK_EQUAL(v[1000], 99); // should fail -- index out of bounds
}
TEST_CASE_END


} // namespace utility
} // namespace test
