// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14743 $
//  $Date: 2007-05-05 01:01:44 -0400 (Sat, 05 May 2007) $
//  $Author: bqian $


// Rosetta Headers
#include "torsion_bbmove_trials.h"
#include "chuck.h"
#include "counters.h"
#include "crankshaft.h"
#include "current_pose.h"
#include "elliptic_msd.h"
#include "force_barcode.h"
#include "fragments.h"
#include "gunn.h"
//#include "gl_graphics.h"
#include "jumping_loops.h"
#include "loops.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "force_barcode.h"
#include "param.h"
#include "pose.h"
#include "pose_rotamer_trials.h"
#include "pose_symmetric_docking.h"
#include "recover.h"
#include "refold.h"
#include "score.h"
#include "smallmove.h"
#include "status.h"
#include "wobble.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <iostream>


//car  all fragment trials must leave the phi psi arrays current
//car  (ie call reset_phipsi if a move was not accepted)
//car  position array need not be left current

//car   main_trial_wobble (chuck move, end angles minimized)
//car   main_trial_gunn_wobble (smooth move, end angles minimized)
//car   main_small_trial    (single residue random angle move)
//car   main_chuck_trial  (frag with small change in rot and offset from current)
//car   main_frag_smooth_trial(frag with small change from current, gunn method)
//car   torsion_move_trials_trial (straight frag insertion)
//car   main_double_trial     (2 chuck insertions + wobble)

////////////////////////////////////////////////////////////////////////////////
/// @begin main_wobble_trial
///
/// @brief
/// chuck move followed by wobble (powell minimization of phi/psi) to
/// minimize change in MSD of the position array
///
/// @detailed
///
/// @param  size - [in/out]? - fragment size
/// @param  cutoff_max - [in/out]? - chuck cutoff, if <0 chuck sets
///        cutoff automatic (rec value: 60)
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? - cycle_number to send to mc
///        (>100000 enables convergence check)
/// @param  nwobble - [in/out]? - wobble this many residues
/// @param  max_wobble_gap - [in/out]? - position of first wobbled residue
///        relative to frag end, -1 is first or last frag res,
///        0 is adjacent to frag, positive values are outside frag
///        if max_wobble_gap>0, the actual wobble_gap will be
///        randomly set between 1 and max_wobble_gap
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_wobble_trial(
	int size,
	float cutoff_max,
	Scoring_Function score_fxn,
	int cycle_number,
	int nwobble,
	int max_wobble_gap
)
{
//car local

	int frag_begin,frag_end;
	bool gfrag( true );

	chuck_wobble_move(size,cutoff_max,nwobble,max_wobble_gap,frag_begin,frag_end,
	 gfrag);

	if ( !gfrag ) return;

	mc_global_track::mc_score::score = score_fxn();
	save_status_info("wobble",frag_begin,frag_end-frag_begin+1);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_gunn_wobble_trial
///
/// @brief
///car gunn (or smooth) frag insertion, followed by wobble (powell minimization)
///car see comments to main_trial_wobble
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  wobble_gap - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car NOTE: this function hasn't been extensively debugged; use at your
///car own risk
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_gunn_wobble_trial(
	int size,
	float cutoff_max,
	Scoring_Function score_fxn,
	int cycle_number,
	int nwobble,
	int wobble_gap
)
{
	using namespace misc;

//car local
	int ntries_wobble,nfail;
	int total_begin,total_end;
	int min_frag_begin,max_frag_begin;
	float rama_score,dist_score;
	int frag_begin;
	int loop_begin,loop_end,loop_num;
	bool gfrag( true ); // if true, powell minimized sucessfully

	gfrag = true;
	ntries_wobble = 0;
	nfail = 0;

L170:; // loop back here if  wobble fails

L171:; // loop back here if choose gunn fails

	retrieve_best_pose();
	choose_fragment_gunn(size,cutoff_max,frag_begin);
	std::cout << "gunn frag begin " << frag_begin << std::endl;
	if ( get_loop_flag() ) {
		identify_loop(frag_begin,frag_begin,loop_begin,loop_end,loop_num);
		if ( loop_begin != 1 && loop_end != total_residue ) {
			min_frag_begin = std::max(loop_begin,loop_begin+nwobble+wobble_gap);
			max_frag_begin = loop_end-std::max(size,size+nwobble+wobble_gap);
		} else {
			min_frag_begin = std::max(loop_begin+3,loop_begin+3+nwobble+wobble_gap);
			max_frag_begin = loop_end-std::max(size+4,size+4+nwobble+wobble_gap);
		}
	} else {
		min_frag_begin = std::max(4,4+nwobble+wobble_gap);
		max_frag_begin = total_residue-std::max(size+4,size+4+nwobble+wobble_gap);
	}


	if ( min_frag_begin > max_frag_begin || frag_begin > max_frag_begin ||
	 frag_begin < min_frag_begin) {
		resetphipsi();
		++nfail;
		if ( nfail > 1000 ) {
			std::cout << "Can't find an insertion point" << std::endl;
			return;
		}
		goto L171; // too close to end
	}

	refold(frag_begin,frag_begin+size-1);
	add_fast_wobble(frag_begin,size,nwobble,wobble_gap,rama_score,dist_score,
	 gfrag,total_begin,total_end);

	if ( !gfrag ) return;

	++ntries_wobble;
	if ( rama_score > 7.0 || dist_score > 30.0 ) {
		resetphipsi();
		goto L170; // try again
	}

	mc_global_track::mc_score::score = score_fxn();
	save_status_info("wobble",total_begin,total_end-total_begin+1);
	monte_carlo(cycle_number);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_small_trial
///
/// @brief
///car random small change in phi/psi of a single residue
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_small_trial(
	Scoring_Function score_fxn,
	int cycle_number
)
{
	int const NMOVES = { 1 };

	FArray1D_int list( NMOVES ); // position of small move
	int first,last,length;

	small_moves(NMOVES,list,length,first,last);
	if ( length == 0 ) return; // small_move failed
	refold(first,last);
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("small",list(1),1);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_shear_trial
///
/// @brief
///car random small change in phi/psi of a single residue
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_shear_trial(
	Scoring_Function score_fxn,
	int cycle_number
)
{
	int const NMOVES = { 1 };

	FArray1D_int list( NMOVES*2 ); // position of small move
	int first,last,length;

	shear_moves(NMOVES,list,length,first,last);
	if ( length == 0 ) return;
	refold(first,last);
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("shear",list(1),1);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_chuck_trial
///
/// @brief
///car insert frag that changes msd of position array minimally
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  tag1 - [in/out]? - tag1, tag2: min and max frag insertion points
/// @param  tag2 - [in/out]? - tag2 also sent as cycle number to monte carlo
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_chuck_trial(
	int size,
	float & cutoff_max,
	Scoring_Function score_fxn,
	int tag1,
	int tag2
)
{
	int size_bin,best_frag,frag_begin;
	float cost;
	float chuck_cutoff;

	chuck_cutoff = cutoff_max; // local copy, may get changed
	choose_chuck(size,chuck_cutoff,tag1,tag2,best_frag,cost,frag_begin);

	if ( best_frag >= 0 ) {
		size_bin = get_index_by_frag_size(size);
		insert_frag(size,size_bin,frag_begin,best_frag);
		refold(frag_begin,frag_begin+size-1);
		mc_global_track::mc_score::score = score_fxn();
		save_status_info("chuck",frag_begin,size);
		monte_carlo(tag2);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_smooth_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_smooth_trial(
	int size,
	float & cutoff,
	Scoring_Function score_fxn,
	int cycle_number
)
{
	int frag_begin;

	retrieve_best_pose();
	choose_fragment_gunn(size,cutoff,frag_begin);
	refold(frag_begin,frag_begin+size-1);
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("smooth",frag_begin,size);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_frag_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///pb  size may be changed by choose_fragment in jumping mode, but
///pb  this function is often called with a constant value for size
///pb  hence the local var size_inout
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_frag_trial(
	int const size,
	Scoring_Function score_fxn,
	int const cycle_number
)
{
	int begin;
	int size_inout = size; // see notes
	choose_fragment(size_inout,begin);
	refold(begin,begin+size_inout-1);
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("standard",begin,size);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_frag_cst_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  tolerance - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_frag_cst_trial(
	int size,
	Scoring_Function score_fxn,
	int cycle_number,
	float tolerance
)
{

//car monte_carlo_reset, not monte_carlo

	int frag_begin;

	choose_fragment_cst(tolerance,size,frag_begin);
	refold(frag_begin,frag_begin+size-1);
	mc_global_track::mc_score::score = score_fxn();
	save_status_info("trial",frag_begin,size);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_double_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? - chuck cutoff for first frag insertion
/// @param  score_fxn - [in/out]? -
/// @param  min_spacing - [in/out]? - min distance between frags
/// @param  max_spacing - [in/out]? - max distance between frags
///        also sent as cycle number to monte_carlo
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_double_trial(
	int size,
	float & cutoff_max,
	Scoring_Function score_fxn,
	int & min_spacing,
	int & max_spacing
)
{
//car local
	int frag_end,frag_begin;

	if ( get_loop_flag() ) {
		std::cout << "no double moves allowed in loops" << std::endl;
		return;
	}

	double_move(size,cutoff_max,min_spacing,max_spacing,frag_begin,frag_end);
	if ( frag_begin == 0 && frag_end == 0 ) return; // double_move failed

	mc_global_track::mc_score::score = score_fxn();
	save_status_info("double",frag_begin,frag_end-frag_begin+1);
	monte_carlo(max_spacing);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_coupled_phipsi_trial
///
/// @brief
/// ctsa - bill w's coupled phipsi move,
/// call with minimize_type = 'nomin' for non-min move
///
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  minimize_type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_coupled_phipsi_trial(
	int nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & minimize_type
)
{
	using namespace param;

	std::string move_type;

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length,begin,end;
	bool gfrag( true );
//------------------------------------------------------------------------------
	move_type = "couple";
	if ( minimize_type != "nomin" ) {
		move_type += "_min";
	}

	coupled_phipsi_moves(nmoves,move_list,length,begin,end);
	if ( length == 0 ) return; // coupled_moves failed
	refold(begin,end);

	if ( minimize_type != "nomin" ) {
		minimize_exclude_residues(move_list,length);
		minimize( minimize_type, move_type, score_fxn, begin, end, gfrag );
		if ( !gfrag ) return;
	} else {
		mc_global_track::mc_score::score = score_fxn();
	}

	save_status_info( move_type, move_list(1), nmoves );
	monte_carlo( cycle_number );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin main_minimize_trial
///
/// @brief
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car note this trial _forces_ an accept by monte_carlo
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_minimize_trial(
	Scoring_Function score_fxn,
	std::string const & type
)
{
	using namespace misc;

	std::string const move_type("minimize");

	retrieve_best_pose();

	bool gfrag( true );
	minimize(type,move_type,score_fxn,total_residue+1,0,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,1,total_residue);
	monte_carlo_reset();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_frag_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///pb  size may be changed by choose_fragment in jumping mode, but
///pb  this function is often called with a constant value for size
///pb  hence the local var size_inout
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_frag_min_trial(
	int const size,
	Scoring_Function score_fxn,
	int const cycle_number,
	std::string const & type
)
{
	std::string const move_type( "standardmin" );

//car local
	int begin;
	bool gfrag( true );

	int size_inout = size;
	choose_fragment( size_inout, begin );
	refold( begin, begin + size_inout - 1 );

	if ( get_loop_fold_with_dunbrack() ) close_loop_after_moves();

	minimize_exclude_frag( begin, size_inout );
	minimize( type, move_type, score_fxn, begin, begin + size-1, gfrag );

	save_status_info( move_type, begin, size );
	monte_carlo( cycle_number );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_double_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff_max - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  min_space - [in/out]? - min distance between frags
/// @param  max_space - [in/out]? - max distance between frags
///        also sent as cycle_number to monte_carlo
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_double_min_trial(
	int & size,
	float & cutoff_max,
	Scoring_Function score_fxn,
	int & min_space,
	int & max_space,
	std::string const & type
)
{
	std::string const move_type( "doublemin" ); //

//car local
	int frag_end,frag_begin;
	bool gfrag( true );

	if ( get_loop_flag() ) {
//W		std::cout << "no double moves allowed in loops" << std::endl;
		return;
	}

	double_move(size,cutoff_max,min_space,max_space,frag_begin,frag_end);
	if ( frag_begin == 0 && frag_end == 0 ) return; // double_move failed

	minimize_exclude_frag(frag_begin,frag_end-frag_begin+1);
	minimize(type,move_type,score_fxn,frag_begin,frag_end,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,frag_begin,frag_end-frag_begin+1);
	monte_carlo(max_space);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_smooth_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  size - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_smooth_min_trial(
	int & size,
	float & cutoff,
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	std::string const move_type( "smoothmin" );

//car local
	bool gfrag( true );
	int frag_begin;

	retrieve_best_pose();
	choose_fragment_gunn(size,cutoff,frag_begin);
	refold(frag_begin,frag_begin+size-1);

	minimize_exclude_frag(frag_begin,size);
	minimize(type,move_type,score_fxn,frag_begin, frag_begin+size-1,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,frag_begin,size);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_small_wobble_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_small_wobble_min_trial(
	int const nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int const cycle_number,
	int const nwobble,
	int const max_wobble_gap,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "smwblmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length, begin, end;
	bool gfrag( true );

	small_wobble_move(1,move_list,length,nwobble,max_wobble_gap,begin,end);
	if ( length == 0 ) return; // small_moves failed

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( ! gfrag ) return;

	save_status_info(move_type,move_list(1),nmoves);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_small_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_small_min_trial(
	int nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "smallmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length,begin,end;
	bool gfrag( true );

	small_moves(nmoves,move_list,length,begin,end);
	if ( length == 0 ) return; // small_moves failed
	refold(begin,end);

	if ( get_loop_fold_with_dunbrack() ) close_loop_after_moves();

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) return;

	save_status_info( move_type, move_list(1), nmoves );
	monte_carlo( cycle_number );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_shear_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_shear_min_trial(
	int nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "shearmin" );

//car local
	FArray1D_int move_list( MAX_RES()() * 2 );
	int length,begin,end;
	bool gfrag( true );

	shear_moves(nmoves,move_list,length,begin,end);
	if ( length == 0 ) return; // shear_moves failed
	refold(begin,end);

	if ( get_loop_fold_with_dunbrack() ) close_loop_after_moves();

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,move_list(1),nmoves);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_wobble_min_trial
///
/// @brief
/// chuck move followed by wobble (powell minimization of phi/psi) to
/// minimize change in MSD of the position array
///
/// @detailed
///
/// @param  size - [in/out]? - fragment size
/// @param  cutoff_max - [in/out]? - huck cutoff, if <0 chuck sets cutoff
///        automatic (rec value: 28)
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  nwobble - [in/out]? - wobble this many residues
/// @param  max_wobble_gap - [in/out]? - position of first wobbled residue
///        relative to frag end, -1 is first or last frag res,
///        0 is adjacent to frag, positive values are outside frag
///        if max_wobble_gap>0, the actual wobble_gap will be
///        randomly set between 1 and max_wobble_gap
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_wobble_min_trial(
	int size,
	float cutoff_max,
	Scoring_Function score_fxn,
	int cycle_number,
	int nwobble,
	int max_wobble_gap,
	std::string const & type
)
{
	std::string const move_type( "wobblemin" );

//car local
	bool gfrag( true ); // if true, powell minimized sucessfully
	int frag_begin,frag_end;

	chuck_wobble_move(size,cutoff_max,nwobble,max_wobble_gap,frag_begin,frag_end,
	 gfrag);
	if ( !gfrag ) return;

	if ( get_loop_fold_with_dunbrack() ) close_loop_after_moves();

	if (get_minimize_exclude_frag_in_wobble())  minimize_exclude_frag(frag_begin,frag_end-frag_begin+1);
	minimize(type,move_type,score_fxn,frag_begin,frag_end,gfrag);

	save_status_info(move_type,frag_begin,frag_end-frag_begin+1);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_barcode_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian
///
/// @last_modified 1/31/05
/////////////////////////////////////////////////////////////////////////////////
void
main_barcode_min_trial(
	int nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "barcomin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length,begin,end;
	bool gfrag( true );

	barcode_moves( nmoves, move_list, length, 7, 0, begin, end );
	if ( length == 0 ) return; // barcode_moves failed
	refold(begin,end);

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) {
		return;
	}

	save_status_info(move_type,begin, begin-end+1);
	monte_carlo(cycle_number);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin main_native_barcode_move
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_native_barcode_move(
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "nat_code" );

//car local
	//FArray1D_int move_list( MAX_RES()() );
	std::vector< int > moved_residues;
	int begin,end;
	bool gfrag( true );

	make_nat_code_decoy( begin, end, moved_residues );

	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,0,1);
//	monte_carlo_reset();
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_barcode_min_cycle
///
/// @brief an interface for barcode_iterative_trial
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_barcode_min_cycle(
	int iteration,
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	for ( int i = 0; i < iteration; ++i ) {
		barcode_iterative_trial( score_fxn, cycle_number, type);
	}

//	monte_carlo( cycle_number );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_feature_small_min_trial
///
/// @brief specify a residue and do small move on that residue
///
/// @detailed
///
/// @param  which_res - [in/out]? - which residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_feature_small_min_trial(
	int num_in, // how many residue to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "feasmmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length,begin,end;
	bool gfrag( true );

	std::vector< int > feature_res = get_feature_residues( num_in );

	target_small_moves(feature_res,move_list,length,begin,end);
	if ( length == 0 ) return; // small_moves failed
	refold(begin,end);

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,move_list(1), length);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_feature_shear_min_trial
///
/// @brief specify a residue and do small move on that residue
///
/// @detailed
///
/// @param  which_res - [in/out]? - which residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_feature_shear_min_trial(
	int num_in, // how many residue to change
	Scoring_Function score_fxn,
	int cycle_number,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "feashmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length,begin,end;
	bool gfrag( true );

	std::vector< int > feature_res = get_feature_residues( num_in );

	target_shear_moves(feature_res, move_list,length,begin,end);
	if ( length == 0 ) return; // small_moves failed
	refold(begin,end);

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( !gfrag ) return;

	save_status_info(move_type,move_list(1), length);
	monte_carlo(cycle_number);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_feature_small_wobble_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  nmoves - [in/out]? - how many residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  nwobble - [in/out]? -
/// @param  max_wobble_gap - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_feature_small_wobble_min_trial(
	int const nmoves, // how many residues to change
	Scoring_Function score_fxn,
	int const cycle_number,
	int const nwobble,
	int const max_wobble_gap,
	std::string const & type
)
{
	using namespace param;

	std::string const move_type( "feawbmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	int length, begin, end;
	bool gfrag( true );

	target_small_wobble_move(1,move_list,length,nwobble,max_wobble_gap,begin,end);
	if ( length == 0 ) return; // small_wobble_move failed

	minimize_exclude_residues(move_list,length);
	minimize(type,move_type,score_fxn,begin,end,gfrag);
	if ( ! gfrag ) return;

	save_status_info(move_type,move_list(1),nmoves);
	monte_carlo(cycle_number);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin pose_small_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  nmoves
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_small_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int nmoves,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose );

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	FArray1D_int move_list( MAX_RES()() );
	int num_out, first, last;

	small_moves( nmoves, move_list, num_out, first, last );


	for ( int i=1; i<= num_out; ++i ) {
		int const pos( move_list(i) );
		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
	}

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "small-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_symm_rb_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  nmoves
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_symm_rb_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const float transmag,
	const float rotmag,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose );

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	symm_move( pose, transmag, rotmag );

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "symrb-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}


void
pose_rb_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	std::string const & min_type,
	int const jump_number,
	int const dir,
	float const trans_mag,
	float const rot_mag,
	const bool try_rotamers,
	const float energycut
)
{
	// choose the jump
	pose_ns::Jump jump( pose.get_jump( jump_number ) );
	jump.gaussian_move( dir, trans_mag, rot_mag );

	pose.set_jump( jump_number, jump );

	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE(
		pose_ns::RESENERGY, energycut, pose.total_residue(),
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1 );

	// calls score --> rottrials inside here
	pose.main_minimize( mc.weight_map(), min_type );

	score_set_try_rotamers( false );

	mc.boltzmann( pose, "rb-min" );

}

void
pose_small_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const int nmoves
)
{
	using namespace param;
	moves_set_current_pose( pose );

	FArray1D_int move_list( MAX_RES()() );
	int num_out, first, last;

	small_moves( nmoves, move_list, num_out, first, last );

	for ( int i=1; i<= num_out; ++i ) {
		int const pos( move_list(i) );
		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
	}

	mc.boltzmann( pose, "small" );

	moves_reset_current_pose();// un-hack

}

void
pose_shear_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const int nmoves
)
{
	using namespace param;
	moves_set_current_pose( pose );

	FArray1D_int move_list( MAX_RES()() );
	int num_out, first, last;

	shear_moves( nmoves, move_list, num_out, first, last );

	for ( int i=1; i<= num_out; ++i ) {
		int const pos( move_list(i) );
		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
	}

	mc.boltzmann( pose, "shear" );

	moves_reset_current_pose();// un-hack

}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_shear_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  nmoves
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_shear_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int nmoves,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose );

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	FArray1D_int move_list( MAX_RES()() );
	int num_out, first, last;

	shear_moves( nmoves, move_list, num_out, first, last );

	for ( int i=1; i<= num_out; ++i ) {
		int const pos( move_list(i) );
		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
	}

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres, mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "shear-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_ccd_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  loop_begin
/// @param  loop_end
/// @param  ccd_cycles
/// @param  try_rotamers
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_ccd_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	int const loop_begin,
	int const loop_end,
	int const cutpoint,
	int const ccd_cycles,
	const bool try_rotamers,
	const float energycut
)
{
	using namespace param;

	float const ccd_threshold( 0.01 );

	float const max_helix_delta( 3 );
	float const max_strand_delta( 5 );
	float const max_loop_delta( 15 );

	float fdev, bdev, torsion_delta, rama_delta;
	fast_ccd_loop_closure( pose, loop_begin, loop_end, cutpoint, ccd_cycles,
												 ccd_threshold, true, 2.0, max_helix_delta,
												 max_strand_delta, max_loop_delta,
												 fdev, bdev, torsion_delta, rama_delta);

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ),1);

	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );

	mc.boltzmann( pose );

}
////////////////////////////////////////////////////////////////////////////////
/// @begin main_wiggle_junction_min_trial
///
/// @brief specify a residue and do small move on that residue
///
/// @detailed
///
/// @param  which_res - [in/out]? - which residues to change
/// @param  score_fxn - [in/out]? -
/// @param  cycle_number - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_wiggle_junction_min_trial(
	int junction_res,
	Scoring_Function score_fxn,
	int cycle,
	std::string const & type
)
{
	using namespace param;
	using namespace misc;

	std::string const move_type( "jxnmin" );

//car local
	FArray1D_int move_list( MAX_RES()() );
	bool gfrag( true );
	int length = 1;

	target_junction_moves( junction_res );
	refold(junction_res, junction_res);

	minimize_include_one_residue( junction_res );
	minimize(type, move_type, score_fxn, junction_res, junction_res, gfrag);
	if ( !gfrag ) return;
	save_status_info(move_type, junction_res, length);
	monte_carlo(cycle);
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/// @begin pose_small_wobble_min_trial
///
/// @brief
///   Wrapper around small_wobble_move for poses.
///
/// @detailed
///   This is kind of a hack, as it uses the standard
///   misc routines for wobble, etc.
///   Doesn't really treat fold-tree chainbreaks properly.
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  nmoves
/// @param  nwobble
/// @param  max_wobble_gap
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors rhiju
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_small_wobble_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int nmoves,
	const int nwobble,
	const int max_wobble_gap,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose ); // triggers a copy_to_misc.

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	//wobble needs best to have the coordinates of the pose to figure
	// out where to return after putting in the fragment insertion.
	accept_best_coord(); //copies misc to best....

	FArray1D_int move_list( MAX_RES()() );
	int length, begin, end, total_begin, total_end;

	small_wobble_move(nmoves, move_list, length, nwobble, max_wobble_gap,
										begin, end, total_begin, total_end);

	if ( length != 0 && begin < end ) {

		for ( int pos=total_begin; pos<= total_end; ++pos ) {
			// Wobble will occasionally shift residues at jumps. One option
			// is to disallow the phi/psi of such residues to move inside wobble,
			// but its currently coded to accept only continuous chunks of phi/psi.
			// The option implemented here is to override allow_bb_move,
			// and allow those phi/psi's to change.
			bool temp_allow_bb_move = pose.get_allow_bb_move( pos );
			pose.set_allow_bb_move( pos, true );

			pose.set_phi      ( pos, misc::phi      ( pos ));
			pose.set_psi      ( pos, misc::psi      ( pos ));
			pose.set_omega    ( pos, misc::omega    ( pos ));
			pose.set_secstruct( pos, misc::secstruct( pos ));
			pose.set_allow_bb_move( pos, temp_allow_bb_move );
		}

		const int nres( pose.total_residue_for_scoring() );
		score_set_try_rotamers( try_rotamers );
		set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
																	mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
		pose.main_minimize( weight_map, min_type ); // calls score inside here
		score_set_try_rotamers( false );

	}

	mc.boltzmann( pose, "smwbl-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}

////////////////////////////////////////////////////////////////////////////////
/// @begin pose_wobble_min_trial
///
/// @brief
///   Wrapper around fragment insertion followed by wobble for poses.
///
/// @detailed
///   This is kind of a hack, as it uses the standard
///   misc routines for wobble, etc.
///   Doesn't really treat fold-tree chainbreaks properly.
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  size
/// @param  cutoff_max
/// @param  nwobble
/// @param  max_wobble_gap
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors rhiju
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_wobble_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int size,
	const float cutoff_max,
	const int nwobble,
	const int max_wobble_gap,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose ); // triggers a copy_to_misc.

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	//wobble needs best to have the coordinates of the pose to figure
	// out where to return after putting in the fragment insertion.
	accept_best_coord(); //copies misc to best....

	bool gfrag( true );
	int frag_begin, frag_end;
	chuck_wobble_move(size,cutoff_max,nwobble,max_wobble_gap,frag_begin,frag_end,
	 gfrag);
	if ( !gfrag ){
		moves_reset_current_pose();// un-hack
		return;
	}
	for ( int pos=frag_begin; pos<= frag_end; ++pos ) {
		// Wobble will occasionally shift residues at jumps. One option
		// is to disallow the phi/psi of such residues to move inside wobble,
		// but its currently coded to accept only continuous chunks of phi/psi.
		// The option implemented here is to override allow_bb_move,
		// and allow those phi/psi's to change.
		bool temp_allow_bb_move = pose.get_allow_bb_move( pos );
		pose.set_allow_bb_move( pos, true );

		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
		pose.set_allow_bb_move( pos, temp_allow_bb_move );
	}

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "wobble-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}


////////////////////////////////////////////////////////////////////////////////
/// @begin pose_crank_min_trial
///
/// @brief
///   Wrapper around crankshaft (fragment insertion followed by two separated wobbles) for poses.
///
/// @detailed
///   This is kind of a hack, as it uses the standard
///   misc routines for wobble, etc.
///   Doesn't really treat fold-tree chainbreaks properly.
///
/// @param  pose
/// @param  Monte_carlo object
/// @param  weight_map - scoring function
/// @param  min_type - minimization protocol
/// @param  size
/// @param  cutoff_max
/// @param  nwobble1
/// @param  nwobble2
/// @param  max_wobble_gap
/// @param  try rotamers (bool)
/// @param  energycut
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors rhiju
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
pose_crank_min_trial(
	pose_ns::Pose & pose,
	pose_ns::Monte_carlo & mc,
	const pose_ns::Score_weight_map & weight_map,
	std::string const & min_type,
	const int size,
	const float cutoff_max,
	const int nwobble1,
	const int nwobble2,
	const bool try_rotamers,
	const float energycut,
	std::string const tag_suffix // = ""
)
{
	using namespace param;
	moves_set_current_pose( pose ); // triggers a copy_to_misc.

	//Will trigger an ediff comparison to the starting score:
	pose.set_extra_score( "EDIFF_SCORE", pose.get_0D_score( pose_ns::SCORE ) );

	//wobble needs best to have the coordinates of the pose to figure
	// out where to return after putting in the fragment insertion.
	accept_best_coord(); //copies misc to best....

	bool gfrag( true );
	int crank_begin, crank_end;
	float bump_score, wobble_cost;

	crankshaft_move(size,cutoff_max,nwobble1,nwobble2,crank_begin,crank_end,
	 wobble_cost,bump_score,gfrag);
	if ( !gfrag ) {
		moves_reset_current_pose();// un-hack
		return;
	}

	for ( int pos=crank_begin; pos<= crank_end; ++pos ) {
		// Wobble will occasionally shift residues at jumps. One option
		// is to disallow the phi/psi of such residues to move inside wobble,
		// but its currently coded to accept only continuous chunks of phi/psi.
		// The option implemented here is to override allow_bb_move,
		// and allow those phi/psi's to change.
		bool temp_allow_bb_move = pose.get_allow_bb_move( pos );
		pose.set_allow_bb_move( pos, true );

		pose.set_phi      ( pos, misc::phi      ( pos ));
		pose.set_psi      ( pos, misc::psi      ( pos ));
		pose.set_omega    ( pos, misc::omega    ( pos ));
		pose.set_secstruct( pos, misc::secstruct( pos ));
		pose.set_allow_bb_move( pos, temp_allow_bb_move );
	}

	const int nres( pose.total_residue_for_scoring() );
	score_set_try_rotamers( try_rotamers );
	set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
		mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
	pose.main_minimize( weight_map, min_type ); // calls score inside here
	score_set_try_rotamers( false );
	mc.boltzmann( pose, "crank-min"+tag_suffix );

	moves_reset_current_pose();// un-hack

}
