// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 6379 $
//  $Date: 2005-06-20 22:05:46 -0400 (Mon, 20 Jun 2005) $
//  $Author: mentzer $


// Rosetta Headers
#include "util_interpolate.h"
#include "util_basic.h"
#include "wobble.h"


////////////////////////////////////////////////////////////////////////////////
/// @begin interpolate_bilinear_by_value
///
/// @brief get bilinear interpolate, given four points of a 2d periodic function
///
/// @detailed
///
///     Value and derivatives of bilinear interpolation between four
///     specified input values, which are the function values
///     corresponding to four points which bracket the interpolated point
///     in 2-dimensions.  (see diagram below) (see Num. Recipes v2, sec
///     3.6, "Interpolation in two or more dimensions")
///
///     Note that it is automatically assumed that the arguments are
///     periodic, that is f(a,b),a,b=periodic value; the *value* of the
///     function can be treated as periodic ( fixed to a periodicity of
///     360.) as well when the input parameter angle is set to true
///
///     The derivatives are of the bilinear interpolation only. This is
///     not the same value as a numerical derivative of function
///     values. The derivatives here are discontinuous every time you
///     cross over a bin boundary in either dimension
///
///
/// -  bilinear interpolation:
///
///    x0y1 +----------+ x1y1
///         |          |
///   1-yd  |    (x,y) |
///         - - +      |
///     yd  |   .      |
///         |   .      |
///    x0y0 +---|------+ x1y0
///          xd   1-xd
///
///
///
/// @param[in]   x0y0 - in - input bracketing function value (see diagram)
/// @param[in]   x1y0 - in - " "
/// @param[in]   x0y1 - in - " "
/// @param[in]   x1y1 - in - " "
/// @param[in]   xd - in - error value in the 1st dim between lower lst dim
///                   bracket value 'x0' and point to be interpolated, 'x'
/// @param[in]   yd - in - " 2nd dim "
/// @param[in]   binrange - in - range of bin in angles ( for both dimensions )
/// @param[in]   angles - in - if true, treat the the *values* of xy_func as
///                       as having a periodicity of 360. ( note that
///                       the bin ranges are already assumed periodic )
/// @param[out]   val - out - biliinear interpolated value
/// @param[out]   dval_dx - out - derivative of 'val' in the 1st ('x') dim
/// @param[out]   dval_dy - out - " 2nd dim "
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
interpolate_bilinear_by_value(
	float const x0y0,
	float const x1y0,
	float const x0y1,
	float const x1y1,
	float const xd,
	float const yd,
	float const binrange,
	bool const angles,
	float & val,
	float & dval_dx,
	float & dval_dy
)
{
	float const w1 = ( 1.0f - xd ) * ( 1.0f - yd );
	float const w2 = xd * ( 1.0f - yd );
	float const w3 = ( 1.0f - xd ) * yd;
	float const w4 = xd * yd;

	if ( angles ) {
		//// ctsa - hack method for weighted angle averaging, if we could burn
		////  cpu time, then the right thing to do would probably be to
		////  break func into x and y components, add up, interpolate,
		////  and find the direction

		float const w12 = w1 + w2;
		float a12;
		if ( w12 != 0.0 ) {
			a12 = ( w1 * x0y0 + w2 * ( subtract_degree_angles(x1y0,x0y0) + x0y0 ) ) / w12;
		} else {
			a12 = 0.0;
		}

		float const w34 = w3 + w4;
		float a34;
		if ( w34 != 0.0 ) {
			a34 = ( w3 * x0y1 + w4 * ( subtract_degree_angles(x1y1,x0y1) + x0y1 ) ) / w34;
		} else {
			a34 = 0.0;
		}

		val = w12 * a12 + w34 * ( subtract_degree_angles(a34,a12) + a12 );
		angle_in_range(val);

		dval_dx = ( ( 1.0f - yd ) * subtract_degree_angles(x1y0,x0y0) +
		 yd * subtract_degree_angles(x1y1,x0y1) ) / binrange;
		dval_dy = ( ( 1.0f - xd ) * subtract_degree_angles(x0y1,x0y0) +
		 xd * subtract_degree_angles(x1y1,x1y0) ) / binrange;
	} else {
		val = x0y0 * w1 + x1y0 * w2 + x0y1 * w3 + x1y1 * w4;

		dval_dx = ( ( 1.0f - yd ) * ( x1y0 - x0y0 ) + yd * ( x1y1 - x0y1 ) ) / binrange;
		dval_dy = ( ( 1.0f - xd ) * ( x0y1 - x0y0 ) + xd * ( x1y1 - x1y0 ) ) / binrange;
	}
}
