// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 11669 $
//  $Date: 2006-12-02 14:31:01 -0500 (Sat, 02 Dec 2006) $
//  $Author: snoeyink $

#ifndef INCLUDED_util_vector
#define INCLUDED_util_vector


// Rosetta Headers
#include "rotate.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>

// C++ Headers
#include <cassert>
#include <cmath>
#include <iosfwd>


// util_vector Function Declarations


//Objexx: Linear, zero-based indexing used for speed (dimension calls not needed)


////////////////////////////////////////////////////////////////////////////////
/// @begin vec_dist
///  simple euclidean dist functions
///  |v1-v2|
///
/// @brief
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
double
Dvec_dist(
	FArray1Da_double v1,
	FArray1Da_double v2
)
{
	double const d_1 = v1[ 0 ] - v2[ 0 ]; // v1(1) - v2(1);
	double const d_2 = v1[ 1 ] - v2[ 1 ]; // v1(2) - v2(2);
	double const d_3 = v1[ 2 ] - v2[ 2 ]; // v1(3) - v2(3);
	return std::sqrt( ( d_1 * d_1 ) + ( d_2 * d_2 ) + ( d_3 * d_3 ) );
}

inline
float
vec_dist(
	FArray1Da_float v1,
	FArray1Da_float v2
)
{
	float const d_1 = v1[ 0 ] - v2[ 0 ]; // v1(1) - v2(1);
	float const d_2 = v1[ 1 ] - v2[ 1 ]; // v1(2) - v2(2);
	float const d_3 = v1[ 2 ] - v2[ 2 ]; // v1(3) - v2(3);
	return std::sqrt( ( d_1 * d_1 ) + ( d_2 * d_2 ) + ( d_3 * d_3 ) );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin vec_dist2
///
/// @brief
///  |v1-v2|**2
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
double
Dvec_dist2(
	FArray1Da_double v1,
	FArray1Da_double v2
)
{
	double const d_1 = v1[ 0 ] - v2[ 0 ]; // v1(1) - v2(1);
	double const d_2 = v1[ 1 ] - v2[ 1 ]; // v1(2) - v2(2);
	double const d_3 = v1[ 2 ] - v2[ 2 ]; // v1(3) - v2(3);
	return ( d_1 * d_1 ) + ( d_2 * d_2 ) + ( d_3 * d_3 );
}

inline
float
vec_dist2(
	FArray1Da_float v1,
	FArray1Da_float v2
)
{
	float const d_1 = v1[ 0 ] - v2[ 0 ]; // v1(1) - v2(1);
	float const d_2 = v1[ 1 ] - v2[ 1 ]; // v1(2) - v2(2);
	float const d_3 = v1[ 2 ] - v2[ 2 ]; // v1(3) - v2(3);
	return ( d_1 * d_1 ) + ( d_2 * d_2 ) + ( d_3 * d_3 );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin divvec
///
/// @brief
///    v2 = v1/x
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  x - [in/out]? -
/// @param  v2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
void
divvec(
	FArray1DB< T > const & v1,
	T x,
	FArray1DB< T > & v2
)
{
	assert( ( v1.size() == 3 ) );
	assert( ( v2.size() == 3 ) );
	if ( x == T( 0.0 ) ) x = T( 0.0001 ); // Modifies local copy of x
	//cems changed division to inverse multiply
	T const inv_x = T( 1.0 ) / x;
	v2[ 0 ] = v1[ 0 ] * inv_x;
	v2[ 1 ] = v1[ 1 ] * inv_x;
	v2[ 2 ] = v1[ 2 ] * inv_x;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin dotprod
///
/// @brief
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
double
Ddotprod(
	FArray1Da_double v1,
	FArray1Da_double v2
)
{
	return ( v1[ 0 ] * v2[ 0 ] ) + ( v1[ 1 ] * v2[ 1 ] ) + ( v1[ 2 ] * v2[ 2 ] );
}

inline
float
dotprod(
	FArray1Da_float v1,
	FArray1Da_float v2
)
{
	return ( v1[ 0 ] * v2[ 0 ] ) + ( v1[ 1 ] * v2[ 1 ] ) + ( v1[ 2 ] * v2[ 2 ] );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin subvec
///
/// @brief v3 = v1 - v2
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
/// @param  v3 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
Dsubvec(
	FArray1Da_double v1,
	FArray1Da_double v2,
	FArray1Da_double v3
)
{
	v3[ 0 ] = v1[ 0 ] - v2[ 0 ];
	v3[ 1 ] = v1[ 1 ] - v2[ 1 ];
	v3[ 2 ] = v1[ 2 ] - v2[ 2 ];
}

inline
void
subvec(
	FArray1Da_float v1,
	FArray1Da_float v2,
	FArray1Da_float v3
)
{
	v3[ 0 ] = v1[ 0 ] - v2[ 0 ];
	v3[ 1 ] = v1[ 1 ] - v2[ 1 ];
	v3[ 2 ] = v1[ 2 ] - v2[ 2 ];
}


////////////////////////////////////////////////////////////////////////////////
/// @begin vector_sum
///
/// @brief v3 = v1 + v2
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
/// @param  v3 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
vector_sum(
	FArray1Da_double v1,
	FArray1DB_double const & v2,
	FArray1DB_double & v3
)
{
	assert( ( v2.size() == 3 ) );
	assert( ( v3.size() == 3 ) );
	v3[ 0 ] = v1[ 0 ] + v2[ 0 ];
	v3[ 1 ] = v1[ 1 ] + v2[ 1 ];
	v3[ 2 ] = v1[ 2 ] + v2[ 2 ];
}

inline
void
vector_sum(
	FArray1Da_float v1,
	FArray1DB_float const & v2,
	FArray1DB_float & v3
)
{
	assert( ( v2.size() == 3 ) );
	assert( ( v3.size() == 3 ) );
	v3[ 0 ] = v1[ 0 ] + v2[ 0 ];
	v3[ 1 ] = v1[ 1 ] + v2[ 1 ];
	v3[ 2 ] = v1[ 2 ] + v2[ 2 ];
}


////////////////////////////////////////////////////////////////////////////////
/// @begin unitvec
///
/// @brief v2 = v1/|v1|
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
Dunitvec(
	FArray1Da_double v1,
	FArray1Da_double v2
)
{
	double length = std::sqrt(
	 ( v1[ 0 ] * v1[ 0 ] ) +
	 ( v1[ 1 ] * v1[ 1 ] ) +
	 ( v1[ 2 ] * v1[ 2 ] ) );
	if ( length == 0.0 ) length = 0.0001;

	//cems changed division to inverse multiply
	double const inv_length = 1.0 / length;
	v2[ 0 ] = v1[ 0 ] * inv_length;
	v2[ 1 ] = v1[ 1 ] * inv_length;
	v2[ 2 ] = v1[ 2 ] * inv_length;
}

inline
void
unitvec(
	FArray1Da_float v1,
	FArray1Da_float v2
)
{
	float length = std::sqrt(
	 ( v1[ 0 ] * v1[ 0 ] ) +
	 ( v1[ 1 ] * v1[ 1 ] ) +
	 ( v1[ 2 ] * v1[ 2 ] ) );
	if ( length == 0.0f ) length = 0.0001f;

	//cems changed division to inverse multiply
	float const inv_length = 1.0f / length;
	v2[ 0 ] = v1[ 0 ] * inv_length;
	v2[ 1 ] = v1[ 1 ] * inv_length;
	v2[ 2 ] = v1[ 2 ] * inv_length;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin cros
///
/// @brief v3 = v1 X v2
///
/// @detailed
///
/// @param  v1 - [in/out]? -
/// @param  v2 - [in/out]? -
/// @param  v3 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
Dcros(
	FArray1Da_double v1,
	FArray1Da_double v2,
	FArray1Da_double v3
)
{
	v3[ 0 ] = v1[ 1 ] * v2[ 2 ] - v1[ 2 ] * v2[ 1 ];
	v3[ 1 ] = v1[ 2 ] * v2[ 0 ] - v1[ 0 ] * v2[ 2 ];
	v3[ 2 ] = v1[ 0 ] * v2[ 1 ] - v1[ 1 ] * v2[ 0 ];
}

inline
void
cros(
	FArray1Da_float v1,
	FArray1Da_float v2,
	FArray1Da_float v3
)
{
	v3[ 0 ] = v1[ 1 ] * v2[ 2 ] - v1[ 2 ] * v2[ 1 ];
	v3[ 1 ] = v1[ 2 ] * v2[ 0 ] - v1[ 0 ] * v2[ 2 ];
	v3[ 2 ] = v1[ 0 ] * v2[ 1 ] - v1[ 1 ] * v2[ 0 ];
}


void
align(
	FArray1Da_double a1,
	FArray1Da_double a2,
	FArray1Da_double a3,
	FArray1Da_double b1,
	FArray2Da_double mat,
	FArray1Da_double vec
);


void
lineup(
	FArray1Da_double a1,
	FArray1Da_double a2,
	FArray1Da_double b1,
	FArray1Da_double b2,
	FArray2Da_double mat,
	FArray1Da_double vec
);


inline
void
mover(
	FArray1Da_double c,
	FArray2DB_double const & mat,
	FArray1Da_double vec
)
{
	c.dimension( 3 );
	vec.dimension( 3 );

	static FArray1D_double t( 3, 0.0 );

	rotate(mat,c,t);
	c(1) = t(1) + vec(1);
	c(2) = t(2) + vec(2);
	c(3) = t(3) + vec(3);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin vector_norm
///
/// @brief
///
/// @detailed
///
/// @param  a - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
T
vector_norm( FArray1DB< T > const & a )
{
	assert( ( a.size() == 3 ) );
	return std::sqrt(
	 ( a[ 0 ] * a[ 0 ] ) +
	 ( a[ 1 ] * a[ 1 ] ) +
	 ( a[ 2 ] * a[ 2 ] ) );
}


void
vector_midpoint(
	FArray1Da_float a,
	FArray1Da_float b,
	FArray1Da_float c
);


void
vector_copy(
	FArray1Da_float a,
	FArray1Da_float b
);


////////////////////////////////////////////////////////////////////////////////
/// @begin vector_normalize
///
/// @brief
///
/// @detailed
///
/// @param  v - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
void
vector_normalize( FArray1DB< T > & v )
{
	assert( ( v.size() == 3 ) );
	T length = std::sqrt(
	 ( v[ 0 ] * v[ 0 ] ) +
	 ( v[ 1 ] * v[ 1 ] ) +
	 ( v[ 2 ] * v[ 2 ] ) );
	if ( length == T( 0.0 ) ) length = T( 0.0001 );
	T const inv_length = T( 1.0 ) / length;
	v[ 0 ] *= inv_length;
	v[ 1 ] *= inv_length;
	v[ 2 ] *= inv_length;
}


void
vector_project_normal(
	FArray1Da_float a,
	FArray1Da_float b
);


void
vector_write(
	std::ostream & unit,
	FArray1Da_float a
);


void
vector_write_line(
	std::ostream & unit,
	FArray1Da_float a
);


void
vector_name_write_line(
	std::ostream & unit,
	FArray1Da_float a,
	std::string const & name
);


void
Dvector_name_write_line(
	std::ostream & unit,
	FArray1Da_double a,
	std::string const & name
);


void
matrix_write(
	std::ostream & unit,
	FArray2Da_float M
);


void
matrix_name_write(
	std::ostream & unit,
	FArray2Da_float M,
	std::string const & name
);


void
Dmatrix_name_write(
	std::ostream & unit,
	FArray2Da_double M,
	std::string const & name
);


float
vec_angle(
	FArray1Da_float a,
	FArray1Da_float b,
	FArray1Da_float c
);

void
rotate_in_plane(
								FArray1Da_float a1,
								FArray1Da_float a2,
								FArray1Da_float b1,
								FArray1Da_float b2,
								FArray1Da_float b3,
								float & rot_ang // rotate angle
								);


////////////////////////////////////////////////////////////////////////////////
/// @begin transpose3
///
/// @brief
///
/// @detailed
///
/// @param  mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
void
transpose3( FArray2DB< T > & mat )
{
	//Objexx Make sure it is a 3x3 matrix
	assert( ( mat.size1() == 3 ) && ( mat.size2() == 3 ) );

	T temp = mat[1];
	mat[1] = mat[3];
	mat[3] = temp;

	temp = mat[2];
	mat[2] = mat[6];
	mat[6] = temp;

	temp = mat[5];
	mat[5] = mat[7];
	mat[7] = temp;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin mat_multiply3
///
/// @brief C = A * B    Using (col,row) subscripting
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  c - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
void
mat_multiply3(
	FArray2DB< T > const & a,
	FArray2DB< T > const & b,
	FArray2DB< T > & c
)
{
	//Objexx Make sure they are 3x3 matrices
	assert( ( a.l1() == 1 ) && ( a.u1() == 3 ) && ( a.l2() == 1 ) && ( a.u2() == 3 ) );
	assert( ( b.l1() == 1 ) && ( b.u1() == 3 ) && ( b.l2() == 1 ) && ( b.u2() == 3 ) );
	assert( ( c.l1() == 1 ) && ( c.u1() == 3 ) && ( c.l2() == 1 ) && ( c.u2() == 3 ) );

	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			c(i,j) = b(i,1) * a(1,j) + b(i,2) * a(2,j) + b(i,3) * a(3,j);
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin mat_multiply_transpose3
///
/// @brief C = A^T * B    Using (col,row) subscripting
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  c - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
template< typename T >
inline
void
mat_multiply_transpose3(
	FArray2DB< T > const & a,
	FArray2DB< T > const & b,
	FArray2DB< T > & c
)
{
	//Objexx Make sure they are 3x3 matrices
	assert( ( a.l1() == 1 ) && ( a.u1() == 3 ) && ( a.l2() == 1 ) && ( a.u2() == 3 ) );
	assert( ( b.l1() == 1 ) && ( b.u1() == 3 ) && ( b.l2() == 1 ) && ( b.u2() == 3 ) );
	assert( ( c.l1() == 1 ) && ( c.u1() == 3 ) && ( c.l2() == 1 ) && ( c.u2() == 3 ) );

	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			c(i,j) = b(i,1) * a(j,1) + b(i,2) * a(j,2) + b(i,3) * a(j,3);
		}
	}
}


#endif
