#!/usr/bin/perl
##
## Copyright 2002, University of Washington, the Baker Lab, and Dylan Chivian.
##   This document contains private and confidential information and its 
##   disclosure does not constitute publication.  All rights are reserved by 
##   University of Washington, the Baker Lab, and Dylan Chivian, except those 
##   specifically granted by license.
##
##  Initial Author: Dylan Chivian (dylan@lazy8.com)
##  $Revision: 1.1 $
##  $Date: 2002/01/01 00:00:00 $
##  $Author: dylan $
##
###############################################################################
# Vanita Sood 25 May 2004
# Make a loop library from the Vall
# Pass in the Vall and optionally the output loop library name, as well as the 
#   start and end residue numbers of the loop
# Will make every possible loop of that length from the Vall, excluding any 
#   loops that begin or end on a chain break (bad phi/psi)
###############################################################################
# conf
###############################################################################

$| = 1;                                              # disable stdout buffering

###############################################################################
# init
###############################################################################

# argv
my %opts = &getCommandLineOptions ();
my $file    = $opts{vall}; # Vall
my $s = $opts{start};
my $e = $opts{end};
my $outfile = $opts{lib}; # output loop library

my @filebuf = &fileBufArray ($file);

my $length  = $e - $s +1; 
$length = sprintf("%3s", $length);

my $start = sprintf("%5s", $s);
my $end = sprintf("%5s", $e);
my $bump = sprintf("%9.4f", -999);
my $deviation = sprintf("%9.4f", -999);
my $blank = sprintf("%78s"," ");
###############################################################################
# main
###############################################################################
my @outbuf = ();
for (my $i = 0; $i <= ($#filebuf - $length + 1); $i++){
    my $loop_line = '';
    my $parent = substr($filebuf[$i],0,4);
    my $dssp = '';
    my $angles = '';
    for (my $j = $i; $j <= $i+$length-1; $j++) {
	my $phi = substr($filebuf[$j],52,8);#print $phi."\n";
	my $psi = substr($filebuf[$j],61,8);#print $psi."\n";
	my $omega = substr($filebuf[$j],71,8);#print $omega."\n";
	#check for untrustworthy zero angles in the middle of the loop
	unless ($j == $i || $j == $i+$length-1) {
	    if ($phi == 0 || $psi == 0 ){
		$angles = "reset";
		last;
	    }
	}
        # add the loop to the library
	$dssp .= substr($filebuf[$j],8,1);
	$phi = sprintf ("%7.2f",$phi);
	$psi = sprintf ("%7.2f",$psi);
	$omega = sprintf ("%7.2f",$omega);
	$angles .= $phi." ".$psi." ".$omega."   ";
	}
#}
    if ($angles ne "reset" ) {
    $loop_line = $length.$start.$end." ".$dssp." ".$deviation." ".$bump.$blank.$parent." ".$angles;
	push (@outbuf, $loop_line);
    }
}
open OUTFILE, ">$outfile";
foreach (@outbuf){
    print OUTFILE $_."\n";
}
close OUTFILE;

exit 0;

###############################################################################
# subs
###############################################################################

# getCommandLineOptions()
#
#  rets: \%opts  pointer to hash of kv pairs of command line options
#
sub getCommandLineOptions {
    use Getopt::Long;
    my $usage = qq{usage: $0
\t -vall <Vall>
\t -start <start residue of extension>
\t -end <end residue of extension>
\t[-lib <outputfilename>] (default : "start_end.loops")
};

    # Get args
    my %opts = ();
    &GetOptions (\%opts, "vall=s", "lib=s", "start=i", "end=i");

    # Check for legal invocation
    if (! defined $opts{vall} || ! defined $opts{start} || ! defined $opts{end}
        ) {
        print STDERR "$usage\n";
        exit -1;
    }
    &checkExist ('f', $opts{vall});

    if (! defined $opts{lib}) {
	$opts{lib} = $opts{start}."_".$opts{end}.".loops";
    }

    return %opts;
}

###############################################################################
# util
###############################################################################

# readFiles
#
sub readFiles {
    my ($dir, $fullpath_flag) = @_;
    my $inode;
    my @inodes = ();
    my @files = ();
    
    opendir (DIR, $dir);
    @inodes = sort readdir (DIR);
    closedir (DIR);
    foreach $inode (@inodes) {
	next if (! -f "$dir/$inode");
	next if ($inode =~ /^\./);
	push (@files, ($fullpath_flag) ? "$dir/$inode" : "$inode");
    }
    return @files;
}

# createDir
#
sub createDir {
    my $dir = shift;
    if (! -d $dir && (system (qq{mkdir -p $dir}) != 0)) {
	print STDERR "$0: unable to mkdir -p $dir\n";
	exit -2;
    }
    return $dir;
}

# copyFile
#
sub copyFile {
    my ($src, $dst) = @_;
    if (system (qq{cp $src $dst}) != 0) {
	print STDERR "$0: unable to cp $src $dst\n";
	exit -2;
    }
    return $dst;
}

# zip
#
sub zip {
    my $file = shift;
    if ($file =~ /^\.Z$/ || $file =~ /\.gz$/) {
	print STDERR "$0: ABORT: already a zipped file $file\n";
	exit -2;
    }
    if (system (qq{gzip -9 $file}) != 0) {
	print STDERR "$0: unable to gzip -9 $file\n";
	exit -2;
    }
    $file .= ".gz";
    return $file;
}

# unzip
#
sub unzip {
    my $file = shift;
    if ($file !~ /^\.Z$/ && $file !~ /\.gz$/) {
	print STDERR "$0: ABORT: not a zipped file $file\n";
	exit -2;
    }
    if (system (qq{gzip -d $file}) != 0) {
	print STDERR "$0: unable to gzip -d $file\n";
	exit -2;
    }
    $file =~ s/\.Z$|\.gz$//;
    return $file;
}

# remove
#
sub remove {
    my $inode = shift;
    if (system (qq{rm -rf $inode}) != 0) {
	print STDERR "$0: unable to rm -rf $inode\n";
	exit -2;
    }
    return $inode;
}
     
# runCmd
#
sub runCmd {
    my ($cmd, $nodie) = @_;
    my $ret;
    my $date = `date +'%Y-%m-%d_%T'`;  chomp $date;
    print "[$date]:$0:RUNNING: $cmd\n" if ($debug);
    $ret = system ($cmd);
    #$ret = ($?>>8)-256;
    if ($ret != 0) {
	$date = `date +'%Y-%m-%d_%T'`;  chomp $date;
	print STDERR ("[$date]:$0: FAILURE (exit: $ret): $cmd\n");
	if ($nodie) {
	    return $ret;
	} else {
	    exit $ret;
	}
    }
    return 0;
}

# logMsg()
#
sub logMsg {
    my ($msg, $logfile) = @_;
    my $date = `date +'%Y-%m-%d_%T'`;  chomp $date;

    if ($logfile) {
        open (LOGFILE, ">".$logfile);
        select (LOGFILE);
    }
    else {
	select (STDERR);
    }
    print "[$date]:$0: $msg\n";
    if ($logfile) {
        close (LOGFILE);
    }
    select (STDOUT);

    return 'true';
}

# checkExist()
#
sub checkExist {
    my ($type, $path) = @_;
    if ($type eq 'd') {
	if (! -d $path) { 
            print STDERR "$0: dirnotfound: $path\n";
            exit -3;
	}
    }
    elsif ($type eq 'f') {
	if (! -f $path) {
            print STDERR "$0: filenotfound: $path\n";
            exit -3;
	}
	elsif (! -s $path) {
            print STDERR "$0: emptyfile: $path\n";
            exit -3;
	}
    }
}

# abort()
#
sub abort {
    my $msg = shift;
    my $date = `date +'%Y-%m-%d_%T'`;  chomp $date;
    print STDERR "[$date]:$0:ABORT: $msg\n";
    exit -2;
}

# writeBufToFile()
#
sub writeBufToFile {
    my ($file, $bufptr) = @_;
    if (! open (FILE, '>'.$file)) {
	&abort ("$0: unable to open file $file for writing");
    }
    print FILE join ("\n", @{$bufptr}), "\n";
    close (FILE);
    return;
}

# fileBufString()
#
sub fileBufString {
    my $file = shift;
    my $oldsep = $/;
    undef $/;
    if ($file =~ /\.gz$|\.Z$/) {
	if (! open (FILE, "gzip -dc $file |")) {
	    &abort ("$0: unable to open file $file for gzip -dc");
	}
    }
    elsif (! open (FILE, $file)) {
	&abort ("$0: unable to open file $file for reading");
    }
    my $buf = <FILE>;
    close (FILE);
    $/ = $oldsep;
    return $buf;
}

# fileBufArray()
#
sub fileBufArray {
    my $file = shift;
    my $oldsep = $/;
    undef $/;
    if ($file =~ /\.gz$|\.Z$/) {
	if (! open (FILE, "gzip -dc $file |")) {
	    &abort ("$0: unable to open file $file for gzip -dc");
	}
    }
    elsif (! open (FILE, $file)) {
	&abort ("$0: unable to open file $file for reading");
    }
    my $buf = <FILE>;
    close (FILE);
    $/ = $oldsep;
    @buf = split (/$oldsep/, $buf);
    pop (@buf)  if ($buf[$#buf] eq '');
    return @buf;
}

# bigFileBufArray()
#
sub bigFileBufArray {
    my $file = shift;
    my $buf = +[];
    if ($file =~ /\.gz$|\.Z$/) {
        if (! open (FILE, "gzip -dc $file |")) {
            &abort ("$0: unable to open file $file for gzip -dc");
        }
    }
    elsif (! open (FILE, $file)) {
        &abort ("$0: unable to open file $file for reading");
    }
    while (<FILE>) {
        chomp;
        push (@$buf, $_);
    }
    close (FILE);
    return $buf;
}     

###############################################################################
# end
1;                                                     # in case it's a package
###############################################################################
