// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/moves/SidechainMover.cc
/// @brief implementation of SidechainMover class and functions
/// @author Colin A. Smith (colin.smith@ucsf.edu)


#include <protocols/moves/SidechainMover.hh>

// Core Headers
#include <core/chemical/ResidueType.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/dunbrack/DunbrackRotamer.hh>
#include <core/scoring/dunbrack/SingleResidueDunbrackLibrary.hh>
#include <core/scoring/dunbrack/RotamerLibraryScratchSpace.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>

// Numeric Headers
#include <numeric/angle.functions.hh>
// AUTO-REMOVED #include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/random/random.hh>

// C++ Headers
#include <sstream>

using namespace core;
using namespace core::pose;

static numeric::random::RandomGenerator RG(38627225);
static util::Tracer TR("protocols.moves.SidechainMover");

namespace protocols {
namespace moves {

SidechainMover::SidechainMover():
	rotamer_library_(core::scoring::ScoringManager::get_instance()->get_RotamerLibrary()),
	prob_uniform_(0.1),
	prob_withinrot_(0.0),
	last_proposal_density_ratio_(1),
	scratch_( new core::scoring::dunbrack::RotamerLibraryScratchSpace )
{}

SidechainMover::SidechainMover(
	core::scoring::dunbrack::RotamerLibrary const & rotamer_library
):
	rotamer_library_(rotamer_library),
	prob_uniform_(0.1),
	prob_withinrot_(0.0),
	last_proposal_density_ratio_(1),
	scratch_( new core::scoring::dunbrack::RotamerLibraryScratchSpace )
{}

SidechainMover::~SidechainMover() {}

/// @detailed
/// Check to make sure that a packer task exists and matches the numer of residues in
/// the given pose. If that isn't the case, create a new one with the task factory.
/// Exits with an error if no task factory exists.
void
SidechainMover::init_task(
	core::pose::Pose & pose
)
{
	// check to see if a valid task already exists
	if (!task_ || task_->total_residue() != pose.total_residue()) {
		// if not, create one using the task factory
		if (task_factory_) {
			set_task(task_factory_->create_task_and_apply_taskoperations( pose ));
		} else {
			utility_exit_with_message("Cannot create task because no task factory is set");
		}
	}
}

/// @detailed
void
SidechainMover::apply(
	Pose & pose
)
{
	using numeric::conversions::degrees;
	using numeric::conversions::radians;

	// reset the last move tracking information
	last_mutation_ = false;
	last_uniform_ = true;
	last_withinrot_ = false;

	init_task(pose);

	// select a residue to change
	runtime_assert(packed_residues_.size() > 0);
	Size const resnum(packed_residues_[RG.random_range(1, packed_residues_.size())]);

	core::chemical::ResidueType const & previous_residue_type( pose.residue_type(resnum) );
	utility::vector1<core::Real> previous_chi_angles( pose.residue(resnum).chi() );

	core::pack::task::ResidueLevelTask const & residue_task(task_->residue_task(resnum));
	core::pack::task::ResidueLevelTask::ResidueTypeCAPList const & residue_types(residue_task.allowed_residue_types());

	// select a residue type
	core::chemical::ResidueTypeCAP residue_type;
	do {
		Size const restypenum(RG.random_range(1, residue_types.size()));
		core::pack::task::ResidueLevelTask::ResidueTypeCAPListConstIter iter(residue_types.begin());
		for (Size i = 1; i < restypenum; ++i) ++iter;
		residue_type = *iter;
	}
	while (residue_type->aa() == core::chemical::aa_pro); // temporary hack to exlcude sampling of proline sidechains

	last_nchi_ = residue_type->nchi();

	///code note: stl uses lazy resizes; if a vector doesn't have to allocate more space, it doesn't, it just
	/// updates its end-of-memory pointer.  By always performing the resize, the vector size() will reflect
	/// the actual number of chi.
	///if (last_chi_angles_.size() < last_nchi_) last_chi_angles_.resize(last_nchi_);
	last_chi_angles_.resize( last_nchi_ );

	TR.Debug << pose.residue_type(resnum).name() << " " << resnum << " -> " << residue_type->name() << std::endl;

	if (residue_type->aa() != pose.residue_type(resnum).aa()) last_mutation_ = true;

	core::scoring::dunbrack::SingleResidueRotamerLibraryCAP residue_rotamer_library(
		rotamer_library_.get_rsd_library(*residue_type)
	);
	core::scoring::dunbrack::SingleResidueDunbrackLibraryCAP residue_dunbrack_library(
		dynamic_cast< core::scoring::dunbrack::SingleResidueDunbrackLibrary const * >(residue_rotamer_library.get())
	);

	// 	TR << "original chi angles:";
	// 	utility::vector1< Real > const & ochis(pose.residue(resnum).chi());
	// 	for (core::Size i(1); i <= ochis.size(); ++i)
	// 		TR << " " << ochis[i];
	// 	TR << std::endl;

	/// last_chi_angles_ holds degrees for a short period;
	/// Apply uniform sampling to rotamers with a small probability, or with probability 1 if
	/// the residue's library is not a dunbrack library. (Is this what we would want for pSer?).
	/// At the moment, the ligand rotamer library does not respond to the assign_random_rotamer_with_bias
	/// method, so this is an ok check.
	core::Real move_type_prob( RG.uniform() );
	if (move_type_prob > prob_uniform_ && residue_dunbrack_library) {

		core::Real const phi( residue_dunbrack_library->get_phi_from_rsd( pose.residue(resnum) ) );
		core::Real const psi( residue_dunbrack_library->get_psi_from_rsd( pose.residue(resnum) ) );

		utility::vector1< core::scoring::dunbrack::DunbrackRotamerSampleData > rotamer_sample_data(
			residue_dunbrack_library->get_all_rotamer_samples(phi, psi)
		);

		Size rotnum = 0;

		if (move_type_prob > prob_uniform_ + prob_withinrot_ || last_mutation_) {

			// choose a random rotamer according to its probability
			core::Real random_prob = RG.uniform();

			/// go through rotamers in decreasing order by probability and stop when the
			while ( random_prob > 0 ) {
				random_prob -= rotamer_sample_data[++rotnum].probability();
				// loop condition might end up satisfied even if we've walked through all possible rotamers
				// if the chosen random number was nearly 1
				// (and interpolation introduced a tiny bit of numerical noise).
				if ( rotnum == rotamer_sample_data.size() ) break;
			}

		} else {

			// find the rotamer that has the highest probability of proposing the previous chi angles
			core::Real max_rot_prob = 0;
			for (core::Size ii = 1; ii <= rotamer_sample_data.size(); ++ii) {
				core::Real candidate_rot_prob( rotamer_sample_data[ii].chi_probability(previous_chi_angles) );
				if (candidate_rot_prob > max_rot_prob) {
					rotnum = ii;
					max_rot_prob = candidate_rot_prob;
				}
			}
			last_withinrot_ = true;
		}

		rotamer_sample_data[rotnum].assign_random_chi(last_chi_angles_, RG);

		//residue_rotamer_library->assign_random_rotamer_with_bias(
		//	pose.residue( resnum ), *scratch_, RG,
		//	last_chi_angles_, true );

		last_uniform_ = false;
	} else {
		for ( Size ii = 1; ii <= last_nchi_; ++ii ) {
			last_chi_angles_[ ii ] = RG.uniform() * 360.0 - 180.0;
		}
	}

	core::Real proposal_density_reverse(1);
	core::Real proposal_density_forward(1);

	if (preserve_detailed_balance_) {
		proposal_density_reverse = proposal_density(pose.residue(resnum), resnum, *residue_type, last_chi_angles_);
	}

	// swap in a new residue if necessary
	if (residue_type->name() != pose.residue_type(resnum).name()) {

		core::conformation::ResidueOP new_residue(
			core::conformation::ResidueFactory::create_residue(*residue_type, pose.residue(resnum), pose.conformation(),
			                                                   residue_task.preserve_c_beta())
		);
		pose.replace_residue(resnum, *new_residue, false);
	}

	/// last_chi_angles_ holds radians after this code executes;
	for ( Size ii = 1; ii <= pose.residue_type( resnum ).nchi(); ++ii ) {
		last_chi_angles_[ ii ] = numeric::principal_angle_degrees(last_chi_angles_[ ii ]);
		pose.set_chi( ii, resnum, last_chi_angles_[ ii ] );
		last_chi_angles_[ ii ] = radians( last_chi_angles_[ ii ] );
	}

	if (preserve_detailed_balance_) {
		proposal_density_forward = proposal_density(pose.residue(resnum), resnum, previous_residue_type, previous_chi_angles);
		last_proposal_density_ratio_ = proposal_density_reverse / proposal_density_forward;
	}

	//Size chinum(1);

	// use Dunbrack statistics to bias chi angle sampling
	//if (RG.uniform() > prob_uniform_ && residue_dunbrack_library) {
	//
	//	// get indices of the phi and psi bins, plus extra information we don't need
	//	Size phibin, psibin, phibin_next, psibin_next;
	//	Real phi_err, psi_err;
	//	residue_dunbrack_library->get_phi_interpolate_bin(pose.phi(resnum), pose.psi(resnum), phibin, psibin,
	//		phibin_next, psibin_next, phi_err, psi_err);

	//	// choose a random rotamer according to its probability
	//	Real accprob = RG.uniform();
	//	for (Size rotnum = 1; rotnum <= residue_dunbrack_library->nrots_per_bin(); rotnum++) {
	//
	//		core::scoring::dunbrack::DunbrackRotamer< core::scoring::dunbrack::FOUR > rotamer(
	//			residue_dunbrack_library->retrieve_rotamer(phibin, psibin, rotnum)
	//		);

	//		// keep looking if we haven't decremented the accumulated probability below zero
	//		accprob -= rotamer.rotamer_probability();
	//		if (accprob > 0) continue;

	//		// for chi angles which have statistics, sample according to rotamer SD and mean
	//		TR.Debug << "Using rotamer " << rotnum << ": ";
	//		for ( ; chinum <= residue_dunbrack_library->nchi_aa(); ++chinum) {
	//			TR.Debug << " " << rotamer.chi_mean(chinum) << "(" << rotamer.chi_sd(chinum) << ")";
	//			last_chi_angles_[chinum] = radians(RG.gaussian()*rotamer.chi_sd(chinum) + rotamer.chi_mean(chinum));
	//			pose.set_chi(chinum, resnum, degrees(last_chi_angles_[chinum]));
	//		}
	//		TR.Debug << std::endl;

	//		last_uniform_ = false;
	//		break;
	//	}
	//}

	// any/all remaning chi angles are chosen uniformly
	//for ( ; chinum <= residue_type->nchi(); ++chinum) {
	//	last_chi_angles_[chinum] = RG.uniform()*numeric::constants::d::pi_2 - numeric::constants::d::pi;
	//	pose.set_chi(chinum, resnum, degrees(last_chi_angles_[chinum]));
	//}

	TR.Debug << "Set residue chi angles to:";
	for (Size i = 1; i <= residue_type->nchi(); ++i) TR.Debug << " " << degrees(last_chi_angles_[i]);
	TR.Debug << std::endl;

	// 	TR << "new chi angles:";
	// 	utility::vector1< Real > const & nchis(pose.residue(resnum).chi());
	// 	for (core::Size i = 1; i <= nchis.size(); ++i)
	// 		TR << " " << nchis[i];
	// 	TR << std::endl;

	update_type();
}

core::Real
SidechainMover::proposal_density(
	core::conformation::Residue const & proposed_residue,
	core::Size const proposed_resnum,
	core::chemical::ResidueType const & initial_residue_type,
	utility::vector1<core::Real> const & initial_chi_angles
) const
{
	utility::vector1<core::Real> const & proposed_chi_angles( proposed_residue.chi() );

	core::Real density(0);

	core::scoring::dunbrack::SingleResidueRotamerLibraryCAP residue_rotamer_library(
		rotamer_library_.get_rsd_library(proposed_residue.type())
	);
	core::scoring::dunbrack::SingleResidueDunbrackLibraryCAP residue_dunbrack_library(
		dynamic_cast< core::scoring::dunbrack::SingleResidueDunbrackLibrary const * >(residue_rotamer_library.get())
	);

	if (residue_dunbrack_library) {

		core::Real const phi( residue_dunbrack_library->get_phi_from_rsd( proposed_residue ) );
		core::Real const psi( residue_dunbrack_library->get_psi_from_rsd( proposed_residue ) );

		utility::vector1< core::scoring::dunbrack::DunbrackRotamerSampleData > rotamer_sample_data(
			residue_dunbrack_library->get_all_rotamer_samples(phi, psi)
		);

		core::Real rot_density(0);

		core::Real max_proposed_rot_prob(0);
		core::Real max_initial_rot_prob(0);

		bool calc_withinrot( prob_withinrot_ && proposed_residue.type().aa() == initial_residue_type.aa() );

		for (core::Size ii = 1; ii <= rotamer_sample_data.size(); ++ii) {

			core::Real const proposed_rot_prob( rotamer_sample_data[ii].chi_probability(proposed_chi_angles) );

			rot_density += rotamer_sample_data[ii].probability() * proposed_rot_prob;

			if (calc_withinrot) {

				core::Real const initial_rot_prob( rotamer_sample_data[ii].chi_probability(initial_chi_angles) );

				if (initial_rot_prob > max_initial_rot_prob) {
					max_initial_rot_prob = initial_rot_prob;
					max_proposed_rot_prob = proposed_rot_prob;
				}
			}
		}

		rot_density *= 1 - prob_uniform_ - prob_withinrot_;

		density += rot_density;

		if (calc_withinrot) {

			core::Real const withinrot_density( prob_withinrot_ * max_proposed_rot_prob );

			density += withinrot_density;
		}
	}

	if (prob_uniform_ || !residue_dunbrack_library) {

		core::Real uniform_density(1);

		for (core::Size ii = 1; ii <= proposed_residue.nchi(); ++ii) {
			uniform_density *= 1.0/360.0;
		}

		if (residue_dunbrack_library) uniform_density *= prob_uniform_;

		density += uniform_density;
	}

	// Probability of selecting the given ResidueType
	// This isn't needed now, but should be changed if the way the ResidueTypes
	// are sampled changes.
	core::pack::task::ResidueLevelTask const & residue_task(task_->residue_task(proposed_resnum));
	density *= 1/residue_task.allowed_residue_types().size();

	return density;
}

/// @detailed
void
SidechainMover::test_move(
	Pose &
)
{

}

/// @detailed
/// all sidechains that might be changed are replaced with ideal coordinates that have
/// the original chi angles
void
SidechainMover::idealize_sidechains(
	core::pose::Pose & pose
)
{
	utility::vector1<Real> chi;

	init_task(pose);

	for (Size i = 1; i <= packed_residues_.size(); ++i) {

		Size const resnum(packed_residues_[i]);

		// get the residue type and residue level task
		chemical::ResidueType const & residue_type(pose.residue_type(resnum));
		pack::task::ResidueLevelTask const & residue_task(task_->residue_task(resnum));

		// disable proline idealization for now
		if (residue_type.aa() == chemical::aa_pro) continue;

		// save original chi angles
		if (residue_type.nchi() > chi.size()) chi.resize(residue_type.nchi());
		for (Size chinum = 1; chinum <= residue_type.nchi(); ++chinum) {
			chi[chinum] = pose.chi(chinum, resnum);
		}

		// create a new residue and replace the old one with it
		conformation::ResidueOP new_residue(
			conformation::ResidueFactory::create_residue(residue_type, pose.residue(resnum), pose.conformation(),
			                                             residue_task.preserve_c_beta())
		);
		pose.replace_residue(resnum, *new_residue, false);

		// put original chi angles back
		for (Size chinum = 1; chinum <= residue_type.nchi(); ++chinum) {
			pose.set_chi(chinum, resnum, chi[chinum]);
		}
	}
}

core::scoring::dunbrack::RotamerLibrary const &
SidechainMover::rotamer_library() const
{
	return rotamer_library_;
}

core::pack::task::TaskFactoryCOP
SidechainMover::task_factory() const
{
	return task_factory_;
}

void
SidechainMover::set_task_factory(
	core::pack::task::TaskFactoryCOP task_factory
)
{
	task_factory_ = task_factory;
}

core::pack::task::PackerTaskCOP
SidechainMover::task() const
{
	return task_;
}

void
SidechainMover::set_task(
	core::pack::task::PackerTaskCOP task
)
{
	using core::pack::task::ResidueLevelTask;

	task_ = task;

	// update the list of residues being packed
	packed_residues_.clear();
	for (Size i = 1; i <= task_->total_residue(); ++i) {

		// loop over all possible residue types and check if any have chi angles
		core::pack::task::ResidueLevelTask const & residue_task(task->residue_task(i));
		for (ResidueLevelTask::ResidueTypeCAPListConstIter iter(residue_task.allowed_residue_types_begin());
		     iter != residue_task.allowed_residue_types_end(); ++iter) {

			// temporarily exclude proline residues from side chain sampling
			if ((*iter)->nchi() > 0 && (*iter)->aa() != core::chemical::aa_pro) {
				packed_residues_.push_back(i);
				break;
			}
		}
	}
}

core::Real
SidechainMover::prob_uniform() const
{
	return prob_uniform_;
}

void
SidechainMover::set_prob_uniform(
	core::Real prob_uniform
)
{
	prob_uniform_ = prob_uniform;
}

core::Real
SidechainMover::prob_withinrot() const
{
	return prob_withinrot_;
}

void
SidechainMover::set_prob_withinrot(
	core::Real prob_withinrot
)
{
	prob_withinrot_ = prob_withinrot;
}

bool
SidechainMover::preserve_detailed_balance() const
{
	return preserve_detailed_balance_;
}

void
SidechainMover::set_preserve_detailed_balance(
	bool preserve_detailed_balance
)
{
	preserve_detailed_balance_ = preserve_detailed_balance;
}

utility::vector1<core::Size> const &
SidechainMover::packed_residues() const
{
	return packed_residues_;
}

core::Size
SidechainMover::last_nchi() const
{
	return last_nchi_;
}

bool
SidechainMover::last_mutation() const
{
	return last_mutation_;
}

bool
SidechainMover::last_uniform() const
{
	return last_uniform_;
}

bool
SidechainMover::last_withinrot() const
{
	return last_withinrot_;
}

core::Real
SidechainMover::last_proposal_density_ratio() const
{
	return last_proposal_density_ratio_;
}

/// @detailed
/// All move types are prefixed with "sc". Sections are divided by underscores.
/// The next section indicates whether a mutation was made ("mut") or not ("chi").
/// The last section indicates wehter chi sampling was uniform ("unif"), used
/// Dunbrack rotamer statistics ("rot"), or whether no chi angles existed in the
/// placed residue ("none").
void
SidechainMover::update_type()
{
	std::stringstream mt;

	mt << "sc_" << (last_mutation_ ? "mut" : "chi") << "_" << (last_nchi_ ? (last_uniform_ ? "unif" : (last_withinrot_ ? "withinrot" : "rot")) : "none");

	std::string const new_type(mt.str());
	type(new_type);
}

} // moves
} // protocols
