// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_DimensionExpressionMin_HH
#define INCLUDED_ObjexxFCL_DimensionExpressionMin_HH


// DimensionExpressionMin: DimensionExpression Binary Min Function
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpression.hh>
#include <ObjexxFCL/DimensionExpressionCon.hh>

// C++ Standard Library Headers
#include <algorithm>


namespace ObjexxFCL {


/// @brief DimensionExpressionMin: DimensionExpression Binary Min Function
class DimensionExpressionMin :
	public DimensionExpression
{


private: // Types


	typedef  DimensionExpression  Super;


public: // Creation


	/// @brief Copy Constructor
	inline
	DimensionExpressionMin( DimensionExpressionMin const & exp ) :
		Super(),
		exp1_p_( exp.exp1_p_ ? exp.exp1_p_->clone() : static_cast< DimensionExpression * >( 0 ) ),
		exp2_p_( exp.exp2_p_ ? exp.exp2_p_->clone() : static_cast< DimensionExpression * >( 0 ) )
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
	}


	/// @brief Expression Constructor
	inline
	DimensionExpressionMin( DimensionExpression const & exp1, DimensionExpression const & exp2 ) :
		exp1_p_( exp1.clone() ),
		exp2_p_( exp2.clone() )
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
	}


	/// @brief Expression Pointer Constructor (Ownership Transfer)
	inline
	DimensionExpressionMin( DimensionExpression * exp1_p_a, DimensionExpression * exp2_p_a ) :
		exp1_p_( exp1_p_a ),
		exp2_p_( exp2_p_a )
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
	}


	/// @brief Clone
	inline
	DimensionExpression *
	clone() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		if ( constant() ) {
			if ( integer() ) {
				return new DimensionExpressionCon( std::min( exp1_p_->ivalue(), exp2_p_->ivalue() ) );
			} else {
				return new DimensionExpressionCon( std::min( exp1_p_->value(), exp2_p_->value() ) );
			}
		} else {
			return new DimensionExpressionMin( exp1_p_->clone(), exp2_p_->clone() );
		}
	}


	/// @brief Clone with Dimension Substitution
	inline
	DimensionExpression *
	clone( Dimension const & dim ) const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		if ( constant() ) {
			if ( integer() ) {
				return new DimensionExpressionCon( std::min( exp1_p_->ivalue(), exp2_p_->ivalue() ) );
			} else {
				return new DimensionExpressionCon( std::min( exp1_p_->value(), exp2_p_->value() ) );
			}
		} else {
			return new DimensionExpressionMin( exp1_p_->clone( dim ), exp2_p_->clone( dim ) );
		}
	}


	/// @brief Destructor
	inline
	virtual
	~DimensionExpressionMin()
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		delete exp1_p_;
		delete exp2_p_;
	}


public: // Inspector


	/// @brief Initialized?
	inline
	bool
	initialized() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( ( exp1_p_->initialized() ) && ( exp2_p_->initialized() ) );
	}


	/// @brief Integer?
	inline
	bool
	integer() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( ( exp1_p_->integer() ) && ( exp2_p_->integer() ) );
	}


	/// @brief Constant?
	inline
	bool
	constant() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( ( exp1_p_->constant() ) && ( exp2_p_->constant() ) );
	}


	/// @brief Reference?
	inline
	bool
	reference() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( ( exp1_p_->reference() ) || ( exp2_p_->reference() ) );
	}


	/// @brief Reducible?
	inline
	bool
	reducible() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( ( constant() ) || ( exp1_p_->reducible() ) || ( exp2_p_->reducible() ) );
	}


	/// @brief Value
	inline
	double
	operator ()() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( std::min( exp1_p_->operator ()(), exp2_p_->operator ()() ) );
	}


	/// @brief Value
	inline
	double
	value() const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		return ( std::min( exp1_p_->value(), exp2_p_->value() ) );
	}


	/// @brief Insert an Observer
	inline
	void
	insert_observer( Observer & observer ) const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		exp1_p_->insert_observer( observer );
		exp2_p_->insert_observer( observer );
	}


	/// @brief Remove an Observer
	inline
	void
	remove_observer( Observer & observer ) const
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		exp1_p_->remove_observer( observer );
		exp2_p_->remove_observer( observer );
	}


public: // Modifier


	/// @brief Update for Destruction of a Subject
	inline
	void
	destructed( Subject const & subject )
	{
		assert( exp1_p_ );
		assert( exp2_p_ );
		exp1_p_->destructed( subject );
		exp2_p_->destructed( subject );
	}


private: // Data


	/// @brief Pointer to expression 1
	DimensionExpression * exp1_p_;

	/// @brief Pointer to expression 2
	DimensionExpression * exp2_p_;


}; // DimensionExpressionMin


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_DimensionExpressionMin_HH
