// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_FArray1DB_HH
#define INCLUDED_ObjexxFCL_FArray1DB_HH


// FArray1DB: Fortran-Compatible 1D Array Abstract Base Class
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArray1DB.fwd.hh>
#include <ObjexxFCL/FArrayB.hh>

// C++ Standard Library Headers
#include <cmath>


namespace ObjexxFCL {


// Forward Declarations
template< typename > class FArray1D;
template< typename > class FArray1Dp;
template< typename > class FArray1Da;
template< typename > class KeyFArray1D;


/// @brief FArray1DB: Fortran-Compatible 1D Array Abstract Base Class
template< typename T >
class FArray1DB :
	public FArrayB< T >
{


private: // Types


	typedef  FArrayB< T >  Super;
	typedef  FArray1D< T >  real_FArray;
	typedef  FArray1Dp< T >  proxy_FArray;
	typedef  FArray1Da< T >  arg_FArray;


private: // Friend


	template< typename > friend class FArray1DB;
	template< typename > friend class FArray1D;
	template< typename > friend class FArray1Dp;
	template< typename > friend class FArray1Da;
	template< typename > friend class KeyFArray1D;


protected: // Types


	typedef  internal::InitializerSentinel  InitializerSentinel;
	typedef  internal::ProxySentinel  ProxySentinel;


public: // Types


	typedef  typename Super::Base  Base;
	typedef  typename Base::Section  Section;
	typedef  typename Base::IR  IR;

	// STL Style
	typedef  typename Base::value_type  value_type;
	typedef  typename Base::reference  reference;
	typedef  typename Base::const_reference  const_reference;
	typedef  typename Base::pointer  pointer;
	typedef  typename Base::const_pointer  const_pointer;
	typedef  typename Base::size_type  size_type;
	typedef  typename Base::difference_type  difference_type;

	// C++ Style
	typedef  typename Base::Value  Value;
	typedef  typename Base::Reference  Reference;
	typedef  typename Base::ConstReference  ConstReference;
	typedef  typename Base::Pointer  Pointer;
	typedef  typename Base::ConstPointer  ConstPointer;
	typedef platform::Size Size;
	typedef  typename Base::Difference  Difference;

	using Super::array_;
	using Super::array_size_;
	using Super::dimensions_initialized;
	using Super::npos;
	using Super::sarray_;
	using Super::shift_;

	// Types to prevent compile failure when std::distance is in scope
	typedef  void  iterator_category;


protected: // Creation


	/// @brief Default Constructor
	inline
	FArray1DB()
	{}


	/// @brief Copy Constructor
	inline
	FArray1DB( FArray1DB const & a ) :
		Super( a )
	{}


	/// @brief Copy Constructor Template
	template< typename U >
	inline
	explicit
	FArray1DB( FArray1DB< U > const & a ) :
		Super( a )
	{}


	/// @brief Size Constructor
	inline
	explicit
	FArray1DB( size_type const size_a ) :
		Super( size_a )
	{}


	/// @brief Size + InitializerSentinel Constructor
	inline
	FArray1DB( size_type const size_a, InitializerSentinel const & initialized ) :
		Super( size_a, initialized )
	{}


	/// @brief Default Proxy Constructor
	inline
	FArray1DB( ProxySentinel const & proxy ) :
		Super( proxy )
	{}


	/// @brief Copy Proxy Constructor
	inline
	FArray1DB( FArray1DB const & a, ProxySentinel const & proxy ) :
		Super( a, proxy )
	{}


	/// @brief Base Proxy Constructor
	inline
	FArray1DB( Base const & a, ProxySentinel const & proxy ) :
		Super( a, proxy )
	{}


	/// @brief Section Proxy Constructor
	inline
	FArray1DB( Section const & s, ProxySentinel const & proxy ) :
		Super( s, proxy )
	{}


	/// @brief Value Proxy Constructor
	inline
	FArray1DB( value_type const & t, ProxySentinel const & proxy ) :
		Super( t, proxy )
	{}


public: // Creation


	/// @brief Destructor
	inline
	virtual
	~FArray1DB()
	{}


public: // Assignment


	/// @brief Copy Assignment
	inline
	FArray1DB &
	operator =( FArray1DB const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension_assign( a.I() );
			Super::operator =( a );
		}
		return *this;
	}


	/// @brief Copy Assignment Template
	template< typename U >
	inline
	FArray1DB &
	operator =( FArray1DB< U > const & a )
	{
		if ( ! equal_dimension( a ) ) dimension_assign( a.I() );
		Super::operator =( a );
		return *this;
	}


	/// @brief += Array Template
	template< typename U >
	inline
	FArray1DB &
	operator +=( FArray1DB< U > const & a )
	{
		assert( equal_dimension( a ) );
		Super::operator +=( a );
		return *this;
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	FArray1DB &
	operator -=( FArray1DB< U > const & a )
	{
		assert( equal_dimension( a ) );
		Super::operator -=( a );
		return *this;
	}


	/// @brief = Value
	inline
	FArray1DB &
	operator =( value_type const & t )
	{
		Super::operator =( t );
		return *this;
	}


	/// @brief += Value
	inline
	FArray1DB &
	operator +=( value_type const & t )
	{
		Super::operator +=( t );
		return *this;
	}


	/// @brief -= Value
	inline
	FArray1DB &
	operator -=( value_type const & t )
	{
		Super::operator -=( t );
		return *this;
	}


	/// @brief *= Value
	inline
	FArray1DB &
	operator *=( value_type const & t )
	{
		Super::operator *=( t );
		return *this;
	}


	/// @brief /= Value
	inline
	FArray1DB &
	operator /=( value_type const & t )
	{
		Super::operator /=( t );
		return *this;
	}


public: // Subscript


	/// @brief array( i ) const
	inline
	value_type const &
	operator ()( int const i ) const
	{
		assert( contains( i ) );
		return sarray_[ i ];
	}


	/// @brief array( i )
	inline
	value_type &
	operator ()( int const i )
	{
		assert( contains( i ) );
		return sarray_[ i ];
	}


	/// @brief Section Starting at array( i )
	inline
	Section const
	a( int const i ) const
	{
		assert( contains( i ) );
		return Section( ( array_size_ != npos ) ? array_size_ - ( i - shift_ ) : npos, sarray_ + i );
	}


	/// @brief Linear Index
	inline
	size_type
	index( int const i ) const
	{
		assert( dimensions_initialized() );
		return ( i - shift_ );
	}


public: // Predicate


	/// @brief Contains Indexed Element?
	virtual
	inline
	bool
	contains( int const i ) const
	{
		return ( I().contains( i ) );
	}


	/// @brief Equal Dimension?
	template< typename U >
	inline
	bool
	equal_dimension( FArray1DB< U > const & a ) const
	{
		return ( I() == a.I() );
	}


public: // Inspector


	/// @brief IndexRange
	virtual
	IR const &
	I1() const = 0;


	/// @brief Lower Index
	virtual
	int
	l1() const = 0;


	/// @brief Upper Index
	virtual
	int
	u1() const = 0;


	/// @brief Size
	virtual
	size_type
	size1() const = 0;


	/// @brief IndexRange
	virtual
	IR const &
	I() const = 0;


	/// @brief Lower Index
	virtual
	int
	l() const = 0;


	/// @brief Upper Index
	virtual
	int
	u() const = 0;


	/// @brief Length
	inline
	value_type
	length() const
	{
		value_type length_sq( value_type( 0 ) );
		for ( int i = l(), e = u(); i <= e; ++i ) {
			value_type const length_i( sarray_[ i ] );
			length_sq += length_i * length_i;
		}
		return std::sqrt( length_sq );
	}


	/// @brief Length Squared
	inline
	value_type
	length_squared() const
	{
		value_type length_sq( value_type( 0 ) );
		for ( int i = l(), e = u(); i <= e; ++i ) {
			value_type const length_i( sarray_[ i ] );
			length_sq += length_i * length_i;
		}
		return length_sq;
	}


public: // Modifier


	/// @brief Clear
	inline
	FArray1DB &
	clear()
	{
		Super::clear();
		return *this;
	}


	/// @brief Assign Default Value to all Elements
	inline
	FArray1DB &
	to_default()
	{
		Super::to_default();
		return *this;
	}


	/// @brief Normalize to Unit Length
	inline
	FArray1DB &
	normalize()
	{
		value_type const length_( length() );
		assert( length_ > value_type( 0 ) );
		operator /=( length_ );
		return *this;
	}


public: // Comparison


	/// @brief FArray1DB == FArray1DB
	friend
	inline
	bool
	operator ==( FArray1DB const & a, FArray1DB const & b )
	{
		if ( &a == &b ) { // Same objects
			return true;
		} else if ( equal_dimensions( a, b ) ) { // Index ranges match
			return ( static_cast< Super const & >( a ) == static_cast< Super const & >( b ) );
		} else { // Index ranges differ
			return false;
		}
	}


	/// @brief FArray1DB != FArray1DB
	friend
	inline
	bool
	operator !=( FArray1DB const & a, FArray1DB const & b )
	{
		return !( a == b );
	}


public: // Friend


	/// @brief Dot Product
	friend
	inline
	value_type
	dot_product( FArray1DB const & a, FArray1DB const & b )
	{
		assert( equal_dimensions( a, b ) );
		value_type sum( value_type( 0 ) );
		for ( int i = a.l(), e = a.u(); i <= e; ++i ) {
			sum += a( i ) * b( i );
		}
		return sum;
	}


	/// @brief Dot Product
	friend
	inline
	value_type
	dot( FArray1DB const & a, FArray1DB const & b )
	{
		assert( equal_dimensions( a, b ) );
		value_type sum( value_type( 0 ) );
		for ( int i = a.l(), e = a.u(); i <= e; ++i ) {
			sum += a( i ) * b( i );
		}
		return sum;
	}


	/// @brief Distance
	friend
	inline
	value_type
	distance( FArray1DB const & a, FArray1DB const & b )
	{
		assert( equal_dimensions( a, b ) );
		value_type distance_sq( value_type( 0 ) );
		for ( int i = a.l(), e = a.u(); i <= e; ++i ) {
			value_type const distance_i( a( i ) - b( i ) );
			distance_sq += distance_i * distance_i;
		}
		return std::sqrt( distance_sq );
	}


	/// @brief Distance Squared
	friend
	inline
	value_type
	distance_squared( FArray1DB const & a, FArray1DB const & b )
	{
		assert( equal_dimensions( a, b ) );
		value_type distance_sq( value_type( 0 ) );
		for ( int i = a.l(), e = a.u(); i <= e; ++i ) {
			value_type const distance_i( a( i ) - b( i ) );
			distance_sq += distance_i * distance_i;
		}
		return distance_sq;
	}


protected: // Functions


	/// @brief Dimension by IndexRange
	virtual
	void
	dimension_assign( IR const & I_a ) = 0;


	/// @brief Swap
	inline
	void
	swap1DB( FArray1DB & v )
	{
		swapB( v );
	}


}; // FArray1DB


/// @brief Equal Dimensions?
template< typename U, typename V >
inline
bool
equal_dimensions( FArray1DB< U > const & a, FArray1DB< V > const & b )
{
	return ( a.I() == b.I() );
}


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArray1DB_HH
