// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_FArray1Dp_HH
#define INCLUDED_ObjexxFCL_FArray1Dp_HH


// FArray1Dp: Fortran-Compatible 1D Proxy Array
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Dp.fwd.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/ObserverMulti.hh>
#include <ObjexxFCL/DynamicIndexRange.hh>


namespace ObjexxFCL {


/// @brief FArray1Dp: Fortran-Compatible 1D Proxy Array
template< typename T >
class FArray1Dp :
	public FArray1DB< T >,
	public ObserverMulti
{


private: // Types


	typedef  FArray1DB< T >  Super;
	typedef  typename Super::real_FArray  real_FArray;
	typedef  typename Super::proxy_FArray  proxy_FArray;
	typedef  typename Super::arg_FArray  arg_FArray;
	typedef  internal::ProxySentinel  ProxySentinel;


private: // Friend


	friend class FArray1Da< T >;


public: // Types


	typedef  typename Super::Base  Base;
	typedef  typename Base::Section  Section;
	typedef  typename Super::IR  SIR;
	typedef  DynamicIndexRange  IR;

	// STL Style
	typedef  typename Base::value_type  value_type;
	typedef  typename Base::reference  reference;
	typedef  typename Base::const_reference  const_reference;
	typedef  typename Base::pointer  pointer;
	typedef  typename Base::const_pointer  const_pointer;
	typedef  typename Base::size_type  size_type;
	typedef  typename Base::difference_type  difference_type;

	// C++ Style
	typedef  typename Base::Value  Value;
	typedef  typename Base::Reference  Reference;
	typedef  typename Base::ConstReference  ConstReference;
	typedef  typename Base::Pointer  Pointer;
	typedef  typename Base::ConstPointer  ConstPointer;
	typedef platform::Size Size;
	typedef  typename Base::Difference  Difference;

	using Super::array_;
	using Super::array_size_;
	using Super::npos;
	using Super::sarray_;
	using Super::shift_;
	using Super::shift_set;
	using Super::size_set;


public: // Creation


	/// @brief Default Constructor
	inline
	FArray1Dp() :
		Super( ProxySentinel() ),
		source_( 0 )
	{}


	/// @brief Copy Constructor
	inline
	FArray1Dp( FArray1Dp const & a ) :
		Super( a, ProxySentinel() ),
		ObserverMulti(),
		I_( a.I_ ),
		source_( &a )
	{
		shift_set( a.shift_ );
		insert_as_observer();
	}


	/// @brief Real Constructor
	inline
	FArray1Dp( real_FArray const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I_ ),
		source_( &a )
	{
		shift_set( a.shift_ );
		insert_as_observer();
	}


	/// @brief Super Constructor
	inline
	FArray1Dp( Super const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I() ),
		source_( dynamic_cast< SubjectMulti const * >( &a ) )
	{
		shift_set( a.shift_ );
		insert_as_observer();
	}


	/// @brief Base Constructor
	inline
	FArray1Dp( Base const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.size() ),
		source_( dynamic_cast< SubjectMulti const * >( &a ) )
	{
		shift_set( 1 );
		insert_as_observer();
	}


	/// @brief Section Constructor
	inline
	FArray1Dp( Section const & s ) :
		Super( s, ProxySentinel() ),
		I_( s.size() ),
		source_( 0 )
	{
		shift_set( 1 );
		insert_as_observer();
	}


	/// @brief Value Constructor
	inline
	FArray1Dp( value_type const & t ) :
		Super( t, ProxySentinel() ),
		I_( star ), // Unbounded
		source_( 0 )
	{
		shift_set( 1 );
		insert_as_observer();
	}


	/// @brief Copy + IndexRange Constructor
	inline
	FArray1Dp( FArray1Dp const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a ),
		source_( &a )
	{
		dimension_proxy();
		insert_as_observer();
	}


	/// @brief Super + IndexRange Constructor
	inline
	FArray1Dp( Super const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a ),
		source_( dynamic_cast< SubjectMulti const * >( &a ) )
	{
		dimension_proxy();
		insert_as_observer();
	}


	/// @brief Base + IndexRange Constructor
	inline
	FArray1Dp( Base const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a ),
		source_( dynamic_cast< SubjectMulti const * >( &a ) )
	{
		dimension_proxy();
		insert_as_observer();
	}


	/// @brief Section + IndexRange Constructor
	inline
	FArray1Dp( Section const & s, IR const & I_a ) :
		Super( s, ProxySentinel() ),
		I_( I_a ),
		source_( 0 )
	{
		dimension_proxy();
		insert_as_observer();
	}


	/// @brief Value + IndexRange Constructor
	inline
	FArray1Dp( value_type const & t, IR const & I_a ) :
		Super( t, ProxySentinel() ),
		I_( I_a ),
		source_( 0 )
	{
		dimension_proxy();
		insert_as_observer();
	}


	/// @brief Destructor
	inline
	virtual
	~FArray1Dp()
	{
		if ( source_ ) source_->remove_observer( *this );
	}


public: // Assignment


	/// @brief Copy Assignment
	inline
	FArray1Dp &
	operator =( FArray1Dp const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment
	inline
	FArray1Dp &
	operator =( Super const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment Template
	template< typename U >
	inline
	FArray1Dp &
	operator =( FArray1DB< U > const & a )
	{
		if ( ! equal_dimension( a ) ) dimension( a );
		Base::operator =( a );
		return *this;
	}


	/// @brief += Array Template
	template< typename U >
	inline
	FArray1Dp &
	operator +=( FArray1DB< U > const & a )
	{
		Super::operator +=( a );
		return *this;
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	FArray1Dp &
	operator -=( FArray1DB< U > const & a )
	{
		Super::operator -=( a );
		return *this;
	}


	/// @brief = Value
	inline
	FArray1Dp &
	operator =( value_type const & t )
	{
		Super::operator =( t );
		return *this;
	}


	/// @brief += Value
	inline
	FArray1Dp &
	operator +=( value_type const & t )
	{
		Super::operator +=( t );
		return *this;
	}


	/// @brief -= Value
	inline
	FArray1Dp &
	operator -=( value_type const & t )
	{
		Super::operator -=( t );
		return *this;
	}


	/// @brief *= Value
	inline
	FArray1Dp &
	operator *=( value_type const & t )
	{
		Super::operator *=( t );
		return *this;
	}


	/// @brief /= Value
	inline
	FArray1Dp &
	operator /=( value_type const & t )
	{
		Super::operator /=( t );
		return *this;
	}


public: // Subscript


	/// @brief array( i ) const
	inline
	value_type const &
	operator ()( int const i ) const
	{
		assert( I_.contains( i ) );
		return sarray_[ i ];
	}


	/// @brief array( i )
	inline
	value_type &
	operator ()( int const i )
	{
		assert( I_.contains( i ) );
		return sarray_[ i ];
	}


	/// @brief Section Starting at array( i )
	inline
	Section const
	a( int const i ) const
	{
		assert( I_.contains( i ) );
		return Section( ( array_size_ != npos ) ? array_size_ - ( i - shift_ ) : npos, sarray_ + i );
	}


	/// @brief Linear Index
	inline
	size_type
	index( int const i ) const
	{
		assert( I_.initialized() );
		return ( i - shift_ );
	}


public: // Predicate


	/// @brief Dimensions Initialized?
	inline
	bool
	dimensions_initialized() const
	{
		return I_.initialized();
	}


	/// @brief Contains Indexed Element?
	inline
	bool
	contains( int const i ) const
	{
		return I_.contains( i );
	}


	/// @brief Initializer Active?
	inline
	bool
	initializer_active() const
	{
		return false;
	}


public: // Inspector


//	/// @brief Derived Type
//	inline
//	FArray1Dp const &
//	type() const
//	{
//		return *this;
//	}


	/// @brief IndexRange
	inline
	IR const &
	I1() const
	{
		return I_;
	}


	/// @brief Lower Index
	inline
	int
	l1() const
	{
		return I_.l();
	}


	/// @brief Upper Index
	inline
	int
	u1() const
	{
		return I_.u();
	}


	/// @brief Size
	inline
	size_type
	size1() const
	{
		return I_.size();
	}


	/// @brief IndexRange
	inline
	IR const &
	I() const
	{
		return I_;
	}


	/// @brief Lower Index
	inline
	int
	l() const
	{
		return I_.l();
	}


	/// @brief Upper Index
	inline
	int
	u() const
	{
		return I_.u();
	}


public: // Modifier


	/// @brief Clear
	inline
	FArray1Dp &
	clear()
	{
		Super::clear();
		I_.clear_no_notify();
		source_ = 0;
		return *this;
	}


	/// @brief Dimension by IndexRange
	inline
	FArray1Dp &
	dimension( IR const & I_a )
	{
		I_.assign_no_notify( I_a );
		dimension_proxy();
		return *this;
	}


	/// @brief Dimension by Array
	template< typename U >
	inline
	FArray1Dp &
	dimension( FArray1DB< U > const & a )
	{
		I_.assign_no_notify( a.I() );
		dimension_proxy();
		return *this;
	}


	/// @brief Attach to Proxy Array
	inline
	FArray1Dp &
	attach( FArray1Dp const & a )
	{
		Base::attach( a );
		I_ = a.I_;
		if ( source_ ) source_->remove_observer( *this );
		source_ = &a;
		a.insert_observer( *this );
		return *this;
	}


	/// @brief Attach to Real Array
	inline
	FArray1Dp &
	attach( real_FArray const & a )
	{
		Base::attach( a );
		I_ = a.I_;
		if ( source_ ) source_->remove_observer( *this );
		source_ = &a;
		a.insert_observer( *this );
		return *this;
	}


	/// @brief Attach to Super Array
	inline
	FArray1Dp &
	attach( Super const & a )
	{
		Base::attach( a );
		I_ = a.I();
		if ( source_ ) source_->remove_observer( *this );
		source_ = dynamic_cast< SubjectMulti const * >( &a );
		if ( source_ ) source_->insert_observer( *this );
		return *this;
	}


	/// @brief Attach to Base Array
	inline
	FArray1Dp &
	attach( Base const & a )
	{
		Base::attach( a, 1 );
		I_ = a.size();
		if ( source_ ) source_->remove_observer( *this );
		source_ = dynamic_cast< SubjectMulti const * >( &a );
		if ( source_ ) source_->insert_observer( *this );
		return *this;
	}


	/// @brief Attach to Section
	inline
	FArray1Dp &
	attach( Section const & s )
	{
		Base::attach( s, 1 );
		I_ = s.size();
		if ( source_ ) source_->remove_observer( *this );
		source_ = 0;
		return *this;
	}


	/// @brief Attach to Value
	inline
	FArray1Dp &
	attach( value_type const & t )
	{
		Base::attach( t, 1 );
		I_ = star; // Unbounded
		if ( source_ ) source_->remove_observer( *this );
		source_ = 0;
		return *this;
	}


	/// @brief Detach from Source Array
	inline
	FArray1Dp &
	detach()
	{
		Base::detach();
		I_.clear();
		if ( source_ ) source_->remove_observer( *this );
		source_ = 0;
		return *this;
	}


public: // Observer Modifier


	/// @brief Update
	inline
	void
	update()
	{
		if ( source_ ) update_to( *dynamic_cast< Base const * >( source_ ) );
		dimension_proxy();
	}


	/// @brief Update for Destruction of a Subject
	inline
	void
	destructed( Subject const & subject )
	{
		if ( ( source_ ) && ( &subject == source_ ) ) { // Source array is being destructed
			Base::detach();
			I_.clear();
			source_ = 0;
		}
	}


protected: // Functions


	/// @brief Dimension by IndexRange
	inline
	void
	dimension_assign( SIR const & I_a )
	{
		I_.assign_no_notify( I_a );
		dimension_proxy();
	}


private: // Functions


	/// @brief Setup for Current IndexRange Dimensions
	inline
	void
	dimension_proxy()
	{
		if ( dimensions_initialized() ) {
			if ( I_.bounded_value() ) { // Bounded
				size_set( I_.size() );
			} else if ( array_size_ != npos ) { // Unbounded with bounded data array
				// Infer upper index and size
				I_.u_no_notify( I_.lz() + array_size_ - 1 );
				size_set( I_.size() );
			} else { // Unbounded with unbounded data array
				size_set( npos );
			}
			shift_set( I_.lz() );
		} else {
			size_set( 0 );
			shift_set( 0 );
		}
	}


	/// @brief Insert as Observer
	inline
	void
	insert_as_observer()
	{
		I_.insert_observer( *this );
		if ( source_ ) source_->insert_observer( *this );
	}


	/// @brief Remove as Observer
	inline
	void
	remove_as_observer()
	{
		I_.remove_observer( *this );
		if ( source_ ) source_->remove_observer( *this );
	}


private: // Data


	/// @brief Index range
	IR I_;

	/// @brief Pointer (non-owning) to source array (0 if unknown)
	SubjectMulti const * source_;


}; // FArray1Dp


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArray1Dp_HH
