// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_FArray4DB_HH
#define INCLUDED_ObjexxFCL_FArray4DB_HH


// FArray4DB: Fortran-Compatible 4D Array Abstract Base Class
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArray4DB.fwd.hh>
#include <ObjexxFCL/FArrayB.hh>


namespace ObjexxFCL {


// Forward Declarations
template< typename > class FArray4D;
template< typename > class FArray4Dp;
template< typename > class FArray4Da;


/// @brief FArray4DB: Fortran-Compatible 4D Array Abstract Base Class
template< typename T >
class FArray4DB :
	public FArrayB< T >
{


private: // Types


	typedef  FArrayB< T >  Super;
	typedef  FArray4D< T >  real_FArray;
	typedef  FArray4Dp< T >  proxy_FArray;
	typedef  FArray4Da< T >  arg_FArray;


private: // Friend


	template< typename > friend class FArray4DB;
	template< typename > friend class FArray4D;
	template< typename > friend class FArray4Dp;
	template< typename > friend class FArray4Da;


protected: // Types


	typedef  internal::InitializerSentinel  InitializerSentinel;
	typedef  internal::ProxySentinel  ProxySentinel;


public: // Types


	typedef  typename Super::Base  Base;
	typedef  typename Base::Section  Section;
	typedef  typename Base::IR  IR;

	// STL Style
	typedef  typename Base::value_type  value_type;
	typedef  typename Base::reference  reference;
	typedef  typename Base::const_reference  const_reference;
	typedef  typename Base::pointer  pointer;
	typedef  typename Base::const_pointer  const_pointer;
	typedef  typename Base::size_type  size_type;
	typedef  typename Base::difference_type  difference_type;

	// C++ Style
	typedef  typename Base::Value  Value;
	typedef  typename Base::Reference  Reference;
	typedef  typename Base::ConstReference  ConstReference;
	typedef  typename Base::Pointer  Pointer;
	typedef  typename Base::ConstPointer  ConstPointer;
	typedef platform::Size Size;
	typedef  typename Base::Difference  Difference;

	using Super::array_;
	using Super::array_size_;
	using Super::dimensions_initialized;
	using Super::npos;
	using Super::sarray_;
	using Super::shift_;


protected: // Creation


	/// @brief Default Constructor
	inline
	FArray4DB() :
		s1_( 0 ),
		s2_( 0 ),
		s3_( 0 )
	{}


	/// @brief Copy Constructor
	inline
	FArray4DB( FArray4DB const & a ) :
		Super( a ),
		s1_( a.s1_ ),
		s2_( a.s2_ ),
		s3_( a.s3_ )
	{}


	/// @brief Copy Constructor Template
	template< typename U >
	inline
	explicit
	FArray4DB( FArray4DB< U > const & a ) :
		Super( a ),
		s1_( a.s1_ ),
		s2_( a.s2_ ),
		s3_( a.s3_ )
	{}


	/// @brief Size Constructor
	inline
	explicit
	FArray4DB( size_type const size_a ) :
		Super( size_a )
	{}


	/// @brief Size + InitializerSentinel Constructor
	inline
	FArray4DB( size_type const size_a, InitializerSentinel const & initialized ) :
		Super( size_a, initialized )
	{}


	/// @brief Default Proxy Constructor
	inline
	FArray4DB( ProxySentinel const & proxy ) :
		Super( proxy ),
		s1_( 0 ),
		s2_( 0 ),
		s3_( 0 )
	{}


	/// @brief Copy Proxy Constructor
	inline
	FArray4DB( FArray4DB const & a, ProxySentinel const & proxy ) :
		Super( a, proxy )
	{}


	/// @brief Base Proxy Constructor
	inline
	FArray4DB( Base const & a, ProxySentinel const & proxy ) :
		Super( a, proxy )
	{}


	/// @brief Section Proxy Constructor
	inline
	FArray4DB( Section const & s, ProxySentinel const & proxy ) :
		Super( s, proxy )
	{}


	/// @brief Value Proxy Constructor
	inline
	FArray4DB( value_type const & t, ProxySentinel const & proxy ) :
		Super( t, proxy )
	{}


public: // Creation


	/// @brief Destructor
	inline
	virtual
	~FArray4DB()
	{}


public: // Assignment


	/// @brief Copy Assignment
	inline
	FArray4DB &
	operator =( FArray4DB const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension_assign( a.I1(), a.I2(), a.I3(), a.I4() );
			Super::operator =( a );
		}
		return *this;
	}


	/// @brief Copy Assignment Template
	template< typename U >
	inline
	FArray4DB &
	operator =( FArray4DB< U > const & a )
	{
		if ( ! equal_dimension( a ) ) dimension_assign( a.I1(), a.I2(), a.I3(), a.I4() );
		Super::operator =( a );
		return *this;
	}


	/// @brief += Array Template
	template< typename U >
	inline
	FArray4DB &
	operator +=( FArray4DB< U > const & a )
	{
		assert( equal_dimension( a ) );
		Super::operator +=( a );
		return *this;
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	FArray4DB &
	operator -=( FArray4DB< U > const & a )
	{
		assert( equal_dimension( a ) );
		Super::operator -=( a );
		return *this;
	}


	/// @brief = Value
	inline
	FArray4DB &
	operator =( value_type const & t )
	{
		Super::operator =( t );
		return *this;
	}


	/// @brief += Value
	inline
	FArray4DB &
	operator +=( value_type const & t )
	{
		Super::operator +=( t );
		return *this;
	}


	/// @brief -= Value
	inline
	FArray4DB &
	operator -=( value_type const & t )
	{
		Super::operator -=( t );
		return *this;
	}


	/// @brief *= Value
	inline
	FArray4DB &
	operator *=( value_type const & t )
	{
		Super::operator *=( t );
		return *this;
	}


	/// @brief /= Value
	inline
	FArray4DB &
	operator /=( value_type const & t )
	{
		Super::operator /=( t );
		return *this;
	}


public: // Subscript


	/// @brief array( i1, i2, i3, i4 ) const
	inline
	value_type const &
	operator ()( int const i1, int const i2, int const i3, int const i4 ) const
	{
		assert( contains( i1, i2, i3, i4 ) );
		return sarray_[ ( ( ( ( ( i4 * s3_ ) + i3 ) * s2_ ) + i2 ) * s1_ ) + i1 ];
	}


	/// @brief array( i1, i2, i3, i4 )
	inline
	value_type &
	operator ()( int const i1, int const i2, int const i3, int const i4 )
	{
		assert( contains( i1, i2, i3, i4 ) );
		return sarray_[ ( ( ( ( ( i4 * s3_ ) + i3 ) * s2_ ) + i2 ) * s1_ ) + i1 ];
	}


	/// @brief Section Starting at array( i1, i2, i3, i4 )
	inline
	Section const
	a( int const i1, int const i2, int const i3, int const i4 ) const
	{
		assert( contains( i1, i2, i3, i4 ) );
		size_type const offset( ( ( ( ( ( ( i4 * s3_ ) + i3 ) * s2_ ) + i2 ) * s1_ ) + i1 ) - shift_ );
		return Section( ( array_size_ != npos ) ? array_size_ - offset : npos, array_ + offset );
	}


	/// @brief Linear Index
	inline
	size_type
	index( int const i1, int const i2, int const i3, int const i4 ) const
	{
		assert( dimensions_initialized() );
		return ( ( ( ( ( ( ( i4 * s3_ ) + i3 ) * s2_ ) + i2 ) * s1_ ) + i1 ) - shift_ );
	}


public: // Predicate


	/// @brief Contains Indexed Element?
	virtual
	inline
	bool
	contains( int const i1, int const i2, int const i3, int const i4 ) const
	{
		return ( ( I1().contains( i1 ) ) && ( I2().contains( i2 ) ) && ( I3().contains( i3 ) ) && ( I4().contains( i4 ) ) );
	}


	/// @brief Equal Dimension?
	template< typename U >
	inline
	bool
	equal_dimension( FArray4DB< U > const & a ) const
	{
		return ( ( I1() == a.I1() ) && ( I2() == a.I2() ) && ( I3() == a.I3() ) && ( I4() == a.I4() ) );
	}


public: // Inspector


	/// @brief IndexRange of Dimension 1
	virtual
	IR const &
	I1() const = 0;


	/// @brief Lower Index of Dimension 1
	virtual
	int
	l1() const = 0;


	/// @brief Upper Index of Dimension 1
	virtual
	int
	u1() const = 0;


	/// @brief Size of Dimension 1
	inline
	size_type
	size1() const
	{
		return s1_;
	}


	/// @brief IndexRange of Dimension 2
	virtual
	IR const &
	I2() const = 0;


	/// @brief Lower Index of Dimension 2
	virtual
	int
	l2() const = 0;


	/// @brief Upper Index of Dimension 2
	virtual
	int
	u2() const = 0;


	/// @brief Size of Dimension 2
	inline
	size_type
	size2() const
	{
		return s2_;
	}


	/// @brief IndexRange of Dimension 3
	virtual
	IR const &
	I3() const = 0;


	/// @brief Lower Index of Dimension 3
	virtual
	int
	l3() const = 0;


	/// @brief Upper Index of Dimension 3
	virtual
	int
	u3() const = 0;


	/// @brief Size of Dimension 3
	inline
	size_type
	size3() const
	{
		return s3_;
	}


	/// @brief IndexRange of Dimension 4
	virtual
	IR const &
	I4() const = 0;


	/// @brief Lower Index of Dimension 4
	virtual
	int
	l4() const = 0;


	/// @brief Upper Index of Dimension 4
	virtual
	int
	u4() const = 0;


	/// @brief Size of Dimension 4
	virtual
	size_type
	size4() const = 0;


public: // Modifier


	/// @brief Clear
	inline
	FArray4DB &
	clear()
	{
		Super::clear();
		s1_ = s2_ = s3_ = 0;
		return *this;
	}


	/// @brief Assign Default Value to all Elements
	inline
	FArray4DB &
	to_default()
	{
		Super::to_default();
		return *this;
	}


public: // Comparison


	/// @brief FArray4DB == FArray4DB
	friend
	inline
	bool
	operator ==( FArray4DB const & a, FArray4DB const & b )
	{
		if ( &a == &b ) { // Same objects
			return true;
		} else if ( equal_dimensions( a, b ) ) { // Index ranges match
			return ( static_cast< Super const & >( a ) == static_cast< Super const & >( b ) );
		} else { // Index ranges differ
			return false;
		}
	}


	/// @brief FArray4DB != FArray4DB
	friend
	inline
	bool
	operator !=( FArray4DB const & a, FArray4DB const & b )
	{
		return !( a == b );
	}


protected: // Functions


	/// @brief Dimension by IndexRange
	virtual
	void
	dimension_assign( IR const & I1_a, IR const & I2_a, IR const & I3_a, IR const & I4_a ) = 0;


	/// @brief Swap
	inline
	void
	swap4DB( FArray4DB & v )
	{
		swapB( v );
		std::swap( s1_, v.s1_ );
		std::swap( s2_, v.s2_ );
		std::swap( s3_, v.s3_ );
	}


protected: // Data


	/// @brief Dimension 1 size
	size_type s1_;

	/// @brief Dimension 2 size
	size_type s2_;

	/// @brief Dimension 3 size
	size_type s3_;


}; // FArray4DB


/// @brief Equal Dimensions?
template< typename U, typename V >
inline
bool
equal_dimensions( FArray4DB< U > const & a, FArray4DB< V > const & b )
{
	return ( ( a.I1() == b.I1() ) && ( a.I2() == b.I2() ) && ( a.I3() == b.I3() ) && ( a.I4() == b.I4() ) );
}


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArray4DB_HH
