// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_FArrayInitializer_HH
#define INCLUDED_ObjexxFCL_FArrayInitializer_HH


// FArrayInitializer: FArray Initializer Class Template
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArrayInitializer.fwd.hh>
#include <ObjexxFCL/FArrayTraits.hh>

// C++ Standard Library Headers
#include <cassert>


namespace ObjexxFCL {


/// @brief FArrayInitializer: FArray Initializer Class Template
template<
	typename T,
	template< typename > class Array
>
class FArrayInitializer
{


public: // Types


	typedef  FArrayTraits< T >  Traits;

	// STL style
	typedef  T  value_type;
	typedef  void (*function_type)( Array< T > & );

	// C++ style
	typedef  T  Value;
	typedef  void (*Function)( Array< T > & );


private: // Types


	enum State { INACTIVE, VALUE, FUNCTION };


public: // Creation


	/// @brief Default Constructor
	inline
	FArrayInitializer() :
		state_( INACTIVE ),
		value_( Traits::initial_value() ),
		function_( static_cast< function_type >( 0 ) )
	{}


	/// @brief Value Constructor
	inline
	explicit
	FArrayInitializer( value_type const & value_a ) :
		state_( VALUE ),
		value_( value_a ),
		function_( static_cast< function_type >( 0 ) )
	{}


	/// @brief Function Constructor
	inline
	explicit
	FArrayInitializer( function_type const & function_a ) :
		state_( function_a ? FUNCTION : INACTIVE ),
		value_( Traits::initial_value() ),
		function_( function_a ? function_a : static_cast< function_type >( 0 ) )
	{}


public: // Assignment


	/// @brief Value Assignment
	inline
	FArrayInitializer &
	operator =( value_type const & value_a )
	{
		state_ = VALUE;
		value_ = value_a;
		function_ = static_cast< function_type >( 0 );
		return *this;
	}


	/// @brief Function Assignment
	inline
	FArrayInitializer &
	operator =( function_type const & function_a )
	{
		state_ = ( function_a ? FUNCTION : INACTIVE );
		value_ = Traits::initial_value();
		function_ = ( function_a ? function_a : static_cast< function_type >( 0 ) );
		return *this;
	}


public: // Inspector


	/// @brief Active?
	inline
	bool
	is_active() const
	{
		return ( state_ != INACTIVE );
	}


	/// @brief Value?
	inline
	bool
	is_value() const
	{
		return ( state_ == VALUE );
	}


	/// @brief Function?
	inline
	bool
	is_function() const
	{
		return ( state_ == FUNCTION );
	}


	/// @brief Value
	inline
	value_type const &
	value() const
	{
		assert( state_ == VALUE );
		return value_;
	}


	/// @brief Function
	inline
	function_type const &
	function() const
	{
		assert( state_ == FUNCTION );
		return function_;
	}


public: // Modifier


	/// @brief Clear
	inline
	void
	clear()
	{
		state_ = INACTIVE;
		value_ = Traits::initial_value();
		function_ = static_cast< function_type >( 0 );
	}


private: // Data


	/// @brief State
	State state_;

	/// @brief Value
	value_type value_;

	/// @brief Function
	function_type function_;


}; // FArrayInitializer


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArrayInitializer_HH
