// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_Fmath_HH
#define INCLUDED_ObjexxFCL_Fmath_HH


// Fortran Intrinsic-Compatible and General Math Functions
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// C++ Standard Library Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <limits>


namespace ObjexxFCL {


// min Functions


/// @brief min( short int, short int )
inline
short int
min( short int const a, short int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( int, int )
inline
int
min( int const a, int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( long int, long int )
inline
long int
min( long int const a, long int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( unsigned short int, unsigned short int )
inline
unsigned short int
min( unsigned short int const a, unsigned short int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( unsigned int, unsigned int )
inline
unsigned int
min( unsigned int const a, unsigned int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( unsigned long int, unsigned long int )
inline
unsigned long int
min( unsigned long int const a, unsigned long int const b )
{
	return ( a < b ? a : b );
}


/// @brief min( float, float )
inline
float
min( float const a, float const b )
{
	return ( a < b ? a : b );
}


/// @brief min( double, double )
inline
double
min( double const a, double const b )
{
	return ( a < b ? a : b );
}


/// @brief min( long double, long double )
inline
long double
min( long double const a, long double const b )
{
	return ( a < b ? a : b );
}


/// @brief Use std::min for 2 arguments
using std::min;


/// @brief min( a, b, c )
template< typename T >
inline
T const &
min( T const & a, T const & b, T const & c )
{
	return ( a < b ? ( a < c ? a : c ) : ( b < c ? b : c ) );
}


/// @brief min( a, b, c, d )
template< typename T >
inline
T const &
min( T const & a, T const & b, T const & c, T const & d )
{
	return std::min( std::min( a, b ), std::min( c, d ) );
}


/// @brief min( a, b, c, d, e )
template< typename T >
inline
T const &
min( T const & a, T const & b, T const & c, T const & d, T const & e )
{
	return min( std::min( a, b ), std::min( c, d ), e );
}


/// @brief min( a, b, c, d, e, f )
template< typename T >
inline
T const &
min( T const & a, T const & b, T const & c, T const & d, T const & e, T const & f )
{
	return min( std::min( a, b ), std::min( c, d ), std::min( e, f ) );
}


// max Functions


/// @brief max( short int, short int )
inline
short int
max( short int const a, short int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( int, int )
inline
int
max( int const a, int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( long int, long int )
inline
long int
max( long int const a, long int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( unsigned short int, unsigned short int )
inline
unsigned short int
max( unsigned short int const a, unsigned short int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( unsigned int, unsigned int )
inline
unsigned int
max( unsigned int const a, unsigned int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( unsigned long int, unsigned long int )
inline
unsigned long int
max( unsigned long int const a, unsigned long int const b )
{
	return ( a < b ? b : a );
}


/// @brief max( float, float )
inline
float
max( float const a, float const b )
{
	return ( a < b ? b : a );
}


/// @brief max( double, double )
inline
double
max( double const a, double const b )
{
	return ( a < b ? b : a );
}


/// @brief max( long double, long double )
inline
long double
max( long double const a, long double const b )
{
	return ( a < b ? b : a );
}


/// @brief Use std::max for 2 arguments
using std::max;


/// @brief max( a, b, c )
template< typename T >
inline
T const &
max( T const & a, T const & b, T const & c )
{
	return ( a < b ? ( b < c ? c : b ) : ( a < c ? c : a ) );
}


/// @brief max( a, b, c, d )
template< typename T >
inline
T const &
max( T const & a, T const & b, T const & c, T const & d )
{
	return std::max( std::max( a, b ), std::max( c, d ) );
}


/// @brief max( a, b, c, d, e )
template< typename T >
inline
T const &
max( T const & a, T const & b, T const & c, T const & d, T const & e )
{
	return max( std::max( a, b ), std::max( c, d ), e );
}


/// @brief max( a, b, c, d, e, f )
template< typename T >
inline
T const &
max( T const & a, T const & b, T const & c, T const & d, T const & e, T const & f )
{
	return max( std::max( a, b ), std::max( c, d ), std::max( e, f ) );
}


// Math Functions


/// @brief abs( x ) == | x |
template< typename T >
inline
T
abs( T const & x )
{
	return ( x < T( 0 ) ? -x : x );
}


/// @brief square( x ) == x^2
template< typename T >
inline
T
square( T const & x )
{
	return x * x;
}


/// @brief cube( x ) == x^3
template< typename T >
inline
T
cube( T const & x )
{
	return x * x * x;
}


/// @brief sign( x )
template< typename T >
inline
int
sign( T const & x )
{
	return ( x >= T( 0 ) ? +1 : -1 );
}


/// @brief Sign Transfer from Second Argument to First Argument
template< typename T >
inline
T
sign( T const & x, T const & y )
{
	return ( y >= T( 0 ) ? abs( x ) : -abs( x ) );
}


/// @brief i(mod n)
template< typename T >
inline
T
mod( T const & i, T const & n )
{
	assert( n != T( 0 ) );
	return ( n == T( 0 ) ? T( 0 ) : i - ( std::numeric_limits< T >::is_integer ? i / n : T( static_cast< long int >( i / n ) ) ) * n );
}


/// @brief i(mod n) : float Arguments
inline
float
mod( float const & i, float const & n )
{
	assert( n != 0.0f );
	return ( n == 0.0f ? 0.0f : std::fmod( i, n ) );
}


/// @brief i(mod n) : double Arguments
inline
double
mod( double const & i, double const & n )
{
	assert( n != 0.0 );
	return ( n == 0.0 ? 0.0 : std::fmod( i, n ) );
}


/// @brief i(mod n) : long double Arguments
inline
long double
mod( long double const & i, long double const & n )
{
	assert( n != 0.0l );
	return ( n == 0.0l ? 0.0l : std::fmod( i, n ) );
}


/// @brief nint( x ): Nearest int
template< typename T >
inline
int
nint( T const & x )
{
	return static_cast< int >( x + ( sign( x ) * T( 0.5 ) ) );
}


/// @brief nsint( x ): Nearest short int
template< typename T >
inline
short int
nsint( T const & x )
{
	return static_cast< short int >( x + ( sign( x ) * T( 0.5 ) ) );
}


/// @brief nlint( x ): Nearest long int
template< typename T >
inline
long int
nlint( T const & x )
{
	return static_cast< long int >( x + ( sign( x ) * T( 0.5 ) ) );
}


/// @brief nearest( x ): Nearest R to a T
template< typename R, typename T >
inline
R
nearest( T const & x )
{
	if ( ( std::numeric_limits< R >::is_integer ) && ( ! std::numeric_limits< T >::is_integer ) ) {
		return R( x + ( sign( x ) * T( 0.5 ) ) );
	} else {
		return R( x );
	}
}


/// @brief Greatest Common Divisor
template< typename T >
inline
T
gcd( T const & m, T const & n )
{
	T lo( min( m, n ) );
	T hi( max( m, n ) );
	while ( lo > T( 0 ) ) {
		T const rem( mod( hi, lo ) );
		hi = lo;
		lo = rem;
	}
	return hi;
}


// Bit Functions


/// @brief Bitwise Not
template< typename T >
inline
T
bit_not( T const & x )
{
	return ( ~x );
}


/// @brief Bitwise And
template< typename T >
inline
T
bit_and( T const & x, T const & y )
{
	return ( x & y );
}


/// @brief Bitwise Inclusive Or
template< typename T >
inline
T
bit_or( T const & x, T const & y )
{
	return ( x | y );
}


/// @brief Bitwise Exclusive Or
template< typename T >
inline
T
bit_xor( T const & x, T const & y )
{
	return ( x ^ y );
}


/// @brief Bit Value Set to 1
template< typename T >
inline
T
bit_set( T const & x, T const & pos )
{
	assert( pos >= T( 0 ) );
	return ( x | ( T( 1 ) << pos ) );
}


/// @brief Bit Value Set to 0
template< typename T >
inline
T
bit_clr( T const & x, T const & pos )
{
	assert( pos >= T( 0 ) );
	return ( x & ~( T( 1 ) << pos ) );
}


/// @brief Bit Value Test
template< typename T >
inline
bool
bit_test( T const & x, T const & pos )
{
	assert( pos >= T( 0 ) );
	return ( x & ( T( 1 ) << pos ) );
}


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_Fmath_HH
