// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_formatted_i_HH
#define INCLUDED_ObjexxFCL_formatted_i_HH


// Fortran-Compatible Formatted Input Functions and Classes
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/sbyte.hh>

// C++ Standard Library Headers
#include <complex>
#include <istream>
#include <limits>
#include <sstream>
#include <string>


namespace ObjexxFCL {


// Types
class Fstring;


namespace fmt {


char const SPACE( ' ' );


// Bite: Inputs a String of Given Width from the Input Stream into a Value
template< typename T >
class Bite
{


public: // Creation


	/// @brief Constructor
	inline
	Bite( int const w, T & t ) :
		w_( w ),
		t_( t )
	{}


	/// @brief Destructor
	inline
	~Bite()
	{}


public: // I/O


	/// @brief Input a Bite from Stream
	friend
	inline
	std::istream &
	operator >>( std::istream & stream, Bite const & bite )
	{
		std::stringstream ss;
		char c;
		int i( 0 );
		while ( ( i < bite.w_ ) && ( stream ) && ( stream.peek() != '\n' ) ) {
			stream.get( c );
			if ( stream ) ss << c;
			++i;
		}
		bite.assign( ss );
		stream.setstate( stream.rdstate() | ( ss.rdstate() & ~std::ios_base::eofbit ) );
		return stream;
	}


private: // I/O


	/// @brief Assign Stream Bite to Value: Generic Implementation
	inline
	void
	assign( std::stringstream & ss ) const
	{
		ss >> t_;
	}


private: // Data


	int w_; // Width
	T & t_; // Reference to value


}; // Bite


/// @brief string is Blank?
inline
bool
is_blank_string( std::string const & s )
{
	if ( s.empty() ) {
		return true;
	} else if ( s.find_first_not_of( ' ' ) == std::string::npos ) {
		return true;
	} else {
		return false;
	}
}


// Bite Explicit Specializations


	/// @brief Assign Stream Bite to Value: bool Specialization
	template<>
	inline
	void
	Bite< bool >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = false;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = false;
		}
	}


	/// @brief Assign Stream Bite to Value: byte Specialization
	template<>
	inline
	void
	Bite< sbyte >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: char Specialization
	template<>
	inline
	void
	Bite< char >::assign( std::stringstream & ss ) const
	{
		t_ = ( is_blank_string( ss.str() ) ? SPACE : ss.str()[ 0 ] );
	}


	/// @brief Assign Stream Bite to Value: signed char Specialization
	template<>
	inline
	void
	Bite< signed char >::assign( std::stringstream & ss ) const
	{
		t_ = ( is_blank_string( ss.str() ) ? SPACE : ss.str()[ 0 ] );
	}


	/// @brief Assign Stream Bite to Value: unsigned char Specialization
	template<>
	inline
	void
	Bite< unsigned char >::assign( std::stringstream & ss ) const
	{
		t_ = ( is_blank_string( ss.str() ) ? SPACE : ss.str()[ 0 ] );
	}


	/// @brief Assign Stream Bite to Value: short int Specialization
	template<>
	inline
	void
	Bite< short int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: unsigned short int Specialization
	template<>
	inline
	void
	Bite< unsigned short int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: int Specialization
	template<>
	inline
	void
	Bite< int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: unsigned int Specialization
	template<>
	inline
	void
	Bite< unsigned int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: long int Specialization
	template<>
	inline
	void
	Bite< long int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: unsigned long int Specialization
	template<>
	inline
	void
	Bite< unsigned long int >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0;
		}
	}


	/// @brief Assign Stream Bite to Value: float Specialization
	template<>
	inline
	void
	Bite< float >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0;
		}
	}


	/// @brief Assign Stream Bite to Value: double Specialization
	template<>
	inline
	void
	Bite< double >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0;
		}
	}


	/// @brief Assign Stream Bite to Value: long double Specialization
	template<>
	inline
	void
	Bite< long double >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0l;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0l;
		}
	}


	/// @brief Assign Stream Bite to Value: complex< float > Specialization
	template<>
	inline
	void
	Bite< std::complex< float > >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0;
		}
	}


	/// @brief Assign Stream Bite to Value: complex< double > Specialization
	template<>
	inline
	void
	Bite< std::complex< double > >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0;
		}
	}


	/// @brief Assign Stream Bite to Value: complex< long double > Specialization
	template<>
	inline
	void
	Bite< std::complex< long double > >::assign( std::stringstream & ss ) const
	{
		if ( is_blank_string( ss.str() ) ) {
			t_ = 0.0l;
		} else {
			ss >> t_;
			if ( ss.fail() ) t_ = 0.0l;
		}
	}


	/// @brief Assign Stream Bite to Value: string Specialization
	template<>
	inline
	void
	Bite< std::string >::assign( std::stringstream & ss ) const
	{
		t_ = ss.str();
	}


	/// @brief Assign Stream Bite to Value: Fstring Specialization
	template<>
	void
	Bite< Fstring >::assign( std::stringstream & ss ) const;


// Bite Makers


/// @brief Bite Maker
template< typename T >
inline
Bite< T >
bite( int const w, T & t )
{
	return Bite< T >( w, t );
}


/// @brief bool Bite Maker: Take One Character
inline
Bite< bool >
bite( bool & t )
{
	return Bite< bool >( 1, t );
}


/// @brief char Bite Maker: Take One Character
inline
Bite< char >
bite( char & t )
{
	return Bite< char >( 1, t );
}


/// @brief string Bite Maker: Take Rest of Line
inline
Bite< std::string >
bite( std::string & t )
{
	return Bite< std::string >( std::numeric_limits< int >::max(), t );
}


/// @brief Fstring Bite Maker: Take Length of Fstring
Bite< Fstring >
bite( Fstring & t );


// Skip: Skips Over a Bite of Specified Width from the Input Stream
class Skip
{


public: // Creation


	/// @brief Constructor
	inline
	explicit
	Skip( int const w = 1 ) :
		w_( w )
	{}


	/// @brief Destructor
	inline
	~Skip()
	{}


public: // I/O


	/// @brief Input a Skip from Stream
	friend
	inline
	std::istream &
	operator >>( std::istream & stream, Skip const & skip )
	{
		char c;
		int i( 0 );
		while ( ( i < skip.w_ ) && ( stream ) && ( stream.peek() != '\n' ) ) {
			stream.get( c );
			++i;
		}
		return stream;
	}


private: // Data


	int w_; // Width


}; // Skip


// Skip Maker and Manipulator


/// @brief Skip Maker
inline
Skip
skip( int const w = 1 )
{
	return Skip( w );
}


/// @brief Skip Rest of Line and Line Terminator (Manipulator)
inline
std::istream &
skip( std::istream & stream )
{
	return stream.ignore( std::numeric_limits< std::streamsize >::max(), '\n' );
}


} // namespace fmt
} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_formatted_i_HH
