// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#ifndef INCLUDED_ObjexxFCL_sbyte_HH
#define INCLUDED_ObjexxFCL_sbyte_HH


// sbyte: One-Byte Integer
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/sbyte.fwd.hh>

// C++ Standard Library Headers
#include <cassert>
#include <cstddef>
#include <istream>


namespace ObjexxFCL {


/// @brief sbyte: One-Byte Integer
class sbyte
{


public: // Creation


	/// @brief Default Constructor
	inline
	sbyte() :
		b_( 0 )
	{}


	/// @brief Constructor
	inline
	explicit
	sbyte( short int const i ) :
		b_( i )
	{}


	/// @brief Destructor
	inline
	~sbyte()
	{}


public: // Conversion


	/// @brief short Conversion
	inline
	operator short int() const
	{
		return static_cast< short int >( b_ );
	}


public: // Assignment


	/// @brief = short
	inline
	sbyte &
	operator =( short int const i )
	{
		b_ = i;
		return *this;
	}


	/// @brief += short
	inline
	sbyte &
	operator +=( short int const i )
	{
		b_ += i;
		return *this;
	}


	/// @brief -= short
	inline
	sbyte &
	operator -=( short int const i )
	{
		b_ -= i;
		return *this;
	}


	/// @brief *= short
	inline
	sbyte &
	operator *=( short int const i )
	{
		b_ *= i;
		return *this;
	}


	/// @brief /= short
	inline
	sbyte &
	operator /=( short int const i )
	{
		assert( i != 0 );
		b_ /= i;
		return *this;
	}


public: // Incrememt/Decrement


	/// @brief ++sbyte
	inline
	sbyte &
	operator ++()
	{
		++b_;
		return *this;
	}


	/// @brief sbyte++
	inline
	sbyte const
	operator ++( int )
	{
		sbyte const old( *this );
		++b_;
		return old;
	}


	/// @brief --sbyte
	inline
	sbyte &
	operator --()
	{
		--b_;
		return *this;
	}


	/// @brief sbyte--
	inline
	sbyte const
	operator --( int )
	{
		sbyte const old( *this );
		--b_;
		return old;
	}


public: // Math


	/// @brief +sbyte
	inline
	sbyte
	operator +() const
	{
		return *this;
	}


	/// @brief -sbyte
	inline
	sbyte
	operator -() const
	{
		return sbyte( -static_cast< short int >( b_ ) );
	}


	/// @brief sbyte + sbyte
	friend
	inline
	sbyte
	operator +( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ + j.b_ );
	}


	/// @brief sbyte - sbyte
	friend
	inline
	sbyte
	operator -( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ - j.b_ );
	}


	/// @brief sbyte * sbyte
	friend
	inline
	sbyte
	operator *( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ * j.b_ );
	}


	/// @brief sbyte / sbyte
	friend
	inline
	sbyte
	operator /( sbyte const & i, sbyte const & j )
	{
		assert( j.b_ != 0 );
		return sbyte( i.b_ / j.b_ );
	}


public: // Bitwise Logical


	/// @brief ~sbyte
	inline
	sbyte
	operator ~() const
	{
		return sbyte( ~b_ );
	}


	/// @brief sbyte >> std::size_t
	inline
	sbyte
	operator >>( std::size_t const n ) const
	{
		return sbyte( b_ >> n );
	}


	/// @brief sbyte >> sbyte
	inline
	sbyte
	operator >>( sbyte const n ) const
	{
		return sbyte( b_ >> static_cast< short int >( n ) );
	}


	/// @brief sbyte << std::size_t
	inline
	sbyte
	operator <<( std::size_t const n ) const
	{
		return sbyte( b_ << n );
	}


	/// @brief sbyte << sbyte
	inline
	sbyte
	operator <<( sbyte const n ) const
	{
		return sbyte( b_ << static_cast< short int >( n ) );
	}


	/// @brief &= sbyte
	inline
	sbyte &
	operator &=( sbyte const & i )
	{
		b_ &= i.b_;
		return *this;
	}


	/// @brief |= sbyte
	inline
	sbyte &
	operator |=( sbyte const & i )
	{
		b_ |= i.b_;
		return *this;
	}


	/// @brief ^= sbyte
	inline
	sbyte &
	operator ^=( sbyte const & i )
	{
		b_ ^= i.b_;
		return *this;
	}


	/// @brief sbyte & sbyte
	friend
	inline
	sbyte
	operator &( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ & j.b_ );
	}


	/// @brief sbyte | sbyte
	friend
	inline
	sbyte
	operator |( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ | j.b_ );
	}


	/// @brief sbyte ^ sbyte
	friend
	inline
	sbyte
	operator ^( sbyte const & i, sbyte const & j )
	{
		return sbyte( i.b_ ^ j.b_ );
	}


public: // Comparison


	/// @brief sbyte == sbyte
	friend
	inline
	bool
	operator ==( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ == j.b_ );
	}


	/// @brief sbyte != sbyte
	friend
	inline
	bool
	operator !=( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ != j.b_ );
	}


	/// @brief sbyte < sbyte
	friend
	inline
	bool
	operator <( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ < j.b_ );
	}


	/// @brief sbyte <= sbyte
	friend
	inline
	bool
	operator <=( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ <= j.b_ );
	}


	/// @brief sbyte > sbyte
	friend
	inline
	bool
	operator >( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ > j.b_ );
	}


	/// @brief sbyte >= sbyte
	friend
	inline
	bool
	operator >=( sbyte const & i, sbyte const & j )
	{
		return ( i.b_ >= j.b_ );
	}


public: // I/O


	/// @brief Stream Input
	friend
	inline
	std::istream &
	operator >>( std::istream & stream, sbyte & b )
	{
		short int s;
		stream >> s;
		b.b_ = s;
		return stream;
	}


private: // Data


	/// @brief Value
	signed char b_;


}; // sbyte


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_sbyte_HH
