// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   rosetta/benchmark/benchmark.cc
///
/// @brief
/// @author Sergey Lyskov

#include <core/chemical/ChemicalManager.hh>
#include <core/scoring/Energies.hh>
#include <core/util/Tracer.hh>
#include <core/init.hh>
#include <core/types.hh>

#include <core/options/option.hh>

#include <numeric/random/random.hh>

#include "benchmark.hh"

#include <time.h>
#include <fstream>

#if  !defined(WINDOWS) && !defined(WIN32)
	#include <sys/time.h>
	#include <sys/resource.h>
#endif


const char results_filename[] = "_performance_";


#include "score.bench.hh"
ScoreBenchmark Score_("core.scoring.Score");

#include "SmallMover.bench.hh"
SmallMoverBenchmark SmallMover_("protocols.moves.SmallMover");

#include "ShearMover.bench.hh"
ShearMoverBenchmark ShearMover_("protocols.moves.ShearMover");

#include "Minimizer.bench.hh"
//MinimizerBenchmark Minimizer_("protocols.optimization.Minimizer");
MinimizerBenchmark_dfpmin Minimizer_dfpmin_("protocols.optimization.Minimizer_dfpmin");
MinimizerBenchmark_dfpmin_armijo MinimizerBenchmark_dfpmin_armijo_("protocols.optimization.Minimizer_dfpmin_armijo");
MinimizerBenchmark_dfpmin_armijo_nonmonotone MinimizerBenchmark_dfpmin_armijo_nonmonotone_("protocols.optimization.Minimizer_dfpmin_armijo_nonmonotone");

#include "Docking.bench.hh"
DockingBenchmark_low DockingLow("protocols.docking.DockingLowRes");
DockingBenchmark_high DockingHigh("protocols.docking.DockingHighRes");

#include "Design.bench.hh"
//DesignBenchmark design("protocols.moves.PackRotamersMover");

#include "LigandDock.bench.hh"
LigandDockBenchmark ligand_dock("protocols.ligand_docking.LigandDockProtocol");

// option key includes

#include <core/options/keys/james.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>



using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("benchmark");

using namespace core;

std::vector<Benchmark *> &Benchmark::allBenchmarks()
{
	static std::vector<Benchmark*> * allBenchmarks = new std::vector<Benchmark*>;
	return *allBenchmarks;
}

double Benchmark::execute(int scaleFactor)
{
	/// Reseting RG system before each performance run.
	numeric::random::RandomGenerator::initializeRandomGenerators(
		 1000, numeric::random::_RND_TestRun_, "mt19937");

	TR << "Setting up "<< name() << "..." << std::endl;
	//for(int i=0; i<3; i++) {
	//	std::cout << "X i="<< i << " R=" << numeric::random::uniform() << std::endl;
	//}

	setUp();

	double t;

	#if  !defined(WINDOWS) && !defined(WIN32)
		TR << "Running(U) " << name() << "..." << std::endl;
		struct rusage R0, R1;

		getrusage(RUSAGE_SELF, &R0);
		run(scaleFactor);

		getrusage(RUSAGE_SELF, &R1);

		t = R1.ru_utime.tv_sec + R1.ru_utime.tv_usec*1e-6 - R0.ru_utime.tv_sec - R0.ru_utime.tv_usec*1e-6;
		TR << "Running(U) " << name() << "... Done. Time:" << t << std::endl;
	#else
		TR << "Running(W) " << name() << "..." << std::endl;
		t = clock();
		run(scaleFactor);
		t = clock() - t;
		t = t / CLOCKS_PER_SEC;
		TR << "Running(W) " << name() << "... Done. Time:" << t << std::endl;
	#endif


	TR << "Tear down "<< name() << "..." << std::endl;
	tearDown();
	TR << "Tear down "<< name() << "... Done." << std::endl << std::endl;

	result_ = t;
	return t;
}

void Benchmark::executeAllBenchmarks(int scaleFactor)
{
	TR << std::endl << "Executing all benchmarks..." << std::endl << std::endl;

	std::vector<Benchmark *> & all( allBenchmarks() );

	for(Size i=0; i<all.size(); i++) {
		Benchmark * B = all[i];
		B->execute(scaleFactor);
	}
	TR << std::endl << "Executing all benchmarks... Done." << std::endl;
}

///
/// Generting report file in python dict format: i.e: { 'Bench1': 1.5, 'Bench2': 1.6 }
///
std::string Benchmark::getReport()
{
	std::vector<Benchmark *> & all( allBenchmarks() );

	char buf[1024];

	std::string res = "{\n";
	for(Size i=0; i<all.size(); i++) {
		Benchmark * B = all[i];
		sprintf(buf, "%f", B->result_);
		res += "    '" + B->name_ + "':" + std::string(buf) + ",\n";
	}
	res += "}\n";
	return res;
}

int real_command_line_argc; char ** real_command_line_argv;
int command_line_argc; char ** command_line_argv;


int main( int argc, char *argv[])
{
	command_line_argc=argc; command_line_argv=argv;
	real_command_line_argc=argc; real_command_line_argv=argv;

	using namespace core;
	using namespace options::OptionKeys;

	core::options::option.add_relevant(run::benchmark_scale);
	core::options::option.add_relevant(in::path::database);

	core::init(argc, argv);

	//TR << "DB:"  << core::options::option[ in::path::database ]() << "\n";

	chemical::ResidueTypeSetCAP residue_set
		( chemical::ChemicalManager::get_instance()->residue_type_set( chemical::FA_STANDARD ) );

	//TR << "Specified()=" << core::options::option[ run::benchmark_scale ].user() << "\n";
	//TR << "Legal" << core::options::option[ run::benchmark_scale ].legal() << "\n";
	//TR << "Active:" << core::options::option[ run::benchmark_scale ].user() << "\n";
	//TR << "native:"  << core::options::option[ james::native ]() << "\n";
	//TR << "DB:"  << core::options::option[ in::path::database ]() << "\n";
	int scale = core::options::option[ run::benchmark_scale ]();

	TR << "Mini Benchmark started! Scale factor: " << scale << " -------------" << std::endl;

	//TR << "CLOCKS_PER_SEC:" << CLOCKS_PER_SEC << "\n";

	Benchmark::executeAllBenchmarks(scale);
	std::string report = Benchmark::getReport();
	TR << "Results:" << std::endl << report;  TR.flush();

	/// Now, saving report to a file
	std::ofstream file(results_filename, std::ios::out | std::ios::binary);
	if(!file) {
		Error() << "Benchmark:: Unable to open file:" << results_filename << " for writing!!!" << std::endl;
		return 1;
	}
	file << report;

	file.close();

	TR << "Mini Benchmark ended.   --------------------------------" << std::endl;
	return 0;
}
