// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file backrub.cc
/// @brief run backrub Monte Carlo
/// @author Colin A. Smith (colin.smith@ucsf.edu)
/// @detailed
/// Currently a work in progress. The goal is to match the features of rosetta++ -backrub_mc

// Protocols Headers
#include <protocols/branch_angle/BranchAngleOptimizer.hh>
#include <protocols/jobdist/Jobs.hh>
#include <protocols/jobdist/standard_mains.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/BackrubMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/SidechainMover.hh>
#include <protocols/viewer/viewers.hh>

// Core Headers
#include <core/chemical/ResidueType.hh>
#include <core/conformation/Conformation.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/init.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/mm/MMBondAngleResidueTypeParamSet.hh>
#include <core/options/option.hh>
#include <core/options/util.hh>
#include <core/options/option_macros.hh>
#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/constraints/util.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>

// Utility Headers
#include <utility/file/file_sys_util.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// Numeric Headers
#include <numeric/random/random.hh>

// Platform Headers
#include <platform/types.hh>

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/backrub.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>



static numeric::random::RandomGenerator RG(222578262);
core::util::Tracer TR("apps.backrub");

OPT_1GRP_KEY(Integer, backrub, ntrials)
OPT_1GRP_KEY(Real, backrub, sc_prob)
OPT_1GRP_KEY(Real, backrub, sm_prob)
OPT_1GRP_KEY(Real, backrub, sc_prob_uniform)
OPT_1GRP_KEY(Real, backrub, mc_kt)
OPT_1GRP_KEY(Real, backrub, mm_bend_weight)
OPT_1GRP_KEY(Boolean, backrub, initial_pack)

void *
my_main( void* );

int
main( int argc, char * argv [] )
{
	OPT(in::path::database);
	OPT(in::file::s);
	OPT(in::file::l);
	OPT(in::file::movemap);
	OPT(in::ignore_unrecognized_res);
	OPT(out::nstruct);
	OPT(packing::resfile);
	OPT(constraints::cst_fa_weight);
	OPT(constraints::cst_fa_file);
	OPT(backrub::pivot_residues);
	OPT(backrub::pivot_atoms);
	OPT(backrub::min_atoms);
	OPT(backrub::max_atoms);
	NEW_OPT(backrub::ntrials, "number of Monte Carlo trials to run", 1000);
	NEW_OPT(backrub::sc_prob, "probability of making a side chain move", 0.25);
	NEW_OPT(backrub::sm_prob, "probability of making a small move", 0);
	NEW_OPT(backrub::sc_prob_uniform, "probability of uniformly sampling chi angles", 0.1);
	NEW_OPT(backrub::mc_kt, "value of kT for Monte Carlo", 0.6);
	NEW_OPT(backrub::mm_bend_weight, "weight of mm_bend bond angle energy term", 1.0);
	NEW_OPT(backrub::initial_pack, "force a repack at the beginning regardless of whether mutations are set in the resfile", false);

	// initialize Rosetta
	core::init(argc, argv);

	protocols::viewer::viewer_main( my_main );

	return 0;
}

bool
read_fold_tree_from_file(
	core::kinematics::FoldTree & foldtree,
	std::string filepath
)
{
	std::ifstream filestream(filepath.c_str());

	while (filestream.good()) {

		std::string line;
		std::string key;

		getline(filestream, line);
		if (filestream.fail()) {
			//TR << "getline() failed" << std::endl;
			return false;
		}

		std::istringstream linestream(line);
		linestream >> key;
		if (key == "FOLD_TREE") {
			linestream.clear();
			linestream.seekg(0, std::ios::beg);
			linestream >> foldtree;
			if (linestream.fail()) {
				TR << "FoldTree parsing failed" << std::endl;
				return false;
			} else {
				return true;
			}
		}
	}

	return false;
}

bool
read_fold_tree_from_file(
	core::pose::Pose & pose,
	std::string filepath
)
{
	core::kinematics::FoldTree foldtree;

	if (read_fold_tree_from_file(foldtree, filepath)) {
		if (foldtree.nres() == pose.total_residue()) {
			pose.fold_tree(foldtree);
			return true;
		} else {
			TR << "Different number of residues in Pose (" << pose.total_residue() << ") and FoldTree (" << foldtree.nres()
			   << ")" << std::endl;
		}
	}

	return false;
}

void
append_fold_tree_to_file(
	core::kinematics::FoldTree const & foldtree,
	std::string file_path
)
{
	std::ofstream filestream(file_path.c_str(), std::ios::out|std::ios::app);
	if (filestream.good()) {
		filestream << foldtree;
		filestream.close();
	} else {
		TR << "couldn't open file to append FoldTree" << std::endl;
	}
}

utility::vector1<core::Size>
positions_incompatible_with_task(
	core::pose::Pose & pose,
	core::pack::task::PackerTask & packertask
)
{
	utility::vector1<core::Size> incompatible_positions;

	assert(pose.total_residue() == packertask.total_residue());

	// iterate over all residues to see if they're compatible
	for (core::Size i = 1; i <= pose.total_residue(); ++i) {

		// only check packable residues for compatibility
		if (packertask.pack_residue(i)) {

			// assume residue is incompatible
			bool incompatible(true);

			// check to see if pose residue type is in list of allowed residue types
			core::pack::task::ResidueLevelTask const & residueleveltask(packertask.residue_task(i));
			for (core::pack::task::ResidueLevelTask::ResidueTypeCAPListConstIter iter(residueleveltask.allowed_residue_types_begin());
			     iter != residueleveltask.allowed_residue_types_end(); ++iter) {

				if ((*iter)->name() == pose.residue_type(i).name()) incompatible = false;
			}

			if (incompatible) incompatible_positions.push_back(i);
		}
	}

	return incompatible_positions;
}

void *
my_main( void* )
{
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace core::pack::task;

	// create a TaskFactory with the resfile
	TaskFactoryOP main_task_factory = new TaskFactory;
	main_task_factory->push_back( new operation::InitializeFromCommandline );
	if ( option[ packing::resfile ].user() ) {
		main_task_factory->push_back( new operation::ReadResfile );
	} else {
		operation::RestrictToRepackingOP rtrop = new operation::RestrictToRepacking;
		main_task_factory->push_back( rtrop );
	}
	// C-beta atoms should not be altered during packing because branching atoms are optimized
	main_task_factory->push_back( new operation::PreserveCBeta );

	// set up the score function and add the bond angle energy term
	core::scoring::ScoreFunctionOP score_fxn = core::scoring::getScoreFunction();
	score_fxn->set_weight(core::scoring::mm_bend, option[ backrub::mm_bend_weight ]);
	core::scoring::methods::EnergyMethodOptions energymethodoptions(score_fxn->energy_method_options());
	energymethodoptions.decompose_bb_hb_into_pair_energies(true);
	energymethodoptions.bond_angle_central_atoms_to_score(option[ backrub::pivot_atoms ]);
	score_fxn->set_energy_method_options(energymethodoptions);
	if ( option[ in::file::centroid_input ].user() ) {
		core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn(*score_fxn);
	} else {
		core::scoring::constraints::add_fa_constraints_from_cmdline_to_scorefxn(*score_fxn);
	}

	// set up the BackrubMover
	protocols::moves::BackrubMover backrubmover;
	// read known and unknown optimization parameters from the database
	backrubmover.branchopt().read_database();
	// tell the branch angle optimizer about the score function MMBondAngleResidueTypeParamSet, if any
	if (energymethodoptions.bond_angle_residue_type_param_set()) {
		backrubmover.branchopt().bond_angle_residue_type_param_set(energymethodoptions.bond_angle_residue_type_param_set());
	}

	// set up the SmallMover
	protocols::moves::SmallMover smallmover;
	smallmover.nmoves(1);
	if (option[ backrub::sm_prob ] > 0) {
		core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap;
		movemap->init_from_file(option[ in::file::movemap ]);
		smallmover.movemap(movemap);
	}

	// set up the SidechainMover
	protocols::moves::SidechainMover sidechainmover;
	sidechainmover.set_task_factory(main_task_factory);
	sidechainmover.set_prob_uniform(option[ backrub::sc_prob_uniform ]);

	// set up the PackRotamersMover
	protocols::moves::PackRotamersMover packrotamersmover;
	packrotamersmover.task_factory(main_task_factory);
	packrotamersmover.score_function(score_fxn);

	utility::vector1< protocols::jobdist::BasicJobOP > input_jobs = protocols::jobdist::load_s_and_l();

	for (core::Size jobnum = 1; jobnum <= input_jobs.size(); ++jobnum) {

		TR << "Processing " << input_jobs[jobnum]->input_tag() << "..." << std::endl;
		bool custom_fold_tree(false);

		// load the PDB file
		core::pose::PoseOP input_pose(new core::pose::Pose());
		if ( option[ in::file::centroid_input ].user() ) {
			TR.Warning << "*** This is untested with centroid mode! ***" << std::endl;
			core::io::pdb::centroid_pose_from_pdb( *input_pose, input_jobs[jobnum]->input_tag() );
			custom_fold_tree = read_fold_tree_from_file( *input_pose, input_jobs[jobnum]->input_tag() );
			core::scoring::constraints::add_constraints_from_cmdline_to_pose(*input_pose);
		} else {
			core::io::pdb::pose_from_pdb( *input_pose, input_jobs[jobnum]->input_tag() );
			custom_fold_tree = read_fold_tree_from_file( *input_pose, input_jobs[jobnum]->input_tag() );
			core::scoring::constraints::add_fa_constraints_from_cmdline_to_pose(*input_pose);
		}
		//input_pose->dump_pdb(input_jobs[jobnum]->output_tag(0) + "_postread.pdb");

		backrubmover.clear_segments();
		backrubmover.set_input_pose(input_pose);

		TR << "Backrub segment lengths: " << option[ backrub::min_atoms ] << "-" << option[ backrub::max_atoms ] << " atoms"
		   << std::endl;

		TR << "Backrub main chain pivot atoms: " << option[ backrub::pivot_atoms ].value_string() << std::endl;

		backrubmover.add_mainchain_segments_from_options();

		TR << "Backrub Segments Added: " << backrubmover.num_segments() << std::endl;

		TR << "Score After PDB Load:" << std::endl;
		score_fxn->show(TR, *input_pose);
		TR.flush();

		backrubmover.optimize_branch_angles(*input_pose);
		//input_pose->dump_pdb(input_jobs[jobnum]->output_tag(0) + "_postoptbranch.pdb");
		sidechainmover.idealize_sidechains(*input_pose);
		//input_pose->dump_pdb(input_jobs[jobnum]->output_tag(0) + "_postidealizesc.pdb");

		TR << "Score After Branch Angle Optimization/Side Chain Idealization:" << std::endl;
		score_fxn->show(TR, *input_pose);
		TR.flush();

		// check to see if we need to force a repack
		bool initial_pack(option[ backrub::initial_pack ]);
		core::pack::task::PackerTaskOP temp_task(main_task_factory->create_task_and_apply_taskoperations(*input_pose));
		utility::vector1<core::Size> incompatible_positions(positions_incompatible_with_task(*input_pose, *temp_task));
		if (incompatible_positions.size()) {
			TR << "Starting ResidueType not allowed in resfile at position(s):";
			for (core::Size i = 1; i <= incompatible_positions.size(); i++) {
				TR << " " << incompatible_positions[i];
			}
			TR << std::endl;
			if (!initial_pack) {
				initial_pack = true;
				TR << "Forcing initial pack" << std::endl;
			}
		}

		protocols::moves::MonteCarlo mc(*input_pose, *score_fxn, option[ backrub::mc_kt ]);

		// create viewer windows if OpenGL is enabled
		protocols::viewer::add_monte_carlo_viewer(mc, "Backrub", 600, 600);

		// iterate to generate multiple structures
		for (int structnum = 1; structnum <= input_jobs[jobnum]->nstruct(); ++structnum)
		{
			// start with a fresh copy of the optimized pose
			core::pose::PoseOP pose(new core::pose::Pose(*input_pose));

			// repack/redesign at the beginning if specified/necessary
			if (initial_pack) packrotamersmover.apply(*pose);
			//pose->dump_pdb(input_jobs[jobnum]->output_tag(structnum) + "_postpack.pdb");

			// reset the Monte Carlo object
			mc.reset(*pose);

			TR << "Running " << option[ backrub::ntrials ] << " trials..." << std::endl;

			for (int i = 1; i <= option[ backrub::ntrials ]; ++i) {

				std::string move_type;

				// could use random mover for this...
				core::Real move_prob = RG.uniform();
				if (move_prob > option[ backrub::sm_prob ] + option[ backrub::sc_prob ]) {
					backrubmover.apply(*pose);
					move_type = backrubmover.type();
				} else if (move_prob > option[ backrub::sc_prob ]) {
					smallmover.apply(*pose);
					move_type = smallmover.type();
				} else {
					sidechainmover.apply(*pose);
					move_type = sidechainmover.type();
				}

				mc.boltzmann(*pose, move_type);
			}

			mc.show_counters();

			// repack II: LAST
			// this could also be done IN the loop, so that we get a minimized structure right after the monte carlo run
			//packrotamersmover.apply(*pose);

			// dump out the low score and last accepted poses
			TR << "Last Score:" << std::endl;
			score_fxn->show(TR, *pose);
			TR.flush();

			*pose = mc.lowest_score_pose();

			// repack II: LOW
			//packrotamersmover.apply(*pose);

			TR << "Low Score:" << std::endl;
			score_fxn->show(TR, *pose);
			TR.flush();

			std::string output_tag(input_jobs[jobnum]->output_tag(structnum));
			mc.last_accepted_pose().dump_pdb(output_tag + "_last.pdb");
			mc.lowest_score_pose().dump_pdb(output_tag + "_low.pdb");

			if (custom_fold_tree) {
				append_fold_tree_to_file(mc.last_accepted_pose().fold_tree(), output_tag + "_last.pdb");
				append_fold_tree_to_file(mc.lowest_score_pose().fold_tree(), output_tag + "_low.pdb");
			}
		}
	}

	// write parameters for any sets of branching atoms for which there were not optimization coefficients
	backrubmover.branchopt().write_database();

	return 0;
}
