// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file Mike Tyka
/// @brief


// libRosetta headers
#include <protocols/moves/ScoreMover.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/chemical/ChemicalManager.hh>

#include <protocols/jobdist/standard_mains.hh>
#include <protocols/cluster/cluster.hh>
#include <protocols/loops/LoopClass.hh>

#include <core/options/option.hh>

#include <core/init.hh>

// C++ headers
//#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <deque>

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/cluster.OptionKeys.gen.hh>

using namespace core;
using namespace ObjexxFCL;
using namespace core::pose;
using namespace protocols;
using namespace core::options;

int
main( int argc, char * argv [] ) {

	using namespace protocols;
	using namespace protocols::jobdist;
	using namespace protocols::moves;
	using namespace core::scoring;
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace utility::file;
	using namespace protocols::cluster;

	register_options_universal_main();
	option.add_relevant( OptionKeys::cluster::input_score_filter           );
	option.add_relevant( OptionKeys::cluster::output_score_filter          );
	option.add_relevant( OptionKeys::cluster::exclude_res                  );
	option.add_relevant( OptionKeys::cluster::thinout_factor               );
	option.add_relevant( OptionKeys::cluster::radius                       );
	option.add_relevant( OptionKeys::cluster::limit_cluster_size           );
	option.add_relevant( OptionKeys::cluster::limit_clusters               );
	option.add_relevant( OptionKeys::cluster::limit_total_structures       );
	option.add_relevant( OptionKeys::cluster::sort_groups_by_energy        );
	option.add_relevant( OptionKeys::cluster::remove_highest_energy_member );
	option.add_relevant( OptionKeys::cluster::limit_dist_matrix            );
	option.add_relevant( OptionKeys::cluster::make_ensemble_cst            );
	ScoreMover::register_options();

	// initialize core
	init(argc, argv);

	std::cout << std::endl;
	std::cout << std::endl;
	std::cout << std::endl;
	std::cout << " Rosetta Tool:  cluster - clustering tool for PDBs and or silent files " << std::endl;
	std::cout << " Usage:                                                                  " << std::endl;
	std::cout << "   PDB input:      -in:file:s *.pdb   or  " << std::endl;
	std::cout << "                   -in:file:l  list_of_pdbs  " << std::endl;
	std::cout << "                   -no_optH                                    Dont change positions of Hydrogen atoms! " << std::endl;
	std::cout << "   Silent input:   -in:file:silent silent.out                  silent input filesname " << std::endl;
	std::cout << "                   -in:file:s                                  specify specific tags to be extracted, if left out all will be taken " << std::endl;
	std::cout << "                   -in:file:fullatom                           for full atom structures " << std::endl;
	std::cout << "                   -in:file:silent_struct_type <type>          specify the input silent-file format " << std::endl;
	std::cout << "   Native:         -in:file:native                             native PDB if CaRMS is required " << std::endl;
	std::cout << "   Scorefunction:  -score:weights  weights                     weight set or weights file " << std::endl;
	std::cout << "                   -score:patch  patch                         patch set " << std::endl;
	std::cout << "                   -rescore:verbose                            display score breakdown " << std::endl;
	std::cout << "                   -rescore:output_only                        don't rescore " << std::endl;
	std::cout << "   Output:         -nooutput                                   don't print PDB structures " << std::endl;
	std::cout << "                   -out:prefix  myprefix                       prefix the output structures with a string " << std::endl;
	std::cout << "   Clustering:     -input_score_filter  <float>                Ignore structures above certain energy " << std::endl;
	std::cout << "                   -exclude_res                                Exclude residue numbers               " << std::endl;
	std::cout << "                   -radius                                     Cluster radius" << std::endl;
	std::cout << "                   -limit_cluster_size                         Maximal cluster size" << std::endl;
	std::cout << "                   -limit_clusters                             Maximal number of clusters" << std::endl;
	std::cout << "                   -limit_total_structures                     Maximal number of structures in total" << std::endl;
	std::cout << "                   -limit_dist_matrix                          <defunc> only pre cluster N structures, then redo." << std::endl;
	std::cout << "                   -sort_groups_by_energy                      Sort clusters by energy." << std::endl;
	std::cout << "                   -remove_highest_energy_member               Remove highest energy member of each cluster" << std::endl;
	std::cout << "            -cluster:radius     RMSD in Angstrom" << std::endl;
	std::cout << " Examples: " << std::endl;
	std::cout << "   cluster -database ~/minirosetta_database -in:file:silent silent.out -in::file::binary_silentfile -in::file::fullatom -native 1a19.pdb " << std::endl;
	std::cout << "clustered Poses are given output names in the form of:" << std::endl;
	std::cout << " c.i.j, which denotes the jth member of the ith cluster." << std::endl;
	std::cout << std::endl;
	std::cout << std::endl;
	std::cout << std::endl;

	if ( !option[ out::output ].user() ) {
		option[ out::nooutput ].value( true );
	}

	int time_start = time(NULL);

	MoverOP mover;
	core::scoring::ScoreFunctionOP sfxn;
	sfxn = core::scoring::getScoreFunction();
	ClusterPhilStyleOP clustering;

	if ( option[ core::options::OptionKeys::cluster::loops ]() ) {
		loops::Loops loops = protocols::loops::get_loops_from_file();
		clustering = new ClusterPhilStyle_Loop(loops );
	} else {
		clustering = new ClusterPhilStyle();
	}

	clustering->set_score_function( sfxn );
	clustering->set_cluster_radius(
		option[ core::options::OptionKeys::cluster::radius ]()
	);

	mover = clustering;
	universal_main(
		*mover, option[ core::options::OptionKeys::cluster::thinout_factor]
	);


	int time_readin = time(NULL);
	clustering->do_clustering();
	int time_initialc = time(NULL);
	std::cout << "e... " << std::endl;
	clustering->do_redistribution();

	std::cout << "Assigning extra structures ... " << std::endl;

	if ( option[ core::options::OptionKeys::cluster::thinout_factor] > 0.0 ) {
		AssignToClustersMoverOP mover_add_structures
			= new AssignToClustersMover( clustering );
		mover_add_structures->set_score_function( sfxn );
		universal_main( *mover_add_structures ); // not - no thinout factor here !!!
	}
	int time_total = time(NULL);

	using namespace core::options::OptionKeys::cluster;

	clustering->sort_each_group_by_energy();
	if ( option[ sort_groups_by_energy ].user() ) clustering->sort_groups_by_energy();
	if ( option[ remove_singletons ].user() ) clustering->remove_singletons();
	if ( option[ limit_cluster_size ].user() ) clustering->limit_groupsize( option[ limit_cluster_size ] );
	if ( option[ limit_clusters ].user() )	clustering->limit_groups( option[ limit_clusters ] );
	if ( option[ limit_total_structures ].user() )	clustering->limit_total_structures( option[ limit_total_structures] );
	if ( option[ remove_highest_energy_member ].user() )	clustering->remove_highest_energy_member_of_each_group();

	// --------------------------------------------------------------------
	// Results:
	clustering->print_summary();
	clustering->print_cluster_PDBs( option[ out::prefix ]() );

	if ( option[ core::options::OptionKeys::cluster::make_ensemble_cst]() ) {
		EnsembleConstraints_Simple cec( 1.0 );
		clustering->create_constraints( option[ out::prefix ](), cec );
	}

	std::cout << "Timing: " << std::endl;
	std::cout <<    "  Readin:" << time_readin - time_start
						<< "s\n  Cluster: "    << time_initialc - time_readin
						<< "s\n  Additional Clustering: " << time_total - time_initialc
						<< "s\n  Total: " << time_total - time_start
						<< std::endl;

	return 0;
}

