// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file /src/apps/public/design/fixbb.cc
/// @brief  Fixed backbone design.  Can do side chain minimization after PackRotamers by using the flag -minimize_sidechains.  This is SLOW.

//core library
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/options/option.hh>
#include <core/options/util.hh>

#include <core/kinematics/MoveMap.hh>

//protocols library (Movers)
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/TaskAwareMinMover.hh>
#include <protocols/moves/MoverContainer.hh>

//utilities
#include <protocols/jd2/JobDistributor.hh>
#include <core/init.hh>

// option key includes
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>

//local options
namespace core{ namespace options{ namespace OptionKeys{
core::options::BooleanOptionKey const minimize_sidechains("minimize_sidechains");
}}}//core::options::OptionKeys

///////////////////////////////////////////////////////////////////////////////
int
main( int argc, char * argv [] )
{
	using namespace core::options;
	using namespace core::options::OptionKeys;

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// setup
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	option.add( minimize_sidechains, "Do minimization of side chains after rotamer packing").def(false);

	core::init(argc, argv);

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// end of setup
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//create a task factory: this will create a new PackerTask for each input pose
	core::pack::task::TaskFactoryOP main_task_factory = new core::pack::task::TaskFactory;
	main_task_factory->push_back( new core::pack::task::operation::InitializeFromCommandline );
	if ( option[ packing::resfile ].user() ) {
		main_task_factory->push_back( new core::pack::task::operation::ReadResfile );
	}

	//create a ScoreFunction from commandline options (default is score12)
	core::scoring::ScoreFunctionOP score_fxn = core::scoring::getScoreFunction();

	//create the PackRotamersMover which will do the packing
	protocols::moves::PackRotamersMoverOP pack_mover = new protocols::moves::PackRotamersMover;
	pack_mover->task_factory( main_task_factory );
	pack_mover->score_function( score_fxn );

	//This sequence mover will contain packing for sure, and may contain minimization
	protocols::moves::SequenceMoverOP seq_mover = new protocols::moves::SequenceMover;
	seq_mover->add_mover( pack_mover );

	//If sidechain minimization is requested, include that too
	if( option[ minimize_sidechains ] ){
		core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap;
		protocols::moves::MinMoverOP min_mover ( new protocols::moves::MinMover(
				movemap,
				score_fxn,
				core::options::option[ core::options::OptionKeys::run::min_type ].value(),
				0.01,
				true ) );
		protocols::moves::TaskAwareMinMoverOP TAmin_mover = new protocols::moves::TaskAwareMinMover(min_mover, main_task_factory);
		seq_mover->add_mover( TAmin_mover );
	} // end optional side chain minimization

	protocols::jd2::JobDistributor::get_instance()->go(seq_mover);

}
