// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file apps/pilot/smlewis/FloppyTail.cc
/// @brief This app is intended for modeling the binding of a long unstructured C-terminal tail to some other part of a protein.  It will work for internal regions too.
/// @author Steven Lewis

// Unit Headers

// Project Headers
#include <core/conformation/Conformation.hh>

#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/PDBPoseMap.hh>
#include <core/pose/PDBInfo.hh>

#include <core/kinematics/MoveMap.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/util.hh>

#include <core/fragment/ConstantLengthFragSet.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <protocols/toolbox/TaskOperations/RestrictToNeighborhoodOperation.hh>

#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreFunction.hh>

#include <core/scoring/constraints/util.hh>

#include <core/chemical/ChemicalManager.hh> //CENTROID, FA_STANDARD

#include <protocols/moves/MonteCarlo.hh>

//movers
#include <protocols/moves/BackboneMover.hh> //SmallMover
#include <protocols/abinitio/FragmentMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MoverContainer.hh> //Sequence Mover
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh> //typeset swapping
#include <protocols/moves/ReturnSidechainMover.hh>
#include <protocols/moves/TaskAwareMinMover.hh>
#include <protocols/moves/OutputMovers.hh> //pdbdumpmover

//JD headers
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/JobOutputter.hh>
#include <protocols/jd2/Job.hh>

// Utility Headers
#include <core/init.hh>
#include <core/options/util.hh>
#include <core/options/option.hh>
#include <core/util/Tracer.hh>
#include <utility/exit.hh>
#include <numeric/xyz.functions.hh>

// option key includes
#include <core/options/keys/run.OptionKeys.gen.hh>
#include <core/options/keys/AnchoredDesign.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("apps.pilot.smlewis.FloppyTail");

namespace core{ namespace options{ namespace OptionKeys{
core::options::IntegerOptionKey const flexible_start_resnum("flexible_start_resnum");
core::options::IntegerOptionKey const flexible_stop_resnum("flexible_stop_resnum");
core::options::StringOptionKey const flexible_chain("flexible_chain");
core::options::RealOptionKey const shear_on("shear_on");
core::options::RealOptionKey const short_tail_fraction("short_tail_fraction");
core::options::RealOptionKey const short_tail_off("short_tail_off");
core::options::BooleanOptionKey const pair_off("pair_off");
}}}//core::options::OptionKeys

///@details This function is specific to the original system for which this code was written - if you are not trying to duplicate the initial results you should remove it!
void create_extra_output( core::pose::Pose & pose, core::scoring::ScoreFunctionCOP score_fxn ){

	//score main pose
	core::Real const full_score((*score_fxn)(pose));

	//constants for hacking up pose
	core::Size const E2_chain(pose.conformation().num_chains()); //E2 is last chain
	core::Size const E2_start(pose.conformation().chain_begin(E2_chain));

	//create and score E3_RING portion
	core::pose::Pose E3_RING(pose);
	E3_RING.conformation().delete_residue_range_slow(E2_start, E3_RING.total_residue());
	core::Real const E3_RING_score((*score_fxn)(E3_RING));

	//create and score E2 portion
	core::pose::Pose E2 = pose.split_by_chain(E2_chain);
	core::Real const E2_score((*score_fxn)(E2));

	//print Job
	using protocols::jd2::JobDistributor;
	protocols::jd2::JobOP job_me( JobDistributor::get_instance()->current_job() );
	//debugging - check these pdbs
	//JobDistributor::get_instance()->job_outputter()->other_pose( job_me, E2, "E2");
	//JobDistributor::get_instance()->job_outputter()->other_pose( job_me, E3_RING, "E3_RING");

	//print Binding_Energy
	core::Real const BE(full_score - E3_RING_score - E2_score);
	job_me->add_string("Binding_Energy = Complex - E2 alone - E3/RING alone");
	job_me->add_string_real_pair("Binding energy", BE);
	//print crosslink distance
	//magic numbers: crosslink experiment tested these residues
	bool const NEDD8(E2_chain == 4);
	core::Size const cl_base(pose.pdb_info()->pdb2pose('A', (NEDD8 ? 1676: 679)));   //K1676??
	core::Size const cl_tail(pose.pdb_info()->pdb2pose('C', 223));

	core::Real const cl_dist(distance( pose.residue(cl_base).atom("CA").xyz(), pose.residue(cl_tail).atom("CA").xyz() ));
	job_me->add_string("Crosslink = distance between atom CA on residues (chain A, K679/K1676; chain C, 223)");
	job_me->add_string_real_pair("crosslink distance", cl_dist);

	//std::cout << full_score << " " << E3_RING_score << " " << E2_score << std::endl;

	return;
}

///@brief FloppyTail mover
class FloppyTailMover : public protocols::moves::Mover {
public:
	FloppyTailMover() : start_(0), stop_(0), init_for_input_yet_(false), centroid_scorefunction_(NULL),
											fullatom_scorefunction_(NULL), task_factory_(NULL), movemap_(NULL), movemap_lesstail_(NULL),
											fragset3mer_(NULL)
	{
		//this should be per-input, not in the ctor, if multiple frags files
		if (core::options::option[ core::options::OptionKeys::in::file::frag3].user()){
			fragset3mer_ = new core::fragment::ConstantLengthFragSet( 3 );
			fragset3mer_->read_fragment_file( core::options::option[ core::options::OptionKeys::in::file::frag3].value() );
		}

		bool const pair_off(core::options::option[ core::options::OptionKeys::pair_off ].value() );

		//set up centroid scorefunction
		using namespace core::scoring;
		centroid_scorefunction_ = new ScoreFunction;
		centroid_scorefunction_->set_weight( env,         1.0 );
		centroid_scorefunction_->set_weight( cbeta,       1.0 );
		centroid_scorefunction_->set_weight( vdw,         1.0 );
		centroid_scorefunction_->set_weight( pair, (pair_off ? 0.0 : 1.0) ); //no pair term experiment - not for general use
		centroid_scorefunction_->set_weight( cenpack,     1.0 );
		centroid_scorefunction_->set_weight( rama,        1.0 );
		centroid_scorefunction_->set_weight( hbond_lr_bb, 1.0 );
		centroid_scorefunction_->set_weight( hbond_sr_bb, 1.0 );
		core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn( *centroid_scorefunction_ ); //protected if(option) internally
		TR << "Using centroid scorefunction\n" << *centroid_scorefunction_;

		//set up fullatom scorefunction
		fullatom_scorefunction_ = ScoreFunctionFactory::create_score_function( STANDARD_WTS, SCORE12_PATCH );
		if( pair_off ) fullatom_scorefunction_->set_weight( fa_pair, 0.0 ); //not for general use
		TR << "Using fullatom scorefunction (STANDARD_WTS, SCORE12_PATCH), pair may be modified\n"
			 << *fullatom_scorefunction_;

	}

	///@brief init_on_new_input system allows for initializing these details the first time apply() is called.  the job distributor will reinitialize the whole mover when the input changes (a freshly constructed mover, which will re-run this on first apply().
	virtual
	void
	init_on_new_input(core::pose::Pose const & pose) {
		init_for_input_yet_ = true;

		//determine where the flexible tail is
		using namespace core::options;
		using namespace core::options::OptionKeys;
		char const chain(option[flexible_chain].value()[0]); //just take the first one
		start_ = pose.pdb_info()->pdb2pose().find(chain, option[flexible_start_resnum].value());
		if(option[flexible_stop_resnum].user())
			stop_ = pose.pdb_info()->pdb2pose().find(chain, option[flexible_stop_resnum].value());
		else
			stop_ = pose.total_residue();

		//setup of TaskFactory
		using namespace core::pack::task;
		task_factory_ = new TaskFactory;
		task_factory_->push_back( new operation::InitializeFromCommandline );
		if ( option[ packing::resfile ].user() ) {
			task_factory_->push_back( new operation::ReadResfile );
		}
		std::set< core::Size > crset;
		TR << "Tail is from " << start_ << " to " << stop_ << std::endl;
		for(core::Size i(start_); i<=stop_; ++i) crset.insert(i);
		task_factory_->push_back( new protocols::toolbox::TaskOperations::RestrictToNeighborhoodOperation( crset ) );

		//setup MoveMap
		movemap_ = new core::kinematics::MoveMap;
		for(core::Size i(start_); i<=stop_; ++i) {
			movemap_->set_bb(i, true); //backbone mobile
			movemap_->set_chi(i, true); // chi of loop residues
		}

		movemap_lesstail_ = new core::kinematics::MoveMap(*movemap_);
		if (stop_ == pose.total_residue()){
			core::Size const taillength = stop_ - start_;
			core::Size const substop = (start_ + core::Size(core::Real(taillength) * (1.0 - option[ short_tail_fraction ] ) ) );
			for(core::Size i(start_); i <= substop; ++i) {
				movemap_lesstail_->set_bb(i, false);
			}
		}

	}

	virtual ~FloppyTailMover(){};

	virtual
	void
	apply( core::pose::Pose & pose ){
		if( !init_for_input_yet_ ) init_on_new_input(pose);

		//for nonterminal sections, this fold tree might help (or might make it worse...)
		if( stop_ != pose.total_residue() ){
			TR << "non-terminal floppy section, using a linear fold tree to try to ensure downstream residues follow\n"
				 << "Old tree: " << pose.fold_tree();
			pose.fold_tree(core::kinematics::linearize_fold_tree(pose.fold_tree()));
			TR << pose.fold_tree();
		}

		TR << "foldtree, movemap: " << std::endl;
		core::kinematics::simple_visualize_fold_tree_and_movemap( pose.fold_tree(), *movemap_, TR);
		if( stop_ == pose.total_residue() ){
			TR << "foldtree, movemap for first part of refine: " << std::endl;
			core::kinematics::simple_visualize_fold_tree_and_movemap( pose.fold_tree(), *movemap_lesstail_, TR);
		}

		core::scoring::constraints::add_constraints_from_cmdline_to_pose( pose ); //protected internally if no constraints

		//centroid
		clock_t starttime = clock();
		TR << "entering perturb steps" << std::endl;

		core::pose::Pose const saved_input_pose( pose ); //used to return sidechains later

		protocols::moves::SwitchResidueTypeSetMover typeset_swap(core::chemical::CENTROID);
		typeset_swap.apply( pose );
		//centroid score
		TR << "centroid score of starting PDB: " << (*centroid_scorefunction_)(pose) << std::endl;
		centroid_scorefunction_->show( TR, pose );
		TR << std::flush; //show doesn't flush the buffer

		/*
			perturbmover of either type
			minimize every once in a while
			MC evaluate
		*/

		////////////////////////////////////backbone_mover_cent/////////////////////////////////////
		protocols::moves::RandomMoverOP backbone_mover_cen( new protocols::moves::RandomMover() );

		using protocols::moves::SmallMover;
		using protocols::moves::BackboneMoverOP;
		BackboneMoverOP small_mover_cen = new SmallMover(movemap_, 0.8, 0);
		small_mover_cen->angle_max( 'H', 180.0 );
		small_mover_cen->angle_max( 'E', 180.0 );
		small_mover_cen->angle_max( 'L', 180.0 );
		backbone_mover_cen->add_mover(small_mover_cen, 1.0);

		BackboneMoverOP shear_mover_cen = new protocols::moves::ShearMover(movemap_, 0.8, 0);
		shear_mover_cen->angle_max( 'H', 180.0 );
		shear_mover_cen->angle_max( 'E', 180.0 );
		shear_mover_cen->angle_max( 'L', 180.0 );
		//backbone_mover_cen->add_mover(shear_mover_cen, 1.0); //not yet

		if(fragset3mer_){ //if we have fragments
			using protocols::abinitio::ClassicFragmentMover;
			protocols::abinitio::ClassicFragmentMoverOP frag_mover = new ClassicFragmentMover(fragset3mer_, movemap_);
			frag_mover->enable_end_bias_check(false);
			backbone_mover_cen->add_mover(frag_mover, 0.5);
		}

		/////////////////////////minimizer mover/////////////////////////////////////////
		using protocols::moves::MinMoverOP;
		using protocols::moves::MinMover;
		MinMoverOP min_mover_cen = new MinMover(
																						movemap_,
																						centroid_scorefunction_,
																						core::options::option[ core::options::OptionKeys::run::min_type ].value(),
																						0.01,
																						true /*use_nblist*/ );

		/////////////////////////Monte Carlo//////////////////////////////////////////////////////////
		//make the monte carlo object
		using protocols::moves::MonteCarlo;
		using protocols::moves::MonteCarloOP;
		using core::options::option;
		using namespace core::options::OptionKeys::AnchoredDesign;
		MonteCarloOP mc_cen( new MonteCarlo( pose, *centroid_scorefunction_, option[ perturb_temp ].value() ) );

		///////////////////////////////////for loop///////////////////////////////////////////////////
		protocols::moves::PDBDumpMover cen_out("cen_cycle");
		core::Size const perturb_applies = option[ perturb_cycles ].value(); //default 5
		core::Size shear_on_cyc(core::Size(core::Real(perturb_applies) * option[ core::options::OptionKeys::shear_on ]));
		if(shear_on_cyc == 0) shear_on_cyc = 1; //0 should mean shear on immediately, but the if below never sees 0.
		TR << "shear on at " << shear_on_cyc << std::endl;
		TR << "   Current     Low    total cycles =" << perturb_applies << std::endl;
		for ( core::Size i = 1; i <= perturb_applies; ++i ) {
			if( i == shear_on_cyc ) backbone_mover_cen->add_mover(shear_mover_cen, 1.0);
			if( (i % 20 == 0) || (i == perturb_applies) ) min_mover_cen->apply(pose);
			else backbone_mover_cen->apply(pose);

			if( option[ debug ] )cen_out.apply(pose); //check trajectory
			mc_cen->boltzmann(pose);

			TR << i << "  " << mc_cen->last_accepted_score() << "  " << mc_cen->lowest_score() << std::endl;

			//constraint report
			//using core::scoring::atom_pair_constraint;
			//core::Real const cstscore_in(pose.energies().total_energies()[atom_pair_constraint] * pose.energies().weights()[atom_pair_constraint]);
			//TR << "cst score " << cstscore_in << std::endl;
		}//end the exciting for loop
		mc_cen->recover_low(pose);

		//filter based on constraints score - if not less than 1 (close to 0), cancel this trajectory
		using core::scoring::atom_pair_constraint;
		core::Real const cstscore(pose.energies().total_energies()[atom_pair_constraint] * pose.energies().weights()[atom_pair_constraint]);
		if (cstscore > 1.0) {
			TR << "centroid constraints not satisfied; final constraint score: " << cstscore << ", restarting centroid" << std::endl;
			set_last_move_status(protocols::moves::FAIL_RETRY);
			return;
		}

		//dump centroid-stage result pose
		if ( core::options::option[core::options::OptionKeys::AnchoredDesign::perturb_show ].value() ) {
			using namespace protocols::jd2;
			JobCOP job_me( JobDistributor::get_instance()->current_job() );
			JobDistributor::get_instance()->job_outputter()->other_pose( job_me, pose, "centroid");
		}

		//show centroid score (duplicates last line above)
		TR << "centroid score of final perturbed PDB: " << (*centroid_scorefunction_)(pose) << std::endl;
		centroid_scorefunction_->show( TR, pose );
		TR << std::flush; //show doesn't flush the buffer

		clock_t stoptime = clock();
		TR << "One perturb took " << ((double) stoptime - starttime )/CLOCKS_PER_SEC << " seconds" << std::endl;
		TR << "perturb steps complete" << std::endl;
		starttime = clock();

		///////////////////////////////////////////////////////////////////////////////////////////////////////
		////////////////////////////////////fullatom///////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////////////////////////////////

		protocols::moves::ReturnSidechainMover return_sidechains( saved_input_pose );
		return_sidechains.apply( pose );

		/////////////////////////////generate full repack&minimize mover//////////////////////////////
		protocols::moves::PackRotamersMoverOP pack_mover = new protocols::moves::PackRotamersMover;
		pack_mover->task_factory( task_factory_ );
		pack_mover->score_function( fullatom_scorefunction_ );

		MinMoverOP min_mover_fa = new MinMover(
																					 movemap_,
																					 fullatom_scorefunction_,
																					 core::options::option[ core::options::OptionKeys::run::min_type ].value(),
																					 0.01,
																					 true /*use_nblist*/ );

		//definitely want sidechain minimization here
		using protocols::moves::TaskAwareMinMoverOP;
		using protocols::moves::TaskAwareMinMover;
		TaskAwareMinMoverOP TAmin_mover_fa = new TaskAwareMinMover(min_mover_fa, task_factory_);

		/////////////////////////repack/minimize once to fix sidechains//////////////////////////////////
		pack_mover->apply(pose);
		TAmin_mover_fa->apply(pose);

		//////////////////////////////////////// backbone mover/////////////////////////////////////////
		protocols::moves::RandomMoverOP backbone_mover_fa( new protocols::moves::RandomMover() );

		BackboneMoverOP small_mover_fa = new SmallMover(movemap_lesstail_, 0.8, 0);
		small_mover_fa->angle_max( 'H', 4.0 );
		small_mover_fa->angle_max( 'E', 4.0 );
		small_mover_fa->angle_max( 'L', 4.0 );

		BackboneMoverOP shear_mover_fa = new protocols::moves::ShearMover(movemap_lesstail_, 0.8, 0);
		shear_mover_fa->angle_max( 'H', 4.0 );
		shear_mover_fa->angle_max( 'E', 4.0 );
		shear_mover_fa->angle_max( 'L', 4.0 );

		backbone_mover_fa->add_mover(small_mover_fa, 1.0);
		backbone_mover_fa->add_mover(shear_mover_fa, 1.0);

		/////////////////fullatom Monte Carlo//////////////////////////////////////////////////////////
		//make the monte carlo object
		MonteCarloOP mc_fa( new MonteCarlo( pose, *fullatom_scorefunction_, option[ refine_temp ].value() ) );

		/////////////////////////////////rotamer trials mover///////////////////////////////////////////
		using protocols::moves::RotamerTrialsMoverOP;
		using protocols::moves::EnergyCutRotamerTrialsMover;
		RotamerTrialsMoverOP rt_mover(new EnergyCutRotamerTrialsMover(
																																	fullatom_scorefunction_,
																																	task_factory_,
																																	mc_fa,
																																	0.01 /*energycut*/ ) );

		/////////////////////////////////////////refine loop///////////////////////////////////////////
		core::Size const refine_applies = option[ refine_cycles ].value(); //default 5
		core::Size const repack_cycles = option[ refine_repack_cycles ].value();
		core::Size const min_cycles = repack_cycles/2;
		core::Size const switch_movemaps(core::Size(core::Real(refine_applies) * option[ core::options::OptionKeys::short_tail_off ]));
		TR << "   Current     Low    total cycles =" << refine_applies << std::endl;
		for ( core::Size i(1); i <= refine_applies; ++i ) {
			if( i == switch_movemaps ){
				small_mover_fa->movemap(movemap_);
				shear_mover_fa->movemap(movemap_);
			}
			if( (i % repack_cycles == 0) || (i == refine_applies) ) { //full repack
				pack_mover->apply(pose);
				TAmin_mover_fa->apply(pose);
			} else if ( i % min_cycles == 0 ) { //minimize
				TAmin_mover_fa->apply(pose);
			} else {
				backbone_mover_fa->apply(pose);
				rt_mover->apply(pose);
			}

			mc_fa->boltzmann(pose);
			TR << i << "  " << mc_fa->last_accepted_score() << "  " << mc_fa->lowest_score() << std::endl;
		}//end the exciting for loop
		mc_fa->recover_low( pose );

		//let's store some energies/etc of interest
		//this code is specific to the E2/RING/E3 system for which this code was written; it is refactored elsewhere
		if (true) create_extra_output(pose, fullatom_scorefunction_);

		(*fullatom_scorefunction_)(pose);
		set_last_move_status(protocols::moves::MS_SUCCESS); //this call is unnecessary but let's be safe
		return;
	}

	virtual
	protocols::moves::MoverOP
	fresh_instance() const {
		return new FloppyTailMover;
	}

	virtual
	bool
	reinitialize_for_each_job() const { return false; }

	virtual
	bool
	reinitialize_for_new_input() const { return true; }

private:
	core::Size start_;
	core::Size stop_;
	bool init_for_input_yet_;

	core::scoring::ScoreFunctionOP centroid_scorefunction_;
	core::scoring::ScoreFunctionOP fullatom_scorefunction_;
	core::pack::task::TaskFactoryOP task_factory_;
	core::kinematics::MoveMapOP movemap_;
	core::kinematics::MoveMapOP movemap_lesstail_;
	core::fragment::ConstantLengthFragSetOP fragset3mer_;
};

typedef utility::pointer::owning_ptr< FloppyTailMover > FloppyTailMoverOP;

int main( int argc, char* argv[] )
{
	using core::options::option;
	using namespace core::options::OptionKeys;
	option.add( flexible_start_resnum, "start res for flexible region").def(180);
	option.add( flexible_stop_resnum, "stop res for flexible region").def(0);
	option.add( flexible_chain, "chain for flexible region").def("C");
	option.add( shear_on, "fraction of perturb moves when shear turns on (0.5 = halfway through").def(1.0/3.0);
	option.add( short_tail_fraction, "what fraction of the flexible segment is used in the short-tail section of refinement (not compatible with non-terminal flexible regions)").def(1.0/3.0);
	option.add( short_tail_off, "fraction of refine cycles where movemap reverts to full tail (0.5 = halfway through)").def(0.1);
	option.add( pair_off, "turn off Epair electrostatics term").def(false);
	core::init(argc, argv);

	protocols::jd2::JobDistributor::get_instance()->go(new FloppyTailMover);

	TR << "************************d**o**n**e**************************************" << std::endl;

	return 0;
}
