// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/chemical/ChemicalManager.cc
/// @brief  Chemical manager class
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)

#include <core/chemical/ChemicalManager.hh>
// Commented by inclean daemon #include <core/chemical/AtomTypeSet.hh>
// Commented by inclean daemon #include <core/chemical/MMAtomTypeSet.hh>
#include <core/chemical/ResidueTypeSet.hh>

#include <utility/vector1.hh>

// Project headers
#include <core/io/database/open.hh>
#include <core/util/Tracer.hh>

#include <core/options/option.hh>

#include <utility/file/file_sys_util.hh>

// option key includes

#include <core/options/keys/in.OptionKeys.gen.hh>



namespace core {
namespace chemical {

static util::Tracer tr("core.chemical.ChemicalManager");

/// @brief set initial value as no instance
ChemicalManager* ChemicalManager::instance_( 0 );

/// @brief static function to get the instance of ( pointer to) this singleton class
ChemicalManager * ChemicalManager::get_instance()
{
	if ( instance_ == 0 )
	{
		 instance_ = new ChemicalManager();
	}
	return instance_;
}

/// @brief private constructor to guarantee the singleton
ChemicalManager::ChemicalManager(){}

/// @details if the tag is not in the map, input it from a database file and add it
/// to the map for future look-up.
AtomTypeSetCAP
ChemicalManager::atom_type_set( std::string const & tag )
{
	AtomTypeSets::const_iterator iter( atom_type_sets_.find( tag ) );
	if ( iter == atom_type_sets_.end() ) {
		// read from file
		std::string const directory( io::database::full_name( "chemical/atom_type_sets/"+tag+"/" ) );
		AtomTypeSetOP new_set( new AtomTypeSet( directory ) );
		iter = atom_type_sets_.insert( std::make_pair( tag, new_set ) ).first;
	}
	return iter->second();
}

/// @details if the tag is not in the map, input it from a database file and add it
/// to the map for future look-up.
MMAtomTypeSetCAP
ChemicalManager::mm_atom_type_set( std::string const & tag )
{
	MMAtomTypeSets::const_iterator iter( mm_atom_type_sets_.find( tag ) );
	if ( iter == mm_atom_type_sets_.end() ) {
		// read from file
		std::string const filename( io::database::full_name( "chemical/mm_atom_type_sets/"+tag+"/mm_atom_properties.txt" ));
		MMAtomTypeSetOP new_set( new MMAtomTypeSet() );
		new_set->read_file( filename );
		iter = mm_atom_type_sets_.insert( std::make_pair( tag, new_set ) ).first;
	}
	return iter->second();
}


// /// @details if the tag is not in the map, input it from a database file and add it
// /// to the map for future look-up.
//CSDAtomTypeSetCAP
//ChemicalManager::csd_atom_type_set( std::string const & tag )
//{
//	CSDAtomTypeSets::const_iterator iter( csd_atom_type_sets_.find( tag ) );
//	if ( iter == csd_atom_type_sets_.end() ) {
		// read from file
//		std::string const filename( io::database::full_name( "chemical/csd_atom_type_sets/"+tag+"/csd_atom_properties.txt" ));
//		CSDAtomTypeSetOP new_set( new CSDAtomTypeSet() );
//		new_set->read_file( filename );
//		iter = csd_atom_type_sets_.insert( std::make_pair( tag, new_set ) ).first;
//	}
//	return iter->second();
//}


///@ details if the tag is not in the map, input it from a database file and add it
///to the map for future look-up.
ResidueTypeSetCAP
ChemicalManager::residue_type_set( std::string const & tag )
{

	using namespace util;

	ResidueTypeSets::const_iterator iter( residue_type_sets_.find( tag ) );
	if ( iter == residue_type_sets_.end() ) {
		// Look for additional residue .params files specified on the cmd line
		std::vector< std::string > extra_params_files;
		if(tag == FA_STANDARD) {
			utility::options::FileVectorOption & fvec
				= core::options::option[ core::options::OptionKeys::in::file::extra_res_fa ];
			for(Size i = 1, e = fvec.size(); i <= e; ++i) {
				utility::file::FileName fname = fvec[i];
				extra_params_files.push_back(fname.name());
			}

			utility::options::PathVectorOption & pvec
				= core::options::option[core::options::OptionKeys::in::file::extra_res_path];
			// convert Pathname->string->char*, glob it, convert char*->string
			for(Size i=1, e= pvec.size(); i<=e; i++){
				utility::vector1<std::string> files;
				std::string directory=pvec[i].name();

				utility::file::list_dir(directory, files);
				tr.Debug<< std::endl;
				for(size_t j=1; j<= files.size(); j++){
					if (files[j].find("param")!=std::string::npos){
						tr.Debug << files[j]<< ", ";
						std::string path= directory+'/'+files[j];
						extra_params_files.push_back(path);
					}
				}
				tr.Debug<< std::endl;
			}

		} else if(tag == CENTROID) {
			utility::options::FileVectorOption & fvec
				= core::options::option[ core::options::OptionKeys::in::file::extra_res_cen ];
			for(Size i = 1, e = fvec.size(); i <= e; ++i) {
				utility::file::FileName fname = fvec[i];
				extra_params_files.push_back(fname.name());
			}
		}
		// read from file
		tr.Debug << "CHEMICAL_MANAGER: read residue types: " << tag << std::endl;
		// redirecting to new icoor folder
		std::string temp_str( io::database::full_name( "chemical/residue_type_sets/"+tag ) );
		if(tag == FA_STANDARD) {
			if (options::option[core::options::OptionKeys::in::new_icoor]) {
				temp_str += "_05.2009_icoor";
			}
		}
		temp_str += "/";

		std::string const directory( temp_str );
		ResidueTypeSetOP new_set( new ResidueTypeSet( tag, directory, extra_params_files ) );
		iter = residue_type_sets_.insert( std::make_pair( tag, new_set ) ).first;
	}
	return iter->second();
}


///@ details if the tag is not in the map, input it from a database file and add it
///to the map for future look-up.
ResidueTypeSet &
ChemicalManager::nonconst_residue_type_set( std::string const & tag )
{
	// trigger initialization if necessary:
	residue_type_set( tag );

	return *( residue_type_sets_.find( tag )->second );
}



// global data
/// @brief tag name for querying fullatom chemical type set.
std::string const FA_STANDARD( "fa_standard" );
/// @brief tag name for querying centroid chemical type set.
std::string const CENTROID( "centroid" );
/// @brief tag name for querying coarse-grained chemical type set.
std::string const COARSE_TWO_BEAD( "coarse_two_bead" );
/// @brief tag name for querying hybrid fullatom+centroid chemical type set.
std::string const HYBRID_FA_STANDARD_CENTROID( "hybrid_fa_standard_centroid" );
/// @brief tag name for querying RNA chemical type set.
std::string const RNA( "rna" );

} // namespace core
} // namespace chemical


/// THIS TURNED OUT TO BE MORE TROUBLE THAN IT WAS WORTH: but maybe some day...


/**
/// @details  Duplicate a ResidueTypeSet, preparatory to modifying it in some way DOES NOT DUPLICATE ITS ATOMTYPESETS
/// Uses ResidueTypeSet::clone()

void
ChemicalManager::copy_residue_type_set(
																			 std::string const & old_name,
																			 std::string const & new_name
																			 )
{
	residue_type_set( old_name ); // triggers initialization if necessary
	if ( residue_type_sets_.find( new_name ) ) {
		utility_exit_with_message( "new name is already being used!" );
	}
	residue_type_sets_.insert( std::make_pair( new_name, residue_type_sets_.find( old_name )->second->clone() ) );
}

/// @details  Duplicate an AtomTypeSet, preparatory to modifying it in some way
/// Uses AtomTypeSet::clone()

void
ChemicalManager::copy_atom_type_set(
																		std::string const & old_name,
																		std::string const & new_name
																		)
{
	atom_type_set( old_name ); // triggers initialization if necessary
	if ( atom_type_sets_.find( new_name ) ) {
		utility_exit_with_message( "new name is already being used!" );
	}
	atom_type_sets_.insert( std::make_pair( new_name, atom_type_sets_.find( old_name )->second->clone() ) );
}
**/

