// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 sw=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file core/chemical/disulfide_util.hh
/// @brief A collection of procedures for manipulating disulfide bonds
/// @author Spencer Bliven <blivens@u.washington.edu>
/// @date 4/30/2009

#ifndef INCLUDED_core_chemical_disulfide_util_HH_
#define INCLUDED_core_chemical_disulfide_util_HH_


// Project Headers
#include <core/pack/task/PackerTask.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/types.hh>
#include <core/pose/Pose.hh>

// Utility headers
#include <utility/vector1.hh>

// C++ headers
#include <utility>

// Unit headers

namespace core {
namespace chemical {

/// @brief Replace a CYS with a CYD or vice-versa for changing disulfide bonds.
/// @param[in] index Position of the residue to replace.
/// @param[in] cys_type_name3 The 3-letter name of the cys type to use: either CYS
///  or CYD.
/// @param[inout] conf The conformation to modify
/// @return true if the replacement was successful, false otherwise.
bool change_cys_state( Size const index, std::string cys_type_name3, core::conformation::Conformation & conf );

/// @brief Introduce cysteines at the specified location and define a
///  disulfide bond between them.
/// @details Does not do the repacking & minimization required to place the
///   disulfide correctly.
void form_disulfide(core::pose::Pose & pose, core::Size lower_res, core::Size upper_res);

/// @brief Rebuild a pair of cysteines (and possibly surrounding residues) so
///  that they form a near-ideal disulfide bond
void rebuild_disulfide( core::pose::Pose & pose,
	core::Size lower_res, core::Size upper_res,
	core::pack::task::PackerTaskOP packer_task = 0,
	core::scoring::ScoreFunctionOP packer_score = 0,
	core::kinematics::MoveMapOP mm = 0,
	core::scoring::ScoreFunctionOP minimizer_score = 0 );

/// @brief Rebuild a number of pairs of cysteines (and possibly surrounding
///  residues) so that they form near-ideal disulfide bonds
void rebuild_disulfide( core::pose::Pose & pose,
	utility::vector1<std::pair<core::Size, core::Size> > disulfides,
	core::pack::task::PackerTaskOP packer_task = 0,
	core::scoring::ScoreFunctionOP packer_score = 0,
	core::kinematics::MoveMapOP mm = 0,
	core::scoring::ScoreFunctionOP minimizer_score = 0 );

/// @brief Find whether there is a disulfide defined between two residues
bool is_disulfide_bond( core::pose::Pose const& pose,
	core::Size residueA_pos, core::Size residueB_pos);

/// @brief Generate a list of all disulfide bonds in the pose
void disulfide_bonds( core::pose::Pose const& pose,
	utility::vector1< std::pair<core::Size,core::Size> > & disulfides );

} // chemical
} // core

#endif //INCLUDED_core_chemical_disulfide_util_HH_

